package org.gcube.dataanalysis.copernicus.motu.client;

import org.gcube.dataanalysis.copernicus.motu.model.MotuUnmarshaller;
import org.gcube.dataanalysis.copernicus.motu.model.ProductMetadataInfo;
import org.gcube.dataanalysis.copernicus.motu.model.RequestSize;
import org.gcube.dataanalysis.copernicus.motu.model.StatusModeResponse;
import org.gcube.dataanalysis.copernicus.motu.util.NetworkUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This is a plain implementation of a Motu client, exposing exactly the
 * operations provided by the server.
 * @author Paolo Fabriani
 */
public class PlainMotuClient {

    /**
     * A logger for this class.
     */
    private static Logger logger = LoggerFactory.getLogger(PlainMotuClient.class);

    /**
     * The endpoint of the Motu server.
     */
    private String serviceURL;

    /**
     * The proxy to the CAS server.
     */
    private CASClient casProxy;

    /**
     * Credentials for the Motu server: username
     */
    private String username;

    /**
     * Credentials for the Motu server: password
     */
    private String password;

    public PlainMotuClient(String serviceURL) {
        this.serviceURL = serviceURL;
        this.casProxy = new CASClient();
    }

    /**
     * Set the username to use when connecting to the Motu server.
     * @param username
     *            the username.
     */
    public void setUsername(String username) {
        this.username = username;
        this.casProxy.setUsername(username);
    }

    /**
     * Set the passowrd to use when connecting to the Motu server.
     * @param password
     *            the password.
     */
    public void setPassword(String password) {
        this.password = password;
        this.casProxy.setPassword(password);
    }

    /**
     * Return the endpoint of the Motu server.
     * @return the endpoint of the Motu sever.
     */
    public String getServiceURL() {
        return serviceURL;
    }

    /**
     * Asynchronously submit a request for a dataset
     * 
     * @param request
     * @return the status of the submitted request
     * @throws Exception
     */
    public StatusModeResponse queueProductDownload(
            final DownloadRequest request) throws Exception {
        request.setAction("productdownload");
        logger.info("queuing request for product download...");
        String url = this.getServiceURL() + "?"
                + NetworkUtils.getDataString(request.getParametersMap());
        String xmlStatus = NetworkUtils.doGet(url, this.getCasProxy());
        logger.debug("status xml is " + xmlStatus);
        return MotuUnmarshaller.getStatusModeResponse(xmlStatus);
    }

    /**
     * Retrieves an estimation of the size of the response
     * 
     * @param request
     *            all parameters also accepted by "downloadProdyct"
     * @return the estimated size of the output, including the maximum size
     *         allowed by the server
     * @throws Exception
     */
    public RequestSize getSize(final DownloadRequest request) throws Exception {
        request.setAction("getsize");
        logger.info("requesting response size...");
        String url = this.getServiceURL() + "?"
                + NetworkUtils.getDataString(request.getParametersMap());
        String xmlStatus = NetworkUtils.doGet(url, this.getCasProxy());
        logger.debug("status xml is " + xmlStatus);
        return MotuUnmarshaller.getRequestSize(xmlStatus);
    }

    /**
     * Retrieve metadata about a product
     * 
     * @param request
     *            mandatory parameters are "service" and "product"
     * @return request metadata
     * @throws Exception
     */
    public ProductMetadataInfo describeProduct(final DownloadRequest request)
            throws Exception {
        request.setAction("describeproduct");
        logger.info("getting product metadata");
        String url = this.getServiceURL() + "?"
                + NetworkUtils.getDataString(request.getParametersMap());
        String xmlMetadata = NetworkUtils.doGet(url, this.getCasProxy());
        // logger.debug("metadata xml is " + xmlMetadata);
        return MotuUnmarshaller.getProductMetadataInfo(xmlMetadata);
    }

    /**
     * @param request
     *            mandatory parameters are "service" and "product"
     * @return request metadata
     * @throws Exception
     */
    public void describeCoverage(final DownloadRequest request)
            throws Exception {
        request.setAction("describecoverage");
        logger.info("getting product coverage");
        String url = this.getServiceURL() + "?"
                + NetworkUtils.getDataString(request.getParametersMap())
                + "&datasetID=" + request.getProduct();
        String xmlCoverage = NetworkUtils.doGet(url, this.getCasProxy());
        logger.debug("coverage xml is " + xmlCoverage);
        // return MotuUnmarshaller.getProductMetadataInfo(xmlMetadata);
    }

    /**
     * Check the status of a submitted request
     * 
     * @param requestId
     *            the id of the request to check status for.
     * @return the status of the ongoing request
     * @throws Exception
     */
    public StatusModeResponse checkStatus(final String requestId)
            throws Exception {
        logger.info("checking status for request " + requestId);
        MultiValueParameters params = new MultiValueParameters();
        params.put("action", "getreqstatus");
        params.put("requestid", requestId);
        String url = this.getServiceURL() + "?"
                + NetworkUtils.getDataString(params);
        String xmlStatus = NetworkUtils.doGet(url, this.getCasProxy());
        logger.debug("status xml is " + xmlStatus);
        return MotuUnmarshaller.getStatusModeResponse(xmlStatus);
    }

    /**
     * Polls the server until the request has been fulfilled. Blocks the caller.
     * @param requestId
     *            the id of the request
     * @return the status response of the last (ready or failed) check
     * @throws Exception
     */
    public StatusModeResponse waitForComplete(final String requestId)
            throws Exception {
        StatusModeResponse statusResponse = null;
        while (true) {
            statusResponse = this.checkStatus(requestId);
            logger.debug(statusResponse.getStatus());
            if (statusResponse.isInProgress()) {
                logger.info("Product is not yet ready. Waiting...");
            } else if (statusResponse.isReady()) {
                logger.info("Product is ready for download from "
                        + statusResponse.getRemoteUri());
                break;
            } else if (statusResponse.isError()) {
                logger.error(statusResponse.getMessage());
                break;
            }
            Thread.sleep(20000);
        }
        return statusResponse;
    }

    /**
     * Return the proxy to the CAS sever.
     * @return the CAS server proxy.
     */
    public CASClient getCasProxy() {
        return casProxy;
    }

}
