package org.gcube.dataanalysis.copernicus.motu.client;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.gcube.dataanalysis.copernicus.motu.util.NetworkUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A class implementing interaction mechanisms with a CAS server.
 *
 * @author Paolo Fabriani
 */
public class CASClient {

    /**
     * A Logger for this class.
     */
    private static Logger logger = LoggerFactory.getLogger(CASClient.class);

    /**
     * The CAS endpoint.
     */
    private String endpoint;

    /**
     * CAS credentials: username.
     */
    private String username;

    /**
     * CAS credentials: password.
     */
    private String password;

    /**
     * Empty constructor.
     */
    public CASClient() {
    }

    /**
     * Build a client for the given endpoint.
     * 
     * @param endpoint
     *            the endpoint of the CAS server
     */
    public CASClient(final String endpoint) {
        this();
        this.endpoint = endpoint;
    }

    /**
     * Build a client for the given endpoint, using the given credentials.
     * 
     * @param endpoint
     *            the CAS server endpoint
     * @param username
     *            the username
     * @param password
     *            the password
     */
    public CASClient(final String endpoint, final String username,
            final String password) {
        this(endpoint);
        this.setUsername(username);
        this.setPassword(password);
    }

    /**
     * Return the endpoint of the CAS server
     * 
     * @return the server endpoint
     */
    public String getEndpoint() {
        if(this.endpoint.startsWith("http:"))
            return this.endpoint.replaceFirst("http:", "https:");
        return endpoint;
    }

    /**
     * @param endpoint
     * @return
     */
    public boolean isAuthnEndpoint(String endpoint) {
        return endpoint.indexOf("/login?") != -1;
    }

    public void setEndpoint(String endpoint) {
        if (endpoint.indexOf("/login?") != -1) {
            endpoint = endpoint.substring(0, endpoint.indexOf("/login?"));
            endpoint += "/v1/tickets";
        }
        this.endpoint = endpoint;
    }

    /**
     * Return the username.
     *
     * @return the username set for this client.
     */
    public String getUsername() {
        return username;
    }

    /**
     * Set the username.
     * @param username
     *            the username to set for this client
     */
    public void setUsername(final String username) {
        this.username = username;
    }

    /**
     * Return the password.
     * @return the password set for this client
     */
    public String getPassword() {
        return password;
    }

    /**
     * Set the password.
     *
     * @param password
     *            the password to set
     */
    public void setPassword(final String password) {
        this.password = password;
    }

    /**
     * Utility operation to extract the TGT from the CAS reponse.
     *
     * @param response
     * @return the TGT
     */
    private static String extractTGT(final String response) {
        String tgt = null;
        Pattern p = Pattern.compile("/(TGT[^\"]+)");
        Matcher m = p.matcher(response);
        if (m.find()) {
            tgt = m.group(1);
        }
        return tgt;
    }

    /**
     * Contact the server to get a Ticket-Granting-Ticket (TGT).
     *
     * @return
     * @throws Exception
     */
    private String getTicketGrantTicket() throws Exception {
        logger.info("requesting a TGT...");
        MultiValueParameters params = new MultiValueParameters();
        params.put("username", this.getUsername());
        params.put("password", this.getPassword());
        String response = NetworkUtils.post(this.getEndpoint(), params);
        logger.debug("got response " + response);
        String tgt = CASClient.extractTGT(response);
        logger.info("got TGT: " + tgt);
        return tgt;
    }

    /**
     * Contact the server to retrieve a Service Ticket (ST) for the given
     * endpoint. The method includes the connection to the server to get a
     * Ticket-Granting-Ticket (TGT).
     * 
     * @param serviceRequest
     * @return
     * @throws Exception
     */
    private String getServiceTicketFor(String serviceRequest) throws Exception {
        logger.info("requesting a ST for " + serviceRequest);
        String tgt = this.getTicketGrantTicket();
        if(tgt==null) {
            throw new Exception("got an empty TGT");
        }
        String url = this.getEndpoint() + "/" + tgt;
        MultiValueParameters params = new MultiValueParameters();
        params.put("service", serviceRequest.toString());
        String serviceTicket = NetworkUtils.post(url, params).trim();
        logger.info("got ST: " + serviceTicket);
        return serviceTicket;
    }

    /**
     * Enrich the given url with a service ticket (i.e. a 'ticket' query
     * parameter) issued by the CAS server.
     * 
     * @param url
     * @return
     * @throws Exception
     */
    public String authenticateForURL(String url) throws Exception {
        String serviceTicket = this.getServiceTicketFor(url);
        return url + "&ticket=" + serviceTicket;
    }

    /**
     * Enrich the given url with a service ticket (i.e. a 'ticket' query
     * parameter) issued by the CAS server.
     * 
     * @param url
     * @param parameters
     * @return
     * @throws Exception
     */
    public String authenticateForURL(String url, MultiValueParameters parameters)
            throws Exception {
        String serviceURL = url;
        if (parameters != null) {
            serviceURL += "?" + NetworkUtils.getDataString(parameters);
        }
        return this.authenticateForURL(serviceURL);
    }

}
