package org.gcube.dataanalysis.copernicus.motu.client;

import java.util.Collection;
import java.util.List;
import java.util.Vector;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 * @author Paolo Fabriani
 *
 */
public abstract class ThreadedWorker<T> {

    /**
     * A logger for this class.
     */
    private static Logger logger = LoggerFactory.getLogger(ThreadedWorker.class);

    /**
     * Default number of concurrent work threads.
     */
    private static final Integer DEFAULT_MAX_THREADS = 2;

    /**
     * The queue of work items to be managed.
     */
    private List<T> workItems;

    /**
     * The maximum number of parallel working threads.
     */
    private Integer maxThreads;

    /**
     * A collection holding active working threads.
     */
    private Collection<Runnable> workThreads;

    /**
     * A listener to be notified when all items are completed.
     */
    private WorkCompleteListener<T> listener;

    /**
     * Basic constructor.
     */
    public ThreadedWorker() {
        this.workItems = new Vector<>();
        this.maxThreads = DEFAULT_MAX_THREADS;
        this.workThreads = new Vector<>();
    }

    /**
     * Push a new work item in the queue, ready to be processed.
     *
     * @param workItem
     *            The work item to push
     */
    public synchronized void push(final T workItem) {
        logger.info("pushing a new chunk");
        this.workItems.add(workItem);
        this.startWorking();
    }

    /**
     * Pop a request, if any, from the queue.
     *
     * @return The work item
     */
    private synchronized T pop() {
        logger.debug("queue size is " + this.workItems.size());
        if (!workItems.isEmpty()) {
            T workItem = this.workItems.get(0);
            this.workItems.remove(workItem);
            return workItem;
        }
        return null;
    }

    /**
     * Evaluates whether a new item can be processed.
     *
     * @return
     */
    private synchronized boolean canStartWorkThread() {
        return this.workThreads.size() < this.getMaxThreads();
    }

    /**
     * Print the status of the work item queue and running threads.
     */
    private synchronized void printStatus() {
        int max = this.getMaxThreads();
        int curThreads = this.workThreads.size();
        int waitChunks = this.workItems.size();
        logger.info("Threads:  " + curThreads + "/" + max);
        logger.info("Requests: " + waitChunks + " waiting");
    }

    /**
     *
     */
    public synchronized void startWorking() {
        if (this.canStartWorkThread()) {
            T workItem = this.pop();
            if (workItem == null) {
                return;
            }
            Runnable r = new Runnable() {
                public void run() {
                    try {
                        doWork(workItem);
                    } catch (Exception e) {
                        e.printStackTrace();
                    } finally {
                        workItemComplete(this, workItem);
                    }
                }
            };
            this.workThreads.add(r);
            new Thread(r).start();
            this.printStatus();
        } else {
            logger.info(
                    "maximum number of threads reached. Waiting for one to finish.");
            this.printStatus();
        }
    }

    public abstract void doWork(T workItem) throws Exception;

    /**
     * Callback method, invoked when a worker completed.
     *
     * @param t
     */
    private synchronized void workItemComplete(Runnable t, T workItem) {
        this.workThreads.remove(t);
        this.startWorking();
        if (this.listener != null) {
            listener.workComplete(workItem);
        }
    }

    /**
     * The work is complete if there's no queued work and all threads are
     * inactive.
     *
     * @return
     */
    public synchronized boolean isComplete() {
        return this.workItems.size() == 0 && this.workThreads.size() == 0;
    }

    /**
     * Return the maximum number of parallel threads allowed.
     *
     * @return the max number of threads.
     */
    public Integer getMaxThreads() {
        return this.maxThreads;
    }

    /**
     * Set the maximum number of parallel threads allowed.
     *
     * @param maxThreads the maximum number of threads.
     */
    public void setMaxThreads(final Integer maxThreads) {
        this.maxThreads = maxThreads;
    }

    /**
     * Return the listener notified upon completion.
     *
     * @return the listener.
     */
    public WorkCompleteListener<T> getListener() {
        return this.listener;
    }

    /**
     * Set the listener to be notified when all items are completed.
     *
     * @param listener the listener.
     */
    public void setListener(final WorkCompleteListener<T> listener) {
        this.listener = listener;
    }

}