/**
 * 
 */
package org.gcube.dataanalysis.copernicus.cmems.model;

import java.text.ParseException;
import java.util.Calendar;
import java.util.Collection;
import java.util.Vector;
import java.util.logging.Logger;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlElementWrapper;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.lang.builder.ToStringBuilder;
import org.gcube.dataanalysis.datasetimporter.util.TimeUtil;

/**
 * @author Paolo Fabriani
 *
 */
@XmlRootElement(name = "product")
@XmlAccessorType(XmlAccessType.PROPERTY)
public class CmemsProduct {

    private Collection<CmemsDataset> tds;
    private Collection<CmemsDataset> wms;
    private Collection<CmemsDataset> dgf;
    private Collection<CmemsDataset> ftp;

    private Collection<CmemsDocument> documents;

    private Collection<CmemsVariable> oceanVariables;
    private Collection<CmemsKeyword> oceanKeys;
    private Collection<CmemsKeyword> allKeywords;
    private Collection<CmemsRegion> regions;

    @XmlTransient
    private RawCmemsProduct wrapped;

    public CmemsProduct() {
        this.wrapped = new RawCmemsProduct();
    }

    public CmemsProduct(RawCmemsProduct raw) {
        this.wrapped = raw;
        this.setTdsDatasets(this.extractDatasets(this.wrapped.getTds(), "TDS"));
        this.setWmsDatasets(this.extractDatasets(this.wrapped.getTds(), "WMS"));
        this.setFtpDatasets(this.extractDatasets(this.wrapped.getTds(), "FTP"));
        this.setDgfDatasets(this.extractDatasets(this.wrapped.getTds(), "DGF"));

        Collection<CmemsVariable> variables = new Vector<>();
        if (this.wrapped.getOceanVariables() != null) {
            String[] vars = this.wrapped.getOceanVariables().split("\\s*,\\s*");
            for (String var : vars) {
                if (var != null && !var.trim().isEmpty()
                        && !var.trim().equals("undefined")) {
                    variables.add(new CmemsVariable(var));
                }
            }
        }
        this.setOceanVariables(variables);

        Collection<CmemsKeyword> keys = new Vector<>();
        if (this.wrapped.getOceanKeys() != null) {
            String[] kk = this.wrapped.getOceanKeys().split("\\s*,\\s*");
            for (String k : kk) {
                if (k != null && !k.trim().isEmpty()
                        && !k.trim().equals("undefined")) {
                    keys.add(new CmemsKeyword(k));
                }
            }
        }
        this.setOceanKeys(keys);

        Vector<CmemsRegion> regions = new Vector<>();
        if (this.wrapped.getGeographicalArea() != null) {
            String[] rr = this.wrapped.getGeographicalArea().split("\\s*,\\s*");
            for (String r : rr) {
                if (r != null && !r.trim().isEmpty()
                        && !r.trim().equals("undefined")) {
                    regions.add(new CmemsRegion(r));
                }
            }
        }
        this.setGeographicalArea(regions);

        Collection<CmemsKeyword> keywords = new Vector<>();
        if (this.wrapped.getAllKeywords() != null) {
            String[] kw = this.wrapped.getAllKeywords().split("\\s*,\\s*");
            for (String k : kw) {
                if (k != null && !k.trim().isEmpty()
                        && !k.trim().equals("undefined")) {
                    keywords.add(new CmemsKeyword(k));
                }
            }
        }
        this.setAllKeywords(keywords);

        Collection<CmemsDocument> docs = new Vector<>();
        if (this.wrapped.getDocumentation() != null) {
            String[] documents = this.wrapped.getDocumentation().split(",");
            for (String d : documents) {
                CmemsDocument doc = new CmemsDocument();
                String[] attrs = d.split(";");
                if (attrs.length > 0)
                    doc.setTitle(attrs[0].trim());
                if (attrs.length > 1)
                    doc.setUrl(attrs[1].trim());
                if (attrs.length > 2)
                    doc.setDate(attrs[2].trim());
                if (!doc.isEmpty())
                    docs.add(doc);
            }
        }
        this.setDocumentation(docs);

    }

    public String getAbstract() {
        return wrapped.getAbstract();
    }

    public void setAbstract(String abztract) {
        this.wrapped.setAbstract(abztract);
    }

    public String getId() {
        return this.wrapped.getId();
    }

    public void setId(String id) {
        this.wrapped.setId(id);
    }

    public String getDisplayPriority() {
        return this.wrapped.getDisplayPriority();
    }

    public void setDisplayPriority(String display_priority) {
        this.wrapped.setDisplayPriority(display_priority);
    }

    public String getExternalShortname() {
        return this.wrapped.getExternalShortname();
    }

    public void setExternalShortname(String external_shortname) {
        this.wrapped.setExternalShortname(external_shortname);
    }

    public String getVersion() {
        return this.wrapped.getVersion();
    }

    public void setVersion(String version) {
        this.wrapped.setVersion(version);
    }

    public String getShortDescription() {
        return this.wrapped.getShortDescription();
    }

    public void setShortDescription(String short_description) {
        this.wrapped.setShortDescription(short_description);
    }

    public Calendar getCreationDate() throws ParseException {
        if (this.wrapped.getCreationDate() != null) {
            return TimeUtil.toCalendar(this.wrapped.getCreationDate());
        } else {
            return null;
        }
    }

    public void setCreationDate(Calendar creation_date) {
        this.wrapped.setCreationDate(TimeUtil.toString(creation_date));
    }

    public Calendar getUpdateDate() throws ParseException {
        if (this.wrapped.getUpdateDate() != null) {
            return TimeUtil.toCalendar(this.wrapped.getUpdateDate());
        } else {
            return null;
        }
    }

    public void setUpdateDate(Calendar update_date) {
        this.wrapped.setUpdateDate(TimeUtil.toString(update_date));
    }

    public String getQuicklooks() {
        return this.wrapped.getQuicklooks();
    }

    public void setQuicklooks(String quicklooks) {
        this.wrapped.setQuicklooks(quicklooks);
    }

    public String getFeatureType() {
        return this.wrapped.getFeatureType();
    }

    public void setFeatureType(String feature_type) {
        this.wrapped.setFeatureType(feature_type);
    }

    public String getForecastLength() {
        return this.wrapped.getForecastLength();
    }

    public void setForecastLength(String forecast_length) {
        this.wrapped.setForecastLength(forecast_length);
    }

    public String getMissionType() {
        return this.wrapped.getMissionType();
    }

    public void setMissionType(String mission_type) {
        this.wrapped.setMissionType(mission_type);
    }

    public String getModelAssimilation() {
        return this.wrapped.getModelAssimilation();
    }

    public void setModelAssimilation(String model_assimilation) {
        this.wrapped.setModelAssimilation(model_assimilation);
    }

    public String getProcessingLevel() {
        return this.wrapped.getProcessingLevel();
    }

    public void setProcessingLevel(String processing_level) {
        this.wrapped.setProcessingLevel(processing_level);
    }

    public Double getWest() {
        if (this.wrapped.getWest() != null)
            return Double.parseDouble(this.wrapped.getWest());
        else
            return null;
    }

    public void setWest(Double west) {
        this.wrapped.setWest(west.toString());
    }

    public Double getEast() {
        if (this.wrapped.getEast() != null)
            return Double.parseDouble(this.wrapped.getEast());
        else
            return null;
    }

    public void setEast(Double east) {
        this.wrapped.setEast(east.toString());
    }

    public Double getSouth() {
        if (this.wrapped.getSouth() != null)
            return Double.parseDouble(this.wrapped.getSouth());
        else
            return null;
    }

    public void setSouth(Double south) {
        this.wrapped.setSouth(south.toString());
    }

    public Double getNorth() {
        if (this.wrapped.getNorth() != null)
            return Double.parseDouble(this.wrapped.getNorth());
        else
            return null;
    }

    public void setNorth(Double north) {
        this.wrapped.setNorth(north.toString());
    }

    public Double getSpatialResolutionRow() {
        String res = this.wrapped.getSpatialResolutionRow();
        if (res != null && !"undefined".equalsIgnoreCase(res.trim()))
            return Double.parseDouble(this.wrapped.getSpatialResolutionRow());
        else
            return null;
    }

    public void setSpatialResolutionRow(Double spatial_resolution_row) {
        this.wrapped.setSpatialResolutionRow(spatial_resolution_row.toString());
    }

    public String getSpatialResolutionRowUnit() {
        return this.wrapped.getSpatialResolutionRowUnit();
    }

    public void setSpatialResolutionRowUnit(
            String spatial_resolution_row_unit) {
        this.wrapped.setSpatialResolutionRowUnit(spatial_resolution_row_unit);
    }

    public Double getSpatialResolutionColumn() {
        String res = this.wrapped.getSpatialResolutionColumn();
        if (res != null && !"undefined".equalsIgnoreCase(res.trim()))
            return Double
                    .parseDouble(this.wrapped.getSpatialResolutionColumn());
        else
            return null;
    }

    public void setSpatialResolutionColumn(Double spatial_resolution_column) {
        this.wrapped
                .setSpatialResolutionRow(spatial_resolution_column.toString());
    }

    public String getSpatialResolutionColumnUnit() {
        return this.wrapped.getSpatialResolutionColumnUnit();
    }

    public void setSpatialResolutionColumnUnit(
            String spatial_resolution_column_unit) {
        this.wrapped
                .setSpatialResolutionColumnUnit(spatial_resolution_column_unit);
    }

    public String getCoordRefSys() {
        return this.wrapped.getCoordRefSys();
    }

    public void setCoordRefSys(String coord_ref_sys) {
        this.wrapped.setCoordRefSys(coord_ref_sys);
    }

    public Double getVerticalMin() {
        String res = this.wrapped.getVerticalMin();
        if (res != null && !"undefined".equalsIgnoreCase(res.trim()))
            return Double.parseDouble(this.wrapped.getVerticalMin());
        else
            return null;
    }

    public void setVerticalMin(Double vertical_min) {
        this.wrapped.setVerticalMin(vertical_min.toString());
    }

    public Double getVerticalMax() {
        String res = this.wrapped.getVerticalMax();
        if (res != null && !"undefined".equalsIgnoreCase(res.trim()))
            return Double.parseDouble(this.wrapped.getVerticalMax());
        else
            return null;
    }

    public void setVerticalMax(String vertical_max) {
        this.wrapped.setVerticalMin(vertical_max.toString());
    }

    public String getVerticalCrs() {
        return this.wrapped.getVerticalCrs();
    }

    public void setVerticalCrs(String vertical_crs) {
        this.wrapped.setVerticalCrs(vertical_crs);
    }

    public String getVerticalLevels() {
        return this.wrapped.getVerticalLevels();
    }

    public void setVerticalLevels(String vertical_levels) {
        this.wrapped.setVerticalLevels(vertical_levels);
    }

    public String getTemporalResolution() {
        return this.wrapped.getTemporalResolution();
    }

    public void setTemporalResolution(String temporal_resolution) {
        this.wrapped.setTemporalResolution(temporal_resolution);
    }

    public String getVerticalCoverage() {
        return this.wrapped.getVerticalCoverage();
    }

    public void setVerticalCoverage(String vertical_coverage) {
        this.wrapped.setVerticalCoverage(vertical_coverage);
    }

    public Calendar getTemporalBegin() throws ParseException {
        if (this.wrapped.getTemporalBegin() != null) {
            return TimeUtil.toCalendar(this.wrapped.getTemporalBegin());
        } else {
            return null;
        }
    }

    public void setTemporalBegin(Calendar temporal_begin) {
        this.wrapped.setTemporalBegin(TimeUtil.toString(temporal_begin));
    }

    public Calendar getTemporalEnd() throws ParseException {
        if (this.wrapped.getTemporalEnd() != null) {
            return TimeUtil.toCalendar(this.wrapped.getTemporalEnd());
        } else {
            return null;
        }
    }

    public void setTemporalEnd(Calendar temporal_end) {
        this.wrapped.setTemporalBegin(TimeUtil.toString(temporal_end));
    }

    public String getTemporalExtent() {
        return this.wrapped.getTemporalExtent();
    }

    public void setTemporalExtent(String temporal_extent) {
        this.wrapped.setTemporalExtent(temporal_extent);
    }

    public String getUpdateFrequency() {
        return this.wrapped.getUpdateFrequency();
    }

    public void setUpdateFrequency(String update_frequency) {
        this.wrapped.setUpdateFrequency(update_frequency);
    }

    public String getUpdateSchedule() {
        return this.wrapped.getUpdateSchedule();
    }

    public void setUpdateSchedule(String update_schedule) {
        this.wrapped.setUpdateSchedule(update_schedule);
    }

    public String getProductionUnit() {
        return this.wrapped.getProductionUnit();
    }

    public void setProductionUnit(String production_unit) {
        this.wrapped.setProductionUnit(production_unit);
    }

    public String getCredit() {
        return this.wrapped.getCredit();
    }

    public void setCredit(String credit) {
        this.wrapped.setCredit(credit);
    }

    @XmlElementWrapper(name = "documentation")
    @XmlElement(name = "document")
    public Collection<CmemsDocument> getDocumentation() {
        return this.documents;

    }

    public void setDocumentation(Collection<CmemsDocument> documents) {
        this.documents = documents;
    }

    public String getMail() {
        return this.wrapped.getMail();
    }

    public void setMail(String mail) {
        this.wrapped.setMail(mail);
    }

    public String getServiceDesk() {
        return this.wrapped.getServiceDesk();
    }

    public void setServiceDesk(String service_desk) {
        this.wrapped.setServiceDesk(service_desk);
    }

    public String getUpstreamProduction() {
        return this.wrapped.getUpstreamProduction();
    }

    public void setUpstreamProduction(String upstream_production) {
        this.wrapped.setUpstreamProduction(upstream_production);
    }

    public String getUpstreamValidation() {
        return this.wrapped.getUpstreamValidation();
    }

    public void setUpstreamValidation(String upstream_validation) {
        this.wrapped.setUpstreamValidation(upstream_validation);
    }

    public String getDownstreamValidation() {
        return this.wrapped.getDownstreamValidation();
    }

    public void setDownstreamValidation(String downstream_validation) {
        this.wrapped.setDownstreamValidation(downstream_validation);
    }

    public String getDownstreamProduction() {
        return this.wrapped.getDownstreamProduction();
    }

    public void setDownstreamProduction(String downstream_production) {
        this.wrapped.setDownstreamProduction(downstream_production);
    }

    public String getMYO_OK() {
        return this.wrapped.getMYO_OK();
    }

    public void setMYO_OK(String mYO_OK) {
        this.wrapped.setMYO_OK(mYO_OK);
    }

    public String getVersion_() {
        return this.wrapped.getVersion();
    }

    public void setVersion_(String _version) {
        this.wrapped.setVersion(_version);
    }

    @XmlTransient
    public String getMotuServer() {
        for (CmemsDataset ds : this.getTdsDatasets()) {
            String url = ds.getUrl();
            if (url != null && !url.equalsIgnoreCase("undefined"))
                return url.substring(0, url.indexOf("?"));
        }
        return null;
    }

    @XmlElementWrapper(name = "tds")
    @XmlElement(name = "dataset")
    public Collection<CmemsDataset> getTdsDatasets() {
        return this.tds;
    }

    public void setTdsDatasets(Collection<CmemsDataset> datasets) {
        this.tds = datasets;
        this.wrapped.setTds(this.toString(datasets));
    }

    @XmlElementWrapper(name = "dgf")
    @XmlElement(name = "dataset")
    public Collection<CmemsDataset> getDgfDatasets() {
        return this.dgf;
    }

    public void setDgfDatasets(Collection<CmemsDataset> datasets) {
        this.dgf = datasets;
    }

    @XmlElementWrapper(name = "ftp")
    @XmlElement(name = "dataset")
    public Collection<CmemsDataset> getFtpDatasets() {
        return this.ftp;
    }

    public void setFtpDatasets(Collection<CmemsDataset> datasets) {
        this.ftp = datasets;
    }

    @XmlElementWrapper(name = "wms")
    @XmlElement(name = "dataset")
    public Collection<CmemsDataset> getWmsDatasets() {
        return this.wms;
    }

    public void setWmsDatasets(Collection<CmemsDataset> datasets) {
        this.wms = datasets;
    }

    public CmemsDataset getTdsDataset(String datasetName) {
        for (CmemsDataset d : this.getTdsDatasets()) {
            if (datasetName.equals(d.getName())) {
                return d;
            }
        }
        return null;
    }

    public CmemsDataset getDgfDataset(String datasetName) {
        for (CmemsDataset d : this.getDgfDatasets()) {
            if (datasetName.equals(d.getName())) {
                return d;
            }
        }
        return null;
    }

    public CmemsDataset getFtpDataset(String datasetName) {
        for (CmemsDataset d : this.getFtpDatasets()) {
            if (datasetName.equals(d.getName())) {
                return d;
            }
        }
        return null;
    }

    private Collection<CmemsDataset> extractDatasets(String datasetString,
            String protocol) {
        Collection<CmemsDataset> out = new Vector<>();
        if (datasetString != null) {
            String[] parts = datasetString.split(";");
            if (parts.length > 1) {
                String[] urls = parts[0].split(",");
                String[] names = parts[1].split(",");
                if (names.length == urls.length && names.length > 0) {
                    for (int i = 0; i < urls.length; i++) {
                        CmemsDataset ds = new CmemsDataset();
                        ds.setProtocol(protocol);
                        ds.setName(names[i].trim());
                        ds.setUrl(urls[i].trim());
                        out.add(ds);
                    }
                }
            }
        }
        return out;
    }

    private String toString(Collection<CmemsDataset> datasets) {
        if (datasets != null && datasets.size() > 0) {
            String urls = "";
            String names = "";
            for (CmemsDataset ds : datasets) {
                if (!urls.isEmpty()) {
                    urls += ",";
                }
                urls += ds.getUrl();
                if (!names.isEmpty()) {
                    names += ",";
                }
                names += ds.getName();
            }
            return urls + ";" + names;
        } else {
            return "";
        }
    }

    public String getTemporalScale() {
        return this.wrapped.getTemporalScale();
    }

    public void setTemporalScale(String temporal_scale) {
        this.wrapped.setTemporalScale(temporal_scale);
    }

    @XmlElementWrapper(name = "oceanVariables")
    @XmlElement(name = "variable")
    public Collection<CmemsVariable> getOceanVariables() {
        return this.oceanVariables;
    }

    public void setOceanVariables(Collection<CmemsVariable> ocean_variables) {
        this.oceanVariables = ocean_variables;
    }

    @XmlElementWrapper(name = "oceanKeys")
    @XmlElement(name = "key")
    public Collection<CmemsKeyword> getOceanKeys() {
        return this.oceanKeys;
    }

    public void setOceanKeys(Collection<CmemsKeyword> ocean_keys) {
        this.oceanKeys = ocean_keys;
    }

    @XmlElementWrapper(name = "allKeywords")
    @XmlElement(name = "keyword")
    public Collection<CmemsKeyword> getAllKeywords() {
        return this.allKeywords;
    }

    public void setAllKeywords(Collection<CmemsKeyword> all_keywords) {
        this.allKeywords = all_keywords;
    }

    @XmlElementWrapper(name = "geographicalAreas")
    @XmlElement(name = "area")
    public Collection<CmemsRegion> getGeographicalArea() {
        return this.regions;
    }

    public void setGeographicalArea(Collection<CmemsRegion> geographical_area) {
        this.regions = geographical_area;
    }

    public String toString() {
        return ToStringBuilder.reflectionToString(this);
    }

}
