/**
 * 
 */
package org.gcube.dataanalysis.copernicus.cmems.importer.seplugin.thredds;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URL;
import java.text.ParseException;
import java.util.Calendar;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.NoSuchElementException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.commons.io.IOUtils;
import org.gcube.common.authorization.library.provider.SecurityTokenProvider;
import org.gcube.dataanalysis.copernicus.cmems.importer.api.ImportOptions;
import org.gcube.dataanalysis.datasetimporter.exception.ServiceUnreachableException;
import org.gcube.dataanalysis.datasetimporter.util.TextUtil;
import org.gcube.dataanalysis.datasetimporter.util.TimeUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 *
 */
public class ThreddsClient {

    /**
     * A logger for this class.
     */
    private static Logger logger = LoggerFactory.getLogger(ThreddsClient.class);

    /**
     * The pattern for chunk file names.
     */
    // private final static Pattern p = Pattern.compile("([^-]+)-([0-9]+).*");

    /**
     * The endpoint of the thredds server.
     */
    private URL endpoint;

    /**
     * The persistenceId in the thredds server.
     */
    private String persistenceId;

    /**
     * The name of the catalogue (currently a folder) within the thredds server.
     */
    private String catalogue;

    /**
     * Basic constructor.
     * @param endpoint
     * @param catalogue
     */
    public ThreddsClient(URL endpoint, String persistenceId, String catalogue) {
        this.endpoint = endpoint;
        this.persistenceId = persistenceId;
        this.catalogue = catalogue;
    }

    /**
     * Return the endpoint of the thredds server
     * @return
     */
    private URL getEndpoint() {
        return endpoint;
    }

    /**
     * Return the persistence id associated with this client
     * @return
     */
    private String getPersistenceId() {
        return persistenceId;
    }
    
    /**
     * Return the catalogue name, within the thredds server
     * @return
     */
    private String getCatalogue() {
        return catalogue;
    }

    /**
     * Given an import request, return all available chunks.
     * @param request
     * @return
     * @throws Exception
     */
    public ThreddsDataset getDataset(ImportOptions request) throws Exception {
        return this.getDataset(request.getHash());
    }
    
    public boolean containsDataset(ImportOptions request) {
        return this.containsDataset(request.getHash());
    }
    
    public boolean containsDataset(String hash) {
        try {
            InputStream stream = this.getNcmlInputStream(hash);
            stream.close();
            return true;
        } catch(Exception e) {
            return false;
        }
    }

    /**
     * Given an dataset hash, return all available chunks.
     * @param request
     * @return
     * @throws Exception
     */
    public ThreddsDataset getDataset(String hash) throws Exception {

        try {
            // get chunks from ncml
            Map<String, ThreddsDatasetChunk> cs1 = this.getChunksInNcml(hash);
            logger.info("found " + cs1.size() + " chunks in ncml");

            // get chunks in catalogue
            Map<String, ThreddsDatasetChunk> cs2 = this.getChunksInCatalogue(hash);
            logger.info("found " + cs2.size() + " chunks in catalogue");

            // update ncml chunks with update time taken from catalogue.xml
            for (Entry<String, ThreddsDatasetChunk> e : cs1.entrySet()) {
                ThreddsDatasetChunk c2 = cs2.get(e.getKey());
                e.getValue().setChunkUpdate(c2.getChunkUpdate());
            }

            // parse the comments in ncml file to get request info
            ImportOptions opts = this.getRequestByHash(hash);

            // build output
            ThreddsDataset out = new ThreddsDataset();
            out.setOptions(opts);
            for(ThreddsDatasetChunk chunk:cs1.values()) {
                out.addChunk(chunk);
            }

            return out;
            
        } catch (Exception e) {
            e.printStackTrace();
            throw new Exception("unable to find/parse a dataset at " + this.getNcmlUrl(hash));
        }
        
    }

    /**
     * Get the NCML document for the given hash
     * @param hash
     * @return
     * @throws Exception
     */
    
    private Document getNcmlDocument(String hash) throws Exception {
        return this.getXmlDocument(this.getNcmlInputStream(hash));
    }

    /**
     * Return the thredds catalog for the given hash
     * @param hash
     * @return
     * @throws Exception
     */
    private Document getCatalog(String hash) throws Exception {
        return this.getXmlDocument(this.getCatalogInputStream(hash));
    }
    
    /**
     * @param hash
     * @return
     * @throws IOException
     */
    private InputStream getNcmlInputStream(String hash) throws IOException {
        String url = getNcmlUrl(hash);
        return new URL(url).openStream();        
    }

    /**
     * @param hash
     * @return
     * @throws IOException
     */
    private InputStream getCatalogInputStream(String hash) throws IOException {
        String url = getCatalogUrl(hash);
        logger.debug(url);
        return new URL(url).openStream();        
    }
    /**
     * Compute the URL of the ncml with the given hash.
     * @param hash
     * @return
     */
    private String getNcmlUrl(String hash) {
        return this.getEndpoint() + "/" + this.getPersistenceId() + "/fileServer/" + this.getCatalogue() + "/" + hash + ".ncml";
    }

    /**
     * Compute the URL of the catalog.xml for the given hash.
     * @param hash
     * @return
     */
    private String getCatalogUrl(String hash) {
        return this.getEndpoint() + "/" + this.getPersistenceId() + "/" + this.getCatalogue() + "/catalog.xml";
    }
    
    /**
     * Read and parse the xml available in the given stream.
     * @param inputStream
     * @return
     * @throws Exception
     */
    private Document getXmlDocument(InputStream inputStream) throws Exception {
        DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
        dbf.setIgnoringComments(false);
        DocumentBuilder db = dbf.newDocumentBuilder();
        Document doc = db.parse(inputStream);
        doc.getDocumentElement().normalize();
        return doc;
    }

    /**
     * Parse an ncml (actually a comment in it) associated with the given hash and extract import parameters.
     * @param hash
     * @return
     * @throws Exception
     */
    private ImportOptions getRequestByHash(String hash) throws Exception {
        InputStream stream = this.getNcmlInputStream(hash);
        String ncml = IOUtils.toString(stream, "UTF-8"); 
        return this.getRequestInNcml(ncml);
    }
    
    private Map<String, ThreddsDatasetChunk> getChunksInNcml(String hash) throws Exception {
        return this.getChunksInNcml(this.getNcmlDocument(hash));
    }
    
    /**
     * Parse the ncml for chunks, get them from the catalogue and return.
     * @param ncml
     * @return
     */
    private Map<String, ThreddsDatasetChunk> getChunksInNcml(Document ncml) throws Exception {
        Map<String, ThreddsDatasetChunk> out = new HashMap<>();
        if (ncml != null) {
            NodeList nList = ncml.getElementsByTagName("netcdf");
            for (int i = 0; i < nList.getLength(); i++) {
                Node nNode = nList.item(i);
                Element eElement = (Element) nNode;
                if (eElement.hasAttribute("location")) {
                    ThreddsDatasetChunk chunk = new ThreddsDatasetChunk(
                            eElement.getAttribute("location"));
                    out.put(chunk.getFileName(), chunk);
                }
            }
        }
        return out;
    }

    /**
     * Retrieve all chunks in a catalogue with the given hash as part of the
     * name.
     * 
     * The name of a chunk in the catalogue has the format
     * <hash>-<year>[<month>[<day>]].nc
     * 
     * The update time is taken from the xml.
     * 
     * The chunk timespan (day, month, year) is computed by looking at the
     * timestamp in the file name.
     * 
     * @param hash
     * @return
     * @throws Exception
     */
    private Map<String, ThreddsDatasetChunk> getChunksInCatalogue(String hash) throws Exception {

        Document doc = this.getCatalog(hash);
        
        Map<String, ThreddsDatasetChunk> out = new HashMap<>();
        if (doc != null) {

            NodeList nList = doc.getElementsByTagName("dataset");
            for (int i = 0; i < nList.getLength(); i++) {
                Node nNode = nList.item(i);
                Element eElement = (Element) nNode;
                String name = eElement.getAttribute("name");
                ThreddsDatasetChunk chunk;
                try {
                    chunk = new ThreddsDatasetChunk(name);
                } catch (ParseException e) {
                    continue;
                }
                // chunk.setName
                String h = chunk.getName();
                // String h =
                if (h.equals(hash)) {
                    try {
                        chunk.setName(h);
//                        chunk.setChunkStart(extractChunkStart(name));
                        NodeList children = eElement.getElementsByTagName("date");
                        for (int j = 0; j < children.getLength();) {
                            Element dateNode = (Element) children.item(j);
                            String dateString = dateNode.getTextContent();
                            Calendar date = TimeUtil.toCalendar(dateString);
                            chunk.setChunkUpdate(date);
//                            chunk.setTimeSpan(getTimeSpan());
                            break;
                        }
                        out.put(name, chunk);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            }
        }
        return out;
    }


    /**
     * Parse an ncml (actually a comment in it) looking for import parameters.
     * @param ncml
     * @return The ImportOptions class containing import parameters.
     * @throws ParseException
     */
    private ImportOptions getRequestInNcml(String ncml) throws ParseException {
        // rebuild the request from info in the comment
        int start = ncml.indexOf("<!--") + 4;
        int end = ncml.indexOf("-->", start);
        String comment = ncml.substring(start, end).trim();
        Map<String, String> out = new HashMap<>();
        for (String entry : comment.split(";")) {
            String[] parts = entry.split(":");
            String key = parts[0].trim();
            String value = parts[1].trim();
            out.put(key, value);
        }
        ImportOptions options = new ImportOptions(out);
        return options;
    }

    /**
     * Publishes a file in the current catalog, using Data Transfer
     * facilities.
     * @param localFile
     */
    public void upload(File localFile) throws Exception {
        this.uploadUsingCurl(localFile);
    }

    /**
     * Upload a local file to the thredds service, under the current persistencyId and catalog.
     * @param localFile
     * @throws Exception
     */
    private void uploadUsingCurl(File localFile) throws IOException {

        // TODO: return the log of the command
        
        String command = "curl";
        command += " -F uploadedFile=@" + localFile.getAbsolutePath();
        command += " --header gcube-token:"
                + SecurityTokenProvider.instance.get();
        command += " " + this.getEndpoint()
                + "/data-transfer-service/gcube/service/REST/"
                + this.getPersistenceId() + "/" + this.getCatalogue();
        command += "?on-existing-file=REWRITE";
        command += "&on-existing-dir=APPEND";
        command += "&create-dirs=true";

        logger.debug(TextUtil.removePasswords(command));

        try {
            // using the Runtime exec method:
            Process p = Runtime.getRuntime().exec(command);

            BufferedReader stdInput = new BufferedReader(
                    new InputStreamReader(p.getInputStream()));

            BufferedReader stdError = new BufferedReader(
                    new InputStreamReader(p.getErrorStream()));

            // read the output from the command
            logger.info("Here is the standard output of the command:");
            String s;
            while ((s = stdInput.readLine()) != null) {
                logger.info(s);
            }

            // read any errors from the attempted command
            logger.info(
                    "Here is the standard error of the command (if any):");
            while ((s = stdError.readLine()) != null) {
                logger.info(s);
            }
        } catch (IOException e) {
            e.printStackTrace();
            throw e;
        }
    }

    public void checkThreddsIsReachable() throws ServiceUnreachableException {
        String endpoint = this.getEndpoint() + "/" + this.getPersistenceId();
        try(InputStream stream = new URL(endpoint).openStream()) {
        } catch(IOException e) {
            throw new ServiceUnreachableException(endpoint);
        }
    }

    private void checkThreddsDatasetExists(String hash) throws NoSuchElementException {
        String endpoint = this.getNcmlUrl(hash);
        try(InputStream stream = new URL(endpoint).openStream()) {
        } catch(IOException e) {
            throw new NoSuchElementException(endpoint);
        }
    }

}
