/**
 * 
 */
package org.gcube.dataanalysis.copernicus.cmems.importer.seplugin.thredds;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Collection;
import java.util.Map.Entry;
import java.util.Vector;

import org.gcube.dataanalysis.copernicus.cmems.importer.api.ChunkTimespan;
import org.gcube.dataanalysis.copernicus.cmems.importer.api.ImportOptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A chunked dataset on thredds. 
 * 
 * @author Paolo Fabriani
 *
 */
public class ThreddsDataset {

    private static Logger logger = LoggerFactory.getLogger(ThreddsDataset.class);
    
    /**
     * The import request.
     */
    private ImportOptions options;

    /**
     * The chunks building to the imported dataset.
     */
    private Collection<ThreddsDatasetChunk> chunks;

    public ThreddsDataset() {
        this.chunks = new Vector<>();
    }
    
    public ThreddsDataset(ImportOptions options) {
        this();
        this.setOptions(options);
        
        // TODO: consider backTime here
        
        // build chunks
        
//        int chunkSpan = Calendar.DAY_OF_YEAR;
//        if(ChunkTimespan.MONTH.equals(options.getChunkSpan()))
//            chunkSpan = Calendar.MONTH;
//        else if(ChunkTimespan.YEAR.equals(options.getChunkSpan()))
//            chunkSpan = Calendar.YEAR;
        
        ChunkTimespan chunkSpan = options.getChunkSpan();
        
        // where to start from
        Calendar c = Calendar.getInstance();
        c.setTimeInMillis(options.gettLo().getTimeInMillis());
        
        // where to end
        Calendar end = options.gettHi();
        if(end==null) {
            logger.debug("No end time in options. Assuming 'now'");
            end = Calendar.getInstance();
            Integer backTime = options.getBackTime();
            if(backTime!=null && backTime>0) {
                logger.debug("End time set back by " + backTime + " days");
                end.add(Calendar.DAY_OF_YEAR, -backTime);
            }
        }
        
        while(!c.after(end)) {
            // build the chunk
            ThreddsDatasetChunk chunk = new ThreddsDatasetChunk();
            Calendar startc = Calendar.getInstance();
            startc.setTimeInMillis(c.getTimeInMillis());
            chunk.setChunkStart(startc);
            chunk.setName(options.getHash());
            chunk.setTimeSpan(chunkSpan);

            // print it out
            logger.debug(chunk.getFileName());
            
            // go to the next chunk
            switch (chunkSpan) {
            case MONTH:
                c.add(Calendar.MONTH, 1);
                c.set(Calendar.DAY_OF_MONTH, 1);
                break;
            case YEAR:
                c.add(Calendar.YEAR, 1);
                c.set(Calendar.MONTH, 0);
                c.set(Calendar.DAY_OF_MONTH, 1);
                break;
            default:
                c.add(Calendar.DAY_OF_MONTH, 1);
            }
            this.addChunk(chunk);
        }
    }
    
    public Collection<ThreddsDatasetChunk> getChunks() {
        if(this.chunks==null)
            return new Vector<>();
        return chunks;
    }

    public void addChunk(ThreddsDatasetChunk chunk) {
        if(this.chunks==null) {
            this.chunks = new Vector<>();
        }
        this.chunks.add(chunk);
    }
    
    public ImportOptions getRequest() {
        return options;
    }

    public void setOptions(ImportOptions request) {
        this.options = request;
    }

    public int size() {
        return this.chunks.size();
    }

    public String toString() {
        String out = "";
        for (ThreddsDatasetChunk c : this.getChunks()) {
            out += c.toString() + "\n";
        }
        return out;
    }
    
    public String getHash() {
        return this.options.getHash();
    }
    
    public String generateNCML() {
        StringBuffer out = new StringBuffer();
        out.append("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n");
        out.append("<netcdf xmlns=\"http://www.unidata.ucar.edu/namespaces/netcdf/ncml-2.2\">\n");
        out.append("  <!--\n");
        for(Entry<String, String> entry:this.getRequest().toMap().entrySet()) {
            String k = entry.getKey();
            String v = entry.getValue();
            if(v!=null) {
                out.append(String.format("    %s: %s;\n", k, v));
            }
        }
        out.append("  -->\n");
        out.append("  <attribute name=\"title\" value=\"some title\"/>\n");
        out.append("  <aggregation type=\"joinExisting\" dimName=\"time\">\n");
        for(ThreddsDatasetChunk c:this.getChunks()) {
            this.options.getChunkSpan();
            out.append("    <netcdf location=\""+this.formatChunkName(c)+"\"/>\n");
        }
        out.append("  </aggregation>\n");
        out.append("</netcdf>\n");
        return out.toString();
    }
    
    private String formatDatePart(Calendar c) {
        DateFormat df;
        String datePart;
        switch (this.options.getChunkSpan()) {
        case YEAR:
            df = new SimpleDateFormat("yyyy");
            datePart = df.format(c.getTime());
            break;
        case MONTH:
            df = new SimpleDateFormat("yyyyMM");
            datePart = df.format(c.getTime());
            break;
        default:
            df = new SimpleDateFormat("yyyyMMdd");
            datePart = df.format(c.getTime());
        }
        return datePart;
    }
    
    private String formatChunkName(ThreddsDatasetChunk chunk) {
        return String.format("%s-%s.nc", this.getHash(), this.formatDatePart(chunk.getChunkStart()));
    }

    public boolean contains(ThreddsDatasetChunk chunk) {
        for(ThreddsDatasetChunk c:this.getChunks()) {
            if(c.getFileName().equals(chunk.getFileName())) {
                return true;
            }
        }
        return false;
    }

}
