/**
 * 
 */
package org.gcube.dataanalysis.copernicus.cmems.importer.service.service;

import java.util.Calendar;
import java.util.Collection;

import org.gcube.dataanalysis.copernicus.cmems.client.CmemsClient;
import org.gcube.dataanalysis.copernicus.cmems.client.SearchOptions;
import org.gcube.dataanalysis.copernicus.cmems.importer.api.ImportOptions;
import org.gcube.dataanalysis.copernicus.cmems.importer.service.util.ParamUtils;
import org.gcube.dataanalysis.copernicus.cmems.model.CmemsDataset;
import org.gcube.dataanalysis.copernicus.cmems.model.CmemsKeyword;
import org.gcube.dataanalysis.copernicus.cmems.model.CmemsProduct;
import org.gcube.dataanalysis.copernicus.cmems.model.CmemsRegion;
import org.gcube.dataanalysis.copernicus.cmems.model.CmemsVariable;
import org.gcube.dataanalysis.copernicus.motu.client.DownloadRequest;
import org.gcube.dataanalysis.copernicus.motu.client.MotuClient;
import org.gcube.dataanalysis.copernicus.motu.model.MotuServer;
import org.gcube.dataanalysis.copernicus.motu.model.ProductMetadataInfo;
import org.gcube.dataanalysis.datasetimporter.util.ConfigurationUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Paolo Fabriani
 *
 */
public class DatasetBrowser {

    private static Logger logger = LoggerFactory.getLogger(DatasetBrowser.class); 

    private String getMotuFor(String cmemsProduct) throws Exception {
        CmemsProduct p = this.getProduct(cmemsProduct);
        String motu = p.getMotuServer();
        logger.info(motu);
        return motu;
    }
    
    public Long datasetSize(ImportOptions options) throws Exception {
        // build the request
        DownloadRequest req = ParamUtils.importOptionsToDownloadRequest(options);
        
        // retrieve motu server
        String motu = options.getMotu();
        if(motu==null)
            motu = this.getMotuFor(options.getProduct());
        
        // build the client
        MotuClient mc = this.getMotuClient(motu);
        
        // get the size
        return mc.getSize(req).getSizeInBytes();
    }
    
    public Long chunkSize(ImportOptions options) throws Exception {

        // get the dataset size
        Long datasetSize = new DatasetBrowser().datasetSize(options);
        logger.info(datasetSize.toString());
        // get the product
        CmemsProduct product = this.getProduct(options.getProduct(), options.getDataset());
        logger.info(product.toString());
        
        // dataset length (in days)
        Calendar end = product.getTemporalEnd();
        if(end==null)
            end = Calendar.getInstance();
        logger.info(end.toString());
        Calendar start = product.getTemporalBegin();
        if(start==null)
            start = this.getCmemsClient().getMinTime();
        logger.info(start.toString());
        Long datasetTimespan = (end.getTimeInMillis()-start.getTimeInMillis())/1000L/60/60/24;
        logger.info(datasetTimespan.toString());
        
        // chunk length (in days);
        Long chunkTimespan = 1L;
        if(options.getChunkSpan()!=null) {
            switch (options.getChunkSpan()) {
            case YEAR:
                chunkTimespan = 365L;
                break;
            case MONTH:
                chunkTimespan = 365L / 12;
                break;
            case DAY:
                chunkTimespan = 1L;
                break;
            }
        }
        logger.info(chunkTimespan.toString());

        // how many chunks for the whole dataset?
        Double chunkRatio = (double)datasetTimespan/(double)chunkTimespan;
        logger.info(chunkRatio.toString());

        // compute approximate chunk size
        Long approxChunkSize = Math.round((double)datasetSize/chunkRatio);        
        logger.info(approxChunkSize.toString());
        
        return approxChunkSize;
    }
    
    private MotuClient getMotuClient(String motuEndpoint) {
        // build a configuration manager
        ConfigurationUtil cu = new ConfigurationUtil();
        String userHome = System.getenv("HOME");
        cu.setLocalConfigurationFile(userHome+"/.cmems");
        // build a client
        MotuClient mc = new MotuClient(motuEndpoint);
        mc.setUsername(cu.getProperty("CMEMS_USERNAME"));
        mc.setPassword(cu.getProperty("CMEMS_PASSWORD"));
        return mc;
    }
    
    private CmemsClient getCmemsClient() {
        CmemsClient cc = new CmemsClient();
        return cc;
    }
    
    public Collection<CmemsProduct> searchProducts(SearchOptions options) throws Exception {
        return this.getCmemsClient().searchProducts(options);
    }
    
    public CmemsProduct getProduct(String product, String dataset) throws Exception {
        Collection<CmemsProduct> products = this.searchProducts(new SearchOptions());
        for(CmemsProduct p:products) {
            if(product.equals(p.getExternalShortname()) && p.getTdsDataset(dataset)!=null) {
                return p;
            }
        }
        return null;
    }
    
    public CmemsProduct getProduct(String product) throws Exception {
        Collection<CmemsProduct> products = this.searchProducts(new SearchOptions());
        for(CmemsProduct p:products) {
            if(product.equals(p.getExternalShortname())) {
                return p;
            }
        }
        return null;
    }
    
    public ProductMetadataInfo getMotuDataset(String product, String dataset) throws Exception {
        
        // retrieve the cmems dataset
        CmemsProduct p = this.getProduct(product, dataset);
        logger.debug("got product " + p.getExternalShortname());
        
        // get the corresponding motu server
        String motuServer = p.getMotuServer();
        logger.debug("motu server is " + motuServer);

        // get the actual product name on motu
        CmemsDataset ds = p.getTdsDataset(dataset);
        String productNameOnMotu = ds.getProductOnMotu();
        String datasetNameOnMotu = ds.getDatasetOnMotu();
        logger.debug("product on motu: " + productNameOnMotu);
        logger.debug("dataset on motu: " + datasetNameOnMotu);
        
        // if not empty, rewrite cmems product/dataset with motu ones
        if(productNameOnMotu!=null && !productNameOnMotu.trim().isEmpty())
            product = productNameOnMotu;
        if(datasetNameOnMotu!=null && !datasetNameOnMotu.trim().isEmpty())
            dataset = datasetNameOnMotu;
        
        // build a request for the motu
        DownloadRequest request = new DownloadRequest();
        request.setService(product);
        request.setProduct(dataset);
        logger.debug("built a request for " + request.getService()+"/"+request.getProduct());
        
        // retrieve dataset info from motu
        MotuClient mc = this.getMotuClient(motuServer);
        logger.debug("built client: " + mc);
        ProductMetadataInfo pmi = null;
        try {
            pmi = mc.describeProduct(request);
            logger.debug("got metadata (1)");
        } catch(Exception e) {
            logger.info(e.toString());
            request.setService(product);
            logger.info("trying another service");
            pmi = mc.describeProduct(request);
            logger.debug("got metadata (2)");
        }
        return pmi;
    }
    
    public Collection<MotuServer> getMotuServers() throws Exception {
        return new CmemsClient().getMotuServers();
    }

    public Collection<CmemsRegion> getRegionalDomains() throws Exception {
        return new CmemsClient().getRegionalDomains();
    }

    public Collection<CmemsVariable> getOceanKeys() throws Exception {
        return new CmemsClient().getOceanKeys();
    }

    public Collection<CmemsKeyword> getKeywords() throws Exception {
        return new CmemsClient().getAllKewords();
    }

    public Calendar getMinTime() throws Exception {
        return new CmemsClient().getMinTime();
    }

    public Calendar getMaxTime() throws Exception {
        return new CmemsClient().getMaxTime();
    }

}
