/**
 * 
 */
package org.gcube.dataanalysis.copernicus.cmems.importer.service.service.validation;

import java.util.Calendar;
import java.util.Collection;
import java.util.Vector;

import org.gcube.dataanalysis.copernicus.cmems.importer.api.ChunkTimespan;
import org.gcube.dataanalysis.copernicus.cmems.importer.api.ImportOptions;
import org.gcube.dataanalysis.copernicus.cmems.importer.api.ValidationError;
import org.gcube.dataanalysis.copernicus.cmems.importer.service.exception.InvalidParameterException;
import org.gcube.dataanalysis.copernicus.cmems.model.CmemsProduct;
import org.gcube.dataanalysis.copernicus.motu.model.ProductMetadataInfo;
import org.gcube.dataanalysis.copernicus.motu.model.Variable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RequestValidator {

    private static Logger logger = LoggerFactory.getLogger(RequestValidator.class); 

    private static final String MOTU = "motu (m)";
    private static final String PRODUCT = "product (p)";
    private static final String DATASET = "dataset (d)";
    private static final String XLO = "xlo";
    private static final String XHI = "xhi";
    private static final String YLO = "ylo";
    private static final String YHI = "yhi";
    private static final String ZLO = "zlo";
    private static final String ZHI = "zhi";
    private static final String TLO = "tlo";
    private static final String THI = "thi";
    private static final String VARIABLE = "variable (v)";

    private static final String BACKTIME = "backTime (b)";
    private static final String CHUNKSPAN = "chunkSpan (s)";
    private static final String SCHEDULE = "importSchedule (f)";

    /**
     * Validate for parameters needed to retrieve product metadata.
     * @param options
     */
    public static Collection<ValidationError> validateForProductMetadata(ImportOptions options) {
        Collection<ValidationError> out =new Vector<>();
        out.addAll(checkSet(PRODUCT, options.getProduct()));
        out.addAll(checkSet(DATASET, options.getDataset()));
        return out;
    }

    public static Collection<ValidationError> checkScheduleOptions(ImportOptions options) throws Exception {
        Collection<ValidationError> errors = new Vector<>();
        
        Integer backTime = options.getBackTime();
        if(backTime!=null && backTime<0)
            errors.add(new ValidationError(BACKTIME, BACKTIME + " can't be negative"));
        
        return errors;
    }
    
    
    /**
     * Add values from the CMEMS product
     * @param options
     * @param product
     * @return
     */
    public static ImportOptions applyDefaults(ImportOptions options, CmemsProduct product) {
        if(isEmpty(options.getMotu()))
            options.setMotu(product.getMotuServer());
        return options;
    }
    
    /**
     * Add values from the Motu dataset
     * @param options
     * @param product
     * @return
     */
    public static ImportOptions applyDefaults(ImportOptions options, ProductMetadataInfo product) {
        // time
        if(isEmpty(options.gettLo()))
            options.settLo(product.getFirstAvailableTimeCode());
        if(isEmpty(options.gettHi()))
            options.settHi(product.getLastAvailableTimeCode());
        // x
        if(isEmpty(options.getxLo()))
            options.setxLo(Double.parseDouble(product.getXAxis().getLower()));
        if(isEmpty(options.getxHi()))
            options.setxHi(Double.parseDouble(product.getXAxis().getUpper()));
        // y
        if(isEmpty(options.getyLo()))
            options.setyLo(Double.parseDouble(product.getYAxis().getLower()));
        if(isEmpty(options.getyHi()))
            options.setyHi(Double.parseDouble(product.getYAxis().getUpper()));
        // z
        if(isEmpty(options.getzLo()))
            options.setzLo(Double.parseDouble(product.getZAxis().getLower()));
        if(isEmpty(options.getzHi()))
            options.setzHi(Double.parseDouble(product.getZAxis().getUpper()));
        // variables
        if(options.getVariables()==null || options.getVariables().isEmpty()) {
            for(Variable var: product.getVariables()) {
                options.addVariable(var.getName());
            }
        }
        // backtime
        if(isEmpty(options.getBackTime()))
            options.setBackTime(0);
        // chunkspan
        if(isEmpty(options.getChunkSpan()))
            options.setChunkSpan(ChunkTimespan.MONTH);
        // frequency
        if(isEmpty(options.getImportSchedule())) {
            options.setImportSchedule(generateRandomMonthlyCron());
        }
        return options;
    }
    
    private static String generateRandomMonthlyCron() {
        Long hour = Math.round(Math.random()*24);
        Long minute = Math.round(Math.random()*60);
        Long day = Math.round(Math.random()*28);
        return String.format("0 %d %d %d 1/1 ? *", minute, hour, day);
    }

    /**
     * Validate the request.
     * @param options
     * @param product
     * @return
     * @throws InvalidParameterException
     */
    public static Collection<ValidationError> validate(ImportOptions options, ProductMetadataInfo product) throws InvalidParameterException {
        Collection<ValidationError> errors = new Vector<>();

        logger.debug("checking basic metadata.");
        errors.addAll(checkSet(MOTU, options.getMotu()));
        errors.addAll(checkSet(PRODUCT, options.getProduct()));
        errors.addAll(checkSet(DATASET, options.getDataset()));
        logger.debug(errors.size()+" errors so far.");

        logger.debug("checking variables.");
        if(options.getVariables().isEmpty())
            errors.add(new ValidationError(VARIABLE, "At least one variable must be set"));
        logger.debug(errors.size()+" errors so far.");
        
        logger.debug("checking longitude range");
        errors.addAll(checkRange(options.getxLo(), options.getxHi(),
                product.getXAxis().getLower(), product.getXAxis().getUpper(),
                XLO, XHI));
        logger.debug(errors.size()+" errors so far.");
        
        logger.debug("checking latitude range");
        errors.addAll(checkRange(options.getyLo(), options.getyHi(),
                product.getYAxis().getLower(), product.getYAxis().getUpper(),
                YLO, YHI));
        logger.debug(errors.size()+" errors so far.");
        
        logger.debug("checking depth range");
        errors.addAll(checkRange(options.getzLo(), options.getzHi(),
                product.getZAxis().getLower(), product.getZAxis().getUpper(),
                ZLO, ZHI));
        logger.debug(errors.size()+" errors so far.");
        
        logger.debug("checking time range");
        errors.addAll(checkRange(options.gettLo(), options.gettHi(), 
                product.getFirstAvailableTimeCode(), product.getLastAvailableTimeCode(),
                TLO, THI));
        // do not proceed if there are problems with ranges. Following checks require ranges are ok.
        if(errors.size()>0)
            return errors;
        logger.debug(errors.size()+" errors so far.");
        
        logger.debug("checking at least one time tick");
        // check at least one time tick
        logger.debug(product.getAvailableTimeCodes().size() + " time ticks in the dataset");
        if(product.getTimeCodesInInterval(options.gettLo(), options.gettHi()).isEmpty())
            errors.add(new ValidationError(TLO+", "+THI, "No data available for the given time range"));
        logger.debug(errors.size()+" errors so far.");
        
//        logger.debug("checking at least one depth level");
//        // check at least one depth level
//        if(product.getDepthsInInterval(options.getzLo(), options.getzHi()).isEmpty())
//            errors.add(new ValidationError(ZLO+", "+ZHI, "No data available for the given depth range"));
//        logger.debug(errors.size()+" errors so far.");

        // check import schedule
        // TODO
        
        // check chunk size
        // TODO
        
        // check backtime
        // TODO
        
        return errors;

    }

    /**
     * Check if the value is not null or, in case of String, is not empty.
     * @param value
     * @return
     */
    public static boolean isEmpty(Object value) {
        if(value==null)
            return true;
        if(value instanceof String &&  ((String)value).trim().isEmpty())
            return true;
        return false;
    }
    
    /**
     * Check if min is actually lesser than max.
     * @param min
     * @param max
     * @return
     */
    private static boolean ordered(Double min, Double max) {
        if(!isEmpty(min) && !isEmpty(max)) {
            if(max<min) {
                return false;
            }
        }
        return true;
    }


    private static Collection<ValidationError> checkRange(Double rMin, Double rMax, String dMin, String dMax, String minKey, String maxKey) {
        return checkRange(rMin, rMax, Double.parseDouble(dMin), Double.parseDouble(dMax), minKey, maxKey);
    }

    private static Collection<ValidationError> checkRange(Calendar rMin, Calendar rMax, Calendar dMin, Calendar dMax, String minKey, String maxKey) {
        return checkRange(rMin.getTimeInMillis(), rMax.getTimeInMillis(), dMin.getTimeInMillis(), dMax.getTimeInMillis(), minKey, maxKey);
    }

    private static Collection<ValidationError> checkRange(Long rMin, Long rMax, Long dMin, Long dMax, String minKey, String maxKey) {
        return checkRange(new Double(rMin), new Double(rMax), new Double(dMin), new Double(dMax), minKey, maxKey);
    }
    
    private static Collection<ValidationError> checkRange(Double rMin, Double rMax, Double dMin, Double dMax, String minKey, String maxKey) {
        Collection<ValidationError> errors = new Vector<>();

        final String ORDERED = "'%s' must be lower than '%s'";
        final String TOO_SMALL = "'%s' is less than range lower bound (%f)";
        final String TOO_BIG = "'%s' is greater than range upper bound (%s)";
        
        // check null values
        errors.addAll(checkSet(minKey, rMin));
        errors.addAll(checkSet(maxKey, rMax));
        
        // chech that min <= max
        if(!ordered(rMin, rMax))
            errors.add(new ValidationError(minKey+","+maxKey, String.format(ORDERED,  minKey, maxKey)));
        
        // check values are inside the range
        if(rMin < dMin)
            errors.add(new ValidationError(minKey, String.format(TOO_SMALL, minKey, dMin)));
        if(rMax > dMax)
            errors.add(new ValidationError(maxKey, String.format(TOO_BIG, maxKey, dMax)));
        
        for(ValidationError err: errors) {
            System.out.println(err.getMessage());
        }
        
        return errors;
    }

    private static Collection<ValidationError> checkSet(String key, Object value) {
        final String MUST_BE_SET = "Field '%s' must be set";
        Collection<ValidationError> errors = new Vector<>();
        if(isEmpty(value))
            errors.add(new ValidationError(key, String.format(MUST_BE_SET, key)));
        return errors;
    }

    
}
