package org.gcube.dataanalysis.ecoengine.transducers.charts;

import java.awt.Image;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;

import org.gcube.contentmanagement.graphtools.data.conversions.ImageTools;
import org.gcube.contentmanagement.lexicalmatcher.analysis.core.LexicalEngineConfiguration;
import org.gcube.contentmanagement.lexicalmatcher.utils.AnalysisLogger;
import org.gcube.dataanalysis.ecoengine.configuration.AlgorithmConfiguration;
import org.gcube.dataanalysis.ecoengine.datatypes.PrimitiveType;
import org.gcube.dataanalysis.ecoengine.datatypes.StatisticalType;
import org.gcube.dataanalysis.ecoengine.datatypes.enumtypes.PrimitiveTypes;
import org.gcube.dataanalysis.ecoengine.interfaces.StandardLocalExternalAlgorithm;
import org.gcube.dataanalysis.ecoengine.utils.DatabaseFactory;
import org.gcube.dataanalysis.ecoengine.utils.DatabaseUtils;
import org.gcube.dataanalysis.ecoengine.utils.IOHelper;
import org.hibernate.SessionFactory;
import org.jfree.chart.JFreeChart;

public abstract class AbstractChartsProducer extends StandardLocalExternalAlgorithm {

	static String inputTableParameter = "InputTable";
	static String attributesParameter = "Attributes";
	static String quantitiesParameter = "Quantities";
	static String timeParameter = "Time";
	public HashMap<String, Image> producedImages = new HashMap<String, Image>();
	public boolean displaycharts = true;
	static int maxSeries = 10;

	SessionFactory connection = null;

	@Override
	protected abstract void setInputParameters();
	
	@Override
	public StatisticalType getOutput() {
		PrimitiveType images = new PrimitiveType(HashMap.class.getName(), producedImages, PrimitiveTypes.IMAGES, "images", "Charts");
		LinkedHashMap<String, StatisticalType> outputmap = new LinkedHashMap<String, StatisticalType>();
		outputmap.put("Images", images);

		PrimitiveType output = new PrimitiveType(HashMap.class.getName(), outputmap, PrimitiveTypes.MAP, "ResultsMap", "Results Map");
		return output;
	}

	@Override
	public void init() throws Exception {
		AnalysisLogger.getLogger().debug("ChartsProducer Initialized");
	}

	@Override
	public abstract String getDescription();

	public String[] getDimensions() {
		String dimensionS = IOHelper.getInputParameter(config, attributesParameter);
		String[] dimensions = null;
		try{dimensions = dimensionS.split(AlgorithmConfiguration.getListSeparator());}catch(Exception e){}
		if (dimensions!=null && dimensions.length==1 && dimensions[0].trim().length()==0)
			dimensions = null;
		return dimensions;
	}

	public String[] getQuantities() {
		String quantitieS = IOHelper.getInputParameter(config, quantitiesParameter);
		String[] quantities = quantitieS.split(AlgorithmConfiguration.getListSeparator());
		return quantities;
	}

	public String getTimeDimension(){
		String timeS = IOHelper.getInputParameter(config, timeParameter);
		return timeS;
	}
	
	public String InfoRetrievalQuery(String table, String[] dimensions, String quantity,String time) {
		if (time!=null){
			if (dimensions!=null && dimensions.length>0)
				return "select * from (select distinct " + Arrays.toString(dimensions).replace("[", "").replace("]", "") + " , " + quantity +","+time+" from "+table+" order by " + quantity + " limit " + maxSeries+") as a order by "+time;
			else
				return "select * from (select distinct " +quantity +","+time+" from "+table+" order by " + quantity + " limit " + maxSeries+") as a order by "+time;
		}
		else{
			if (dimensions!=null && dimensions.length>0)
				return "select distinct " + Arrays.toString(dimensions).replace("[", "").replace("]", "") + " , " + quantity +" from "+table+" order by " + quantity + " limit " + maxSeries;
			else
				return "select distinct row_number() over(), " + quantity +" from "+table+" order by " + quantity + " limit " + maxSeries;
		}
	}
	
	@Override
	protected void process() throws Exception {
		status = 0;
		AnalysisLogger.setLogger(config.getConfigPath() + AlgorithmConfiguration.defaultLoggerFile);

		long t0 = System.currentTimeMillis();
		try {
			AnalysisLogger.getLogger().debug("ChartsProducer started");
			String driver = config.getParam("DatabaseDriver");
			String username = config.getParam("DatabaseUserName");
			String pwd = config.getParam("DatabasePassword");
			String url = config.getParam("DatabaseURL");
			String table = IOHelper.getInputParameter(config, inputTableParameter);

			AnalysisLogger.getLogger().debug("ChartsProducer: Driver: " + driver + " User " + username + " URL " + url + " Table: " + table);
			connection = DatabaseUtils.initDBSession(config);

			AnalysisLogger.getLogger().debug("ChartsProducer: Connection initialized");

			LexicalEngineConfiguration conf = new LexicalEngineConfiguration();
			conf.setDatabaseUserName(username);
			conf.setDatabasePassword(pwd);
			conf.setDatabaseDriver(driver);
			conf.setDatabaseURL(url);
			conf.setDatabaseDialect("org.hibernate.dialect.PostgreSQLDialect");

			String[] dimensions = getDimensions();
			String[] quantities = getQuantities();
			String time = getTimeDimension();
			
			//one quantity for each chart 
			for (String quantity : quantities) {
				//produce chart with dimensions,quantity, time
				String query = InfoRetrievalQuery(table,dimensions,quantity,time);
				AnalysisLogger.getLogger().debug("ChartsProducer: Query for retrieving information "+query);
				List<Object> values = DatabaseFactory.executeSQLQuery(query, connection);
				Image[] charts= createCharts(dimensions,quantity,time,values,displaycharts);
				for (Image chart:charts){
					if (chart!=null){
						producedImages.put("Charts focused on "+quantity, chart);
					}
				}
			}

			AnalysisLogger.getLogger().debug("ChartsProducer: finished");

		} catch (Throwable e) {
			e.printStackTrace();
			throw new Exception(e.getMessage());
		} finally {
			shutdown();
			status = 100;
		}
	}

	public abstract Image[] createCharts(String[] dimensions,String quantity,String time,List<Object> rows, boolean displaychart);
		
	
	@Override
	public void shutdown() {
		AnalysisLogger.getLogger().debug("ChartsProducer shutdown");
		DatabaseUtils.closeDBConnection(connection);
	}

}
