package org.gcube.dataanalysis.copernicus.motu.client;

import java.io.File;
import java.util.Collection;
import java.util.UUID;
import java.util.Vector;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.io.FileUtils;
import org.apache.log4j.Logger;
import org.gcube.dataanalysis.copernicus.motu.model.Axis;
import org.gcube.dataanalysis.copernicus.motu.model.MotuCatalogue;
import org.gcube.dataanalysis.copernicus.motu.model.MotuMarshaller;
import org.gcube.dataanalysis.copernicus.motu.model.ProductMetadataInfo;
import org.gcube.dataanalysis.copernicus.motu.model.RequestSize;
import org.gcube.dataanalysis.copernicus.motu.model.ServiceMetadata;
import org.gcube.dataanalysis.copernicus.motu.model.StatusModeResponse;
import org.gcube.dataanalysis.copernicus.motu.util.NetworkUtils;
import org.gcube.dataanalysis.copernicus.motu.util.MotuWorkspace;

public class MotuClient extends PlainMotuClient {

    /**
     * A logger for this class.
     */
    private static final Logger LOGGER = Logger.getLogger(MotuClient.class);

    /**
     * The preferred size for downloaded chunks.
     */
    private Long preferredDownloadSize;

    /**
     * This is the default order in which requests are split when needed for
     * size constraints.
     */
    private static final String[] DEFAULT_VAR_SPLIT = { "v", "t", "x", "y", "z" };

    /**
     * How often to check for an update about ongoing submission/downloads.
     */
    private static final Long CHECK_INTERVAL = 10000L;

    /**
     *
     * @param serviceURL
     *            the URL of the Motu server.
     */
    public MotuClient(final String serviceURL) {
        super(serviceURL);
    }

    /**
     * Return the preferred download size set for this client. If no size has
     * been set, max_value is returned; the maximum enforced by the server will
     * be used.
     * @return The preferred size, as set by the client.
     */
    public Long getPreferredDownloadSize() {
        if (this.preferredDownloadSize != null) {
            return this.preferredDownloadSize;
        } else {
            return Long.MAX_VALUE;
        }
    }

    /**
     * Set the preferred size of chunks. The client will break the request to
     * get chunks as close as possible to this size.
     * @param preferredDownloadSize
     *            the preferred size of downloaded chunks.
     */
    public void setPreferredDownloadSize(final Long preferredDownloadSize) {
        this.preferredDownloadSize = preferredDownloadSize;
    }

    /**
     * Download the dataset corresponding to the given request, by splitting
     * requests if needed.
     * @param request
     *            the request for dataset.
     * @throws Exception
     */
    public void downloadProduct(final DownloadRequest request) throws Exception {
        this.downloadProduct(request, DEFAULT_VAR_SPLIT);
    }

    /**
     * A more robust implementation of the super.getSize, as it makes sure that
     * all parameters are set.
     */
    @Override
    public RequestSize getSize(final DownloadRequest request) throws Exception {
        this.ensureParameters(request);
        return super.getSize(request);
    }

    /**
     * Return the full size of the given product, within the given service.
     * @param service
     * @param product
     * @return
     * @throws Exception
     */
    public RequestSize getSize(final String service, final String product) throws Exception {
        DownloadRequest request = new DownloadRequest();
        request.setService(service);
        request.setProduct(product);
        return this.getSize(request);
    }

    /**
     * Make sure all parameters are set in the request. If not, they're set with
     * values retrieved from the server as metadata.
     * @param request
     * @return
     * @throws Exception
     */
    private DownloadRequest ensureParameters(final DownloadRequest request)
            throws Exception {
        ProductMetadataInfo pmi = this.describeProduct(request);
        if (request.gettLo() == null) {
//            request.settLo(pmi.getTimeCoverage().getStart());
            request.settLo(pmi.getFirstAvailableTimeCode());
        }
        if (request.gettHi() == null) {
//            request.settHi(pmi.getTimeCoverage().getEnd());
            request.settHi(pmi.getLastAvailableTimeCode());
        }
        for (Axis axis : pmi.getDataGeospatialCoverage()) {
            if (axis.getName().equals("lon")) {
                if (request.getxLo() == null) {
                    request.setxLo(Double.parseDouble(axis.getLower()));
                }
                if (request.getxHi() == null) {
                    request.setxHi(Double.parseDouble(axis.getUpper()));
                }
            }
            if (axis.getName().equals("lat")) {
                if (request.getyLo() == null) {
                    request.setyLo(Double.parseDouble(axis.getLower()));
                }
                if (request.getyHi() == null) {
                    request.setyHi(Double.parseDouble(axis.getUpper()));
                }
            }
            if (axis.getName().equals("depth")) {
                if (request.getzLo() == null) {
                    request.setzLo(Double.parseDouble(axis.getLower()));
                }
                if (request.getzHi() == null) {
                    request.setzHi(Double.parseDouble(axis.getUpper()));
                }
            }
        }
        return request;
    }

    public ProductMetadataInfo describeProduct(String service, String product) throws Exception {
        DownloadRequest request = new DownloadRequest();
        request.setService(service);
        request.setProduct(product);
        return super.describeProduct(request);
    }

    /**
     * Download the dataset corresponding to the given request, by splitting
     * requests if needed.
     * @param request
     *            the request for dataset.
     * @param splitParameters
     *            the parameters along which the request has to be split.
     * @throws Exception
     */
    public File downloadProduct(final DownloadRequest request, File destFile, 
            final String... splitParameters) throws Exception {

        // create a workspace
        String executionId = UUID.randomUUID().toString();
        MotuWorkspace workspace = new MotuWorkspace(executionId);
        workspace.setExecutionsRoot("/tmp/motu-downloads");
//        workspace.setBinariesLocation(".");
        workspace.setInputLocation("download");
        workspace.setOutputLocation("output");
        workspace.ensureStructureExists();

        // to avoid submitting too much requests
        ThreadedSubmitter ts = new ThreadedSubmitter(this);

        // to avoid downloading too many files concurrently
        ThreadedDownloader td = new ThreadedDownloader();
        td.setDestinationDir(workspace.getInputLocation());

        // a callback to be invoked when a chunk is ready to be donwloaded
        ts.setListener(new WorkCompleteListener<DownloadRequestEnvelope>() {
            public void workComplete(final DownloadRequestEnvelope chunk) {
                td.push(chunk);
            }
        });

        // set any missing parameter by getting metadata from the server
        this.ensureParameters(request);

        // split the request into chunks acceptable by the server
        RequestSplitter splitter = new RequestSplitter(request);
        splitter.setMotuClient(this);
        for (DownloadRequestEnvelope envelope : splitter
                .splitRequest(splitParameters)) {
            // push chunks to the submitter
            ts.push(envelope);
        }

        // wait for all chunks to complete
        while (true) {
            if (ts.isComplete() && td.isComplete()) {
                break;
            } else {
                Thread.sleep(CHECK_INTERVAL);
            }
        }
        
        // do merge the output
        File mergedFile = new File(workspace.getOutputLocation(), request.getProduct()+".nc");
        ChunkMerger cm = new ChunkMerger();
        cm.mergeAll(workspace.getInputLocation(), mergedFile);

        if(destFile==null) {
            destFile = new File("/tmp", executionId+"-"+request.getProduct()+".nc");
        }
        FileUtils.moveFile(mergedFile, destFile);

        // TODO: clean the workspace (unless there are errors)
        // workspace.destroy();
        return destFile;
    }
    
    public File downloadProduct(final DownloadRequest request,
            final String... splitParameters) throws Exception {
        return this.downloadProduct(request,  null, splitParameters);
    }
    
    /**
     * Return the list of services exposed by this Motu server
     * @return
     */
    public Collection<ServiceMetadata> listServices() throws Exception {

        // retrieve the html page
        String url = this.getServiceURL() + "?action=listservices";
        String html = NetworkUtils.doGet(url, this.getCasProxy());

        // build the pattern to look for 'a' tags
        String str = "<tr> <td> <a href=\\?action=listcatalog&service=([^>]+)> ([^<]+) </a> </td> <td> ([^<]+) </td> </tr>";
        str = str.replaceAll("\\s", "\\\\s*");
        Pattern pattern = Pattern.compile(str);

        // scan the html page and populate output
        Collection<ServiceMetadata> output = new Vector<>();
        Matcher m = pattern.matcher(html);
        while (m.find()) {
            ServiceMetadata service = new ServiceMetadata();
            service.setName(m.group(1).trim());
            service.setDescription(m.group(2).trim());
            service.setType(m.group(3).trim());
            output.add(service);
        }

        return output;
    }

    /**
     * Return the list of products (i.e. datasets) available for the given
     * service.
     * @param service
     * @return
     */
    public Collection<String> listProductsNames(final String service) throws Exception {

        // retrieve the html page
        String url = this.getServiceURL() + "?action=listcatalog&service="+ service;
        String html = NetworkUtils.doGet(url, this.getCasProxy());

        // build the pattern to look for 'a' tags
        String str = "<a href=\\?action=productdownloadhome&service=([^&]+)&product=([^&>]+)>";
        str = str.replaceAll("\\s", "\\\\s*");
        Pattern pattern = Pattern.compile(str);

        // scan the html page and populate output
        Collection<String> output = new Vector<>();
        Matcher m = pattern.matcher(html);
        while (m.find()) {
            output.add(m.group(2));
        }

        return output;
    }

    /**
     * Retrieves the status of all given requests.
     * @param requests
     *            the requests whose status has to be checked.
     * @return a collection with the status of all requests.
     * @throws Exception
     */
    public Collection<StatusModeResponse> checkAllStatus(
            final Collection<StatusModeResponse> requests) throws Exception {
        Collection<StatusModeResponse> out = new Vector<>();
        for (StatusModeResponse smr : requests) {
            out.add(super.checkStatus(smr.getRequestId()));
        }
        for (StatusModeResponse smr : out) {
            LOGGER.info("Status of " + smr.getRequestId() + " is "
                    + smr.getStatus());
        }
        return out;
    }

    /**
     * Query the server to get a catalog of available services and associated
     * products.
     * @return a catalog
     * @throws Exception
     */
    public MotuCatalogue getCatalogue() throws Exception {
        MotuCatalogue catalogue = new MotuCatalogue();
        catalogue.setUrl(this.getServiceURL());
        for (ServiceMetadata service : this.listServices()) {
            if (service.getType().equals("Subsetter")) {
                for (String productName : this
                        .listProductsNames(service.getName())) {
                    ProductMetadataInfo pmi = this
                            .describeProduct(service.getName(), productName);
                    service.addProduct(pmi);
                }
            }
            if (service.hasProducts()) {
                catalogue.addService(service);
            }
        }
        return catalogue;
    }
    
}
