/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.math.linear;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import org.apache.commons.math.MathRuntimeException;
import org.apache.commons.math.MaxIterationsExceededException;
import org.apache.commons.math.linear.ArrayRealVector;
import org.apache.commons.math.linear.DecompositionSolver;
import org.apache.commons.math.linear.EigenDecomposition;
import org.apache.commons.math.linear.InvalidMatrixException;
import org.apache.commons.math.linear.MatrixUtils;
import org.apache.commons.math.linear.RealMatrix;
import org.apache.commons.math.linear.RealVector;
import org.apache.commons.math.linear.SingularMatrixException;
import org.apache.commons.math.linear.TriDiagonalTransformer;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class EigenDecompositionImpl
implements EigenDecomposition {
    private static final double TOLERANCE = (double)1.110223E-14f;
    private static final double TOLERANCE_2 = 1.232595164407831E-28;
    private double splitTolerance;
    private double[] main;
    private double[] secondary;
    private double[] squaredSecondary;
    private TriDiagonalTransformer transformer;
    private double lowerSpectra;
    private double upperSpectra;
    private double minPivot;
    private double sigma;
    private double sigmaLow;
    private double tau;
    private double[] work;
    private int pingPong;
    private double qMax;
    private double eMin;
    private int tType;
    private double dMin;
    private double dMin1;
    private double dMin2;
    private double dN;
    private double dN1;
    private double dN2;
    private double g;
    private double[] realEigenvalues;
    private double[] imagEigenvalues;
    private ArrayRealVector[] eigenvectors;
    private RealMatrix cachedV;
    private RealMatrix cachedD;
    private RealMatrix cachedVt;

    public EigenDecompositionImpl(RealMatrix matrix, double splitTolerance) throws InvalidMatrixException {
        if (!this.isSymmetric(matrix)) {
            throw new InvalidMatrixException("eigen decomposition of assymetric matrices not supported yet", new Object[0]);
        }
        this.splitTolerance = splitTolerance;
        this.transformToTridiagonal(matrix);
        this.decompose();
    }

    public EigenDecompositionImpl(double[] main, double[] secondary, double splitTolerance) throws InvalidMatrixException {
        this.main = (double[])main.clone();
        this.secondary = (double[])secondary.clone();
        this.transformer = null;
        this.squaredSecondary = new double[secondary.length];
        for (int i = 0; i < this.squaredSecondary.length; ++i) {
            double s = secondary[i];
            this.squaredSecondary[i] = s * s;
        }
        this.splitTolerance = splitTolerance;
        this.decompose();
    }

    private boolean isSymmetric(RealMatrix matrix) {
        int rows = matrix.getRowDimension();
        int columns = matrix.getColumnDimension();
        double eps = (double)(10 * rows * columns) * (double)1.110223E-16f;
        for (int i = 0; i < rows; ++i) {
            for (int j = i + 1; j < columns; ++j) {
                double mji;
                double mij = matrix.getEntry(i, j);
                if (!(Math.abs(mij - (mji = matrix.getEntry(j, i))) > Math.max(Math.abs(mij), Math.abs(mji)) * eps)) continue;
                return false;
            }
        }
        return true;
    }

    private void decompose() {
        this.cachedV = null;
        this.cachedD = null;
        this.cachedVt = null;
        this.work = new double[6 * this.main.length];
        this.computeGershgorinCircles();
        this.findEigenvalues();
        this.eigenvectors = null;
    }

    @Override
    public RealMatrix getV() throws InvalidMatrixException {
        if (this.cachedV == null) {
            if (this.eigenvectors == null) {
                this.findEigenVectors();
            }
            int m = this.eigenvectors.length;
            this.cachedV = MatrixUtils.createRealMatrix(m, m);
            for (int k = 0; k < m; ++k) {
                this.cachedV.setColumnVector(k, this.eigenvectors[k]);
            }
        }
        return this.cachedV;
    }

    @Override
    public RealMatrix getD() throws InvalidMatrixException {
        if (this.cachedD == null) {
            this.cachedD = MatrixUtils.createRealDiagonalMatrix(this.realEigenvalues);
        }
        return this.cachedD;
    }

    @Override
    public RealMatrix getVT() throws InvalidMatrixException {
        if (this.cachedVt == null) {
            if (this.eigenvectors == null) {
                this.findEigenVectors();
            }
            int m = this.eigenvectors.length;
            this.cachedVt = MatrixUtils.createRealMatrix(m, m);
            for (int k = 0; k < m; ++k) {
                this.cachedVt.setRowVector(k, this.eigenvectors[k]);
            }
        }
        return this.cachedVt;
    }

    @Override
    public double[] getRealEigenvalues() throws InvalidMatrixException {
        return (double[])this.realEigenvalues.clone();
    }

    @Override
    public double getRealEigenvalue(int i) throws InvalidMatrixException, ArrayIndexOutOfBoundsException {
        return this.realEigenvalues[i];
    }

    @Override
    public double[] getImagEigenvalues() throws InvalidMatrixException {
        return (double[])this.imagEigenvalues.clone();
    }

    @Override
    public double getImagEigenvalue(int i) throws InvalidMatrixException, ArrayIndexOutOfBoundsException {
        return this.imagEigenvalues[i];
    }

    @Override
    public RealVector getEigenvector(int i) throws InvalidMatrixException, ArrayIndexOutOfBoundsException {
        if (this.eigenvectors == null) {
            this.findEigenVectors();
        }
        return this.eigenvectors[i].copy();
    }

    @Override
    public double getDeterminant() {
        double determinant = 1.0;
        for (double lambda : this.realEigenvalues) {
            determinant *= lambda;
        }
        return determinant;
    }

    @Override
    public DecompositionSolver getSolver() {
        if (this.eigenvectors == null) {
            this.findEigenVectors();
        }
        return new Solver(this.realEigenvalues, this.imagEigenvalues, this.eigenvectors);
    }

    private void transformToTridiagonal(RealMatrix matrix) {
        this.transformer = new TriDiagonalTransformer(matrix);
        this.main = this.transformer.getMainDiagonalRef();
        this.secondary = this.transformer.getSecondaryDiagonalRef();
        this.squaredSecondary = new double[this.secondary.length];
        for (int i = 0; i < this.squaredSecondary.length; ++i) {
            double s = this.secondary[i];
            this.squaredSecondary[i] = s * s;
        }
    }

    private void computeGershgorinCircles() {
        int m = this.main.length;
        int lowerStart = 4 * m;
        int upperStart = 5 * m;
        this.lowerSpectra = Double.POSITIVE_INFINITY;
        this.upperSpectra = Double.NEGATIVE_INFINITY;
        double eMax = 0.0;
        double eCurrent = 0.0;
        for (int i = 0; i < m - 1; ++i) {
            double upper;
            double lower;
            double dCurrent = this.main[i];
            double ePrevious = eCurrent;
            eCurrent = Math.abs(this.secondary[i]);
            eMax = Math.max(eMax, eCurrent);
            double radius = ePrevious + eCurrent;
            this.work[lowerStart + i] = lower = dCurrent - radius;
            this.lowerSpectra = Math.min(this.lowerSpectra, lower);
            this.work[upperStart + i] = upper = dCurrent + radius;
            this.upperSpectra = Math.max(this.upperSpectra, upper);
        }
        double dCurrent = this.main[m - 1];
        this.work[lowerStart + m - 1] = dCurrent - eCurrent;
        this.work[upperStart + m - 1] = dCurrent + eCurrent;
        this.minPivot = Double.MIN_NORMAL * Math.max(1.0, eMax * eMax);
    }

    private void findEigenvalues() throws InvalidMatrixException {
        List<Integer> splitIndices = this.computeSplits();
        this.realEigenvalues = new double[this.main.length];
        this.imagEigenvalues = new double[this.main.length];
        int begin = 0;
        for (int end : splitIndices) {
            int n = end - begin;
            switch (n) {
                case 1: {
                    this.process1RowBlock(begin);
                    break;
                }
                case 2: {
                    this.process2RowsBlock(begin);
                    break;
                }
                case 3: {
                    this.process3RowsBlock(begin);
                    break;
                }
                default: {
                    int i;
                    double[] range = this.eigenvaluesRange(begin, n);
                    double oneFourth = 0.25 * (3.0 * range[0] + range[1]);
                    int oneFourthCount = this.countEigenValues(oneFourth, begin, n);
                    double threeFourth = 0.25 * (range[0] + 3.0 * range[1]);
                    int threeFourthCount = this.countEigenValues(threeFourth, begin, n);
                    boolean chooseLeft = oneFourthCount - 1 >= n - threeFourthCount;
                    double lambda = chooseLeft ? range[0] : range[1];
                    this.tau = (range[1] - range[0]) * (double)1.110223E-16f * (double)n + 2.0 * this.minPivot;
                    this.ldlTDecomposition(lambda, begin, n);
                    this.processGeneralBlock(n);
                    if (chooseLeft) {
                        for (i = 0; i < n; ++i) {
                            this.realEigenvalues[begin + i] = lambda + this.work[4 * i];
                        }
                    } else {
                        for (i = 0; i < n; ++i) {
                            this.realEigenvalues[begin + i] = lambda - this.work[4 * i];
                        }
                    }
                }
            }
            begin = end;
        }
        Arrays.sort(this.realEigenvalues);
        int i = 0;
        for (int j = this.realEigenvalues.length - 1; i < j; ++i, --j) {
            double tmp = this.realEigenvalues[i];
            this.realEigenvalues[i] = this.realEigenvalues[j];
            this.realEigenvalues[j] = tmp;
        }
    }

    private List<Integer> computeSplits() {
        ArrayList<Integer> list = new ArrayList<Integer>();
        double absDCurrent = Math.abs(this.main[0]);
        for (int i = 0; i < this.secondary.length; ++i) {
            double absDPrevious = absDCurrent;
            absDCurrent = Math.abs(this.main[i + 1]);
            double max = this.splitTolerance * Math.sqrt(absDPrevious * absDCurrent);
            if (!(Math.abs(this.secondary[i]) <= max)) continue;
            list.add(i + 1);
            this.secondary[i] = 0.0;
            this.squaredSecondary[i] = 0.0;
        }
        list.add(this.secondary.length + 1);
        return list;
    }

    private void process1RowBlock(int index) {
        this.realEigenvalues[index] = this.main[index];
    }

    private void process2RowsBlock(int index) throws InvalidMatrixException {
        double largestRoot;
        double q0 = this.main[index];
        double q1 = this.main[index + 1];
        double s = q0 + q1;
        double e12 = this.squaredSecondary[index];
        double p = q0 * q1 - e12;
        double delta = s * s - 4.0 * p;
        if (delta < 0.0) {
            throw new InvalidMatrixException("cannot solve degree {0} equation", 2);
        }
        this.realEigenvalues[index] = largestRoot = 0.5 * (s + Math.sqrt(delta));
        this.realEigenvalues[index + 1] = p / largestRoot;
    }

    private void process3RowsBlock(int index) throws InvalidMatrixException {
        double t;
        double q0 = this.main[index];
        double q1 = this.main[index + 1];
        double q2 = this.main[index + 2];
        double q1q2Me22 = q1 * q2 - this.squaredSecondary[index + 1];
        double e12 = this.squaredSecondary[index];
        double c = q0 * q1 + q0 * q2 + q1q2Me22 - e12;
        double b = -(q0 + q1 + q2);
        double b2 = b * b;
        double q = (3.0 * c - b2) / 9.0;
        double d = q2 * e12 - q0 * q1q2Me22;
        double r = ((9.0 * c - 2.0 * b2) * b - 27.0 * d) / 54.0;
        double delta = q * q * q + r * r;
        if (delta >= 0.0) {
            throw new InvalidMatrixException("cannot solve degree {0} equation", 3);
        }
        double sqrtMq = Math.sqrt(-q);
        double theta = Math.acos(r / (-q * sqrtMq));
        double alpha = 2.0 * sqrtMq;
        double beta = b / 3.0;
        double z0 = alpha * Math.cos(theta / 3.0) - beta;
        double z1 = alpha * Math.cos((theta + Math.PI * 2) / 3.0) - beta;
        double z2 = alpha * Math.cos((theta + Math.PI * 4) / 3.0) - beta;
        if (z0 < z1) {
            t = z0;
            z0 = z1;
            z1 = t;
        }
        if (z1 < z2) {
            t = z1;
            z1 = z2;
            z2 = t;
        }
        if (z0 < z1) {
            t = z0;
            z0 = z1;
            z1 = t;
        }
        this.realEigenvalues[index] = z0;
        this.realEigenvalues[index + 1] = z1;
        this.realEigenvalues[index + 2] = z2;
    }

    private void processGeneralBlock(int n) throws InvalidMatrixException {
        double sumOffDiag = 0.0;
        for (int i = 0; i < n - 1; ++i) {
            int fourI = 4 * i;
            double ei = this.work[fourI + 2];
            sumOffDiag += ei;
        }
        if (sumOffDiag == 0.0) {
            return;
        }
        this.flipIfWarranted(n, 2);
        this.initialSplits(n);
        this.tType = 0;
        this.dMin1 = 0.0;
        this.dMin2 = 0.0;
        this.dN = 0.0;
        this.dN1 = 0.0;
        this.dN2 = 0.0;
        this.tau = 0.0;
        int i0 = 0;
        int n0 = n;
        while (n0 > 0) {
            double qMax;
            this.sigma = n0 == n ? 0.0 : -this.work[4 * n0 - 2];
            this.sigmaLow = 0.0;
            double eMin = i0 == n0 ? 0.0 : this.work[4 * n0 - 6];
            double eMax = 0.0;
            double qMin = qMax = this.work[4 * n0 - 4];
            i0 = 0;
            for (int i = 4 * (n0 - 2); i >= 0; i -= 4) {
                if (this.work[i + 2] <= 0.0) {
                    i0 = 1 + i / 4;
                    break;
                }
                if (qMin >= 4.0 * eMax) {
                    qMin = Math.min(qMin, this.work[i + 4]);
                    eMax = Math.max(eMax, this.work[i + 2]);
                }
                qMax = Math.max(qMax, this.work[i] + this.work[i + 2]);
                eMin = Math.min(eMin, this.work[i + 2]);
            }
            this.work[4 * n0 - 2] = eMin;
            this.dMin = -Math.max(0.0, qMin - 2.0 * Math.sqrt(qMin * eMax));
            this.pingPong = 0;
            int maxIter = 30 * (n0 - i0);
            int k = 0;
            while (i0 < n0) {
                if (k >= maxIter) {
                    throw new InvalidMatrixException(new MaxIterationsExceededException(maxIter));
                }
                n0 = this.goodStep(i0, n0);
                this.pingPong = 1 - this.pingPong;
                if (this.pingPong == 0 && n0 - i0 > 3 && this.work[4 * n0 - 1] <= 1.232595164407831E-28 * qMax && this.work[4 * n0 - 2] <= 1.232595164407831E-28 * this.sigma) {
                    int split = i0 - 1;
                    qMax = this.work[4 * i0];
                    eMin = this.work[4 * i0 + 2];
                    double previousEMin = this.work[4 * i0 + 3];
                    for (int i = 4 * i0; i < 4 * n0 - 11; i += 4) {
                        if (this.work[i + 3] <= 1.232595164407831E-28 * this.work[i] && this.work[i + 2] <= 1.232595164407831E-28 * this.sigma) {
                            this.work[i + 2] = -this.sigma;
                            split = i / 4;
                            qMax = 0.0;
                            eMin = this.work[i + 6];
                            previousEMin = this.work[i + 7];
                            continue;
                        }
                        qMax = Math.max(qMax, this.work[i + 4]);
                        eMin = Math.min(eMin, this.work[i + 2]);
                        previousEMin = Math.min(previousEMin, this.work[i + 3]);
                    }
                    this.work[4 * n0 - 2] = eMin;
                    this.work[4 * n0 - 1] = previousEMin;
                    i0 = split + 1;
                }
                ++k;
            }
        }
    }

    private void initialSplits(int n) {
        this.pingPong = 0;
        for (int k = 0; k < 2; ++k) {
            int i;
            double d = this.work[4 * (n - 1) + this.pingPong];
            for (i = 4 * (n - 2) + this.pingPong; i >= 0; i -= 4) {
                if (this.work[i + 2] <= 1.232595164407831E-28 * d) {
                    this.work[i + 2] = -0.0;
                    d = this.work[i];
                    continue;
                }
                d *= this.work[i] / (d + this.work[i + 2]);
            }
            d = this.work[this.pingPong];
            for (i = 2 + this.pingPong; i < 4 * n - 2; i += 4) {
                int j = i - 2 * this.pingPong - 1;
                this.work[j] = d + this.work[i];
                if (this.work[i] <= 1.232595164407831E-28 * d) {
                    this.work[i] = -0.0;
                    this.work[j] = d;
                    this.work[j + 2] = 0.0;
                    d = this.work[i + 2];
                    continue;
                }
                if (Double.MIN_NORMAL * this.work[i + 2] < this.work[j] && Double.MIN_NORMAL * this.work[j] < this.work[i + 2]) {
                    double tmp = this.work[i + 2] / this.work[j];
                    this.work[j + 2] = this.work[i] * tmp;
                    d *= tmp;
                    continue;
                }
                this.work[j + 2] = this.work[i + 2] * (this.work[i] / this.work[j]);
                d *= this.work[i + 2] / this.work[j];
            }
            this.work[4 * n - 3 - this.pingPong] = d;
            this.pingPong = 1 - this.pingPong;
        }
    }

    private int goodStep(int start, int end) {
        this.g = 0.0;
        int deflatedEnd = end;
        boolean deflating = true;
        while (deflating) {
            if (start >= deflatedEnd) {
                return deflatedEnd;
            }
            int k = 4 * deflatedEnd + this.pingPong - 1;
            if (start == deflatedEnd - 1 || start != deflatedEnd - 2 && (this.work[k - 5] <= 1.232595164407831E-28 * (this.sigma + this.work[k - 3]) || this.work[k - 2 * this.pingPong - 4] <= 1.232595164407831E-28 * this.work[k - 7])) {
                this.work[4 * deflatedEnd - 4] = this.sigma + this.work[4 * deflatedEnd - 4 + this.pingPong];
                --deflatedEnd;
                continue;
            }
            if (start == deflatedEnd - 2 || this.work[k - 9] <= 1.232595164407831E-28 * this.sigma || this.work[k - 2 * this.pingPong - 8] <= 1.232595164407831E-28 * this.work[k - 11]) {
                if (this.work[k - 3] > this.work[k - 7]) {
                    double tmp = this.work[k - 3];
                    this.work[k - 3] = this.work[k - 7];
                    this.work[k - 7] = tmp;
                }
                if (this.work[k - 5] > 1.232595164407831E-28 * this.work[k - 3]) {
                    double t = 0.5 * (this.work[k - 7] - this.work[k - 3] + this.work[k - 5]);
                    double s = this.work[k - 3] * (this.work[k - 5] / t);
                    s = s <= t ? this.work[k - 3] * this.work[k - 5] / (t * (1.0 + Math.sqrt(1.0 + s / t))) : this.work[k - 3] * this.work[k - 5] / (t + Math.sqrt(t * (t + s)));
                    t = this.work[k - 7] + (s + this.work[k - 5]);
                    int n = k - 3;
                    this.work[n] = this.work[n] * (this.work[k - 7] / t);
                    this.work[k - 7] = t;
                }
                this.work[4 * deflatedEnd - 8] = this.sigma + this.work[k - 7];
                this.work[4 * deflatedEnd - 4] = this.sigma + this.work[k - 3];
                deflatedEnd -= 2;
                continue;
            }
            deflating = false;
        }
        int l = 4 * deflatedEnd + this.pingPong - 1;
        if ((this.dMin <= 0.0 || deflatedEnd < end) && this.flipIfWarranted(deflatedEnd, 1)) {
            this.dMin2 = Math.min(this.dMin2, this.work[l - 1]);
            this.work[l - 1] = Math.min(this.work[l - 1], Math.min(this.work[3 + this.pingPong], this.work[7 + this.pingPong]));
            this.work[l - 2 * this.pingPong] = Math.min(this.work[l - 2 * this.pingPong], Math.min(this.work[6 + this.pingPong], this.work[6 + this.pingPong]));
            this.qMax = Math.max(this.qMax, Math.max(this.work[3 + this.pingPong], this.work[7 + this.pingPong]));
            this.dMin = -0.0;
        }
        if (this.dMin < 0.0 || Double.MIN_NORMAL * this.qMax < Math.min(this.work[l - 1], Math.min(this.work[l - 9], this.dMin2 + this.work[l - 2 * this.pingPong]))) {
            this.computeShiftIncrement(start, deflatedEnd, end - deflatedEnd);
            boolean loop = true;
            while (loop) {
                this.dqds(start, deflatedEnd);
                if (this.dMin >= 0.0 && this.dMin1 > 0.0) {
                    this.updateSigma(this.tau);
                    return deflatedEnd;
                }
                if (this.dMin < 0.0 && this.dMin1 > 0.0 && this.work[4 * deflatedEnd - 5 - this.pingPong] < (double)1.110223E-14f * (this.sigma + this.dN1) && Math.abs(this.dN) < (double)1.110223E-14f * this.sigma) {
                    this.work[4 * deflatedEnd - 3 - this.pingPong] = 0.0;
                    this.dMin = 0.0;
                    this.updateSigma(this.tau);
                    return deflatedEnd;
                }
                if (this.dMin < 0.0) {
                    if (this.tType < -22) {
                        this.tau = 0.0;
                        continue;
                    }
                    if (this.dMin1 > 0.0) {
                        this.tau = (this.tau + this.dMin) * 0.9999999999999998;
                        this.tType -= 11;
                        continue;
                    }
                    this.tau *= 0.25;
                    this.tType -= 12;
                    continue;
                }
                if (Double.isNaN(this.dMin)) {
                    this.tau = 0.0;
                    continue;
                }
                loop = false;
            }
        }
        this.dqd(start, deflatedEnd);
        return deflatedEnd;
    }

    private boolean flipIfWarranted(int n, int step) {
        if (1.5 * this.work[this.pingPong] < this.work[4 * (n - 1) + this.pingPong]) {
            int i = 0;
            for (int j = 4 * n - 1; i < j; i += 4, j -= 4) {
                for (int k = 0; k < 4; k += step) {
                    double tmp = this.work[i + k];
                    this.work[i + k] = this.work[j - k];
                    this.work[j - k] = tmp;
                }
            }
            return true;
        }
        return false;
    }

    private double[] eigenvaluesRange(int index, int n) {
        double middle;
        double range;
        int i;
        int lowerStart = 4 * this.main.length;
        int upperStart = 5 * this.main.length;
        double lower = Double.POSITIVE_INFINITY;
        double upper = Double.NEGATIVE_INFINITY;
        for (int i2 = 0; i2 < n; ++i2) {
            lower = Math.min(lower, this.work[lowerStart + index + i2]);
            upper = Math.max(upper, this.work[upperStart + index + i2]);
        }
        double tNorm = Math.max(Math.abs(lower), Math.abs(upper));
        double relativeTolerance = Math.sqrt(1.110223E-16f);
        double absoluteTolerance = 4.0 * this.minPivot;
        int maxIter = 2 + (int)((Math.log(tNorm + this.minPivot) - Math.log(this.minPivot)) / Math.log(2.0));
        double margin = 2.0 * (tNorm * (double)1.110223E-16f * (double)n + 2.0 * this.minPivot);
        double left = lower - margin;
        double right = upper + margin;
        for (i = 0; i < maxIter && !((range = right - left) < absoluteTolerance) && !(range < relativeTolerance * Math.max(Math.abs(left), Math.abs(right))); ++i) {
            middle = 0.5 * (left + right);
            if (this.countEigenValues(middle, index, n) >= 1) {
                right = middle;
                continue;
            }
            left = middle;
        }
        lower = Math.max(lower, left - (double)1.110223E-14f * Math.abs(left));
        left = lower - margin;
        right = upper + margin;
        for (i = 0; i < maxIter && !((range = right - left) < absoluteTolerance) && !(range < relativeTolerance * Math.max(Math.abs(left), Math.abs(right))); ++i) {
            middle = 0.5 * (left + right);
            if (this.countEigenValues(middle, index, n) >= n) {
                right = middle;
                continue;
            }
            left = middle;
        }
        upper = Math.min(upper, right + (double)1.110223E-14f * Math.abs(right));
        return new double[]{lower, upper};
    }

    private int countEigenValues(double t, int index, int n) {
        double ratio = this.main[index] - t;
        int count = ratio > 0.0 ? 0 : 1;
        for (int i = 1; i < n; ++i) {
            if (!((ratio = this.main[index + i] - this.squaredSecondary[index + i - 1] / ratio - t) <= 0.0)) continue;
            ++count;
        }
        return count;
    }

    private void ldlTDecomposition(double lambda, int index, int n) {
        double di = this.main[index] - lambda;
        this.work[0] = Math.abs(di);
        for (int i = 1; i < n; ++i) {
            int fourI = 4 * i;
            double eiM1 = this.secondary[index + i - 1];
            double ratio = eiM1 / di;
            this.work[fourI - 2] = ratio * ratio * Math.abs(di);
            di = this.main[index + i] - lambda - eiM1 * ratio;
            this.work[fourI] = Math.abs(di);
        }
    }

    private void dqds(int start, int end) {
        double tmp;
        int j4;
        double d;
        this.eMin = this.work[4 * start + this.pingPong + 4];
        this.dMin = d = this.work[4 * start + this.pingPong] - this.tau;
        this.dMin1 = -this.work[4 * start + this.pingPong];
        if (this.pingPong == 0) {
            for (j4 = 4 * start + 3; j4 <= 4 * (end - 3); j4 += 4) {
                this.work[j4 - 2] = d + this.work[j4 - 1];
                tmp = this.work[j4 + 1] / this.work[j4 - 2];
                d = d * tmp - this.tau;
                this.dMin = Math.min(this.dMin, d);
                this.work[j4] = this.work[j4 - 1] * tmp;
                this.eMin = Math.min(this.work[j4], this.eMin);
            }
        } else {
            for (j4 = 4 * start + 3; j4 <= 4 * (end - 3); j4 += 4) {
                this.work[j4 - 3] = d + this.work[j4];
                tmp = this.work[j4 + 2] / this.work[j4 - 3];
                d = d * tmp - this.tau;
                this.dMin = Math.min(this.dMin, d);
                this.work[j4 - 1] = this.work[j4] * tmp;
                this.eMin = Math.min(this.work[j4 - 1], this.eMin);
            }
        }
        this.dN2 = d;
        this.dMin2 = this.dMin;
        j4 = 4 * (end - 2) - this.pingPong - 1;
        int j4p2 = j4 + 2 * this.pingPong - 1;
        this.work[j4 - 2] = this.dN2 + this.work[j4p2];
        this.work[j4] = this.work[j4p2 + 2] * (this.work[j4p2] / this.work[j4 - 2]);
        this.dN1 = this.work[j4p2 + 2] * (this.dN2 / this.work[j4 - 2]) - this.tau;
        this.dMin1 = this.dMin = Math.min(this.dMin, this.dN1);
        j4p2 = (j4 += 4) + 2 * this.pingPong - 1;
        this.work[j4 - 2] = this.dN1 + this.work[j4p2];
        this.work[j4] = this.work[j4p2 + 2] * (this.work[j4p2] / this.work[j4 - 2]);
        this.dN = this.work[j4p2 + 2] * (this.dN1 / this.work[j4 - 2]) - this.tau;
        this.dMin = Math.min(this.dMin, this.dN);
        this.work[j4 + 2] = this.dN;
        this.work[4 * end - this.pingPong - 1] = this.eMin;
    }

    private void dqd(int start, int end) {
        double tmp;
        double tmp2;
        int j4;
        double d;
        this.eMin = this.work[4 * start + this.pingPong + 4];
        this.dMin = d = this.work[4 * start + this.pingPong];
        if (this.pingPong == 0) {
            for (j4 = 4 * start + 3; j4 < 4 * (end - 3); j4 += 4) {
                this.work[j4 - 2] = d + this.work[j4 - 1];
                if (this.work[j4 - 2] == 0.0) {
                    this.work[j4] = 0.0;
                    this.dMin = d = this.work[j4 + 1];
                    this.eMin = 0.0;
                } else if (Double.MIN_NORMAL * this.work[j4 + 1] < this.work[j4 - 2] && Double.MIN_NORMAL * this.work[j4 - 2] < this.work[j4 + 1]) {
                    tmp2 = this.work[j4 + 1] / this.work[j4 - 2];
                    this.work[j4] = this.work[j4 - 1] * tmp2;
                    d *= tmp2;
                } else {
                    this.work[j4] = this.work[j4 + 1] * (this.work[j4 - 1] / this.work[j4 - 2]);
                    d *= this.work[j4 + 1] / this.work[j4 - 2];
                }
                this.dMin = Math.min(this.dMin, d);
                this.eMin = Math.min(this.eMin, this.work[j4]);
            }
        } else {
            for (j4 = 4 * start + 3; j4 < 4 * (end - 3); j4 += 4) {
                this.work[j4 - 3] = d + this.work[j4];
                if (this.work[j4 - 3] == 0.0) {
                    this.work[j4 - 1] = 0.0;
                    this.dMin = d = this.work[j4 + 2];
                    this.eMin = 0.0;
                } else if (Double.MIN_NORMAL * this.work[j4 + 2] < this.work[j4 - 3] && Double.MIN_NORMAL * this.work[j4 - 3] < this.work[j4 + 2]) {
                    tmp2 = this.work[j4 + 2] / this.work[j4 - 3];
                    this.work[j4 - 1] = this.work[j4] * tmp2;
                    d *= tmp2;
                } else {
                    this.work[j4 - 1] = this.work[j4 + 2] * (this.work[j4] / this.work[j4 - 3]);
                    d *= this.work[j4 + 2] / this.work[j4 - 3];
                }
                this.dMin = Math.min(this.dMin, d);
                this.eMin = Math.min(this.eMin, this.work[j4 - 1]);
            }
        }
        this.dN2 = d;
        this.dMin2 = this.dMin;
        j4 = 4 * (end - 2) - this.pingPong - 1;
        int j4p2 = j4 + 2 * this.pingPong - 1;
        this.work[j4 - 2] = this.dN2 + this.work[j4p2];
        if (this.work[j4 - 2] == 0.0) {
            this.work[j4] = 0.0;
            this.dMin = this.dN1 = this.work[j4p2 + 2];
            this.eMin = 0.0;
        } else if (Double.MIN_NORMAL * this.work[j4p2 + 2] < this.work[j4 - 2] && Double.MIN_NORMAL * this.work[j4 - 2] < this.work[j4p2 + 2]) {
            tmp = this.work[j4p2 + 2] / this.work[j4 - 2];
            this.work[j4] = this.work[j4p2] * tmp;
            this.dN1 = this.dN2 * tmp;
        } else {
            this.work[j4] = this.work[j4p2 + 2] * (this.work[j4p2] / this.work[j4 - 2]);
            this.dN1 = this.work[j4p2 + 2] * (this.dN2 / this.work[j4 - 2]);
        }
        this.dMin1 = this.dMin = Math.min(this.dMin, this.dN1);
        j4p2 = (j4 += 4) + 2 * this.pingPong - 1;
        this.work[j4 - 2] = this.dN1 + this.work[j4p2];
        if (this.work[j4 - 2] == 0.0) {
            this.work[j4] = 0.0;
            this.dMin = this.dN = this.work[j4p2 + 2];
            this.eMin = 0.0;
        } else if (Double.MIN_NORMAL * this.work[j4p2 + 2] < this.work[j4 - 2] && Double.MIN_NORMAL * this.work[j4 - 2] < this.work[j4p2 + 2]) {
            tmp = this.work[j4p2 + 2] / this.work[j4 - 2];
            this.work[j4] = this.work[j4p2] * tmp;
            this.dN = this.dN1 * tmp;
        } else {
            this.work[j4] = this.work[j4p2 + 2] * (this.work[j4p2] / this.work[j4 - 2]);
            this.dN = this.work[j4p2 + 2] * (this.dN1 / this.work[j4 - 2]);
        }
        this.dMin = Math.min(this.dMin, this.dN);
        this.work[j4 + 2] = this.dN;
        this.work[4 * end - this.pingPong - 1] = this.eMin;
    }

    private void computeShiftIncrement(int start, int end, int deflated) {
        double cnst1 = 0.563;
        double cnst2 = 1.01;
        double cnst3 = 1.05;
        if (this.dMin <= 0.0) {
            this.tau = -this.dMin;
            this.tType = -1;
            return;
        }
        int nn = 4 * end + this.pingPong - 1;
        switch (deflated) {
            case 0: {
                if (this.dMin == this.dN || this.dMin == this.dN1) {
                    int np;
                    double gam;
                    double b1 = Math.sqrt(this.work[nn - 3]) * Math.sqrt(this.work[nn - 5]);
                    double b2 = Math.sqrt(this.work[nn - 7]) * Math.sqrt(this.work[nn - 9]);
                    double a2 = this.work[nn - 7] + this.work[nn - 5];
                    if (this.dMin == this.dN && this.dMin1 == this.dN1) {
                        double gap2 = this.dMin2 - a2 - this.dMin2 * 0.25;
                        double gap1 = a2 - this.dN - (gap2 > 0.0 && gap2 > b2 ? b2 / gap2 * b2 : b1 + b2);
                        if (gap1 > 0.0 && gap1 > b1) {
                            this.tau = Math.max(this.dN - b1 / gap1 * b1, 0.5 * this.dMin);
                            this.tType = -2;
                            break;
                        }
                        double s = 0.0;
                        if (this.dN > b1) {
                            s = this.dN - b1;
                        }
                        if (a2 > b1 + b2) {
                            s = Math.min(s, a2 - (b1 + b2));
                        }
                        this.tau = Math.max(s, 0.333 * this.dMin);
                        this.tType = -3;
                        break;
                    }
                    this.tType = -4;
                    double s = 0.25 * this.dMin;
                    if (this.dMin == this.dN) {
                        gam = this.dN;
                        a2 = 0.0;
                        if (this.work[nn - 5] > this.work[nn - 7]) {
                            return;
                        }
                        b2 = this.work[nn - 5] / this.work[nn - 7];
                        np = nn - 9;
                    } else {
                        np = nn - 2 * this.pingPong;
                        b2 = this.work[np - 2];
                        gam = this.dN1;
                        if (this.work[np - 4] > this.work[np - 2]) {
                            return;
                        }
                        a2 = this.work[np - 4] / this.work[np - 2];
                        if (this.work[nn - 9] > this.work[nn - 11]) {
                            return;
                        }
                        b2 = this.work[nn - 9] / this.work[nn - 11];
                        np = nn - 13;
                    }
                    a2 += b2;
                    for (int i4 = np; i4 >= 4 * start + 2 + this.pingPong && b2 != 0.0; i4 -= 4) {
                        b1 = b2;
                        if (this.work[i4] > this.work[i4 - 2]) {
                            return;
                        }
                        if (100.0 * Math.max(b2, b1) < (a2 += (b2 *= this.work[i4] / this.work[i4 - 2])) || 0.563 < a2) break;
                    }
                    if ((a2 = 1.05 * a2) < 0.563) {
                        s = gam * (1.0 - Math.sqrt(a2)) / (1.0 + a2);
                    }
                    this.tau = s;
                    break;
                }
                if (this.dMin == this.dN2) {
                    this.tType = -5;
                    double s = 0.25 * this.dMin;
                    int np = nn - 2 * this.pingPong;
                    double b1 = this.work[np - 2];
                    double b2 = this.work[np - 6];
                    double gam = this.dN2;
                    if (this.work[np - 8] > b2 || this.work[np - 4] > b1) {
                        return;
                    }
                    double a2 = this.work[np - 8] / b2 * (1.0 + this.work[np - 4] / b1);
                    if (end - start > 2) {
                        b2 = this.work[nn - 13] / this.work[nn - 15];
                        a2 += b2;
                        for (int i4 = nn - 17; i4 >= 4 * start + 2 + this.pingPong && b2 != 0.0; i4 -= 4) {
                            b1 = b2;
                            if (this.work[i4] > this.work[i4 - 2]) {
                                return;
                            }
                            if (100.0 * Math.max(b2, b1) < (a2 += (b2 *= this.work[i4] / this.work[i4 - 2])) || 0.563 < a2) break;
                        }
                        a2 = 1.05 * a2;
                    }
                    if (a2 < 0.563) {
                        this.tau = gam * (1.0 - Math.sqrt(a2)) / (1.0 + a2);
                        break;
                    }
                    this.tau = s;
                    break;
                }
                this.g = this.tType == -6 ? (this.g += 0.333 * (1.0 - this.g)) : (this.tType == -18 ? 0.08325 : 0.25);
                this.tau = this.g * this.dMin;
                this.tType = -6;
                break;
            }
            case 1: {
                if (this.dMin1 == this.dN1 && this.dMin2 == this.dN2) {
                    double a2;
                    double gap2;
                    this.tType = -7;
                    double s = 0.333 * this.dMin1;
                    if (this.work[nn - 5] > this.work[nn - 7]) {
                        return;
                    }
                    double b1 = this.work[nn - 5] / this.work[nn - 7];
                    double b2 = b1;
                    if (b2 != 0.0) {
                        for (int i4 = 4 * end - 10 + this.pingPong; i4 >= 4 * start + 2 + this.pingPong; i4 -= 4) {
                            double oldB1 = b1;
                            if (this.work[i4] > this.work[i4 - 2]) {
                                return;
                            }
                            if (100.0 * Math.max(b1, oldB1) < (b2 += (b1 *= this.work[i4] / this.work[i4 - 2]))) break;
                        }
                    }
                    if ((gap2 = 0.5 * this.dMin2 - (a2 = this.dMin1 / (1.0 + (b2 = Math.sqrt(1.05 * b2)) * b2))) > 0.0 && gap2 > b2 * a2) {
                        this.tau = Math.max(s, a2 * (1.0 - 1.01 * a2 * (b2 / gap2) * b2));
                        break;
                    }
                    this.tau = Math.max(s, a2 * (1.0 - 1.01 * b2));
                    this.tType = -8;
                    break;
                }
                this.tau = 0.25 * this.dMin1;
                if (this.dMin1 == this.dN1) {
                    this.tau = 0.5 * this.dMin1;
                }
                this.tType = -9;
                break;
            }
            case 2: {
                if (this.dMin2 == this.dN2 && 2.0 * this.work[nn - 5] < this.work[nn - 7]) {
                    this.tType = -10;
                    double s = 0.333 * this.dMin2;
                    if (this.work[nn - 5] > this.work[nn - 7]) {
                        return;
                    }
                    double b1 = this.work[nn - 5] / this.work[nn - 7];
                    double b2 = b1;
                    if (b2 != 0.0) {
                        for (int i4 = 4 * end - 9 + this.pingPong; i4 >= 4 * start + 2 + this.pingPong; i4 -= 4) {
                            if (this.work[i4] > this.work[i4 - 2]) {
                                return;
                            }
                            if (100.0 * (b1 *= this.work[i4] / this.work[i4 - 2]) < (b2 += b1)) break;
                        }
                    }
                    b2 = Math.sqrt(1.05 * b2);
                    double a2 = this.dMin2 / (1.0 + b2 * b2);
                    double gap2 = this.work[nn - 7] + this.work[nn - 9] - Math.sqrt(this.work[nn - 11]) * Math.sqrt(this.work[nn - 9]) - a2;
                    if (gap2 > 0.0 && gap2 > b2 * a2) {
                        this.tau = Math.max(s, a2 * (1.0 - 1.01 * a2 * (b2 / gap2) * b2));
                        break;
                    }
                    this.tau = Math.max(s, a2 * (1.0 - 1.01 * b2));
                    break;
                }
                this.tau = 0.25 * this.dMin2;
                this.tType = -11;
                break;
            }
            default: {
                this.tau = 0.0;
                this.tType = -12;
            }
        }
    }

    private void updateSigma(double tau) {
        if (tau < this.sigma) {
            this.sigmaLow += tau;
            double t = this.sigma + this.sigmaLow;
            this.sigmaLow -= t - this.sigma;
            this.sigma = t;
        } else {
            double t = this.sigma + tau;
            this.sigmaLow += this.sigma - (t - tau);
            this.sigma = t;
        }
    }

    private void findEigenVectors() {
        int i;
        double di;
        int m = this.main.length;
        this.eigenvectors = new ArrayRealVector[m];
        double[] d = new double[m];
        double[] l = new double[m - 1];
        d[0] = di = this.main[0];
        for (i = 1; i < m; ++i) {
            double eiM1 = this.secondary[i - 1];
            double ratio = eiM1 / di;
            di = this.main[i] - eiM1 * ratio;
            l[i - 1] = ratio;
            d[i] = di;
        }
        for (i = 0; i < m; ++i) {
            this.eigenvectors[i] = this.findEigenvector(this.realEigenvalues[i], d, l);
        }
    }

    private ArrayRealVector findEigenvector(double eigenvalue, double[] d, double[] l) {
        int i;
        int m = this.main.length;
        this.stationaryQuotientDifferenceWithShift(d, l, eigenvalue);
        this.progressiveQuotientDifferenceWithShift(d, l, eigenvalue);
        int r = m - 1;
        double minG = Math.abs(this.work[6 * r] + this.work[6 * r + 3] + eigenvalue);
        int i2 = 0;
        int sixI = 0;
        while (i2 < m - 1) {
            double g = this.work[sixI] + d[i2] * this.work[sixI + 9] / this.work[sixI + 10];
            double absG = Math.abs(g);
            if (absG < minG) {
                r = i2;
                minG = absG;
            }
            ++i2;
            sixI += 6;
        }
        double[] eigenvector = new double[m];
        double n2 = 1.0;
        eigenvector[r] = 1.0;
        double z = 1.0;
        for (i = r - 1; i >= 0; --i) {
            eigenvector[i] = z *= -this.work[6 * i + 2];
            n2 += z * z;
        }
        z = 1.0;
        for (i = r + 1; i < m; ++i) {
            eigenvector[i] = z *= -this.work[6 * i - 1];
            n2 += z * z;
        }
        double inv = 1.0 / Math.sqrt(n2);
        int i3 = 0;
        while (i3 < m) {
            int n = i3++;
            eigenvector[n] = eigenvector[n] * inv;
        }
        return this.transformer == null ? new ArrayRealVector(eigenvector, false) : new ArrayRealVector(this.transformer.getQ().operate(eigenvector), false);
    }

    private void stationaryQuotientDifferenceWithShift(double[] d, double[] l, double lambda) {
        int nM1 = d.length - 1;
        double si = -lambda;
        int i = 0;
        int sixI = 0;
        while (i < nM1) {
            double di = d[i];
            double li = l[i];
            double diP1 = di + si;
            double liP1 = li * di / diP1;
            this.work[sixI] = si;
            this.work[sixI + 1] = diP1;
            this.work[sixI + 2] = liP1;
            si = li * liP1 * si - lambda;
            ++i;
            sixI += 6;
        }
        this.work[6 * nM1 + 1] = d[nM1] + si;
        this.work[6 * nM1] = si;
    }

    private void progressiveQuotientDifferenceWithShift(double[] d, double[] l, double lambda) {
        int nM1 = d.length - 1;
        double pi = d[nM1] - lambda;
        int i = nM1 - 1;
        int sixI = 6 * i;
        while (i >= 0) {
            double di = d[i];
            double li = l[i];
            double diP1 = di * li * li + pi;
            double t = di / diP1;
            this.work[sixI + 9] = pi;
            this.work[sixI + 10] = diP1;
            this.work[sixI + 5] = li * t;
            pi = pi * t - lambda;
            --i;
            sixI -= 6;
        }
        this.work[3] = pi;
        this.work[4] = pi;
    }

    private static class Solver
    implements DecompositionSolver {
        private double[] realEigenvalues;
        private double[] imagEigenvalues;
        private final ArrayRealVector[] eigenvectors;

        private Solver(double[] realEigenvalues, double[] imagEigenvalues, ArrayRealVector[] eigenvectors) {
            this.realEigenvalues = realEigenvalues;
            this.imagEigenvalues = imagEigenvalues;
            this.eigenvectors = eigenvectors;
        }

        public double[] solve(double[] b) throws IllegalArgumentException, InvalidMatrixException {
            if (!this.isNonSingular()) {
                throw new SingularMatrixException();
            }
            int m = this.realEigenvalues.length;
            if (b.length != m) {
                throw MathRuntimeException.createIllegalArgumentException("vector length mismatch: got {0} but expected {1}", b.length, m);
            }
            double[] bp = new double[m];
            for (int i = 0; i < m; ++i) {
                ArrayRealVector v = this.eigenvectors[i];
                double[] vData = v.getDataRef();
                double s = v.dotProduct(b) / this.realEigenvalues[i];
                for (int j = 0; j < m; ++j) {
                    int n = j;
                    bp[n] = bp[n] + s * vData[j];
                }
            }
            return bp;
        }

        public RealVector solve(RealVector b) throws IllegalArgumentException, InvalidMatrixException {
            if (!this.isNonSingular()) {
                throw new SingularMatrixException();
            }
            int m = this.realEigenvalues.length;
            if (b.getDimension() != m) {
                throw MathRuntimeException.createIllegalArgumentException("vector length mismatch: got {0} but expected {1}", b.getDimension(), m);
            }
            double[] bp = new double[m];
            for (int i = 0; i < m; ++i) {
                ArrayRealVector v = this.eigenvectors[i];
                double[] vData = v.getDataRef();
                double s = v.dotProduct(b) / this.realEigenvalues[i];
                for (int j = 0; j < m; ++j) {
                    int n = j;
                    bp[n] = bp[n] + s * vData[j];
                }
            }
            return new ArrayRealVector(bp, false);
        }

        public RealMatrix solve(RealMatrix b) throws IllegalArgumentException, InvalidMatrixException {
            if (!this.isNonSingular()) {
                throw new SingularMatrixException();
            }
            int m = this.realEigenvalues.length;
            if (b.getRowDimension() != m) {
                throw MathRuntimeException.createIllegalArgumentException("dimensions mismatch: got {0}x{1} but expected {2}x{3}", b.getRowDimension(), b.getColumnDimension(), m, "n");
            }
            int nColB = b.getColumnDimension();
            double[][] bp = new double[m][nColB];
            for (int k = 0; k < nColB; ++k) {
                for (int i = 0; i < m; ++i) {
                    int j;
                    ArrayRealVector v = this.eigenvectors[i];
                    double[] vData = v.getDataRef();
                    double s = 0.0;
                    for (j = 0; j < m; ++j) {
                        s += v.getEntry(j) * b.getEntry(j, k);
                    }
                    s /= this.realEigenvalues[i];
                    for (j = 0; j < m; ++j) {
                        double[] dArray = bp[j];
                        int n = k;
                        dArray[n] = dArray[n] + s * vData[j];
                    }
                }
            }
            return MatrixUtils.createRealMatrix(bp);
        }

        public boolean isNonSingular() {
            for (int i = 0; i < this.realEigenvalues.length; ++i) {
                if (this.realEigenvalues[i] != 0.0 || this.imagEigenvalues[i] != 0.0) continue;
                return false;
            }
            return true;
        }

        public RealMatrix getInverse() throws InvalidMatrixException {
            if (!this.isNonSingular()) {
                throw new SingularMatrixException();
            }
            int m = this.realEigenvalues.length;
            double[][] invData = new double[m][m];
            for (int i = 0; i < m; ++i) {
                double[] invI = invData[i];
                for (int j = 0; j < m; ++j) {
                    double invIJ = 0.0;
                    for (int k = 0; k < m; ++k) {
                        double[] vK = this.eigenvectors[k].getDataRef();
                        invIJ += vK[i] * vK[j] / this.realEigenvalues[k];
                    }
                    invI[j] = invIJ;
                }
            }
            return MatrixUtils.createRealMatrix(invData);
        }
    }
}

