package it.eng.rdlab.soa3.connector.utils;

import java.io.File;
import java.io.FileReader;
import java.security.KeyPair;
import java.security.KeyStore;
import java.security.PrivateKey;
import java.security.SecureRandom;
import java.security.Security;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.List;

import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManagerFactory;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.bouncycastle.openssl.PEMReader;

/**
 * 
 * Manages the security features of the connection with the Policy Manager: the class accepts the paths to certificate, private key and truststore directory
 * it also accepts the extension of the trust files.
 * 
 * Default values are:
 * 
 * 	certificate /etc/grid-security/hostcert.pem
 *	private key /etc/grid-security/hostkey.pem
 *	trust store directory /etc/grid-security/certificates
 * 	trust files extension .0
 * 
 * 
 * @author Ciro Formisano (ENG)
 *
 */
public class SecurityManager 
{
	private Log log;
	
	private static SecurityManager instance;
	public static final String DEFAULT_CERT_FILE = "/etc/grid-security/hostcert.pem";
	public static final String DEFAULT_KEY_FILE = "/etc/grid-security/hostkey.pem";
	public static final String DEFAULT_TRUST_DIR = "/etc/grid-security/certificates/";
	public static final String DEFAULT_TRUST_FILE_EXTENSION = ".0";
	
	private String 	certFile,
					keyFile,
					trustDir,
					trustExt;
	
	private boolean invalidateTrustDirectory;
	
	private List<String> trustedCertificates;
	
	
	private SecurityManager ()
	{
		this.log = LogFactory.getLog(this.getClass());
		Security.addProvider(new org.bouncycastle.jce.provider.BouncyCastleProvider());
		certFile = DEFAULT_CERT_FILE;
		keyFile = DEFAULT_KEY_FILE;
		trustDir = DEFAULT_TRUST_DIR;
		trustExt = DEFAULT_TRUST_FILE_EXTENSION;
		this.invalidateTrustDirectory = false;
		this.trustedCertificates = new ArrayList<String>();
	}
	
	
	/**
	 * 
	 * @return a singleton instance
	 */
	public static SecurityManager getInstance ()
	{
		if (instance == null) instance = new SecurityManager();
		
		return instance;
	}
	

	/**
	 * 
	 * Sets the certificate
	 * 
	 * @param certFile the complete path to the certificate PEM file
	 */
	public void setCertFile(String certFile) {
		
		if (certFile != null)  this.certFile = certFile;
	}


	/**
	 * 
	 * Sets the key file
	 * 
	 * @param keyFile the complete path to the key PEM file
	 */
	public void setKeyFile(String keyFile) {
		
		if (keyFile != null)  this.keyFile = keyFile;
	}


	/**
	 * 
	 * Sets the trust directory
	 * 
	 * @param trustDir the path to the trust dir
	 */
	public void setTrustDir(String trustDir) 
	{
		if (trustDir != null)
		{
			if (!trustDir.endsWith("/")) trustDir = trustDir+"/";
			
			this.trustDir = trustDir;
		}
	}


	/**
	 * 
	 * The trust extension
	 * 
	 * @param trustExt the trust files extension
	 */
	public void setTrustExt(String trustExt) 
	{
		if (trustExt != null) this.trustExt = trustExt;

	}


	/**
	 * 
	 * If true is passed, the trust dir won't be taken into account
	 * 
	 * @param invalidate
	 */
	public void invalidateTrustedDir (boolean invalidate)
	{
		this.invalidateTrustDirectory = invalidate;
	}

	/**
	 * 
	 * Adds a single trusted certificate
	 * 
	 * @param file the complete path to the trusted certificate PEM file
	 */
	public void addTrustedCert (String file)
	{
		this.trustedCertificates.add(file);
	}
	
	
	private KeyStore getKeyStore () throws Exception
	{
		FileReader reader = new FileReader(keyFile);
		PEMReader pem = new PEMReader(reader);
	    PrivateKey key = ((KeyPair)pem.readObject()).getPrivate();
	    pem.close();
	    reader.close();

	    // Get the certificate      
	    reader = new FileReader(certFile);
	    pem = new PEMReader(reader);
	    X509Certificate cert = (X509Certificate)pem.readObject();
	    pem.close();
	    reader.close();
	
		KeyStore keystore = KeyStore.getInstance(KeyStore.getDefaultType());
		keystore.load(null);
		keystore.setKeyEntry("certs", key,"changeit".toCharArray(), new java.security.cert.Certificate[]{cert});
		
		List<String> trustedCertFiles = getTrustedCerts();
		this.log.debug ("trusted files");
		
		for (int i=0;i<trustedCertFiles.size();i++)
		{
			this.log.debug (trustedCertFiles.get(i));
		    reader = new FileReader(trustedCertFiles.get(i));
		    pem = new PEMReader(reader);
		    X509Certificate trustCert = (X509Certificate)pem.readObject();
		    pem.close();
		    reader.close();
		    keystore.setCertificateEntry("trust"+i, trustCert);
		}

		return keystore;
	}
	
	private List<String> getTrustedCerts ()
	{
		List<String> response = new ArrayList<String>(this.trustedCertificates);
		
		if (!this.invalidateTrustDirectory)
		{
			this.log.debug("Reading the trust directory");
			File dir = new File(this.trustDir);
			String [] fileNames = dir.list();
			this.log.debug("Files in the trust directory");
			
			for (String fileName : fileNames)
			{
				this.log.debug(fileName);
				if (fileName.endsWith(trustExt)) response.add(this.trustDir+fileName);
			}
		}
		
		return response;
		
	}
	
	/**
	 * 
	 * Loads the configured certificates
	 * 
	 * @throws Exception
	 */
	public void loadCertificate () throws Exception
	{
	
//		KeyStore keystore = KeyStore.getInstance("pkcs12");
//		keystore.load(new FileInputStream("/home/ciro/Varie/host.p12"),"changeit".toCharArray());
		KeyStore keystore = getKeyStore();
		KeyManagerFactory keyFactory = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
		keyFactory.init(keystore,"changeit".toCharArray());
		TrustManagerFactory trustFactory = TrustManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
		trustFactory.init(keystore);
		SSLContext sslContext = SSLContext.getInstance("TLS");
		//sslContext.init(keyFactory.getKeyManagers(),new TrustManager [] {new NullTrustManager(keystore)}, new SecureRandom());
		sslContext.init(keyFactory.getKeyManagers(),trustFactory.getTrustManagers(), new SecureRandom());
		HttpsURLConnection.setDefaultSSLSocketFactory(sslContext.getSocketFactory());
		
	}
	

	
	public static void main(String[] args) throws Exception {
		Security.addProvider(new org.bouncycastle.jce.provider.BouncyCastleProvider());
		SecurityManager.getInstance().loadCertificate();
	}
}
