package org.gcube.security.soa3.connector;

import java.security.Principal;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import javax.security.auth.Subject;
import javax.ws.rs.core.MediaType;
import javax.xml.rpc.handler.MessageContext;

import org.gcube.common.core.contexts.GCUBEServiceContext;
import org.gcube.common.core.security.GCUBESecurityController;
import org.gcube.common.core.security.GCUBEServiceSecurityController;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.security.soa3.cache.SOA3EhcacheWrapper;
import org.gcube.security.soa3.configuration.ConfigurationManager;
import org.gcube.security.soa3.connector.credentials.TicketCredentials;
import org.gcube.security.soa3.connector.engine.RestManager;
import org.gcube.security.soa3.connector.integration.utils.Utils;
import org.globus.wsrf.impl.security.authentication.Constants;
import org.w3c.dom.Element;

/**
 * 
 * Implementation of the security controller for using the SOA3 Security Framework
 * 
 * @author Ciro Formisano (ENG)
 *
 */
public class SOA3SecurityController implements GCUBESecurityController {

	private GCUBELog log;
	private static final String 	DN = "DN";
	private final String AUTHORIZATION_HEADER = "Authorization";
	private String serviceName;
	private String soa3Endpoint;
	private boolean credentialPropagationPolicy;

	private class CredentialsBean 
	{
		public String 	type,
						credentialString;
		
		public boolean propagate = true;
	}
	
	
	public SOA3SecurityController ()
	{
		this.log = new GCUBELog(this);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void init(GCUBEServiceContext serviceContext) 
	{
		this.serviceName = serviceContext.getName();
		log.debug("Initializing security manager for service "+serviceName);
		this.soa3Endpoint = ConfigurationManager.getInstance().getServerUrl(serviceName);
		this.credentialPropagationPolicy = ConfigurationManager.getInstance().getCredentialPropagationPolicy(serviceName);
	}


	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean checkAccess(Map<String, Object> parameters) 
	{
		log.debug("Checking access");
		
		if (!ConfigurationManager.getInstance().isSecurityEnabled(this.serviceName))
		{
			log.debug("Security disabled");
			return true;
		}
		else
		{
			log.debug("Security enabled");
			return applySecurityPolicies(parameters);
		}
	}
	
	/**
	 * 
	 * @param parameters
	 * @return 
	 */
	private boolean applySecurityPolicies (Map<String, Object> parameters)
	{
		log.debug("Checking the acces rights");
		Map<String, String> headers = (Map<String, String>) parameters.get(GCUBEServiceSecurityController.HEADERS);
		String securityHeader = headers.get(Utils.BINARY_SECURITY_TOKEN_LABEL);		
		MessageContext messageContext = (MessageContext) parameters.get(GCUBEServiceSecurityController.MESSAGE_CONTEXT);
		CredentialsBean credentials = getCredentialsBean(messageContext, securityHeader);
		log.debug("Response = "+credentials.credentialString);
		setCredentials(credentials, messageContext);
		return credentials.credentialString != null;
	}
	
	/**
	 * 
	 * @param credentials
	 * @param messageContext
	 */
	private void setCredentials (CredentialsBean credentials,MessageContext messageContext)
	{
		log.debug("Setting credentials in the messageContext");
		
		if (credentials.credentialString != null && credentials.propagate && this.credentialPropagationPolicy)
		{
			log.debug("Setting...");
			
			try 
			{
				log.debug("Generating security header");
				Element element = Utils.generateBinaryTokenElement(TicketCredentials.SES, credentials.credentialString);
				log.debug("Security Header generated");
				messageContext.setProperty(Utils.SECURITY_TOKEN, element);
			} 
			catch (Exception e) 
			{
				log.debug("Unable to generate the security header",e);
			}
		}
		else log.debug("Propagation not set");
	}
	
	/**
	 * 
	 * @param messageContext
	 * @param securityHeader
	 * @return
	 */
	private CredentialsBean getCredentialsBean (MessageContext messageContext, String securityHeader)
	{
		this.log.debug("Get Credentials bean");
		CredentialsBean response = new CredentialsBean();
		
		if (securityHeader != null)
		{
			log.debug("Security Header not null");
			
			try 
			{
				String [] secHeaderElements = securityHeader.split(" ");
				response.type = secHeaderElements [0];
				String value = secHeaderElements [1];
				log.debug("Type = "+response.type);
				log.debug("id = "+value);
				response.credentialString = getAuthenticationString(response.type, value);
				
			} catch (Exception e)
			{
				log.error("Invalid auth header, triyng to find DN");
				response.type = DN;
				response.credentialString = getDn(messageContext);
				response.propagate = false;
				log.debug("DN = "+response.credentialString);
			}
			
//			messageContext.removeProperty(SECURITY_TOKEN);
//			type = securityHeader.getAttribute(SoapMessageManager.ENCODING_TYPE_LABEL);
//			String value = securityHeader.getAttribute(SoapMessageManager.ID_LABEL);

		}
		else 
		{
			log.debug("Security Header null, trying to find DN");
			response.type = DN;
			response.credentialString = getDn(messageContext);
			response.propagate = false;
			log.debug("DN = "+response.credentialString);
			
		}
		
		this.log.debug("Credentials bean generated");
		return response;
	}
	
	
	/**
	 * 
	 * @param messageContext
	 * @return
	 */
	private String getDn (MessageContext messageContext)
	{
		log.debug("No security header found");
		log.debug("Looking for the Distinguished Name");
		String response = null;
		Subject subject = (Subject) messageContext.getProperty(Constants.PEER_SUBJECT);
		
		if (subject == null)
		{
			log.error("No Distinguished name found");
		}
		else
		{
			log.debug("External subject "+subject);
			Set<Principal> principals = subject.getPrincipals();
			
			if (principals == null || principals.isEmpty())
			{
				log.error("Unable to find subject identity");
			}
			else 
			{
				log.debug("Identities found, looking for the DNs");
				Iterator<Principal> principalIterator = principals.iterator();
				
				while (principalIterator.hasNext() && response == null)
				{
					String dn = principalIterator.next().getName();
					log.debug("Distinguished name "+dn);
					response = getAuthenticationString(DN, dn);
					log.debug("Response = "+response);
					
				}

			}

		}
		
		return response;
	}
	
	/**
	 * 
	 * @param type
	 * @param value
	 * @return
	 */
	private String getAuthenticationString (String type, String value)
	{
		log.debug("Asking the cache...");
		String cacheString = type+value;
		log.debug("Cache string "+cacheString);
		String response = SOA3EhcacheWrapper.getInstance().get(cacheString);
		
		if (response == null)
		{
			log.debug("Response null, asking to SOA3");
			response = askSoa3(type, value);
			if (response != null) 
			{
				log.debug("Response found populating the cache");
				SOA3EhcacheWrapper.getInstance().put(cacheString, response);
				log.debug("Cache populated");
			}
		}
		else
		{
			log.debug("Response found in the cache");
		}
		
		log.debug("Response = "+response);
		return response;
	}
	
	/**
	 * 
	 * @param type
	 * @param value
	 * @return
	 */
	private String askSoa3 (String type, String value)
	{
		log.debug("Sending authentication message to SOA3");
		Map<String, String> headers = new HashMap<String, String> ();
		headers.put(AUTHORIZATION_HEADER, type+ " "+value);
		String response = RestManager.getInstance(soa3Endpoint).sendMessage("access", headers, null, MediaType.APPLICATION_JSON_TYPE, MediaType.APPLICATION_JSON_TYPE);
		log.debug("Authentication response = "+response);
		return response;

	}
	
	



}
