package org.gcube.couchbase.helpers;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.commons.lang.StringUtils;
import org.gcube.couchbase.entities.ForwardIndexDocument;
import org.gcube.couchbase.entities.Operator;
import org.gcube.indexmanagement.bdbwrapper.BDBGcqlQueryContainer.SingleTerm;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.couchbase.client.CouchbaseClient;
import com.couchbase.client.protocol.views.ComplexKey;
import com.couchbase.client.protocol.views.Query;
import com.couchbase.client.protocol.views.Stale;
import com.couchbase.client.protocol.views.View;
import com.couchbase.client.protocol.views.ViewResponse;
import com.couchbase.client.protocol.views.ViewResponseReduced;
import com.couchbase.client.protocol.views.ViewRow;
import com.google.gson.Gson;
import com.google.gson.internal.StringMap;

/**
 * 
 * @author Alex Antoniadis
 * 
 */
public class QueryHelper {
	
	private static final Logger logger = LoggerFactory.getLogger(QueryHelper.class);
	private static Gson gson = new Gson();

	private static final String WILDCARD = "*";
	private static final String CONJUCTION = " AND ";
	private static int MAX_RESULTS = 1000;

	/**
	 * Returns a string that is the representations of the terms given but
	 * excludes the fields with the wildcard
	 * 
	 * @param terms
	 * @return
	 */
	public static String pruneQuery(List<SingleTerm> terms) {
		List<String> validTerms = new ArrayList<String>();
		List<String> excludedFields = getExcludedFields(terms);

		logger.info("All terms : ");
		for (SingleTerm t : terms)
			logger.info("\t" + t.getField());
		logger.info("Excluded fields : " + excludedFields);

		for (SingleTerm term : terms) {
			if (excludedFields.contains(term.getField()))
				continue;
			validTerms.add(term.getField() + " " + term.getRelation() + " " + term.getValue());
		}
		return StringUtils.join(validTerms, CONJUCTION);
	}

	/**
	 * Excludes the fields with the wildcards
	 * 
	 * @param terms
	 * @return
	 */
	public static List<String> getExcludedFields(List<SingleTerm> terms) {
		List<String> excludedTerms = new ArrayList<String>();

		for (SingleTerm term : terms) {
			if (term.getValue().equals(WILDCARD))
				excludedTerms.add(term.getField());
		}
		
		excludedTerms.add(ForwardIndexDocument.LANGUAGE_FIELD);
		
		return excludedTerms;
	}

	/**
	 * Adds the fields that may be in searchables with wildcards but not appear
	 * in projections
	 * 
	 * @param queries
	 * @param projections
	 */
	public static void addExcludedFields(List<ArrayList<SingleTerm>> queries, List<String> projections) {
		logger.debug("adding excluded fields in projections");

		logger.debug("projections : " + projections);
		for (List<SingleTerm> query : queries) {
			logger.debug("\tquery : " + query);
			List<String> excludedFields = QueryHelper.getExcludedFields(query);

			logger.debug("\texcludedFields : " + excludedFields);
			for (String field : excludedFields) {
				if (!projections.contains(field)) {
					logger.debug("\t\tfield : " + field + " not in projections");
					projections.add(field);
				}
			}
		}
	}

	/**
	 * Performs a dummy query with stale value FALSE in order to force index to
	 * be recreated
	 * 
	 * @param client
	 * @param designDocumentName
	 * @param viewName
	 */
	public static void dummyQuery(CouchbaseClient client, String designDocumentName, String viewName) {
		logger.info("Forcing update on index : " + viewName);
		View view = client.getView(designDocumentName, viewName);
		Query query = new Query();
		query.setStale(Stale.FALSE);
		client.query(view, query);
	}

	/**
	 * Performs projection on the given documents. Each document is a pair of
	 * key (id) of the doc and the JSON that contains all contents, so we apply
	 * the projection list and get a new collection of the projected documents
	 * only;
	 * 
	 * @param docs
	 * @param projections
	 * @return
	 */
	public static List<Map<String, String>> applyProjection(Map<String, Object> docs, List<String> projections) {
		List<Map<String, String>> projectedDocs = new ArrayList<Map<String, String>>();

		for (Entry<String, Object> doc : docs.entrySet()) {
			String docValue = doc.getValue().toString();

			@SuppressWarnings("unchecked")
			Map<String, StringMap<?>> docMap = gson.fromJson(docValue, Map.class);
			StringMap<?> fieldsMap = docMap.get("fields");

			logger.trace("docID : " + doc.getKey());
			logger.trace("\tfieldsJson : " + fieldsMap);

			Map<String, String> projectedDoc = new HashMap<String, String>();
			for (String projection : projections) {
				Object fieldContent = fieldsMap.get(projection);
				String fieldValue = null;

				if (fieldContent != null) {
					if (fieldContent instanceof Object[])
						fieldValue = StringUtils.join((Object[]) fieldContent);
					else if (fieldContent instanceof ArrayList)
						fieldValue = StringUtils.join((ArrayList<?>) fieldContent, " ");
					else
						fieldValue = fieldContent.toString();
				} else {
					fieldValue = "";
				}

				projectedDoc.put(projection, fieldValue);
			}
			projectedDocs.add(projectedDoc);

		}

		return projectedDocs;
	}

	/**
	 * Returns only the keys and the matching docs (not the document). User
	 * needs to perform a multiget to get the documents
	 * 
	 * @param client
	 * @param designDocumentName
	 * @param viewName
	 * @param startKey
	 * @param endKey
	 * @return
	 */
	public static List<String> queryCouchBase(CouchbaseClient client, String designDocumentName, String viewName,
			Operator op, Object startKey, Object endKey) {
		View view = client.getView(designDocumentName, viewName);

		Query query = QueryHelper.createTwoOperantQuery(op, startKey, endKey);
		ViewResponse vr = client.query(view, query);

		return getViewResponse(vr);
	}

	/**
	 * Returns only the keys and the matching docs (not the document). User
	 * needs to perform a multiget to get the documents
	 * 
	 * @param client
	 * @param designDocumentName
	 * @param viewName
	 * @param op1
	 * @param key1
	 * @param op2
	 * @param key2
	 * @return
	 */
	public static List<String> queryCouchBase(CouchbaseClient client, String designDocumentName, String viewName,
			Operator op1, Object key1, Operator op2, Object key2) {
		View view = client.getView(designDocumentName, viewName);

		Query query = QueryHelper.createTwoOperantQuery(op1, key1, op2, key2);
		ViewResponse vr = client.query(view, query);

		return getViewResponse(vr);
	}

	/**
	 * Returns only the keys and the matching docs (not the document). User
	 * needs to perform a multiget to get the documents
	 * 
	 * @param client
	 * @param designDocumentName
	 * @param viewName
	 * @param op
	 * @param key
	 * @return
	 */
	public static List<String> queryCouchBase(CouchbaseClient client, String designDocumentName, String viewName,
			Operator op, Object key) {
		View view = client.getView(designDocumentName, viewName);

		Query query = QueryHelper.createOneOperantQuery(op, key);
		ViewResponse vr = client.query(view, query);

		return getViewResponse(vr);
	}

	/**
	 * Returns a list of the keys of the matching documents of the ViewResponse
	 * 
	 * @param vr
	 * @return
	 */
	public static List<String> getViewResponse(ViewResponse vr) {
		List<String> ret = new LinkedList<String>();
		Iterator<ViewRow> it = vr.iterator();

		while (it.hasNext()) {
			ViewRow viewRow = it.next();
			//logger.info("Found : " + viewRow.getId() + ", key : " + viewRow.getKey());
			logger.trace("Found : " + viewRow.getId() + ", key : " + viewRow.getKey());
			ret.add(viewRow.getId());
		}
		return ret;
	}

	/**
	 * Returns a document that its key matches the id given
	 * 
	 * @param client
	 * @param id
	 * @return
	 */
	public static Object singleGetCouchBase(CouchbaseClient client, String id) {
		return client.get(id);
	}

	/**
	 * Returns a Map of <key, doc_contents_in_json> for the given collection of
	 * ids. It performs a multiget request. If distinct option is true the ids
	 * are first de-duplicated.
	 * 
	 * @param client
	 * @param ids
	 * @param distinct
	 * @return
	 */
	public static Map<String, Object> multiGetCouchBase(CouchbaseClient client, Collection<String> ids, boolean distinct) {
		return multiGetCouchBase(client, ids, distinct, MAX_RESULTS);
	}

	/**
	 * Returns a Map of <key, doc_contents_in_json> for the given collection of
	 * ids. It performs a multiget request for the first numOfResults ids. If
	 * distinct option is true the ids are first de-duplicated.
	 * 
	 * @param client
	 * @param ids
	 * @param distinct
	 * @return
	 */
	public static Map<String, Object> multiGetCouchBase(CouchbaseClient client, Collection<String> ids,
			boolean distinct, int numOfResults) {
		Collection<String> newIds = null;

		if (distinct)
			newIds = com.google.common.collect.Ordering.natural().greatestOf(new HashSet<String>(ids), numOfResults);
		else
			newIds = com.google.common.collect.Ordering.natural().greatestOf(ids, numOfResults);

		return client.getBulk(newIds);
	}

	/**
	 * Return a Map that contains the Relations per key that are extracted from
	 * the queryString Relations contain the operator and the value for each
	 * key. e.g. queryString = "x < 1 AND 1 < y < 5" will give { x , <LESS 1> },
	 * { y, (<GREATER, 1>, <LESS, 5>) }
	 * 
	 * @param queryString
	 * @return
	 * @throws Exception
	 */
	public static Map<String, List<Relation>> getKeys(String queryString) throws Exception {
		Map<String, List<Relation>> keys = new HashMap<String, List<Relation>>();

		String[] subStrings = queryString.split("[aA][nN][dD]");
		for (String subString : subStrings) {
			subString = subString.trim();
			logger.info("subString : " + subString);

			String[] terms = subString.split("\\s");
			logger.info("terms size : " + terms.length);
			for (String t : terms) {
				logger.info("\t" + t);
			}

			if (terms.length == 3) {
				String field = terms[0];
				String operator = terms[1];
				String value = terms[2];

				Relation r = new Relation(operator, value);

				List<Relation> rels = keys.get(field);// keys.containsKey(field)
														// ? keys.get(field) :
														// new
														// ArrayList<Relation>();
				if (rels == null) {
					rels = new ArrayList<Relation>();
					keys.put(field, rels);
				}

				rels.add(r);
			} else if (terms.length == 5) {
				String valueLeft = terms[0];
				String operatorLeft = terms[1];
				String field = terms[2];
				String operatorRight = terms[3];
				String valueRight = terms[4];

				Relation relLeft = new Relation(Operator.flipOperator(operatorLeft), valueLeft);
				Relation relRight = new Relation(operatorRight, valueRight);

				List<Relation> rels = keys.get(field);
				if (rels == null) {
					rels = new ArrayList<Relation>();
					keys.put(field, rels);
				}

				rels.add(relLeft);
				rels.add(relRight);
			} else {
				throw new Exception("relation : " + subString
						+ " cannot be parsed. Usually a relation has 3 or 5 terms. This one has : " + terms.length);
			}
		}
		return keys;
	}

	/**
	 * Performs a validation on a Map that contains the Relations per key. If a
	 * key has 2 relations then it should be in the form 1 < x < 5 and not 1 = x
	 * < 9. Only LESS-GREATER and LESS_THAN-GREATER_THAN can be combined so far.
	 * 
	 * @param keys
	 * @return
	 * @throws Exception
	 */
	public static Map<String, List<Relation>> validateAndSimplifyKeys(Map<String, List<Relation>> keys)
			throws Exception {
		Map<String, List<Relation>> newKeys = new HashMap<String, List<Relation>>();

		for (Entry<String, List<Relation>> e : keys.entrySet()) {
			String field = e.getKey();
			List<Relation> rels = e.getValue();
			if (rels.size() == 1) {
				newKeys.put(field, rels);
			} else if (rels.size() == 2) {
				Relation left = null;
				Relation right = null;

				for (Relation rel : rels)
					if (rel.operator.equals(Operator.LESS_EQUAL) || rel.operator.equals(Operator.LESS_THAN))
						left = rel;
					else if (rel.operator.equals(Operator.GREATER_EQUAL) || rel.operator.equals(Operator.GREATER_THAN))
						right = rel;

				if (left == null || right == null)
					throw new Exception("Error simplifying relations : " + rels);

				newKeys.put(field, Arrays.asList(left, right));
			} else
				throw new Exception(
						"Sympifying is not implementing yet. The following relation contains more than 2 relations : "
								+ rels);
		}

		return newKeys;
	}

	/**
	 * Returns all the ids that match the given queryString in a list. Since the
	 * queryString can have only conjunctions the result is the intersection of
	 * the query of each relation separately.
	 * 
	 * @param client
	 * @param bucketName
	 * @param designDocumentName
	 * @param keys
	 * @param queryString
	 * @return
	 * @throws Exception
	 */
	public static List<String> queryString(CouchbaseClient client, String bucketName, String designDocumentName,
			Map<String, CouchBaseDataTypesHelper.DataType> keys, String queryString) throws Exception {
		Map<String, List<Relation>> relations = getKeys(queryString);
		return queryRelations(client, bucketName, designDocumentName, relations, keys);
	}

	/**
	 * Returns all the ids that match the given Relations in a list. Since the
	 * Relations can have only conjunctions the result is the intersection of
	 * the query of each relation separately.
	 * 
	 * @param client
	 * @param bucketName
	 * @param designDocumentName
	 * @param keys
	 * @param queryString
	 * @return
	 * @throws Exception
	 */
	public static List<String> queryRelations(CouchbaseClient client, String bucketName, String designDocumentName,
			Map<String, List<Relation>> relations, Map<String, CouchBaseDataTypesHelper.DataType> keys)
			throws Exception {
		List<String> docIDs = new LinkedList<String>();

		// Since the relations are conjunctions each loop should be intersected
		// with the current result
		boolean first = true;
		for (Entry<String, List<Relation>> relation : relations.entrySet()) {
			// TODO: check if any query count returns 0 so that the result is
			// the empty set

			String field = relation.getKey();
			logger.info("querying on field : " + field + " started");
			
			String viewName = ViewHelper.constructViewName(bucketName, field, keys);

			List<String> queryResult = null;
			List<Relation> rels = relation.getValue();

			if (rels.size() == 1) {
				Operator op = rels.get(0).operator;
				Object value = CouchBaseDataTypesHelper.convertToObject(rels.get(0).value, keys.get(field));

				queryResult = queryCouchBase(client, designDocumentName, viewName, op, value);
			} else if (rels.size() == 2) {
				Operator opLeft = rels.get(0).operator;
				Object valueLeft = CouchBaseDataTypesHelper.convertToObject(rels.get(0).value, keys.get(field));

				Operator opRight = rels.get(1).operator;
				Object valueRight = CouchBaseDataTypesHelper.convertToObject(rels.get(1).value, keys.get(field));

				queryResult = queryCouchBase(client, designDocumentName, viewName, opLeft, valueLeft, opRight,
						valueRight);
			} else {
				throw new Exception("Error in relations : " + rels);
			}

			logger.info("querying on field : " + field + " ended");
			if (first) {
				docIDs.addAll(queryResult);
				first = false;
			} else {
				docIDs.retainAll(queryResult);
			}
		}

		return docIDs;
	}

	/**
	 * Returns the number of keys for each field. No distinction is made for the
	 * key-value
	 * 
	 * @param client
	 * @param bucketName
	 * @param designDocumentName
	 * @param fields
	 * @param keys
	 * @return
	 * @throws Exception
	 */
	public static Long keyCount(CouchbaseClient client, String bucketName, String designDocumentName,
			Set<String> fields, Map<String, CouchBaseDataTypesHelper.DataType> keys) throws Exception {
		Long count = Long.valueOf(0);

		// Since the relations are conjuctions each loop should be intersected
		// with the current result
		for (String field : fields)
			count += queryCountCouchBase(client, bucketName, designDocumentName, field, keys);

		return count;
	}

	/**
	 * Returns the number of keys for each field. No distinction is made for the
	 * key-value
	 * 
	 * @param client
	 * @param bucketName
	 * @param designDocumentName
	 * @param fieldName
	 * @param keys
	 * @return
	 * @throws Exception
	 */
	public static Long queryCountCouchBase(CouchbaseClient client, String bucketName, String designDocumentName,
			String fieldName, Map<String, CouchBaseDataTypesHelper.DataType> keys) throws Exception {
		String viewName = ViewHelper.constructViewName(bucketName, fieldName, keys);
		return queryCountCouchBase(client, designDocumentName, viewName);
	}

	/**
	 * Returns the number of keys for each field. No distinction is made for the
	 * key-value
	 * 
	 * @param client
	 * @param designDocumentName
	 * @param viewName
	 * @return
	 * @throws Exception
	 */
	public static Long queryCountCouchBase(CouchbaseClient client, String designDocumentName, String viewName)
			throws Exception {
		View view = client.getView(designDocumentName, viewName);
		Query query = new Query();
		ViewResponseReduced vrr = (ViewResponseReduced) client.query(view, query);

		Long count = null;
		int rows = 0;
		Iterator<ViewRow> it = vrr.iterator();

		while (it.hasNext()) {
			ViewRow viewRow = it.next();
			count = Long.valueOf(viewRow.getValue());
			rows++;
			if (rows > 1)
				throw new Exception("reduce returned more than 1 results");
		}
		return count;
	}

	/**
	 * Creates a 2-operant Query from the given operators and values
	 * 
	 * @param op1
	 * @param value1
	 * @param op2
	 * @param value2
	 * @return
	 */
	public static Query createTwoOperantQuery(Operator op1, Object value1, Operator op2, Object value2) {
		Query query = new Query();

		query = createOneOperantQuery(query, op1, value1);
		query = createOneOperantQuery(query, op2, value2);

		return query;
	}

	/**
	 * Creates a 2-operant Query from the given operators and values
	 * 
	 * @param op
	 * @param value1
	 * @param value2
	 * @return
	 */
	public static Query createTwoOperantQuery(Operator op, Object value1, Object value2) {
		Query query = new Query();
		query.setReduce(false);

		ComplexKey key1 = ComplexKey.of(value1);
		ComplexKey key2 = ComplexKey.of(value2);

		switch (op) {
		case GREATER_THAN_AND_LESS_THAN:
			query.setRange(key1, key2);
			query.setInclusiveEnd(false);
			break;

		case GREATER_EQUAL_AND_LESS_EQUAL:
			query.setRange(key1, key2);
			// TODO: skip the equals
			break;

		default:
			break;
		}

		return query;
	}

	/**
	 * Creates a 1-operant Query from the given operator and value
	 * 
	 * @param op
	 * @param value
	 * @return
	 */
	public static Query createOneOperantQuery(Operator op, Object value) {
		Query query = new Query();
		return createOneOperantQuery(query, op, value);
	}

	/**
	 * Creates a 1-operant Query from the given operator and value
	 * 
	 * @param query
	 * @param op
	 * @param value
	 * @return
	 */
	public static Query createOneOperantQuery(Query query, Operator op, Object value) {
		query.setReduce(false);

		ComplexKey key = ComplexKey.of(value);

		switch (op) {
		case EQUAL:
			query.setKey(key);
			break;

		case GREATER_THAN:
			query.setRangeStart(key);
			// TODO: skip the equals
			break;

		case GREATER_EQUAL:
			query.setRangeStart(key);
			break;

		case LESS_THAN:
			query.setRangeEnd(key);
			query.setInclusiveEnd(false);
			break;

		case LESS_EQUAL:
			query.setRangeEnd(key);
			query.setInclusiveEnd(true);
			break;

		default:
			break;
		}

		return query;
	}

	@SuppressWarnings("unused")
	private static void printQuery(ViewResponse vr) {
		System.out.println("Query return " + vr.size() + " results");

		int i = 1;
		Iterator<ViewRow> it = vr.iterator();

		while (it.hasNext()) {
			ViewRow viewRow = it.next();
			System.out.println(i + " #. " + viewRow.getKey() + " : " + viewRow.getValue());
			i++;
		}
	}

	public static void printQueryDate(ViewResponse vr) {
		System.out.println("Query return " + vr.size() + " results");

		int i = 1;
		Iterator<ViewRow> it = vr.iterator();

		while (it.hasNext()) {
			ViewRow viewRow = it.next();
			if (viewRow.getKey() == null || viewRow.getKey().equalsIgnoreCase("null"))
				continue;

			logger.info(i + " #. " + DateHelper.toCalendarDateString(Long.parseLong(viewRow.getKey())) + " : "
					+ viewRow.getValue());
			i++;
		}
	}

	public static Map<String, String> getViewResponseKeyValues(ViewResponse vr) {
		Map<String, String> ret = new HashMap<String, String>();
		Iterator<ViewRow> it = vr.iterator();

		while (it.hasNext()) {
			ViewRow viewRow = it.next();
			logger.info("Found : " + viewRow.getId() + ", key : " + viewRow.getKey());
			ret.put(viewRow.getId(), viewRow.getKey());
		}
		return ret;
	}

	public static void main(String[] args) throws Exception {
		Map<String, List<Relation>> keys = getKeys("a <= 1   AND   b >= 2 and b <= 9 And a >= 0 AND 1 < x < 5");

		System.out.println(keys);

		System.out.println(validateAndSimplifyKeys(keys));
	}
}

class Relation implements Serializable {
	private static final long serialVersionUID = 1L;

	Operator operator;
	String value;

	public Relation(String operator, String value) throws Exception {
		this.operator = Operator.getOperatorFromString(operator);
		if (this.operator == null)
			throw new Exception("Unknown operator : " + operator);
		this.value = value;
	}

	@Override
	public String toString() {
		return "Relation [operator=" + operator + ", value=" + value + "]";
	}

	/*
	 * public static Triple getDominant(Triple t1, Triple t2){ if
	 * (!t1.leftOperant.equalsIgnoreCase(t2.leftOperant)) return null;
	 * 
	 * if (t1.operator.equals(Operator.EQUAL)){ if
	 * (t2.operator.equals(Operator.EQUAL)){ if
	 * (t1.rightOperant.equals(t2.rightOperant)) return t1; } }
	 */
	/*
	 * public static Triple getCombo(Triple t1, Triple t2) { if
	 * (!t1.leftOperant.equalsIgnoreCase(t2.leftOperant)) return null; if
	 * (t1.equals(Operator.LESS_THAN)) }
	 */

}
