package elasticsearchindex.components;
/*
 * IndexField.java
 *
 * $Author: tsakas $
 * $Date: 2007/12/20 14:37:39 $
 * $Id: IndexField.java,v 1.1 2007/12/20 14:37:39 tsakas Exp $
 *
 * <pre>
 *             Copyright (c) : 2006 Fast Search & Transfer ASA
 *                             ALL RIGHTS RESERVED
 * </pre>
 */



import java.util.ArrayList;

/**
 * Represent the characteristics of a single field in a RowSet.
 * 
 * @see FullTextIndexType
 */
public class IndexField {

    /** The name of the field */
    public String name;

    /** An indicator of whether to index the field */
    public boolean index;

    /** An indicator of whether to tokenize the field */
    public boolean tokenize;

    /** An indicator of whether to store the field */
    public boolean store;

    /** An indicator of whether to return the field (in the results from a query) */
    public boolean returned;

    /** An indicator of whether to sort the field */
    public boolean sort;

    /** The boost value to use for this field during ranking */
    public float boost = 1.0f;

    /** The child fields of this field */
    public ArrayList childrenFields = null;

    /**
     * Empty constructor. The field attributes must be set in subsequent calls.
     */
    public IndexField() {
        this.childrenFields = new ArrayList();
    }

    /**
     * Constructor setting all field attributes.
     * 
     * @param name -
     *            The name of the new field
     * @param index -
     *            An indicator of whether to index the new field
     * @param tokenize -
     *            An indicator of whether to tokenize the new field
     * @param store -
     *            An indicator of whether to store the new field
     * @param returned -
     *            An indicator of whether to return the new field (in the
     *            results from a query)
     * @param sort -
     *            An indicator of whether to sort the new field
     * @param boost -
     *            The boost value to use for the new field during ranking
     */
    public IndexField(String name, boolean index, boolean tokenize,
            boolean store, boolean returned, boolean sort, float boost) {
        this.name = name;
        this.index = index;
        this.tokenize = tokenize;
        this.store = store;
        this.returned = returned;
        this.sort = sort;
        this.boost = boost;
        this.childrenFields = new ArrayList();
    }

    /**
     * Adds a <code>IndexField</code> sub field to this field
     * 
     * @param field
     *            <code>IndexField</code> - the sub field to be added
     */
    public void addChildField(IndexField field) {
        this.childrenFields.add(field);
    }

    /**
     * {@inheritDoc}
     */
    public String toString() {
        return toString(0);
    }

    /**
     * A recursive version of toString used to format the output properly.
     * 
     * @param indent -
     *            the number of characters to indent the line
     * @return a String representation of the field
     */
    private String toString(int indent) {
        String returnString = "\n";
        for (int i = 0; i < indent; i++)
            returnString += " ";
        returnString += name + "( indexed=" + index + ", tokenized=" + tokenize
                + ", store=" + store + ", return=" + returned + ", sort="
                + sort + ", boost=" + boost + " )";
        for (int i = 0; i < childrenFields.size(); i++) {
            returnString += ((IndexField) childrenFields.get(i))
                    .toString(indent + 3);
        }
        return returnString;
    }
}
