package elasticsearchindex.components;
/*
 * XMLProfileParser.java
 *
 * $Author: spiros $
 * $Date: 2008/01/09 16:35:22 $
 * $Id: XMLProfileParser.java,v 1.3 2008/01/09 16:35:22 spiros Exp $
 *
 * <pre>
 *             Copyright (c) : 2006 Fast Search & Transfer ASA
 *                             ALL RIGHTS RESERVED
 * </pre>
 */



import java.io.InputStream;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;

import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.apache.xerces.dom.TextImpl;
import org.apache.xerces.parsers.DOMParser;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.w3c.dom.Document;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;

/**
 * Provides various XML read only functions for the indexing service.
 * 
 * @see FullTextIndexType
 */
public class XMLProfileParser {
    /** Log4j logger */
    private static GCUBELog logger = new GCUBELog(XMLProfileParser.class);

    /** The DOM parser this parser builds on */
    private DOMParser parser;

    /** The parsed DOM Document */
    private Document doc;

    /** A list of the child or sibling nodes of the current node */
    private NodeList children = null;

    /**
     * The root node of the document, or of the parts of the document which is
     * of interest
     */
    private Node rootNode;

    /** The node reflecting the users current node of interest */
    private Node currentNode = null;

    /**
     * A counter to keep track of the current nodes place in the current
     * "children" NodeList
     */
    int childNextIndex = -1;

    /**
     * Creates a parser instance.
     */
    public XMLProfileParser() {
        parser = new DOMParser();
    }

    /**
     * Sets the current document to the parsed XML string.
     * @param XMLString - the XML document.
     * @param schemaLocation - The loactaion of the schema.
     * @throws IndexException - in case if failure
     */
    public void readString(String XMLString, String schemaLocation)
            throws Exception {
        if (schemaLocation != null && !schemaLocation.equals("")) {
            String propId = "http://apache.org/xml/properties"
                    + "/schema/external-noNamespaceSchemaLocation";
            parser.setFeature(
                    "http://apache.org/xml/features/validation/schema", true);
            parser.setProperty(propId, schemaLocation);
        } else {
            parser.setFeature(
                    "http://apache.org/xml/features/validation/schema", false);
        }

        parser.parse(new InputSource(new StringReader(XMLString)));
        doc = parser.getDocument();
    }

    /**
     * Sets the current document to the parsed file.
     * @param filename - the name of the file to parse.
     * @param schemaLocation - the location of the schema
     * @throws Exception - in case of failure.
     */
    public void readDoc(String filename, String schemaLocation)
            throws Exception {
        if (schemaLocation != null && !schemaLocation.equals("")) {
            String propId = "http://apache.org/xml/properties"
                    + "/schema/external-noNamespaceSchemaLocation";
            parser.setFeature(
                    "http://apache.org/xml/features/validation/schema", true);
            parser.setProperty(propId, schemaLocation);
        } else {
            parser.setFeature(
                    "http://apache.org/xml/features/validation/schema", false);
        }

        parser.parse(filename);
        doc = parser.getDocument();
    }

    /**
     * Sets the current document to the parsed InputStream.
     * @param is - the input stream to parse.
     * @param schemaLocation - the location of the schema.
     * @throws Exception - in case of failure.
     */
    public void readInputStream(InputStream is, String schemaLocation)
            throws Exception {
        if (schemaLocation != null && !schemaLocation.equals("")) {
            String propId = "http://apache.org/xml/properties"
                    + "/schema/external-noNamespaceSchemaLocation";
            parser.setFeature(
                    "http://apache.org/xml/features/validation/schema", true);
            parser.setProperty(propId, schemaLocation);
        } else {
            parser.setFeature(
                    "http://apache.org/xml/features/validation/schema", false);
        }

        parser.parse(new InputSource(is));
        doc = parser.getDocument();
    }

    /**
     * Retreives the #text element of the first instance of
     * <code>fieldname</code>.
     * 
     * @param fieldname
     *            name of the element to retreive
     * @return the #text element
     * @throws Exception - in case of failure.
     * @see FullTextIndexType
     */
    public String getFieldText(String fieldname) throws Exception {
        NodeList nodes = doc.getElementsByTagName(fieldname);
        Node n = nodes.item(0); // get first instance
        if (n == null) {
            return ("error: field not found");
        }
        nodes = n.getChildNodes();
        for (int i = 0; i < nodes.getLength(); i++) {
            Node nn = nodes.item(i);
            if (nn.getNodeType() == Node.TEXT_NODE) {
                return ((TextImpl) nn).getData().trim();
            }
        }
        return "error: no TEXT_NODE found";
    }

    /**
     * Retreives the #text elements of all instances of <code>fieldname</code>.
     * 
     * @param fieldname
     *            name of the element to retreive
     * @return the #text element
     * @throws Exception - in case of failure.
     * @see FullTextIndexType
     */
    public String[] getFieldTextArray(String fieldname) throws Exception {
        NodeList nodes = doc.getElementsByTagName(fieldname);
        ArrayList<String> nodeList = new ArrayList<String>();
        Node n = nodes.item(0); // get first instance
        if (n == null) {
            return null;
        }
        nodes = n.getChildNodes();
        for (int i = 0; i < nodes.getLength(); i++) {
            Node nn = nodes.item(i);
            if (nn.getNodeType() == Node.TEXT_NODE) {
                nodeList.add(((TextImpl) nn).getData().trim());
            }
        }
        return nodeList.toArray(new String[nodeList.size()]);
    }

    /**
     * Recursivly traverses the XML document printing elements to System.out.
     * @throws Exception - in case of failure.
     */
    private void printTree() throws Exception {
        Node n = doc.getDocumentElement();
        iterate(n, "");
    }

    /**
     * The recursive part of printTree
     * 
     * @param n -
     *            the node to print information about
     * @param s -
     *            indent
     */
    private void iterate(Node n, String s) {
        NodeList nodes = n.getChildNodes();
        for (int i = 0; i < nodes.getLength(); i++) {
            Node nn = nodes.item(i);

            if (nn.getNodeType() == Node.TEXT_NODE) {
                TextImpl textImpl = (TextImpl) nn;
                String tmpString = textImpl.getData().trim();
                if (tmpString.length() > 0) {
                    logger.info(s + "data: " + tmpString);
                }
            } else {
                logger.info(s + "*** " + nn.getNodeName() + " ***");
                NamedNodeMap map = nn.getAttributes();
                if (map.getNamedItem("name") != null) {
                    logger.info(s + map.getNamedItem("name"));
                }
                if (map.getNamedItem("value") != null) {
                    logger.info(s + map.getNamedItem("value"));
                }
                iterate(nn, s + "  ");
            }
        }
    }

    /**
     * Sets the Field Node to the first child of the first instance of the field
     * matching the argument. Following calls to <code>
     * setNextField()</code>
     * and <code>getFieldByValue(String
     * elementName)</code> will give 2nd
     * layerd text.
     * 
     * @return int nr of child elements, #text elements excluded.
     * @throws Exception - in case of failure.
     */
    public int setRootNode(String fieldname) throws Exception {
        NodeList nodes = doc.getElementsByTagName(fieldname);
        rootNode = nodes.item(0); // get root instance
        if (rootNode == null) {
            // logger.error("getFieldNodes: field not found");
            return -1;
        }
        children = rootNode.getChildNodes();
        childNextIndex = 0;
        currentNode = rootNode;
        return children.getLength() / 2; // ignore #text elements on this
                                            // level
    }

    /**
     * Counts the number of descendant elements with a specified element name
     * 
     * @param elementName
     *            <code>String</code> - the name of the elements to count
     * @return <code>int</code> - the number of descendants with the specified
     *         elementName
     */
    public int countDescendants(String elementName) {
        return countDescendants(rootNode, elementName);
    }

    /**
     * Counts the number of descendant elements of a specified root element with
     * a specified element name
     * 
     * @param rootNode -
     *            the node to count the descendents of
     * @param elementName
     *            <code>String</code> - the name of the elements to count
     * @return
     */
    private int countDescendants(Node rootNode, String elementName) {
        NodeList list = rootNode.getChildNodes();
        int count = 0;
        for (int i = 0; i < list.getLength(); i++) {
            String name = list.item(i).getLocalName();
            if (elementName.equals(name))
                count++;
            count += countDescendants(list.item(i), elementName);
        }
        return count;
    }

    /**
     * Counts the descendants of a specific node
     * 
     * @param rootNode -
     *            the node to count the descendants of
     * @return the number of descendants of the specified node
     */
    private int countDescendants(Node rootNode) {
        NodeList list = rootNode.getChildNodes();
        int count = list.getLength();
        for (int i = 0; i < list.getLength(); i++) {
            count += countDescendants(list.item(i));
        }
        return count;
    }

    /**
     * Sets current node to next node.
     * 
     * @return true if the node was present
     * @see #setRootNode(String fieldname)
     * @throws Exception - in case of failure.     
     */
    public boolean setNextField() throws Exception {
        if (children == null) {
            return false;
        }
        if (childNextIndex < children.getLength()) {
            currentNode = children.item(childNextIndex++);
            if (currentNode.getNodeType() == Node.TEXT_NODE) { // if text,
                // ignore
                if (childNextIndex < children.getLength()) {
                    currentNode = children.item(childNextIndex++);
                } else {
                    return false;
                }
            }
        } else {
            return false;
        }
        return true;
    }

    /**
     * Sets the current node to the first child (of the current node) element
     * with a specified element name
     * 
     * @param elementName
     *            <code>String</code> - the name of the element to go to
     * @return <code>boolean</code> - true if the action was successful
     */
    public boolean goChildElement(String elementName) {
        // logger.info("------> goChild called");
        Node tempNode = getChildElement(currentNode, elementName);
        if (tempNode != null) {
            currentNode = tempNode;
            return true;
        } else
            return false;
    }

    /**
     * returns the first child element of the specified parentNode with a
     * specified element name
     * 
     * @param parentNode -
     *            the node of which to look for children
     * @param childElementName -
     *            the ElementName of the child node to return
     * @return the requested child element, or null if none were found
     */
    private Node getChildElement(Node parentNode, String childElementName) {
        NodeList currentChildren = parentNode.getChildNodes();
        Node returnNode = null;
        if (currentChildren == null)
            return null;
        for (int i = 0; i < currentChildren.getLength(); i++) {
            if (childElementName.equals(currentChildren.item(i).getLocalName())) {
                return currentChildren.item(i);
            }// ie: nodes bellow non-fitting nodes are regarded as on the
                // same level
            if ((returnNode = getChildElement(currentChildren.item(i),
                    childElementName)) != null)
                return returnNode;
        }
        return null;
    }

    /**
     * Sets the current node to the first sibling (of the current node) element
     * with a specified element name
     * 
     * @param elementName
     *            <code>String</code> - the name of the element to go to
     * @return <code>boolean</code> - true if the action was successful
     */
    public boolean goSiblingElement(String elementName) {
        // logger.info("------> goSibling called");
        Node tempNode = getSiblingElement(currentNode, elementName);
        if (tempNode != null) {
            currentNode = tempNode;
            return true;
        } else
            return false;
    }

    /**
     * returns the first sibling of the specified parentNode element with a
     * specified element name
     * 
     * @param node -
     *            the node of which to look for siblings
     * @param siblingElementName -
     *            the ElementName of the sibling node to return
     * @return the requested sibling element, or null if none were found
     */
    private Node getSiblingElement(Node node, String siblingElementName) {
        Node siblingNode = node.getNextSibling();
        if (siblingNode == null)
            return null;
        if (siblingElementName.equals(siblingNode.getLocalName())) {
            return siblingNode;
        }
        return getSiblingElement(siblingNode, siblingElementName);
    }

    /**
     * Sets the current node to the first ascendant (of the current node)
     * element with a specified element name
     * 
     * @param elementName
     *            <code>String</code> - the name of the element to go to
     * @return <code>boolean</code> - true if the action was successful
     */
    public boolean goParentElement(String elementName) {
        // logger.info("------> goParent called");
        Node tempNode = getParentElement(currentNode, elementName);
        if (tempNode != null) {
            currentNode = tempNode;
            return true;
        } else
            return false;
    }

    /**
     * returns the first ascendant of the specified child node element with a
     * specified element name
     * 
     * @param child -
     *            the node of which to look for siblings
     * @param parentElementName -
     *            the ElementName of the sibling node to return
     * @return the requested ascendant element, or null if none were found
     */
    private Node getParentElement(Node child, String parentElementName) {
        Node parentNode = child.getParentNode();
        if (parentNode == null || parentNode == rootNode)
            return null;
        if (parentElementName.equals(parentNode.getLocalName())) {
            return parentNode;
        }
        return getParentElement(parentNode, parentElementName);
    }

    /**
     * Determines whether the current node has one or more descendant elements
     * with a specified element name
     * 
     * @param elementName
     *            <code>String</code> - the element name to search for
     * @return <code>boolean</code> - true if one or more descendants were
     *         found
     */
    public boolean hasDescendant(String elementName) {
        return (hasDescendant(currentNode, elementName));
    }

    /**
     * Determines whether the specified node has one or more descendant elements
     * with a specified element name
     * 
     * @param parentNode -
     *            the node for which to look for descendants
     * @param childElementName -
     *            the element name to search for
     * @return - true if one or more descendants were found
     */
    private boolean hasDescendant(Node parentNode, String childElementName) {
        NodeList currentChildren = parentNode.getChildNodes();
        if (currentChildren == null)
            return false;
        for (int i = 0; i < currentChildren.getLength(); i++) {
            if (childElementName.equals(currentChildren.item(i).getLocalName()))
                return true;
            if (hasDescendant(currentChildren.item(i), childElementName))
                return true;
        }
        return false;
    }

    /**
     * Determines whether the current node has one or more sibling elements with
     * a specified element name
     * 
     * @param elementName
     *            <code>String</code> - the element name to search for
     * @return <code>boolean</code> - true if one or more siblings were found
     */
    public boolean hasSibling(String elementName) {
        return currentNode.getNextSibling() != null;
    }

    /**
     * Gets the #text element of the specified element in the current field.
     * 
     * @param elementName
     *            element identifier
     * @return the #text element
     * @throws Exception - in case of failure.
     * @see #setRootNode(String fieldname)
     * @see #setNextField()
     */
    public String getFieldByValue(String elementName) throws Exception {
        if (currentNode.getNodeType() == Node.TEXT_NODE) {
            //logger.info("(" + ((TextImpl) currentNode).getData().trim() + ")");
            // logger.error("getFieldByValue got a Text Node");
        } else {

            NamedNodeMap attributes = currentNode.getAttributes();
            if (attributes != null) {
                Node attr = attributes.getNamedItem(elementName);
                if (attr != null) {
                    // logger.info("name attribute: " +
                    // attr.getNodeValue() );
                    return attr.getNodeValue();
                }
            }

            // logger.info("rootnavn: " +childNode.getNodeName());
            NodeList nodes = currentNode.getChildNodes();
            for (int i = 0; i < nodes.getLength(); i++) {
                Node nn = nodes.item(i);
                // logger.info("navn: " +nn.getNodeName());
                if (nn.getNodeName().equals(elementName)) {
                    // logger.info( nn.getNodeName() );
                    /*
                	nodes = nn.getChildNodes();
                    for (int ii = 0; ii < nodes.getLength(); ii++) {
                        nn = nodes.item(ii);
                        if (nn.getNodeType() == Node.TEXT_NODE) {
                            return ((TextImpl) nn).getData();
                        }
                    }
                    */
                	
                	/* Transform the whole field to a string */
        	        TransformerFactory tFactory = TransformerFactory.newInstance();
        	        Transformer transformer = tFactory.newTransformer();
        	        transformer.setOutputProperty("omit-xml-declaration", "yes");
        	        StringWriter sw = new StringWriter();
        	        StreamResult result = new StreamResult(sw);
        	        DOMSource source = new DOMSource(nn);
        	        transformer.transform(source, result);
        	        
        	        /* strip off the tag representing the current
        	         * field (we only want the contents as a string)
        	         */
        	        String sValue = sw.getBuffer().toString();
        	        sValue = sValue.substring(sValue.indexOf('>') + 1);
        	        return sValue.substring(0, sValue.lastIndexOf('<'));
                }
            }
        }
        return "";
    }

    /**
     * Gets the (element name, #text value) pairs for all child nodes in the
     * current field.
     * 
     * @return string matrix of the elements of the current field
     * @throws Exception - in case of failure.
     * @see #setNextField()
     * @see #setRootNode(String fieldname)
     */
    public String[][] getSubFields() throws Exception {
        if (currentNode.getNodeType() == Node.TEXT_NODE) {
            logger.error("getFieldByValue got a Text Node");
            return null;
        } else {
            NodeList nodes = currentNode.getChildNodes();
            int elementCount = 0;
            for (int i = 0; i < nodes.getLength(); i++) {
                if (nodes.item(i).getNodeType() == Node.ELEMENT_NODE) {
                    elementCount++;
                }
            }
            String[][] ret = new String[2][elementCount];
            Node nn, nnn;
            //NodeList nodes2;

            elementCount = 0;
            for (int i = 0; i < nodes.getLength(); i++) {
                nn = nodes.item(i);
                if (nn.getNodeType() == Node.ELEMENT_NODE) {
                    NamedNodeMap map = nn.getAttributes();
                    if (map.getNamedItem("name") != null) {
                        nnn = map.getNamedItem("name");
                        ret[0][elementCount] = nnn.getNodeValue();
                    }
                    /*
                    ret[1][elementCount] = null;
                    nodes2 = nn.getChildNodes();
                    for (int ii = 0; ii < nodes2.getLength(); ii++) {
                        nnn = nodes2.item(ii);
                        if (nnn.getNodeType() == Node.TEXT_NODE) {
                            ret[1][elementCount] = ((TextImpl) nnn).getData();
                            break;
                        }
                    }
                    */
                    
                    //XPath xpath = XPathFactory.newInstance().newXPath();
                    //ret[1][elementCount] = (String) xpath.evaluate("./text()", nn, XPathConstants.STRING);
                    
                    
                    //Transform the whole field to a string
        	        TransformerFactory tFactory = TransformerFactory.newInstance();
        	        Transformer transformer = tFactory.newTransformer();
        	        transformer.setOutputProperty("omit-xml-declaration", "yes");
        	        StringWriter sw = new StringWriter();
        	        StreamResult result = new StreamResult(sw);
        	        DOMSource source = new DOMSource(nn);
        	        transformer.transform(source, result);
        	        
        	        // strip off the tag representing the current
        	        // field (we only want the contents as a string)
        	        String sValue = sw.getBuffer().toString();
        	        sValue = sValue.substring(sValue.indexOf('>') + 1);
        	        ret[1][elementCount] = sValue.substring(0, sValue.lastIndexOf('<'));
                    
                    elementCount++;
                }
            }
            return ret;
        }
    }

    /**
     * Inserts Entity References in the place of certain illegal characters in
     * order to make an xml file valid
     * 
     * @param text
     *            <code>String</code> - the xml file insert Entity References
     *            into
     * @return <code>String</code> - a version of the xml file with Entity
     *         References in the place of certain illegal characters
     */
    public static String escapeForXML(String text) {
        StringBuffer xmlText = new StringBuffer();

        int length = text.length();
        for (int i = 0; i < length; i++) {
            char ch = text.charAt(i);
            switch (ch) {
            case '>':
                xmlText.append("&gt;");
                break;
            case '<':
                xmlText.append("&lt;");
                break;
            case '&':
                xmlText.append("&amp;");
                break;
            case '\'':
                xmlText.append("&apos;");
                break;
            case '"':
                xmlText.append("&quot;");
                break;
            default:
                xmlText.append(ch);
            }
        }
        return xmlText.toString();
    }

}
