package org.gcube.index.fulltextindexnode;

import static org.gcube.resources.discovery.icclient.ICFactory.clientFor;
import static org.gcube.resources.discovery.icclient.ICFactory.queryFor;

import java.net.URI;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.xml.rpc.ServiceException;

import org.apache.axis.message.addressing.AttributedURI;
import org.apache.axis.message.addressing.EndpointReferenceType;
import org.apache.axis.types.URI.MalformedURIException;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.SearchHit;
import org.gcube.common.core.contexts.GCUBERemotePortTypeContext;
import org.gcube.common.core.faults.GCUBEFault;
import org.gcube.common.core.porttypes.GCUBEPortType;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.types.VOID;
import org.gcube.common.resources.gcore.ServiceInstance;
import org.gcube.common.scope.api.ScopeProvider;
import org.globus.wsrf.ResourceException;
import org.oasis.wsrf.lifetime.Destroy;

import org.gcube.elasticsearch.*;

import org.gcube.index.fulltextindexnode.stubs.CreateResourceResponse;
import org.gcube.index.fulltextindexnode.stubs.FullTextIndexNodeFactoryPortType;
import org.gcube.index.fulltextindexnode.stubs.FullTextIndexNodePortType;
import org.gcube.index.fulltextindexnode.stubs.StringArray;
import org.gcube.index.fulltextindexnode.stubs.GetIndexInformationResponse;
import org.gcube.index.fulltextindexnode.stubs.service.FullTextIndexNodeFactoryServiceAddressingLocator;
import org.gcube.index.fulltextindexnode.stubs.service.FullTextIndexNodeServiceAddressingLocator;

import org.gcube.index.fulltextindexnode.stubs.Query;
import org.gcube.index.fulltextindexnode.stubs.QueryLimited;
import org.gcube.index.fulltextindexnode.stubs.FeedLocator;
import org.gcube.index.fulltextindexnode.stubs.FeedLocatorUnit;

import org.gcube.indexmanagement.common.FullTextIndexType;
import org.gcube.resources.discovery.client.api.DiscoveryClient;
import org.gcube.resources.discovery.client.queries.api.SimpleQuery;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FullTextIndexNode extends GCUBEPortType {

	protected static final String FACTORY = "Factory";
	protected static final String SERVICE_NAME = "FullTextIndexNode";
	protected static final String SERVICE_CLASS = "Index";
	private static final String USE_CLUSTER_ID = "useClusterId";
	protected static final int TIMEOUT = 700000;

	private static final Logger logger = LoggerFactory.getLogger(FullTextIndexNode.class);

//	private String clusterID = null;

	@Override
	protected void onReady() throws Exception {
		super.onReady();
	}

	/** {@inheritDoc} */
	protected ServiceContext getServiceContext() {
		return ServiceContext.getContext();
	}

	public synchronized GetIndexInformationResponse getIndexInformation(VOID request) throws GCUBEFault {
		try {
			Resource managementResource = getResource();

			if (managementResource.isInitializing())
				throw new Exception("Resource is not initialized yet");

			GetIndexInformationResponse response = new GetIndexInformationResponse();
			response.setIndexID(managementResource.getIndexID());

			StringArray colStringArray = new StringArray();
			colStringArray.setArray(managementResource.getCollectionID());
			response.setCollectionID(colStringArray);

			StringArray fieldStringArray = new StringArray();
			fieldStringArray.setArray(managementResource.getFields());
			response.setFields(fieldStringArray);

			return response;
		} catch (Exception e) {
			logger.error("Exception", e);
			throw new GCUBEFault(e.getMessage());
		}
	}

	private String initializeClusterID()
	{
		logger.info("resource key  is : " + getResouceKey());
		String clusterID = null;
		try {
			clusterID = (String) getResource().getResourcePropertySet().get(Resource.RP_CLUSTER_ID).get(0);
			logger.info("after get resource clusterID is : " + clusterID);
		} catch (ResourceException e) {
			logger.error("Error while initializing cluster id",e);
		}
		StatefulContext pctx = (StatefulContext) StatefulContext.getContext();
		boolean useClusterID = (Boolean) pctx.getProperty(USE_CLUSTER_ID);
		if (!useClusterID)
			clusterID = ServiceContext.getContext().getScope().toString();
		return clusterID;
	}
	
	public boolean activateIndex(String indexName) {
		logger.info("activating index : " + indexName);
		final String clusterID = initializeClusterID();
		logger.info("clusterID = "+clusterID);
		Resource resource = null;
		try {
			resource = this.getResource();
		} catch (ResourceException e) {
			logger.error("Error acquiring resource", e);
			return false;
		}
		
		final FullTextNode fulltextnode = getFullTextNode();
		boolean activateResult = fulltextnode.activateIndex(indexName);
		logger.info("Activate result was: " + activateResult);
		if (activateResult){
			try {
				updateManagerProperties(fulltextnode, resource, clusterID);
			} catch (Exception e) {
				logger.error("Error while updating manager properties", e);
				return false;
			}
		}
		return activateResult;
	}
	
	public Long collectionCount(String collectionID) {
		logger.info("collectionCount : " + collectionID);
		final FullTextNode fulltextnode = getFullTextNode();
		Long collectionCount = fulltextnode.getCollectionDocuments(collectionID);
		
		logger.info("collectionCount result : " + collectionCount);
		return collectionCount;
	}
	
	public StringArray getCollectionsOfIndex(String indexName) {
		final FullTextNode fulltextnode = getFullTextNode();
		Set<String> collections = fulltextnode.getCollectionsOfIndex(indexName);
		StringArray res = new StringArray();
		res.setArray(collections.toArray(new String[0]));
		
		return res;
	}
	
	
	public StringArray getIndicesOfCollection(String collectionID) {
		final FullTextNode fulltextnode = getFullTextNode();
		Set<String> indices = fulltextnode.getIndicesOfCollection(collectionID);
		StringArray res = new StringArray();
		res.setArray(indices.toArray(new String[0]));
		
		return res;
	}
	
	public boolean deactivateIndex(String indexName) {
		final FullTextNode fulltextnode = getFullTextNode();
		boolean deactivateResult = fulltextnode.deactivateIndex(indexName);
		logger.info("Deactivate result was: " + deactivateResult);
		
		return deactivateResult;
	}
	
	public boolean deleteIndex(String indexName) {
		logger.info("deleting index : " + indexName);
		final String clusterID = initializeClusterID();
		logger.info("clusterID = "+clusterID);
		Resource resource = null;
		try {
			resource = this.getResource();
		} catch (ResourceException e) {
			logger.error("Error acquiring resource", e);
			return false;
		}
		
		final FullTextNode fulltextnode = getFullTextNode();
		boolean deleteIndexResult = fulltextnode.deleteIndex(indexName);
		logger.info("DeleteIndex result was: " + deleteIndexResult);
		if (deleteIndexResult){
			try {
				updateManagerProperties(fulltextnode, resource, clusterID);
			} catch (Exception e) {
				logger.error("Error while updating manager properties", e);
				return false;
			}
		}
		
		return deleteIndexResult;
	}

	public boolean feedLocator(FeedLocator request) {
		logger.info("in feed locator");
		Set<String> sids = new HashSet<String>();
		
		StringArray rsids = request.getSids();
		if (rsids != null && rsids.getArray() != null){
			for (String rsi : rsids.getArray()){
				sids.add(rsi);
			}
		}
		
		return this.feedLocatorFlatten(request.getResultSetLocation(), request.getIndexName(), request.isActivate(), sids);
	}
	
	
	private boolean feedLocatorFlatten(final String resultSetLocation, final String indexName, final Boolean activate, final Set<String> sids) {
		logger.info("in feed locator flatten");
		logger.info("feedLocator activate : " + activate);
		
		final String clusterID = initializeClusterID();
		logger.info("clusterID = "+clusterID);
		final String key = getResouceKey();
		logger.info("resource key in feeding : " + key);
		final FullTextNode fulltextnode = getFullTextNode();
		Resource myResource = null;
		try {
			myResource = this.getResource();
		} catch (ResourceException e) {
			logger.error("Error acquiring resource", e);
			return false;
		}
		final Resource resource = myResource;
		
		new Thread() {
			@Override
			public void run() {
				try {
					final HashMap<String, FullTextIndexType> colForField = new HashMap<String, FullTextIndexType>();
					Boolean result = null;
					if (indexName != null )
						result = fulltextnode.feedLocator(resultSetLocation, colForField, indexName, sids);
					else
						result = fulltextnode.feedLocator(resultSetLocation, colForField, sids);
					logger.info("Feeding completed, result was: " + result);
					if (result) {
						if (activate){
							boolean activateResult = fulltextnode.activateIndex(indexName);
							logger.info("Activate result was: " + activateResult);
							updateManagerProperties(fulltextnode, resource, clusterID);
						}
					}
					logger.info("Properties updating completed");
				} catch (Exception e) {
					logger.error("Exception", e);
				}
			}
		}.start();
		return true;
	}
	
	
	
	public boolean feedLocatorUnit(FeedLocatorUnit request) {
		logger.info("in feed locator unit");
		Set<String> sids = new HashSet<String>();
		
		StringArray rsids = request.getSids();
		if (rsids != null && rsids.getArray() != null){
			for (String rsi : rsids.getArray()){
				sids.add(rsi);
			}
		}
		
		return this.feedLocatorUnitFlatten(request.getResultSetLocation(), request.getIndexName(), sids);
	}
	
	
	///////////////
	private boolean feedLocatorUnitFlatten(final String resultSetLocation, final String indexName, final Set<String> sids) {
		logger.info("in feed locator unit flatten");
		
		final String clusterID = initializeClusterID();
		logger.info("clusterID = "+clusterID);
		System.out.println("clusterID = "+clusterID);
		final String key = getResouceKey();
		logger.info("resource key in feeding : " + key);
		final FullTextNode fulltextnode = getFullTextNode();
		Resource myResource = null;
		try {
			myResource = this.getResource();
		} catch (ResourceException e) {
			logger.error("Error acquiring resource", e);
			return false;
		}
		final Resource resource = myResource;
		
		Boolean result = null;
		try {
			final HashMap<String, FullTextIndexType> colForField = new HashMap<String, FullTextIndexType>();
			if (indexName != null )
				result = fulltextnode.feedLocator(resultSetLocation, colForField, indexName, sids);
			else
				result = fulltextnode.feedLocator(resultSetLocation, colForField, sids);
			logger.info("Feeding completed, result was: " + result);
		} catch (Exception e) {
			result = Boolean.FALSE;
			logger.error("Exception", e);
		}
		return result;
	}
	///////////////
	
	
	
	
	public boolean flush(VOID v)
	{
		final String clusterID = initializeClusterID();
		final FullTextNode fulltextnode = getFullTextNode();
		try
		{
			logger.info("Flushing index...");
			boolean flushResult = updateManagerProperties(fulltextnode,  this.getResource(), clusterID);
			logger.info("Flushing index result : " + flushResult);
			return flushResult;
		}catch(Exception e)
		{
			logger.error("Failed to flush index",e);
			return false;
		}
			
	}
	
	public String query(Query request) {
		Set<String> sids = new HashSet<String>();
		
		StringArray rsids = request.getSids();
		if (rsids != null && rsids.getArray() != null){
			for (String rsi : rsids.getArray()){
				sids.add(rsi);
			}
		}
		
		final String clusterID = initializeClusterID();
		try {
			return getFullTextNode().query(request.getQuery(), sids);
		} catch (Exception e) {
			logger.error("Exception", e);
			return "Caught Exception " + e.getMessage();
		}
	}
	
	public String queryLimited(QueryLimited request) {
		Set<String> sids = new HashSet<String>();
		
		StringArray rsids = request.getSids();
		if (rsids != null && rsids.getArray() != null){
			for (String rsi : rsids.getArray()){
				sids.add(rsi);
			}
		}
		
		final String clusterID = initializeClusterID();
		try {
			return getFullTextNode().query(request.getQuery(), request.getCount(), request.getFrom(), sids);
		} catch (Exception e) {
			logger.error("Exception", e);
			return "Caught Exception " + e.getMessage();
		}
	}

	public boolean shutdown(String nothing) {
		final String clusterID = initializeClusterID();
		try {
			if (nothing != null && nothing.trim().equalsIgnoreCase("DELETE")) {
				getFullTextNode().delete();
				updateManagerProperties(getFullTextNode(), this.getResource(), clusterID);
			}
			getFullTextNode().close();

			return true;
		} catch (Exception e) {
			logger.error("Exception", e);
			return false;
		}
	}
	
	
	public boolean deleteCollection(String collectionID) {
		final String clusterID = initializeClusterID();
		try {
			Boolean result = getFullTextNode().deleteCollection(collectionID);
			logger.info("Feeding completed, result was: " + result);
			if (result)
				updateManagerProperties(getFullTextNode(), getResource(), clusterID);
			logger.info("Properties updating completed");
			
			return result;
		} catch (Exception e) {
			logger.error("Exception", e);
			return false;
		}
	}

	public boolean destroyNode(VOID voidType) {
		final String clusterID = initializeClusterID();

		boolean hasFailed = false;
		HashMap<String, Set<String>> endpoints = discoverFulltextIndexNodes(Arrays.asList(getFullTextNodeClientScope()), clusterID);

		logger.info("Found endpoints to shutdown " + endpoints);

		logger.info("Deleting indices...");
		getFullTextNode().delete();
		logger.info("Deleting indices DONE");

		GCUBEScope scope = GCUBEScope.getScope(getFullTextNodeClientScope());
		
		for (String endpoint : endpoints.keySet()) {
			for (String key : endpoints.get(endpoint)) {
				logger.info("Destoying resource " + endpoint + " " + key);
				try {
					EndpointReferenceType factoryEPR = new EndpointReferenceType(new AttributedURI(endpoint + FACTORY));
					FullTextIndexNodeFactoryPortType fpt = new FullTextIndexNodeFactoryServiceAddressingLocator()
							.getFullTextIndexNodeFactoryPortTypePort(factoryEPR);
					fpt = GCUBERemotePortTypeContext.getProxy(fpt, scope);

					org.gcube.index.fulltextindexnode.stubs.CreateResource cr = new org.gcube.index.fulltextindexnode.stubs.CreateResource();
					cr.setIndexID(key);
					CreateResourceResponse crr = fpt.createResource(cr);
					logger.info("created resource at endpoint " + crr.getEndpointReference());
					FullTextIndexNodePortType pt = new FullTextIndexNodeServiceAddressingLocator().getFullTextIndexNodePortTypePort(crr.getEndpointReference());
					pt = GCUBERemotePortTypeContext.getProxy(pt, scope, TIMEOUT);

					if (pt.shutdown("") == false) { // will not do delete
						logger.error("error deleting index at : " + endpoint);
						hasFailed = true;
					} else {
						logger.info("Destroying of discovered resource...");
						pt.destroy(new Destroy());
						logger.info("Destroying of discovered resource...OK");
					}
				} catch (MalformedURIException e) {
					logger.error("Exception", e);
					hasFailed = true;
				} catch (ServiceException e) {
					logger.error("Exception", e);
					hasFailed = true;
				} catch (RemoteException e) {
					logger.error("Exception", e);
					hasFailed = true;
				} catch (Exception e) {
					logger.error("Exception", e);
					hasFailed = true;
				}
			}
		}

		// Delete this resource

		return !(hasFailed);
	}

	public boolean refresh(VOID voidType) {
		final String clusterID = initializeClusterID();
		try {
			getFullTextNode().invalidateCache();
			getFullTextNode().refreshIndexTypesOfIndex();
		} catch (Exception e) {
			logger.error("Exception ", e);
			return false;
		}
		return true;
	}

	public boolean rebuildMetaIndex(VOID voidType) {
		final String clusterID = initializeClusterID();
		try {
			Resource managementResource = getResource();

			if (managementResource.isInitializing())
				throw new Exception("Resource is not initialized yet");

			String[] collectionIds = managementResource.getCollectionID();
			String[] fields = managementResource.getFields();

			logger.info("Adding collectionsIDs: " + Arrays.toString(collectionIds));
			logger.info("Adding fields: " + Arrays.toString(fields));
			getFullTextNode().rebuildMetaIndex(Arrays.asList(collectionIds), Arrays.asList(fields));

		} catch (Exception e) {
			logger.error("Exception", e);
			return false;
		}
		return true;
	}

	public boolean setCollections(StringArray collectionIDs) throws GCUBEFault {
		logger.info("settingCollections");
		try {
			Resource resource = getResource();
			resource.setCollectionID(collectionIDs.getArray());
			resource.store();
		} catch (RemoteException re) {
			throw new GCUBEFault(re.getMessage());
		} catch (Exception e) {
			logger.error("Exception", e);
		}
		return true;
	}

	public boolean setFields(StringArray fields) throws GCUBEFault {
		logger.info("settingFields");
		try {
			Resource resource = getResource();
			resource.setFields(fields.getArray());
			resource.store();
		} catch (RemoteException re) {
			throw new GCUBEFault(re.getMessage());
		} catch (Exception e) {
			logger.error("Exception", e);
		}
		return true;
	}

	@SuppressWarnings("unchecked")
	private boolean updateManagerProperties(final FullTextNode fulltextnode, final Resource resource, final String clusterID) throws GCUBEFault {
		Map<String, Object> result = null;
		try {
			SearchResponse response = fulltextnode.getIndexClient().prepareSearch(FullTextNode.META_INDEX).setQuery(QueryBuilders.matchAllQuery()).execute()
					.actionGet();
			for (SearchHit hit : response.getHits().getHits()) {
				result = hit.getSource();
			}

		} catch (Exception e) {
			logger.warn("Meta Index missing");
		}
		ArrayList<String> collectionIdsToBeAdded;
		ArrayList<String> fieldsToBeAdded;
		if (result != null) {
			collectionIdsToBeAdded = (ArrayList<String>) result.get("collectionIDs");
			fieldsToBeAdded = (ArrayList<String>) result.get("fields");
		} else {
			throw new RuntimeException("no meta-index document found");
		}

		logger.info("fields to be added: " + fieldsToBeAdded);
		logger.info("Collections to be added: " + collectionIdsToBeAdded);

		HashMap<String, Set<String>> endpoints = discoverFulltextIndexNodes(Arrays.asList(fulltextnode.getScope()), clusterID);

		StringArray sac = new StringArray();
		sac.setArray(Arrays.copyOf(collectionIdsToBeAdded.toArray(), collectionIdsToBeAdded.toArray().length, String[].class));
		StringArray saf = new StringArray();
		saf.setArray(Arrays.copyOf(fieldsToBeAdded.toArray(), fieldsToBeAdded.toArray().length, String[].class));

//		try {
//			resource.setCollectionID(sac.getArray());
//			resource.setFields(saf.getArray());
//			resource.store();
//		} catch (RemoteException re) {
//			throw new GCUBEFault(re.getMessage());
//		} catch (Exception e) {
//			logger.error("Exception", e);
//		}

		for (String endpoint : endpoints.keySet()) {
			for (String key : endpoints.get(endpoint)) {
				logger.info("Recreating resource " + endpoint + " " + key);
				try {
					EndpointReferenceType factoryEPR = new EndpointReferenceType(new AttributedURI(endpoint + FACTORY));
					FullTextIndexNodeFactoryPortType fpt = new FullTextIndexNodeFactoryServiceAddressingLocator()
							.getFullTextIndexNodeFactoryPortTypePort(factoryEPR);
					String scope = fulltextnode.getScope();
					fpt = GCUBERemotePortTypeContext.getProxy(fpt, GCUBEScope.getScope(scope));

					org.gcube.index.fulltextindexnode.stubs.CreateResource cr = new org.gcube.index.fulltextindexnode.stubs.CreateResource();
					cr.setIndexID(key);
					CreateResourceResponse crr = fpt.createResource(cr);
					logger.info("created resource at endpoint " + crr.getEndpointReference());
					FullTextIndexNodePortType pt = new FullTextIndexNodeServiceAddressingLocator().getFullTextIndexNodePortTypePort(crr.getEndpointReference());
					pt = GCUBERemotePortTypeContext.getProxy(pt, GCUBEScope.getScope(fulltextnode.getScope()), TIMEOUT);
					pt.setCollections(sac);
					pt.setFields(saf);
					logger.info("refreshing cache in " + crr.getEndpointReference());
					pt.refresh(new VOID());
				} catch (MalformedURIException e) {
					logger.error("Exception", e);
				} catch (ServiceException e) {
					logger.error("Exception", e);
				} catch (RemoteException e) {
					logger.error("Exception", e);
				} catch (Exception e) {
					logger.error("Exception", e);
				}
			}

		}

		return true;
	}

	public static HashMap<String, Set<String>> discover(String serviceName, String serviceClass, List<String> scopes, String clusterID) {
		logger.info("Discovering : serviceName " + serviceName + " serviceClass, " + serviceClass + " scopes : " + scopes + " clusterID : " + clusterID);

		HashMap<String, Set<String>> endpoints = new HashMap<String, Set<String>>();

		for (String scope : scopes) {
			ScopeProvider.instance.set(scope);

			SimpleQuery query = queryFor(ServiceInstance.class);
			query.addNamespace("ns1", URI.create("http://gcube-system.org/namespaces/index/FullTextIndexNode/service"))
					.addCondition("$resource/Data/gcube:ServiceClass/text() eq '" + serviceClass + "'")
					.addCondition("$resource/Data/gcube:ServiceName/text() eq '" + serviceName + "'")
					.addCondition("$resource/Data/ns1:ClusterID/text() eq '" + clusterID + "'");

			DiscoveryClient<ServiceInstance> client = clientFor(ServiceInstance.class);
			List<ServiceInstance> resources = client.submit(query);

			for (ServiceInstance se : resources) {
				String endpoint = se.endpoint().toString();
				if (endpoints.containsKey(endpoint)) {
					Set<String> wsr = endpoints.get(endpoint);
					wsr.add(se.key());
					endpoints.put(endpoint, wsr);
				} else {
					Set<String> wsr = new HashSet<String>();
					wsr.add(se.key());
					endpoints.put(endpoint, wsr);
				}

			}
		}
		return endpoints;
	}

	public static HashMap<String, Set<String>> discoverByIndexID(String serviceName, String serviceClass, List<String> scopes, String indexID, String epr) {
		logger.info("Discovering : serviceName " + serviceName + " serviceClass, " + serviceClass + " scopes : " + scopes + " indexID : " + indexID);

		HashMap<String, Set<String>> endpoints = new HashMap<String, Set<String>>();

		for (String scope : scopes) {
			ScopeProvider.instance.set(scope);

			SimpleQuery query = queryFor(ServiceInstance.class);
			query.addNamespace("ns1", URI.create("http://gcube-system.org/namespaces/index/FullTextIndexNode/service"))
					.addCondition("$resource/Data/gcube:ServiceClass/text() eq '" + serviceClass + "'")
					.addCondition("$resource/Data/gcube:ServiceName/text() eq '" + serviceName + "'")
					.addCondition("$resource/Data/ns1:IndexID/text() eq '" + indexID + "'")
					.addCondition("$resource/Source/text() eq '" + epr + "'");

			DiscoveryClient<ServiceInstance> client = clientFor(ServiceInstance.class);
			List<ServiceInstance> resources = client.submit(query);

			for (ServiceInstance se : resources) {
				String endpoint = se.endpoint().toString();
				if (endpoints.containsKey(endpoint)) {
					Set<String> wsr = endpoints.get(endpoint);
					wsr.add(se.key());
					endpoints.put(endpoint, wsr);
				} else {
					Set<String> wsr = new HashSet<String>();
					wsr.add(se.key());
					endpoints.put(endpoint, wsr);
				}

			}
		}
		return endpoints;
	}

	
	public static HashMap<String, Set<String>> discoverFulltextIndexNodes(List<String> scopes, String clusterID) {
		return discover(SERVICE_NAME, SERVICE_CLASS, scopes, clusterID);
	}

	private Resource getResource() throws ResourceException {
		return (Resource) StatefulContext.getContext().getWSHome().find();
	}

	private String getResouceKey() {
		try {
			String resourceKey = this.getResource().getResourceKey();

			// String resourceKey = (String)
			// getResource().getResourcePropertySet().get(Resource.RP_INDEX_ID).get(0);
			logger.info("getResouceKey : " + resourceKey);
			return resourceKey;
		} catch (Exception e) {
			logger.error("Exception", e);
			return null;
		}
	}

	private FullTextNode getFullTextNode() {
		try {
			return getResource().getFullTextNodeClient().getFullTextNode();
		} catch (Exception e) {
			logger.error("Exception", e);
			return null;
		}
	}

	private String getFullTextNodeClientScope() {
		try {
			return getResource().getFullTextNodeClient().getScope();
		} catch (Exception e) {
			logger.error("Exception", e);
			return null;
		}
	}

}