package org.gcube.rest.index.service.resources;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;

import org.gcube.elasticsearch.FullTextNode;
import org.gcube.rest.commons.resourceawareservice.resources.ResourceFactory;
import org.gcube.rest.commons.resourceawareservice.resources.exceptions.StatefulResourceException;
import org.gcube.rest.index.common.Constants;
import org.gcube.rest.index.common.resources.IndexResource;
import org.gcube.rest.index.service.IndexClientWrapper;
import org.gcube.rest.resourceawareservice.exceptions.ResourceNotFoundException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.io.Resources;
import com.google.gson.Gson;
import com.google.inject.Inject;
import com.google.inject.Provider;

public class IndexResourceFactory extends ResourceFactory<IndexResource> {

	private static final Logger logger = LoggerFactory
			.getLogger(IndexResourceFactory.class);

	private static final String DEFAULT_SAME_CLUSTER_PROP = "defaultSameCluster";

	private Map<String, IndexClientWrapper> indexClientWrappers = new ConcurrentHashMap<String, IndexClientWrapper>();

	private Provider<IndexClientWrapper> ftnClientProvider;

	@Inject
	public IndexResourceFactory(Provider<IndexClientWrapper> ftnClientProvider) {
		this.ftnClientProvider = ftnClientProvider;
	}

	public FullTextNode getIndexNode(IndexResource resource)
			throws ResourceNotFoundException {
		if (!this.indexClientWrappers.containsKey(resource.getResourceID()))
			throw new ResourceNotFoundException("resource with id : "
					+ resource.getResourceID() + " not in map factory : "
					+ this.indexClientWrappers.keySet());
		return this.indexClientWrappers.get(resource.getResourceID())
				.getFullTextNode();
	}

	@Override
	public IndexResource createResource(String resourceID, String params)
			throws StatefulResourceException {
		logger.info("IndexResource createResource");

		IndexResource resource = new Gson().fromJson(params,
				IndexResource.class);

		if (resource.getScope() != null
				&& resource.getScope().equalsIgnoreCase(this.getScope()) == false) {
			logger.error("scope set to : " + resource.getScope()
					+ " but different to : " + this.getScope());
			throw new StatefulResourceException("scope set to : "
					+ resource.getScope() + " but different to : "
					+ this.getScope());
		}

		resource.setResourceID(resourceID);

		logger.info("IndexClientWrapper initializing");

		IndexClientWrapper clientWrapper = this.ftnClientProvider.get();
		initICWToResource(clientWrapper, resource);
		this.indexClientWrappers.put(resourceID, clientWrapper);

		resource.setHostname(clientWrapper.getFullTextNode().getHostname());
		logger.info("hostname set to resource : " + resource.getHostname());

		return resource;
	}

	@Override
	public void loadResource(IndexResource resource)
			throws StatefulResourceException {
		logger.info("IndexResource loadResource");

		if (resource.getScope() != null
				&& resource.getScope().equalsIgnoreCase(this.getScope()) == false) {
			logger.error("scope set to : " + resource.getScope()
					+ " but different to : " + this.getScope());
			throw new StatefulResourceException("scope set to : "
					+ resource.getScope() + " but different to : "
					+ this.getScope());
		}

		logger.info("IndexClientWrapper loading");
		IndexClientWrapper clientWrapper = this.ftnClientProvider.get();
		loadICWToResource(clientWrapper, resource);

		resource.setHostname(clientWrapper.getFullTextNode().getHostname());
		logger.info("hostname set to resource : " + resource.getHostname());

		resource.onLoad();
		this.indexClientWrappers.put(resource.getResourceID(), clientWrapper);
	}

	@Override
	public void closeResource(IndexResource resource)
			throws StatefulResourceException {
		IndexClientWrapper icw = this.indexClientWrappers.get(resource
				.getResourceID());
		logger.info("Closing index...");
		icw.getFullTextNode().close();
		logger.info("Closing index...OK");
		this.indexClientWrappers.remove(resource.getResourceID());
	}

	@Override
	public void destroyResource(IndexResource resource)
			throws StatefulResourceException {
		IndexClientWrapper icw = this.indexClientWrappers.get(resource
				.getResourceID());

		try {
			logger.info("Deleting index");
			icw.getFullTextNode().deleteIndex();
		} catch (Exception e) {
			logger.error(
					"Error while deleting the index. Maybe it does not exist",
					e);
			throw new StatefulResourceException(
					"Error while deleting the index. Maybe it does not exist",
					e);
		} finally {
			this.closeResource(resource);
		}
	}

	private static void initICWToResource(
			IndexClientWrapper indexClientWrapper, IndexResource resource)
			throws StatefulResourceException {
		if (resource.getIndexID() == null
				|| resource.getIndexID().trim().length() == 0) {
			logger.info("No indexID given, assigning a new one: "
					+ resource.getResourceID());
			resource.setIndexID(resource.getResourceID());
		}

		String transformedClusterName = transformClusterName(
				resource.getClusterID(), resource.getResourceID(),
				resource.getScope());
		
		resource.setClusterID(changeClusterID(resource.getClusterID(), resource.getResourceID(),resource.getScope()));

//		logger.info("cluster id will change to : " + transformedClusterName);
//		resource.setClusterID(transformedClusterName);
		
		try {
			logger.info("initializing FullTextNodeClient : ");
			indexClientWrapper.initialize(transformedClusterName, resource.getClusterID());
		} catch (Exception e) {
			throw new StatefulResourceException(
					"error while initializing the fulltext index client", e);
		}

		String transportAddress = indexClientWrapper.getFullTextNode()
				.getESTransportAddress();
		resource.setEsTransportAddress(transportAddress);

		logger.info("getting meta index values for existing index");
		List<String> collectionsOfMetaIndex = indexClientWrapper
				.getFullTextNode().getCollectionOfIndex();
		List<String> fieldsOfMetaIndex = indexClientWrapper.getFullTextNode()
				.getFieldsOfIndex();

		if (collectionsOfMetaIndex != null && fieldsOfMetaIndex != null) {
			resource.setCollections(collectionsOfMetaIndex);
			resource.setFields(fieldsOfMetaIndex);
		} else {
			resource.setCollections(Collections.<String> emptyList());
			resource.setFields(Collections.<String> emptyList());
		}

		resource.setSupportedRelations(IndexResource.getSupportedRelationsSet());

	}

	private static void loadICWToResource(
			IndexClientWrapper indexClientWrapper, IndexResource resource)
			throws StatefulResourceException {

		String transformedClusterName = transformClusterName(
				resource.getClusterID(), resource.getResourceID(),
				resource.getScope());
		
		resource.setClusterID(changeClusterID(resource.getClusterID(), resource.getResourceID(),resource.getScope()));
		
		try {
			logger.info("initializing FullTextNodeClient : ");
			indexClientWrapper.initialize(transformedClusterName, resource.getClusterID());
		} catch (Exception e) {
			throw new StatefulResourceException(
					"error while initializing the fulltext index client", e);
		}

		String transportAddress = indexClientWrapper.getFullTextNode()
				.getESTransportAddress();
		resource.setEsTransportAddress(transportAddress);

		logger.info("getting meta index values for existing index");
		List<String> collectionsOfMetaIndex = indexClientWrapper
				.getFullTextNode().getCollectionOfIndex();
		List<String> fieldsOfMetaIndex = indexClientWrapper.getFullTextNode()
				.getFieldsOfIndex();

		if (collectionsOfMetaIndex != null && fieldsOfMetaIndex != null) {
			resource.setCollections(collectionsOfMetaIndex);
			resource.setFields(fieldsOfMetaIndex);
		} else {
			resource.setCollections(Collections.<String> emptyList());
			resource.setFields(Collections.<String> emptyList());
		}

		resource.setSupportedRelations(IndexResource.getSupportedRelationsSet());
	}

	private static String transformClusterName(String initClusterID,
			String resourceID, String scope) {

		logger.info("calling transformedClusterName for parameters. initClusterID : "
				+ initClusterID
				+ ", resourceID : "
				+ resourceID
				+ ", scope : "
				+ scope);

		String clusterName = changeClusterID(initClusterID, resourceID, scope);

		String ret = "es-cluster-" + scope + "-" + clusterName;
		
		ret = ret.replace("/", "-");
		logger.info("transformedClusterName : " + ret);

		return ret;

	}
	
	static String changeClusterID(String initClusterID,
			String resourceID, String scope){
		
		final Properties properties = new Properties();
		try (InputStream is = Resources.getResource(Constants.PROPERTIES_FILE)
				.openStream()) {
			properties.load(is);
		} catch (Exception e) {
			throw new IllegalArgumentException(
					"could not load property file  : "
							+ Constants.PROPERTIES_FILE);
		}

		Boolean defaultSameCluster = Boolean.valueOf(properties
				.getProperty(DEFAULT_SAME_CLUSTER_PROP));

		String clusterName = null;

		if (initClusterID != null) {
			clusterName = initClusterID;
		} else {
			if (defaultSameCluster == true) {
				clusterName = "default-cluster-name";
			} else {
				clusterName = resourceID;
			}
		}
		
		return clusterName;
	}

}
