/**
 * BDBWrapper.java
 *
 * $Author: spiros $
 * $Author: vasilis verroios $
 * $Date: 2008/03/13 10:10:59 $
 * $Id: BDBWrapper.java,v 1.0 2009/03/03 10:10:59 verroios Exp $
 *
 * <pre>
 *             Copyright (c) : 2007 Fast Search & Transfer ASA
 *                             ALL RIGHTS RESERVED
 * </pre>
 */

package org.gcube.indexmanagement.bdbwrapper;


import gr.uoa.di.madgik.grs.proxy.tcp.TCPWriterProxy;
import gr.uoa.di.madgik.grs.record.GenericRecord;
import gr.uoa.di.madgik.grs.record.GenericRecordDefinition;
import gr.uoa.di.madgik.grs.record.RecordDefinition;
import gr.uoa.di.madgik.grs.record.field.FieldDefinition;
import gr.uoa.di.madgik.grs.record.field.StringFieldDefinition;
import gr.uoa.di.madgik.grs.writer.GRS2WriterException;
import gr.uoa.di.madgik.grs.writer.RecordWriter;

import java.io.File;
import java.io.FileInputStream;
import java.io.StringReader;
import java.net.URI;
import java.nio.ByteBuffer;
import java.nio.channels.ReadableByteChannel;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.ListIterator;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.Set;
import java.util.Vector;

import jdbm.RecordManager;
import jdbm.btree.BTree;
import jdbm.helper.Tuple;

import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.common.searchservice.searchlibrary.resultset.elements.PropertyElementBase;
import org.gcube.common.searchservice.searchlibrary.resultset.elements.ResultElementGeneric;
import org.gcube.common.searchservice.searchlibrary.rsclient.elements.RSLocator;
import org.gcube.common.searchservice.searchlibrary.rsclient.elements.RSResourceWSRFType;
import org.gcube.common.searchservice.searchlibrary.rswriter.RSXMLWriter;
import org.gcube.indexmanagement.bdbwrapper.BDBGcqlQueryContainer.SingleTerm;
import org.gcube.indexmanagement.common.ForwardIndex;
import org.gcube.indexmanagement.common.ForwardIndexField;
import org.gcube.indexmanagement.common.ForwardIndexType;
import org.gcube.indexmanagement.common.IndexException;
import org.gcube.indexmanagement.common.IndexType;
import org.gcube.indexmanagement.common.PropertyElementForwardIndex;
import org.gcube.indexmanagement.common.ResultType;
import org.gcube.indexmanagement.common.XMLTokenReplacer;
import org.gcube.indexmanagement.common.linguistics.languageidplugin.LanguageIdPlugin;
import org.gcube.indexmanagement.resourceregistry.RRadaptor;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.SAXParseException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.DefaultHandler;
import org.xml.sax.helpers.XMLReaderFactory;

import com.sleepycat.bind.tuple.FloatBinding;
import com.sleepycat.bind.tuple.IntegerBinding;
import com.sleepycat.bind.tuple.StringBinding;
import com.sleepycat.je.Cursor;
import com.sleepycat.je.Database;
import com.sleepycat.je.DatabaseConfig;
import com.sleepycat.je.DatabaseEntry;
import com.sleepycat.je.Environment;
import com.sleepycat.je.EnvironmentConfig;
import com.sleepycat.je.OperationStatus;


enum LookupType { EQUAL, NOTEQUAL, LT, LE, GT, GE, GTANDLT, GTANDLE, GEANDLT, GEANDLE };

/**
 *  Class to store key and value pairs in the berkeley database
 */
public class BDBWrapper implements ForwardIndex {
	
	//Supported Relations
	public static final String EXACT = "==";

	public static final String WITHIN = "within";
	
	public static String[] SupportedRelations = { EXACT, WITHIN };
	
    /**
     * @param dbName   Name of the DB.
     */    
    private String dbName;
    
    /**
     * @param dbName   ID of the resource.
     */    
    private String resourceId;
    
    /**
     * @param treeName   Name of the tree to store within the storage file.
     */
    private String treeName;
    
    /**
     * @param env Environment of the BerkeleyDB databases
     */
    private Environment env;

    /**
     * @param db the BerkeleyDB databases, one for each field
     */
    private Database[] db;
    
    /**
     * @param dbMain the database to store the actual values
     */
    private Database dbMain;

    /* Log4j */
    static GCUBELog logger = new GCUBELog(BDBWrapper.class);

    /**
     * @param comparator that is used to compare the keys.
     */
    private Comparator[] comparator;
        
    /**
     * @param keyTypes stores info for every indexed field 
     */
    private HashMap<String, Integer> keyTypes = new HashMap<String, Integer>();
    
    /**
     * @param dbT stores info for the type of every indexed field
     */
    private ForwardIndexType[] dbT;
    
    /**
     * @param numKeys maintains the number of entries in each indexed field
     */
    private int[] numKeys;
    
    /**
     * @param numDocs maintains the number of values stored
     */
    private int numDocs;
    
    /*
     * used in internal naming
     */
    private static final String contentName = "Objects";

	static final String NODOCID = "NoDocID";
    
    /**
     * @param packageSize determines how keys and values are multiplexed
     */
    private long packageSize = 10;
        
    /**
     * @param envDir the directory used for storing indexes
     */
    private File envDir;
    
    /**
     * @param envConfig the configuration of the BerkeleyDB environment
     */
    private EnvironmentConfig envConfig;
    
    /**
     * @param dbConfig the configuration of the BerkeleyDB databases for indexed fields
     */
    private DatabaseConfig dbConfig[];
    
    /**
     * @param dbConfigMain the configuration of the BerkeleyDB database for value store
     */
    private DatabaseConfig dbConfigMain;
    
    /**
     * @param keyNames the names of the indexed fields
     */
    private String[] keyNames;

	private RRadaptor adaptor = null;

    /**
     * Provides the ability to store simple key-value pairs. 
     * @param baseDir      The base directory - it must exist before the call to the constructor
     * @param dbName       The name of the database to be used
     * @param resourceId The ID of the collection to be indexed
     * @param keyNames     The names of the keys. One index is created for each keyName 
     * @param dbT          The corresponding type of each keyName(int,string or float,string or string,string
     *                   date,string).
     * @throws IndexException - in case of failure.
     */
    public BDBWrapper(String baseDir,String dbName, String resourceId, String[] keyNames, ForwardIndexType[] dbT)
        throws IndexException {
        // Logger
    	logger.debug(" >>> BDBWrapper \n"  
                + "baseDir: " + baseDir 
                + "dbName: " + dbName
                + "treeName: " + treeName
                + "dbT " + dbT);
    	//create new environment handler
    	envDir = new File(baseDir);
    	envConfig = new EnvironmentConfig();
		envConfig.setAllowCreate(true);
		try{
			this.env = new Environment(envDir, envConfig);
		}catch(Exception e){
			logger.error("Corrupted Environment in the directory: " + baseDir);
			throw new IndexException("Corrupted Environment in the directory: " + baseDir);
		}
		//store dbName and collectionID and types
		this.dbName = dbName;
    	this.resourceId = resourceId;
    	this.dbT = dbT;
    	//the two arrays must have the same length
    	if(keyNames.length != dbT.length)
    	{
    		throw new IndexException("keyNames length isn't equal to dbT length");
    	}
    	
    	this.keyNames = keyNames;
    	
    	try{
    		//create an array to store Database handlers
    		this.db = new Database[keyNames.length];
    		this.comparator = new Comparator[keyNames.length]; 
    		this.numKeys = new int[keyNames.length];
    		this.dbConfig = new DatabaseConfig[keyNames.length];
    		//For each keyName create a new B-tree(if it doesn't already exist in this environment for this dbName and collectio ID)
    		for(int i=0; i<keyNames.length; i++)
    		{
    			keyTypes.put(keyNames[i], i);
    			//create the database configuration to be used
    			dbConfig[i] = new DatabaseConfig();
    			dbConfig[i].setAllowCreate(true);
    			dbConfig[i].setDeferredWrite(true);
    			dbConfig[i].setSortedDuplicates(true);
    			if(dbT[i].getKeyField().getDataType() == ForwardIndexField.DataType.DATE)
    			{
    				//comparator is necessary because of different date formats that may be used
    				dbConfig[i].setBtreeComparator(DateComparator.class);
    				comparator[i] = new DateComparator();
    			}else if (dbT[i].getKeyField().getDataType() == ForwardIndexField.DataType.STRING)
    			{
    				//it is not sure if a comparator in this case is necessary
    				dbConfig[i].setBtreeComparator(StringComparator.class);
    				comparator[i] = new StringComparator();
    			}else if (dbT[i].getKeyField().getDataType() == ForwardIndexField.DataType.INT)
    			{
    				//do not set Comparator in this case for higher performance
    				//dbConfig[i].setBtreeComparator(IntegerComparator.class);
    				comparator[i] =  new IntegerComparator();
    			}else if (dbT[i].getKeyField().getDataType() == ForwardIndexField.DataType.FLOAT){
    				//do not set Comparator in this case for higher performance
    				//dbConfig[i].setBtreeComparator(FloatComparator.class);
    				comparator[i] = new FloatComparator();
    			}
    			this.db[i] = env.openDatabase(null, convertToIndexName(keyNames[i]), dbConfig[i]);
    			this.numKeys[i] = (int)this.db[i].count();
    		}
    		dbConfigMain = new DatabaseConfig();
    		dbConfigMain.setAllowCreate(true);
    		dbConfigMain.setDeferredWrite(true);
    		this.dbMain = env.openDatabase(null, convertToIndexNameMain(contentName), dbConfigMain);
    		//initialise counters 
    		this.numDocs = (int)this.dbMain.count();
    		logger.debug(" <<< BDBWrapper");  
    	}catch (Exception e) {
            logger.error("db create failed ", e);
            // Must be handled
        }
    }
    
    /**
     * Returns the number of stored documents(values)
     * @return the number of stored documents
     */
    public int getNumDocs()
    {
    	return this.numDocs; 
    }
    
    /**
     * Returns the number of entries for each indexed field
     * @return array with the number of entries for each indexed field
     */
    public int[] getNumKeys()
    {
    	return this.numKeys;
    }
    
    /**
     * Returns the package size that determines how keys and values are multiplexed. Package size refers to 
     * how many key values will be sent before the corresponding values are sent.
     * @return package size
     */
    public long getPackageSize()
    {
    	return packageSize;
    }
    
    /**
     * Sets the package size that determines how keys and values are multiplexed. Package size refers to 
     * how many key values will be sent before the corresponding values are sent.
     * @param packageSize - package size
     */
    public void setPackageSize(long packageSize)
    {
    	this.packageSize = packageSize;
    }
    
    private String convertToIndexName(String keyName)
    {
    	return dbName + "_" + resourceId + "_" + keyName;  
    }
    
    private String convertToIndexNameMain(String name)
    {
    	return dbName + "_" + name + "_" + resourceId;   
    }

    /**
     * Closes this BDBWrapper by deleting the whole index
     * @throws Exception
     */
    public void close() throws Exception {
    	
    	// close database handles
        for(int i=0; i<this.db.length; i++)
        	this.db[i].close();
        this.dbMain.close();
    	//delete key indices
    	Set<String> c = keyTypes.keySet();
    	Iterator<String> iter = c.iterator();
    	while(iter.hasNext())
    		env.removeDatabase(null, convertToIndexName(iter.next()));
    	//delete Data index
    	env.removeDatabase(null, convertToIndexNameMain(contentName));
    	env.close();
    }
    
    protected void finalize() throws Throwable {
        try {
        	// close database, environment handlers
        	for(int i=0; i<this.db.length; i++)
        		this.db[i].close();
        	this.dbMain.close();
        	this.env.close();
        } finally {
            super.finalize();
        }
    }

    
    /**
     * Insert a key and value pair in the tree.
     * 
     * @param keyName - determines the indexed field where the key will be stored.
     * @param key -   the tuple key.
     * @param value - the tuple value.
     * @throws IndexException - when IOException is received.
     */
    private boolean insertPairInDb(String keyName, String key, DatabaseEntry value) throws IndexException {
        logger.debug(" >>> insertPairInDb key/valueid:  "+ key + " / " + StringBinding.entryToString(value));
        
        DatabaseEntry keyValue = new DatabaseEntry();
        boolean added = false;
        
        //check if the keyName is declared
        if(!keyTypes.containsKey(keyName))
        {
        	logger.error("keyName: " + keyName + " was not declared when the constructor was called. " +
			"There is no index for this keyName");
    		throw new IndexException("keyName: " + keyName + " was not declared when the constructor was called. " +
    				"There is no index for this keyName");
        }
        //find where info for this keyName is stored
        int pos = keyTypes.get(keyName);
        if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.DATE)
        	keyValue.setData(convertToDate(key,dbT[pos].getKeyField().getDataTypeFormat()).getBytes());
        else if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.FLOAT)
        	FloatBinding.floatToEntry(convertToFloat(key), keyValue);
        else if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.INT)      
        	IntegerBinding.intToEntry(convertToInt(key), keyValue);
        else if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.STRING)        
        	StringBinding.stringToEntry(key, keyValue);
        //store this pair in the corresponding index
        try{
        	if(this.db[pos].getSearchBoth(null, keyValue, value, null) == OperationStatus.NOTFOUND)
        		if(this.db[pos].put(null, keyValue, value) == OperationStatus.SUCCESS)
        		{
        			this.numKeys[pos]++;
        			added = true;
        		}
        }catch(Exception e)
        {
        	logger.error(" *** insertPairInDb error message: " + e.getMessage());
        	throw new IndexException("Could not insert key/valueid:  "+ key + " / " + StringBinding.entryToString(value) + " for keyName: " + keyName + 
        			" error: " + e.getMessage());
        }
        logger.debug(" <<< insertPairInDb");
        return added;
    }
    
    /**
     * Deletes all the keys that refer to the specific id in the field keyName. 
     * 
     * @param keyName -   the indexed field.
     * @param id - the id
     * @throws IndexException - when IOException is received.
     */
    private int deleteValueFromDb(String keyName, String id) throws Exception {
    	logger.debug(" >>> deleteValueFromDb: " + id);
    	DatabaseEntry keyValue = new DatabaseEntry();
        DatabaseEntry bindKey = new DatabaseEntry();
        DatabaseEntry bindKey2 = new DatabaseEntry();
        StringBinding.stringToEntry(id, bindKey2);
        
    	//check if the keyName is declared
        if(!keyTypes.containsKey(keyName))
    		throw new IndexException("keyName: " + keyName + " was not declared when the constructor was called. " +
    				"There is no index for this keyName");
        //find where info for this keyName is stored
        int pos = keyTypes.get(keyName);
        int refs = 0;
        //delete every key that refers to this id, from the corresponding index
        Cursor cursor = this.db[pos].openCursor(null, null);
        try{        	
        	OperationStatus opSt = OperationStatus.SUCCESS;
        	while(opSt == OperationStatus.SUCCESS)
        	{
        		opSt = cursor.getNext(keyValue, bindKey, null);
        		if(bindKey.equals(bindKey2)){
        			if(cursor.delete() == OperationStatus.SUCCESS){
        				logger.debug("deleted key from Index: " + keyName);
        				this.numKeys[pos]--;
        				refs++;
        			}else{
        				throw new Exception("cursor.delete() was not successful");
        			}
        		}
        	}
        	cursor.close();
        }catch(Exception e){
        	cursor.close();
        	logger.error("Could not delete entry from index with keyName: " +keyName+ " that refers to id: " + id);
			throw new Exception("Could not delete entry from index with keyName: " +keyName+ " that refers to id: " + id + " error: " +e.getMessage());
		}
        return refs;
    }

    /**
     * Deletes a key and value pair from the index keyName.
     * 
     * @param keyName - the name of the field.
     * @param key - the key
     * @param id - the id of the value.
     * @throws IndexException - when IOException is received.
     */
    private int deletePairInDb(String keyName, String key, String id) throws Exception {
        logger.debug(" >>> deletePairInDb: "+ key);
        DatabaseEntry keyValue = new DatabaseEntry();
        DatabaseEntry bindKey = new DatabaseEntry();
        StringBinding.stringToEntry(id, bindKey);
        
        //check if the keyName is declared
        if(!keyTypes.containsKey(keyName))
    		throw new IndexException("keyName: " + keyName + " was not declared when the constructor was called. " +
    				"There is no index for this keyName");
        //find where info for this keyName is stored
        int pos = keyTypes.get(keyName);
        if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.DATE)
        	keyValue.setData(convertToDate(key,dbT[pos].getKeyField().getDataTypeFormat()).getBytes());
        else if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.FLOAT)
        	FloatBinding.floatToEntry(convertToFloat(key), keyValue);
        else if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.INT)      
        	IntegerBinding.intToEntry(convertToInt(key), keyValue);
        else if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.STRING)        
        	StringBinding.stringToEntry(key, keyValue);
        //delete this pair from the corresponding index
        Cursor cursor = this.db[pos].openCursor(null, null);
        try{        	
        	if(cursor.getSearchBoth(keyValue, bindKey, null) == OperationStatus.NOTFOUND)
        	{
        		logger.error("Could not find key: " + key + " for keyName: " + keyName + "and for id: " + id);
        		return 0;
        	}
        	if(cursor.delete() == OperationStatus.SUCCESS)
        		this.numKeys[pos]--;
        	else{
        		throw new IndexException("Could not delete key:  "+ key + " for keyName: " + keyName + 
            			" and for id: " + id);
        	}
        	
        	cursor.close();
        	
        }
        catch(Exception e)
        {
        	cursor.close();
        	logger.error(" *** deletePairInDb error message: " + e.getMessage());
        	throw new IndexException("Could not delete key/id: " + key + "/" + id + " for keyName: " + keyName + 
        			" error: " + e.getMessage());
        } 
        return 1;
    }

    /**
     *  Inserts tuples in a rowset into the tree, parsing the values from a String.
     *  Deletes tuples in a rowset from the tree, parsing the values from a String.
     * @param rowset - the rowset containing the tuples
     * @throws IndexException - when parsing goes bonk.
     */
  /*  public void insertRowSet(String rowset) throws IndexException {
        logger.debug(" >>> insertRowSet:\n" + rowset);
        XMLProfileParser XMLparser = new XMLProfileParser();
        String[][] fieldData = null;
        try {
            XMLparser.readString(rowset, null);
            XMLparser.setRootNode("ROWSET");
            
            fieldData = XMLparser.getSubFields();
            
            if ((fieldData[0][0].equalsIgnoreCase("insert")) ||
                (fieldData[0][1].equalsIgnoreCase("insert"))) {
                XMLparser.setRootNode("tuple");                
                while (XMLparser.setNextField()) {
                    fieldData = XMLparser.getSubFields();
                    if (!fieldData[0][0].equals("key")) {
                        logger.error(" *** insertRowSet error, insert error, key not defined");
                        throw new IndexException(" *** insertRowSet error ***: " +
                                                 "*** Error 'key' is not first field in rowset");
                    }
                    if (!fieldData[0][1].equals("value")) {
                        logger.error(" *** insertRowSet error, insert error, value not defined");
                        throw new IndexException(" *** insertRowSet error ***: " +
                                                 "*** Error 'value' is not second field in rowset");
                    }
                    Object key = fieldData[1][0];
                    Object value = fieldData[1][1];
                    
                    // NB! null values are converted to "null" by soap
                    // if( value.equals("null") || value == null)
                    // System.out.println( "- Deleting "+key );
                    // Convert the key from String to the correct type
                    if (this.dbType == DbTypeEnum.INT_TYPE) {
                        insertPairInDb(convertToInt(key),value);
                    }
                    else if (this.dbType == DbTypeEnum.FLOAT_TYPE) {
                        insertPairInDb(convertToFloat(key),value);
                    }
                    else if (this.dbType == DbTypeEnum.STRING_TYPE) {
                        insertPairInDb(convertToString(key),value);
                    }
                    else if (this.dbType == DbTypeEnum.DATE_TYPE) {
                        insertPairInDb(convertToDate(key),value);
                    }
                }
            }
            if ((fieldData[0][0].equalsIgnoreCase("delete")) ||
                (fieldData[0][1].equalsIgnoreCase("delete"))) {
                XMLparser.setRootNode("tuple");                
                while (XMLparser.setNextField()) {
                    fieldData = XMLparser.getSubFields();
                    if (!fieldData[0][0].equals("key")) {
                        logger.error(" *** insertRowSet error, delete error, key not defined");
                        throw new IndexException(" *** deleteRowSet error ***: " +
                                                 "*** Error 'key' is not first field in rowset");
                    }
                    Object key = fieldData[1][0];                    
                    // NB! null values are converted to "null" by soap
                    // if( value.equals("null") || value == null)
                    // System.out.println( "- Deleting "+key );
                    // Convert the key from String to the correct type
                    if (this.dbType == DbTypeEnum.INT_TYPE) {
                        deletePairInDb(convertToInt(key));
                    }
                    else if (this.dbType == DbTypeEnum.FLOAT_TYPE) {
                        deletePairInDb(convertToFloat(key));
                    }
                    else if (this.dbType == DbTypeEnum.STRING_TYPE) {
                        deletePairInDb(convertToString(key));
                    }
                    else if (this.dbType == DbTypeEnum.DATE_TYPE) {
                        deletePairInDb(convertToDate(key));
                    }
                }
            }
        } 
        catch (Exception e) {
            logger.error(" *** insertRowSet error " + e.getMessage());
            throw new IndexException(" *** insertRowSet error ***: "
                                     + e.getMessage());
        }
        logger.debug(" <<< insertRowSet");
    }*/

    /**
     * Adds a tuple to a resultset writer.
     * 
     * @param writer the RSXMLWriter to add results to
     * @param key the tuple's key
     * @param id the tuple's value id
     */
    private void addToResults(RSXMLWriter writer, Object key, long id) throws Exception {
    	String xmlResult = null;

		String resolvedKey = XMLTokenReplacer.XMLResolve(key.toString());
    	
		xmlResult = "<tuple>";
		xmlResult += "<key>";
		xmlResult += resolvedKey;
		xmlResult += "</key>";
		xmlResult += "<valueid>";
		xmlResult += id;
		xmlResult += "</valueid>";
		xmlResult += "</tuple>";

		writer.addResults(new ResultElementGeneric("1", "1", xmlResult));
		
    	logger.debug("=================================");
    	logger.debug("BDBWrapper added key/dataID: " + xmlResult);
    	logger.debug("=================================");
    }
    
    /**
     * Adds a key to a resultset writer.
     * 
     * @param writer the RSXMLWriter to add results to
     * @param key the tuple's key
     */
    private void addToResults(RSXMLWriter writer, Object key) throws Exception {
    	String resolvedKey = XMLTokenReplacer.XMLResolve(key.toString());
		
    	String xmlResult = "<key>" + resolvedKey + "</key>";
		writer.addResults(new ResultElementGeneric("1", "1", xmlResult));

    	logger.debug("=================================");
    	logger.debug("BDBWrapper added key: " + xmlResult);
    	logger.debug("=================================");
    }
    
    private void addData(RSXMLWriter writer, Vector<String> data, boolean bDataOnly) throws Exception {
    	
    	for(int i=0; i<data.size(); i++)
    	{
    		logger.debug("adding " + i + " data element to RS");
    		
    		long time1 = Calendar.getInstance().getTimeInMillis();
    		long time2 = Calendar.getInstance().getTimeInMillis();
    		String stringXML = XMLTokenReplacer.XMLResolve(data.get(i));
    		
    		time1 = time2;
    		time2 = Calendar.getInstance().getTimeInMillis();
    		logger.debug("XMLResolve after " + (time2-time1));
    		
    		time1 = time2;
    		time2 = Calendar.getInstance().getTimeInMillis();
    		logger.debug("RS_fromXML after " + (time2-time1));
    		
    		String xmlResult = null;
    		if(!bDataOnly){
    			xmlResult = "<value>" + stringXML + "</value>";
    			time1 = time2;
        		time2 = Calendar.getInstance().getTimeInMillis();
        		logger.debug("String concat after " + (time2-time1));
        	}else{
    			xmlResult = stringXML;
    		}
    		
    		ResultElementGeneric tmp= new ResultElementGeneric("1","1",xmlResult);
    		time1 = time2;
    		time2 = Calendar.getInstance().getTimeInMillis();
    		logger.debug("new ResultElementGeneric after" + (time2-time1));
    		
    		//tmp.setRecordAttributes(element.getRecordAttributes());
    		time1 = time2;
    		time2 = Calendar.getInstance().getTimeInMillis();
    		logger.debug("set atributes after " + (time2-time1));
    		
    		writer.addResults(tmp);
    		time1 = time2;
    		time2 = Calendar.getInstance().getTimeInMillis();
    		logger.debug("added to RS writer after " + (time2-time1));
    		logger.debug("=================================");
    		logger.debug("BDBWrapper added data. Local iteration: " + i);
    		logger.debug("=================================");
    	}
    }
    
    /** 
     * Gets the values where the key equals the input key.
     * @param key - The equality key for the search
     * @param bKeysOnly return only keys, or full tuples?
     * @return String - The EPR of the result set containing the result.
     * @throws IndexException - when IOException is received.
     */
    public String getEQ(final String keyName, final Object key, final ResultType resType) throws IndexException {
    	RSLocator locator = null;

    	try {
	    	int maxHits = 1000;
	    	PropertyElementBase[] properties = {new PropertyElementForwardIndex(maxHits)};
	 	    final RSXMLWriter rsWriter =  RSXMLWriter.getRSXMLWriter(properties);
	 	    locator = rsWriter.getRSLocator(new RSResourceWSRFType());
	 	    	 	    
	 	    (new Thread() {
	 	    	public void run() {
	 	    		logger.debug(" >>> getEQ for key: "+ key);
	 	    		
	 	    		try {
	 	    			scanIndex(rsWriter, keyName, resType, key, true, key, true, false);
	 	    			rsWriter.close();
	 	    		}
	 	    		catch(Exception e){
	 	    			logger.error(" *** getEQ error key: " + key + " message: " + e.getMessage());
	 	    			try {
    						rsWriter.close();
    					}
    					catch (Exception e1) { }
	 	    		}
	 	    		logger.debug(" <<< getEQ");
	 	    	}
	 	    }).start();
 	    } catch (Exception e) {
 	    	logger.error(e);
 	    }
 	    return locator.getLocator();
    }
    
    /** 
     * Gets values >= keyGE.
     * @param keyGE - the key
     * @param bKeysOnly return only keys, or full tuples?
     * @return String - The EPR of the result set containing the result.
     * @throws IndexException - when IOException is received.
     */
    public String getGE(final String keyName, final Object keyGE, final ResultType resType) throws IndexException {
    	RSLocator locator = null;

    	try {
    		int maxHits = 1000;
    		PropertyElementBase[] properties = {new PropertyElementForwardIndex(maxHits)};
    		final RSXMLWriter rsWriter =  RSXMLWriter.getRSXMLWriter(properties);
    		locator = rsWriter.getRSLocator(new RSResourceWSRFType());

    		(new Thread() {
    			public void run() {
    				logger.debug(" >>> getGE for key: "+ keyGE);
    				
    				try {
    					scanIndex(rsWriter, keyName, resType, keyGE, true, null, false, false);
    					rsWriter.close();
    				} 
    				catch(Exception e){
    					logger.error(" *** getGE error for key: "+ keyGE + " message: " + e.getMessage());
    					try {
    						rsWriter.close();
    					}
    					catch (Exception e1) { }
    				}
    				logger.debug(" <<< getGE");
    			}
    		}).start();
    	} catch (Exception e) {
    		logger.error(e);
    	}
    	return locator.getLocator();
    }
    
    /** 
     * Gets keyGT < values 
     * @param keyGT - The lower exclusive bound key
     * @param bKeysOnly return only keys, or full tuples?
     * @return String - The EPR of the result set containing the result.
     * @throws IndexException - when IOException is received.
     */
    public String getGT(final String keyName, final Object keyGT, final ResultType resType) throws IndexException {
    	RSLocator locator = null;

    	try {
    		int maxHits = 1000;
    		PropertyElementBase[] properties = {new PropertyElementForwardIndex(maxHits)};
    		final RSXMLWriter rsWriter =  RSXMLWriter.getRSXMLWriter(properties);
    		locator = rsWriter.getRSLocator(new RSResourceWSRFType());

    		(new Thread() {
    			public void run() {
    				logger.debug(" >>> getGT for key: "+ keyGT);
    				
    				try {
    					scanIndex(rsWriter, keyName, resType, keyGT, false, null, false, false);
    					rsWriter.close();
    				} 
    				catch(Exception e){
    					logger.error(" *** getGT error for key: "+ keyGT + " message: " + e.getMessage());
    					try {
    						rsWriter.close();
    					}
    					catch (Exception e1) { }
    				}
    				logger.debug(" <<< getGT");
    			}
    		}).start();
    	} catch (Exception e) {
    		logger.error(e);
    	}
    	return locator.getLocator();
    }

    /** 
     * Gets values <= keyLE 
     * @param keyLE - The upper inclusive bound key
     * @param bKeysOnly return only keys, or full tuples?
     * @return String - The EPR of the result set containing the result.
     * @throws IndexException - when IOException is received.
     */
    public String getLE(final String keyName, final Object keyLE, final ResultType resType) throws IndexException {
    	RSLocator locator = null;

    	try {
    		int maxHits = 1000;
    		PropertyElementBase[] properties = {new PropertyElementForwardIndex(maxHits)};
    		final RSXMLWriter rsWriter =  RSXMLWriter.getRSXMLWriter(properties);
    		locator = rsWriter.getRSLocator(new RSResourceWSRFType());

    		(new Thread() {
    			public void run() {
    				logger.debug(" >>> getLE for key: "+ keyLE);
    				
    				try {
    					scanIndex(rsWriter, keyName, resType, null, false, keyLE, true, false);
    					rsWriter.close();
    				} 
    				catch(Exception e){
    					logger.error(" *** getLE error for key: "+ keyLE + " message: " + e.getMessage());
    					try {
    						rsWriter.close();
    					}
    					catch (Exception e1) { }
    				}
    				logger.debug(" <<< getLE");
    			}
    		}).start();
    	} catch (Exception e) {
    		logger.error(e);
    	}
    	return locator.getLocator();
    }
    
    /** 
     * Gets values < keyLT 
     * @param keyLT - The upper exclusive bound key
     * @param bKeysOnly return only keys, or full tuples?
     * @return String - The EPR of the result set containing the result.
     * @throws IndexException - when IOException is received.
     */
    public String getLT(final String keyName, final Object keyLT, final ResultType resType) throws IndexException {
    	RSLocator locator = null;

    	try {
    		int maxHits = 1000;
    		PropertyElementBase[] properties = {new PropertyElementForwardIndex(maxHits)};
    		final RSXMLWriter rsWriter =  RSXMLWriter.getRSXMLWriter(properties);
    		locator = rsWriter.getRSLocator(new RSResourceWSRFType());

    		(new Thread() {
    			public void run() {
    				logger.debug(" >>> getLT for key: "+ keyLT);
    				
    				try {
    					scanIndex(rsWriter, keyName, resType, null, false, keyLT, false, false);
    					rsWriter.close();
    				} 
    				catch(Exception e){
    					logger.error(" *** getLT error for key: "+ keyLT + " message: " + e.getMessage());
    					try {
    						rsWriter.close();
    					}
    					catch (Exception e1) { }
    				}
    				logger.debug(" <<< getLT");
    			}
    		}).start();
    	} catch (Exception e) {
    		logger.error(e);
    	}
    	return locator.getLocator();
    }
    
    /** 
     * Gets keyGE <= values <= keyLE 
     * @param keyGE - The lower inclusive bound
     * @param keyLE - The upper inclusive bound
     * @param bKeysOnly return only keys, or full tuples?
     * @return String - The EPR of the result set containing the result.
     * @throws IndexException - when IOException is received.
     */
    public String getGEandLE(final String keyName, final Object keyGE, final Object keyLE, final ResultType resType) throws IndexException {
    	RSLocator locator = null;

    	try {
    		int maxHits = 1000;
    		PropertyElementBase[] properties = {new PropertyElementForwardIndex(maxHits)};
    		final RSXMLWriter rsWriter =  RSXMLWriter.getRSXMLWriter(properties);
    		locator = rsWriter.getRSLocator(new RSResourceWSRFType());

    		(new Thread() {
    			public void run() {
    				logger.debug(" >>> getGEandLE for keyGE "+ keyGE + " keyLE " + keyLE);
    				
    				try {
    					scanIndex(rsWriter, keyName, resType, keyGE, true, keyLE, true, false);
    					rsWriter.close();
    				} 
    				catch(Exception e){
    					logger.error(" *** getGEandLE for keyGE and keyLE: "+ keyGE 
    	                         + " / " + keyLE + " message: " + e.getMessage());
    					try {
    						rsWriter.close();
    					}
    					catch (Exception e1) { }
    				}
    				logger.debug(" <<< getGEandLE");
    			}
    		}).start();
    	} catch (Exception e) {
    		logger.error(e);
    	}
    	return locator.getLocator();
    }
    
    /** 
     * Gets keyGE <= values < keyLT 
     * @param keyGE - The lower inclusive bound key
     * @param keyLT - The upper exclusive bound key
     * @param bKeysOnly return only keys, or full tuples?
     * @return String - The EPR of the result set containing the result.
     * @throws IndexException - when IOException is received.
     */
    public String getGEandLT(final String keyName, final Object keyGE, final Object keyLT, final ResultType resType) throws IndexException {
    	RSLocator locator = null;

    	try {
    		int maxHits = 1000;
    		PropertyElementBase[] properties = {new PropertyElementForwardIndex(maxHits)};
    		final RSXMLWriter rsWriter =  RSXMLWriter.getRSXMLWriter(properties);
    		locator = rsWriter.getRSLocator(new RSResourceWSRFType());

    		(new Thread() {
    			public void run() {
    				logger.debug(" >>> getGEandLT for keyGE "+ keyGE + " keyLT " + keyLT);
    				
    				try {
    					scanIndex(rsWriter, keyName, resType, keyGE, true, keyLT, false, false);
    					rsWriter.close();
    				} 
    				catch(Exception e){
    					logger.error(" *** getGEandLT for keyGE and keyLT: "
    	                         + keyGE + " / " + keyLT + " message: " + e.getMessage());
    					try {
    						rsWriter.close();
    					}
    					catch (Exception e1) { }
    				}
    				logger.debug(" <<< getGEandLT");
    			}
    		}).start();
    	} catch (Exception e) {
    		logger.error(e);
    	}
    	return locator.getLocator();
    }
    
    /** 
     * Gets keyGT < values <= keyLE 
     * @param keyGT - The lower exclusive bound
     * @param keyLE - The upper inclusive bound
     * @param bKeysOnly return only keys, or full tuples?
     * @return String - The EPR of the result set containing the result.
     * @throws IndexException - when IOException is received.
     */
    public String getGTandLE(final String keyName, final Object keyGT, final Object keyLE, final ResultType resType) throws IndexException {
    	RSLocator locator = null;

    	try {
    		int maxHits = 1000;
    		PropertyElementBase[] properties = {new PropertyElementForwardIndex(maxHits)};
    		final RSXMLWriter rsWriter =  RSXMLWriter.getRSXMLWriter(properties);
    		locator = rsWriter.getRSLocator(new RSResourceWSRFType());

    		(new Thread() {
    			public void run() {
    				logger.debug(" >>> getGTandLE for keyGT "+ keyGT + " keyLE " + keyLE);
    				
    				try {
    					scanIndex(rsWriter, keyName, resType, keyGT, false, keyLE, true, false);
    					rsWriter.close();
    				} 
    				catch(Exception e){
    					logger.error(" *** getGTandLE for keyGT and keyLE: "
    	                         + keyGT + " / " + keyLE + " message: " + e.getMessage());
    					try {
    						rsWriter.close();
    					}
    					catch (Exception e1) { }
    				}
    				logger.debug(" <<< getGTandLE");
    			}
    		}).start();
    	} catch (Exception e) {
    		logger.error(e);
    	}
    	return locator.getLocator();
    }
    
    /** 
     * Gets keyGT < values < keyLT 
     * @param keyGT - The lower exclusive bound key
     * @param keyLT - The upper exclusive bound key
     * @param bKeysOnly return only keys, or full tuples?
     * @return String - The EPR of the result set containing the result.
     * @throws IndexException - when IOException is received.
     */
    public String getGTandLT(final String keyName, final Object keyGT, final Object keyLT, final ResultType resType) throws IndexException {
    	RSLocator locator = null;

    	try {
    		int maxHits = 1000;
    		PropertyElementBase[] properties = {new PropertyElementForwardIndex(maxHits)};
    		final RSXMLWriter rsWriter =  RSXMLWriter.getRSXMLWriter(properties);
    		locator = rsWriter.getRSLocator(new RSResourceWSRFType());

    		(new Thread() {
    			public void run() {
    				logger.debug(" >>> getGTandLT for keyGT "+ keyGT + " keyLT " + keyLT);
    				
    				try {
    					scanIndex(rsWriter, keyName, resType, keyGT, false, keyLT, false, false);
    					rsWriter.close();
    				} 
    				catch(Exception e){
    					logger.error(" *** getGTandLT for keyGT and keyLT: "
    	                         + keyGT + " / " + keyLT + " message: " + e.getMessage());
    					try {
    						rsWriter.close();
    					}
    					catch (Exception e1) { }
    				}
    				logger.debug(" >>> getGTandLT");
    			}
    		}).start();
    	} catch (Exception e) {
    		logger.error(e);
    	}
    	return locator.getLocator();
    }
    
    /** 
     * Gets all.
     * @param bKeysOnly return only keys, or full tuples?
     * @return String - The EPR of the result set containing the result.
     * @throws IndexException - when IOException is received.
     */
    public String getAllDec(final String keyName, final ResultType resType) throws IndexException {
    	RSLocator locator = null;

    	try {
    		int maxHits = 1000;
    		PropertyElementBase[] properties = {new PropertyElementForwardIndex(maxHits)};
    		final RSXMLWriter rsWriter =  RSXMLWriter.getRSXMLWriter(properties);
    		locator = rsWriter.getRSLocator(new RSResourceWSRFType());

    		(new Thread() {
    			public void run() {
    				logger.debug(" >>> getAllDec");
    				
    				try {
    					scanIndex(rsWriter, keyName, resType, null, false, null, false, true);
    					rsWriter.close();
    				} 
    				catch(Exception e){
    					logger.error(" *** getAllDec  message: " + e.getMessage());
    					try {
    						rsWriter.close();
    					}
    					catch (Exception e1) { }
    				}
    				logger.debug(" >>> getAllDec");
    			}
    		}).start();
    	} catch (Exception e) {
    		logger.error(e);
    	}
    	return locator.getLocator();
    }
    
    /** 
     * Gets all.
     * @param bKeysOnly return only keys, or full tuples?
     * @return String - The EPR of the result set containing the result.
     * @throws IndexException - when IOException is received.
     */
    public String getAllInc(final String keyName, final ResultType resType) throws IndexException {
    	RSLocator locator = null;

    	try {
    		int maxHits = 1000;
    		PropertyElementBase[] properties = {new PropertyElementForwardIndex(maxHits)};
    		final RSXMLWriter rsWriter =  RSXMLWriter.getRSXMLWriter(properties);
    		locator = rsWriter.getRSLocator(new RSResourceWSRFType());

    		(new Thread() {
    			public void run() {
    				logger.debug(" >>> getAllInc");
    				
    				try {    					
    					scanIndex(rsWriter, keyName, resType, null, false, null, false, false);
    					rsWriter.close();
    				}
    				catch(Exception e){
    					logger.error(" *** getAllInc  message: " + e.getMessage());
    					try {
    						rsWriter.close();
    					}
    					catch (Exception e1) { }
    				}
    				logger.debug(" >>> getAllInc");
    			}
    		}).start();
    	} catch (Exception e) {
    		logger.error(e);
    	}
    	return locator.getLocator();
    }
    
    private void scanIndex(RSXMLWriter rsWriter, String keyName, ResultType resType, Object left, boolean gequal, Object right, boolean lequal, boolean dec) throws Exception{
    	//get type of results
    	boolean bDataOnly = false;
    	boolean bKeysOnly = false;
    	if(resType == ResultType.KEYS)
 	       	bKeysOnly = true;
 	    else if(resType == ResultType.DATA)
 	    	bDataOnly = true;	    
    	
    	//get info for this keyName
		Integer pos = keyTypes.get(keyName);
		if(pos == null)
		{
			throw new IndexException("keyName: " + keyName + " was not declared when the constructor was called. " +
			"There is no index for this keyName");
		}
		DatabaseEntry greater = null;
		DatabaseEntry lower = null;
		if(left != null){
			greater =  new DatabaseEntry();
			if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.DATE)
				greater.setData(convertToDate(left, dbT[pos].getKeyField().getDataTypeFormat()).getBytes());
			else if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.FLOAT)
				FloatBinding.floatToEntry(convertToFloat(left), greater);
			else if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.INT)      
				IntegerBinding.intToEntry(convertToInt(left), greater);
			else if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.STRING)        
				StringBinding.stringToEntry((String)left, greater);
		}
		if(right != null){
			lower =  new DatabaseEntry();
			if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.DATE)
				lower.setData(convertToDate(right, dbT[pos].getKeyField().getDataTypeFormat()).getBytes());
			else if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.FLOAT)
				FloatBinding.floatToEntry(convertToFloat(right), lower);
			else if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.INT)      
				IntegerBinding.intToEntry(convertToInt(right), lower);
			else if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.STRING)        
				StringBinding.stringToEntry((String)right, lower);
		}
		//variables for browsing data
		HashMap<String, Long> sentData = new HashMap<String, Long>();
		Vector<String> current = new Vector<String>();
		long currentLength = 0;
		long id = 1;
		DatabaseEntry keyValue = new DatabaseEntry();
		DatabaseEntry bindKey = new DatabaseEntry();
		DatabaseEntry value = new DatabaseEntry();
		Object key = null;
		OperationStatus opSt = OperationStatus.SUCCESS;
		//new cursor for this index
		Cursor cursor = db[pos].openCursor(null, null);
		try{
		//Move the cursor to a specific point if we want results greater than a key
		if(greater != null)
		{
			keyValue.setData(greater.getData());
			opSt = cursor.getSearchKeyRange(keyValue, bindKey, null);
			//if we don't want equal keys
			if(!gequal)
			{
				if(comparator[pos].compare(keyValue.getData(), greater.getData()) == 0)
				{
					opSt = cursor.getNextNoDup(keyValue, bindKey, null);
				}
			}
		//If we want getAllDec
		}else if(dec){
			opSt = cursor.getPrev(keyValue, bindKey, null);
		}else{
			opSt = cursor.getNext(keyValue, bindKey, null);
		}
		while(opSt == OperationStatus.SUCCESS)
		{
			//if we don't want results greater than lower
			if(lower != null)
			{
				if(comparator[pos].compare(keyValue.getData(), lower.getData()) > 0)
					break;
				if(comparator[pos].compare(keyValue.getData(), lower.getData())==0 && !lequal)
					break;
			}
			//find if this data item is already sent
			Long dataId = sentData.get(StringBinding.entryToString(bindKey));
			if(dataId == null)
				logger.debug("sentData.get() returned null for valueid(bindKey)"+ StringBinding.entryToString(bindKey) + "(" + bindKey + ")");
			else
				logger.debug("sentData.get() returned: " + dataId + " for valueid(bindKey)"+ StringBinding.entryToString(bindKey) + "(" + bindKey + ")");
			//if not and we don't want keys only
			if(dataId == null && !bKeysOnly)
			{
				logger.debug("valueID(bindkey):" + StringBinding.entryToString(bindKey) + "(" + bindKey + ") is added to sentData");
				//get the data item from the Main Index
				if(dbMain.get(null, bindKey, value, null) == OperationStatus.NOTFOUND)
				{
    				logger.error("Reference to actual data not found. Index is corrupted!");
    				throw new Exception("Reference to actual data not found. Index is corrupted!");
    			}
				//add the data item for sending it in the current package
				String s = StringBinding.entryToString(value);
				s = extractValueFromEntry(s);
				current.add(s);
				//add the item to the sent data
				sentData.put(StringBinding.entryToString(bindKey), id);
				
				//add results to the Result Set
				if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.DATE)
		        	key = CustomDate.toCustomDate(keyValue.getData());
		        else if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.FLOAT)
		        	key = FloatBinding.entryToFloat(keyValue); 
		        else if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.INT)      
		        	key = IntegerBinding.entryToInt(keyValue);
		        else if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.STRING)        
		        	key = StringBinding.entryToString(keyValue);
				if(!bDataOnly)
					addToResults(rsWriter, key, id);
				logger.debug("Found key/id: " + key + "/" + StringBinding.entryToString(bindKey));
				//increment the counter for tagging data items
				id++;
			//if it is sent or we want keys only	
			}else{
				logger.debug("valueID(bindkey):" + StringBinding.entryToString(bindKey) + "(" + bindKey + ") is already added to sentData");
				//TMP
				/*Iterator<String> iter = sentData.keySet().iterator();
				while(iter.hasNext())
					logger.debug("iter.next():"+iter.next());*/
				//add results to the Result Set
				if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.DATE)
		        	key = CustomDate.toCustomDate(keyValue.getData());
		        else if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.FLOAT)
		        	key = FloatBinding.entryToFloat(keyValue); 
		        else if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.INT)      
		        	key = IntegerBinding.entryToInt(keyValue);
		        else if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.STRING)        
		        	key = StringBinding.entryToString(keyValue);
				if(bKeysOnly){
					addToResults(rsWriter, key);
				}else if(!bDataOnly){
					addToResults(rsWriter, key, dataId);
				}
				logger.debug("Found key/id: " + key + "/" + StringBinding.entryToString(bindKey));
			}
			currentLength++;
			//if we have reached the end of this package send data 
			if(currentLength == packageSize && !bKeysOnly)
			{
				addData(rsWriter, current, bDataOnly);
				currentLength = 0;
				current.clear();
			}
			if(dec)
				if(bKeysOnly)
					opSt = cursor.getPrevNoDup(keyValue, bindKey, null);
				else
					opSt = cursor.getPrev(keyValue, bindKey, null);
			else
				if(bKeysOnly)
					opSt = cursor.getNextNoDup(keyValue, bindKey, null);
				else
					opSt = cursor.getNext(keyValue, bindKey, null);
		}
		//Send remaining data 
		if(current.size() > 0 && !bKeysOnly)
			addData(rsWriter, current, bDataOnly);
			
		cursor.close();
		}catch(Exception e){
			cursor.close();
			throw e;
		}
    }
    
    public URI processQuery(String cqlQuery, ArrayList<String> presentable, 
    		ArrayList<String> searchable, RRadaptor adaptor) throws GRS2WriterException{
    	
    	RecordWriter<GenericRecord> rsWriter = null;
    	this.adaptor  = adaptor;
		
    	try {
    		BDBGcqlProcessor processor = new BDBGcqlProcessor();
    		
    		BDBGcqlQueryContainer queryCont = 
    			(BDBGcqlQueryContainer)processor.processQuery(presentable, searchable, cqlQuery, adaptor);
    		ArrayList<?> queries = queryCont.getBdbQueries();
    		LinkedHashMap<String, String> projections = queryCont.getProjectedFields();
    		
    		//if there are no projections return all the returnable fields
    		/*if(projections == null || projections.size() > 0) {
    			projections = (ArrayList<String>)presentable.clone();
    		}*/
    		    		
    		boolean distinct = queryCont.getDistinct();
    		//give the definition for the record
    		FieldDefinition[] fieldDef = createFieldDefinition(projections, distinct);
    		RecordDefinition[] definition = new RecordDefinition[]{new GenericRecordDefinition(fieldDef)};
    		rsWriter = new RecordWriter<GenericRecord>(new TCPWriterProxy(), definition);
			
    		(new BDBQueryExecutor(keyTypes, dbT, db, dbMain, comparator, queries, rsWriter, projections, distinct)).start();
    		  		    		
    	} catch (Exception e) {
    		logger.error(e);
    	}
    	return rsWriter.getLocator();
    }
    
    private FieldDefinition[] createFieldDefinition(LinkedHashMap<String, String> projections, boolean distinct) throws Exception{
    	ArrayList<FieldDefinition> fieldDef = new ArrayList<FieldDefinition>();
		
    	//if it is not the distinct case
    	if(!distinct) {
	    	//add a field for the docID
			fieldDef.add(new StringFieldDefinition(IndexType.DOCID_FIELD));
    	}
		
		for(Entry<String, String> current : projections.entrySet()) {
			fieldDef.add(new StringFieldDefinition(current.getKey()));
		}
		
		return fieldDef.toArray(new FieldDefinition[fieldDef.size()]);
    }
    
    /**
     * THIS METHOD IS NEVER CALLED IN FORWARD INDEX.
     * Merge deletion is used to update the index with documents to be deleted.
     * @param deletionFile - the file with the content to delete .
     * @throws IndexException - in case of failure.
     *
     */
    public synchronized int mergeDeletion(File deletionFile) throws IndexException {
        try {
            logger.debug(" >>> mergeDeletion");
            logger.debug(" <<< mergeDeletion");
            return 0;
        }
        catch (Exception ex) {
            logger.error(" *** mergeDeletion error  message: " + ex.getMessage());
            return 1;
        }
    }
    
    static int numberOfChars(String input, String countString) {
    	return input.split("\\Q"+countString+"\\E", -1).length - 1;
    }


    /**
     * Receives the file with rowsets that is to be merged into the index.
     * @param inputIndex - The file with rowsets that is merged into the index.
     * @return - number of added documents.
     * @throws IndexException - in case of error.
     */
    public synchronized void mergeAddition(File inputIndex)
        throws IndexException {
        logger.debug(" >>> mergeAddition file:" + inputIndex);
        int index = 0;
        try {
            FileInputStream inputStream = new FileInputStream(inputIndex);
            int available = inputStream.available();
            ReadableByteChannel deltaChannel = inputStream.getChannel();
            if (available > 0) {
                ByteBuffer buffer = ByteBuffer.allocate(available);
                int bytesRead = deltaChannel.read(buffer);
                buffer.flip();
                byte[] c = new byte[buffer.capacity()];
                buffer.get(c);
                String rowSet = "<ROWSET>" + new String(c, "UTF-8") + "</ROWSET>";
                logger.debug("    rowset\n" 
                             + rowSet);

                // Parse
                UpdateXMLParser parser = new UpdateXMLParser();
                parser.parse(rowSet);
                // Flush out to disk - It seems that sync is not enough. We must close the handlers
                // this.dbMain.sync();
                // for(int i=0; i<this.db.length; i++)
                //	this.db[i].sync();
                // close database, environment handlers
                for(int i=0; i<this.db.length; i++)
                	this.db[i].close();
                this.dbMain.close();
                this.env.close();
                // re-open them
                this.env = new Environment(envDir, envConfig);
                this.dbMain = env.openDatabase(null, convertToIndexNameMain(contentName), dbConfigMain);
                for(int i=0; i<this.db.length; i++)
                	this.db[i] = env.openDatabase(null, convertToIndexName(keyNames[i]), dbConfig[i]);
                
                logger.debug(" <<< mergeAddition");
            }
            else {
                logger.error(" *** mergeAddition empty file, available bytes: " + available);
                throw new IndexException(" Empty insert file received");
            }
        }
        catch(Exception ex) {
            logger.error(" *** mergeAddition error  message: " + ex.getMessage());
        }
    }

    /**
     * Adds a tuple to the index.
     * 
     * @param key the tuple's key
     * @param value the tuple's value
     */
    private void addTuple(Vector<String> keyNames, Vector<String> keyValues, String value) throws Exception {
    	
    	DatabaseEntry bindKey = new DatabaseEntry();
    	DatabaseEntry valueEntry = new DatabaseEntry();
    	           	
    	logger.debug("=================================");
    	logger.debug("BDBWrapper.addTuple merging current index with a new tuple");
    	logger.debug("=================================");
    	
    	if(keyNames.size() != keyValues.size())
    	{
    		logger.error("keyNames don't have the same size with keyValues");
    		return;
    	}
    	
    	//get the language of this tuple
    	String lang = LanguageIdPlugin.LANG_UNKNOWN;
    	for(int i=0; i<keyNames.size(); i++)
    	{
    		if(keyNames.get(i).equals(IndexType.LANGUAGE_FIELD))
    		{
    			lang = keyValues.get(i);
    		}
    	}
    	
    	//produce bindKey
    	String stringXML = XMLTokenReplacer.XMLResolve(value);
		String docID = getDocIDFromXML(stringXML);
		docID = enhanceDocID(docID, lang);
		StringBinding.stringToEntry(docID, bindKey);
		
		logger.debug("adding value with id: " + docID);
         
		int additions = 0;
    	//place an entry in the corresponding index for each keyname-keyvalue
    	for(int i=0; i<keyNames.size(); i++)
    		if(insertPairInDb(keyNames.get(i), keyValues.get(i), bindKey))
    			additions++;
    	
    	logger.debug("addition of keys completed");
                
    	//check if there is already a value with this docID
    	if(dbMain.get(null, bindKey, valueEntry, null) == OperationStatus.NOTFOUND)
		{
    		logger.debug("this DocID is inserted for the first time");
    		//construct the value to be inserted
    		value = convertValueToEntry(additions, value);
        	StringBinding.stringToEntry(value, valueEntry);
        	logger.debug("inserting DocID: "+docID+ ", to the value index");
    		//place the actual data in the Main index    	
        	if(this.dbMain.put(null, bindKey, valueEntry) == OperationStatus.SUCCESS)
        	{
        		this.numDocs++;
        		logger.debug("DocID: "+docID+", inserted successfully");
        	}
        	else
        	{
        		logger.error("could not put actual data to value store: " + value);
        		throw new Exception("could not put actual data to value store: " + value);
        	}
		}else{
			logger.debug("this DocID is already inserted");
    		//construct the value to be inserted	
			String s = StringBinding.entryToString(valueEntry);
			int refs = extractRefsFromEntry(s)+additions;
    		value = convertValueToEntry(refs, value);
        	StringBinding.stringToEntry(value, valueEntry);
        	logger.debug("inserting the updated entry for DocID: "+docID+ ", to the value index");
    		//place the actual data in the Main index    	
        	if(this.dbMain.put(null, bindKey, valueEntry) != OperationStatus.SUCCESS)
        	{
        		logger.error("could not put actual data to value store: " + value);
        		throw new Exception("could not put actual data to value store: " + value);
        	}else{
        		logger.debug("updated entry for DocID: "+docID+", inserted successfully");
        	}
        }    	
    	    	
    }
    
    /**
     * Parses the XML containing the presentable information for a document
     * (FORMAT: <FIELD name="fieldname"> fieldvalue </FIELD>) and returns 
     * a map containing the field name - value pairs.
     * @param fieldsString - XML containing the presentable information for a document
     * @return {@link HashMap} containing the field pairs
     */
    public static HashMap<String, String> getPresentationFieldsFromXML(
			String fieldsString) {
		HashMap<String, String> fields =  new HashMap<String, String>();
		
		//create a new builder to process the string
		StringBuilder tmpBuilder =  new StringBuilder(fieldsString);
		
		//for all the fields
		while(tmpBuilder.indexOf(">") > -1) {
			
			//find the name
			int from = tmpBuilder.indexOf("name=\"");
			if(from == -1)
				break;
			
			from += "name=\"".length();
			tmpBuilder.replace(0, from, "");
			int to = tmpBuilder.indexOf("\"");
			String name = tmpBuilder.substring(0, to).trim();
			tmpBuilder.replace(0, to, "");
			
			//find the value
			from = tmpBuilder.indexOf(">") + ">".length();
			tmpBuilder.replace(0, from, "");
			to = tmpBuilder.indexOf("<");
			if(to == -1)
				break;
			
			String value = tmpBuilder.substring(0, to).trim();
			tmpBuilder.replace(0, to, "");
			
			//put the field in the map
			if(name != null && !name.equals("") && value != null && !value.equals("")) {
				
				String sumValue = fields.get(name);
				if(sumValue == null) {
					sumValue = value;
				} else {
					sumValue += (" " + value);
				}
				
				fields.put(name, sumValue);
			}
		}
		
		return fields;
	}
    
    /**
     * Parses the XML containing the presentable information for a document
     * (FORMAT: <FIELD name="fieldname"> fieldvalue </FIELD>) and returns 
     * an array containing the unique field names.
     * @param fieldsString - XML containing the presentable information for a document
     * @return {@link ArrayList} containing the unique field names
     */
    public static ArrayList<String> getPresentableFromXML(
			String fieldsString, String colID, String colLang) {
    	HashSet<String> presentable = new HashSet<String>();
		
		//create a new builder to process the string
		StringBuilder tmpBuilder =  new StringBuilder(fieldsString);
		
		//for all the fields
		while(tmpBuilder.indexOf(">") > -1) {
			
			//find the name
			int from = tmpBuilder.indexOf("name=\"") + "name=\"".length();
			tmpBuilder.replace(0, from, "");
			int to = tmpBuilder.indexOf("\"");
			String name = tmpBuilder.substring(0, to).trim();
			tmpBuilder.replace(0, to, "");
			
			//put the name in the set
			if(name != null) {
				
				//The docID is chosen to be a default attribute in each result, 
				//so we won't add it to the presentable fields
				if(name.equalsIgnoreCase(IndexType.DOCID_FIELD))
					continue;
				
				presentable.add(colID + IndexType.SEPERATOR_FIELD_INFO 
    					+ colLang + IndexType.SEPERATOR_FIELD_INFO 
    					+ IndexType.PRESENTABLE_TAG + IndexType.SEPERATOR_FIELD_INFO
    					+ name);
			}
		}
		
		return new ArrayList<String>(presentable);
	}
    
    static String getDocIDFromXML(
			String fieldsString) {
		HashMap<String, String> fields =  new HashMap<String, String>();
		
		//create a new builder to process the string
		StringBuilder tmpBuilder =  new StringBuilder(fieldsString);
		
		//for all the fields
		while(tmpBuilder.indexOf(">") > -1) {
			
			//find the name
			int from = tmpBuilder.indexOf("name=\"") + "name=\"".length();
			tmpBuilder.replace(0, from, "");
			int to = tmpBuilder.indexOf("\"");
			String name = tmpBuilder.substring(0, to).trim();
			tmpBuilder.replace(0, to, "");
			
			//find the value
			from = tmpBuilder.indexOf(">") + ">".length();
			tmpBuilder.replace(0, from, "");
			to = tmpBuilder.indexOf("<");
			String value = tmpBuilder.substring(0, to).trim();
			tmpBuilder.replace(0, to, "");
			
			//put the field in the map
			if(name.equalsIgnoreCase(IndexType.DOCID_FIELD)) {
				
				if(value == null) {
					logger.error("No docID value inside presentation field for docID");
					return NODOCID;
				}
				
				return value;
			}
		}
		logger.error("No docID value found");
		return NODOCID;
	}

	private String enhanceDocID(String docID, String lang) {
		return lang + IndexType.SEPERATOR_FIELD_INFO + docID;
	}
    
    private String filterDocID(String docID) {
		return docID.substring(docID.indexOf(IndexType.SEPERATOR_FIELD_INFO) 
				+ IndexType.SEPERATOR_FIELD_INFO.length());
	}

	private String convertValueToEntry(int refs, String value) {
		return refs + ":" + value;
	}
    
    static int extractRefsFromEntry(String value) {
		return Integer.parseInt(value.substring(0, value.indexOf(":")));
	}
    
    static String extractValueFromEntry(String value) {
		return value.substring(value.indexOf(":")+1);
	}
    
    /**
     * Deletes all the references for this id in this Index lookup resource
     * @param id - language + IndexType.SEPERATOR_FIELD_INFO{@link IndexType} + docID
     *  The parameter has changed since version 3.0.0, because a document is identified
     *  by the language and the gDocID since then. 
     * @throws Exception
     */
    private void deleteAllRefs(String id) throws Exception{
    	int refs = 0;
    	Iterator<String> iter = keyTypes.keySet().iterator();
    	while(iter.hasNext())
    	{
    		refs += deleteValueFromDb(iter.next(), id);
    	}
    	if(refs > 0)
    	{
    		DatabaseEntry value = new DatabaseEntry();
    		DatabaseEntry bindKey = new DatabaseEntry();
    		StringBinding.stringToEntry(id, bindKey);
    		if(this.dbMain.get(null, bindKey, value, null) == OperationStatus.NOTFOUND)
    		{
    			logger.error("Reference to actual data not found. Index is corrupted!");
    			throw new Exception("Reference to actual data not found. Index is corrupted!");
    		}
    		String s = StringBinding.entryToString(value);
    		int newRefs = extractRefsFromEntry(s)-refs;
    		//if this was the last reference to actual data delete them from the main index
    		if(newRefs == 0){
    			if(this.dbMain.delete(null, bindKey) == OperationStatus.SUCCESS)
    				this.numDocs--;
    		}else{
    			logger.error("Data for id: " + id + " are not in valid form. Index is corrupted!");
    			throw new Exception("Data for id: " + id + " are not in valid form. Index is corrupted!");
    		}
    	}
    }

	/**
	 * for each keyName-keyValue/id pair: delete it from its index and decrement the 
	 * references number from actual data
	 * @param keyNames - specifies the keyNames 
	 * @param keyValues - specifies the keyValues (* for all the values of this keyName)
	 * @param id - language + IndexType.SEPERATOR_FIELD_INFO{@link IndexType} + docID
     *  The parameter has changed since version 3.0.0, because a document is identified
     *  by the language and the gDocID since then.  @throws Exception
	 */
    private void deleteTuple(Vector<String> keyNames, Vector<String> keyValues, String id) throws Exception{
    	    	
    	if(keyNames.size() != keyValues.size())
    	{
    		logger.error("keyNames don't have the same size with keyValues");
    		return;
    	}
    	
    	DatabaseEntry bindKey = new DatabaseEntry();
        StringBinding.stringToEntry(id, bindKey);
    	
    	//for each keyName-keyValue/id pair: delete it from its index and decrement the 
    	//references number from actual data 
    	for(int i=0; i<keyNames.size(); i++)
    	{
    		int refs = 0;
    		if(keyValues.get(i).compareTo("*") == 0)
    			refs = deleteValueFromDb(keyNames.get(i), id);
    		else
    			refs = deletePairInDb(keyNames.get(i), keyValues.get(i), id);
    		if(refs == 0)
				continue;
			
    		DatabaseEntry value = new DatabaseEntry();
    		if(this.dbMain.get(null, bindKey, value, null) == OperationStatus.NOTFOUND)
    		{
    			logger.error("Reference to actual data not found. Index is corrupted!");
    			throw new Exception("Reference to actual data not found. Index is corrupted!");
    		}
    		String s = StringBinding.entryToString(value);
    		int newRefs = extractRefsFromEntry(s)-refs;
    		//if this was the last reference to actual data delete them from the main index
    		if(newRefs == 0){
    			if(this.dbMain.delete(null, bindKey) == OperationStatus.SUCCESS)
    				this.numDocs--;
    		}else if(newRefs < 0){
    			logger.error("Data for id: " + id + " are not in valid form. Index is corrupted!");
    			throw new Exception("Data for id: " + id + " are not in valid form. Index is corrupted!");
    		}else{
    			s = convertValueToEntry(newRefs, extractValueFromEntry(s));
    			StringBinding.stringToEntry(s, value);
    			this.dbMain.put(null, bindKey, value);
    		}
    		    				
    	}
    	
    }
    
    /** 
     * Converts the String to an int value
     * @param in - the object to convert to int
     * @return - the int value of the String input
     * @throws -IndexException in case of failure
     */
    static final int convertToInt(Object in) throws IndexException {
        logger.debug(" >>> convertToInt input value: " + in);
        try {
            // The input is an String 
            int res = new Integer(((String)(in))).intValue();
            // Integer res = new Integer(((String)(in)));
            logger.debug(" <<< convertToInt value: " + res);
            return res;
        }
        catch (Exception ex) {
            logger.error(" *** convertToInt error  message: " + ex.getMessage());
            throw new IndexException(" *** convertToInt error", ex);
        }
    }

    /** 
     * Converts the String to a float value
     * @param in - the object to convert to float
     * @return - the float value of the String input
     * @throws -IndexException in case of failure
     */
    static final float convertToFloat(Object in) throws IndexException {
        logger.debug(" >>> convertToFloat input value: " + in);
        try {
            // The input is an String 
            float res = new Float(((String)(in))).floatValue();
            logger.debug(" <<< convertToFloat float: " + res);
            return res;
        }
        catch (Exception ex) {
            logger.error(" *** convertToFloat error  message: " + ex.getMessage());
            throw new IndexException(" *** convertToFloat error", ex);
        }
    }

    /** 
     * Converts the String to a date value
     * @param in - the object to convert to date
     * @return - the date value of the String input
     * @throws -IndexException in case of failure
     */
    static final CustomDate convertToDate(Object in, String format) throws IndexException {
        logger.debug(" >>> convertToDate input value: " + in);
        try {
            // The input is an String 
        	SimpleDateFormat df = new SimpleDateFormat(format);            
            CustomDate res = new CustomDate((String) in, df);
            logger.debug(" <<< convertToDate date: " + res);
            return res;
        }
        catch (Exception ex) {
            logger.error(" *** convertToDate error  message: " + ex.getMessage());
            throw new IndexException(" *** convertToDate error", ex);
        }
    }

    /** 
     * Converts the Object to a string value
     * @param in - the object to convert to date
     * @return - the date value of the String input
     * @throws -IndexException in case of failure
     */
    static final String convertToString(Object in) throws IndexException {
        logger.debug(" >>> convertToString input value: " + in);
        try {
            // The input is an String
            // May be a too complicated conversion since it is already a String
            // but this method will at least throw exception if it fails
            String  res = new String((String)(in));
            logger.debug(" <<< convertToString string: " + res);
            return res;
        }
        catch (Exception ex) {
            logger.error(" *** convertToString error  message: " + ex.getMessage());
            throw new IndexException(" *** convertToString error", ex);
        }
    }

    /**
     * Updates the index.
     * NOT USED IN THE FORWARDINDEX
     * @throws IndexException - in case of error
     */
    public synchronized void updateIndex()
        throws IndexException {
        logger.debug(" >>> updateIndex");
        logger.debug(" <<< updateIndex");
        try {
            ;
        }
        catch(Exception ex) {
            logger.error(" *** updateIndex error  message: " + ex.getMessage());
            ;
        }
    }

    /**
     * Prints out the values in the iterator.
     * @param result - the resultSet to print.
     */
    private void printRs(String result) {
        // The result in the vector is a repetition of "key value"
        logger.info("Result set: " + result);
    }

    /**
     * Prints out the values in the iterator.
     * @param result - the resultSet to print.
     */
    private void printList(LinkedList result) {
        // The result in the vector is a repetition of "key value"
        int hits = 0;
        String xmlResult = "<forwardIndexResult>";
        ListIterator<Tuple> it = result.listIterator();
        while (it.hasNext()) {
            Tuple tuple = it.next();
            hits++;
            xmlResult += "<tuple>";
            xmlResult += "<key>";
            xmlResult += tuple.getKey();
            xmlResult += "</key>";
            xmlResult += "<value>";
            xmlResult += tuple.getValue();
            xmlResult += "</value>";
            xmlResult += "</tuple>";
        }
        xmlResult = xmlResult + "</forwardIndexResult>";
        logger.info(xmlResult);
    }

    /**
     * Test main.
     * @param args - the array of arguments.
     */
/*    public static void main(String[] args) {
	String res;
        boolean printRes=true;
        //        WriteResults writeResult = new WriteResults();
        DbTypeEnum dbtAll[] =DbTypeEnum.values();
        try {
            int startTime = (int) Calendar.getInstance().getTimeInMillis();
            System.out.println(" Create Object");
            DbTypeEnum dbt = DbTypeEnum.INT_TYPE;
            BDBWrapper db = new BDBWrapper(".","namedb4","nametree4",dbt);
            db.insertPairInDb(40001, 21000);
            System.out.println("Insert second 40001/21101");
            db.insertPairInDb(40001, 21101);
            System.out.println("GetEQ");
            res=db.getEQ(40001, false);
            //            writeResult.write(res,"ForwardIndexLookup",true);
            db.printRs(res);;
            db.insertPairInDb(40001, 21101);
            res=db.getEQ(40001, false);
            db.printRs(res);
            //            db.deletePairInDb(40000);
            db.deletePairInDb(40001);
            db.insertPairInDb(1,1000);
            db.insertPairInDb(100001,1000);
            int key1=0;
            for (int i = 10;i < 10000; i++) {
                db.insertPairInDb(i, i+1000);
                res=db.getEQ(i, false);
                if (printRes)
                    db.printRs(res);
                res=db.getGE(i, false);
                if (printRes)
                    db.printRs(res);
                res=db.getGT(i, false);
                if (printRes)
                    db.printRs(res);
                res=db.getLE(i, false);
                if (printRes)
                    db.printRs(res);
                res=db.getLT(i, false);
                if (printRes)
                    db.printRs(res);
                res=db.getGEandLE(key1,i, false);
                if (printRes)
                    db.printRs(res);
                res=db.getGEandLT(key1,i, false);
                if (printRes)
                    db.printRs(res);
                res=db.getGTandLE(key1,i, false);
                if (printRes)
                    db.printRs(res);
                res=db.getGTandLT(key1,i, false);
                if (printRes)
                    db.printRs(res);
            }

           long usedTime = (long) Calendar.getInstance().getTimeInMillis()-startTime;
           System.out.println(" Used time: millis: " + usedTime);
        }
        catch (Exception e) {
            e.printStackTrace();
        }
    }
*/    
    
    
    
    private enum ParserState { IDLE, IN_INSERT, IN_DELETE, IN_TUPLE, IN_KEY, IN_KEYNAME, IN_KEYVALUE, IN_VALUE, IN_ID };
 
    /**
     * Class that handles the SAX parsing of updates to the index.
     * 
     * @author Spyros Boutsis, UoA
     * @author Vasilis Verroios, UoA
     */
    class UpdateXMLParser extends DefaultHandler {
    	
    	private static final String INSERT_ELEMENT = "INSERT";
    	private static final String DELETE_ELEMENT = "DELETE";
    	private static final String TUPLE_ELEMENT = "TUPLE";
    	private static final String KEY_ELEMENT = "KEY";
    	private static final String KEYNAME_ELEMENT = "KEYNAME";
    	private static final String KEYVALUE_ELEMENT = "KEYVALUE";
    	private static final String VALUE_ELEMENT = "VALUE";
    	private static final String ID_ELEMENT = "ID";
    	
    	private ParserState currState;
    	private StringBuffer currKeyName;
    	private StringBuffer currKeyValue;
    	private StringBuffer currValue;
    	private StringBuffer currId;
    	private Vector<String> keyNames;
    	private Vector<String> keyValues;    	
    	private boolean bDeletion;
    	private boolean bKey;
    	private Exception e;
    	
    	public UpdateXMLParser() {
    		currState = ParserState.IDLE;
    		currKeyName = new StringBuffer();
    		currKeyValue = new StringBuffer();
    		currValue = new StringBuffer();
    		currId = new StringBuffer();
    		keyNames = new Vector<String>();
    		keyValues = new Vector<String>();
    		bDeletion = false;
    		e = null;
    	}
    	
    	public void parse(String updateData) throws Exception {
    		XMLReader xr = XMLReaderFactory.createXMLReader();
    		xr.setContentHandler(this);
    		xr.setErrorHandler(this);
    		xr.setFeature("http://xml.org/sax/features/validation", false);
    		xr.parse(new InputSource(new StringReader(updateData)));
    		if (e != null)
    			throw e;
    	}
    	
        public void startElement (String uri, String name, String qName, Attributes atts) {
        	boolean bError = false;
        	
        	if (qName.equalsIgnoreCase(INSERT_ELEMENT)) {
        		if (currState.equals(ParserState.IDLE)) {
        			currState = ParserState.IN_INSERT;
        			bDeletion = false;
        		}
        		else
        		{
        			logger.error("Invalid XML data in rowset. INSERT was found in a wrong place");
        			bError = true;
        		}
        	}
        	else if (qName.equalsIgnoreCase(DELETE_ELEMENT)) {
        		if (currState.equals(ParserState.IDLE)) {
        			currState = ParserState.IN_DELETE;
        			bDeletion = true;
        		}
        		else
        		{
        			logger.error("Invalid XML data in rowset. DELETE was found in a wrong place");
        			bError = true;
        		}
        	}
        	else if (qName.equalsIgnoreCase(TUPLE_ELEMENT)) {
        		if (currState.equals(ParserState.IN_INSERT) || currState.equals(ParserState.IN_DELETE))
        		{
        			currState = ParserState.IN_TUPLE;
        		}else
        		{
        			logger.error("Invalid XML data in rowset. TUPLE was found in a wrong place");
        			bError = true;
        		}
        	}
        	else if (qName.equalsIgnoreCase(KEY_ELEMENT)) {
        		if (currState.equals(ParserState.IN_TUPLE))
        		{
        			currState = ParserState.IN_KEY;
        		}
        		else
        		{
        			logger.error("Invalid XML data in rowset. KEY was found in a wrong place");
        			bError = true;
        		}
        	}
        	else if (qName.equalsIgnoreCase(KEYNAME_ELEMENT)) {
        		if (currState.equals(ParserState.IN_KEY))
        		{
        			if(bKey)
        			{
            			logger.error("Invalid XML data in rowset. bkey is already set");
            			bError = true;
        			}else{
        				bKey = true;
        				currState = ParserState.IN_KEYNAME;
        			}
        		}else
        			bError = true;
        	}
        	else if (qName.equalsIgnoreCase(KEYVALUE_ELEMENT)) {
        		if (currState.equals(ParserState.IN_KEY))
        		{
        			currState = ParserState.IN_KEYVALUE;
        		}
        		else
        		{
        			logger.error("Invalid XML data in rowset. KEYVALUE was found in a wrong place");
        			bError = true;
        		}
        	}
        	else if (qName.equalsIgnoreCase(VALUE_ELEMENT)) {
        		if (currState.equals(ParserState.IN_TUPLE))
        		{
        			currState = ParserState.IN_VALUE;
        		}
        		else
        		{
        			logger.error("Invalid XML data in rowset. VALUE was found in a wrong place");
        			bError = true;
        		}
        	}
        	else if (qName.equalsIgnoreCase(ID_ELEMENT)) {
        		if (currState.equals(ParserState.IN_TUPLE))
        		{
        			currState = ParserState.IN_ID;
        		}
        		else
        		{
        			logger.error("Invalid XML data in rowset. ID was found in a wrong place");
        			bError = true;
        		}
        	}
        	
        	if (bError)
        		e = new Exception("Invalid XML in rowset data");
        }

        public void endElement (String uri, String name, String qName) {
        	boolean bError = false;
        	
        	if (qName.equalsIgnoreCase(INSERT_ELEMENT)) {
        		if (currState.equals(ParserState.IN_INSERT)) {
        			currState = ParserState.IDLE;
        		}
        		else
        		{
        			logger.error("Invalid XML data in rowset. /INSERT was found in a wrong place");
        			bError = true;
        		}
        	}
        	else if (qName.equalsIgnoreCase(DELETE_ELEMENT)) {
        		if (currState.equals(ParserState.IN_DELETE)) {
        			currState = ParserState.IDLE;
        		}
        		else
        		{
        			logger.error("Invalid XML data in rowset. /DELETE was found in a wrong place");
        			bError = true;
        		}
        	}
        	else if (qName.equalsIgnoreCase(TUPLE_ELEMENT)) {
        		if (currState.equals(ParserState.IN_TUPLE)) {
        			if (bDeletion) {	/* Deletion */
        				String _currId = currId.toString().trim();
        				if (keyNames.size()>0 && _currId!=null) {
        					try{
        						deleteTuple(keyNames, keyValues, _currId);
        					}catch(Exception ex){
        						logger.error(ex.getMessage());
        						e = ex;
        					}
        				}else if(keyNames.size()==0 && _currId!=null) {
        					try{
        						deleteAllRefs(_currId);
        					}catch(Exception ex){
        						logger.error(ex.getMessage());
        						e = ex;
        					}
        				}else{
        					logger.warn("Could not make deletion bacause the value ID to be deleted is null!");
        				}
        				currState = ParserState.IN_DELETE;
        			}
        			else {	/* Insertion */
        				String _currValue = currValue.toString().trim();
        				if (keyNames.size()>0 && _currValue!=null) {
        					try{
        						//log statistics for memory usage
        	        			Runtime rt = Runtime.getRuntime();
        	        			logger.debug("Trying to add Tuple---Max Memory: "+rt.maxMemory()+", Total Memory: "+rt.totalMemory()+", Free Memory: "+rt.freeMemory());
        	        			addTuple(keyNames, keyValues, _currValue);
        						logger.debug("Just returned from addTuple");
        					}catch(Exception ex){
        						logger.error(ex.getMessage());
        						e = ex;
        					}
        				}else{
            				logger.warn("Could not make insertion! keyNames.size()=" + keyNames.size() + " currValue='" + _currValue + "'");
            			}
        				currState = ParserState.IN_INSERT;
        			}
        			
        			keyNames.clear();
        			keyValues.clear();
        			currValue = new StringBuffer();
        			currId = new StringBuffer();
        		}
        		else
        			bError = true;
        	}
        	else if (qName.equalsIgnoreCase(KEY_ELEMENT)) {
        		if (currState.equals(ParserState.IN_KEY)){
        			currState = ParserState.IN_TUPLE;
        			bKey = false;
        			if(currKeyValue.length()>0 && currKeyName.length()>0){
        				keyNames.add(currKeyName.toString().trim());
        				keyValues.add(currKeyValue.toString().trim());
        			}
        			currKeyName = new StringBuffer();
        			currKeyValue = new StringBuffer();
        		}else
        		{
        			logger.error("Invalid XML data in rowset. /KEY was found in a wrong place");
        			bError = true;
        		}
        	}
        	else if (qName.equalsIgnoreCase(KEYNAME_ELEMENT)) {
        		if (currState.equals(ParserState.IN_KEYNAME))
        			currState = ParserState.IN_KEY;
        		else
        		{
        			logger.error("Invalid XML data in rowset. /KEYNAME was found in a wrong place");
        			bError = true;
        		}
        	}
        	else if (qName.equalsIgnoreCase(KEYVALUE_ELEMENT)) {
        		if (currState.equals(ParserState.IN_KEYVALUE))
        			currState = ParserState.IN_KEY;
        		else
        		{
        			logger.error("Invalid XML data in rowset. /KEYVALUE was found in a wrong place");
        			bError = true;
        		}
        	}
        	else if (qName.equalsIgnoreCase(VALUE_ELEMENT)) {
        		if (currState.equals(ParserState.IN_VALUE))
        		{
        			currState = ParserState.IN_TUPLE;
        			//log statistics for memory usage
        			Runtime rt = Runtime.getRuntime();
        		}else
        		{
        			logger.error("Invalid XML data in rowset. /VALUE was found in a wrong place");
        			bError = true;
        		}
        	}
        	else if (qName.equalsIgnoreCase(ID_ELEMENT)) {
        		if (currState.equals(ParserState.IN_ID))
        			currState = ParserState.IN_TUPLE;
        		else
        		{
        			logger.error("Invalid XML data in rowset. /ID was found in a wrong place");
        			bError = true;
        		}
        	}
        	
        	if (bError)
        		e = new Exception("Invalid XML in rowset data");
        }
        
        public void characters (char ch[], int start, int length) {
        	if (currState == ParserState.IN_KEYNAME) {
        		currKeyName.append(ch, start, length);
        	}
        	else if (currState == ParserState.IN_KEYVALUE) {
        		currKeyValue.append(ch, start, length);
        	}
        	else if (currState == ParserState.IN_VALUE) {
        		currValue.append(ch, start, length);
        	}
        	else if (currState == ParserState.IN_ID) {        		
        		currId.append(ch, start, length);
        	}
        }
        
        public void error(SAXParseException e1) {
        	e = e1;
        }
        
        public void fatalError(SAXParseException e1) {
        	e = e1;
        }
    }
}
