/**
 * 
 */
package org.gcube.indexmanagement.common;

import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;

import javax.xml.namespace.QName;

import org.apache.axis.message.addressing.EndpointReferenceType;
import org.gcube.common.core.contexts.GCUBEServiceContext;
import org.gcube.common.core.informationsystem.notifier.ISNotifier.TopicData;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.state.GCUBEWSResource;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.indexmanagement.common.notifications.IsRegisteredCallback;
import org.gcube.indexmanagement.common.notifications.IsRegisteredNotificationRequest;
import org.gcube.indexmanagement.common.notifications.NotifierRequestQueue;
import org.gcube.indexmanagement.common.notifications.RegisterNotificationRequest;
import org.gcube.indexmanagement.common.notifications.UnregisterNotificationRequest;
import org.globus.wsrf.ResourceException;
import org.globus.wsrf.ResourceProperty;
import org.globus.wsrf.impl.SimpleResourceProperty;
import org.globus.wsrf.impl.SimpleTopic;

/**
 * @author Spyros Boutsis, NKUA
 *
 */
public abstract class IndexWSResource extends GCUBEWSResource  {

	/** The two possible modes of index resource destruction:
	 * FULL_DESTRUCTION: Remove the resource and perform full clean-up of any objects created by it (e.g. objects on the CMS)
	 * REMOVAL_ONLY: Just remove the resource from the container
	 */
	public enum DestructionMode { FULL_DESTRUCTION, REMOVAL_ONLY };
	
	/** The list of standard resource properties defined by every index resource */
    public static final String RP_INDEX_ID = "IndexID";
    public static final String RP_INDEX_TYPE_NAME = "IndexTypeName";
    public static final String RP_COLLECTION_ID = "CollectionID";
    public static final String RP_FIELDS = "Fields";
    public static final String RP_MODIFIED = "Modified";
    public static final String RP_CREATED = "Created";

	/** The list of property names for this resource */
	protected List<String> propNames;

	/** The service's namespace */
	protected String namespace;
	
	/** Is the resource in the initialization state or not? */
	protected boolean bInitializing;
	/** The corresponding lock Object */
	protected Object initLock = new Object();
	
	/** The object that handles the registration of notification topics */
	protected TopicRegistrar topicRegistrar;
	
	/** The resource's destruction mode */
	protected DestructionMode destructionMode;
	
	/** The class logger */
	static GCUBELog logger = new GCUBELog(IndexWSResource.class);
	 
	/**
	 * Constructs a new IndexWSResource object.
	 */
	protected IndexWSResource() {
		/* Initialize the list of resource property names and topics */
		this.propNames = new LinkedList<String>();
		this.bInitializing = false;
		this.topicRegistrar = new TopicRegistrar();
	}
	
	/**
	 * Initializes a IndexWSResource object.
	 * @throws Exception
	 */
	public void initialise(String namespace, String indexID, String indexTypeName, String collectionID[]) throws Exception {
		try {
			this.namespace = namespace;
			this.destructionMode = DestructionMode.FULL_DESTRUCTION;
			
			/* Create and initialize the standard resource properties */
			createProperty(RP_INDEX_ID);
			this.getResourcePropertySet().get(RP_INDEX_ID).clear();
	    	this.getResourcePropertySet().get(RP_INDEX_ID).add(indexID);

			createProperty(RP_INDEX_TYPE_NAME);
			this.getResourcePropertySet().get(RP_INDEX_TYPE_NAME).clear();
            this.getResourcePropertySet().get(RP_INDEX_TYPE_NAME).add(indexTypeName);

			createProperty(RP_COLLECTION_ID);
			int size = 0;
			if(collectionID != null) {
				size = collectionID.length;
			}
	    	this.getResourcePropertySet().get(RP_COLLECTION_ID).clear();
	    	for (int i=0; i<size; i++)
	    		this.getResourcePropertySet().get(RP_COLLECTION_ID).add(collectionID[i]);
	    	
	    	//The Fields property is filled during the feeding of the index
	    	createProperty(RP_FIELDS);
	    	this.getResourcePropertySet().get(RP_FIELDS).clear();
	    	
			createProperty(RP_CREATED);
			this.getResourcePropertySet().get(RP_CREATED).clear();
	    	this.getResourcePropertySet().get(RP_CREATED).add(Calendar.getInstance());

			createProperty(RP_MODIFIED);
			this.getResourcePropertySet().get(RP_MODIFIED).clear();
	    	this.getResourcePropertySet().get(RP_MODIFIED).add(Calendar.getInstance());

		} catch (Exception e) {
			logger.error("Failed to initialize index WS resource.", e);
			throw new Exception("Failed to initialize index WS resource.", e);
		}
	}
	
	/**
	 * Invoked when a resource is being created from a serialized, previously saved state.
	 * 
	 * @param ois the input stream through which the state can be read
	 * @param indicates if the resource is being loaded for the first time (hard load) or not (soft load)
	 * @throws Exception an error occured during resource deserialization
	 */
	protected void onLoad(ObjectInputStream ois, boolean firstLoad) throws Exception {
		
		this.getServiceContext().setScope(GCUBEScope.getScope(this.getResourcePropertySet().getScope().get(0)));
		
		this.destructionMode = DestructionMode.FULL_DESTRUCTION;
		this.namespace = (String) ois.readObject();
		
		/* Create and initialize the standard resource properties from the given input stream */
		String indexID = (String) ois.readObject();
		createProperty(RP_INDEX_ID);
		this.getResourcePropertySet().get(RP_INDEX_ID).clear();
    	this.getResourcePropertySet().get(RP_INDEX_ID).add(indexID);

    	String indexTypeName = (String) ois.readObject();
		createProperty(RP_INDEX_TYPE_NAME);
		this.getResourcePropertySet().get(RP_INDEX_TYPE_NAME).clear();
        this.getResourcePropertySet().get(RP_INDEX_TYPE_NAME).add(indexTypeName);

        int size = ois.readInt();
		createProperty(RP_COLLECTION_ID);
    	this.getResourcePropertySet().get(RP_COLLECTION_ID).clear();
    	for (int i=0; i<size; i++)
    		this.getResourcePropertySet().get(RP_COLLECTION_ID).add((String) ois.readObject());
    	
    	size = ois.readInt();
		createProperty(RP_FIELDS);
    	this.getResourcePropertySet().get(RP_FIELDS).clear();
    	for (int i=0; i<size; i++)
    		this.getResourcePropertySet().get(RP_FIELDS).add((String) ois.readObject());

    	Calendar created = (Calendar) ois.readObject();
		createProperty(RP_CREATED);
		this.getResourcePropertySet().get(RP_CREATED).clear();
    	this.getResourcePropertySet().get(RP_CREATED).add(created);

    	Calendar modified = (Calendar) ois.readObject();
		createProperty(RP_MODIFIED);
		this.getResourcePropertySet().get(RP_MODIFIED).clear();
    	this.getResourcePropertySet().get(RP_MODIFIED).add(modified);
	}
	
	/**
	 * Invoked when the state of the resource must be saved (resource serialization)
	 * 
	 * @param oos the output stream to write the resource state to
	 * @throws Exception an error occured during resource serialization
	 */
	protected void onStore(ObjectOutputStream oos) throws Exception {
		
		oos.writeObject(namespace);

		oos.writeObject((String) this.getResourcePropertySet().get(RP_INDEX_ID).get(0));
		
		oos.writeObject((String) this.getResourcePropertySet().get(RP_INDEX_TYPE_NAME).get(0));

		int size = this.getResourcePropertySet().get(RP_COLLECTION_ID).size();
		oos.writeInt(size);
		for (int i=0; i<size; i++)
			oos.writeObject((String) this.getResourcePropertySet().get(RP_COLLECTION_ID).get(i));

		size = this.getResourcePropertySet().get(RP_FIELDS).size();
		oos.writeInt(size);
		for (int i=0; i<size; i++)
			oos.writeObject((String) this.getResourcePropertySet().get(RP_FIELDS).get(i));

		oos.writeObject((Calendar) this.getResourcePropertySet().get(RP_CREATED).get(0));
		
		oos.writeObject((Calendar) this.getResourcePropertySet().get(RP_MODIFIED).get(0));
	}
	
	/*
	 * (non-Javadoc)
	 * @see org.gcube.common.core.state.GCUBEStatefulResource#store()
	 */
	public void store() {
		/* Store the resource only if it's not currently being initialized */
		if (!this.isInitializing())
			super.store();
	}
	
	/**
	 * Adds a new ResourceProperty to the resource, given a property name.
	 * 
	 * @param propName the name of the new resource property
	 * @return the generated ResourceProperty object
	 * @throws Exception
	 */
	public ResourceProperty createProperty(String propName) throws Exception {
		ResourceProperty prop = this.getProperty(propName);
		this.getResourcePropertySet().add(prop);
		this.propNames.add(propName);
		return prop;
	}

	/**
	 * Adds a new ResourceProperty to the resource, given a property name and a namespace.
	 * 
	 * @param propName the name of the new resource property
	 * @param nameSpace the namespace of the new resource property
	 * @return the generated ResourceProperty object
	 * @throws Exception
	 */
	public ResourceProperty createProperty(String propName, String nameSpace) throws Exception {
		ResourceProperty prop = new SimpleResourceProperty(new QName(nameSpace, propName));
		this.getResourcePropertySet().add(prop);
		this.propNames.add(propName);
		return prop;
	}

	/**
	 * Adds a new topic to the resource, given a topic name and a namespace.
	 * 
	 * @param topicName the name of the new topic
	 * @param nameSpace the namespace of the new topic
	 * @return the generated ResourcePropertyTopic object
	 * @throws Exception
	 */
	public SimpleTopic createTopic(String topicName, String nameSpace) throws Exception {
		SimpleTopic topic = new SimpleTopic(new QName(nameSpace, topicName));
		//this.getTopicList().addTopic(topic);
		this.topicRegistrar.addTopicToRegister(topic);
		return topic;
	}
	
	/**
	 * Adds a new topic to the resource, given a topic name.
	 * 
	 * @param topicName the name of the new topic
	 * @return the generated ResourcePropertyTopic object
	 * @throws Exception
	 */
	public SimpleTopic createTopic(String topicName) throws Exception {
		SimpleTopic topic = new SimpleTopic(new QName(this.getPorttypeContext().getNamespace(), topicName));
		//this.getTopicList().addTopic(topic);
		this.topicRegistrar.addTopicToRegister(topic);
		return topic;
	}
	
	@Override
	protected String[] getPropertyNames() {
		return propNames.toArray(new String[propNames.size()]);
	}
	
	/**
	 * Returns the namespace of the service this resource belongs to
	 * @return the namespace
	 */
	public String getNamespace() {
		return this.namespace;
	}
	
	/**
     * Getter method for the IndexID Resource Property
     * @return <code>String</code> the requested indexID 
     */
    public String getIndexID() {
        return (String) this.getResourcePropertySet().get(RP_INDEX_ID).get(0);
    }

    /**
     * Setter method for the IndexID Resource Property
     * @param indexID <code>String</code> the new IndexId
     */
    public void setIndexID(String indexID) { 
    	this.getResourcePropertySet().get(RP_INDEX_ID).clear();
    	this.getResourcePropertySet().get(RP_INDEX_ID).add(indexID);    	
    }
    
    /**
     * Getter method for the CollectionID list Resource Property
     * @return <code>String[]</code> the requested CollectionIDs
     */
    public String[] getCollectionID() {
    	int size = this.getResourcePropertySet().get(RP_COLLECTION_ID).size();
    	String[] colIDs = new String[size];
    	for (int i=0; i<size; i++)
    		colIDs[i] = (String) this.getResourcePropertySet().get(RP_COLLECTION_ID).get(i);
    	return colIDs;
    }
    
    /**
     * Getter method for the Fields list Resource Property
     * @return <code>String[]</code> the requested Fields
     */
    public String[] getFields() {
    	int size = this.getResourcePropertySet().get(RP_FIELDS).size();
    	String[] fields = new String[size];
    	for (int i=0; i<size; i++)
    		fields[i] = (String) this.getResourcePropertySet().get(RP_FIELDS).get(i);
    	return fields;
    }
    
    /**
     * Setter method for the CollectionID list Resource Property
     * @param <code>String[]</code> the new CollectionIDs
     */
    public void setCollectionID(String[] collectionIDs) throws Exception {
    	
    	if(collectionIDs == null) {
    		logger.warn("Null for the collectionIDs to be set");
    		return;
    	}
    	
    	int size = collectionIDs.length;
    	this.getResourcePropertySet().get(RP_COLLECTION_ID).clear();
    	for (int i=0; i<size; i++)
    		this.getResourcePropertySet().get(RP_COLLECTION_ID).add(collectionIDs[i]);
    }
    
    /**
     * Setter method for the Fields list Resource Property
     * @param <code>String[]</code> the new Fields
     */
    public void setFields(String[] fields) throws Exception {
    	
    	if(fields == null) {
    		logger.warn("Null for the fields to be set");
    		return;
    	}
    	
    	int size = fields.length;
    	this.getResourcePropertySet().get(RP_FIELDS).clear();
    	for (int i=0; i<size; i++)
    		this.getResourcePropertySet().get(RP_FIELDS).add(fields[i]);
    }

    /**
     * Adds a collection id to the collectionID resource property
     * @param collectionID <code>String</code> -the ID to be added
     */ 
    public void addCollectionID(String collectionID){
        String[] collections = getCollectionID();
        String [] newCollections = new String[collections.length +1];
        //if there is already a collection with the same ID return
        for(int i = 0; i < collections.length; i++){
            newCollections[i] = collections[i];
            if(collections[i].equals(collectionID))
        	{
        		logger.warn("Info: ColID:" + collectionID + ", is already added");
        		return;
        	}
        }
        newCollections[collections.length] = collectionID;
        try {
        	this.setCollectionID(newCollections);
        } catch (Exception e) { 
        	logger.error("Failed to set CollectionID resource property.", e);
        }
    }
    
    /**
     * Adds a field to the Fields resource property
     * @param field <code>String</code> -the field to be added in the following format: 
     * [collectionID]:[language]:["s"|"p"]:[fieldName]
     */ 
    public void addField(String field){
        String[] fields = getFields();
        String [] newFields = new String[fields.length +1];
        //if there is already a field describing the same information return
        for(int i = 0; i < fields.length; i++){
            newFields[i] = fields[i];
            if(fields[i].equals(field))
        	{
        		logger.warn("Info: Field:" + field + ", is already added");
        		return;
        	}
        }
        newFields[fields.length] = field;
        try {
        	this.setFields(newFields);
        } catch (Exception e) { 
        	logger.error("Failed to set Fields resource property.", e);
        }
    }
    
    /**
     * Getter method for the IndexTypeName Resource Property
     * @return <code>String</code> the requested IndexTypeName
     */
    public String getIndexTypeName() {
        return (String) this.getResourcePropertySet().get(RP_INDEX_TYPE_NAME).get(0);
    }        
    /**
     * Setter method for the IndexTypeName Resource Property
     * @param indexTypeName <code>String</code> the new IndexTypeName 
     */
    public void setIndexTypeName(String indexTypeName) throws Exception {
        try{
            this.getResourcePropertySet().get(RP_INDEX_TYPE_NAME).clear();
            this.getResourcePropertySet().get(RP_INDEX_TYPE_NAME).add(indexTypeName);
        }
        catch(Exception e){
            logger.error(e.toString());
        }
    }
    
    /**
     * Getter method for the Created Resource Property
     * @return <code>Calendar</code> the requested time of creation for this index replication
     */
    public Calendar getCreated() {
        return (Calendar) this.getResourcePropertySet().get(RP_CREATED).get(0); 
    }
    
    /**
     * Setter method for the Created Resource Property
     * @param created <code>Calendar</code> representation of the time the resource was created.
     */
    public void setCreated(Calendar created) {
    	this.getResourcePropertySet().get(RP_CREATED).clear();
    	this.getResourcePropertySet().get(RP_CREATED).add(created);
    }

    /**
     * Getter method for the Modified Resource Property
     * @return <code>Calendar</code> the requested lastModified indicator
     */
    public Calendar getModified() {
        return (Calendar) this.getResourcePropertySet().get(RP_MODIFIED).get(0);
    }
    
    /**
     * Setter method for the Modified Resource Property
     * @param modified <code>Calendar</code> representation of the time the resource was modified
     */
    public void setModified(Calendar modified) {
    	this.getResourcePropertySet().get(RP_MODIFIED).clear();
    	this.getResourcePropertySet().get(RP_MODIFIED).add(modified);
    }
    
    /**
     * Returns the current initialization state of the resource
     * @return true if the resoure is currently initializing, false otherwise
     */
    public boolean isInitializing() {
    	return bInitializing;
    }
    
    /**
     * Specifies the current state of the resource (initializing or not)
     * @param isInitializing the initialization state to set
     */
    public void setIsInitializing(boolean isInitializing) {
    	synchronized (initLock) {
    		bInitializing = isInitializing;
        	if(!isInitializing) {
        		initLock.notifyAll();
        	}
		}    	
    }
    
    /**
     * Blocks thread execution until WS resource is initialized
     */
    protected void waitUntilInitialized() {
    	synchronized (initLock) {
    		while(isInitializing()) {
        		try {
        			initLock.wait();
        		} catch (InterruptedException e) {
					logger.warn("Interrupted Exception while waiting for WS resource initialization: ", e);
				}
        	}
		}    	
    }
    
    protected void filterFieldInfo(ArrayList<String> presentableFields, ArrayList<String> searchableFields){
		
    	//first clear the arrays
    	presentableFields.clear();
    	searchableFields.clear();
    	
    	//to ensure each field is added only once
    	HashSet<String> searchableSet = new HashSet<String>();
    	HashSet<String> presentableSet =  new HashSet<String>();
    	
    	//create the arrays of presentable and searchable fields
		for(String field : this.getFields())
		{
			//add the field to the related array, depending on its type
			String fieldName;
			if((fieldName = getFieldName(field, IndexType.PRESENTABLE_TAG)) != null)
			{
				if(presentableSet.add(fieldName)) {
					logger.trace("added presentable field: " + fieldName);
					presentableFields.add(fieldName);
				}
				
			} else {
				fieldName = getFieldName(field, IndexType.SEARCHABLE_TAG);
				if(searchableSet.add(fieldName)) {
					logger.trace("added searchable field: " + fieldName);
					searchableFields.add(fieldName);
				}
			}
		}
	}
	
	private static String getFieldName(String field, String type) {
		//get the string after the second seperator, which indicates the type
		int from = field.indexOf(IndexType.SEPERATOR_FIELD_INFO, 
				field.indexOf(IndexType.SEPERATOR_FIELD_INFO) + 1) + 1;
		String t = field.substring(from, from + type.length());
		if(!t.equals(type))
			return null;
		//get the string after the third seperator, which indicates the fieldName
		from = field.indexOf(IndexType.SEPERATOR_FIELD_INFO, from) + 1;
		return field.substring(from);		
	}
    
    /**
     * Returns the resource's destruction mode
     * @return the resource's destruction mode
     */
    public DestructionMode getDestructionMode() {
    	return destructionMode;
    }
    
    /**
     * Sets the resource's destruction mode
     * @param mode the destruction mode to be set
     */
    public void setDestructionMode(DestructionMode mode) {
    	destructionMode = mode;
    }

    /**
     * Performs clean-up operations before the resource is removed from the home
     */
    public void onResourceRemoval() { }
    
    /*
     * (non-Javadoc)
     * @see org.gcube.common.core.state.GCUBEWSResource#publish(org.gcube.common.core.scope.GCUBEScope[])
     */
    public void publish(final GCUBEScope ... scopes) throws ResourceException {
    	/* This method is overriden because we want to handle the topic registration manually,
    	 * without having gCore register automatically for us. So, first the super implementation
    	 * of this method is invoked, and then the custom registrar begins the registration. Before
    	 * beginning the registration process, all topics are added to the resource's topicList,
    	 * because globus must find them in there in order for the registration to succeed. 
    	 */
    	super.publish(scopes);
    	for (SimpleTopic t : this.topicRegistrar.getTopicsToBeRegistered())
    		this.getTopicList().addTopic(t);
    	this.topicRegistrar.unregisterAndReregisterTopics();
    	
    }
    
    /**
     * Checks if the registration of the given topic has been completed.
     * @param topic the topic whose registration to check
     * @return true if the registration has been completed, else false
     * @throws Exception
     */
    public boolean isTopicRegistered(SimpleTopic topic) throws Exception {
    	return this.topicRegistrar.isTopicRegistered(topic);
    	/*
    	ISNotifier notifier = GHNContext.getImplementation(ISNotifier.class);
    	GCUBEServiceContext sctx = getServiceContext();
		GCUBEScope scope = (sctx.getScope()!=null ? sctx.getScope() : GCUBEScope.getScope(getResourcePropertySet().getScope().get(0)));
		return notifier.isTopicRegistered(topic.getName(), getEPR(), getServiceContext(), scope);
		*/
    }
    
    
    
    
    /**
     * This class handles the registration of topics exposed by this resource, as well as
     * the removal of old registrations of the same topics.
     * 
     * @author Spyros Boutsis, NKUA
     */
    private class TopicRegistrar extends Thread {
    	
    	//private ISNotifier notifier;
		private LinkedList<SimpleTopic> topicsToBeRegistered;
    	private LinkedList<SimpleTopic> registeredTopics;
    	
    	/**
    	 * Class constructor
    	 */
    	TopicRegistrar() {
    		try {
    			//notifier = GHNContext.getImplementation(ISNotifier.class);
    			topicsToBeRegistered = new LinkedList<SimpleTopic>();
    			registeredTopics = new LinkedList<SimpleTopic>();
    		} catch (Exception e) { }
    	}

    	/**
    	 * Adds a topic to the list of topics to be registered
    	 * @param topic
    	 */
    	synchronized void addTopicToRegister(SimpleTopic topic) throws Exception {
//    		GCUBEServiceContext sctx = getServiceContext();
//    		GCUBEScope scope = (sctx.getScope()!=null ? sctx.getScope() : GCUBEScope.getScope(getResourcePropertySet().getScope().get(0)));
//    		boolean bIsAlreadyRegistered = false;
//    		try {
//    			List<TopicData> l = new LinkedList<TopicData>();
//    			l.add(new TopicData(topic.getName(), getEPR()));
//    			bIsAlreadyRegistered = notifier.isTopicRegistered(sctx, scope, l)[0];
//    		} catch (Exception e) { }
    		topicsToBeRegistered.add(topic);
    		logger.debug("Topic added to the list of topics to be registered: " + topic.getName());
//    		if (bIsAlreadyRegistered) {
//    			alreadyRegisteredTopics.add(topic);
//    			logger.debug("A registration already exists for this topic, it will be first unregistered.");
//    		}
    	}
    	
    	/**
    	 * Starts the un-registration and re-registration of the topics that are already registered
    	 */
    	void unregisterAndReregisterTopics() {
    		/* If no old topic registrations need to be removed, do the registration in the current
    		 * thread, by invoking run() directly. Else, start a new thread by invoking start(). */
//    		if (alreadyRegisteredTopics.isEmpty())
//    			this.run();
//    		else
    		if (this.topicsToBeRegistered.size() > 0)
    			this.start();
    	}

    	/**
    	 * Returns the list of topics to be registered
    	 * @return
    	 */
    	List<SimpleTopic> getTopicsToBeRegistered() {
    		return topicsToBeRegistered;
    	}
    	
    	/**
    	 * Checks if the given topic is registered or not.
    	 * @param topicName the topic
    	 * @return
    	 */
    	synchronized boolean isTopicRegistered(SimpleTopic topic) throws Exception {
    		return this.registeredTopics.contains(topic);
    	}
    	
    	/*
    	 * (non-Javadoc)
    	 * @see java.lang.Thread#run()
    	 */
    	public void run() {
    		GCUBEServiceContext sctx = getServiceContext();
    		GCUBEScope scope = (sctx.getScope()!=null ? sctx.getScope() : GCUBEScope.getScope(getResourcePropertySet().getScope().get(0)));
        	LinkedList<SimpleTopic> alreadyRegisteredTopics = new LinkedList<SimpleTopic>();
    		LinkedList<SimpleTopic> topicsToBeRemoved = new LinkedList<SimpleTopic>();
    		
    		/* Get the resource's epr */
    		EndpointReferenceType myEpr = null;
    		try {
    			myEpr = getEPR();
    		} catch (Exception e) {
    			logger.error("Failed to get my own EPR!", e);
    			return;
    		}
    		
    		/* See if any of the topics are already registered, and add them to the 'alreadyRegisteredTopics' list */
    		List<TopicData> l = new LinkedList<TopicData>();
    		for (SimpleTopic t : this.topicsToBeRegistered)
    			l.add(new TopicData(t.getName(), myEpr));
    		
    		final Object monitor = new Object();
    		final List<Boolean> bIsAlreadyRegistered = new LinkedList<Boolean>();
    		IsRegisteredCallback callback = new IsRegisteredCallback() {
    			public void isRegisteredResponse(boolean[] response) {
    				for (boolean b : response)
    					bIsAlreadyRegistered.add(b);
    				
    				synchronized (monitor) {
    					monitor.notify();
    				}
    			}
    		};
    		synchronized (monitor) {
    			NotifierRequestQueue.getInstance().add(new IsRegisteredNotificationRequest(l, sctx, scope, callback));
    			while (true) {
	    			try {
	    				monitor.wait();
	    				break;
	    			} catch (InterruptedException e) { }
    			}
    		}
    		logger.info("Old registrations test done.");    		
    		for (int i=0; i<bIsAlreadyRegistered.size(); i++) {
    			if (bIsAlreadyRegistered.get(i))
    				alreadyRegisteredTopics.add(this.topicsToBeRegistered.get(i));
    		}
    		
    		/* Unregister the registered topics, and wait until all topics have been unregistered */
    		if (!alreadyRegisteredTopics.isEmpty()) {
    			logger.debug("Starting removal of old topic registrations...");
    			NotifierRequestQueue.getInstance().add(new UnregisterNotificationRequest(myEpr, alreadyRegisteredTopics, sctx, scope));
	    		while (true) {
					try {
						Thread.sleep(20000);	// Poll every 20 seconds
					} catch (Exception e) { }

					l.clear();
					for (SimpleTopic t : alreadyRegisteredTopics)
		    			l.add(new TopicData(t.getName(), myEpr));
					
					final Object monitor1 = new Object();
		    		final List<Boolean> bRegistered = new LinkedList<Boolean>();
		    		IsRegisteredCallback callback1 = new IsRegisteredCallback() {
		    			public void isRegisteredResponse(boolean[] response) {
		    				for (boolean b : response)
		    					bRegistered.add(b);
		    				synchronized (monitor1) {
		    					monitor1.notify();
		    				}
		    			}
		    		};
		    		synchronized (monitor1) {
		    			NotifierRequestQueue.getInstance().add(new IsRegisteredNotificationRequest(l, sctx, scope, callback1));
		    			while (true) {
			    			try {
			    				monitor1.wait();
			    				break;
			    			} catch (InterruptedException e) { }
		    			}
		    		}
		    		
		    		for (int i=0; i<bRegistered.size(); i++) {
		    			if (!bRegistered.get(i)) {
		    				SimpleTopic t = alreadyRegisteredTopics.get(i);
		    				topicsToBeRemoved.add(t);
		    				logger.debug("Topic unregistration completed: " + t.getName());
		    			}
					}
					
					synchronized (this) {
						if (topicsToBeRemoved.size() > 0) {
							alreadyRegisteredTopics.removeAll(topicsToBeRemoved);
							if (alreadyRegisteredTopics.isEmpty()) {
								logger.info("All old topic registrations removed.");
								break;
							}
							topicsToBeRemoved.clear();
						}
					}
					logger.info("Still waiting for " + alreadyRegisteredTopics.size() + " old topic registrations to be removed.");
				}
    		}

    		/* Now register the topics */
    		try {
    			logger.debug("Starting registration of new topics...");
    			NotifierRequestQueue.getInstance().add(new RegisterNotificationRequest(myEpr, topicsToBeRegistered, sctx, scope));
    			logger.debug("Topic registration started.");	    		
    		} catch (Exception e) {
    			logger.error("Error while registering topics.");
    		}
    		
    		/* Wait to see if the registration were successful */
    		while (true) {
				try {
					Thread.sleep(20000);	// Poll every 20 seconds
				} catch (Exception e) { }

	    		l.clear();
	    		topicsToBeRemoved.clear();
				for (SimpleTopic t : topicsToBeRegistered)
	    			l.add(new TopicData(t.getName(), myEpr));
				
				final Object monitor1 = new Object();
				final List<Boolean> bRegistered = new LinkedList<Boolean>();
				IsRegisteredCallback callback1 = new IsRegisteredCallback() {
	    			public void isRegisteredResponse(boolean[] response) {
	    				for (boolean b : response) {
	    					bRegistered.add(b);
	    				}
	    				synchronized (monitor1) {
	    					monitor1.notify();
	    				}
	    			}
	    		};
	    		synchronized (monitor1) {
	    			NotifierRequestQueue.getInstance().add(new IsRegisteredNotificationRequest(l, sctx, scope, callback1));
	    			while (true) {
		    			try {
		    				monitor1.wait();
		    				break;
		    			} catch (InterruptedException e) { }
	    			}
	    		}
				
	    		for (int i=0; i<bRegistered.size(); i++) {
	    			if (bRegistered.get(i)) {
	    				SimpleTopic t = topicsToBeRegistered.get(i);
	    				topicsToBeRemoved.add(t);
	    				synchronized (this) {
							this.registeredTopics.add(t);
						}
	    				logger.debug("Topic registration completed: " + t.getName());
	    			}
				}
	    		
				if (topicsToBeRemoved.size() > 0) {
					topicsToBeRegistered.removeAll(topicsToBeRemoved);
					if (topicsToBeRegistered.isEmpty()) {
						logger.info("All topic registrations completed!");
						break;
					}
				}
    		}
    	}
    }
}
