package org.gcube.indexmanagement.common.mergesort;

import gr.uoa.di.madgik.grs.record.GenericRecord;

import java.util.ArrayList;

/**
 * This class is the container for results that are produced by <numberOfworkers> 
 * workers and are consumed by a MergeSorter. The MergeSorter must pause until
 * all the queues of the active workers have at least one element to consume OR
 * there is no active worker left. A worker must pause until 
 * <resultsRetrieved> >= <getUntil> AND it has at least one element in its queue. 
 * The MergeSorter must wake up the workers when <getUntil> > <resultsRetrieved> OR
 * one queue was emptied. 
 * @author bill
 *
 */
public class MergeSortPipe {
	
	/**
	 * The number of workers using the pipe
	 */
	private int numberOfworkers;
	/**
	 * The number of results retrieved and placed in the queues
	 */
	private long resultsRetrieved = 0;
	/**
	 * The number of results sent to the clients
	 */
	private long resultsSent = 0;	
	/**
	 * The number of results that must be retrieved until the client requests more
	 */
	private long getUntil = 0;
	
	/**
	 * The queues used by the workers
	 */
	private ArrayList<ArrayList<MergeSortElement>> queues;
	/**
	 * The active array indicating which workers are active
	 */
	private boolean[] active;
	/**
	 * This flag indicates if there is no need for generating more elements
	 */
	private boolean stop = false;
	
	
	/**
	 * constructor
	 * 
	 * @param numberOfworkers - the number of workers
	 */
	public MergeSortPipe(int numberOfworkers) {
		this.numberOfworkers = numberOfworkers;
		this.active = new boolean[numberOfworkers];
		for(int i=0; i<numberOfworkers; i++)
			this.active[i] = true;
		this.queues = new ArrayList<ArrayList<MergeSortElement>>(numberOfworkers);
		for(int i=0; i<numberOfworkers; i++)
			this.queues.add(new ArrayList<MergeSortElement>());
	}
	
	/**
	 * sets the workerID worker as inactive
	 * @param workerID
	 */
	public void setInActive(int workerID)
	{
		this.active[workerID] = false;
	}
	
	/**
	 * returns the state of the workerID
	 * @param workerID
	 * @return true if workerID is active / false otherwise
	 */
	public boolean isActive(int workerID)
	{
		return this.active[workerID];
	}
	
	/**
	 * gets the sum of the results retrieved 
	 * @return resultsRetrieved
	 */
	public long getResultsRetrieved() {
		return resultsRetrieved;
	}

	/**
	 * sets the resultsRetrieved
	 * @param resultsRetrieved
	 */
	public void setResultsRetrieved(long resultsRetrieved) {
		this.resultsRetrieved = resultsRetrieved;
	}
	
	public long getResultsSent() {
		return resultsSent;
	}

	/**
	 * gets the sum of the results requested by the client, 
	 * until now
	 * @return getUntil
	 */
	public long getGetUntil() {
		return getUntil;
	}

	/**
	 * sets the number of results requested by the client
	 * @param getUntil
	 */
	public void setGetUntil(long getUntil) {
		this.getUntil = getUntil;
	}
	
	/**
	 * Adds a new elements' array in the queue
	 * @param workerID - the ID of the owrker that adds the element
	 * @param elements - the elements to be added in the queue
	 */
	public void add(int workerID, ArrayList<MergeSortElement> elements)
	{
		//add to the queue for this worker this element
		this.queues.get(workerID).addAll(elements);
		this.resultsRetrieved += elements.size();
	}
	
	public MergeSortElement getNext()
	{
		//if there is even one active queue empty this method 
		//should not be called - return null
		if(!areAllQueuesNonEmpty())
			return null;
		double max = -1.0;
		int queueID = -1;
		//examine all the active workers OR the queues that 
		//have at least one element
		for(int i=0; i<numberOfworkers; i++)
		{
			if(isActive(i) || queues.get(i).size()>0)
			{
				MergeSortElement element = queues.get(i).get(0); 
				if(max == -1.0 || element.getRank() > max)
				{
					max = element.getRank();
					queueID = i;
				}
			}
		}
		//if all the queues are inactive and empty, 
		//there is nothing to send
		if(queueID == -1)
		{
			stop = true;
			return null;
		}
		this.resultsSent++;
		return queues.get(queueID).remove(0);
	}
	
	/**
	 * Adds a new element in the queue
	 * @param workerID - the ID of the owrker that adds the element
	 * @param element - the element to be added in the queue
	 */
	public void add(int workerID, MergeSortElement element)
	{
		//add to the queue for this worker this element
		this.queues.get(workerID).add(element);
		this.resultsRetrieved++;
	}
	
	/**
	 * checks if all the queues, of the active workers, 
	 * have at least one element
	 * @return true if yes / false otherwise
	 */
	public boolean areAllQueuesNonEmpty() {
		for(int i=0; i<numberOfworkers; i++)
			//if there is one queue empty and the 
			//corresponding worker is still active 
			//return false
			if(queues.get(i).size() == 0 && active[i])
				return false;
		//all the queues have at least one element
		return true;			
	}
	
	/**
	 * checks if there is any worker left
	 * @return true if yes / false otherwise
	 */
	public boolean isAnyWorkerLeft() {
		for(boolean isActive : active)
			//if there is one active return true
			if(isActive)
				return true;
		//no worker is left
		return false;
	}
	
	/**
	 * checks if a specific worker can sleep
	 * @param workerID - the ID of the worker
	 * @return true if yes / false otherwise
	 */
	public boolean canWorkerPause(int workerID) {
		if(queues.get(workerID).size() > 0)
			if((getUntil != -1) && (resultsRetrieved >= getUntil))
				return true;
		return false;
	}
	
	/**
	 * checks if a MergeSorter can pause and let the 
	 * workers do the processing, until there is enough
	 * data to do merge sorting, or there are more results
	 * than the results requested by the client, or there 
	 * is no worker left
	 * 
	 * @return true if yes / false otherwise
	 */
	public boolean canMergeSorterPause() {
		//if the results retrieved are more than the results requested
		//the mergeSorter must not pause on the pipe, but on the interaction with the client
		if((getUntil != -1) && (resultsRetrieved >= getUntil))
			return false;
		//if there is at least one worker left and at least one active queue empty the mergeSorter
		//can pause
		if(isAnyWorkerLeft())
			if(!areAllQueuesNonEmpty())
				return true;
		return false;
	}
	
	/**
	 * checks if a MergeSorter must receive a notification.
	 * If we are in a state that the MergeSorter should not 
	 * be paused then a notification must be sent.
	 * 
	 * @return true if yes / false otherwise
	 */
	public boolean sendMergerNotification() {
		
		if(canMergeSorterPause())
			return false;
		else
			return true;
	}
	
	/**
	 * checks if workers must receive notifications. If there 
	 * is at least one active queue that is empty or the results 
	 * retrieved are less than the results requested by the 
	 * client.
	 * 
	 * @return true if yes / false otherwise 
	 */
	public boolean sendWorkersNotification() {
		if(!areAllQueuesNonEmpty())
			return true;
		if((getUntil == -1) || (resultsRetrieved < getUntil))
			return true;
		return false;
	}

	/**
	 * raises the stop flag
	 */
	public void setStop() {
		this.stop = true;
	}

	/**
	 * returns the stop flag
	 * @return the stop flag
	 */
	public boolean isStoped() {
		return stop;
	}

}
