package org.gcube.indexmanagement.geo.refinement;

import java.util.Iterator;
import java.util.List;

import org.gcube.indexmanagement.geo.DataWrapper;
import org.gcube.indexmanagement.geo.GeoIndexType;
import org.gcube.indexmanagement.geo.InclusionType;
import org.gcube.indexmanagement.geo.shape.Polygon;
import org.gcube.indexmanagement.geo.shape.Rectangle;

import com.vividsolutions.jts.geom.Envelope;

/** The base class for all Refiners */
public abstract class Refiner {

    /** The query polygon */
    protected Envelope query;

    /** The query Minimal Bounding Rectangle */
    protected Polygon polygon;

    /** The index type of the Index using this Refiner */
    protected GeoIndexType indexType = null;

    /** The containment method used in the query */
    protected InclusionType containmentMethod;
    
    /** This flags indicates if we want to return the non-hits
     * instead of the hits
     */
    protected boolean not = false;

    /**
     * A method called by the GeoIndexResource after creating an instance of a
     * Refiner
     * 
     * @param polygon -
     *            A polygon representation of the original query of which this
     *            Refiner will refine the results
     * @param containmentMethod -
     *            The containment method used in the original query
     * @param indexType -
     *            The geo index type of the GeoIndex using this Refiner
     * @param args -
     *            Any additional plugin specific initialization arguments.
     * @throws RefinerInitializationException -
     *             An error occurex while initializing the Refiner
     */
    final public void init(Polygon polygon, InclusionType containmentMethod,
            GeoIndexType indexType, boolean not, String args[])
            throws RefinerInitializationException {
        this.polygon = polygon;
        Rectangle rect = polygon.getBoundingBox();
        this.query = new Envelope(rect.getMinX(), rect.getMaxX(), rect
                .getMinY(), rect.getMaxY());
        this.containmentMethod = containmentMethod;

        this.indexType = indexType;
        
        this.not = not;

        initialize(args);
    }

    /**
     * A method used to refine a list of GeoIndex entries using a plugin
     * specific isHit(entry) method to determine which entries are valid
     * 
     * @param entries -
     *            The list of entries to be refined
     * @return A refined list only containing entries which a valid according to
     *         this refiners isHit(entry) method
     */
    final public int refine(List<DataWrapper> entries) {
        Iterator<DataWrapper> iterator = entries.iterator();
        int removedCount = 0;

        while (iterator.hasNext()) {
            if ((!not && !isHit(iterator.next())) 
            		|| 
            	(not && isHit(iterator.next()))) {
                iterator.remove();
                removedCount++;
            }
        }
        return removedCount;
    }

    /**
     * The plugin specific initialization method. Initializes an instance of a
     * Refiner plugin using an array of Strings. All other initialization
     * methods have been completed when this method is called, so it can make
     * use of any of the standard Refiner instance variables.
     * 
     * @param args -
     *            A plugin specific array of Strings used to initialize a
     *            Refiner plugin.
     * @throws RefinerInitializationException -
     *             An error occurex while initializing the Refiner
     */
    abstract protected void initialize(String args[])
            throws RefinerInitializationException;

    /**
     * A method used to check if this Refiner plugin is compatible with
     * indices/entries with a specific GeoIndexType
     * 
     * @param indexType -
     *            The GeoIndexType of which to check the compatability with this
     *            plugin
     * @return TRUE if this Refiner plugin is compatible with indices/entries
     *         with the specified GeoIndexType
     */
    abstract public boolean isIndexTypeCompatible(GeoIndexType indexType);

    /**
     * A plugin specific method used to determine which index entries are hits
     * according to the given query.
     * 
     * @param entry -
     *            the entry to determine if is a hit.
     * @return TRUE if the entry is a hit according to the Refiner's query.
     */
    abstract public boolean isHit(DataWrapper entry);

}
