/*
 * IndexGenerator.java
 *
 * $Author: tsakas $
 * $Date: 2007/12/20 14:37:39 $
 * $Id: IndexGenerator.java,v 1.1 2007/12/20 14:37:39 tsakas Exp $
 *
 * <pre>
 *             Copyright (c) : 2006 Fast Search & Transfer ASA
 *                             ALL RIGHTS RESERVED
 * </pre>
 */

package org.gcube.indexmanagement.common;

import java.io.File;

/**
 * Interface for Index Generating Wrapper Classes.
 */
public interface IndexGenerator {

    /**
     * Inserts rowset of supported a supported indexType.
     * 
     * @param rowset
     *            xml that contain data to insert
     * @return number of handled documents
     * @throws IndexException -
     *             an error inserting the ROWSET
     */
    public int insertRowSet(String rowset) throws IndexException;

    /**
     * Deletes the listed documents from the local storage
     * 
     * @param documentIDs -
     *            an array of the IDs of the documents to delete
     * @return the number of documents deleted
     * @throws IndexException -
     *             an error deleting documents
     */
    public int deleteDocumentsLocally(String[] documentIDs)
            throws IndexException;

    /**
     * Deletes the listed documents both from the local storage and the main
     * index
     * 
     * @param documentIDs -
     *            an array of the IDs of the documents to delete
     * @return a new deletionFile
     * @throws IndexException -
     *             an error deleting documents
     */
    public File deleteDocuments(String[] documentIDs) throws IndexException;

    /**
     * Commits inserted rowsets
     * 
     * @throws IndexException -
     *             an error committing
     */
    public void commit() throws IndexException;

    /**
     * Deletes inserted rowsets since last <code>commit()</code>
     * 
     * @throws IndexException -
     *             an error aborting
     */
    public void abort() throws IndexException;

    /**
     * Returns the size of the committed files
     * 
     * @throws IndexException -
     *             an error getting the fileSize
     */
    public long getCommittedFileSize() throws IndexException;

    /**
     * Returns the size of the uncommitted files
     * 
     * @throws IndexException -
     *             an error getting the fileSize
     */
    public long getUnCommittedFileSize() throws IndexException;

    /**
     * Creates a new index
     * 
     * @param indexName -
     *            name of the index
     * @param collectionID -
     *            the collectionID of the index to create
     * @param indexType -
     *            the IndexType of the index to create
     * @throws IndexException -
     *             an error creating the index to create
     */
    public void createIndex(String indexName,
    		FullTextIndexType indexType, boolean forceCreate) throws IndexException;

    /**
     * Deletes the entire index with all it's inserted rowsets.
     * <p>
     * Warning: there is no undelete method.
     * 
     * @throws IndexException -
     *             an error deleting documents
     */
    public void deleteIndex() throws IndexException;

    /**
     * Opens the index for insertion of rowsets
     * 
     * @param indexName
     *            name of the index
     * @throws IndexException -
     *             an error opening the index
     */
    public void openIndex(String indexName) throws IndexException;

    /**
     * Updates the Index' state object
     * 
     * @throws IndexException -
     *             an error setting the indexType
     */
    public void setIndexType(FullTextIndexType idxType) throws IndexException;

    /**
     * Shows the current file size of the index
     * 
     * @throws IndexException -
     *             an error deleting documents
     */
    public long getIndexFileSize() throws IndexException;

    /**
     * Merges the named index into the open index
     * 
     * @param inputIndexName
     *            name of the input index
     * @throws IndexException -
     *             an merging the index
     */
    public int mergeIndex(File inputIndex) throws IndexException;

    /**
     * Deletes the documents from the file from the index
     * 
     * @param deletionFile
     *            a file identifying the documents to delete
     * @throws IndexException -
     *             an error merging the deletion file
     */
    public int mergeDeletionFile(File deletionFile) throws IndexException;

    /**
     * Closes the index committing any uncommitted insertions
     * 
     * @throws IndexException -
     *             an error closing the Index
     */
    public void closeIndex() throws IndexException;

    /**
     * Frees any resources used for updating indexes if any.
     * 
     * @throws IndexException -
     *             an error closing the IndexGenerator
     */
    public void close() throws IndexException;
}
