package org.gcube.indexmanagement.geo.ranking;

import java.util.Comparator;
import java.util.List;

/**
 * A utility class used to sort objects using the quick sort and partial quick
 * sort sorting algorithms. See http://en.wikipedia.org/wiki/Quicksort (or any
 * data structures & algorithm text book) for an explanation of the quick sort
 * algorithm.
 */
public class Sorter {

    /**
     * A method used to sort a list of objects using a specified comparator and
     * the quick sort algorithm
     * 
     * @param <T>
     *            The Type of the List to sort
     * @param objects -
     *            The list to sort
     * @param comparator -
     *            The comparator to use during sorting
     */
    public static <T> void quickSort(List<T> objects, Comparator<T> comparator) {
        quickSort(objects, 0, objects.size() - 1, 0, objects.size(), comparator);
    }

    /**
     * A method used to sort a list of objects using a specified comparator and
     * the partial quick sort algorithm
     * 
     * @param <T>
     *            The Type of the List to sort
     * @param objects -
     *            The list to sort
     * @param start -
     *            The starting point of the list part to sort (inclusive)
     * @param end -
     *            The ending point of the list part to sort (inclusive)
     * @param isSeriallySorted -
     *            an indicator of whether the list is being serially sorted (ie.
     *            are all parts before the current start index already sorted?)
     * @param comparator -
     *            The comparator to use during sorting
     */
    public static <T> void partialQuickSort(List<T> objects, int start,
            int end, boolean isSeriallySorted, Comparator<T> comparator) {
        if (isSeriallySorted)
            quickSort(objects, start, objects.size() - 1, start, end,
                    comparator);
        else
            quickSort(objects, 0, objects.size() - 1, start, end, comparator);
    }

    /**
     * A method used to sort a list of objects using a specified comparator and
     * the partial quick sort algorithm
     * 
     * @param <T>
     *            The Type of the List to sort
     * @param objects -
     *            The list to sort
     * @param left -
     *            The left most point of the current partition to sort
     *            (inclusive)
     * @param right -
     *            The right most point of the current partition to sort
     *            (inclusive)
     * @param start -
     *            The starting point of the list part to sort (inclusive)
     * @param end -
     *            The ending point of the list part to sort (inclusive)
     * @param comparator -
     *            The comparator to use during sorting
     */
    private static <T> void quickSort(List<T> objects, int left, int right,
            int start, int end, Comparator<T> comparator) {
        if (right > left) {
            int pivotIndex = selectPivot(objects, left, right);
            int alteredPivotIndex = partition(objects, left, right, pivotIndex,
                    comparator);
            if (alteredPivotIndex > start)
                quickSort(objects, left, alteredPivotIndex - 1, start, end,
                        comparator);
            if (alteredPivotIndex < end)
                quickSort(objects, alteredPivotIndex + 1, right, start, end,
                        comparator);
        }
    }

    /**
     * A simple method of selecting the pivot. Could be improved :)
     * 
     * @param objects -
     *            The list of objects being sorted
     * @param left -
     *            The left most index in the part of the list to find a pivot
     *            (inclusive)
     * @param right -
     *            The right most index in the part of the list to find a pivot
     *            (inclusive)
     * @return the requested pivot
     */
    private static int selectPivot(List objects, int left, int right) {
        return left;
    }

    /**
     * Puts all smaller objects smaller than or equal to a specified pivot
     * object to the left of the pivot, and all the object bigger than the pivot
     * object to the right of the pivot.
     * 
     * @param <T>
     *            The Type of the List to partition
     * @param objects -
     *            The list of which a part is to be partitioned
     * @param left -
     *            The index of the left most object in the list part to
     *            partition (inclusive)
     * @param right -
     *            The index of the left most object in the list part to
     *            partition (inclusive)
     * @param pivotIndex -
     *            The index of the pivot
     * @param comparator -
     *            The comparator to use in order to compare objects
     * @return the new (and final) index of the pivot object in the list to be
     *         sorted
     */
    private static <T> int partition(List<T> objects, int left, int right,
            int pivotIndex, Comparator<T> comparator) {
        T temp = objects.get(right);
        T pivot = objects.get(pivotIndex);
        objects.set(right, pivot);
        objects.set(pivotIndex, temp);

        int currentPivotIndex = left;
        for (int i = left; i < right; i++) {
            if (comparator.compare(objects.get(i), pivot) <= 0) {
                temp = objects.get(currentPivotIndex);
                objects.set(currentPivotIndex, objects.get(i));
                objects.set(i, temp);
                currentPivotIndex++;
            }
        }

        temp = objects.get(currentPivotIndex);
        objects.set(currentPivotIndex, pivot);
        objects.set(right, temp);

        return currentPivotIndex;
    }
};
