/*
 * QueryTermFilter.java
 *
 * $Author: tsakas $
 * $Date: 2007/12/20 14:37:39 $
 * $Id: QueryTermFilter.java,v 1.1 2007/12/20 14:37:39 tsakas Exp $
 *
 * <pre>
 *             Copyright (c) : 2006 Fast Search & Transfer ASA
 *                             ALL RIGHTS RESERVED
 * </pre>
 */
package org.gcube.indexmanagement.lucenewrapper;

import java.util.HashSet;

import org.apache.lucene.index.Term;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.FuzzyQuery;
import org.apache.lucene.search.PhraseQuery;
import org.apache.lucene.search.PrefixQuery;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.RangeQuery;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.WildcardQuery;

/**
 * Extracts terms from queries.
 */
public final class QueryTermFilter {
    /**
     * Extracts all query terms of the given Query.
     * 
     * @param query
     *            The query to extract terms
     * @return QueryTerm[] An array extracted query terms.
     */

    public static final QueryTerm[] getTerms(Query query) {
        HashSet terms = new HashSet();
        getTerms(query, terms);
        return (QueryTerm[]) terms.toArray(new QueryTerm[0]);
    }

    /**
     * Internal method to extract terms from different query types.
     * 
     * @param query
     *            The query.
     * @param terms
     *            List of extracted query terms.
     */
    private static final void getTerms(Query query, HashSet terms) {
        if (query instanceof BooleanQuery) {
            BooleanQuery bq = (BooleanQuery) query;
            BooleanClause[] queryClauses = bq.getClauses();

            for (int i = 0; i < queryClauses.length; i++)
                // if (!queryClauses[i].prohibited)
                if (!(queryClauses[i].getOccur() == BooleanClause.Occur.MUST_NOT))
                    getTerms(queryClauses[i].getQuery(), terms);
        } else if (query instanceof PhraseQuery) {
            PhraseQuery pq = (PhraseQuery) query;
            Term[] queryTerms = pq.getTerms();

            for (int i = 0; i < queryTerms.length; i++)
                terms.add(new QueryTerm(queryTerms[i].field(), queryTerms[i]
                        .text()));
        } else if (query instanceof TermQuery) {
            TermQuery tq = (TermQuery) query;
            terms.add(new QueryTerm(tq.getTerm().field(), tq.getTerm().text()));
        } else if (query instanceof WildcardQuery) {
            WildcardQuery wq = (WildcardQuery) query;
            terms.add(new QueryTerm(wq.getTerm().field(), wq.getTerm().text()));
        } else if (query instanceof PrefixQuery) {
            PrefixQuery pq = (PrefixQuery) query;
            terms.add(new QueryTerm(pq.getPrefix().field(), pq.getPrefix()
                    .text()));
        } else if (query instanceof FuzzyQuery) {
            FuzzyQuery fq = (FuzzyQuery) query;
            terms.add(new QueryTerm(fq.getTerm().field(), fq.getTerm().text()));
        } else if (query instanceof RangeQuery) {
            RangeQuery rq = (RangeQuery) query;
            terms.add(new QueryTerm(rq.getLowerTerm().field(), rq
                    .getLowerTerm().text()));
            terms.add(new QueryTerm(rq.getUpperTerm().field(), rq
                    .getUpperTerm().text()));
        }

    }

}
