package org.gcube.informationsystem.icproxy.resources;

import static org.gcube.resources.discovery.icclient.ICFactory.clientFor;
import static org.gcube.resources.discovery.icclient.ICFactory.queryFor;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.Authorization;

import java.util.List;

import javax.validation.constraints.NotNull;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.MediaType;

import lombok.extern.slf4j.Slf4j;

import org.gcube.common.resources.gcore.GenericResource;
import org.gcube.common.scope.api.ScopeProvider;
import org.gcube.informationsystem.icproxy.swagger.Bootstrap;
import org.gcube.resources.discovery.client.api.DiscoveryClient;
import org.gcube.resources.discovery.client.queries.api.SimpleQuery;
import org.slf4j.LoggerFactory;

@Path("GenericResource")
@Slf4j
@Api(value="GenericResource", authorizations={@Authorization(value = Bootstrap.GCUBE_TOKEN_IN_QUERY_DEF), @Authorization(value = Bootstrap.GCUBE_TOKEN_IN_HEADER_DEF)})
public class GenericResourceResource {
	
	private static final org.slf4j.Logger log = LoggerFactory.getLogger(GenericResourceResource.class);

	
	@GET
	@Path("/{secondaryType}/{name}")
	@Produces(MediaType.APPLICATION_XML)
	@ApiOperation(value = "Retrieve a Generic Resource by specifying the secondary type and name within the path.", response=String.class, 
	notes="Retrieve a Generic Resource by specifying the secondary type and name within the path. For example: "
			+ "/icproxy/gcube/service/GenericResource/GenericResource/ScenarioCollectionInfo retrieves the Generic Resource having secondary type GenericResource and name ScenarioCollectionInfo."
			, nickname="get-generic-resource")
	public List<GenericResource> retrieveByTypeAndName(@NotNull @PathParam("secondaryType") String secondaryType, 
			@NotNull @PathParam("name") String resourceName) {
		
		String scope = ScopeProvider.instance.get();
		log.info("genericResource called with secondaryType {} and name {} in scope {}",secondaryType, resourceName, scope);
		SimpleQuery query = createSecondaryTypeQuery(secondaryType)
				.addCondition(String.format("$resource/Profile/Name/text() eq '%s'",resourceName));
		DiscoveryClient<GenericResource> client = clientFor(GenericResource.class);
		List<GenericResource> endpoints = client.submit(query);
		log.debug("retrieved resources are "+endpoints.size());
		return endpoints; 
	}
	
	@GET
	@Path("/{secondaryType}")
	@Produces(MediaType.APPLICATION_XML)
	@ApiOperation(value = "Retrieve a list of Generic Resources by specifying the secondary type within the path.", response=String.class, 
	notes="Retrieve a list of Generic Resources by specifying the secondary type within the path. For example: "
			+ "/ic-proxy/gcube/service/GenericResource/GenericResource/ retrieves the Generic Resources having secondary type GenericResource."
			, nickname="get-generic-resources-by-secondary-type")
	public List<GenericResource> retrieveByType(@NotNull @PathParam("secondaryType") String secondaryType) {
		String scope = ScopeProvider.instance.get();
		log.info("genericResource called with secondaryType {} in scope {}",secondaryType, scope);
		SimpleQuery query = createSecondaryTypeQuery(secondaryType);
		DiscoveryClient<GenericResource> client = clientFor(GenericResource.class);
		List<GenericResource> endpoints = client.submit(query);
		log.debug("retrieved resources are "+endpoints.size());
		return endpoints; 
	}
	
	SimpleQuery createSecondaryTypeQuery(String secondaryTpe){
		SimpleQuery query = queryFor(GenericResource.class);
		query.addCondition(String.format("$resource/Profile/SecondaryType/text() eq '%s'",secondaryTpe));
		return query;
	}
	
}
