package org.gcube.informationsystem.resourceregistry.client;

import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import org.gcube.common.gxhttp.reference.GXConnection;
import org.gcube.common.gxhttp.request.GXHTTPStringRequest;
import org.gcube.informationsystem.base.reference.Element;
import org.gcube.informationsystem.base.reference.IdentifiableElement;
import org.gcube.informationsystem.context.reference.entities.Context;
import org.gcube.informationsystem.model.reference.entities.Entity;
import org.gcube.informationsystem.model.reference.entities.Facet;
import org.gcube.informationsystem.model.reference.entities.Resource;
import org.gcube.informationsystem.model.reference.relations.ConsistsOf;
import org.gcube.informationsystem.model.reference.relations.IsRelatedTo;
import org.gcube.informationsystem.model.reference.relations.Relation;
import org.gcube.informationsystem.resourceregistry.api.contexts.ContextCache;
import org.gcube.informationsystem.resourceregistry.api.contexts.ContextCacheRenewal;
import org.gcube.informationsystem.resourceregistry.api.contexts.ContextUtility;
import org.gcube.informationsystem.resourceregistry.api.exceptions.AvailableInAnotherContextException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.NotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.context.ContextNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.query.InvalidQueryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.schema.SchemaNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.rest.AccessPath;
import org.gcube.informationsystem.resourceregistry.api.rest.httputils.HTTPUtility;
import org.gcube.informationsystem.resourceregistry.api.utils.Utility;
import org.gcube.informationsystem.types.TypeMapper;
import org.gcube.informationsystem.types.reference.Type;
import org.gcube.informationsystem.utils.ElementMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Luca Frosini (ISTI - CNR)
 */
public class ResourceRegistryClientImpl implements ResourceRegistryClient {
	
	private static final Logger logger = LoggerFactory.getLogger(ResourceRegistryClientImpl.class);
	
	protected final String address;
	
	private GXHTTPStringRequest includeAdditionalQueryParameters(GXHTTPStringRequest gxHTTPStringRequest) throws UnsupportedEncodingException{
		return includeAdditionalQueryParameters(gxHTTPStringRequest, null);
	}
	
	private GXHTTPStringRequest includeAdditionalQueryParameters(GXHTTPStringRequest gxHTTPStringRequest, Map<String,String> queryParams) throws UnsupportedEncodingException{
		gxHTTPStringRequest = checkHierarchicalMode(gxHTTPStringRequest, queryParams);
		return checkIncludeContextsInInstanceHeader(gxHTTPStringRequest, queryParams);
	}
	
	private GXHTTPStringRequest checkHierarchicalMode(GXHTTPStringRequest gxHTTPStringRequest, Map<String,String> queryParams) throws UnsupportedEncodingException{
		if(ResourceRegistryClientFactory.isHierarchicalMode()) {
			if(queryParams==null) {
				queryParams = new HashMap<>();
			}
			queryParams.put(AccessPath.HIERARCHICAL_MODE_PARAM, Boolean.toString(true));
		}
		return gxHTTPStringRequest.queryParams(queryParams);
	}
	
	private GXHTTPStringRequest checkIncludeContextsInInstanceHeader(GXHTTPStringRequest gxHTTPStringRequest, Map<String,String> queryParams) throws UnsupportedEncodingException{
		if(ResourceRegistryClientFactory.includeContextsInInstanceHeader()) {
			if(queryParams==null) {
				queryParams = new HashMap<>();
			}
			queryParams.put(AccessPath.INCLUDE_CONTEXTS_IN_HEADER_PARAM, Boolean.toString(true));
		}
		return gxHTTPStringRequest.queryParams(queryParams);
	}
	
	
	protected ContextCacheRenewal contextCacheRenewal = new ContextCacheRenewal() {
		
		@Override
		public List<Context> renew() throws ResourceRegistryException {
			return getAllContextFromServer();
		}
		
	};
	
	public ResourceRegistryClientImpl(String address) {
		this.address = address;
		ContextCache contextCache = ContextCache.getInstance();
		contextCache.setContextCacheRenewal(contextCacheRenewal);
	}
	
	public List<Context> getAllContextFromServer() throws ResourceRegistryException {
		try {
			logger.info("Going to read all {}s", Context.NAME);
			GXHTTPStringRequest gxHTTPStringRequest = GXHTTPStringRequest.newRequest(address); 
			gxHTTPStringRequest.from(ResourceRegistryClient.class.getSimpleName());
			gxHTTPStringRequest.header("Accept", GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(AccessPath.ACCESS_PATH_PART);
			gxHTTPStringRequest.path(AccessPath.CONTEXTS_PATH_PART);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.get();
			String ret = HTTPUtility.getResponse(String.class, httpURLConnection);
			
			logger.debug("Got Contexts are {}", ret);
			return ElementMapper.unmarshalList(Context.class, ret);
			
		} catch(ResourceRegistryException e) {
			// logger.trace("Error while getting {} schema for {}", polymorphic ?
			// AccessPath.POLYMORPHIC_PARAM + " " : "",
			// type, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error while getting {}schema for {}", polymorphic ?
			// AccessPath.POLYMORPHIC_PARAM + " " : "",
			// type, e);
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public List<Context> getAllContext() throws ResourceRegistryException {
		ContextCache contextCache = ContextCache.getInstance();
		return contextCache.getContexts();
	}
	
	protected Context getContextFromServer(String id) throws ContextNotFoundException, ResourceRegistryException {
		try {
			// TODO use cache
			
			logger.info("Going to get current {} ", Context.NAME);
			GXHTTPStringRequest gxHTTPStringRequest = GXHTTPStringRequest.newRequest(address); 
			gxHTTPStringRequest.from(ResourceRegistryClient.class.getSimpleName());
			gxHTTPStringRequest.header("Accept", GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(AccessPath.ACCESS_PATH_PART);
			gxHTTPStringRequest.path(AccessPath.CONTEXTS_PATH_PART);
			gxHTTPStringRequest.path(id);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.get();
			Context context = HTTPUtility.getResponse(Context.class, httpURLConnection);
			
			logger.debug("Got Context is {}", ElementMapper.marshal(context));
			return context;
		} catch(ResourceRegistryException e) {
			// logger.trace("Error while getting {} schema for {}", polymorphic ?
			// AccessPath.POLYMORPHIC_PARAM + " " : "",
			// type, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error while getting {}schema for {}", polymorphic ?
			// AccessPath.POLYMORPHIC_PARAM + " " : "",
			// type, e);
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public Context getContext(UUID uuid) throws ContextNotFoundException, ResourceRegistryException {
		return ContextCache.getInstance().getContextByUUID(uuid);
	}
	
	@Override
	public Context getCurrentContext() throws ContextNotFoundException, ResourceRegistryException {
		String contextFullName = ResourceRegistryClientFactory.getCurrentContextFullName();
		ContextCache contextCache = ContextCache.getInstance();
		UUID uuid = contextCache.getUUIDByFullName(contextFullName);
		Context context = null;
		if(uuid == null) {
			context = getContextFromServer(AccessPath.CURRENT_CONTEXT);
			contextCache.cleanCache();
			contextCache.refreshContextsIfNeeded();
			Context c = contextCache.getContextByUUID(context.getHeader().getUUID());
			if(c!=null){
				context = c;
			}else {
				logger.error("Current Context is {}. It is possibile to get it from the server but not from the cache. This is very strange and should not occur.", contextFullName);
			}
		}else {
			context = contextCache.getContextByUUID(uuid);
		}
		return context;
	}
	
	@Override
	public <E extends Element> List<Type> getSchema(Class<E> clazz, Boolean polymorphic)
			throws SchemaNotFoundException, ResourceRegistryException {
		String type = Utility.getTypeName(clazz);
		try {
			logger.info("Going to get {} schema", type);
			GXHTTPStringRequest gxHTTPStringRequest = GXHTTPStringRequest.newRequest(address); 
			gxHTTPStringRequest.from(ResourceRegistryClient.class.getSimpleName());
			gxHTTPStringRequest.header("Accept", GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(AccessPath.ACCESS_PATH_PART);
			gxHTTPStringRequest.path(AccessPath.TYPES_PATH_PART);
			gxHTTPStringRequest.path(type);
			
			Map<String,String> parameters = new HashMap<>();
			parameters.put(AccessPath.POLYMORPHIC_PARAM, polymorphic.toString());
			gxHTTPStringRequest.queryParams(parameters);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.get();
			String json = HTTPUtility.getResponse(String.class, httpURLConnection);

			logger.debug("Got schema for {} is {}", type, json);
			return TypeMapper.deserializeTypeDefinitions(json);
		} catch(ResourceRegistryException e) {
			// logger.trace("Error while getting {} schema for {}", polymorphic ? AccessPath.POLYMORPHIC_PARAM + " " : "",
			//		type, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error while getting {}schema for {}", polymorphic ?
			// AccessPath.POLYMORPHIC_PARAM + " " : "",
			// type, e);
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public <IE extends IdentifiableElement> boolean exists(Class<IE> clazz, UUID uuid)
			throws NotFoundException, AvailableInAnotherContextException, ResourceRegistryException {
		String type = Utility.getTypeName(clazz);
		return exists(type, uuid);
	}
	
	@Override
	public boolean exists(String type, UUID uuid)
			throws NotFoundException, AvailableInAnotherContextException, ResourceRegistryException {
		try {
			logger.info("Going to check if {} with UUID {} exists", type, uuid);
			GXHTTPStringRequest gxHTTPStringRequest = GXHTTPStringRequest.newRequest(address); 
			gxHTTPStringRequest.from(ResourceRegistryClient.class.getSimpleName());
			gxHTTPStringRequest.header("Accept", GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(AccessPath.ACCESS_PATH_PART);
			gxHTTPStringRequest.path(AccessPath.INSTANCES_PATH_PART);
			gxHTTPStringRequest.path(type);
			gxHTTPStringRequest.path(uuid.toString());
			
			includeAdditionalQueryParameters(gxHTTPStringRequest);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.head();
			HTTPUtility.getResponse(String.class, httpURLConnection);
			
			logger.debug("{} with UUID {} exists", type, uuid);
			return true;
		} catch(ResourceRegistryException e) {
			// logger.trace("Error while checking if {} with UUID {} exists.", type, uuid,
			// e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error while checking if {} with UUID {} exists.", type, uuid,
			// e);
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public <IE extends IdentifiableElement> IE getInstance(Class<IE> clazz, UUID uuid)
			throws NotFoundException, AvailableInAnotherContextException, ResourceRegistryException {
		String type = Utility.getTypeName(clazz);
		String ret = getInstance(type, uuid);
		try {
			return ElementMapper.unmarshal(clazz, ret);
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	public String getInstance(String type, UUID uuid)
			throws NotFoundException, AvailableInAnotherContextException, ResourceRegistryException {
		try {
			logger.info("Going to get {} with UUID {}", type, uuid);
			GXHTTPStringRequest gxHTTPStringRequest = GXHTTPStringRequest.newRequest(address); 
			gxHTTPStringRequest.from(ResourceRegistryClient.class.getSimpleName());
			gxHTTPStringRequest.header("Accept", GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(AccessPath.ACCESS_PATH_PART);
			gxHTTPStringRequest.path(AccessPath.INSTANCES_PATH_PART);
			gxHTTPStringRequest.path(type);
			gxHTTPStringRequest.path(uuid.toString());
			
			includeAdditionalQueryParameters(gxHTTPStringRequest);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.get();
			String ret = HTTPUtility.getResponse(String.class, httpURLConnection);
			
			logger.debug("Got {} with UUID {} is {}", type, uuid, ret);
			return ret;
		} catch(ResourceRegistryException e) {
			// logger.trace("Error while getting {} with UUID {}", type, uuid, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error while getting {} with UUID {}", type, uuid, e);
			throw new RuntimeException(e);
		}
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public <IE extends IdentifiableElement, R extends Resource> List<R> getInstances(Class<IE> clazz, Boolean polymorphic)
			throws ResourceRegistryException {
		String type = Utility.getTypeName(clazz);
		String ret = getInstances(type, polymorphic);
		try {
			return (List<R>) ElementMapper.unmarshalList(Resource.class, ret);
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public String getInstances(String type, Boolean polymorphic) throws ResourceRegistryException {
		try {
			logger.info("Going to get all instances of {} ", type);
			GXHTTPStringRequest gxHTTPStringRequest = GXHTTPStringRequest.newRequest(address); 
			gxHTTPStringRequest.from(ResourceRegistryClient.class.getSimpleName());
			gxHTTPStringRequest.header("Accept", GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(AccessPath.ACCESS_PATH_PART);
			gxHTTPStringRequest.path(AccessPath.INSTANCES_PATH_PART);
			gxHTTPStringRequest.path(type);
			
			Map<String,String> parameters = new HashMap<>();
			parameters.put(AccessPath.POLYMORPHIC_PARAM, polymorphic.toString());
			includeAdditionalQueryParameters(gxHTTPStringRequest, parameters);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.get();
			
			String ret = HTTPUtility.getResponse(String.class, httpURLConnection);
			
			logger.debug("Got instances of {} are {}", type, ret);
			return ret;
		} catch(ResourceRegistryException e) {
			// logger.trace("Error while getting {} instances", type, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error while getting {} instances", type, e);
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public String query(String query, int limit, String fetchPlan)
			throws InvalidQueryException, ResourceRegistryException {
		return query(query, limit, fetchPlan, false);
	}
	
	@Override
	public String query(String query, int limit, String fetchPlan, boolean raw)
			throws InvalidQueryException, ResourceRegistryException {
		
		try {
			logger.info("Going to query. {}", query);
			GXHTTPStringRequest gxHTTPStringRequest = GXHTTPStringRequest.newRequest(address); 
			gxHTTPStringRequest.from(ResourceRegistryClient.class.getSimpleName());
			gxHTTPStringRequest.header("Accept", GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(AccessPath.ACCESS_PATH_PART);
			gxHTTPStringRequest.path(AccessPath.QUERY_PATH_PART);
			
			Map<String,String> parameters = new HashMap<>();
			parameters.put(AccessPath.QUERY_PARAM, query);
			if(limit <= 0) {
				limit = AccessPath.UNBOUNDED;
			}
			parameters.put(AccessPath.LIMIT_PARAM, Integer.toString(limit));
			parameters.put(AccessPath.RAW_PARAM, Boolean.toString(raw));
			
			if(fetchPlan != null && fetchPlan.compareTo("")!=0) {
				parameters.put(AccessPath.FETCH_PLAN_PARAM, fetchPlan);
			}
			
			includeAdditionalQueryParameters(gxHTTPStringRequest, parameters);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.get();

			String ret = HTTPUtility.getResponse(String.class, httpURLConnection);
			
			logger.debug("Query result is {}", ret);
			return ret;
		} catch(ResourceRegistryException e) {
			// logger.trace("Error while querying", e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error while querying", e);
			throw new RuntimeException(e);
		}
	}
	
	protected String getRelated(String entityType, String relationType, String referenceEntityType,
			UUID referenceEntity, Direction direction, Boolean polymorphic, Map<String,String> map)
			throws ResourceRegistryException {
		
		try {
			GXHTTPStringRequest gxHTTPStringRequest = GXHTTPStringRequest.newRequest(address); 
			gxHTTPStringRequest.from(ResourceRegistryClient.class.getSimpleName());
			gxHTTPStringRequest.header("Accept", GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(AccessPath.ACCESS_PATH_PART);
			gxHTTPStringRequest.path(AccessPath.QUERY_PATH_PART);
			gxHTTPStringRequest.path(entityType);
			gxHTTPStringRequest.path(relationType);
			gxHTTPStringRequest.path(referenceEntityType);
			
			Map<String,String> parameters = new HashMap<>();
			parameters.put(AccessPath.DIRECTION_PARAM, direction.name());
			parameters.put(AccessPath.POLYMORPHIC_PARAM, polymorphic.toString());
			
			if(referenceEntity == null) {
				if(map != null && map.size() > 0) {
					logger.info("Going to get {} linked by a {} Relation to a {} having {}", entityType, relationType,
							referenceEntityType, map);
					parameters.putAll(map);
				} else {
					logger.info("Going to get {} linked by a {} Relation to a {}", entityType, relationType,
							referenceEntityType);
				}
			} else {
				logger.info("Going to get {} linked by {} to {} with UUID {}", entityType, relationType,
						referenceEntityType, referenceEntity);
				parameters.put(AccessPath.REFERENCE_PARAM, referenceEntity.toString());
			}
			
			includeAdditionalQueryParameters(gxHTTPStringRequest, parameters);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.get();

			String json = HTTPUtility.getResponse(String.class, httpURLConnection);
			
			if(referenceEntity == null) {
				logger.info("{} linked by {} to/from {} having {} are {}", entityType, relationType,
						referenceEntityType, map, json);
				
			} else {
				logger.info("{} linked by {} to/from {} with UUID {} are", entityType, relationType,
						referenceEntityType, referenceEntity, json);
			}
			
			return json;
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
		
	}
	
	@Override
	public <R extends Resource, C extends ConsistsOf<?,?>, F extends Facet> List<R> getResourcesFromReferenceFacet(
			Class<R> resourceClass, Class<C> consistsOfClass, Class<F> facetClass, F referenceFacet,
			boolean polymorphic) throws ResourceRegistryException {
		UUID referenceFacetUUID = referenceFacet.getHeader().getUUID();
		return getResourcesFromReferenceFacet(resourceClass, consistsOfClass, facetClass, referenceFacetUUID,
				polymorphic);
	}
	
	@SuppressWarnings("unchecked")
	public <R extends Resource, C extends ConsistsOf<?,?>, F extends Facet> List<R> getResourcesFromReferenceFacet(
			Class<R> resourceClass, Class<C> consistsOfClass, Class<F> facetClass, UUID referenceFacetUUID,
			boolean polymorphic) throws ResourceRegistryException {
		String resourceType = Utility.getTypeName(resourceClass);
		String consistsOfType = Utility.getTypeName(consistsOfClass);
		String facetType = Utility.getTypeName(facetClass);
		String ret = getResourcesFromReferenceFacet(resourceType, consistsOfType, facetType, referenceFacetUUID,
				polymorphic);
		try {
			return (List<R>) ElementMapper.unmarshalList(Resource.class, ret);
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public String getResourcesFromReferenceFacet(String resourceType, String consistsOfType, String facetType,
			UUID facetUUID, boolean polymorphic) throws ResourceRegistryException {
		return getRelated(resourceType, consistsOfType, facetType, facetUUID, Direction.OUT, polymorphic);
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public <R extends Resource, C extends ConsistsOf<?,?>, F extends Facet> List<R> getFilteredResources(
			Class<R> resourceClass, Class<C> consistsOfClass, Class<F> facetClass, boolean polymorphic,
			Map<String,String> map) throws ResourceRegistryException {
		String resourceType = Utility.getTypeName(resourceClass);
		String consistsOfType = Utility.getTypeName(consistsOfClass);
		String facetType = Utility.getTypeName(facetClass);
		String ret = getFilteredResources(resourceType, consistsOfType, facetType, polymorphic, map);
		try {
			return (List<R>) ElementMapper.unmarshalList(Resource.class, ret);
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public String getFilteredResources(String resourceType, String consistsOfType, String facetType,
			boolean polymorphic, Map<String,String> map) throws ResourceRegistryException {
		return getRelated(resourceType, consistsOfType, facetType, Direction.OUT, polymorphic, map);
	}
	
	@Override
	public <R extends Resource, I extends IsRelatedTo<?,?>, RR extends Resource> List<R> getRelatedResourcesFromReferenceResource(
			Class<R> resourceClass, Class<I> isRelatedToClass, Class<RR> referenceResourceClass, RR referenceResource,
			Direction direction, boolean polymorphic) throws ResourceRegistryException {
		UUID referenceResourceUUID = referenceResource.getHeader().getUUID();
		return getRelatedResourcesFromReferenceResource(resourceClass, isRelatedToClass, referenceResourceClass,
				referenceResourceUUID, direction, polymorphic);
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public <R extends Resource, I extends IsRelatedTo<?,?>, RR extends Resource> List<R> getRelatedResourcesFromReferenceResource(
			Class<R> resourceClass, Class<I> isRelatedToClass, Class<RR> referenceResourceClass,
			UUID referenceResourceUUID, Direction direction, boolean polymorphic) throws ResourceRegistryException {
		String resourceType = Utility.getTypeName(resourceClass);
		String isRelatedToType = Utility.getTypeName(isRelatedToClass);
		String referenceResourceType = Utility.getTypeName(referenceResourceClass);
		String ret = getRelatedResourcesFromReferenceResource(resourceType, isRelatedToType, referenceResourceType,
				referenceResourceUUID, direction, polymorphic);
		try {
			return (List<R>) ElementMapper.unmarshalList(Resource.class, ret);
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public String getRelatedResourcesFromReferenceResource(String resourceType, String isRelatedToType,
			String referenceResourceType, UUID referenceResourceUUID, Direction direction, boolean polymorphic)
			throws ResourceRegistryException {
		return getRelated(resourceType, isRelatedToType, referenceResourceType, referenceResourceUUID, direction,
				polymorphic);
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public <R extends Resource, I extends IsRelatedTo<?,?>, RR extends Resource> List<R> getRelatedResources(
			Class<R> resourceClass, Class<I> isRelatedToClass, Class<RR> referenceResourceClass, Direction direction,
			boolean polymorphic) throws ResourceRegistryException {
		String resourceType = Utility.getTypeName(resourceClass);
		String isRelatedToType = Utility.getTypeName(isRelatedToClass);
		String referenceResourceType = Utility.getTypeName(referenceResourceClass);
		String ret = getRelatedResources(resourceType, isRelatedToType, referenceResourceType, direction, polymorphic);
		try {
			return (List<R>) ElementMapper.unmarshalList(Resource.class, ret);
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public String getRelatedResources(String resourceType, String isRelatedToType, String referenceResourceType,
			Direction direction, boolean polymorphic) throws ResourceRegistryException {
		return getRelated(resourceType, isRelatedToType, referenceResourceType, direction, polymorphic, null);
	}
	
	@SuppressWarnings("unchecked")
	// @Override
	protected <E extends Entity, R extends Relation<?,?>, RE extends Entity> List<E> getRelated(Class<E> entityClass,
			Class<R> relationClass, Class<RE> referenceEntityClass, Direction direction, boolean polymorphic,
			Map<String,String> map) throws ResourceRegistryException {
		String entityType = Utility.getTypeName(entityClass);
		String relationType = Utility.getTypeName(relationClass);
		String referenceEntityType = Utility.getTypeName(referenceEntityClass);
		String ret = getRelated(entityType, relationType, referenceEntityType, direction, polymorphic, map);
		try {
			return (List<E>) ElementMapper.unmarshalList(Resource.class, ret);
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	// @Override
	protected String getRelated(String entityType, String relationType, String referenceEntityType, Direction direction,
			boolean polymorphic, Map<String,String> map) throws ResourceRegistryException {
		return getRelated(entityType, relationType, referenceEntityType, null, direction, polymorphic, map);
	}
	
	// @Override
	protected  <E extends Entity, R extends Relation<?,?>, RE extends Entity> List<E> getRelated(Class<E> entityClass,
			Class<R> relationClass, Class<RE> referenceEntityClass, RE referenceEntity, Direction direction,
			boolean polymorphic) throws ResourceRegistryException {
		UUID referenceEntityUUID = referenceEntity.getHeader().getUUID();
		return getRelated(entityClass, relationClass, referenceEntityClass, referenceEntityUUID, direction,
				polymorphic);
	}
	
	@SuppressWarnings("unchecked")
	// @Override
	protected  <E extends Entity, R extends Relation<?,?>, RE extends Entity> List<E> getRelated(Class<E> entityClass,
			Class<R> relationClass, Class<RE> referenceEntityClass, UUID referenceEntityUUID, Direction direction,
			boolean polymorphic) throws ResourceRegistryException {
		String entityType = Utility.getTypeName(entityClass);
		String relationType = Utility.getTypeName(relationClass);
		String referenceEntityType = Utility.getTypeName(referenceEntityClass);
		String ret = getRelated(entityType, relationType, referenceEntityType, referenceEntityUUID, direction,
				polymorphic);
		try {
			return (List<E>) ElementMapper.unmarshalList(Resource.class, ret);
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	// @Override
	protected  String getRelated(String entityType, String relationType, String referenceEntityType, UUID referenceEntity,
			Direction direction, boolean polymorphic) throws ResourceRegistryException {
		return getRelated(entityType, relationType, referenceEntityType, referenceEntity, direction, polymorphic, null);
	}

	@Override
	public <IE extends IdentifiableElement> Set<UUID> getInstanceContexts(Class<IE> clazz, UUID uuid)
			throws NotFoundException, AvailableInAnotherContextException, ResourceRegistryException {
		String typeName = Utility.getTypeName(clazz);
		return getInstanceContexts(typeName, uuid);
	}

	@Override
	public Set<UUID> getInstanceContexts(String type, UUID uuid)
			throws NotFoundException, AvailableInAnotherContextException, ResourceRegistryException {
		try {
			logger.trace("Going to get contexts of {} with UUID {}", type, uuid);
			GXHTTPStringRequest gxHTTPStringRequest = GXHTTPStringRequest.newRequest(address); 
			gxHTTPStringRequest.from(ResourceRegistryClient.class.getSimpleName());
			gxHTTPStringRequest.header("Accept", GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(AccessPath.ACCESS_PATH_PART);
			gxHTTPStringRequest.path(AccessPath.INSTANCES_PATH_PART);
			gxHTTPStringRequest.path(type);
			gxHTTPStringRequest.path(uuid.toString());
			gxHTTPStringRequest.path(AccessPath.CONTEXTS_PATH_PART);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.get();
			String jsonArray = HTTPUtility.getResponse(String.class, httpURLConnection);

			logger.info("Contexts of {} with UUID {} are {}", type, uuid, jsonArray);
			
			Set<UUID> contexts = ContextUtility.getContextUUIDSet(jsonArray);
			return contexts;
		} catch(ResourceRegistryException e) {
			// logger.trace("Error while getting {} with UUID {}", type, uuid, e);
			throw e;
		} catch(Exception e) {
			// logger.trace("Error while getting {} with UUID {}", type, uuid, e);
			throw new RuntimeException(e);
		}
	}
	
}
