package org.gcube.informationsystem.resourceregistry.contexts;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import org.gcube.com.fasterxml.jackson.core.JsonProcessingException;
import org.gcube.common.gxhttp.reference.GXConnection;
import org.gcube.common.gxhttp.request.GXHTTPStringRequest;
import org.gcube.common.security.providers.SecretManagerProvider;
import org.gcube.common.security.secrets.Secret;
import org.gcube.informationsystem.contexts.reference.ContextState;
import org.gcube.informationsystem.contexts.reference.entities.Context;
import org.gcube.informationsystem.resourceregistry.api.contexts.ContextCache;
import org.gcube.informationsystem.resourceregistry.api.contexts.ContextCacheRenewal;
import org.gcube.informationsystem.resourceregistry.api.exceptions.NotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.contexts.ContextAlreadyPresentException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.contexts.ContextNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.request.BaseRequestInfo;
import org.gcube.informationsystem.resourceregistry.api.rest.ContextPath;
import org.gcube.informationsystem.resourceregistry.api.rest.httputils.HTTPUtility;
import org.gcube.informationsystem.serialization.ElementMapper;
import org.gcube.informationsystem.utils.UUIDManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Luca Frosini (ISTI - CNR)
 */
public class ResourceRegistryContextClientImpl extends BaseRequestInfo implements ResourceRegistryContextClient {
	
	private static final Logger logger = LoggerFactory.getLogger(ResourceRegistryContextClientImpl.class);
	
	private static final String ACCEPT_HTTP_HEADER_KEY = "Accept";
	private static final String CONTENT_TYPE_HTTP_HEADER_KEY = "Content-Type";
	
	/** The base address of the Resource Registry service */
	protected final String address;
	
	/** HTTP headers to be included in requests */
	protected Map<String, String> headers;
	
	/** Cache for context information to improve performance */
	protected ContextCache contextCache;

	private void addOptionalQueryParameters(Map<String,String> queryParams) throws UnsupportedEncodingException {
		addIncludeMeta(queryParams);
	}
	
	private GXHTTPStringRequest includeAdditionalQueryParameters(GXHTTPStringRequest gxHTTPStringRequest) throws UnsupportedEncodingException{
		Map<String,String> queryParams = new HashMap<>();
		return includeAdditionalQueryParameters(gxHTTPStringRequest, queryParams);
	}
	
	private GXHTTPStringRequest includeAdditionalQueryParameters(GXHTTPStringRequest gxHTTPStringRequest, Map<String,String> queryParams) throws UnsupportedEncodingException{
		if(queryParams==null) {
			queryParams = new HashMap<>();
		}
		addOptionalQueryParameters(queryParams);
		return gxHTTPStringRequest.queryParams(queryParams);
	}
	
	private void addIncludeMeta(Map<String,String> queryParams) throws UnsupportedEncodingException{
		addIncludeMeta(queryParams, includeMeta);
	}
	
	private void addIncludeMeta(Map<String,String> queryParams, boolean includeMeta) throws UnsupportedEncodingException{
		if(includeMeta) {
			queryParams.put(ContextPath.INCLUDE_META_QUERY_PARAMETER, Boolean.toString(includeMeta));
		}
	}
	
	/*
	private void addOffset(Map<String,String> queryParams) throws UnsupportedEncodingException{
		addOffset(queryParams, offset);
	}
	*/
	
	private void addOffset(Map<String,String> queryParams, Integer offset) throws UnsupportedEncodingException{
		if(offset!=null) {
			queryParams.put(ContextPath.OFFSET_QUERY_PARAMETER, offset.toString());
		}
	}
	
	/*
	private void addLimit(Map<String,String> queryParams) throws UnsupportedEncodingException{
		addLimit(queryParams, limit);
	}
	*/
	
	private void addLimit(Map<String,String> queryParams, Integer limit) throws UnsupportedEncodingException{
		if(limit!=null) {
			queryParams.put(ContextPath.LIMIT_QUERY_PARAMETER, limit.toString());
		}
	}
	
	/** Context cache renewal strategy for refreshing cached context information */
	protected ContextCacheRenewal contextCacheRenewal = new ContextCacheRenewal() {
		
		@Override
		public List<Context> renew() throws ResourceRegistryException {
			return getAllContextFromServer(true, 0, BaseRequestInfo.UNBOUNDED_LIMIT);
		}
		
	};
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addHeader(String name, String value) {
		headers.put(name, value);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addHeader(String name, boolean value) {
		addHeader(name, Boolean.toString(value));
	}

	/**
	 * Creates and configures a GXHTTPStringRequest with authentication headers.
	 * 
	 * @return A configured GXHTTPStringRequest instance ready for making HTTP requests
	 */
	protected GXHTTPStringRequest getGXHTTPStringRequest() {
		GXHTTPStringRequest gxHTTPStringRequest = GXHTTPStringRequest.newRequest(address);
		
		/* It is a gcube request */
		Secret secret = SecretManagerProvider.get();
		Map<String, String> authorizationHeaders = secret.getHTTPAuthorizationHeaders();
		for(String key : authorizationHeaders.keySet()) {
			gxHTTPStringRequest.header(key, authorizationHeaders.get(key));
		}

		gxHTTPStringRequest.from(this.getClass().getSimpleName());
		for(String name : headers.keySet()) {
			gxHTTPStringRequest.header(name, headers.get(name));
		}
		return gxHTTPStringRequest;
	}
	
	/**
	 * This client does not use the shared context cache 
	 * because it is used by managers to manipulate contexts.
	 * The shared cache has only minimal context information
	 * available to any users.
	 * 
	 * @param address The base address of the Resource Registry service
	 */
	public ResourceRegistryContextClientImpl(String address) {
		this.address = address;
		this.headers = new HashMap<>();
		this.includeMeta = false;
		this.contextCache = new ContextCache();
		contextCache.setContextCacheRenewal(contextCacheRenewal);
	}
	
	private void forceCacheRefresh() {
		try {
			contextCache.cleanCache();
			contextCache.refreshContextsIfNeeded();
		}catch (Exception e) {
			logger.warn("Unable to force cache refresh.", e);
		}
	}
	
	
	/**
	 * It reads all the contexts from server. 
	 * The cache used for contexts is bypassed and not updated.
	 * 
	 * @return All Contexts read from server
	 * @throws ResourceRegistryException If an error occurs during the operation
	 */
	public List<Context> getAllContextFromServer() throws ResourceRegistryException {
		return getAllContextFromServer(includeMeta, offset, limit);
	}
	
	/**
	 * Retrieves all contexts from the Resource Registry server with specified parameters.
	 * 
	 * @param includeMeta Whether to include metadata in the response
	 * @param offset The offset for pagination
	 * @param limit The limit for pagination
	 * @return List of all contexts from the server
	 * @throws ResourceRegistryException If an error occurs during the operation
	 */
	protected List<Context> getAllContextFromServer(boolean includeMeta, Integer offset, Integer limit) throws ResourceRegistryException {
		try {
			logger.info("Going to read all {}s", Context.NAME);
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(ContextPath.CONTEXTS_PATH_PART);
			
			Map<String,String> parameters = new HashMap<>();
			addIncludeMeta(parameters, includeMeta);
			addOffset(parameters, offset);
			addLimit(parameters, limit);
			gxHTTPStringRequest.queryParams(parameters);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.get();
			String all = HTTPUtility.getResponse(String.class, httpURLConnection);
			
			logger.debug("Got contexts are {}", Context.NAME, all);
			return ElementMapper.unmarshalList(Context.class, all);
			
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	
	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>Cache-Based Approach:</strong>
	 * <ul>
	 * <li>This implementation uses a local {@link ContextCache} instance for optimal performance</li>
	 * <li>The cache is automatically populated/refreshed via {@link ContextCacheRenewal} when needed</li>
	 * <li>Cache renewal calls {@code getAllContextFromServer(true, 0, UNBOUNDED_LIMIT)} to fetch all contexts</li>
	 * <li>No direct REST API call is made unless the cache needs refreshing</li>
	 * </ul>
	 * 
	 * <strong>Cache Refresh Strategy:</strong>
	 * <ul>
	 * <li>Cache refresh is triggered automatically when {@code contextCache.getContexts()} detects stale data</li>
	 * <li>The refresh operation fetches contexts from server with full metadata enabled</li>
	 * <li>Uses unlimited result set (no pagination) for complete context hierarchy</li>
	 * <li>Cache refresh failures are logged but don't prevent operation completion</li>
	 * </ul>
	 * 
	 * <strong>Performance Characteristics:</strong>
	 * <ul>
	 * <li><strong>First call</strong>: Moderate (requires initial server fetch via {@code getAllContextFromServer()})</li>
	 * <li><strong>Subsequent calls</strong>: Very fast (served from local cache)</li>
	 * <li><strong>After cache expiry</strong>: Moderate (automatic refresh triggered)</li>
	 * </ul>
	 * 
	 * <strong>REST API Mapping:</strong>
	 * When cache refresh is needed, internally calls:
	 * {@code GET /contexts?includeMeta=true&offset=0&limit=-1}
	 * 
	 * <strong>Authorization Handling:</strong>
	 * <ul>
	 * <li>Uses current security context from {@link SecretManagerProvider} for server requests</li>
	 * <li>Authorization headers are automatically added to REST requests via {@code getGXHTTPStringRequest()}</li>
	 * <li>Filtered results based on user's role and permissions as per server-side authorization</li>
	 * </ul>
	 */
	@Override
	public List<Context> all() throws ResourceRegistryException {
		return contextCache.getContexts();
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public ContextCache getContextCache() {
		return contextCache;
	}
	
	/**
	 * Internal method to create a new context in the Resource Registry.
	 * Handles UUID generation if not present and marshals the context to JSON.
	 * 
	 * @param context The context to create
	 * @return The JSON representation of the created context
	 * @throws ContextAlreadyPresentException If a context with the same UUID already exists
	 * @throws ResourceRegistryException If an error occurs during creation
	 */
	protected String internalCreate(Context context) throws ContextAlreadyPresentException, ResourceRegistryException {
		try {
			UUID uuid = context.getID();
			if(uuid == null) {
				uuid = UUIDManager.getInstance().generateValidUUID();
				context.setID(uuid);
			}
			
			String contextString = ElementMapper.marshal(context);
			
			logger.trace("Going to create {}", contextString);
			
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.header(CONTENT_TYPE_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(ContextPath.CONTEXTS_PATH_PART);
			gxHTTPStringRequest.path(uuid.toString());
			
			includeAdditionalQueryParameters(gxHTTPStringRequest);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.put(contextString);
			String c = HTTPUtility.getResponse(String.class, httpURLConnection);
			
			forceCacheRefresh();
			
			logger.trace("{} successfully created", c);
			return c;
			
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	
	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>UUID Generation:</strong>
	 * <ul>
	 * <li>If the context object has no UUID ({@code context.getID() == null}), generates one via {@link UUIDManager}</li>
	 * <li>Generated UUID is automatically assigned to the context object before creation</li>
	 * </ul>
	 * 
	 * <strong>REST API Implementation:</strong>
	 * <ul>
	 * <li>Marshals the context object to JSON using {@link ElementMapper}</li>
	 * <li>Makes HTTP PUT request to: {@code PUT /contexts/{uuid}}</li>
	 * <li>Content-Type: application/json;charset=UTF-8</li>
	 * <li>Includes additional query parameters via {@code includeAdditionalQueryParameters()}</li>
	 * </ul>
	 * 
	 * <strong>Authorization:</strong>
	 * <ul>
	 * <li>Uses current security context from {@link SecretManagerProvider}</li>
	 * <li>Authorization headers automatically added via {@code getGXHTTPStringRequest()}</li>
	 * </ul>
	 * 
	 * <strong>Cache Management:</strong>
	 * <ul>
	 * <li>Forces cache refresh via {@code forceCacheRefresh()} after successful creation</li>
	 * <li>Ensures subsequent operations see the newly created context</li>
	 * </ul>
	 * 
	 * <strong>Error Handling:</strong>
	 * <ul>
	 * <li>Propagates {@link ResourceRegistryException} from server</li>
	 * <li>Wraps other exceptions in {@link RuntimeException}</li>
	 * </ul>
	 */
	@Override
	public Context create(Context context) throws ContextAlreadyPresentException, ResourceRegistryException {
		try {
			String res = internalCreate(context);
			return ElementMapper.unmarshal(Context.class, res);
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>JSON Processing:</strong>
	 * <ul>
	 * <li>Unmarshals the JSON string to a {@link Context} object using {@link ElementMapper}</li>
	 * <li>Delegates to {@code internalCreate(Context)} for the actual creation logic</li>
	 * <li>Returns the raw JSON response from the server</li>
	 * </ul>
	 * 
	 * <strong>UUID Handling:</strong>
	 * <ul>
	 * <li>If the JSON contains no UUID, one is automatically generated (via {@code internalCreate()})</li>
	 * <li>UUID generation follows the same logic as {@code create(Context)}</li>
	 * </ul>
	 * 
	 * <strong>REST API Implementation:</strong>
	 * <ul>
	 * <li>Same REST implementation as {@code create(Context)}: {@code PUT /contexts/{uuid}}</li>
	 * <li>JSON validation occurs during unmarshaling phase</li>
	 * </ul>
	 * 
	 * <strong>Cache Management:</strong>
	 * <ul>
	 * <li>Forces cache refresh after successful creation (same as {@code create(Context)})</li>
	 * </ul>
	 * 
	 * <strong>Error Handling:</strong>
	 * <ul>
	 * <li>Validates JSON format during unmarshaling</li>
	 * <li>Propagates {@link ResourceRegistryException} from server</li>
	 * <li>Wraps JSON processing and other exceptions in {@link RuntimeException}</li>
	 * </ul>
	 */
	@Override
	public String create(String context) throws ContextAlreadyPresentException, ResourceRegistryException {
		try {
			Context c = ElementMapper.unmarshal(Context.class, context);
			return internalCreate(c);
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * Checks if a context with the specified UUID exists on the server.
	 * 
	 * @param uuid The UUID of the context to check
	 * @return true if the context exists, false otherwise
	 * @throws ContextNotFoundException If the context is not found
	 * @throws ResourceRegistryException If an error occurs during the operation
	 */
	public boolean existFromServer(String uuid) throws ContextNotFoundException, ResourceRegistryException {
		try {
			logger.trace("Going to read {} with UUID {}", Context.NAME, uuid);
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(ContextPath.CONTEXTS_PATH_PART);
			gxHTTPStringRequest.path(uuid);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.head();
			HTTPUtility.getResponse(String.class, httpURLConnection);
			
			return true;
		} catch (NotFoundException e) {
			return false;
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	
	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>UUID Conversion:</strong>
	 * <ul>
	 * <li>Converts the string UUID to {@link UUID} object using {@code UUID.fromString()}</li>
	 * <li>Delegates to {@code exist(UUID)} for the actual existence check</li>
	 * </ul>
	 * 
	 * <strong>Error Handling:</strong>
	 * <ul>
	 * <li>Will throw {@link IllegalArgumentException} if the string is not a valid UUID format</li>
	 * <li>Other exceptions are handled by the delegated {@code exist(UUID)} method</li>
	 * </ul>
	 */
	@Override
	public boolean exist(String uuid) throws ResourceRegistryException {
		return exist(UUID.fromString(uuid));
	}
	
	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>Cache-First Strategy:</strong>
	 * <ul>
	 * <li>Uses {@code read(UUID)} method to check existence</li>
	 * <li>Leverages the cache-first approach of the read operation for performance</li>
	 * <li>If context exists in cache, no server call is made (very fast)</li>
	 * <li>If not in cache, makes REST API call via {@code readFromServer()}</li>
	 * </ul>
	 * 
	 * <strong>Existence Logic:</strong>
	 * <ul>
	 * <li>Returns {@code true} if {@code read(UUID)} succeeds</li>
	 * <li>Returns {@code false} if {@code read(UUID)} throws {@link ContextNotFoundException}</li>
	 * <li>Propagates other {@link ResourceRegistryException} types</li>
	 * </ul>
	 * 
	 * <strong>Performance Characteristics:</strong>
	 * <ul>
	 * <li><strong>Very Fast</strong>: When context is in cache (no server call)</li>
	 * <li><strong>Moderate</strong>: When cache miss occurs (requires server round-trip via read)</li>
	 * </ul>
	 * 
	 * <strong>Authorization Behavior:</strong>
	 * <ul>
	 * <li>Returns {@code false} for contexts that exist but are not accessible to the current user</li>
	 * <li>This is because the underlying {@code read()} operation will fail with authorization errors</li>
	 * <li>Effectively combines existence and accessibility checking</li>
	 * </ul>
	 */
	@Override
	public boolean exist(UUID uuid) throws ResourceRegistryException {
		try {
			read(uuid);
			return true;
		}catch (ContextNotFoundException e) {
			return false;
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>UUID Extraction:</strong>
	 * <ul>
	 * <li>Extracts UUID from the context object via {@code context.getID()}</li>
	 * <li>Delegates to {@code read(UUID)} for the actual retrieval operation</li>
	 * </ul>
	 * 
	 * <strong>Convenience Method:</strong>
	 * <ul>
	 * <li>Provides a convenient way to refresh/reload a context object</li>
	 * <li>Useful when you have a context reference but want the latest server data</li>
	 * </ul>
	 */
	@Override
	public Context read(Context context) throws ContextNotFoundException, ResourceRegistryException {
		return read(context.getID());
	}
	
	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>Cache-First Strategy:</strong>
	 * <ul>
	 * <li>First checks the local {@link ContextCache} via {@code contextCache.getContextByUUID(uuid)}</li>
	 * <li>If found in cache, returns immediately (very fast path)</li>
	 * <li>If not found in cache, calls {@code readFromServer(uuid.toString())} to fetch from server</li>
	 * </ul>
	 * 
	 * <strong>Server Fetch Process:</strong>
	 * <ul>
	 * <li>Makes REST GET request: {@code GET /contexts/{uuid}} with query parameters</li>
	 * <li>Unmarshals the JSON response to {@link Context} object using {@link ElementMapper}</li>
	 * <li>Forces cache refresh via {@code forceCacheRefresh()} after successful fetch</li>
	 * <li>Attempts to get the refreshed context from cache for consistency</li>
	 * </ul>
	 * 
	 * <strong>Cache Synchronization:</strong>
	 * <ul>
	 * <li>After server fetch, tries to get the context from refreshed cache</li>
	 * <li>If cache refresh succeeds, uses the cached version for consistency</li>
	 * <li>If cache refresh fails, logs error but continues with server-fetched context</li>
	 * <li>This ensures cache consistency and handles potential cache refresh issues gracefully</li>
	 * </ul>
	 * 
	 * <strong>REST API Implementation:</strong>
	 * <ul>
	 * <li>Uses {@code readFromServer()} which calls: {@code GET /contexts/{uuid}}</li>
	 * <li>Includes additional query parameters via {@code includeAdditionalQueryParameters()}</li>
	 * <li>Authorization headers automatically added via {@code getGXHTTPStringRequest()}</li>
	 * </ul>
	 * 
	 * <strong>Error Handling:</strong>
	 * <ul>
	 * <li>{@link IOException} during JSON unmarshaling wrapped in {@link RuntimeException}</li>
	 * <li>Server-side errors propagated as {@link ResourceRegistryException}</li>
	 * <li>Cache refresh failures logged but don't affect operation success</li>
	 * </ul>
	 */
	@Override
	public Context read(UUID uuid) throws ContextNotFoundException, ResourceRegistryException {
		Context context = contextCache.getContextByUUID(uuid);;
		if(context == null) {
			String contextJson = readFromServer(uuid.toString());
			try {
				context = ElementMapper.unmarshal(Context.class, contextJson);
			} catch (IOException e) {
				throw new RuntimeException(e);
			}
			forceCacheRefresh();
			Context c = contextCache.getContextByUUID(context.getID());
			if(c!=null){
				context = c;
			}else {
				logger.error("Context with UUID {} is {}. It is possibile to get it from the server but not from the cache. This is very strange and should not occur.", uuid, context);
			}
		}
		return context;
	}
	
	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>Current Context Resolution:</strong>
	 * <ul>
	 * <li>Gets current context full name from {@link SecretManagerProvider} security context</li>
	 * <li>Resolves the full name to UUID using {@code contextCache.getUUIDByFullName()}</li>
	 * <li>Delegates to {@code read(UUID)} for the actual context retrieval</li>
	 * </ul>
	 * 
	 * <strong>Security Context Integration:</strong>
	 * <ul>
	 * <li>Automatically extracts context from JWT token in the current thread</li>
	 * <li>No need to manually specify context UUID or name</li>
	 * <li>Works with the current authorization scope</li>
	 * </ul>
	 * 
	 * <strong>Cache Dependency:</strong>
	 * <ul>
	 * <li>Requires the context cache to have full name to UUID mapping</li>
	 * <li>If mapping is not available, may trigger cache refresh</li>
	 * <li>Uses the same cache-first strategy as {@code read(UUID)}</li>
	 * </ul>
	 */
	@Override
	public Context readCurrentContext() throws ContextNotFoundException, ResourceRegistryException {
		String contextFullName = SecretManagerProvider.get().getContext();
		UUID uuid = contextCache.getUUIDByFullName(contextFullName);
		return read(uuid);
	}
	
	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>UUID Conversion and Delegation:</strong>
	 * <ul>
	 * <li>Converts string UUID to {@link UUID} object via {@code UUID.fromString()}</li>
	 * <li>Delegates to {@code read(UUID)} for the actual retrieval</li>
	 * <li>Marshals the resulting {@link Context} object to JSON using {@link ElementMapper}</li>
	 * </ul>
	 * 
	 * <strong>JSON Marshaling:</strong>
	 * <ul>
	 * <li>Uses {@link ElementMapper} for consistent JSON representation</li>
	 * <li>Same format as returned by server REST API</li>
	 * <li>Marshaling errors wrapped in {@link RuntimeException}</li>
	 * </ul>
	 * 
	 * <strong>Error Handling:</strong>
	 * <ul>
	 * <li>{@link IllegalArgumentException} for invalid UUID format (from {@code UUID.fromString()})</li>
	 * <li>{@link ContextNotFoundException} propagated from underlying read operation</li>
	 * <li>{@link JsonProcessingException} during marshaling wrapped in {@link RuntimeException}</li>
	 * <li>Other {@link ResourceRegistryException} types propagated unchanged</li>
	 * </ul>
	 */
	@Override
	public String read(String uuid) throws ContextNotFoundException, ResourceRegistryException {
		try {
			return ElementMapper.marshal(read(UUID.fromString(uuid)));
		} catch (ContextNotFoundException e) {
			throw e;
		} catch (JsonProcessingException e) {
			throw new RuntimeException(e);
		} catch (ResourceRegistryException e) {
			throw e;
		}
	}
	
	/**
	 * Reads a context from the server by its UUID.
	 * 
	 * @param uuid The UUID of the context to read
	 * @return The JSON representation of the context
	 * @throws ContextNotFoundException If the context with the specified UUID is not found
	 * @throws ResourceRegistryException If an error occurs during the operation
	 */
	public String readFromServer(String uuid) throws ContextNotFoundException, ResourceRegistryException {
		try {
			logger.trace("Going to read {} with UUID {}", Context.NAME, uuid);
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(ContextPath.CONTEXTS_PATH_PART);
			gxHTTPStringRequest.path(uuid);
			
			includeAdditionalQueryParameters(gxHTTPStringRequest);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.get();
			String c = HTTPUtility.getResponse(String.class, httpURLConnection);
			
			logger.debug("Got {} is {}", Context.NAME, c);
			return c;
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * Internal method to update an existing context in the Resource Registry.
	 * 
	 * @param context The context to update
	 * @return The JSON representation of the updated context
	 * @throws ContextNotFoundException If the context with the specified UUID is not found
	 * @throws ResourceRegistryException If an error occurs during the update
	 */
	public String internalUpdate(Context context) throws ContextNotFoundException, ResourceRegistryException {
		try {
			String contextString = ElementMapper.marshal(context);
			logger.trace("Going to update {}", contextString);
			
			UUID uuid = context.getID();
			
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.header(CONTENT_TYPE_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(ContextPath.CONTEXTS_PATH_PART);
			gxHTTPStringRequest.path(uuid.toString());
			
			includeAdditionalQueryParameters(gxHTTPStringRequest);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.put(contextString);
			String c = HTTPUtility.getResponse(String.class, httpURLConnection);
			
			forceCacheRefresh();
			
			logger.trace("{} successfully updated", c);
			return c;
			
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>Context Validation:</strong>
	 * <ul>
	 * <li>Requires a valid UUID in the context object (cannot be null)</li>
	 * <li>Marshals the context object to JSON using {@link ElementMapper}</li>
	 * <li>Delegates to {@code internalUpdate(Context)} for the actual update operation</li>
	 * </ul>
	 * 
	 * <strong>REST API Implementation:</strong>
	 * <ul>
	 * <li>Makes HTTP PUT request to: {@code PUT /contexts/{uuid}}</li>
	 * <li>Content-Type: application/json;charset=UTF-8</li>
	 * <li>Includes additional query parameters via {@code includeAdditionalQueryParameters()}</li>
	 * <li>Uses the context's UUID for the URL path</li>
	 * </ul>
	 * 
	 * <strong>Authorization:</strong>
	 * <ul>
	 * <li>Uses current security context from {@link SecretManagerProvider}</li>
	 * <li>Authorization headers automatically added via {@code getGXHTTPStringRequest()}</li>
	 * </ul>
	 * 
	 * <strong>Cache Management:</strong>
	 * <ul>
	 * <li>Forces cache refresh via {@code forceCacheRefresh()} after successful update</li>
	 * <li>Ensures subsequent operations see the updated context</li>
	 * <li>Cache refresh failures are logged but don't prevent operation completion</li>
	 * </ul>
	 * 
	 * <strong>Response Processing:</strong>
	 * <ul>
	 * <li>Unmarshals the JSON response back to a {@link Context} object</li>
	 * <li>Returns the updated context with any server-side modifications</li>
	 * </ul>
	 * 
	 * <strong>Error Handling:</strong>
	 * <ul>
	 * <li>Propagates {@link ResourceRegistryException} from server</li>
	 * <li>Wraps JSON processing and other exceptions in {@link RuntimeException}</li>
	 * </ul>
	 */
	@Override
	public Context update(Context context) throws ContextNotFoundException, ResourceRegistryException {
		try {
			String res = internalUpdate(context);
			return ElementMapper.unmarshal(Context.class, res);
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>JSON Processing:</strong>
	 * <ul>
	 * <li>Unmarshals the JSON string to a {@link Context} object using {@link ElementMapper}</li>
	 * <li>Delegates to {@code internalUpdate(Context)} for the actual update logic</li>
	 * <li>Returns the raw JSON response from the server</li>
	 * </ul>
	 * 
	 * <strong>UUID Extraction:</strong>
	 * <ul>
	 * <li>Extracts UUID from the JSON representation</li>
	 * <li>UUID must be present in the JSON (cannot update without specifying which context)</li>
	 * <li>JSON validation occurs during unmarshaling phase</li>
	 * </ul>
	 * 
	 * <strong>REST API Implementation:</strong>
	 * <ul>
	 * <li>Same REST implementation as {@code update(Context)}: {@code PUT /contexts/{uuid}}</li>
	 * <li>UUID extracted from JSON is used for the URL path</li>
	 * </ul>
	 * 
	 * <strong>Cache Management:</strong>
	 * <ul>
	 * <li>Forces cache refresh after successful update (same as {@code update(Context)})</li>
	 * </ul>
	 * 
	 * <strong>Error Handling:</strong>
	 * <ul>
	 * <li>Validates JSON format during unmarshaling</li>
	 * <li>Propagates {@link ResourceRegistryException} from server</li>
	 * <li>Wraps JSON processing and other exceptions in {@link RuntimeException}</li>
	 * <li>Missing or invalid UUID in JSON will cause unmarshaling to fail</li>
	 * </ul>
	 */
	@Override
	public String update(String context) throws ContextNotFoundException, ResourceRegistryException {
		try {
			Context c = ElementMapper.unmarshal(Context.class, context);
			return internalUpdate(c);
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
//	protected String internalChangeState(UUID uuid, String context)  throws ContextNotFoundException, ResourceRegistryException {
//		try {
//			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
//			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
//			gxHTTPStringRequest.header(CONTENT_TYPE_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
//			gxHTTPStringRequest.path(ContextPath.CONTEXTS_PATH_PART);
//			gxHTTPStringRequest.path(uuid.toString());
//			
//			includeAdditionalQueryParameters(gxHTTPStringRequest);
//			
//			HttpURLConnection httpURLConnection = gxHTTPStringRequest.post(context);
//			String c = HTTPUtility.getResponse(String.class, httpURLConnection);
//			
//			forceCacheRefresh();
//			
//			logger.trace("{} successfully updated", c);
//			return c;
//		} catch(ResourceRegistryException e) {
//			throw e;
//		} catch(Exception e) {
//			throw new RuntimeException(e);
//		}
//	}
//	
//	@Override
//	public Context changeState(Context context) throws ContextNotFoundException, ResourceRegistryException {
//		try {
//			String contextString = ElementMapper.marshal(context);
//			UUID uuid = context.getID();
//			logger.trace("Going to change the state of {} with name {} id {} to {}", 
//					Context.NAME, context.getName(), uuid, context.getState());
//			
//			contextString = internalChangeState(uuid, contextString);
//			
//			return ElementMapper.unmarshal(Context.class, contextString);
//		} catch(ResourceRegistryException e) {
//			throw e;
//		} catch(Exception e) {
//			throw new RuntimeException(e);
//		}
//	}
//
//	@Override
//	public String changeState(String context) throws ContextNotFoundException, ResourceRegistryException {
//		try {
//			Context c = ElementMapper.unmarshal(Context.class, context);
//			UUID uuid = c.getID();
//			logger.trace("Going to change the state of {} with name {} id {} to {}", 
//					Context.NAME, c.getName(), uuid, c.getState());
//			
//			return internalChangeState(uuid, context);
//
//		} catch(ResourceRegistryException e) {
//			throw e;
//		} catch(Exception e) {
//			throw new RuntimeException(e);
//		}
//	}
	
	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>UUID Extraction:</strong>
	 * <ul>
	 * <li>Extracts UUID from the context object via {@code context.getID()}</li>
	 * <li>Delegates to {@code delete(UUID)} for the actual deletion operation</li>
	 * </ul>
	 * 
	 * <strong>Convenience Method:</strong>
	 * <ul>
	 * <li>Provides a convenient way to delete a context when you have the context object</li>
	 * <li>Useful when deleting contexts obtained from other operations like {@code read()} or {@code all()}</li>
	 * </ul>
	 */
	@Override
	public boolean delete(Context context) throws ContextNotFoundException, ResourceRegistryException {
		return delete(context.getID());
	}
	
	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>UUID Conversion:</strong>
	 * <ul>
	 * <li>Converts the {@link UUID} object to string via {@code uuid.toString()}</li>
	 * <li>Delegates to {@code delete(String)} for the actual deletion operation</li>
	 * </ul>
	 * 
	 * <strong>Type Safety:</strong>
	 * <ul>
	 * <li>Provides type-safe UUID handling</li>
	 * <li>Eliminates the possibility of invalid UUID format errors</li>
	 * </ul>
	 */
	@Override
	public boolean delete(UUID uuid) throws ContextNotFoundException, ResourceRegistryException {
		return delete(uuid.toString());
	}
	
	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>REST API Implementation:</strong>
	 * <ul>
	 * <li>Makes HTTP DELETE request to: {@code DELETE /contexts/{uuid}}</li>
	 * <li>Accept header set to: application/json;charset=UTF-8</li>
	 * <li>No request body is sent (DELETE operation)</li>
	 * <li>Uses the UUID string directly in the URL path</li>
	 * </ul>
	 * 
	 * <strong>Authorization:</strong>
	 * <ul>
	 * <li>Uses current security context from {@link SecretManagerProvider}</li>
	 * <li>Authorization headers automatically added via {@code getGXHTTPStringRequest()}</li>
	 * </ul>
	 * 
	 * <strong>Response Processing:</strong>
	 * <ul>
	 * <li>Server response is processed via {@code HTTPUtility.getResponse()}</li>
	 * <li>Currently always returns {@code true} if no exception is thrown</li>
	 * <li>Server-side validation ensures deletion constraints are respected</li>
	 * </ul>
	 * 
	 * <strong>Cache Management:</strong>
	 * <ul>
	 * <li>Forces cache refresh via {@code forceCacheRefresh()} after deletion attempt (in finally block)</li>
	 * <li>Cache refresh is attempted even if deletion fails to ensure consistency</li>
	 * <li>Cache refresh failures are silently ignored to not interfere with deletion result</li>
	 * </ul>
	 * 
	 * <strong>Error Handling:</strong>
	 * <ul>
	 * <li>{@link ResourceRegistryException} propagated from server (e.g., authorization errors, child context constraints)</li>
	 * <li>Other exceptions wrapped in {@link RuntimeException}</li>
	 * <li>Cache refresh errors are caught and ignored</li>
	 * </ul>
	 * 
	 * <strong>Logging:</strong>
	 * <ul>
	 * <li>Trace-level logging for deletion initiation</li>
	 * <li>Info-level logging for deletion completion with success/failure status</li>
	 * </ul>
	 */
	@Override
	public boolean delete(String uuid) throws ContextNotFoundException, ResourceRegistryException {
		try {
			logger.trace("Going to delete {} with UUID {}", Context.NAME, uuid);
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(ContextPath.CONTEXTS_PATH_PART);
			gxHTTPStringRequest.path(uuid);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.delete();
			HTTPUtility.getResponse(String.class, httpURLConnection);
			
			boolean deleted = true;
			
			logger.info("{} with UUID {} {}", Context.NAME, uuid,
					deleted ? " successfully deleted" : "was NOT deleted");
			return deleted;
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new RuntimeException(e);
		}finally {
			try {
				forceCacheRefresh();
			}catch (Exception e) {
				
			}
		}
	}

	/**
	 * Internal method to change the state of an existing context in the Resource Registry.
	 * 
	 * @param uuid the UUID of the context whose state will be changed
	 * @param context the new context state as a JSON string
	 * @return the updated context as a JSON string
	 * @throws ContextNotFoundException if the specified context UUID does not exist
	 * @throws ResourceRegistryException if an error occurs during the state change operation
	 */
	protected String internalChangeState(UUID uuid, String context) throws ContextNotFoundException, ResourceRegistryException {
		try {
			logger.trace("Going to change state of {} with UUID {}", Context.NAME, uuid);
			GXHTTPStringRequest gxHTTPStringRequest = getGXHTTPStringRequest();
			gxHTTPStringRequest.header(ACCEPT_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.header(CONTENT_TYPE_HTTP_HEADER_KEY, GXConnection.APPLICATION_JSON_CHARSET_UTF_8);
			gxHTTPStringRequest.path(ContextPath.CONTEXTS_PATH_PART);
			gxHTTPStringRequest.path(uuid.toString());
			
			includeAdditionalQueryParameters(gxHTTPStringRequest);
			
			HttpURLConnection httpURLConnection = gxHTTPStringRequest.post(context);
			String c = HTTPUtility.getResponse(String.class, httpURLConnection);
			
			logger.trace("{} state successfully changed", Context.NAME);
			return c;
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new RuntimeException(e);
		} finally {
			try {
				forceCacheRefresh();
			} catch (Exception e) {
				// Cache refresh failures should not interfere with state change result
			}
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>UUID Extraction:</strong>
	 * <ul>
	 * <li>Extracts UUID from the context object via {@code context.getID()}</li>
	 * <li>Delegates to internal change state operation for the actual REST API call</li>
	 * </ul>
	 * 
	 * <strong>JSON Marshalling:</strong>
	 * <ul>
	 * <li>Marshals the context object to JSON using {@code ElementMapper.marshal()}</li>
	 * <li>Only the state property from the JSON is processed by the server</li>
	 * <li>Response JSON is unmarshalled back to a Context object</li>
	 * </ul>
	 * 
	 * <strong>Cache Management:</strong>
	 * <ul>
	 * <li>Forces cache refresh via {@code forceCacheRefresh()} after state change (in finally block)</li>
	 * <li>Cache refresh is attempted even if state change fails to ensure consistency</li>
	 * <li>Cache refresh failures are silently ignored to not interfere with state change result</li>
	 * </ul>
	 * 
	 * <strong>Convenience Method:</strong>
	 * <ul>
	 * <li>Provides object-oriented approach to state changes</li>
	 * <li>Useful when working with Context objects rather than raw JSON</li>
	 * </ul>
	 */
	@Override
	public Context changeState(Context context) throws ContextNotFoundException, ResourceRegistryException {
		try {
			String contextString = ElementMapper.marshal(context);
			UUID uuid = context.getID();
			logger.trace("Going to change the state of {} with name {} id {} to {}", 
					Context.NAME, context.getName(), uuid, context.getState());
			
			contextString = internalChangeState(uuid, contextString);
			
			return ElementMapper.unmarshal(Context.class, contextString);
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>REST API Implementation:</strong>
	 * <ul>
	 * <li>Makes HTTP PATCH request to: {@code PATCH /contexts/{uuid}}</li>
	 * <li>Content-Type header set to: application/json;charset=UTF-8</li>
	 * <li>Accept header set to: application/json;charset=UTF-8</li>
	 * <li>JSON body contains the context with new state (other properties ignored)</li>
	 * </ul>
	 * 
	 * <strong>UUID Extraction:</strong>
	 * <ul>
	 * <li>Parses the input JSON to extract the context UUID</li>
	 * <li>Uses the UUID for the REST API path parameter</li>
	 * <li>Validates that the UUID is present and valid</li>
	 * </ul>
	 * 
	 * <strong>Authorization:</strong>
	 * <ul>
	 * <li>Uses current security context from {@link SecretManagerProvider}</li>
	 * <li>Authorization headers automatically added via {@code getGXHTTPStringRequest()}</li>
	 * </ul>
	 * 
	 * <strong>Response Processing:</strong>
	 * <ul>
	 * <li>Server response processed via {@code HTTPUtility.getResponse()}</li>
	 * <li>Returns complete updated context as JSON string</li>
	 * <li>Response includes all context properties with the updated state</li>
	 * </ul>
	 * 
	 * <strong>Cache Management:</strong>
	 * <ul>
	 * <li>Forces cache refresh via {@code forceCacheRefresh()} after state change (in finally block)</li>
	 * <li>Cache refresh is attempted even if state change fails to ensure consistency</li>
	 * <li>Cache refresh failures are silently ignored to not interfere with state change result</li>
	 * </ul>
	 * 
	 * <strong>Error Handling:</strong>
	 * <ul>
	 * <li>{@link ResourceRegistryException} propagated from server (e.g., authorization errors, invalid state)</li>
	 * <li>JSON parsing exceptions wrapped in {@link RuntimeException}</li>
	 * <li>Cache refresh errors are caught and ignored</li>
	 * </ul>
	 * 
	 * <strong>Logging:</strong>
	 * <ul>
	 * <li>Trace-level logging for state change initiation with context details</li>
	 * <li>Trace-level logging for successful state change completion</li>
	 * </ul>
	 */
	@Override
	public String changeState(String context) throws ContextNotFoundException, ResourceRegistryException {
		try {
			Context c = ElementMapper.unmarshal(Context.class, context);
			UUID uuid = c.getID();
			logger.trace("Going to change the state of {} with name {} id {} to {}", 
					Context.NAME, c.getName(), uuid, c.getState());
			
			return internalChangeState(uuid, context);
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>Minimal Context Construction:</strong>
	 * <ul>
	 * <li>Creates a new Context instance with only UUID and state properties set</li>
	 * <li>Uses {@code ContextImpl} with temporary name for minimal object creation</li>
	 * <li>All other context properties remain null/default values</li>
	 * </ul>
	 * 
	 * <strong>Type Safety:</strong>
	 * <ul>
	 * <li>Uses {@code ContextState} enum for compile-time validation</li>
	 * <li>Automatically converts enum to its string representation via {@code getState()}</li>
	 * <li>Eliminates runtime errors from invalid state strings</li>
	 * </ul>
	 * 
	 * <strong>Delegation:</strong>
	 * <ul>
	 * <li>Delegates to existing {@code changeState(Context)} method for consistency</li>
	 * <li>Inherits all caching, error handling, and REST API interaction behavior</li>
	 * </ul>
	 * 
	 * <strong>Cache Management:</strong>
	 * <ul>
	 * <li>Cache refresh is handled by the delegated {@code changeState(Context)} method</li>
	 * <li>Ensures consistency with other changeState method implementations</li>
	 * </ul>
	 */
	@Override
	public Context changeState(UUID uuid, ContextState state) throws ContextNotFoundException, ResourceRegistryException {
		try {
			// Create minimal context with only UUID and state
			Context context = new org.gcube.informationsystem.contexts.impl.entities.ContextImpl("TempContext");
			context.setID(uuid);
			context.setState(state.getState()); // Convert enum to string
			
			logger.trace("Going to change the state of {} with UUID {} to {}", 
					Context.NAME, uuid, state);
			
			// Delegate to existing changeState(Context) method
			return changeState(context);
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * <strong>Implementation Details:</strong>
	 * 
	 * <strong>String to Enum Conversion:</strong>
	 * <ul>
	 * <li>Converts the state string to {@code ContextState} enum using {@code ContextState.fromString()}</li>
	 * <li>Handles case-insensitive matching via the enum's fromString method</li>
	 * <li>Provides detailed error messages for invalid state strings with valid options listed</li>
	 * </ul>
	 * 
	 * <strong>UUID Validation:</strong>
	 * <ul>
	 * <li>Converts the UUID string to {@code UUID} object using {@code UUID.fromString()}</li>
	 * <li>Validates UUID format and throws {@code IllegalArgumentException} for invalid formats</li>
	 * <li>Provides clear error messages for malformed UUIDs</li>
	 * </ul>
	 * 
	 * <strong>Delegation:</strong>
	 * <ul>
	 * <li>Delegates to {@code changeState(UUID, ContextState)} for type-safe processing</li>
	 * <li>Marshals the resulting Context object back to JSON string using {@code ElementMapper.marshal()}</li>
	 * <li>Returns complete updated context as JSON with new state and all other properties</li>
	 * </ul>
	 * 
	 * <strong>Cache Management:</strong>
	 * <ul>
	 * <li>Cache refresh is handled by the delegated method chain</li>
	 * <li>Ensures consistency with other changeState method implementations</li>
	 * </ul>
	 * 
	 * <strong>Error Handling:</strong>
	 * <ul>
	 * <li>{@code IllegalArgumentException} for invalid UUID format or state string</li>
	 * <li>{@code ResourceRegistryException} propagated from underlying operations</li>
	 * <li>Other exceptions wrapped in {@code RuntimeException}</li>
	 * </ul>
	 */
	@Override
	public String changeState(String uuid, String state) throws ContextNotFoundException, ResourceRegistryException {
		try {
			// Validate and convert string parameters
			UUID contextUuid;
			try {
				contextUuid = UUID.fromString(uuid);
			} catch (IllegalArgumentException e) {
				throw new IllegalArgumentException("Invalid UUID format: " + uuid, e);
			}
			
			ContextState contextState;
			try {
				contextState = ContextState.fromString(state);
				if (contextState == null) {
					throw new IllegalArgumentException("Invalid state value");
				}
			} catch (Exception e) {
				throw new IllegalArgumentException("Invalid state value: " + state + ". Valid states are: " + 
					Arrays.toString(ContextState.values()), e);
			}
			
			logger.trace("Going to change the state of {} with UUID {} to {}", 
					Context.NAME, uuid, state);
			
			// Delegate to UUID/ContextState variant and marshal result to JSON
			Context updatedContext = changeState(contextUuid, contextState);
			return ElementMapper.marshal(updatedContext);
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
}
