package org.gcube.informationsystem.resourceregistry.publisher;

import java.util.List;
import java.util.Map;
import java.util.UUID;

import org.gcube.informationsystem.contexts.reference.entities.Context;
import org.gcube.informationsystem.model.reference.ERElement;
import org.gcube.informationsystem.model.reference.entities.Facet;
import org.gcube.informationsystem.model.reference.entities.Resource;
import org.gcube.informationsystem.model.reference.relations.ConsistsOf;
import org.gcube.informationsystem.model.reference.relations.IsRelatedTo;
import org.gcube.informationsystem.resourceregistry.api.contexts.ContextCache;
import org.gcube.informationsystem.resourceregistry.api.exceptions.AlreadyPresentException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.AvailableInAnotherContextException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.NotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.contexts.ContextNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entities.facet.FacetAlreadyPresentException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entities.facet.FacetAvailableInAnotherContextException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entities.facet.FacetNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entities.resource.ResourceAlreadyPresentException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entities.resource.ResourceAvailableInAnotherContextException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entities.resource.ResourceNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.relations.consistsof.ConsistsOfAlreadyPresentException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.relations.consistsof.ConsistsOfAvailableInAnotherContextException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.relations.consistsof.ConsistsOfNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.relations.isrelatedto.IsRelatedToAlreadyPresentException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.relations.isrelatedto.IsRelatedToAvailableInAnotherContextException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.relations.isrelatedto.IsRelatedToNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.types.SchemaViolationException;
import org.gcube.informationsystem.resourceregistry.api.request.RequestInfo;

/**
 * Client interface for publishing and managing entities, relations, and resources in the Resource Registry.
 * 
 * <strong>Purpose:</strong>
 * This interface provides methods for creating, reading, updating, and deleting Information System model instances
 * (Resources, Facets, ConsistsOf relations, IsRelatedTo relations) in the Resource Registry service.
 * 
 * <strong>Instantiation:</strong>
 * Clients should be created using the factory pattern for proper configuration:
 * <pre>
 * // Recommended way to create a client
 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
 * </pre>
 * 
 * <strong>Supported Operations:</strong>
 * <ul>
 * <li><strong>List Operations</strong>: Retrieve instances of specific types with filtering and pagination support</li>
 * <li><strong>CRUD Operations</strong>: Create, Read, Update, Delete for all Information System model types</li>
 * <li><strong>Context Management</strong>: Add/remove instances to/from specific contexts</li>
 * <li><strong>Specialized Operations</strong>: Type-specific methods for Resources, Facets, and Relations.</li>
 * </ul>
 *
 * <strong>Query Parameter Configuration:</strong>
 * Most methods support optional query parameters that can be configured via the client configuration:
 *  * - <strong>limit</strong>: Maximum number of results per request;
 * <ul>
 * <li><strong>offset</strong>: Number of results to skip for pagination</li>
 * <li><strong>includeMeta</strong>: Include metadata in responses (role-based filtering applied)</li>
 * <li><strong>allMeta</strong>: Include metadata for all nested instances</li>
 * <li><strong>includeContexts</strong>: Show context availability information</li>
 * <li><strong>hierarchical</strong>: Include instances from child contexts (admin-only).</li>
 * </ul>
 *
 * <strong>Authorization:</strong>
 * All operations respect the authorization context derived from the current authentication token.
 * Users can only access and modify instances within their authorized contexts.
 * 
 * <strong>Context Support:</strong>
 * The client automatically operates within the context determined by the authorization token.
 * Additional methods support cross-context operations for authorized users.
 * 
 * @author Luca Frosini (ISTI - CNR)
 */
public interface ResourceRegistryPublisher extends RequestInfo {

	/**
	 * Adds a custom HTTP header to be included in requests.
	 * 
	 * @param name Header name
	 * @param value Header value
	 */
	public void addHeader(String name, String value);
	
	/**
	 * Adds a custom HTTP header to be included in requests.
	 * 
	 * @param name Header name
	 * @param value boolean value
	 */
	public void addHeader(String name, boolean value);

	/**
	 * Gets the context cache used for context-related operations.
	 * 
	 * <strong>Cache Behavior:</strong>
	 * <ul>
	 * <li>Provides access to the internal context cache used by all context methods</li>
	 * <li>Cache improves performance by avoiding repeated server requests</li>
	 * <li>Automatically manages cache updates and invalidation</li>
	 * <li>Used internally by {@code getAllContexts()}, {@code getContext()}, and {@code getCurrentContext()} methods.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>Available to all authenticated users</li>
	 * <li>Cache content is filtered based on user authorization level.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Access context cache for advanced operations
	 * ContextCache cache = publisher.getContextCache();
	 * 
	 * // Force cache refresh if needed
	 * cache.refresh();
	 * </pre>
	 * 
	 * @return The context cache instance
	 */
	public ContextCache getContextCache();
	
	/**
	 * Retrieves all available contexts using internal cache.
	 * 
	 * <strong>Cache Behavior:</strong>
	 * <ul>
	 * <li><strong>Uses internal context cache</strong> - does NOT bypass cache like server-direct methods</li>
	 * <li>First call may trigger server request to populate cache</li>
	 * <li>Subsequent calls return cached data for improved performance</li>
	 * <li>Cache is automatically maintained and refreshed as needed.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters NOT Applicable:</strong>
	 * <ul>
	 * <li>Standard query parameters like {@code includeMeta}, {@code limit}, {@code offset}, {@code hierarchical} are NOT applicable to cached methods</li>
	 * <li>Cache behavior is controlled internally for optimal performance</li>
	 * <li>For direct server access with full query control, use the implementation's FromServer methods.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>Available to all authenticated users</li>
	 * <li>Returns contexts accessible based on user authorization level</li>
	 * <li>Context list is filtered according to user permissions.</li>
	 * </ul>
	 * 
	 * <strong>Context Content:</strong>
	 * <ul>
	 * <li>Includes all contexts where the user has any level of access</li>
	 * <li>Context details include basic information (name, UUID, parent relationships)</li>
	 * <li>Metadata inclusion depends on cache configuration and user role.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Get all contexts from cache
	 * List&lt;Context&gt; contexts = publisher.getAllContexts();
	 * 
	 * // Iterate through available contexts
	 * for (Context context : contexts) {
	 *     System.out.println("Context: " + context.getName() + " (" + context.getUuid() + ")");
	 * }
	 * </pre>
	 * 
	 * @return List of all contexts accessible to the current user (from cache)
	 * @throws ResourceRegistryException If fails to retrieve contexts from cache or server
	 */
	public List<Context> getAllContexts() throws ResourceRegistryException;
	
	/**
	 * Retrieves a specific context by its UUID using internal cache.
	 * 
	 * <strong>Cache Behavior:</strong>
	 * <ul>
	 * <li><strong>Uses internal context cache</strong> - does NOT bypass cache like server-direct methods</li>
	 * <li>First call may trigger cache population from server if needed</li>
	 * <li>Subsequent calls use cached data for improved performance</li>
	 * <li>Cache is automatically maintained and refreshed as needed.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters NOT Applicable:</strong>
	 * <ul>
	 * <li>Standard query parameters like {@code includeMeta}, {@code hierarchical} are NOT applicable to cached methods</li>
	 * <li>Metadata inclusion depends on cache configuration and user role</li>
	 * <li>For direct server access with full query control, use the implementation's FromServer methods.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>Available to all authenticated users</li>
	 * <li>Returns context only if accessible based on user authorization level</li>
	 * <li>Context details are filtered according to user permissions and role.</li>
	 * </ul>
	 * 
	 * <strong>Context Content:</strong>
	 * <ul>
	 * <li>Includes basic context information (name, UUID, parent relationships)</li>
	 * <li>Metadata inclusion depends on user role: IS-Manager/Infrastructure-Manager see complete metadata, others see filtered metadata</li>
	 * <li>Sensitive information (createdBy, lastUpdatedBy) may be obfuscated for non-administrative users.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Get context from cache
	 * UUID contextUUID = UUID.fromString("c0f314e7-2807-4241-a792-2a6c79ed4fd0");
	 * Context context = publisher.getContext(contextUUID);
	 * 
	 * System.out.println("Context name: " + context.getName());
	 * System.out.println("Context UUID: " + context.getUuid());
	 * System.out.println("Parent context: " + context.getParent());
	 * </pre>
	 * 
	 * @param uuid The UUID of the context to retrieve
	 * @return The context with the specified UUID (from cache)
	 * @throws ContextNotFoundException If the context is not found or not accessible
	 * @throws ResourceRegistryException If an error occurs during context retrieval
	 */
	public Context getContext(UUID uuid) throws ContextNotFoundException, ResourceRegistryException;

	/**
	 * Retrieves the current context associated with the publisher's authorization token using internal cache.
	 * 
	 * <strong>Cache Behavior:</strong>
	 * <ul>
	 * <li><strong>Uses internal context cache</strong> - does NOT bypass cache like server-direct methods</li>
	 * <li>Current context is determined from the authorization token/secret</li>
	 * <li>First call may trigger cache population from server if needed</li>
	 * <li>Subsequent calls use cached data for improved performance.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters NOT Applicable:</strong>
	 * <ul>
	 * <li>Standard query parameters like {@code includeMeta}, {@code hierarchical} are NOT applicable to cached methods</li>
	 * <li>Metadata inclusion depends on cache configuration and user role</li>
	 * <li>For direct server access with full query control, use the implementation's FromServer methods.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>Available to all authenticated users</li>
	 * <li>Returns the context associated with the current authorization token</li>
	 * <li>Context details are filtered according to user permissions and role.</li>
	 * </ul>
	 * 
	 * <strong>Context Resolution:</strong>
	 * <ul>
	 * <li>Current context is extracted from the authorization token/secret</li>
	 * <li>Token must contain valid context information</li>
	 * <li>Context must be accessible to the user associated with the token.</li>
	 * </ul>
	 * 
	 * <strong>Context Content:</strong>
	 * <ul>
	 * <li>Includes basic context information (name, UUID, parent relationships)</li>
	 * <li>Metadata inclusion depends on user role: IS-Manager/Infrastructure-Manager see complete metadata, others see filtered metadata</li>
	 * <li>Sensitive information (createdBy, lastUpdatedBy) may be obfuscated for non-administrative users.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Get current context from cache
	 * Context currentContext = publisher.getCurrentContext();
	 * 
	 * System.out.println("Current context name: " + currentContext.getName());
	 * System.out.println("Current context UUID: " + currentContext.getUuid());
	 * System.out.println("Current context parent: " + currentContext.getParent());
	 * </pre>
	 * 
	 * @return The current context associated with the authorization token (from cache)
	 * @throws ContextNotFoundException If the current context cannot be determined or accessed
	 * @throws ResourceRegistryException If an error occurs during context retrieval
	 */
	public Context getCurrentContext() throws ContextNotFoundException, ResourceRegistryException;
		
	/**
	 * Retrieves all instances of the specified type from the Resource Registry.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /instances/{type-name}[?polymorphic={true|false}&limit={number}&offset={number}&includeMeta={true|false}&allMeta={true|false}&includeContexts={true|false}&hierarchical={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Returns all instances of the specified class that are accessible in the current context</li>
	 * <li>Supports polymorphic queries to include instances of subtypes</li>
	 * <li>Results are filtered based on user authorization and context access.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>polymorphic</strong> (automatically set from method parameter):
	 * <ul>
	 * <li>Whether to include instances of subtypes in the response</li>
	 * <li>Default: true (includes all subtypes)</li>
	 * <li>Values: true (includes all subtypes) | false (exact type only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#POLYMORPHIC_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>limit</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Maximum number of instances to return in a single response</li>
	 * <li>Default value: server configuration dependent</li>
	 * <li>Usage: {@code client.setLimit(50)}</li>
	 * <li>Example: Set to 50 to get at most 50 instances per request</li>
	 * <li>Unlimited results: Set to {@code RequestInfo.UNBOUNDED_LIMIT} (-1) for unlimited results (may cause timeout if results are too many)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#LIMIT_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>offset</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Number of instances to skip from the beginning of the result set</li>
	 * <li>Default value: 0</li>
	 * <li>Usage: {@code client.setOffset(10)}</li>
	 * <li>Example: Set to 10 to skip the first 10 contexts (useful for pagination)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#OFFSET_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response instance</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration, admin-only):
	 * <ul>
	 * <li>Whether to include instances from child contexts of the current context</li>
	 * <li>Default value: false (child contexts not included)</li>
	 * <li>Usage: {@code publisher.setHierarchicalMode(true)}</li>
	 * <li>Values: true (includes child contexts) | false (current context only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#HIERARCHICAL_MODE_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Instances successfully retrieved</li>
	 * <li><strong>400 Bad Request</strong>: Invalid type name or query parameters</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to list instances or access hierarchical data</li>
	 * <li><strong>404 Not Found</strong>: Specified type does not exist in the model</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Type name is automatically derived from the provided class using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>Supports all Information System model types and their subtypes: Entities (Resources and all Resource subtypes, Facets and all Facet subtypes) and Relations (ConsistsOf and all ConsistsOf subtypes, IsRelatedTo and all IsRelatedTo subtypes).</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The instance must be accessible in the current context or child contexts (depending on hierarchical mode)</li>
	 * <li>IS-Manager and Infrastructure-Manager are global roles (available in all contexts) and can use hierarchical mode in any contexts</li>
	 * <li>Context-Manager is a context-specific role and can use hierarchical mode only in contexts where they have this role.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to list any instances with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to list any instances with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>Context-Manager</strong>: Full access to list instances within managed contexts with hierarchical querying; receives complete, unfiltered metadata</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Get all EService instances including subtypes
	 * List&lt;EService&gt; services = publisher.list(EService.class, true);
	 * 
	 * // Get only exact ContactFacet instances (no subtypes)
	 * List&lt;ContactFacet&gt; contacts = publisher.list(ContactFacet.class, false);
	 * </pre>
	 * 
	 * @param <ERElem> The specific type of Information System element to retrieve
	 * @param clazz The class representing the type of instances to retrieve
	 * @param polymorphic Whether to include instances of subtypes (true) or only exact type instances (false)
	 * @return List of instances of the specified type accessible in the current context
	 * @throws ResourceRegistryException If an error occurs during instance retrieval or if the type is invalid
	 */	
	public <ERElem extends ERElement> List<ERElem> list(Class<ERElem> clazz, Boolean polymorphic)
			throws ResourceRegistryException;
	
	/**
	 * Retrieves all instances of the specified type from the Resource Registry as JSON string.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /instances/{type-name}[?polymorphic={true|false}&limit={number}&offset={number}&includeMeta={true|false}&allMeta={true|false}&includeContexts={true|false}&hierarchical={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Returns all instances of the specified type that are accessible in the current context</li>
	 * <li>Supports polymorphic queries to include instances of subtypes</li>
	 * <li>Results are filtered based on user authorization and context access</li>
	 * <li>Returns the response as a JSON array string.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>polymorphic</strong> (automatically set from method parameter):
	 * <ul>
	 * <li>Whether to include instances of subtypes in the response</li>
	 * <li>Default: true (includes all subtypes)</li>
	 * <li>Values: true (includes all subtypes) | false (exact type only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#POLYMORPHIC_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>limit</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Maximum number of instances to return in a single response</li>
	 * <li>Default value: server configuration dependent</li>
	 * <li>Usage: {@code client.setLimit(50)}</li>
	 * <li>Example: Set to 50 to get at most 50 instances per request</li>
	 * <li>Unlimited results: Set to {@code RequestInfo.UNBOUNDED_LIMIT} (-1) for unlimited results (may cause timeout if results are too many)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#LIMIT_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>offset</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Number of instances to skip from the beginning of the result set</li>
	 * <li>Default value: 0</li>
	 * <li>Usage: {@code client.setOffset(10)}</li>
	 * <li>Example: Set to 10 to skip the first 10 contexts (useful for pagination)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#OFFSET_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response instance</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration, admin-only):
	 * <ul>
	 * <li>Whether to include instances from child contexts of the current context</li>
	 * <li>Default value: false (child contexts not included)</li>
	 * <li>Usage: {@code publisher.setHierarchicalMode(true)}</li>
	 * <li>Values: true (includes child contexts) | false (current context only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#HIERARCHICAL_MODE_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Supports all Information System model types and their subtypes: Entities (Resources and all Resource subtypes, Facets and all Facet subtypes) and Relations (ConsistsOf and all ConsistsOf subtypes, IsRelatedTo and all IsRelatedTo subtypes)</li>
	 * <li>Type names are case-sensitive and must match exactly the class names defined in the Information System model.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The instance must be accessible in the current context or child contexts (depending on hierarchical mode)</li>
	 * <li>IS-Manager and Infrastructure-Manager are global roles (available in all contexts) and can use hierarchical mode in any contexts</li>
	 * <li>Context-Manager is a context-specific role and can use hierarchical mode only in contexts where they have this role.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and read permissions for the specific context where they want to list instances.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Get all EService instances as JSON
	 * String servicesJson = publisher.list("EService", true);
	 * 
	 * // Get only exact ContactFacet instances as JSON
	 * String contactsJson = publisher.list("ContactFacet", false);
	 * </pre>
	 * 
	 * @param type The name of the Information System type whose instances to retrieve
	 * @param polymorphic Whether to include instances of subtypes (true) or only exact type instances (false)
	 * @return JSON array string containing the instances of the specified type accessible in the current context
	 * @throws ResourceRegistryException If an error occurs during instance retrieval or if the type is invalid
	 */
	public String list(String type, Boolean polymorphic) 
			throws ResourceRegistryException;
	
	/**
	 * Creates a new instance of an Information System element in the Resource Registry.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /instances/{type-name}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Creates a new entity (Resource, Facet) or relation (ConsistsOf, IsRelatedTo) instance</li>
	 * <li>The instance is automatically associated with the current context</li>
	 * <li>Validates the instance against the Information System model schema</li>
	 * <li>Generates a new UUID for the instance if not already set</li>
	 * <li>Returns the created instance with server-generated metadata.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>201 Created</strong>: Instance successfully created</li>
	 * <li><strong>400 Bad Request</strong>: Invalid instance data or schema validation failure</li>
	 * <li><strong>409 Conflict</strong>: Instance with the same UUID already exists</li>
	 * <li><strong>403 Forbidden</strong>: User lacks write permissions in the current context</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 *
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Type name is automatically derived from the provided object using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>Supports all Information System model types and their subtypes: Entities (Resources and all Resource subtypes, Facets and all Facet subtypes) and Relations (ConsistsOf and all ConsistsOf subtypes, IsRelatedTo and all IsRelatedTo subtypes).</li>
	 * </ul>
	 *
	 * <strong>Schema Validation:</strong>
	 * <ul>
	 * <li>All properties and relationships are validated against the Information System model</li>
	 * <li>Required properties must be present and valid</li>
	 * <li>Property types and constraints are enforced</li>
	 * <li>Relation targets must exist and be accessible in compatible contexts.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The instance is automatically associated with the current context.</li>
	 * </ul>
	 *
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and write permissions for the specific context where they want to create instances.</li>
	 * </ul>
	 *
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Create a new EService resource
	 * EService service = new EService();
	 * ...
	 * EService createdService = publisher.create(service);
	 * 
	 * // Create a new ContactFacet
	 * ContactFacet contact = new ContactFacet();
	 * ....
	 * ContactFacet createdContact = publisher.create(contact);
	 * </pre>
	 * 
	 * @param <ERElem> The specific type of Information System element to create
	 * @param er The instance to create in the Resource Registry
	 * @return The created instance with server-generated metadata and UUID
	 * @throws SchemaViolationException If the instance violates Information System model constraints (HTTP 400)
	 * @throws AlreadyPresentException If an instance with the same UUID already exists (HTTP 409)
	 * @throws ResourceRegistryException If an error occurs during instance creation
	 */
	public <ERElem extends ERElement> ERElem create(ERElem er)
			throws SchemaViolationException, AlreadyPresentException, ResourceRegistryException;

	/**
	 * Creates a new instance of an Information System element from JSON representation.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /instances/{type-name}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Creates a new entity (Resource, Facet) or relation (ConsistsOf, IsRelatedTo) instance from JSON</li>
	 * <li>The instance is automatically associated with the current context</li>
	 * <li>Validates the JSON against the Information System model schema</li>
	 * <li>Type is determined from the JSON @class property</li>
	 * <li>Generates a new UUID for the instance if not already set</li>
	 * <li>Returns the created instance as JSON with server-generated metadata.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>201 Created</strong>: Instance successfully created</li>
	 * <li><strong>400 Bad Request</strong>: Invalid JSON data or schema validation failure</li>
	 * <li><strong>409 Conflict</strong>: Instance with the same UUID already exists</li>
	 * <li><strong>403 Forbidden</strong>: User lacks write permissions in the current context</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>JSON Format Requirements:</strong>
	 * <ul>
	 * <li>Must contain a valid @class property specifying the Information System type</li>
	 * <li>Must conform to the Information System model JSON schema</li>
	 * <li>All required properties for the specified type must be present</li>
	 * <li>Nested objects and relations must be properly formatted.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation:</strong>
	 * <ul>
	 * <li>JSON structure is validated against the Information System model</li>
	 * <li>Required properties must be present and valid</li>
	 * <li>Property types and constraints are enforced</li>
	 * <li>Relation targets must exist and be accessible in compatible contexts.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The instance is automatically associated with the current context.</li>
	 * </ul>
	 *
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and write permissions for the specific context where they want to create instances.</li>
	 * </ul>
	 *
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Create a new EService from JSON
	 * String serviceJson = "...";
	 * String createdServiceJson = publisher.create(serviceJson);
	 * 
	 * // Create a new ContactFacet from JSON
	 * String contactJson = "...";
	 * String createdContactJson = publisher.create(contactJson);
	 * </pre>
	 * 
	 * @param json JSON representation of the instance to create
	 * @return JSON representation of the created instance with server-generated metadata and UUID
	 * @throws SchemaViolationException If the JSON violates Information System model constraints (HTTP 400)
	 * @throws AlreadyPresentException If an instance with the same UUID already exists (HTTP 409)
	 * @throws ResourceRegistryException If an error occurs during instance creation or JSON parsing
	 */
	public String create(String json)
			throws SchemaViolationException, AlreadyPresentException, ResourceRegistryException;
	
	/**
	 * Checks if an Information System element instance exists in the Resource Registry.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code HEAD /instances/{type-name}/{uuid}[?hierarchical={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Verifies the existence of the specified instance using its UUID and type</li>
	 * <li>The type name is automatically derived from the provided instance using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>Does not return the instance data, only confirms existence</li>
	 * <li>Checks accessibility within the current context and user authorization.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Instance exists and is accessible in the current context</li>
	 * <li><strong>404 Not Found</strong>: Instance does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: Instance exists but user lacks authorization to access it</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration, admin-only):
	 * <ul>
	 * <li>Whether to include instances from child contexts of the current context</li>
	 * <li>Default value: false (child contexts not included)</li>
	 * <li>Usage: {@code publisher.setHierarchicalMode(true)}</li>
	 * <li>Values: true (includes child contexts) | false (current context only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#HIERARCHICAL_MODE_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The type name is automatically derived from the provided instance using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>Supports all Information System model types and their subtypes: Entities (Resources and all Resource subtypes, Facets and all Facet subtypes) and Relations (ConsistsOf and all ConsistsOf subtypes, IsRelatedTo and all IsRelatedTo subtypes).</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The instance must be accessible in the current context or child contexts (depending on hierarchical mode)</li>
	 * <li>IS-Manager and Infrastructure-Manager are global roles (available in all contexts) and can use hierarchical mode in any contexts</li>
	 * <li>Context-Manager is a context-specific role and can use hierarchical mode only in contexts where they have this role.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and read permissions for the specific context where they want to check existence.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Check if an EService instance exists
	 * EService eService = new EService();
	 * eService.setID(someUUID);
	 * boolean exists = publisher.exist(eService);
	 * 
	 * // Check if a ContactFacet instance exists
	 * ContactFacet contact = new ContactFacet();
	 * contact.setID(anotherUUID);
	 * boolean contactExists = publisher.exist(contact);
	 * </pre>
	 * 
	 * @param <ERElem> The specific type of Information System element to check
	 * @param er The instance to check for existence (must have a valid UUID)
	 * @return true if the instance exists and is accessible in the current context, false if not found
	 * @throws AvailableInAnotherContextException If the instance exists but is not accessible in the current context (HTTP 403)
	 * @throws ResourceRegistryException If an error occurs during the existence check
	 */
	public <ERElem extends ERElement> boolean exist(ERElem er)
			throws AvailableInAnotherContextException, ResourceRegistryException;

	/**
	 * Checks if an Information System element instance exists in the Resource Registry using class and UUID.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code HEAD /instances/{type-name}/{uuid}[?hierarchical={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Verifies the existence of the specified instance using its UUID and class type</li>
	 * <li>The type name is automatically derived from the provided class using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>Does not return the instance data, only confirms existence</li>
	 * <li>Checks accessibility within the current context and user authorization.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Instance exists and is accessible in the current context</li>
	 * <li><strong>404 Not Found</strong>: Instance does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: Instance exists but user lacks authorization to access it</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration, admin-only):
	 * <ul>
	 * <li>Whether to include instances from child contexts of the current context</li>
	 * <li>Default value: false (child contexts not included)</li>
	 * <li>Usage: {@code publisher.setHierarchicalMode(true)}</li>
	 * <li>Values: true (includes child contexts) | false (current context only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#HIERARCHICAL_MODE_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The type name is automatically derived from the provided class using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(clazz)}</li>
	 * <li>Supports all Information System model types and their subtypes: Entities (Resources and all Resource subtypes, Facets and all Facet subtypes) and Relations (ConsistsOf and all ConsistsOf subtypes, IsRelatedTo and all IsRelatedTo subtypes).</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The instance must be accessible in the current context or child contexts (depending on hierarchical mode)</li>
	 * <li>IS-Manager and Infrastructure-Manager are global roles (available in all contexts) and can use hierarchical mode in any contexts</li>
	 * <li>Context-Manager is a context-specific role and can use hierarchical mode only in contexts where they have this role.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and read permissions for the specific context where they want to check existence.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Check if an EService instance exists by class and UUID
	 * UUID serviceUUID = UUID.fromString("...");
	 * boolean exists = publisher.exist(EService.class, serviceUUID);
	 * 
	 * // Check if a ContactFacet instance exists by class and UUID
	 * UUID contactUUID = UUID.fromString("...");
	 * boolean contactExists = publisher.exist(ContactFacet.class, contactUUID);
	 * </pre>
	 * 
	 * @param <ERElem> The specific type of Information System element to check
	 * @param clazz The class representing the type of instance to check
	 * @param uuid The UUID of the instance to check for existence
	 * @return true if the instance exists and is accessible in the current context, false if not found
	 * @throws AvailableInAnotherContextException If the instance exists but is not accessible in the current context (HTTP 403)
	 * @throws ResourceRegistryException If an error occurs during the existence check
	 */
	public <ERElem extends ERElement> boolean exist(Class<ERElem> clazz, UUID uuid)
			throws AvailableInAnotherContextException, ResourceRegistryException;
	
	/**
	 * Checks if an Information System element instance exists in the Resource Registry using type name and UUID.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code HEAD /instances/{type-name}/{uuid}[?hierarchical={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Verifies the existence of the specified instance using its UUID and type name</li>
	 * <li>Does not return the instance data, only confirms existence</li>
	 * <li>Checks accessibility within the current context and user authorization</li>
	 * <li>Most direct method when type name is already known.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Instance exists and is accessible in the current context</li>
	 * <li><strong>404 Not Found</strong>: Instance does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: Instance exists but user lacks authorization to access it</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration, admin-only):
	 * <ul>
	 * <li>Whether to include instances from child contexts of the current context</li>
	 * <li>Default value: false (child contexts not included)</li>
	 * <li>Usage: {@code publisher.setHierarchicalMode(true)}</li>
	 * <li>Values: true (includes child contexts) | false (current context only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#HIERARCHICAL_MODE_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Type name must be a valid Information System model type</li>
	 * <li>Supports all Information System model types and their subtypes: Entities (Resources and all Resource subtypes, Facets and all Facet subtypes) and Relations (ConsistsOf and all ConsistsOf subtypes, IsRelatedTo and all IsRelatedTo subtypes)</li>
	 * <li>Type name is case-sensitive and must match exactly.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The instance must be accessible in the current context or child contexts (depending on hierarchical mode)</li>
	 * <li>IS-Manager and Infrastructure-Manager are global roles (available in all contexts) and can use hierarchical mode in any contexts</li>
	 * <li>Context-Manager is a context-specific role and can use hierarchical mode only in contexts where they have this role.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and read permissions for the specific context where they want to check existence.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Check if an EService instance exists by type name and UUID
	 * UUID serviceUUID = UUID.fromString("...");
	 * boolean exists = publisher.exist("EService", serviceUUID);
	 * 
	 * // Check if a ContactFacet instance exists by type name and UUID
	 * UUID contactUUID = UUID.fromString("...");
	 * boolean contactExists = publisher.exist("ContactFacet", contactUUID);
	 * </pre>
	 * 
	 * @param type The name of the Information System type to check
	 * @param uuid The UUID of the instance to check for existence
	 * @return true if the instance exists and is accessible in the current context, false if not found
	 * @throws AvailableInAnotherContextException If the instance exists but is not accessible in the current context (HTTP 403)
	 * @throws ResourceRegistryException If an error occurs during the existence check
	 */
	public boolean exist(String type, UUID uuid)
			throws AvailableInAnotherContextException, ResourceRegistryException;
	
	/**
	 * Reads an existing instance from the Resource Registry using a provided instance object.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /instances/{type-name}/{uuid}[?includeMeta={true|false}&allMeta={true|false}&includeContexts={true|false}&hierarchical={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves an existing instance (Entity or Relation) by extracting type and UUID from the provided instance</li>
	 * <li>The instance must be accessible in the current context or in child contexts (based on hierarchical mode)</li>
	 * <li>Returns the complete instance with all properties and metadata</li>
	 * <li>The type is automatically determined from the provided instance object.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Instance successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: No instance found with the specified UUID or not accessible in current context</li>
	 * <li><strong>403 Forbidden</strong>: User lacks read permissions for the instance or its context</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response instance</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration, admin-only):
	 * <ul>
	 * <li>Whether to include instances from child contexts of the current context</li>
	 * <li>Default value: false (child contexts not included)</li>
	 * <li>Usage: {@code publisher.setHierarchicalMode(true)}</li>
	 * <li>Values: true (includes child contexts) | false (current context only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#HIERARCHICAL_MODE_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The type name is automatically derived from the provided instance using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>The UUID is extracted from the provided instance's header.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The instance must be accessible in the current context or child contexts (depending on hierarchical mode)</li>
	 * <li>IS-Manager and Infrastructure-Manager are global roles (available in all contexts) and can use hierarchical mode in any contexts</li>
	 * <li>Context-Manager is a context-specific role and can use hierarchical mode only in contexts where they have this role.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and read permissions for the specific context where they want to read instances.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Read an EService Resource
	 * EService eService = new EService();
	 * eService.setHeader(new Header(uuid));
	 * EService readEService = publisher.read(eService);
	 * </pre>
	 * 
	 * @param <ERElem> The specific type of Entity or Relation to read
	 * @param er Instance object containing at least the UUID in the header
	 * @return The complete instance from the Resource Registry
	 * @throws NotFoundException If no instance is found with the specified UUID (HTTP 404)
	 * @throws AvailableInAnotherContextException If the instance exists but is not accessible in the current context (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during instance retrieval
	 */
	public <ERElem extends ERElement> ERElem read(ERElem er)
			throws NotFoundException, AvailableInAnotherContextException, ResourceRegistryException;

	/**
	 * Reads an existing instance from the Resource Registry by its type class and UUID.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /instances/{type-name}/{uuid}[?includeMeta={true|false}&allMeta={true|false}&includeContexts={true|false}&hierarchical={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves an existing instance (Entity or Relation) by its type class and unique identifier</li>
	 * <li>The instance must be accessible in the current context or in child contexts (based on hierarchical mode)</li>
	 * <li>Returns the complete instance with all properties and metadata</li>
	 * <li>Type validation ensures the UUID corresponds to an instance of the specified class.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Instance successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: No instance found with the specified UUID and type, or not accessible in current context</li>
	 * <li><strong>403 Forbidden</strong>: User lacks read permissions for the instance or its context</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response instance</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration, admin-only):
	 * <ul>
	 * <li>Whether to include instances from child contexts of the current context</li>
	 * <li>Default value: false (child contexts not included)</li>
	 * <li>Usage: {@code publisher.setHierarchicalMode(true)}</li>
	 * <li>Values: true (includes child contexts) | false (current context only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#HIERARCHICAL_MODE_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The type name is automatically derived from the provided class using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>The class must be a valid Information System model type.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The instance must be accessible in the current context or child contexts (depending on hierarchical mode)</li>
	 * <li>IS-Manager and Infrastructure-Manager are global roles (available in all contexts) and can use hierarchical mode in any contexts</li>
	 * <li>Context-Manager is a context-specific role and can use hierarchical mode only in contexts where they have this role.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and read permissions for the specific context where they want to read instances.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Read an EService Resource by class and UUID
	 * UUID serviceUuid = UUID.fromString("...");
	 * EService service = publisher.read(EService.class, serviceUuid);
	 * </pre>
	 * 
	 * @param <ERElem> The specific type of Entity or Relation to read
	 * @param clazz The class of the Entity or Relation type to read
	 * @param uuid The unique identifier of the instance to read
	 * @return The complete instance from the Resource Registry
	 * @throws NotFoundException If no instance is found with the specified UUID and type (HTTP 404)
	 * @throws AvailableInAnotherContextException If the instance exists but is not accessible in the current context (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during instance retrieval
	 */
	public <ERElem extends ERElement> ERElem read(Class<ERElem> clazz, UUID uuid)
			throws NotFoundException, AvailableInAnotherContextException, ResourceRegistryException;
	
	/**
	 * Reads an existing instance from the Resource Registry by its type name and UUID.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /instances/{type-name}/{uuid}[?includeMeta={true|false}&allMeta={true|false}&includeContexts={true|false}&hierarchical={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves an existing instance (Entity or Relation) by its type name and unique identifier</li>
	 * <li>The instance must be accessible in the current context or in child contexts (based on hierarchical mode)</li>
	 * <li>Returns the complete instance as JSON with all properties and metadata</li>
	 * <li>Type validation ensures the UUID corresponds to an instance of the specified type.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Instance successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: No instance found with the specified UUID and type, or not accessible in current context</li>
	 * <li><strong>403 Forbidden</strong>: User lacks read permissions for the instance or its context</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response instance</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration, admin-only):
	 * <ul>
	 * <li>Whether to include instances from child contexts of the current context</li>
	 * <li>Default value: false (child contexts not included)</li>
	 * <li>Usage: {@code publisher.setHierarchicalMode(true)}</li>
	 * <li>Values: true (includes child contexts) | false (current context only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#HIERARCHICAL_MODE_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The specified type must be a valid Information System model type name</li>
	 * <li>Supports all Information System model types and their subtypes: Entities (Resources and all Resource subtypes, Facets and all Facet subtypes) and Relations (ConsistsOf and all ConsistsOf subtypes, IsRelatedTo and all IsRelatedTo subtypes)</li>
	 * <li>Type names are case-sensitive and must match exactly the class names defined in the Information System model</li>
	 * <li>UUID must correspond to an instance of exactly the specified type (no polymorphic matching).</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The instance must be accessible in the current context or child contexts (depending on hierarchical mode)</li>
	 * <li>IS-Manager and Infrastructure-Manager are global roles (available in all contexts) and can use hierarchical mode in any contexts</li>
	 * <li>Context-Manager is a context-specific role and can use hierarchical mode only in contexts where they have this role.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and read permissions for the specific context where they want to read instances.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Read an EService Resource by type and UUID
	 * UUID serviceUuid = UUID.fromString("...");
	 * String serviceJson = publisher.read("EService", serviceUuid);
	 * </pre>
	 * 
	 * @param type The exact type name of the Entity or Relation to read
	 * @param uuid The unique identifier of the instance to read
	 * @return JSON representation of the complete instance from the Resource Registry
	 * @throws NotFoundException If no instance is found with the specified UUID and type (HTTP 404)
	 * @throws AvailableInAnotherContextException If the instance exists but is not accessible in the current context (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during instance retrieval
	 */
	public String read(String type, UUID uuid) 
			throws NotFoundException, AvailableInAnotherContextException, ResourceRegistryException;
	
	/**
	 * Updates an existing instance in the Resource Registry using the instance object.
	 * 
	 * <strong>REST Endpoint:</strong> {@code PUT /instances/{type-name}/{instance-uuid}}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>Basic update: {@code EService myService = publisher.update(existingService);}</li>
	 * <li>With metadata in response: {@code publisher.includeMeta(true); EService myService = publisher.update(existingService);}</li>
	 * <li>Full metadata context: {@code publisher.includeMeta(true); publisher.allMeta(true); EService myService = publisher.update(existingService);}</li>
	 * <li>With context information: {@code publisher.includeContexts(true); EService myService = publisher.update(existingService);}.</li>
	 * </ul>
	 * 
	 * <strong>Update Behavior:</strong>
	 * <ul>
	 * <li>Updates an existing instance with the same UUID as the provided instance</li>
	 * <li>The instance must already exist in the Resource Registry</li>
	 * <li>Only modifiable properties are updated; metadata fields are automatically managed</li>
	 * <li>The instance UUID cannot be changed during update operations.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response instance</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 *
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The type name is automatically derived from the provided instance using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>The UUID is extracted from the provided instance's header.</li>
	 * </ul>
	 *
	 * <strong>Schema Validation:</strong>
	 * <ul>
	 * <li>All properties and relationships are validated against the Information System model</li>
	 * <li>Required properties must be present and valid</li>
	 * <li>Property types and constraints are enforced.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The instance must be accessible in the current context.</li>
	 * </ul>
	 *
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>Valid authorization token required for accessing the current context</li>
	 * <li>Update permissions required for the specific instance type in the current context</li>
	 * <li>The current context is derived from the authorization token provided during client creation.</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Instance successfully updated</li>
	 * <li><strong>400 Bad Request</strong>: Invalid request body, malformed instance data, or schema validation failed</li>
	 * <li><strong>404 Not Found</strong>: The instance does not exist</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to update the instance.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns the updated instance object with its properties and relations</li>
	 * <li>Response structure depends on includeMeta, allMeta, and includeContexts settings</li>
	 * <li>All nested elements are updated according to the current configuration.</li>
	 * </ul>
	 * 
	 * @param <ERElem> The generic type extending ERElement (Entity or Relation)
	 * @param er The instance object to update with new values
	 * @return The updated instance object with current data from the Resource Registry
	 * @throws SchemaViolationException If the updated instance violates type schema constraints
	 * @throws NotFoundException If the instance to update does not exist
	 * @throws ResourceRegistryException If an error occurs during instance update
	 */
	public <ERElem extends ERElement> ERElem update(ERElem er)
			throws SchemaViolationException, NotFoundException, ResourceRegistryException;

	/**
	 * Updates an existing instance in the Resource Registry using JSON representation.
	 * 
	 * <strong>REST Endpoint:</strong> {@code PUT /instances/{type-name}/{instance-uuid}}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>Basic update: {@code String result = publisher.update(jsonString);}</li>
	 * <li>With metadata in response: {@code publisher.includeMeta(true); String result = publisher.update(jsonString);}</li>
	 * <li>Full metadata context: {@code publisher.includeMeta(true); publisher.allMeta(true); String result = publisher.update(jsonString);}</li>
	 * <li>With context information: {@code publisher.includeContexts(true); String result = publisher.update(jsonString);}.</li>
	 * </ul>
	 * 
	 * <strong>Update Behavior:</strong>
	 * <ul>
	 * <li>Updates an existing instance using JSON representation</li>
	 * <li>The JSON must contain the UUID of the instance to update</li>
	 * <li>The instance type is automatically detected from the JSON content</li>
	 * <li>The instance must already exist in the Resource Registry</li>
	 * <li>Only modifiable properties are updated; metadata fields are automatically managed.</li>
	 * </ul>
	 * 
	 * <strong>JSON Format Requirements:</strong>
	 * <ul>
	 * <li>Valid JSON representation of an Information System instance</li>
	 * <li>Must include the "id" field with the UUID of the instance to update</li>
	 * <li>Must include the "type" field to identify the instance type</li>
	 * <li>Must conform to the schema of the specified type.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response instance</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 *
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The instance type is automatically detected from the JSON content</li>
	 * <li>The JSON must contain the UUID of the instance to update.</li>
	 * </ul>
	 *
	 * <strong>Schema Validation:</strong>
	 * <ul>
	 * <li>JSON structure is validated against the Information System model</li>
	 * <li>Required properties must be present and valid</li>
	 * <li>Property types and constraints are enforced.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The instance must be accessible in the current context.</li>
	 * </ul>
	 *
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>Valid authorization token required for accessing the current context</li>
	 * <li>Update permissions required for the specific instance type in the current context</li>
	 * <li>The current context is derived from the authorization token provided during client creation.</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Instance successfully updated</li>
	 * <li><strong>400 Bad Request</strong>: Invalid JSON format, missing required fields, or schema validation failed</li>
	 * <li><strong>404 Not Found</strong>: The instance does not exist</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to update the instance.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON string containing the updated instance</li>
	 * <li>Response structure depends on includeMeta, allMeta, and includeContexts settings</li>
	 * <li>All nested elements are updated according to the current configuration.</li>
	 * </ul>
	 * 
	 * @param json JSON representation of the instance to update
	 * @return JSON string containing the updated instance with current data from the Resource Registry
	 * @throws SchemaViolationException If the updated instance violates type schema constraints
	 * @throws NotFoundException If the instance to update does not exist
	 * @throws ResourceRegistryException If an error occurs during instance update
	 */
	public String update(String json) 
			throws SchemaViolationException, NotFoundException, ResourceRegistryException;
	
	/**
	 * Updates an existing instance in the Resource Registry using type name and JSON representation.
	 * 
	 * <strong>REST Endpoint:</strong> {@code PUT /instances/{type-name}/{instance-uuid}}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>Basic update: {@code String result = publisher.update("EService", jsonString);}</li>
	 * <li>With metadata in response: {@code publisher.includeMeta(true); String result = publisher.update("EService", jsonString);}</li>
	 * <li>Full metadata context: {@code publisher.includeMeta(true); publisher.allMeta(true); String result = publisher.update("EService", jsonString);}</li>
	 * <li>With context information: {@code publisher.includeContexts(true); String result = publisher.update("EService", jsonString);}.</li>
	 * </ul>
	 * 
	 * <strong>Update Behavior:</strong>
	 * <ul>
	 * <li>Updates an existing instance using explicit type name and JSON representation</li>
	 * <li>The JSON must contain the UUID of the instance to update</li>
	 * <li>The provided type name must match the actual instance type</li>
	 * <li>The instance must already exist in the Resource Registry</li>
	 * <li>Only modifiable properties are updated; metadata fields are automatically managed.</li>
	 * </ul>
	 * 
	 * <strong>JSON Format Requirements:</strong>
	 * <ul>
	 * <li>Valid JSON representation of an Information System instance</li>
	 * <li>Must include the "id" field with the UUID of the instance to update</li>
	 * <li>May include the "type" field, but the explicitly provided type parameter takes precedence</li>
	 * <li>Must conform to the schema of the specified type.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response instance</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 *
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The provided type name must match the actual instance type</li>
	 * <li>The JSON must contain the UUID of the instance to update.</li>
	 * </ul>
	 *
	 * <strong>Schema Validation:</strong>
	 * <ul>
	 * <li>JSON structure is validated against the Information System model</li>
	 * <li>Required properties must be present and valid</li>
	 * <li>Property types and constraints are enforced.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The instance must be accessible in the current context.</li>
	 * </ul>
	 *
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>Valid authorization token required for accessing the current context</li>
	 * <li>Update permissions required for the specific instance type in the current context</li>
	 * <li>The current context is derived from the authorization token provided during client creation.</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Instance successfully updated</li>
	 * <li><strong>400 Bad Request</strong>: Invalid JSON format, type mismatch, missing required fields, or schema validation failed</li>
	 * <li><strong>404 Not Found</strong>: The instance or specified type does not exist</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to update the instance.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON string containing the updated instance</li>
	 * <li>Response structure depends on includeMeta, allMeta, and includeContexts settings</li>
	 * <li>All nested elements are updated according to the current configuration.</li>
	 * </ul>
	 * 
	 * @param type The name of the Information System type (e.g., "EService", "ContactFacet")
	 * @param json JSON representation of the instance to update
	 * @return JSON string containing the updated instance with current data from the Resource Registry
	 * @throws SchemaViolationException If the updated instance violates type schema constraints
	 * @throws NotFoundException If the instance to update or the specified type does not exist
	 * @throws ResourceRegistryException If an error occurs during instance update
	 */
	public String update(String type, String json)
			throws SchemaViolationException, NotFoundException, ResourceRegistryException;
	
	/**
	 * Updates an existing instance in the Resource Registry using type name, JSON representation, and explicit UUID.
	 * 
	 * <strong>REST Endpoint:</strong> {@code PUT /instances/{type-name}/{instance-uuid}}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>Basic update: {@code String result = publisher.update("EService", jsonString, instanceUuid);}</li>
	 * <li>With metadata in response: {@code publisher.includeMeta(true); String result = publisher.update("EService", jsonString, instanceUuid);}</li>
	 * <li>Full metadata context: {@code publisher.includeMeta(true); publisher.allMeta(true); String result = publisher.update("EService", jsonString, instanceUuid);}</li>
	 * <li>With context information: {@code publisher.includeContexts(true); String result = publisher.update("EService", jsonString, instanceUuid);}.</li>
	 * </ul>
	 * 
	 * <strong>Update Behavior:</strong>
	 * <ul>
	 * <li>Updates an existing instance using explicit type name, JSON representation, and UUID</li>
	 * <li>The explicitly provided UUID takes precedence over any UUID in the JSON content</li>
	 * <li>The provided type name must match the actual instance type</li>
	 * <li>The instance must already exist in the Resource Registry</li>
	 * <li>Only modifiable properties are updated; metadata fields are automatically managed</li>
	 * <li>This method provides the most explicit control over instance identification.</li>
	 * </ul>
	 * 
	 * <strong>JSON Format Requirements:</strong>
	 * <ul>
	 * <li>Valid JSON representation of an Information System instance</li>
	 * <li>The "id" field in JSON is ignored if present; the uuid parameter is used instead</li>
	 * <li>May include the "type" field, but the explicitly provided type parameter takes precedence</li>
	 * <li>Must conform to the schema of the specified type.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response instance</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 *
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The explicitly provided UUID takes precedence over any UUID in the JSON content</li>
	 * <li>The provided type name must match the actual instance type.</li>
	 * </ul>
	 *
	 * <strong>Schema Validation:</strong>
	 * <ul>
	 * <li>JSON structure is validated against the Information System model</li>
	 * <li>Required properties must be present and valid</li>
	 * <li>Property types and constraints are enforced.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The instance must be accessible in the current context.</li>
	 * </ul>
	 *
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>Valid authorization token required for accessing the current context</li>
	 * <li>Update permissions required for the specific instance type in the current context</li>
	 * <li>The current context is derived from the authorization token provided during client creation.</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Instance successfully updated</li>
	 * <li><strong>400 Bad Request</strong>: Invalid JSON format, type mismatch, invalid UUID, or schema validation failed</li>
	 * <li><strong>404 Not Found</strong>: The instance with the specified UUID or the specified type does not exist</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to update the instance.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON string containing the updated instance</li>
	 * <li>Response structure depends on includeMeta, allMeta, and includeContexts settings</li>
	 * <li>All nested elements are updated according to the current configuration.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>Valid authorization token required for accessing the current context</li>
	 * <li>Update permissions required for the specific instance type in the current context</li>
	 * <li>The current context is derived from the authorization token provided during client creation.</li>
	 * </ul>
	 * 
	 * @param type The name of the Information System type (e.g., "EService", "ContactFacet")
	 * @param json JSON representation of the instance to update (UUID in JSON is ignored)
	 * @param uuid The unique identifier of the instance to update
	 * @return JSON string containing the updated instance with current data from the Resource Registry
	 * @throws SchemaViolationException If the updated instance violates type schema constraints
	 * @throws NotFoundException If the instance with the specified UUID or the specified type does not exist
	 * @throws ResourceRegistryException If an error occurs during instance update
	 */
	public String update(String type, String json, UUID uuid)
			throws SchemaViolationException, NotFoundException, ResourceRegistryException;
	
	/**
	 * Deletes an Information System element instance from the Resource Registry.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code DELETE /instances/{type-name}/{uuid}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Permanently deletes the specified instance from the Resource Registry</li>
	 * <li>The type and UUID are automatically extracted from the provided element object</li>
	 * <li>The instance must exist and be accessible in the current context</li>
	 * <li>Supports both Resource and Facet types and their specializations</li>
	 * <li>This is a destructive operation that cannot be undone</li>
	 * <li>Related elements may also be deleted based on propagation constraints defined in the schema.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Instance successfully deleted</li>
	 * <li><strong>404 Not Found</strong>: Instance does not exist or is not accessible in the current context</li>
	 * <li><strong>403 Forbidden</strong>: User lacks permissions to delete the instance</li>
	 * <li><strong>409 Conflict</strong>: Instance cannot be deleted due to existing relationships or schema constraints</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The type name is automatically derived from the provided element object using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>The UUID is extracted from the element's header</li>
	 * <li>Supports all ERElement types: Resources, Facets, ConsistsOf relations, and IsRelatedTo relations.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The instance must be accessible in the current context.</li>
	 * </ul>
	 *
	 * <strong>Deletion Propagation Rules:</strong>
	 * <ul>
	 * <li><strong>Resources</strong>: When a Resource is deleted, all its associated Facets (via ConsistsOf relations) may also be deleted if they become orphaned and the delete propagation constraint is {@code propagate}</li>
	 * <li><strong>Facets</strong>: When a Facet is deleted, the parent Resources that reference it may also be affected based on schema constraints</li>
	 * <li><strong>Relations</strong>: When a relation is deleted, the connected elements remain but the relationship is removed</li>
	 * <li><strong>Cascade Effects</strong>: The actual propagation behavior depends on the schema-defined constraints for each type.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation and Constraints:</strong>
	 * <ul>
	 * <li>Before deletion, the system validates that removing the instance will not violate schema constraints</li>
	 * <li>If deletion would leave required relationships unsatisfied or violate cardinality constraints, a {@link SchemaViolationException} is thrown</li>
	 * <li>This results in an HTTP 409 Conflict response with details about the constraint violation</li>
	 * <li><strong>Example scenarios that prevent deletion:</strong></li>
	 * </ul>
	 *   • Deleting a Facet that is required by an existing Resource;
	 *   • Deleting a Resource that has mandatory relationships that would be violated;
	 *   • Operations that would make related elements incomplete according to their schema definition.
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and delete permissions for the specific context where the instance exists.</li>
	 * </ul>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can delete any instance in any context</li>
	 * <li>Full administrative privileges across all contexts.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can delete any instance in any context</li>
	 * <li>Full administrative privileges across all contexts.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can delete instances only in contexts where they have Context-Manager role</li>
	 * <li>Operations are limited to their assigned contexts.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can delete instances only in contexts where they have appropriate permissions</li>
	 * <li>Operations are limited to instances they have access to in their authorized contexts.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Delete an EService instance
	 * EService service = publisher.read(EService.class, serviceUUID);
	 * boolean deleted = publisher.delete(service);
	 * 
	 * // Delete a ContactFacet instance
	 * ContactFacet contact = publisher.read(ContactFacet.class, contactUUID);
	 * boolean contactDeleted = publisher.delete(contact);
	 * </pre>
	 * 
	 * @param <ERElem> The specific type of Information System element to delete
	 * @param er The instance to delete (must contain a valid UUID in the header)
	 * @return true if the instance was successfully deleted, false otherwise
	 * @throws SchemaViolationException If deleting the instance would violate Information System model constraints (HTTP 409)
	 * @throws NotFoundException If the instance does not exist or is not accessible (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during the delete operation
	 */
	public <ERElem extends ERElement> boolean delete(ERElem er)
			throws SchemaViolationException, NotFoundException, ResourceRegistryException;

	/**
	 * Deletes an Information System element instance from the Resource Registry using type name and UUID.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code DELETE /instances/{type-name}/{uuid}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Permanently deletes the specified instance from the Resource Registry</li>
	 * <li>The instance is identified by its type name and UUID</li>
	 * <li>The instance must exist and be accessible in the current context</li>
	 * <li>Supports both Resource and Facet types and their specializations</li>
	 * <li>This is a destructive operation that cannot be undone</li>
	 * <li>Related elements may also be deleted based on propagation constraints defined in the schema.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Instance successfully deleted</li>
	 * <li><strong>404 Not Found</strong>: Instance does not exist or is not accessible in the current context</li>
	 * <li><strong>403 Forbidden</strong>: User lacks permissions to delete the instance</li>
	 * <li><strong>409 Conflict</strong>: Instance cannot be deleted due to existing relationships or schema constraints</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The type parameter must be a valid ERElement type name registered in the Information System</li>
	 * <li>Type names are case-sensitive and must match exactly</li>
	 * <li>Supports all ERElement types: Resources (e.g., EService, HostingNode), Facets (e.g., StateFacet, ContactFacet), ConsistsOf relations, and IsRelatedTo relations</li>
	 * <li>The UUID must be a valid UUID that identifies an existing instance.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The instance must be accessible in the current context.</li>
	 * </ul>
	 *
	 * <strong>Deletion Propagation Rules:</strong>
	 * <ul>
	 * <li><strong>Resources</strong>: When a Resource is deleted, all its associated Facets (via ConsistsOf relations) may also be deleted if they become orphaned and the delete propagation constraint is {@code propagate}</li>
	 * <li><strong>Facets</strong>: When a Facet is deleted, the parent Resources that reference it may also be affected based on schema constraints</li>
	 * <li><strong>Relations</strong>: When a relation is deleted, the connected elements remain but the relationship is removed</li>
	 * <li><strong>Cascade Effects</strong>: The actual propagation behavior depends on the schema-defined constraints for each type.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation and Constraints:</strong>
	 * <ul>
	 * <li>Before deletion, the system validates that removing the instance will not violate schema constraints</li>
	 * <li>If deletion would leave required relationships unsatisfied or violate cardinality constraints, a {@link SchemaViolationException} is thrown</li>
	 * <li>This results in an HTTP 409 Conflict response with details about the constraint violation</li>
	 * <li><strong>Example scenarios that prevent deletion:</strong></li>
	 * </ul>
	 *   • Deleting a Facet that is required by an existing Resource;
	 *   • Deleting a Resource that has mandatory relationships that would be violated;
	 *   • Operations that would make related elements incomplete according to their schema definition.
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and delete permissions for the specific context where the instance exists.</li>
	 * </ul>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can delete any instance in any context</li>
	 * <li>Full administrative privileges across all contexts.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can delete any instance in any context</li>
	 * <li>Full administrative privileges across all contexts.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can delete instances only in contexts where they have Context-Manager role</li>
	 * <li>Operations are limited to their assigned contexts.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can delete instances only in contexts where they have appropriate permissions</li>
	 * <li>Operations are limited to instances they have access to in their authorized contexts.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Delete an EService instance by type and UUID
	 * UUID serviceUUID = UUID.fromString("12345678-1234-1234-1234-123456789abc");
	 * boolean deleted = publisher.delete("EService", serviceUUID);
	 * 
	 * // Delete a ContactFacet instance by type and UUID
	 * UUID contactUUID = UUID.fromString("87654321-4321-4321-4321-cba987654321");
	 * boolean contactDeleted = publisher.delete("ContactFacet", contactUUID);
	 * </pre>
	 * 
	 * @param type The exact type name of the ERElement to delete
	 * @param uuid The UUID of the instance to delete
	 * @return true if the instance was successfully deleted, false otherwise
	 * @throws SchemaViolationException If deleting the instance would violate Information System model constraints (HTTP 409)
	 * @throws NotFoundException If the instance does not exist or is not accessible (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during the delete operation
	 */
	public boolean delete(String type, UUID uuid)
			throws SchemaViolationException, NotFoundException, ResourceRegistryException;

	/**
	 * Creates a new Facet instance in the Resource Registry.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /instances/{facet-type-name}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Creates a new Facet entity in the Information System</li>
	 * <li>Facets represent specific aspects or properties that can be attached to Resources via ConsistsOf relations</li>
	 * <li>The instance is automatically associated with the current context</li>
	 * <li>Validates the facet against the Information System model schema</li>
	 * <li>Generates a new UUID for the facet if not already set</li>
	 * <li>Returns the created facet with server-generated metadata.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>201 Created</strong>: Facet successfully created</li>
	 * <li><strong>400 Bad Request</strong>: Invalid facet data or schema validation failure</li>
	 * <li><strong>409 Conflict</strong>: Facet with the same UUID already exists</li>
	 * <li><strong>403 Forbidden</strong>: User lacks write permissions in the current context</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The facet type name is automatically derived from the provided object using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>Supports all Facet subtypes defined in the Information System model.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation:</strong>
	 * <ul>
	 * <li>All facet properties are validated against the Information System model</li>
	 * <li>Required properties must be present and valid</li>
	 * <li>Property types and constraints are enforced</li>
	 * <li>Facet-specific validation rules are applied.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The instance is automatically associated with the current context.</li>
	 * </ul>
	 *
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and write permissions for the specific context where they want to create facets.</li>
	 * </ul>
	 *
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Create a new ContactFacet
	 * ContactFacet contact = new ContactFacet();
	 * ...
	 * ContactFacet createdContact = publisher.createFacet(contact);
	 * 
	 * // Create a new SoftwareFacet
	 * SoftwareFacet software = new SoftwareFacet();
	 * ...
	 * SoftwareFacet createdSoftware = publisher.createFacet(software);
	 * </pre>
	 * 
	 * @param <F> The specific type of Facet to create
	 * @param facet The Facet instance to create in the Resource Registry
	 * @return The created Facet with server-generated metadata and UUID
	 * @throws SchemaViolationException If the facet violates Information System model constraints (HTTP 400)
	 * @throws FacetAlreadyPresentException If a facet with the same UUID already exists (HTTP 409)
	 * @throws ResourceRegistryException If an error occurs during facet creation
	 */
	public <F extends Facet> F createFacet(F facet)
			throws SchemaViolationException, FacetAlreadyPresentException, ResourceRegistryException;

	/**
	 * Creates a new Facet instance from JSON representation.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /instances/{facet-type-name}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Creates a new Facet entity from JSON representation</li>
	 * <li>Facets represent specific aspects or properties that can be attached to Resources via ConsistsOf relations</li>
	 * <li>The instance is automatically associated with the current context</li>
	 * <li>Validates the JSON against the Information System model schema</li>
	 * <li>Type is determined from the JSON @class property</li>
	 * <li>Generates a new UUID for the facet if not already set</li>
	 * <li>Returns the created facet as JSON with server-generated metadata.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>201 Created</strong>: Facet successfully created</li>
	 * <li><strong>400 Bad Request</strong>: Invalid JSON data or schema validation failure</li>
	 * <li><strong>409 Conflict</strong>: Facet with the same UUID already exists</li>
	 * <li><strong>403 Forbidden</strong>: User lacks write permissions in the current context</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>JSON Format Requirements:</strong>
	 * <ul>
	 * <li>Must contain a valid @class property specifying a Facet type</li>
	 * <li>Must conform to the Information System model JSON schema for Facets</li>
	 * <li>All required properties for the specified facet type must be present</li>
	 * <li>Facet-specific properties must be properly formatted.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation:</strong>
	 * <ul>
	 * <li>JSON structure is validated against the Information System model</li>
	 * <li>Required facet properties must be present and valid</li>
	 * <li>Property types and constraints are enforced</li>
	 * <li>Facet-specific validation rules are applied.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The instance is automatically associated with the current context.</li>
	 * </ul>
	 *
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and write permissions for the specific context where they want to create facets.</li>
	 * </ul>
	 *
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Create a new ContactFacet from JSON
	 * String contactJson = "...";
	 * String createdContactJson = publisher.createFacet(contactJson);
	 * 
	 * // Create a new SoftwareFacet from JSON
	 * String softwareJson = "...";
	 * String createdSoftwareJson = publisher.createFacet(softwareJson);
	 * </pre>
	 * 
	 * @param facet JSON representation of the Facet to create
	 * @return JSON representation of the created Facet with server-generated metadata and UUID
	 * @throws SchemaViolationException If the JSON violates Information System model constraints (HTTP 400)
	 * @throws FacetAlreadyPresentException If a facet with the same UUID already exists (HTTP 409)
	 * @throws ResourceRegistryException If an error occurs during facet creation or JSON parsing
	 */
	public String createFacet(String facet)
			throws SchemaViolationException, FacetAlreadyPresentException, ResourceRegistryException;

	/**
	 * Reads a Facet instance from the Resource Registry using a provided instance object.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /instances/{facet-type-name}/{uuid}[?includeMeta={true|false}&allMeta={true|false}&includeContexts={true|false}&hierarchical={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves an existing Facet by extracting type and UUID from the provided instance</li>
	 * <li>The Facet must be accessible in the current context or in child contexts (based on hierarchical mode)</li>
	 * <li>Returns the complete Facet instance with all properties and metadata</li>
	 * <li>The type is automatically determined from the provided instance object.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Facet successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: No Facet found with the specified UUID or not accessible in current context</li>
	 * <li><strong>403 Forbidden</strong>: User lacks read permissions for the Facet or its context</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response instance</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration, admin-only):
	 * <ul>
	 * <li>Whether to include Facets from child contexts of the current context</li>
	 * <li>Values: true (includes child contexts) | false (current context only)</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#HIERARCHICAL_MODE_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The Facet type name is automatically derived from the provided instance using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>The UUID is extracted from the provided instance's header</li>
	 * <li>The returned object will be of the same Facet type as the provided instance.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The Facet must be accessible in the current context or child contexts (depending on hierarchical mode)</li>
	 * <li>IS-Manager and Infrastructure-Manager are global roles (available in all contexts) and can use hierarchical mode in any contexts</li>
	 * <li>Context-Manager is a context-specific role and can use hierarchical mode only in contexts where they have this role.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and read permissions for the specific context where they want to read Facets.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Read a ContactFacet
	 * ContactFacet contactFacet = new ContactFacet();
	 * contactFacet.setHeader(new Header(uuid));
	 * ContactFacet readContactFacet = publisher.readFacet(contactFacet);
	 * </pre>
	 * 
	 * @param <F> The specific type of Facet to read
	 * @param facet Instance object containing at least the UUID in the header
	 * @return The complete Facet instance from the Resource Registry
	 * @throws FacetNotFoundException If no Facet is found with the specified UUID (HTTP 404)
	 * @throws FacetAvailableInAnotherContextException If the Facet exists but is not accessible in the current context (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during Facet retrieval
	 */
	public <F extends Facet> F readFacet(F facet) 
			throws FacetNotFoundException, FacetAvailableInAnotherContextException, ResourceRegistryException;

	/**
	 * Reads a Facet instance from the Resource Registry by its type and UUID.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /instances/{facet-type-name}/{uuid}[?includeMeta={true|false}&allMeta={true|false}&includeContexts={true|false}&hierarchical={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves an existing Facet by its type name and unique identifier</li>
	 * <li>The Facet must be accessible in the current context or in child contexts (based on hierarchical mode)</li>
	 * <li>Returns the complete Facet instance as JSON with all properties and metadata</li>
	 * <li>Type validation ensures the UUID corresponds to a Facet of the specified type.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Facet successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: No Facet found with the specified UUID and type, or not accessible in current context</li>
	 * <li><strong>403 Forbidden</strong>: User lacks read permissions for the Facet or its context</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response instance</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration, admin-only):
	 * <ul>
	 * <li>Whether to include instances from child contexts of the current context</li>
	 * <li>Default value: false (child contexts not included)</li>
	 * <li>Usage: {@code publisher.setHierarchicalMode(true)}</li>
	 * <li>Values: true (includes child contexts) | false (current context only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#HIERARCHICAL_MODE_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The specified type must be a valid Facet type in the Information System model</li>
	 * <li>Supports all Facet types and their subtypes as defined in the Information System model</li>
	 * <li>Type names are case-sensitive and must match exactly the class names defined in the Information System model</li>
	 * <li>UUID must correspond to a Facet of exactly the specified type (no polymorphic matching).	 * </li>
	 * </ul>
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The Facet must be accessible in the current context or child contexts (depending on hierarchical mode)</li>
	 * <li>IS-Manager and Infrastructure-Manager are global roles (available in all contexts) and can use hierarchical mode in any contexts</li>
	 * <li>Context-Manager is a context-specific role and can use hierarchical mode only in contexts where they have this role.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and read permissions for the specific context where they want to read Facets.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Read a ContactFacet by type and UUID
	 * UUID facetUuid = UUID.fromString("...");
	 * String contactJson = publisher.readFacet("ContactFacet", facetUuid);
	 * </pre>
	 * 
	 * @param facetType The exact type name of the Facet to read
	 * @param uuid The unique identifier of the Facet to read
	 * @return JSON representation of the complete Facet instance from the Resource Registry
	 * @throws FacetNotFoundException If no Facet is found with the specified UUID and type (HTTP 404)
	 * @throws FacetAvailableInAnotherContextException If the Facet exists but is not accessible in the current context (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during Facet retrieval
	 */
	public String readFacet(String facetType, UUID uuid) 
			throws FacetNotFoundException, FacetAvailableInAnotherContextException, ResourceRegistryException;

	/**
	 * Updates an existing Facet instance in the Resource Registry.
	 * 
	 * <strong>REST Endpoint:</strong> {@code PUT /instances/{facet-type-name}/{facet-uuid}}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>Basic update: {@code ContactFacet updatedFacet = publisher.updateFacet(facetToUpdate);}</li>
	 * <li>With metadata in response: {@code publisher.includeMeta(true); ContactFacet updatedFacet = publisher.updateFacet(facetToUpdate);}</li>
	 * <li>Full metadata context: {@code publisher.includeMeta(true); publisher.allMeta(true); ContactFacet updatedFacet = publisher.updateFacet(facetToUpdate);}</li>
	 * <li>With context information: {@code publisher.includeContexts(true); ContactFacet updatedFacet = publisher.updateFacet(facetToUpdate);}.</li>
	 * </ul>
	 * 
	 * <strong>Update Behavior:</strong>
	 * <ul>
	 * <li>Updates an existing Facet with new property values from the provided instance</li>
	 * <li>The Facet must already exist in the Resource Registry</li>
	 * <li>All modifiable properties are updated; metadata fields are automatically managed</li>
	 * <li>The Facet UUID cannot be changed during update operations</li>
	 * <li>Associated Resources are automatically revalidated after Facet update</li>
	 * <li>The update affects all Resources that include this Facet via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response Facet</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>Valid authorization token required for accessing the current context</li>
	 * <li>Update permissions required for Facets in the current context</li>
	 * <li>The current context is derived from the authorization token provided during client creation.</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Facet successfully updated</li>
	 * <li><strong>400 Bad Request</strong>: Invalid Facet data or schema validation failed</li>
	 * <li><strong>404 Not Found</strong>: The Facet does not exist</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to update the Facet.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns the updated Facet object with its properties</li>
	 * <li>Response structure depends on includeMeta, allMeta, and includeContexts settings</li>
	 * <li>All properties are updated according to the current configuration.</li>
	 * </ul>
	 * 
	 * @param <F> The specific type of Facet to update
	 * @param facet The Facet instance with updated values (must contain valid UUID)
	 * @return The updated Facet object with current data from the Resource Registry
	 * @throws SchemaViolationException If the updated Facet violates type schema constraints
	 * @throws FacetNotFoundException If the Facet to update does not exist
	 * @throws ResourceRegistryException If an error occurs during Facet update
	 */
	public <F extends Facet> F updateFacet(F facet)
			throws SchemaViolationException, FacetNotFoundException, ResourceRegistryException;

	/**
	 * Updates an existing Facet instance in the Resource Registry using JSON representation.
	 * 
	 * <strong>REST Endpoint:</strong> {@code PUT /instances/{facet-type-name}/{facet-uuid}}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>Basic update: {@code String updatedFacet = publisher.updateFacet(facetJsonString);}</li>
	 * <li>With metadata in response: {@code publisher.includeMeta(true); String updatedFacet = publisher.updateFacet(facetJsonString);}</li>
	 * <li>Full metadata context: {@code publisher.includeMeta(true); publisher.allMeta(true); String updatedFacet = publisher.updateFacet(facetJsonString);}</li>
	 * <li>With context information: {@code publisher.includeContexts(true); String updatedFacet = publisher.updateFacet(facetJsonString);}.</li>
	 * </ul>
	 * 
	 * <strong>Update Behavior:</strong>
	 * <ul>
	 * <li>Updates an existing Facet using JSON representation</li>
	 * <li>The JSON must contain the UUID of the Facet to update</li>
	 * <li>The Facet type is automatically detected from the JSON content</li>
	 * <li>The Facet must already exist in the Resource Registry</li>
	 * <li>All modifiable properties are updated; metadata fields are automatically managed</li>
	 * <li>Associated Resources are automatically revalidated after Facet update</li>
	 * <li>The update affects all Resources that include this Facet via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>JSON Format Requirements:</strong>
	 * <ul>
	 * <li>Valid JSON representation of a Facet instance</li>
	 * <li>Must include the "id" field with the UUID of the Facet to update</li>
	 * <li>Must include the "@class" field to identify the Facet type</li>
	 * <li>Must conform to the schema of the specified Facet type.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response Facet</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>Valid authorization token required for accessing the current context</li>
	 * <li>Update permissions required for Facets in the current context</li>
	 * <li>The current context is derived from the authorization token provided during client creation.</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Facet successfully updated</li>
	 * <li><strong>400 Bad Request</strong>: Invalid JSON format, missing required fields, or schema validation failed</li>
	 * <li><strong>404 Not Found</strong>: The Facet does not exist</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to update the Facet.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON string containing the updated Facet</li>
	 * <li>Response structure depends on includeMeta, allMeta, and includeContexts settings</li>
	 * <li>All properties are updated according to the current configuration.</li>
	 * </ul>
	 * 
	 * @param facet JSON representation of the Facet to update
	 * @return JSON string containing the updated Facet with current data from the Resource Registry
	 * @throws SchemaViolationException If the updated Facet violates type schema constraints
	 * @throws FacetNotFoundException If the Facet to update does not exist
	 * @throws ResourceRegistryException If an error occurs during Facet update
	 */
	public String updateFacet(String facet)
			throws SchemaViolationException, FacetNotFoundException, ResourceRegistryException;

	/**
	 * Deletes a Facet instance from the Resource Registry.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code DELETE /instances/{facet-type-name}/{uuid}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Permanently deletes the specified Facet instance from the Resource Registry</li>
	 * <li>The type and UUID are automatically extracted from the provided Facet object</li>
	 * <li>The Facet must exist and be accessible in the current context</li>
	 * <li>This is a destructive operation that cannot be undone</li>
	 * <li>Parent Resources that reference this Facet may also be affected based on propagation constraints defined in the schema.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Facet successfully deleted</li>
	 * <li><strong>404 Not Found</strong>: Facet does not exist or is not accessible in the current context</li>
	 * <li><strong>403 Forbidden</strong>: User lacks permissions to delete the Facet</li>
	 * <li><strong>409 Conflict</strong>: Facet cannot be deleted due to existing relationships or schema constraints</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The type name is automatically derived from the provided Facet object using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>The UUID is extracted from the Facet's header</li>
	 * <li>Supports all Facet types and their specializations (e.g., StateFacet, ContactFacet, AccessPointFacet).</li>
	 * </ul>
	 * 
	 * <strong>Deletion Propagation Rules:</strong>
	 * <ul>
	 * <li><strong>Facets</strong>: When a Facet is deleted, parent Resources that reference it via ConsistsOf relations may also be affected</li>
	 * <li><strong>Parent Resources</strong>: If the deleted Facet was required for a Resource's schema compliance, the Resource may become invalid or be automatically removed</li>
	 * <li><strong>ConsistsOf Relations</strong>: All ConsistsOf relations connecting Resources to this Facet are automatically removed</li>
	 * <li><strong>Cascade Effects</strong>: The actual propagation behavior depends on the schema-defined constraints for each Resource type.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation and Constraints:</strong>
	 * <ul>
	 * <li>Before deletion, the system validates that removing the Facet will not violate schema constraints</li>
	 * <li>If deletion would leave required relationships unsatisfied or violate cardinality constraints, a {@link SchemaViolationException} is thrown</li>
	 * <li>This results in an HTTP 409 Conflict response with details about the constraint violation</li>
	 * <li><strong>Example scenarios that prevent deletion:</strong></li>
	 * </ul>
	 *   • Deleting a Facet that is required by an existing Resource;
	 *   • Deleting Facets that violate cardinality constraints (min occurrences) defined in the schema;
	 *   • Operations that would make parent Resources incomplete according to their schema definition.
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and delete permissions for the specific context where the Facet exists.</li>
	 * </ul>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can delete any Facet in any context</li>
	 * <li>Full administrative privileges across all contexts.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can delete any Facet in any context</li>
	 * <li>Full administrative privileges across all contexts.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can delete Facets only in contexts where they have Context-Manager role</li>
	 * <li>Operations are limited to their assigned contexts.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can delete Facets only in contexts where they have appropriate permissions</li>
	 * <li>Operations are limited to Facets they have access to in their authorized contexts.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Delete a ContactFacet instance
	 * ContactFacet contact = publisher.readFacet(ContactFacet.class, contactUUID);
	 * boolean deleted = publisher.deleteFacet(contact);
	 * 
	 * // Delete a StateFacet instance
	 * StateFacet state = publisher.readFacet(StateFacet.class, stateUUID);
	 * boolean stateDeleted = publisher.deleteFacet(state);
	 * </pre>
	 * 
	 * @param <F> The specific type of Facet to delete
	 * @param facet The Facet instance to delete (must contain a valid UUID in the header)
	 * @return true if the Facet was successfully deleted, false otherwise
	 * @throws SchemaViolationException If deleting the Facet would violate Information System model constraints (HTTP 409)
	 * @throws FacetNotFoundException If the Facet does not exist or is not accessible (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during the delete operation
	 */
	public <F extends Facet> boolean deleteFacet(F facet)
			throws SchemaViolationException, FacetNotFoundException, ResourceRegistryException;

	/**
	 * Deletes a Facet instance from the Resource Registry using type name and UUID.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code DELETE /instances/{facet-type-name}/{uuid}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Permanently deletes the specified Facet instance from the Resource Registry</li>
	 * <li>The Facet is identified by its type name and UUID</li>
	 * <li>The Facet must exist and be accessible in the current context</li>
	 * <li>This is a destructive operation that cannot be undone</li>
	 * <li>Parent Resources that reference this Facet may also be affected based on propagation constraints defined in the schema.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Facet successfully deleted</li>
	 * <li><strong>404 Not Found</strong>: Facet does not exist or is not accessible in the current context</li>
	 * <li><strong>403 Forbidden</strong>: User lacks permissions to delete the Facet</li>
	 * <li><strong>409 Conflict</strong>: Facet cannot be deleted due to existing relationships or schema constraints</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The facetType parameter must be a valid Facet type name registered in the Information System</li>
	 * <li>Type names are case-sensitive and must match exactly</li>
	 * <li>Supports all Facet types and their specializations (e.g., StateFacet, ContactFacet, AccessPointFacet, SoftwareFacet)</li>
	 * <li>The UUID must be a valid UUID that identifies an existing Facet instance.</li>
	 * </ul>
	 * 
	 * <strong>Deletion Propagation Rules:</strong>
	 * <ul>
	 * <li><strong>Facets</strong>: When a Facet is deleted, parent Resources that reference it via ConsistsOf relations may also be affected</li>
	 * <li><strong>Parent Resources</strong>: If the deleted Facet was required for a Resource's schema compliance, the Resource may become invalid or be automatically removed</li>
	 * <li><strong>ConsistsOf Relations</strong>: All ConsistsOf relations connecting Resources to this Facet are automatically removed</li>
	 * <li><strong>Cascade Effects</strong>: The actual propagation behavior depends on the schema-defined constraints for each Resource type.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation and Constraints:</strong>
	 * <ul>
	 * <li>Before deletion, the system validates that removing the Facet will not violate schema constraints</li>
	 * <li>If deletion would leave required relationships unsatisfied or violate cardinality constraints, a {@link SchemaViolationException} is thrown</li>
	 * <li>This results in an HTTP 409 Conflict response with details about the constraint violation</li>
	 * <li><strong>Example scenarios that prevent deletion:</strong></li>
	 * </ul>
	 *   • Deleting a Facet that is required by an existing Resource;
	 *   • Deleting Facets that violate cardinality constraints (min occurrences) defined in the schema;
	 *   • Operations that would make parent Resources incomplete according to their schema definition.
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and delete permissions for the specific context where the Facet exists.</li>
	 * </ul>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can delete any Facet in any context</li>
	 * <li>Full administrative privileges across all contexts.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can delete any Facet in any context</li>
	 * <li>Full administrative privileges across all contexts.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can delete Facets only in contexts where they have Context-Manager role</li>
	 * <li>Operations are limited to their assigned contexts.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can delete Facets only in contexts where they have appropriate permissions</li>
	 * <li>Operations are limited to Facets they have access to in their authorized contexts.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Delete a ContactFacet by type and UUID
	 * UUID contactUUID = UUID.fromString("12345678-1234-1234-1234-123456789abc");
	 * boolean deleted = publisher.deleteFacet("ContactFacet", contactUUID);
	 * 
	 * // Delete a StateFacet by type and UUID
	 * UUID stateUUID = UUID.fromString("87654321-4321-4321-4321-cba987654321");
	 * boolean stateDeleted = publisher.deleteFacet("StateFacet", stateUUID);
	 * </pre>
	 * 
	 * @param facetType The exact type name of the Facet to delete
	 * @param uuid The UUID of the Facet instance to delete
	 * @return true if the Facet was successfully deleted, false otherwise
	 * @throws SchemaViolationException If deleting the Facet would violate Information System model constraints (HTTP 409)
	 * @throws FacetNotFoundException If the Facet does not exist or is not accessible (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during the delete operation
	 */
	public boolean deleteFacet(String facetType, UUID uuid)
			throws SchemaViolationException, FacetNotFoundException, ResourceRegistryException;

	/**
	 * Creates a new Resource instance in the Resource Registry.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /instances/{resource-type-name}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Creates a new Resource entity in the Information System</li>
	 * <li>Resources represent the main entities that can have Facets attached via ConsistsOf relations and can be connected to other Resources via IsRelatedTo relations</li>
	 * <li>The instance is automatically associated with the current context</li>
	 * <li>Validates the resource against the Information System model schema</li>
	 * <li>Generates a new UUID for the resource if not already set</li>
	 * <li>Returns the created resource with server-generated metadata.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>201 Created</strong>: Resource successfully created</li>
	 * <li><strong>400 Bad Request</strong>: Invalid resource data or schema validation failure</li>
	 * <li><strong>409 Conflict</strong>: Resource with the same UUID already exists</li>
	 * <li><strong>403 Forbidden</strong>: User lacks write permissions in the current context</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The resource type name is automatically derived from the provided object using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>Supports all Resource subtypes defined in the Information System model.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation:</strong>
	 * <ul>
	 * <li>All resource properties are validated against the Information System model</li>
	 * <li>Required properties must be present and valid</li>
	 * <li>Property types and constraints are enforced</li>
	 * <li>Resource-specific validation rules are applied.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The instance is automatically associated with the current context.</li>
	 * </ul>
	 *
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and write permissions for the specific context where they want to create resources.</li>
	 * </ul>
	 *
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Create a new EService resource
	 * EService service = new EService();
	 * ...
	 * EService createdService = publisher.createResource(service);
	 * 
	 * // Create a new HostingNode resource
	 * HostingNode node = new HostingNode();
	 * ...
	 * HostingNode createdNode = publisher.createResource(node);
	 * </pre>
	 * 
	 * @param <R> The specific type of Resource to create
	 * @param resource The Resource instance to create in the Resource Registry
	 * @return The created Resource with server-generated metadata and UUID
	 * @throws SchemaViolationException If the resource violates Information System model constraints (HTTP 400)
	 * @throws ResourceAlreadyPresentException If a resource with the same UUID already exists (HTTP 409)
	 * @throws ResourceRegistryException If an error occurs during resource creation
	 */
	public <R extends Resource> R createResource(R resource)
			throws SchemaViolationException, ResourceAlreadyPresentException, ResourceRegistryException;

	/**
	 * Creates a new Resource instance from JSON representation.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /instances/{resource-type-name}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Creates a new Resource entity from JSON representation</li>
	 * <li>Resources represent the main entities that can have Facets attached via ConsistsOf relations and can be connected to other Resources via IsRelatedTo relations</li>
	 * <li>The instance is automatically associated with the current context</li>
	 * <li>Validates the JSON against the Information System model schema</li>
	 * <li>Type is determined from the JSON @class property</li>
	 * <li>Generates a new UUID for the resource if not already set</li>
	 * <li>Returns the created resource as JSON with server-generated metadata.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>201 Created</strong>: Resource successfully created</li>
	 * <li><strong>400 Bad Request</strong>: Invalid JSON data or schema validation failure</li>
	 * <li><strong>409 Conflict</strong>: Resource with the same UUID already exists</li>
	 * <li><strong>403 Forbidden</strong>: User lacks write permissions in the current context</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>JSON Format Requirements:</strong>
	 * <ul>
	 * <li>Must contain a valid @class property specifying a Resource type</li>
	 * <li>Must conform to the Information System model JSON schema for Resources</li>
	 * <li>All required properties for the specified resource type must be present</li>
	 * <li>Resource-specific properties must be properly formatted.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation:</strong>
	 * <ul>
	 * <li>JSON structure is validated against the Information System model</li>
	 * <li>Required resource properties must be present and valid</li>
	 * <li>Property types and constraints are enforced</li>
	 * <li>Resource-specific validation rules are applied.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The instance is automatically associated with the current context.</li>
	 * </ul>
	 *
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and write permissions for the specific context where they want to create resources.</li>
	 * </ul>
	 *
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Create a new EService from JSON
	 * String serviceJson = "...";
	 * String createdServiceJson = publisher.createResource(serviceJson);
	 * 
	 * // Create a new HostingNode from JSON
	 * String nodeJson = "...";
	 * String createdNodeJson = publisher.createResource(nodeJson);
	 * </pre>
	 * 
	 * @param resource JSON representation of the Resource to create
	 * @return JSON representation of the created Resource with server-generated metadata and UUID
	 * @throws SchemaViolationException If the JSON violates Information System model constraints (HTTP 400)
	 * @throws ResourceAlreadyPresentException If a resource with the same UUID already exists (HTTP 409)
	 * @throws ResourceRegistryException If an error occurs during resource creation or JSON parsing
	 */
	public String createResource(String resource)
			throws SchemaViolationException, ResourceAlreadyPresentException, ResourceRegistryException;

	/**
	 * Reads a Resource instance from the Resource Registry using a provided instance object.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /instances/{resource-type-name}/{uuid}[?includeMeta={true|false}&allMeta={true|false}&includeContexts={true|false}&hierarchical={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves an existing Resource by extracting type and UUID from the provided instance</li>
	 * <li>The Resource must be accessible in the current context or in child contexts (based on hierarchical mode)</li>
	 * <li>Returns the complete Resource instance with all properties, ConsistsOf relations, and metadata</li>
	 * <li>The type is automatically determined from the provided instance object.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Resource successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: No Resource found with the specified UUID or not accessible in current context</li>
	 * <li><strong>403 Forbidden</strong>: User lacks read permissions for the Resource or its context</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response instance</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration, admin-only):
	 * <ul>
	 * <li>Whether to include instances from child contexts of the current context</li>
	 * <li>Default value: false (child contexts not included)</li>
	 * <li>Usage: {@code publisher.setHierarchicalMode(true)}</li>
	 * <li>Values: true (includes child contexts) | false (current context only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#HIERARCHICAL_MODE_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The Resource type name is automatically derived from the provided instance using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>The UUID is extracted from the provided instance's header</li>
	 * <li>The returned object will be of the same Resource type as the provided instance.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The Resource must be accessible in the current context or child contexts (depending on hierarchical mode)</li>
	 * <li>IS-Manager and Infrastructure-Manager are global roles (available in all contexts) and can use hierarchical mode in any contexts</li>
	 * <li>Context-Manager is a context-specific role and can use hierarchical mode only in contexts where they have this role.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and read permissions for the specific context where they want to read resources.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Read an EService Resource
	 * EService eService = new EService();
	 * eService.setHeader(new Header(uuid));
	 * EService readEService = publisher.readResource(eService);
	 * </pre>
	 * 
	 * @param <R> The specific type of Resource to read
	 * @param resource Instance object containing at least the UUID in the header
	 * @return The complete Resource instance from the Resource Registry
	 * @throws ResourceNotFoundException If no Resource is found with the specified UUID (HTTP 404)
	 * @throws ResourceAvailableInAnotherContextException If the Resource exists but is not accessible in the current context (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during Resource retrieval
	 */
	public <R extends Resource> R readResource(R resource) 
			throws ResourceNotFoundException, ResourceAvailableInAnotherContextException, ResourceRegistryException;

	/**
	 * Reads a Resource instance from the Resource Registry by its type and UUID.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /instances/{resource-type-name}/{uuid}[?includeMeta={true|false}&allMeta={true|false}&includeContexts={true|false}&hierarchical={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves an existing Resource by its type name and unique identifier</li>
	 * <li>The Resource must be accessible in the current context or in child contexts (based on hierarchical mode)</li>
	 * <li>Returns the complete Resource instance as JSON with all properties, ConsistsOf relations, and metadata</li>
	 * <li>Type validation ensures the UUID corresponds to a Resource of the specified type.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Resource successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: No Resource found with the specified UUID and type, or not accessible in current context</li>
	 * <li><strong>403 Forbidden</strong>: User lacks read permissions for the Resource or its context</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response instance</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration, admin-only):
	 * <ul>
	 * <li>Whether to include instances from child contexts of the current context</li>
	 * <li>Default value: false (child contexts not included)</li>
	 * <li>Usage: {@code publisher.setHierarchicalMode(true)}</li>
	 * <li>Values: true (includes child contexts) | false (current context only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#HIERARCHICAL_MODE_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The specified type must be a valid Resource type in the Information System model</li>
	 * <li>Supports Resource subtypes defined in the Information System model</li>
	 * <li>Type names are case-sensitive and must match exactly the class names defined in the Information System model</li>
	 * <li>UUID must correspond to a Resource of exactly the specified type (no polymorphic matching).</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The Resource must be accessible in the current context or child contexts (depending on hierarchical mode)</li>
	 * <li>IS-Manager and Infrastructure-Manager are global roles (available in all contexts) and can use hierarchical mode in any contexts</li>
	 * <li>Context-Manager is a context-specific role and can use hierarchical mode only in contexts where they have this role.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and read permissions for the specific context where they want to read resources.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Read an EService Resource by type and UUID
	 * UUID resourceUuid = UUID.fromString("...");
	 * String serviceJson = publisher.readResource("EService", resourceUuid);
	 * </pre>
	 * 
	 * @param resourceType The exact type name of the Resource to read
	 * @param uuid The unique identifier of the Resource to read
	 * @return JSON representation of the complete Resource instance from the Resource Registry
	 * @throws ResourceNotFoundException If no Resource is found with the specified UUID and type (HTTP 404)
	 * @throws ResourceAvailableInAnotherContextException If the Resource exists but is not accessible in the current context (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during Resource retrieval
	 */
	public String readResource(String resourceType, UUID uuid)
			throws ResourceNotFoundException, ResourceAvailableInAnotherContextException, ResourceRegistryException;

	/**
	 * Updates an existing Resource instance in the Resource Registry.
	 * 
	 * <strong>REST Endpoint:</strong> {@code PUT /instances/{resource-type-name}/{resource-uuid}}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>Basic update: {@code EService updatedResource = publisher.updateResource(resourceToUpdate);}</li>
	 * <li>With metadata in response: {@code publisher.includeMeta(true); EService updatedResource = publisher.updateResource(resourceToUpdate);}</li>
	 * <li>Full metadata context: {@code publisher.includeMeta(true); publisher.allMeta(true); EService updatedResource = publisher.updateResource(resourceToUpdate);}</li>
	 * <li>With context information: {@code publisher.includeContexts(true); EService updatedResource = publisher.updateResource(resourceToUpdate);}.</li>
	 * </ul>
	 * 
	 * <strong>Update Behavior:</strong>
	 * <ul>
	 * <li>Updates an existing Resource with new property values from the provided instance</li>
	 * <li>The Resource must already exist in the Resource Registry</li>
	 * <li>All modifiable properties are updated; metadata fields are automatically managed</li>
	 * <li>The Resource UUID cannot be changed during update operations</li>
	 * <li>Associated Facets via ConsistsOf relations are preserved unless explicitly modified</li>
	 * <li>Existing ConsistsOf relations and their target Facets remain unchanged unless modified through separate operations.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response Resource</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>Valid authorization token required for accessing the current context</li>
	 * <li>Update permissions required for Resources in the current context</li>
	 * <li>The current context is derived from the authorization token provided during client creation.</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Resource successfully updated</li>
	 * <li><strong>400 Bad Request</strong>: Invalid Resource data or schema validation failed</li>
	 * <li><strong>404 Not Found</strong>: The Resource does not exist</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to update the Resource.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns the updated Resource object with its properties and relations</li>
	 * <li>Response structure depends on includeMeta, allMeta, and includeContexts settings</li>
	 * <li>All nested elements are included according to the current configuration.</li>
	 * </ul>
	 * 
	 * @param <R> The specific type of Resource to update
	 * @param resource The Resource instance with updated values (must contain valid UUID)
	 * @return The updated Resource object with current data from the Resource Registry
	 * @throws SchemaViolationException If the updated Resource violates type schema constraints
	 * @throws ResourceNotFoundException If the Resource to update does not exist
	 * @throws ResourceRegistryException If an error occurs during Resource update
	 */
	public <R extends Resource> R updateResource(R resource)
			throws SchemaViolationException, ResourceNotFoundException, ResourceRegistryException;

	/**
	 * Updates an existing Resource instance in the Resource Registry using JSON representation.
	 * 
	 * <strong>REST Endpoint:</strong> {@code PUT /instances/{resource-type-name}/{resource-uuid}}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>Basic update: {@code String updatedResource = publisher.updateResource(resourceJsonString);}</li>
	 * <li>With metadata in response: {@code publisher.includeMeta(true); String updatedResource = publisher.updateResource(resourceJsonString);}</li>
	 * <li>Full metadata context: {@code publisher.includeMeta(true); publisher.allMeta(true); String updatedResource = publisher.updateResource(resourceJsonString);}</li>
	 * <li>With context information: {@code publisher.includeContexts(true); String updatedResource = publisher.updateResource(resourceJsonString);}.</li>
	 * </ul>
	 * 
	 * <strong>Update Behavior:</strong>
	 * <ul>
	 * <li>Updates an existing Resource using JSON representation</li>
	 * <li>The JSON must contain the UUID of the Resource to update</li>
	 * <li>The Resource type is automatically detected from the JSON content</li>
	 * <li>The Resource must already exist in the Resource Registry</li>
	 * <li>All modifiable properties are updated; metadata fields are automatically managed</li>
	 * <li>Associated Facets via ConsistsOf relations are preserved unless explicitly modified</li>
	 * <li>Existing ConsistsOf relations and their target Facets remain unchanged unless modified through separate operations.</li>
	 * </ul>
	 * 
	 * <strong>JSON Format Requirements:</strong>
	 * <ul>
	 * <li>Valid JSON representation of a Resource instance</li>
	 * <li>Must include the "id" field with the UUID of the Resource to update</li>
	 * <li>Must include the "@class" field to identify the Resource type</li>
	 * <li>Must conform to the schema of the specified Resource type</li>
	 * <li>ConsistsOf relations should be included if they need to be updated.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response Resource</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>Valid authorization token required for accessing the current context</li>
	 * <li>Update permissions required for Resources in the current context</li>
	 * <li>The current context is derived from the authorization token provided during client creation.</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Resource successfully updated</li>
	 * <li><strong>400 Bad Request</strong>: Invalid JSON format, missing required fields, or schema validation failed</li>
	 * <li><strong>404 Not Found</strong>: The Resource does not exist</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to update the Resource.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON string containing the updated Resource</li>
	 * <li>Response structure depends on includeMeta, allMeta, and includeContexts settings</li>
	 * <li>All nested elements are included according to the current configuration.</li>
	 * </ul>
	 * 
	 * @param resource JSON representation of the Resource to update
	 * @return JSON string containing the updated Resource with current data from the Resource Registry
	 * @throws SchemaViolationException If the updated Resource violates type schema constraints
	 * @throws ResourceNotFoundException If the Resource to update does not exist
	 * @throws ResourceRegistryException If an error occurs during Resource update
	 */
	public String updateResource(String resource)
			throws SchemaViolationException, ResourceNotFoundException, ResourceRegistryException;

	/**
	 * Deletes a Resource instance from the Resource Registry.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code DELETE /instances/{resource-type-name}/{uuid}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Permanently deletes the specified Resource instance from the Resource Registry</li>
	 * <li>The type and UUID are automatically extracted from the provided Resource object</li>
	 * <li>The Resource must exist and be accessible in the current context</li>
	 * <li>This is a destructive operation that cannot be undone</li>
	 * <li>Associated Facets and ConsistsOf relations may also be deleted based on propagation constraints defined in the schema.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Resource successfully deleted</li>
	 * <li><strong>404 Not Found</strong>: Resource does not exist or is not accessible in the current context</li>
	 * <li><strong>403 Forbidden</strong>: User lacks permissions to delete the Resource</li>
	 * <li><strong>409 Conflict</strong>: Resource cannot be deleted due to existing relationships or schema constraints</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The type name is automatically derived from the provided Resource object using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>The UUID is extracted from the Resource's header</li>
	 * <li>Supports all Resource types and their specializations (e.g., EService, HostingNode, Software, Dataset).</li>
	 * </ul>
	 * 
	 * <strong>Deletion Propagation Rules:</strong>
	 * <ul>
	 * <li><strong>Resources</strong>: When a Resource is deleted, all its associated Facets (via ConsistsOf relations) may also be deleted if they become orphaned and the delete propagation constraint is {@code propagate}</li>
	 * <li><strong>Associated Facets</strong>: Facets connected to the Resource via ConsistsOf relations are automatically removed if no other Resources reference them</li>
	 * <li><strong>ConsistsOf Relations</strong>: All ConsistsOf relations connecting the Resource to its Facets are automatically removed</li>
	 * <li><strong>IsRelatedTo Relations</strong>: IsRelatedTo relations where this Resource is source or target are automatically removed</li>
	 * <li><strong>Cascade Effects</strong>: The actual propagation behavior depends on the schema-defined constraints for each Resource type.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation and Constraints:</strong>
	 * <ul>
	 * <li>Before deletion, the system validates that removing the Resource will not violate schema constraints</li>
	 * <li>If deletion would leave required relationships unsatisfied or violate cardinality constraints, a {@link SchemaViolationException} is thrown</li>
	 * <li>This results in an HTTP 409 Conflict response with details about the constraint violation</li>
	 * <li><strong>Example scenarios that prevent deletion:</strong></li>
	 * </ul>
	 *   • Deleting a Resource that is required by other Resources via IsRelatedTo relations;
	 *   • Deleting Resources that violate cardinality constraints (min occurrences) defined in the schema;
	 *   • Operations that would make related Resources incomplete according to their schema definition.
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and delete permissions for the specific context where the Resource exists.</li>
	 * </ul>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can delete any Resource in any context</li>
	 * <li>Full administrative privileges across all contexts.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can delete any Resource in any context</li>
	 * <li>Full administrative privileges across all contexts.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can delete Resources only in contexts where they have Context-Manager role</li>
	 * <li>Operations are limited to their assigned contexts.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can delete Resources only in contexts where they have appropriate permissions</li>
	 * <li>Operations are limited to Resources they have access to in their authorized contexts.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Delete an EService instance
	 * EService service = publisher.readResource(EService.class, serviceUUID);
	 * boolean deleted = publisher.deleteResource(service);
	 * 
	 * // Delete a HostingNode instance
	 * HostingNode host = publisher.readResource(HostingNode.class, hostUUID);
	 * boolean hostDeleted = publisher.deleteResource(host);
	 * </pre>
	 * 
	 * @param <R> The specific type of Resource to delete
	 * @param resource The Resource instance to delete (must contain a valid UUID in the header)
	 * @return true if the Resource was successfully deleted, false otherwise
	 * @throws ResourceNotFoundException If the Resource does not exist or is not accessible (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during the delete operation
	 */
	public <R extends Resource> boolean deleteResource(R resource)
			throws ResourceNotFoundException, ResourceRegistryException;

	/**
	 * Deletes a Resource instance from the Resource Registry using type name and UUID.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code DELETE /instances/{resource-type-name}/{uuid}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Permanently deletes the specified Resource instance from the Resource Registry</li>
	 * <li>The Resource is identified by its type name and UUID</li>
	 * <li>The Resource must exist and be accessible in the current context</li>
	 * <li>This is a destructive operation that cannot be undone</li>
	 * <li>Associated Facets and ConsistsOf relations may also be deleted based on propagation constraints defined in the schema.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Resource successfully deleted</li>
	 * <li><strong>404 Not Found</strong>: Resource does not exist or is not accessible in the current context</li>
	 * <li><strong>403 Forbidden</strong>: User lacks permissions to delete the Resource</li>
	 * <li><strong>409 Conflict</strong>: Resource cannot be deleted due to existing relationships or schema constraints</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The resourceType parameter must be a valid Resource type name registered in the Information System</li>
	 * <li>Type names are case-sensitive and must match exactly</li>
	 * <li>Supports all Resource types and their specializations (e.g., EService, HostingNode, Software, Dataset, VirtualMachine)</li>
	 * <li>The UUID must be a valid UUID that identifies an existing Resource instance.</li>
	 * </ul>
	 * 
	 * <strong>Deletion Propagation Rules:</strong>
	 * <ul>
	 * <li><strong>Resources</strong>: When a Resource is deleted, all its associated Facets (via ConsistsOf relations) may also be deleted if they become orphaned and the delete propagation constraint is {@code propagate}</li>
	 * <li><strong>Associated Facets</strong>: Facets connected to the Resource via ConsistsOf relations are automatically removed if no other Resources reference them</li>
	 * <li><strong>ConsistsOf Relations</strong>: All ConsistsOf relations connecting the Resource to its Facets are automatically removed</li>
	 * <li><strong>IsRelatedTo Relations</strong>: IsRelatedTo relations where this Resource is source or target are automatically removed</li>
	 * <li><strong>Cascade Effects</strong>: The actual propagation behavior depends on the schema-defined constraints for each Resource type.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation and Constraints:</strong>
	 * <ul>
	 * <li>Before deletion, the system validates that removing the Resource will not violate schema constraints</li>
	 * <li>If deletion would leave required relationships unsatisfied or violate cardinality constraints, a {@link SchemaViolationException} is thrown</li>
	 * <li>This results in an HTTP 409 Conflict response with details about the constraint violation</li>
	 * <li><strong>Example scenarios that prevent deletion:</strong></li>
	 * </ul>
	 *   • Deleting a Resource that is required by other Resources via IsRelatedTo relations;
	 *   • Deleting Resources that violate cardinality constraints (min occurrences) defined in the schema;
	 *   • Operations that would make related Resources incomplete according to their schema definition.
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and delete permissions for the specific context where the Resource exists.</li>
	 * </ul>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can delete any Resource in any context</li>
	 * <li>Full administrative privileges across all contexts.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can delete any Resource in any context</li>
	 * <li>Full administrative privileges across all contexts.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can delete Resources only in contexts where they have Context-Manager role</li>
	 * <li>Operations are limited to their assigned contexts.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can delete Resources only in contexts where they have appropriate permissions</li>
	 * <li>Operations are limited to Resources they have access to in their authorized contexts.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Delete an EService by type and UUID
	 * UUID serviceUUID = UUID.fromString("12345678-1234-1234-1234-123456789abc");
	 * boolean deleted = publisher.deleteResource("EService", serviceUUID);
	 * 
	 * // Delete a HostingNode by type and UUID
	 * UUID hostUUID = UUID.fromString("87654321-4321-4321-4321-cba987654321");
	 * boolean hostDeleted = publisher.deleteResource("HostingNode", hostUUID);
	 * </pre>
	 * 
	 * @param resourceType The exact type name of the Resource to delete
	 * @param uuid The UUID of the Resource instance to delete
	 * @return true if the Resource was successfully deleted, false otherwise
	 * @throws ResourceNotFoundException If the Resource does not exist or is not accessible (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during the delete operation
	 */
	public boolean deleteResource(String resourceType, UUID uuid)
			throws ResourceNotFoundException, ResourceRegistryException;

	/**
	 * Creates a new ConsistsOf relation instance in the Resource Registry.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /instances/{consistsof-type-name}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Creates a new ConsistsOf relation between a Resource and a Facet</li>
	 * <li>ConsistsOf relations represent the composition relationship where a Resource consists of one or more Facets</li>
	 * <li>The relation instance is automatically associated with the current context</li>
	 * <li>Validates the relation against the Information System model schema</li>
	 * <li>Generates a new UUID for the relation if not already set</li>
	 * <li>Returns the created relation with server-generated metadata.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>201 Created</strong>: ConsistsOf relation successfully created</li>
	 * <li><strong>400 Bad Request</strong>: Invalid relation data or schema validation failure</li>
	 * <li><strong>409 Conflict</strong>: ConsistsOf relation with the same UUID already exists</li>
	 * <li><strong>404 Not Found</strong>: Source Resource or target Facet does not exist</li>
	 * <li><strong>403 Forbidden</strong>: User lacks write permissions in the current context</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The relation type name is automatically derived from the provided object using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>Supports ConsistsOf types and subtypes defined in the Information System model</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation:</strong>
	 * <ul>
	 * <li>All relation properties are validated against the Information System model</li>
	 * <li>Source and target entities must exist and be accessible in compatible contexts</li>
	 * <li>Relation constraints and cardinality rules are enforced</li>
	 * <li>ConsistsOf-specific validation rules are applied.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The instance is automatically associated with the current context.</li>
	 * </ul>
	 *
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>User must have write permissions in the current context</li>
	 * <li>Both source Resource and target Facet must be available in the current context.</li>
	 * </ul>
	 *
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Create a ConsistsOf relation between an EService and ContactFacet
	 * IsIdentifiedBy&lt;EService, ContactFacet&gt; identifiedBy = new IsIdentifiedBy&lt;&gt;();
	 * identifiedBy.setSource(eservice);
	 * identifiedBy.setTarget(contact);
	 * ...
	 * IsIdentifiedBy&lt;EService, ContactFacet&gt; createdRelation = publisher.createConsistsOf(identifiedBy);
	 * </pre>
	 * 
	 * @param <C> The specific type of ConsistsOf relation to create
	 * @param consistsOf The ConsistsOf relation instance to create in the Resource Registry
	 * @return The created ConsistsOf relation with server-generated metadata and UUID
	 * @throws SchemaViolationException If the relation violates Information System model constraints (HTTP 400)
	 * @throws ConsistsOfAlreadyPresentException If a relation with the same UUID already exists (HTTP 409)
	 * @throws ResourceNotFoundException If the source Resource or target Facet does not exist (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during relation creation
	 */
	public <C extends ConsistsOf<? extends Resource, ? extends Facet>> C createConsistsOf(C consistsOf)
			throws SchemaViolationException, ConsistsOfAlreadyPresentException, ResourceNotFoundException, ResourceRegistryException;

	/**
	 * Creates a new ConsistsOf relation instance from JSON representation.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /instances/{consistsof-type-name}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Creates a new ConsistsOf relation between a Resource and a Facet from JSON representation</li>
	 * <li>ConsistsOf relations represent the composition relationship where a Resource consists of one or more Facets</li>
	 * <li>The relation instance is automatically associated with the current context</li>
	 * <li>Validates the JSON against the Information System model schema</li>
	 * <li>Type is determined from the JSON @class property</li>
	 * <li>Generates a new UUID for the relation if not already set</li>
	 * <li>Returns the created relation as JSON with server-generated metadata.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>201 Created</strong>: ConsistsOf relation successfully created</li>
	 * <li><strong>400 Bad Request</strong>: Invalid JSON data or schema validation failure</li>
	 * <li><strong>409 Conflict</strong>: ConsistsOf relation with the same UUID already exists</li>
	 * <li><strong>404 Not Found</strong>: Source Resource or target Facet does not exist</li>
	 * <li><strong>403 Forbidden</strong>: User lacks write permissions in the current context</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>JSON Format Requirements:</strong>
	 * <ul>
	 * <li>Must contain a valid @class property specifying a ConsistsOf type</li>
	 * <li>Must conform to the Information System model JSON schema for ConsistsOf relations</li>
	 * <li>Must include valid source and target references</li>
	 * <li>All required properties for the specified relation type must be present.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation:</strong>
	 * <ul>
	 * <li>JSON structure is validated against the Information System model</li>
	 * <li>Source and target entity references are validated</li>
	 * <li>Relation constraints and cardinality rules are enforced</li>
	 * <li>ConsistsOf-specific validation rules are applied.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The instance is automatically associated with the current context.</li>
	 * </ul>
	 *
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>User must have write permissions in the current context</li>
	 * <li>Both source Resource and target Facet must be available in the current context.</li>
	 * </ul>
	 *
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Create a ConsistsOf relation from JSON
	 * String consistsOfJson = "...";
	 * String createdRelationJson = publisher.createConsistsOf(consistsOfJson);
	 * </pre>
	 * 
	 * @param consistsOf JSON representation of the ConsistsOf relation to create
	 * @return JSON representation of the created ConsistsOf relation with server-generated metadata and UUID
	 * @throws SchemaViolationException If the JSON violates Information System model constraints (HTTP 400)
	 * @throws ConsistsOfAlreadyPresentException If a relation with the same UUID already exists (HTTP 409)
	 * @throws ResourceNotFoundException If the source Resource or target Facet does not exist (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during relation creation or JSON parsing
	 */
	public String createConsistsOf(String consistsOf)
			throws SchemaViolationException, ConsistsOfAlreadyPresentException, ResourceNotFoundException, ResourceRegistryException;

	/**
	 * Reads a ConsistsOf relation instance from the Resource Registry using a provided instance object.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /instances/{consistsof-type-name}/{uuid}[?includeMeta={true|false}&allMeta={true|false}&includeContexts={true|false}&hierarchical={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves an existing ConsistsOf relation by extracting type and UUID from the provided instance</li>
	 * <li>The relation must be accessible in the current context or in child contexts (based on hierarchical mode)</li>
	 * <li>Returns the complete ConsistsOf relation with source Resource, target Facet, and all properties</li>
	 * <li>The type is automatically determined from the provided instance object.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: ConsistsOf relation successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: No ConsistsOf relation found with the specified UUID or not accessible in current context</li>
	 * <li><strong>403 Forbidden</strong>: User lacks read permissions for the relation or its context</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response instance</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration, admin-only):
	 * <ul>
	 * <li>Whether to include instances from child contexts of the current context</li>
	 * <li>Default value: false (child contexts not included)</li>
	 * <li>Usage: {@code publisher.setHierarchicalMode(true)}</li>
	 * <li>Values: true (includes child contexts) | false (current context only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#HIERARCHICAL_MODE_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The ConsistsOf type name is automatically derived from the provided instance using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>The UUID is extracted from the provided instance's header</li>
	 * <li>The returned object will be of the same ConsistsOf relation type as the provided instance.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The ConsistsOf relation must be accessible in the current context or child contexts (depending on hierarchical mode)</li>
	 * <li>IS-Manager and Infrastructure-Manager are global roles (available in all contexts) and can use hierarchical mode in any contexts</li>
	 * <li>Context-Manager is a context-specific role and can use hierarchical mode only in contexts where they have this role.</li>
	 * </ul>
	 * 
	 * <strong>Relation Context:</strong>
	 * <ul>
	 * <li>ConsistsOf relations represent aggregation relationships between Resources and Facets</li>
	 * <li>The source of the relation is always a Resource, the target is always a Facet</li>
	 * <li>The relation defines how a Resource is composed of various Facets.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Read a ConsistsOf relation
	 * ConsistsOf&lt;EService, SoftwareFacet&gt; co = new ConsistsOf&lt;&gt;();
	 * co.setHeader(new Header(uuid));
	 * ConsistsOf&lt;EService, SoftwareFacet&gt; readCO = publisher.readConsistsOf(co);
	 * </pre>
	 * 
	 * @param <C> The specific type of ConsistsOf relation to read
	 * @param consistsOf Instance object containing at least the UUID in the header
	 * @return The complete ConsistsOf relation instance from the Resource Registry
	 * @throws ConsistsOfNotFoundException If no ConsistsOf relation is found with the specified UUID (HTTP 404)
	 * @throws ConsistsOfAvailableInAnotherContextException If the relation exists but is not accessible in the current context (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during relation retrieval
	 */
	public <C extends ConsistsOf<? extends Resource, ? extends Facet>> C readConsistsOf(C consistsOf)
			throws ConsistsOfNotFoundException, ConsistsOfAvailableInAnotherContextException, ResourceRegistryException;

	/**
	 * Reads a ConsistsOf relation instance from the Resource Registry by its type and UUID.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /instances/{consistsof-type-name}/{uuid}[?includeMeta={true|false}&allMeta={true|false}&includeContexts={true|false}&hierarchical={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves an existing ConsistsOf relation by its type name and unique identifier</li>
	 * <li>The relation must be accessible in the current context or in child contexts (based on hierarchical mode)</li>
	 * <li>Returns the complete ConsistsOf relation as JSON with source Resource, target Facet, and all properties</li>
	 * <li>Type validation ensures the UUID corresponds to a ConsistsOf relation of the specified type.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: ConsistsOf relation successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: No ConsistsOf relation found with the specified UUID and type, or not accessible in current context</li>
	 * <li><strong>403 Forbidden</strong>: User lacks read permissions for the relation or its context</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response instance</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration, admin-only):
	 * <ul>
	 * <li>Whether to include instances from child contexts of the current context</li>
	 * <li>Default value: false (child contexts not included)</li>
	 * <li>Usage: {@code publisher.setHierarchicalMode(true)}</li>
	 * <li>Values: true (includes child contexts) | false (current context only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#HIERARCHICAL_MODE_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The specified type must be a valid ConsistsOf relation type in the Information System model</li>
	 * <li>Supports all ConsistsOf relation types and their subtypes as defined in the Information System model</li>
	 * <li>Type names are case-sensitive and must match exactly the class names defined in the Information System model</li>
	 * <li>UUID must correspond to a ConsistsOf relation of exactly the specified type (no polymorphic matching).</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The ConsistsOf relation must be accessible in the current context or child contexts (depending on hierarchical mode)</li>
	 * <li>IS-Manager and Infrastructure-Manager are global roles (available in all contexts) and can use hierarchical mode in any contexts</li>
	 * <li>Context-Manager is a context-specific role and can use hierarchical mode only in contexts where they have this role.</li>
	 * </ul>
	 * 
	 * <strong>Relation Context:</strong>
	 * <ul>
	 * <li>ConsistsOf relations represent aggregation relationships between Resources and Facets</li>
	 * <li>The JSON response includes source Resource reference, target Facet reference, and relation properties</li>
	 * <li>Relation cardinality and constraints are validated according to the Information System model.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Read a ConsistsOf relation by type and UUID
	 * UUID relationUuid = UUID.fromString("...");
	 * String relationJson = publisher.readConsistsOf("ConsistsOf", relationUuid);
	 * </pre>
	 * 
	 * @param consistsOfType The exact type name of the ConsistsOf relation to read
	 * @param uuid The unique identifier of the ConsistsOf relation to read
	 * @return JSON representation of the complete ConsistsOf relation instance from the Resource Registry
	 * @throws ConsistsOfNotFoundException If no ConsistsOf relation is found with the specified UUID and type (HTTP 404)
	 * @throws ConsistsOfAvailableInAnotherContextException If the relation exists but is not accessible in the current context (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during relation retrieval
	 */
	public String readConsistsOf(String consistsOfType, UUID uuid) 
			throws ConsistsOfNotFoundException, ConsistsOfAvailableInAnotherContextException, ResourceRegistryException;

	/**
	 * Updates an existing ConsistsOf relation instance in the Resource Registry.
	 * 
	 * <strong>REST Endpoint:</strong> {@code PUT /instances/{consistsof-type-name}/{consistsof-uuid}}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>Basic update: {@code ConsistsOf updatedRelation = publisher.updateConsistsOf(relationToUpdate);}</li>
	 * <li>With metadata in response: {@code publisher.includeMeta(true); ConsistsOf updatedRelation = publisher.updateConsistsOf(relationToUpdate);}</li>
	 * <li>Full metadata context: {@code publisher.includeMeta(true); publisher.allMeta(true); ConsistsOf updatedRelation = publisher.updateConsistsOf(relationToUpdate);}</li>
	 * <li>With context information: {@code publisher.includeContexts(true); ConsistsOf updatedRelation = publisher.updateConsistsOf(relationToUpdate);}.</li>
	 * </ul>
	 * 
	 * <strong>Update Behavior:</strong>
	 * <ul>
	 * <li>Updates an existing ConsistsOf relation with new property values from the provided instance</li>
	 * <li>The ConsistsOf relation must already exist in the Resource Registry</li>
	 * <li>All modifiable properties are updated; metadata fields are automatically managed</li>
	 * <li>The ConsistsOf UUID cannot be changed during update operations</li>
	 * <li>Source Resource and target Facet cannot be changed during update (only properties)</li>
	 * <li>The source Resource is automatically revalidated after ConsistsOf update to ensure schema compliance.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response ConsistsOf relation</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>Valid authorization token required for accessing the current context</li>
	 * <li>Update permissions required for ConsistsOf relations in the current context</li>
	 * <li>The current context is derived from the authorization token provided during client creation.</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: ConsistsOf relation successfully updated</li>
	 * <li><strong>400 Bad Request</strong>: Invalid relation data or schema validation failed</li>
	 * <li><strong>404 Not Found</strong>: The ConsistsOf relation does not exist</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to update the ConsistsOf relation.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns the updated ConsistsOf relation object with its properties</li>
	 * <li>Response structure depends on includeMeta, allMeta, and includeContexts settings</li>
	 * <li>All nested elements are included according to the current configuration.</li>
	 * </ul>
	 * 
	 * @param <C> The specific type of ConsistsOf relation to update
	 * @param consistsOf The ConsistsOf relation instance with updated values (must contain valid UUID)
	 * @return The updated ConsistsOf relation object with current data from the Resource Registry
	 * @throws SchemaViolationException If the updated ConsistsOf relation violates type schema constraints
	 * @throws ConsistsOfNotFoundException If the ConsistsOf relation to update does not exist
	 * @throws ResourceRegistryException If an error occurs during ConsistsOf relation update
	 */
	public <C extends ConsistsOf<? extends Resource, ? extends Facet>> C updateConsistsOf(C consistsOf)
			throws SchemaViolationException, ConsistsOfNotFoundException, ResourceRegistryException;

	/**
	 * Updates an existing ConsistsOf relation instance in the Resource Registry using JSON representation.
	 * 
	 * <strong>REST Endpoint:</strong> {@code PUT /instances/{consistsof-type-name}/{consistsof-uuid}}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>Basic update: {@code String updatedRelation = publisher.updateConsistsOf(relationJsonString);}</li>
	 * <li>With metadata in response: {@code publisher.includeMeta(true); String updatedRelation = publisher.updateConsistsOf(relationJsonString);}</li>
	 * <li>Full metadata context: {@code publisher.includeMeta(true); publisher.allMeta(true); String updatedRelation = publisher.updateConsistsOf(relationJsonString);}</li>
	 * <li>With context information: {@code publisher.includeContexts(true); String updatedRelation = publisher.updateConsistsOf(relationJsonString);}.</li>
	 * </ul>
	 * 
	 * <strong>Update Behavior:</strong>
	 * <ul>
	 * <li>Updates an existing ConsistsOf relation using JSON representation</li>
	 * <li>The JSON must contain the UUID of the ConsistsOf relation to update</li>
	 * <li>The ConsistsOf type is automatically detected from the JSON content</li>
	 * <li>The ConsistsOf relation must already exist in the Resource Registry</li>
	 * <li>All modifiable properties are updated; metadata fields are automatically managed</li>
	 * <li>Source Resource and target Facet cannot be changed during update (only properties)</li>
	 * <li>The source Resource is automatically revalidated after ConsistsOf update to ensure schema compliance.</li>
	 * </ul>
	 * 
	 * <strong>JSON Format Requirements:</strong>
	 * <ul>
	 * <li>Valid JSON representation of a ConsistsOf relation instance</li>
	 * <li>Must include the "id" field with the UUID of the ConsistsOf relation to update</li>
	 * <li>Must include the "@class" field to identify the ConsistsOf type</li>
	 * <li>Must conform to the schema of the specified ConsistsOf type</li>
	 * <li>Source and target references should remain unchanged.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response ConsistsOf relation</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>Valid authorization token required for accessing the current context</li>
	 * <li>Update permissions required for ConsistsOf relations in the current context</li>
	 * <li>The current context is derived from the authorization token provided during client creation.</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: ConsistsOf relation successfully updated</li>
	 * <li><strong>400 Bad Request</strong>: Invalid JSON format, missing required fields, or schema validation failed</li>
	 * <li><strong>404 Not Found</strong>: The ConsistsOf relation does not exist</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to update the ConsistsOf relation.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON string containing the updated ConsistsOf relation</li>
	 * <li>Response structure depends on includeMeta, allMeta, and includeContexts settings</li>
	 * <li>All nested elements are included according to the current configuration.</li>
	 * </ul>
	 * 
	 * @param consistsOf JSON representation of the ConsistsOf relation to update
	 * @return JSON string containing the updated ConsistsOf relation with current data from the Resource Registry
	 * @throws SchemaViolationException If the updated ConsistsOf relation violates type schema constraints
	 * @throws ConsistsOfNotFoundException If the ConsistsOf relation to update does not exist
	 * @throws ResourceRegistryException If an error occurs during ConsistsOf relation update
	 */
	public String updateConsistsOf(String consistsOf)
			throws SchemaViolationException, ConsistsOfNotFoundException, ResourceRegistryException;

	/**
	 * Deletes a ConsistsOf relation instance from the Resource Registry.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code DELETE /instances/{consistsof-type-name}/{uuid}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Permanently deletes the specified ConsistsOf relation instance from the Resource Registry</li>
	 * <li>The type and UUID are automatically extracted from the provided ConsistsOf object</li>
	 * <li>The ConsistsOf relation must exist and be accessible in the current context</li>
	 * <li>This operation removes the relationship between a Resource and a Facet</li>
	 * <li>This is a destructive operation that cannot be undone</li>
	 * <li>The connected Resource and Facet elements may be affected based on propagation constraints.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: ConsistsOf relation successfully deleted</li>
	 * <li><strong>404 Not Found</strong>: ConsistsOf relation does not exist or is not accessible in the current context</li>
	 * <li><strong>403 Forbidden</strong>: User lacks permissions to delete the ConsistsOf relation</li>
	 * <li><strong>409 Conflict</strong>: ConsistsOf relation cannot be deleted due to schema constraints</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The type name is automatically derived from the provided ConsistsOf object using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>The UUID is extracted from the ConsistsOf relation's header</li>
	 * <li>Supports all ConsistsOf relation types and their specializations.</li>
	 * </ul>
	 * 
	 * <strong>Deletion Effects and Propagation:</strong>
	 * <ul>
	 * <li><strong>ConsistsOf Relations</strong>: When a ConsistsOf relation is deleted, the relationship between the Resource and Facet is removed</li>
	 * <li><strong>Connected Resource</strong>: The source Resource remains but loses the connection to the target Facet</li>
	 * <li><strong>Connected Facet</strong>: The target Facet may become orphaned if this was its only connection to a Resource</li>
	 * <li><strong>Orphaned Facets</strong>: Facets that become orphaned (not connected to any Resource) may be automatically deleted based on schema configuration</li>
	 * <li><strong>Cascade Effects</strong>: The actual propagation behavior depends on the schema-defined constraints for the specific ConsistsOf type.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation and Constraints:</strong>
	 * <ul>
	 * <li>Before deletion, the system validates that removing the ConsistsOf relation will not violate schema constraints</li>
	 * <li>If deletion would leave the Resource incomplete or violate cardinality constraints, a {@link SchemaViolationException} is thrown</li>
	 * <li>This results in an HTTP 409 Conflict response with details about the constraint violation</li>
	 * <li><strong>Example scenarios that prevent deletion:</strong></li>
	 * </ul>
	 *   • Deleting a ConsistsOf relation to a required Facet (violates min cardinality);
	 *   • Removing mandatory relationships that make the Resource invalid according to its schema;
	 *   • Operations that would violate schema-defined constraints for the Resource type.
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and delete permissions for the specific context where the ConsistsOf relation exists.</li>
	 * </ul>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can delete any ConsistsOf relation in any context</li>
	 * <li>Full administrative privileges across all contexts.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can delete any ConsistsOf relation in any context</li>
	 * <li>Full administrative privileges across all contexts.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can delete ConsistsOf relations only in contexts where they have Context-Manager role</li>
	 * <li>Operations are limited to their assigned contexts.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can delete ConsistsOf relations only in contexts where they have appropriate permissions</li>
	 * <li>Operations are limited to relations they have access to in their authorized contexts.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Delete a ConsistsOf relation instance
	 * ConsistsOf&lt;EService, StateFacet&gt; consistsOf = publisher.readConsistsOf(ConsistsOf.class, relationUUID);
	 * boolean deleted = publisher.deleteConsistsOf(consistsOf);
	 * 
	 * // Delete a specific ConsistsOf relation between a HostingNode and a StateFacet
	 * ConsistsOf&lt;HostingNode, StateFacet&gt; hostState = publisher.readConsistsOf(ConsistsOf.class, hostStateUUID);
	 * boolean relationDeleted = publisher.deleteConsistsOf(hostState);
	 * </pre>
	 * 
	 * @param <C> The specific type of ConsistsOf relation to delete
	 * @param consistsOf The ConsistsOf relation instance to delete (must contain a valid UUID in the header)
	 * @return true if the ConsistsOf relation was successfully deleted, false otherwise
	 * @throws SchemaViolationException If deleting the ConsistsOf relation would violate Information System model constraints (HTTP 409)
	 * @throws ConsistsOfNotFoundException If the ConsistsOf relation does not exist or is not accessible (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during the delete operation
	 */
	public <C extends ConsistsOf<? extends Resource, ? extends Facet>> boolean deleteConsistsOf(C consistsOf)
			throws SchemaViolationException, ConsistsOfNotFoundException, ResourceRegistryException;

	/**
	 * Deletes a ConsistsOf relation instance from the Resource Registry using type name and UUID.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code DELETE /instances/{consistsof-type-name}/{uuid}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Permanently deletes the specified ConsistsOf relation instance from the Resource Registry</li>
	 * <li>The ConsistsOf relation is identified by its type name and UUID</li>
	 * <li>The ConsistsOf relation must exist and be accessible in the current context</li>
	 * <li>This operation removes the relationship between a Resource and a Facet</li>
	 * <li>This is a destructive operation that cannot be undone</li>
	 * <li>The connected Resource and Facet elements may be affected based on propagation constraints.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: ConsistsOf relation successfully deleted</li>
	 * <li><strong>404 Not Found</strong>: ConsistsOf relation does not exist or is not accessible in the current context</li>
	 * <li><strong>403 Forbidden</strong>: User lacks permissions to delete the ConsistsOf relation</li>
	 * <li><strong>409 Conflict</strong>: ConsistsOf relation cannot be deleted due to schema constraints</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The consistsOfType parameter must be a valid ConsistsOf relation type name registered in the Information System</li>
	 * <li>Type names are case-sensitive and must match exactly</li>
	 * <li>Supports all ConsistsOf relation types and their specializations</li>
	 * <li>The UUID must be a valid UUID that identifies an existing ConsistsOf relation instance.</li>
	 * </ul>
	 * 
	 * <strong>Deletion Effects and Propagation:</strong>
	 * <ul>
	 * <li><strong>ConsistsOf Relations</strong>: When a ConsistsOf relation is deleted, the relationship between the Resource and Facet is removed</li>
	 * <li><strong>Connected Resource</strong>: The source Resource remains but loses the connection to the target Facet</li>
	 * <li><strong>Connected Facet</strong>: The target Facet may become orphaned if this was its only connection to a Resource</li>
	 * <li><strong>Orphaned Facets</strong>: Facets that become orphaned (not connected to any Resource) may be automatically deleted based on schema configuration</li>
	 * <li><strong>Cascade Effects</strong>: The actual propagation behavior depends on the schema-defined constraints for the specific ConsistsOf type.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation and Constraints:</strong>
	 * <ul>
	 * <li>Before deletion, the system validates that removing the ConsistsOf relation will not violate schema constraints</li>
	 * <li>If deletion would leave the Resource incomplete or violate cardinality constraints, a {@link SchemaViolationException} is thrown</li>
	 * <li>This results in an HTTP 409 Conflict response with details about the constraint violation</li>
	 * <li><strong>Example scenarios that prevent deletion:</strong></li>
	 * </ul>
	 *   • Deleting a ConsistsOf relation to a required Facet (violates min cardinality);
	 *   • Removing mandatory relationships that make the Resource invalid according to its schema;
	 *   • Operations that would violate schema-defined constraints for the Resource type.
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and delete permissions for the specific context where the ConsistsOf relation exists.</li>
	 * </ul>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can delete any ConsistsOf relation in any context</li>
	 * <li>Full administrative privileges across all contexts.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can delete any ConsistsOf relation in any context</li>
	 * <li>Full administrative privileges across all contexts.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can delete ConsistsOf relations only in contexts where they have Context-Manager role</li>
	 * <li>Operations are limited to their assigned contexts.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can delete ConsistsOf relations only in contexts where they have appropriate permissions</li>
	 * <li>Operations are limited to relations they have access to in their authorized contexts.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Delete a ConsistsOf relation by type and UUID
	 * UUID relationUUID = UUID.fromString("12345678-1234-1234-1234-123456789abc");
	 * boolean deleted = publisher.deleteConsistsOf("ConsistsOf", relationUUID);
	 * 
	 * // Delete a specific ConsistsOf relation type
	 * UUID specificRelationUUID = UUID.fromString("87654321-4321-4321-4321-cba987654321");
	 * boolean relationDeleted = publisher.deleteConsistsOf("SpecificConsistsOfType", specificRelationUUID);
	 * </pre>
	 * 
	 * @param consistsOfType The exact type name of the ConsistsOf relation to delete
	 * @param uuid The UUID of the ConsistsOf relation instance to delete
	 * @return true if the ConsistsOf relation was successfully deleted, false otherwise
	 * @throws SchemaViolationException If deleting the ConsistsOf relation would violate Information System model constraints (HTTP 409)
	 * @throws ConsistsOfNotFoundException If the ConsistsOf relation does not exist or is not accessible (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during the delete operation
	 */
	public boolean deleteConsistsOf(String consistsOfType, UUID uuid)
			throws SchemaViolationException, ConsistsOfNotFoundException, ResourceRegistryException;

	/**
	 * Creates a new IsRelatedTo relation instance in the Resource Registry.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /instances/{isrelatedto-type-name}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Creates a new IsRelatedTo relation between two Resources</li>
	 * <li>IsRelatedTo relations represent various types of relationships between Resources (e.g., dependency, hosting, derivation)</li>
	 * <li>The relation instance is automatically associated with the current context</li>
	 * <li>Validates the relation against the Information System model schema</li>
	 * <li>Generates a new UUID for the relation if not already set</li>
	 * <li>Returns the created relation with server-generated metadata.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>201 Created</strong>: IsRelatedTo relation successfully created</li>
	 * <li><strong>400 Bad Request</strong>: Invalid relation data or schema validation failure</li>
	 * <li><strong>409 Conflict</strong>: IsRelatedTo relation with the same UUID already exists</li>
	 * <li><strong>404 Not Found</strong>: Source Resource or target Resource does not exist</li>
	 * <li><strong>403 Forbidden</strong>: User lacks write permissions in the current context</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The relation type name is automatically derived from the provided object using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>Supports all IsRelatedTo subtypes defined in the Information System model.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation:</strong>
	 * <ul>
	 * <li>All relation properties are validated against the Information System model</li>
	 * <li>Source and target Resources must exist and be accessible in compatible contexts</li>
	 * <li>Relation constraints and cardinality rules are enforced</li>
	 * <li>IsRelatedTo-specific validation rules are applied.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The instance is automatically associated with the current context.</li>
	 * </ul>
	 *
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>User must have write permissions in the current context</li>
	 * <li>Both source and target Resources must be available in the current context.</li>
	 * </ul>
	 *
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Create an IsRelatedTo relation between two EServices
	 * Activates&lt;EService, EService&gt; activates = new Activates&lt;&gt;();
	 * activates.setSource(sourceService);
	 * activates.setTarget(targetService);
	 * ...
	 * Activates&lt;EService, EService&gt; createdRelation = publisher.createIsRelatedTo(activates);
	 * </pre>
	 * 
	 * @param <I> The specific type of IsRelatedTo relation to create
	 * @param isRelatedTo The IsRelatedTo relation instance to create in the Resource Registry
	 * @return The created IsRelatedTo relation with server-generated metadata and UUID
	 * @throws SchemaViolationException If the relation violates Information System model constraints (HTTP 400)
	 * @throws IsRelatedToAlreadyPresentException If a relation with the same UUID already exists (HTTP 409)
	 * @throws ResourceNotFoundException If the source or target Resource does not exist (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during relation creation
	 */
	public <I extends IsRelatedTo<? extends Resource, ? extends Resource>> I createIsRelatedTo(I isRelatedTo)
			throws SchemaViolationException, IsRelatedToAlreadyPresentException, ResourceNotFoundException, ResourceRegistryException;

	/**
	 * Creates a new IsRelatedTo relation instance from JSON representation.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /instances/{isrelatedto-type-name}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Creates a new IsRelatedTo relation between two Resources from JSON representation</li>
	 * <li>IsRelatedTo relations represent various types of relationships between Resources (e.g., dependency, hosting, derivation)</li>
	 * <li>The relation instance is automatically associated with the current context</li>
	 * <li>Validates the JSON against the Information System model schema</li>
	 * <li>Type is determined from the JSON @class property</li>
	 * <li>Generates a new UUID for the relation if not already set</li>
	 * <li>Returns the created relation as JSON with server-generated metadata.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>201 Created</strong>: IsRelatedTo relation successfully created</li>
	 * <li><strong>400 Bad Request</strong>: Invalid JSON data or schema validation failure</li>
	 * <li><strong>409 Conflict</strong>: IsRelatedTo relation with the same UUID already exists</li>
	 * <li><strong>404 Not Found</strong>: Source Resource or target Resource does not exist</li>
	 * <li><strong>403 Forbidden</strong>: User lacks write permissions in the current context</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>JSON Format Requirements:</strong>
	 * <ul>
	 * <li>Must contain a valid @class property specifying an IsRelatedTo type</li>
	 * <li>Must conform to the Information System model JSON schema for IsRelatedTo relations</li>
	 * <li>Must include valid source and target Resource references</li>
	 * <li>All required properties for the specified relation type must be present.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation:</strong>
	 * <ul>
	 * <li>JSON structure is validated against the Information System model</li>
	 * <li>Source and target Resource references are validated</li>
	 * <li>Relation constraints and cardinality rules are enforced</li>
	 * <li>IsRelatedTo-specific validation rules are applied.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The instance is automatically associated with the current context.</li>
	 * </ul>
	 *
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>User must have write permissions in the current context</li>
	 * <li>Both source and target Resources must be available in the current context.</li>
	 * </ul>
	 *
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Create an IsRelatedTo relation from JSON
	 * String isRelatedToJson = "...";
	 * String createdRelationJson = publisher.createIsRelatedTo(isRelatedToJson);
	 * </pre>
	 * 
	 * @param isRelatedTo JSON representation of the IsRelatedTo relation to create
	 * @return JSON representation of the created IsRelatedTo relation with server-generated metadata and UUID
	 * @throws SchemaViolationException If the JSON violates Information System model constraints (HTTP 400)
	 * @throws IsRelatedToAlreadyPresentException If a relation with the same UUID already exists (HTTP 409)
	 * @throws ResourceNotFoundException If the source or target Resource does not exist (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during relation creation or JSON parsing
	 */
	public String createIsRelatedTo(String isRelatedTo)
			throws SchemaViolationException, IsRelatedToAlreadyPresentException, ResourceNotFoundException, ResourceRegistryException;

	/**
	 * Reads an IsRelatedTo relation instance from the Resource Registry by its UUID.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /instances/{isrelatedto-type-name}/{uuid}[?includeMeta={true|false}&allMeta={true|false}&includeContexts={true|false}&hierarchical={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves an existing IsRelatedTo relation by its unique identifier</li>
	 * <li>The relation must be accessible in the current context or in child contexts (based on hierarchical mode)</li>
	 * <li>Returns the complete relation instance with all properties and metadata</li>
	 * <li>The type is automatically determined from the provided object instance.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: IsRelatedTo relation successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: No IsRelatedTo relation found with the specified UUID or not accessible in current context</li>
	 * <li><strong>403 Forbidden</strong>: User lacks read permissions for the relation or its context</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response instance</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration, admin-only):
	 * <ul>
	 * <li>Whether to include instances from child contexts of the current context</li>
	 * <li>Default value: false (child contexts not included)</li>
	 * <li>Usage: {@code publisher.setHierarchicalMode(true)}</li>
	 * <li>Values: true (includes child contexts) | false (current context only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#HIERARCHICAL_MODE_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The relation type is automatically derived from the provided instance using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>The returned object will be of the same type as the provided instance.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The relation must be accessible in the current context or child contexts (depending on hierarchical mode)</li>
	 * <li>IS-Manager and Infrastructure-Manager are global roles (available in all contexts) and can use hierarchical mode in any contexts</li>
	 * <li>Context-Manager is a context-specific role and can use hierarchical mode only in contexts where they have this role.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Read an Activates relation
	 * Activates&lt;EService, EService&gt; activates = new Activates&lt;&gt;();
	 * activates.setHeader(new Header(uuid));
	 * Activates&lt;EService, EService&gt; readActivates = publisher.readIsRelatedTo(activates);
	 * </pre>
	 * 
	 * @param <I> The specific type of IsRelatedTo relation to read
	 * @param isRelatedTo Instance object containing at least the UUID in the header
	 * @return The complete IsRelatedTo relation instance from the Resource Registry
	 * @throws IsRelatedToNotFoundException If no IsRelatedTo relation is found with the specified UUID (HTTP 404)
	 * @throws IsRelatedToAvailableInAnotherContextException If the IsRelatedTo relation is not accessible in the current context (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during relation retrieval
	 */
	public <I extends IsRelatedTo<? extends Resource, ? extends Resource>> I readIsRelatedTo(I isRelatedTo)
			throws IsRelatedToNotFoundException, IsRelatedToAvailableInAnotherContextException, ResourceRegistryException;

	/**
	 * Reads an IsRelatedTo relation instance from the Resource Registry by its type and UUID.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /instances/{isrelatedto-type-name}/{uuid}[?includeMeta={true|false}&allMeta={true|false}&includeContexts={true|false}&hierarchical={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves an existing IsRelatedTo relation by its type name and unique identifier</li>
	 * <li>The relation must be accessible in the current context or in child contexts (based on hierarchical mode)</li>
	 * <li>Returns the complete relation instance as JSON with all properties and metadata</li>
	 * <li>Type validation ensures the UUID corresponds to a relation of the specified type.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: IsRelatedTo relation successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: No IsRelatedTo relation found with the specified UUID and type, or not accessible in current context</li>
	 * <li><strong>403 Forbidden</strong>: User lacks read permissions for the relation or its context</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response instance</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration, admin-only):
	 * <ul>
	 * <li>Whether to include instances from child contexts of the current context</li>
	 * <li>Default value: false (child contexts not included)</li>
	 * <li>Usage: {@code publisher.setHierarchicalMode(true)}</li>
	 * <li>Values: true (includes child contexts) | false (current context only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#HIERARCHICAL_MODE_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The specified type must be a valid IsRelatedTo type in the Information System model</li>
	 * <li>Supports all IsRelatedTo types and their subtypes as defined in the Information System model</li>
	 * <li>Type names are case-sensitive and must match exactly the class names defined in the Information System model</li>
	 * <li>UUID must correspond to a relation of exactly the specified type (no polymorphic matching).	 * </li>
	 * </ul>
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The relation must be accessible in the current context or child contexts (depending on hierarchical mode)</li>
	 * <li>IS-Manager and Infrastructure-Manager are global roles (available in all contexts) and can use hierarchical mode in any contexts</li>
	 * <li>Context-Manager is a context-specific role and can use hierarchical mode only in contexts where they have this role.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Read an Activates relation by type and UUID
	 * UUID relationUuid = UUID.fromString("...");
	 * String relationJson = publisher.readIsRelatedTo("Activates", relationUuid);
	 * </pre>
	 * 
	 * @param isRelatedToType The exact type name of the IsRelatedTo relation to read
	 * @param uuid The unique identifier of the IsRelatedTo relation to read
	 * @return JSON representation of the complete IsRelatedTo relation instance from the Resource Registry
	 * @throws IsRelatedToNotFoundException If no relation is found with the specified UUID and type (HTTP 404)
	 * @throws IsRelatedToAvailableInAnotherContextException If the relation exists but is not accessible in the current context (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during relation retrieval
	 */
	public String readIsRelatedTo(String isRelatedToType, UUID uuid)
			throws IsRelatedToNotFoundException, IsRelatedToAvailableInAnotherContextException, ResourceRegistryException;

	/**
	 * Updates an existing IsRelatedTo relation instance in the Resource Registry.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code PUT /instances/{isrelatedto-type-name}/{uuid}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Updates an existing IsRelatedTo relation with new property values</li>
	 * <li>The relation must exist and be accessible in the current context</li>
	 * <li>All provided properties will replace the existing ones (full update, not partial)</li>
	 * <li>Validates the updated relation against the Information System model schema</li>
	 * <li>Preserves the original UUID and creation metadata</li>
	 * <li>Returns the updated relation with current metadata.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: IsRelatedTo relation successfully updated</li>
	 * <li><strong>400 Bad Request</strong>: Invalid relation data or schema validation failure</li>
	 * <li><strong>404 Not Found</strong>: No IsRelatedTo relation found with the specified UUID or not accessible in current context</li>
	 * <li><strong>403 Forbidden</strong>: User lacks write permissions for the relation or its context</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The relation type name is automatically derived from the provided object using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>Type cannot be changed during update (UUID must correspond to existing relation of same type).</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation:</strong>
	 * <ul>
	 * <li>All relation properties are validated against the Information System model</li>
	 * <li>Source and target Resources must exist and be accessible in compatible contexts</li>
	 * <li>Relation constraints and cardinality rules are enforced</li>
	 * <li>IsRelatedTo-specific validation rules are applied.</li>
	 * </ul>
	 * 
	 * <strong>Update Behavior:</strong>
	 * <ul>
	 * <li>This is a complete replacement of the relation (not a merge)</li>
	 * <li>All properties not specified in the update will be lost</li>
	 * <li>Source and target Resource references can be changed if valid</li>
	 * <li>Context association cannot be changed through update.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Update an Activates relation
	 * Activates&lt;EService, EService&gt; relation = ...; // Retrieved relation
	 * // Modify properties
	 * relation.setStartTime(...);
	 * Activates&lt;EService, EService&gt; updatedRelation = publisher.updateIsRelatedTo(relation);
	 * </pre>
	 * 
	 * @param <I> The specific type of IsRelatedTo relation to update
	 * @param isRelatedTo The IsRelatedTo relation instance with updated values
	 * @return The updated IsRelatedTo relation with current server metadata
	 * @throws SchemaViolationException If the updated relation violates Information System model constraints (HTTP 400)
	 * @throws IsRelatedToNotFoundException If no relation is found with the specified UUID (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during relation update
	 */
	public <I extends IsRelatedTo<? extends Resource, ? extends Resource>> I updateIsRelatedTo(I isRelatedTo)
			throws SchemaViolationException, IsRelatedToNotFoundException, ResourceRegistryException;

	/**
	 * Updates an existing IsRelatedTo relation instance from JSON representation.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code PUT /instances/{isrelatedto-type-name}/{uuid}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Updates an existing IsRelatedTo relation with new property values from JSON representation</li>
	 * <li>The relation must exist and be accessible in the current context</li>
	 * <li>All provided properties will replace the existing ones (full update, not partial)</li>
	 * <li>Validates the JSON against the Information System model schema</li>
	 * <li>Type is determined from the JSON @class property</li>
	 * <li>Preserves the original UUID and creation metadata</li>
	 * <li>Returns the updated relation as JSON with current metadata.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: IsRelatedTo relation successfully updated</li>
	 * <li><strong>400 Bad Request</strong>: Invalid JSON data or schema validation failure</li>
	 * <li><strong>404 Not Found</strong>: No IsRelatedTo relation found with the specified UUID or not accessible in current context</li>
	 * <li><strong>403 Forbidden</strong>: User lacks write permissions for the relation or its context</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>JSON Format Requirements:</strong>
	 * <ul>
	 * <li>Must contain a valid @class property specifying an IsRelatedTo type</li>
	 * <li>Must contain a valid header with the UUID of the relation to update</li>
	 * <li>Must conform to the Information System model JSON schema for IsRelatedTo relations</li>
	 * <li>Must include valid source and target Resource references</li>
	 * <li>All required properties for the specified relation type must be present.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation:</strong>
	 * <ul>
	 * <li>JSON structure is validated against the Information System model</li>
	 * <li>Source and target Resource references are validated</li>
	 * <li>Relation constraints and cardinality rules are enforced</li>
	 * <li>IsRelatedTo-specific validation rules are applied.</li>
	 * </ul>
	 * 
	 * <strong>Update Behavior:</strong>
	 * <ul>
	 * <li>This is a complete replacement of the relation (not a merge)</li>
	 * <li>All properties not specified in the JSON will be lost</li>
	 * <li>Source and target Resource references can be changed if valid</li>
	 * <li>Context association cannot be changed through update.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Update an IsRelatedTo relation from JSON
	 * String updatedRelationJson = "..."; // JSON with same UUID but updated properties
	 * String resultJson = publisher.updateIsRelatedTo(updatedRelationJson);
	 * </pre>
	 * 
	 * @param isRelatedTo JSON representation of the IsRelatedTo relation with updated values
	 * @return JSON representation of the updated IsRelatedTo relation with current server metadata
	 * @throws SchemaViolationException If the JSON violates Information System model constraints (HTTP 400)
	 * @throws IsRelatedToNotFoundException If no relation is found with the specified UUID (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during relation update or JSON parsing
	 */
	public String updateIsRelatedTo(String isRelatedTo)
			throws SchemaViolationException, IsRelatedToNotFoundException, ResourceRegistryException;

	/**
	 * Deletes an IsRelatedTo relation instance from the Resource Registry.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code DELETE /instances/{isrelatedto-type-name}/{uuid}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Permanently removes an IsRelatedTo relation from the Resource Registry</li>
	 * <li>The relation must exist and be accessible in the current context</li>
	 * <li>Only the relation is deleted; source and target Resources remain unaffected</li>
	 * <li>The operation is irreversible once completed</li>
	 * <li>Returns boolean indicating successful deletion.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>204 No Content</strong>: IsRelatedTo relation successfully deleted</li>
	 * <li><strong>404 Not Found</strong>: No IsRelatedTo relation found with the specified UUID or not accessible in current context</li>
	 * <li><strong>403 Forbidden</strong>: User lacks write permissions for the relation or its context</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The relation type name is automatically derived from the provided object using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>Only the UUID from the header is used for identification.</li>
	 * </ul>
	 * 
	 * <strong>Deletion Behavior:</strong>
	 * <ul>
	 * <li>The relation is permanently removed from the Information System</li>
	 * <li>Source and target Resources are not affected by the deletion</li>
	 * <li>Any cascading effects depend on Information System model constraints</li>
	 * <li>The operation cannot be undone.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Delete an Activates relation
	 * Activates&lt;EService, EService&gt; relation = new Activates&lt;&gt;();
	 * relation.setHeader(new Header(uuid));
	 * boolean deleted = publisher.deleteIsRelatedTo(relation);
	 * </pre>
	 * 
	 * @param <I> The specific type of IsRelatedTo relation to delete
	 * @param isRelatedTo Instance object containing at least the UUID in the header
	 * @return {@code true} if the IsRelatedTo relation was successfully deleted, {@code false} otherwise
	 * @throws IsRelatedToNotFoundException If no relation is found with the specified UUID (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during relation deletion
	 */
	public <I extends IsRelatedTo<? extends Resource, ? extends Resource>> boolean deleteIsRelatedTo(I isRelatedTo)
			throws IsRelatedToNotFoundException, ResourceRegistryException;

	/**
	 * Deletes an IsRelatedTo relation instance from the Resource Registry by its type and UUID.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code DELETE /instances/{isrelatedto-type-name}/{uuid}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Permanently removes an IsRelatedTo relation from the Resource Registry</li>
	 * <li>The relation must exist and be accessible in the current context</li>
	 * <li>Only the relation is deleted; source and target Resources remain unaffected</li>
	 * <li>Type validation ensures the UUID corresponds to a relation of the specified type</li>
	 * <li>The operation is irreversible once completed</li>
	 * <li>Returns boolean indicating successful deletion.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>204 No Content</strong>: IsRelatedTo relation successfully deleted</li>
	 * <li><strong>404 Not Found</strong>: No IsRelatedTo relation found with the specified UUID and type, or not accessible in current context</li>
	 * <li><strong>403 Forbidden</strong>: User lacks write permissions for the relation or its context</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The specified type must be a valid IsRelatedTo type in the Information System model</li>
	 * <li>Supports all IsRelatedTo types and their subtypes as defined in the Information System model</li>
	 * <li>Type names are case-sensitive and must match exactly the class names defined in the Information System model</li>
	 * <li>UUID must correspond to a relation of exactly the specified type (no polymorphic matching).	 * </li>
	 * </ul>
	 * <strong>Deletion Behavior:</strong>
	 * <ul>
	 * <li>The relation is permanently removed from the Information System</li>
	 * <li>Source and target Resources are not affected by the deletion</li>
	 * <li>Any cascading effects depend on Information System model constraints</li>
	 * <li>The operation cannot be undone.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Delete an Activates relation by type and UUID
	 * UUID relationUuid = UUID.fromString("...");
	 * boolean deleted = publisher.deleteIsRelatedTo("Activates", relationUuid);
	 * </pre>
	 * 
	 * @param isRelatedToType The exact type name of the IsRelatedTo relation to delete
	 * @param uuid The unique identifier of the IsRelatedTo relation to delete
	 * @return {@code true} if the IsRelatedTo relation was successfully deleted, {@code false} otherwise
	 * @throws IsRelatedToNotFoundException If no relation is found with the specified UUID and type (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during relation deletion
	 */
	public boolean deleteIsRelatedTo(String isRelatedToType, UUID uuid)
			throws IsRelatedToNotFoundException, ResourceRegistryException;

	/**
	 * Adds an Information System element instance to a specific context.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code PUT /instances/{type-name}/{uuid}/contexts/{context-uuid}[?dryRun={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Adds an existing instance (Entity or Relation) to the specified target context</li>
	 * <li>The instance must already exist and be accessible in the context where the user is performing the request</li>
	 * <li>The target context must exist and the user must have the rights to perform such a request</li>
	 * <li>Returns the list of affected elements (the instance itself and any related elements that are also added)</li>
	 * <li>Supports dry-run mode to preview changes without applying them.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Instance successfully added to context (or dry-run completed)</li>
	 * <li><strong>404 Not Found</strong>: Instance or target context does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: User lacks permissions to perform the request</li>
	 * <li><strong>400 Bad Request</strong>: Invalid parameters or schema validation failure</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>forceAddToContext</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Forces the addition of an instance to a context even if the instance is not present in the current context</li>
	 * <li>Default value: false (standard validation applies)</li>
	 * <li>Usage: {@code publisher.addHeader(SharingPath.FORCE_ADD_TO_CONTEXT_QUERY_PARAMETER, true)}</li>
	 * <li>Values: true | false</li>
	 * <li>When true: Allows adding an instance to a context even if the instance exists only in other contexts</li>
	 * <li>When false: Standard validation applies - instance must be accessible in the current context for ADD operations</li>
	 * <li><strong>Authorization Required</strong>: Only available to IS-Manager and Infrastructure-Manager</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response for affected instances</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The type name must be a valid Information System model type</li>
	 * <li>Supports all Information System model types and their subtypes: Entities (Resources and all Resource subtypes, Facets and all Facet subtypes) and Relations (ConsistsOf and all ConsistsOf subtypes, IsRelatedTo and all IsRelatedTo subtypes)</li>
	 * <li>Type names are case-sensitive and must match exactly the class names defined in the Information System model.</li>
	 * </ul>
	 * 
	 * <strong>Context Propagation Rules:</strong>
	 * <ul>
	 * <li><strong>Resources</strong>: When added to a context, all their associated Facets (via ConsistsOf relations) are also added to the same context if the add propagation constraint is {@code propagate}</li>
	 * <li><strong>Facets</strong>: When added to a context, the Resource that reference it (via ConsistsOf relation) is also added to the same context</li>
	 * <li><strong>ConsistsOf Relations</strong>: add both facet and resource to the same context</li>
	 * <li><strong>IsRelatedTo Relations</strong>: add both source and target resources to the same context.</li>
	 * </ul>
	 * 
	 * <strong>Propagation Constraints and Cascade Effects:</strong>
	 * <ul>
	 * <li>When an instance is added to a context, related instances may also be automatically added due to propagation constraints</li>
	 * <li>The response includes all instances that were added as part of the cascade effect</li>
	 * <li>Propagation follows the schema-defined relationships and their constraints.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation for ConsistsOf Relations and Facets:</strong>
	 * <ul>
	 * <li>When operations are performed on ConsistsOf relations or Facet instances, the parent Resource is automatically validated against its schema definition</li>
	 * <li>If the Resource is no longer compliant with its schema after the add operation, a {@link SchemaViolationException} is thrown</li>
	 * <li>This results in an HTTP 400 Bad Request response with details about the schema violation</li>
	 * <li><strong>Example scenarios that violate schema constraints:</strong></li>
	 * </ul>
	 *   • Adding instances that violate cardinality constraints (max occurrences) defined in the schema;
	 *   • Operations that make the Resource incomplete or invalid according to its schema definition.
	 * <ul>
	 * <li><strong>Constraint Sources</strong>: Constraints are defined by the specific type definitions that have been installed on the system</li>
	 * <li>Schema validation ensures the integrity and consistency by enforcing the constraints defined in each type's schema during operations.</li>
	 * </ul>
	 * 
    * <strong>Authorization Requirements:</strong>
     * <ul>
     * <li>All users must have a valid token and appropriate permissions for the specific context where they want to add instances.</li>
     * </ul>
     * 
     * <strong>IS-Manager:</strong>
     * <ul>
     * <li>Can perform ADD operations on any instance in any context</li>
     * <li>Can use forceAddToContext=true to add instances from other contexts</li>
     * <li>Full administrative privileges across all contexts</li>
     * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
     * </ul>
     * 
     * <strong>Infrastructure-Manager:</strong>
     * <ul>
     * <li>Can perform ADD operations on any instance in any context</li>
     * <li>Can use forceAddToContext=true to add instances from other contexts</li>
     * <li>Full administrative privileges across all contexts</li>
     * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
     * </ul>
     * 
     * <strong>Context-Manager:</strong>
     * <ul>
     * <li>Can perform ADD operations only on instances accessible in their current context</li>
     * <li>Cannot use forceAddToContext=true (parameter is ignored if provided)</li>
     * <li>Operations are limited to contexts where they have Context-Manager role</li>
     * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
     * </ul>
     * 
     * <strong>Other Users:</strong>
     * <ul>
     * <li>Can perform ADD operations only on instances accessible in their current context</li>
     * <li>Cannot use forceAddToContext=true (parameter is ignored if provided)</li>
     * <li>Operations are limited to instances they have access to in their authorized context</li>
     * <li>See filtered metadata with sensitive fields obfuscated when includeMeta=true.</li>
     * </ul>
	 * 
     * <strong>Response Format:</strong>
     * <ul>
     * <li>Returns a JSON array containing all instances affected by the operation</li>
     * <li>Each instance object includes its UUID, type, and relevant metadata</li>
     * <li>For dryRun=true: Shows what would be affected without making changes</li>
     * <li>For dryRun=false: Shows what was actually affected by the completed operation.</li>
     * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Add an EService to a specific context with actual changes
	 * UUID serviceUUID = UUID.fromString("...");
	 * UUID targetContextUUID = UUID.fromString("...");
	 * List&lt;ERElement&gt; affected = publisher.addToContext("EService", serviceUUID, targetContextUUID, false);
	 * 
	 * // Preview what would be affected by adding a ContactFacet to a context
	 * UUID facetUUID = UUID.fromString("...");
	 * List&lt;ERElement&gt; preview = publisher.addToContext("ContactFacet", facetUUID, targetContextUUID, true);
	 * </pre>
	 * 
	 * @param type The exact type name of the Information System element to add
	 * @param instanceUUID The UUID of the instance to add to the context
	 * @param contextUUID The UUID of the target context where the instance should be added
	 * @param dryRun Whether to perform a dry-run (true) or apply actual changes (false)
	 * @return List of all elements affected by this operation (including the instance itself and any related elements)
	 * @throws SchemaViolationException If the operation violates Information System model constraints (HTTP 400)
	 * @throws NotFoundException If the instance or context is not found or not accessible (HTTP 404)
	 * @throws ContextNotFoundException If the target context does not exist or is not accessible (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during the add operation
	 */
	public List<ERElement> addToContext(String type, UUID instanceUUID, UUID contextUUID, Boolean dryRun)
			throws SchemaViolationException, NotFoundException, ContextNotFoundException, ResourceRegistryException;

	/**
	 * Adds an Information System element instance to a specific context using a provided instance object.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code PUT /instances/{type-name}/{uuid}/contexts/{context-uuid}[?dryRun={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Adds an existing instance (Entity or Relation) to the specified target context</li>
	 * <li>The type and UUID are automatically extracted from the provided instance object</li>
	 * <li>The instance must already exist and be accessible in the context where the user is performing the request</li>
	 * <li>The target context must exist and the user must have the rights to perform such a request</li>
	 * <li>Returns the list of affected elements (the instance itself and any related elements that are also added)</li>
	 * <li>Supports dry-run mode to preview changes without applying them.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Instance successfully added to context (or dry-run completed)</li>
	 * <li><strong>404 Not Found</strong>: Instance or target context does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: User lacks permissions to perform the request</li>
	 * <li><strong>400 Bad Request</strong>: Invalid parameters or schema validation failure</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>forceAddToContext</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Forces the addition of an instance to a context even if the instance is not present in the current context</li>
	 * <li>Default value: false (standard validation applies)</li>
	 * <li>Usage: {@code publisher.addHeader(SharingPath.FORCE_ADD_TO_CONTEXT_QUERY_PARAMETER, true)}</li>
	 * <li>Values: true | false</li>
	 * <li>When true: Allows adding an instance to a context even if the instance exists only in other contexts</li>
	 * <li>When false: Standard validation applies - instance must be accessible in the current context for ADD operations</li>
	 * <li><strong>Authorization Required</strong>: Only available to IS-Manager and Infrastructure-Manager</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response for affected instances</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Type name is automatically derived from the provided instance using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>UUID is extracted from the instance's header using standard Information System model conventions</li>
	 * <li>Supports all Information System model types and their subtypes: Entities (Resources and all Resource subtypes, Facets and all Facet subtypes) and Relations (ConsistsOf and all ConsistsOf subtypes, IsRelatedTo and all IsRelatedTo subtypes)</li>
	 * <li>Type names are case-sensitive and must match exactly the class names defined in the Information System model.</li>
	 * </ul>
	 *
	 * <strong>Context Propagation Rules:</strong>
	 * <ul>
	 * <li><strong>Resources</strong>: When added to a context, all their associated Facets (via ConsistsOf relations) are also added to the same context if the add propagation constraint is {@code propagate}</li>
	 * <li><strong>Facets</strong>: When added to a context, the Resource that reference it (via ConsistsOf relation) is also added to the same context</li>
	 * <li><strong>ConsistsOf Relations</strong>: add both facet and resource to the same context</li>
	 * <li><strong>IsRelatedTo Relations</strong>: add both source and target resources to the same context.</li>
	 * </ul>
	 * 
	 * <strong>Propagation Constraints and Cascade Effects:</strong>
	 * <ul>
	 * <li>When an instance is added to a context, related instances may also be automatically added due to propagation constraints</li>
	 * <li>The response includes all instances that were added as part of the cascade effect</li>
	 * <li>Propagation follows the schema-defined relationships and their constraints.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation for ConsistsOf Relations and Facets:</strong>
	 * <ul>
	 * <li>When operations are performed on ConsistsOf relations or Facet instances, the parent Resource is automatically validated against its schema definition</li>
	 * <li>If the Resource is no longer compliant with its schema after the add operation, a {@link SchemaViolationException} is thrown</li>
	 * <li>This results in an HTTP 400 Bad Request response with details about the schema violation</li>
	 * <li><strong>Example scenarios that violate schema constraints:</strong></li>
	 * </ul>
	 *   • Adding instances that violate cardinality constraints (max occurrences) defined in the schema;
	 *   • Operations that make the Resource incomplete or invalid according to its schema definition.
	 * <ul>
	 * <li><strong>Constraint Sources</strong>: Constraints are defined by the specific type definitions that have been installed on the system</li>
	 * <li>Schema validation ensures the integrity and consistency by enforcing the constraints defined in each type's schema during operations.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and appropriate permissions for the specific context where they want to add instances.</li>
	 * </ul>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD operations on any instance in any context</li>
	 * <li>Can use forceAddToContext=true to add instances from other contexts</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD operations on any instance in any context</li>
	 * <li>Can use forceAddToContext=true to add instances from other contexts</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD operations only on instances accessible in their current context</li>
	 * <li>Cannot use forceAddToContext=true (parameter is ignored if provided)</li>
	 * <li>Operations are limited to contexts where they have Context-Manager role</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can perform ADD operations only on instances accessible in their current context</li>
	 * <li>Cannot use forceAddToContext=true (parameter is ignored if provided)</li>
	 * <li>Operations are limited to instances they have access to in their authorized context</li>
	 * <li>See filtered metadata with sensitive fields obfuscated when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON array containing all instances affected by the operation</li>
	 * <li>Each instance object includes its UUID, type, and relevant metadata</li>
	 * <li>For dryRun=true: Shows what would be affected without making changes</li>
	 * <li>For dryRun=false: Shows what was actually affected by the completed operation.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Add an EService instance to a specific context
	 * EService service = new EService();
	 * service.setHeader(new Header(serviceUUID));
	 * UUID targetContextUUID = UUID.fromString("...");
	 * List&lt;ERElement&gt; affected = publisher.addToContext(service, targetContextUUID, false);
	 * 
	 * // Preview what would be affected by adding a ContactFacet to a context
	 * ContactFacet contact = publisher.read(ContactFacet.class, contactUUID);
	 * List&lt;ERElement&gt; preview = publisher.addToContext(contact, targetContextUUID, true);
	 * </pre>
	 * 
	 * @param er The instance to add to the context (must contain a valid UUID in the header)
	 * @param contextUUID The UUID of the target context where the instance should be added
	 * @param dryRun Whether to perform a dry-run (true) or apply actual changes (false)
	 * @return List of all elements affected by this operation (including the instance itself and any related elements)
	 * @throws SchemaViolationException If the operation violates Information System model constraints (HTTP 400)
	 * @throws NotFoundException If the instance or context is not found or not accessible (HTTP 404)
	 * @throws ContextNotFoundException If the target context does not exist or is not accessible (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during the add operation
	 */
	public List<ERElement> addToContext(ERElement er, UUID contextUUID, Boolean dryRun)
			throws SchemaViolationException, NotFoundException, ContextNotFoundException, ResourceRegistryException;

	/**
	 * Adds an Information System element instance to the current context.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code PUT /instances/{type-name}/{uuid}/contexts[?dryRun={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Adds an existing instance (Entity or Relation) to the current context derived from the authorization token</li>
	 * <li>The instance must already exist and be accessible in the context where the user is performing the request</li>
	 * <li>The user must have the rights to perform such a request in the current context</li>
	 * <li>Returns the list of affected elements (the instance itself and any related elements that are also added)</li>
	 * <li>Supports dry-run mode to preview changes without applying them</li>
	 * <li>This is a convenience method that uses the current context instead of requiring an explicit context UUID.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Instance successfully added to current context (or dry-run completed)</li>
	 * <li><strong>404 Not Found</strong>: Instance does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: User lacks permissions to perform the request in the current context</li>
	 * <li><strong>400 Bad Request</strong>: Invalid parameters or schema validation failure</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>forceAddToContext</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Forces the addition of an instance to a context even if the instance is not present in the current context</li>
	 * <li>Default value: false (standard validation applies)</li>
	 * <li>Usage: {@code publisher.addHeader(SharingPath.FORCE_ADD_TO_CONTEXT_QUERY_PARAMETER, true)}</li>
	 * <li>Values: true | false</li>
	 * <li>When true: Allows adding an instance to a context even if the instance exists only in other contexts</li>
	 * <li>When false: Standard validation applies - instance must be accessible in the current context for ADD operations</li>
	 * <li><strong>Authorization Required</strong>: Only available to IS-Manager and Infrastructure-Manager</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response for affected instances</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The type name must be a valid Information System model type</li>
	 * <li>Supports all Information System model types: Entities (Resources, Facets) and Relations (ConsistsOf, IsRelatedTo) and their specialization</li>
	 * <li>Type name is case-sensitive and must match exactly.</li>
	 * </ul>
	 * 
	 * <strong>Current Context:</strong>
	 * <ul>
	 * <li>The current context is automatically derived from the authorization token provided during client creation</li>
	 * <li>No explicit context UUID is required</li>
	 * <li>The user must have appropriate permissions in this context.</li>
	 * </ul>
	 * 
	 * <strong>Context Propagation Rules:</strong>
	 * <ul>
	 * <li><strong>Resources</strong>: When added to a context, all their associated Facets (via ConsistsOf relations) are also added to the same context if the add propagation constraint is {@code propagate}</li>
	 * <li><strong>Facets</strong>: When added to a context, the Resource that reference it (via ConsistsOf relation) is also added to the same context</li>
	 * <li><strong>ConsistsOf Relations</strong>: add both facet and resource to the same context</li>
	 * <li><strong>IsRelatedTo Relations</strong>: add both source and target resources to the same context.</li>
	 * </ul>
	 * 
	 * <strong>Propagation Constraints and Cascade Effects:</strong>
	 * <ul>
	 * <li>When an instance is added to a context, related instances may also be automatically added due to propagation constraints</li>
	 * <li>The response includes all instances that were added as part of the cascade effect</li>
	 * <li>Propagation follows the schema-defined relationships and their constraints.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation for ConsistsOf Relations and Facets:</strong>
	 * <ul>
	 * <li>When operations are performed on ConsistsOf relations or Facet instances, the parent Resource is automatically validated against its schema definition</li>
	 * <li>If the Resource is no longer compliant with its schema after the add operation, a {@link SchemaViolationException} is thrown</li>
	 * <li>This results in an HTTP 400 Bad Request response with details about the schema violation</li>
	 * <li><strong>Example scenarios that violate schema constraints:</strong></li>
	 * </ul>
	 *   • Adding instances that violate cardinality constraints (max occurrences) defined in the schema;
	 *   • Operations that make the Resource incomplete or invalid according to its schema definition.
	 * <ul>
	 * <li><strong>Constraint Sources</strong>: Constraints are defined by the specific type definitions that have been installed on the system</li>
	 * <li>Schema validation ensures the integrity and consistency by enforcing the constraints defined in each type's schema during operations.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and appropriate permissions for the current context where they want to add instances.</li>
	 * <li>The current context is derived from the authorization token provided during client creation.</li>
	 * </ul>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD operations on any instance in any context</li>
	 * <li>Can use forceAddToContext=true to add instances from other contexts</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD operations on any instance in any context</li>
	 * <li>Can use forceAddToContext=true to add instances from other contexts</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD operations only on instances accessible in their current context</li>
	 * <li>Cannot use forceAddToContext=true (parameter is ignored if provided)</li>
	 * <li>Operations are limited to contexts where they have Context-Manager role</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can perform ADD operations only on instances accessible in their current context</li>
	 * <li>Cannot use forceAddToContext=true (parameter is ignored if provided)</li>
	 * <li>Operations are limited to instances they have access to in their authorized context</li>
	 * <li>See filtered metadata with sensitive fields obfuscated when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON array containing all instances affected by the operation</li>
	 * <li>Each instance object includes its UUID, type, and relevant metadata</li>
	 * <li>For dryRun=true: Shows what would be affected without making changes</li>
	 * <li>For dryRun=false: Shows what was actually affected by the completed operation.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Add an EService to the current context with actual changes
	 * UUID serviceUUID = UUID.fromString("...");
	 * List&lt;ERElement&gt; affected = publisher.addToCurrentContext("EService", serviceUUID, false);
	 * 
	 * // Preview what would be affected by adding a ContactFacet to the current context
	 * UUID facetUUID = UUID.fromString("...");
	 * List&lt;ERElement&gt; preview = publisher.addToCurrentContext("ContactFacet", facetUUID, true);
	 * </pre>
	 * 
	 * @param type The exact type name of the Information System element to add
	 * @param instanceUUID The UUID of the instance to add to the current context
	 * @param dryRun Whether to perform a dry-run (true) or apply actual changes (false)
	 * @return List of all elements affected by this operation (including the instance itself and any related elements)
	 * @throws SchemaViolationException If the operation violates Information System model constraints (HTTP 400)
	 * @throws NotFoundException If the instance is not found or not accessible (HTTP 404)
	 * @throws ContextNotFoundException If the current context cannot be determined or accessed (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during the add operation
	 */
	public List<ERElement> addToCurrentContext(String type, UUID instanceUUID, Boolean dryRun)
			throws SchemaViolationException, NotFoundException, ContextNotFoundException, ResourceRegistryException;

	/**
	 * Adds an Information System element instance to the current context using a provided instance object.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code PUT /instances/{type-name}/{uuid}/contexts[?dryRun={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Adds an existing instance (Entity or Relation) to the current context derived from the authorization token</li>
	 * <li>The type and UUID are automatically extracted from the provided instance object</li>
	 * <li>The instance must already exist and be accessible in the context where the user is performing the request</li>
	 * <li>The user must have the rights to perform such a request in the current context</li>
	 * <li>Returns the list of affected elements (the instance itself and any related elements that are also added)</li>
	 * <li>Supports dry-run mode to preview changes without applying them</li>
	 * <li>This is a convenience method that uses the current context instead of requiring an explicit context UUID.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Instance successfully added to current context (or dry-run completed)</li>
	 * <li><strong>404 Not Found</strong>: Instance does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: User lacks permissions to perform the request in the current context</li>
	 * <li><strong>400 Bad Request</strong>: Invalid parameters or schema validation failure</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>forceAddToContext</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Forces the addition of an instance to a context even if the instance is not present in the current context</li>
	 * <li>Default value: false (standard validation applies)</li>
	 * <li>Usage: {@code publisher.addHeader(SharingPath.FORCE_ADD_TO_CONTEXT_QUERY_PARAMETER, true)}</li>
	 * <li>Values: true | false</li>
	 * <li>When true: Allows adding an instance to a context even if the instance exists only in other contexts</li>
	 * <li>When false: Standard validation applies - instance must be accessible in the current context for ADD operations</li>
	 * <li><strong>Authorization Required</strong>: Only available to IS-Manager and Infrastructure-Manager</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response for affected instances</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Type name is automatically derived from the provided instance using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>UUID is extracted from the instance's header using standard Information System model conventions</li>
	 * <li>Supports all Information System model types and their subtypes: Entities (Resources and all Resource subtypes, Facets and all Facet subtypes) and Relations (ConsistsOf and all ConsistsOf subtypes, IsRelatedTo and all IsRelatedTo subtypes)</li>
	 * <li>Type names are case-sensitive and must match exactly the class names defined in the Information System model.</li>
	 * </ul>
	 *
	 * <strong>Current Context:</strong>
	 * <ul>
	 * <li>The current context is automatically derived from the authorization token provided during client creation</li>
	 * <li>No explicit context UUID is required</li>
	 * <li>The user must have appropriate permissions in this context.</li>
	 * </ul>
	 * 
	 * <strong>Context Propagation Rules:</strong>
	 * <ul>
	 * <li><strong>Resources</strong>: When added to a context, all their associated Facets (via ConsistsOf relations) are also added to the same context if the add propagation constraint is {@code propagate}</li>
	 * <li><strong>Facets</strong>: When added to a context, the Resource that reference it (via ConsistsOf relation) is also added to the same context</li>
	 * <li><strong>ConsistsOf Relations</strong>: add both facet and resource to the same context</li>
	 * <li><strong>IsRelatedTo Relations</strong>: add both source and target resources to the same context.</li>
	 * </ul>
	 * 
	 * <strong>Propagation Constraints and Cascade Effects:</strong>
	 * <ul>
	 * <li>When an instance is added to a context, related instances may also be automatically added due to propagation constraints</li>
	 * <li>The response includes all instances that were added as part of the cascade effect</li>
	 * <li>Propagation follows the schema-defined relationships and their constraints.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation for ConsistsOf Relations and Facets:</strong>
	 * <ul>
	 * <li>When operations are performed on ConsistsOf relations or Facet instances, the parent Resource is automatically validated against its schema definition</li>
	 * <li>If the Resource is no longer compliant with its schema after the add operation, a {@link SchemaViolationException} is thrown</li>
	 * <li>This results in an HTTP 400 Bad Request response with details about the schema violation</li>
	 * <li><strong>Example scenarios that violate schema constraints:</strong></li>
	 * </ul>
	 *   • Adding instances that violate cardinality constraints (max occurrences) defined in the schema;
	 *   • Operations that make the Resource incomplete or invalid according to its schema definition.
	 * <ul>
	 * <li><strong>Constraint Sources</strong>: Constraints are defined by the specific type definitions that have been installed on the system</li>
	 * <li>Schema validation ensures the integrity and consistency by enforcing the constraints defined in each type's schema during operations.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and appropriate permissions for the current context where they want to add instances.</li>
	 * <li>The current context is derived from the authorization token provided during client creation.</li>
	 * </ul>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD operations on any instance in any context</li>
	 * <li>Can use forceAddToContext=true to add instances from other contexts</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD operations on any instance in any context</li>
	 * <li>Can use forceAddToContext=true to add instances from other contexts</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD operations only on instances accessible in their current context</li>
	 * <li>Cannot use forceAddToContext=true (parameter is ignored if provided)</li>
	 * <li>Operations are limited to contexts where they have Context-Manager role</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can perform ADD operations only on instances accessible in their current context</li>
	 * <li>Cannot use forceAddToContext=true (parameter is ignored if provided)</li>
	 * <li>Operations are limited to instances they have access to in their authorized context</li>
	 * <li>See filtered metadata with sensitive fields obfuscated when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON array containing all instances affected by the operation</li>
	 * <li>Each instance object includes its UUID, type, and relevant metadata</li>
	 * <li>For dryRun=true: Shows what would be affected without making changes</li>
	 * <li>For dryRun=false: Shows what was actually affected by the completed operation.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Add an EService instance to the current context
	 * EService service = new EService();
	 * service.setHeader(new Header(serviceUUID));
	 * List&lt;ERElement&gt; affected = publisher.addToCurrentContext(service, false);
	 * 
	 * // Preview what would be affected by adding a ContactFacet to the current context
	 * ContactFacet contact = publisher.read(ContactFacet.class, contactUUID);
	 * List&lt;ERElement&gt; preview = publisher.addToCurrentContext(contact, true);
	 * </pre>
	 * 
	 * @param er The instance to add to the current context (must contain a valid UUID in the header)
	 * @param dryRun Whether to perform a dry-run (true) or apply actual changes (false)
	 * @return List of all elements affected by this operation (including the instance itself and any related elements)
	 * @throws SchemaViolationException If the operation violates Information System model constraints (HTTP 400)
	 * @throws NotFoundException If the instance is not found or not accessible (HTTP 404)
	 * @throws ContextNotFoundException If the current context cannot be determined or accessed (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during the add operation
	 */
	public List<ERElement> addToCurrentContext(ERElement er, Boolean dryRun)
			throws SchemaViolationException, NotFoundException, ContextNotFoundException, ResourceRegistryException;

	/**
	 * Removes an Information System element instance from a specific context.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /sharing/contexts/{context-uuid}/{type-name}/{instance-uuid}?operation=REMOVE[&dryRun={true|false}][&includeMeta={true|false}][&allMeta={true|false}][&includeContexts={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Removes an existing instance (Entity or Relation) from the specified target context</li>
	 * <li>The instance must already exist and be accessible in the target context where the user is performing the request</li>
	 * <li>The target context must exist and the user must have the rights to perform such a request</li>
	 * <li>Returns the list of affected elements (the instance itself and any related elements that are also removed)</li>
	 * <li>Supports dry-run mode to preview changes without applying them.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Instance successfully removed from context (or dry-run completed)</li>
	 * <li><strong>404 Not Found</strong>: Instance or target context does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: User lacks permissions to perform the request</li>
	 * <li><strong>400 Bad Request</strong>: Invalid parameters or schema validation failure</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response for affected instances</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The type name must be a valid Information System model type</li>
	 * <li>Supports all Information System model types and their subtypes: Entities (Resources and all Resource subtypes, Facets and all Facet subtypes) and Relations (ConsistsOf and all ConsistsOf subtypes, IsRelatedTo and all IsRelatedTo subtypes)</li>
	 * <li>Type names are case-sensitive and must match exactly the class names defined in the Information System model.</li>
	 * </ul>
	 * 
	 * <strong>Context Propagation Rules:</strong>
	 * <ul>
	 * <li><strong>Resources</strong>: When removed from a context, all their associated Facets (via ConsistsOf relations) are also removed from the same context if the remove propagation constraint is {@code propagate}</li>
	 * <li><strong>Facets</strong>: When removed from a context, the Resource that reference it (via ConsistsOf relation) is also removed from the same context if the Facet was the last one required by the Resource in that context</li>
	 * <li><strong>ConsistsOf Relations</strong>: Remove both facet and resource from the same context</li>
	 * <li><strong>IsRelatedTo Relations</strong>: Remove both source and target resources from the same context.</li>
	 * </ul>
	 * 
	 * <strong>Propagation Constraints and Cascade Effects:</strong>
	 * <ul>
	 * <li>When an instance is removed from a context, related instances may also be automatically removed due to propagation constraints</li>
	 * <li>The response includes all instances that were removed as part of the cascade effect</li>
	 * <li>Propagation follows the schema-defined relationships and their constraints.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation for ConsistsOf Relations and Facets:</strong>
	 * <ul>
	 * <li>When operations are performed on ConsistsOf relations or Facet instances, the parent Resource is automatically validated against its schema definition</li>
	 * <li>If the Resource is no longer compliant with its schema after the remove operation, a {@link SchemaViolationException} is thrown</li>
	 * <li>This results in an HTTP 400 Bad Request response with details about the schema violation</li>
	 * <li><strong>Example scenarios that violate schema constraints:</strong></li>
	 * </ul>
	 *   • Removing a required/mandatory Facet or ConsistsOf relation from a Resource;
	 *   • Removing instances that violate cardinality constraints (min occurrences) defined in the schema;
	 *   • Operations that make the Resource incomplete or invalid according to its schema definition.
	 * <ul>
	 * <li><strong>Constraint Sources</strong>: Constraints are defined by the specific type definitions that have been installed on the system</li>
	 * <li>Schema validation ensures the integrity and consistency by enforcing the constraints defined in each type's schema during operations.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and appropriate permissions for the specific context where they want to remove instances.</li>
	 * </ul>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations on any instance in any context</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations on any instance in any context</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations only on instances accessible in their current context</li>
	 * <li>Operations are limited to contexts where they have Context-Manager role</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations only on instances accessible in their current context</li>
	 * <li>Operations are limited to instances they have access to in their authorized context</li>
	 * <li>See filtered metadata with sensitive fields obfuscated when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON array containing all instances affected by the operation</li>
	 * <li>Each instance object includes its UUID, type, and relevant metadata</li>
	 * <li>For dryRun=true: Shows what would be affected without making changes</li>
	 * <li>For dryRun=false: Shows what was actually affected by the completed operation.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Remove an EService from a specific context with actual changes
	 * UUID serviceUUID = UUID.fromString("...");
	 * UUID targetContextUUID = UUID.fromString("...");
	 * List&lt;ERElement&gt; affected = publisher.removeFromContext("EService", serviceUUID, targetContextUUID, false);
	 * 
	 * // Preview what would be affected by removing a ContactFacet from a context
	 * UUID facetUUID = UUID.fromString("...");
	 * List&lt;ERElement&gt; preview = publisher.removeFromContext("ContactFacet", facetUUID, targetContextUUID, true);
	 * </pre>
	 * 
	 * @param type The exact type name of the Information System element to remove
	 * @param instanceUUID The UUID of the instance to remove from the context
	 * @param contextUUID The UUID of the target context from which the instance should be removed
	 * @param dryRun Whether to perform a dry-run (true) or apply actual changes (false)
	 * @return List of all elements affected by this operation (including the instance itself and any related elements)
	 * @throws SchemaViolationException If the operation violates Information System model constraints (HTTP 400)
	 * @throws NotFoundException If the instance or context is not found or not accessible (HTTP 404)
	 * @throws ContextNotFoundException If the target context does not exist or is not accessible (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during the remove operation
	 */
	public List<ERElement> removeFromContext(String type, UUID instanceUUID, UUID contextUUID, Boolean dryRun)
			throws SchemaViolationException, NotFoundException, ContextNotFoundException, ResourceRegistryException;

	/**
	 * Removes an Information System element instance from a specific context using a provided instance object.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /sharing/contexts/{context-uuid}/{type-name}/{instance-uuid}?operation=REMOVE[&dryRun={true|false}][&includeMeta={true|false}][&allMeta={true|false}][&includeContexts={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Removes an existing instance (Entity or Relation) from the specified target context</li>
	 * <li>The instance must already exist and be accessible in the target context where the user is performing the request</li>
	 * <li>The target context must exist and the user must have the rights to perform such a request</li>
	 * <li>Returns the list of affected elements (the instance itself and any related elements that are also removed)</li>
	 * <li>Supports dry-run mode to preview changes without applying them</li>
	 * <li>This method accepts an Information System instance object directly, which provides type safety and convenience.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Instance successfully removed from context (or dry-run completed)</li>
	 * <li><strong>404 Not Found</strong>: Instance or target context does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: User lacks permissions to perform the request</li>
	 * <li><strong>400 Bad Request</strong>: Invalid parameters or schema validation failure</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response for affected instances</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The instance object must contain a valid UUID in its header</li>
	 * <li>Supports all Information System model types and their subtypes: Entities (Resources and all Resource subtypes, Facets and all Facet subtypes) and Relations (ConsistsOf and all ConsistsOf subtypes, IsRelatedTo and all IsRelatedTo subtypes)</li>
	 * <li>The actual type is determined from the instance object's class.</li>
	 * </ul>
	 * 
	 * <strong>Context Propagation Rules:</strong>
	 * <ul>
	 * <li><strong>Resources</strong>: When removed from a context, all their associated Facets (via ConsistsOf relations) are also removed from the same context if the remove propagation constraint is {@code propagate}</li>
	 * <li><strong>Facets</strong>: When removed from a context, the Resource that reference it (via ConsistsOf relation) is also removed from the same context if the Facet was the last one required by the Resource in that context</li>
	 * <li><strong>ConsistsOf Relations</strong>: Remove both facet and resource from the same context</li>
	 * <li><strong>IsRelatedTo Relations</strong>: Remove both source and target resources from the same context.</li>
	 * </ul>
	 * 
	 * <strong>Propagation Constraints and Cascade Effects:</strong>
	 * <ul>
	 * <li>When an instance is removed from a context, related instances may also be automatically removed due to propagation constraints</li>
	 * <li>The response includes all instances that were removed as part of the cascade effect</li>
	 * <li>Propagation follows the schema-defined relationships and their constraints.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation for ConsistsOf Relations and Facets:</strong>
	 * <ul>
	 * <li>When operations are performed on ConsistsOf relations or Facet instances, the parent Resource is automatically validated against its schema definition</li>
	 * <li>If the Resource is no longer compliant with its schema after the remove operation, a {@link SchemaViolationException} is thrown</li>
	 * <li>This results in an HTTP 400 Bad Request response with details about the schema violation</li>
	 * <li><strong>Example scenarios that violate schema constraints:</strong></li>
	 * </ul>
	 *   • Removing a required/mandatory Facet or ConsistsOf relation from a Resource;
	 *   • Removing instances that violate cardinality constraints (min occurrences) defined in the schema;
	 *   • Operations that make the Resource incomplete or invalid according to its schema definition.
	 * <ul>
	 * <li><strong>Constraint Sources</strong>: Constraints are defined by the specific type definitions that have been installed on the system</li>
	 * <li>Schema validation ensures the integrity and consistency by enforcing the constraints defined in each type's schema during operations.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and appropriate permissions for the specific context where they want to remove instances.</li>
	 * </ul>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations on any instance in any context</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations on any instance in any context</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations only on instances accessible in their current context</li>
	 * <li>Operations are limited to contexts where they have Context-Manager role</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations only on instances accessible in their current context</li>
	 * <li>Operations are limited to instances they have access to in their authorized context</li>
	 * <li>See filtered metadata with sensitive fields obfuscated when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON array containing all instances affected by the operation</li>
	 * <li>Each instance object includes its UUID, type, and relevant metadata</li>
	 * <li>For dryRun=true: Shows what would be affected without making changes</li>
	 * <li>For dryRun=false: Shows what was actually affected by the completed operation.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Remove an EService instance from a specific context
	 * EService service = publisher.read(EService.class, serviceUUID);
	 * UUID targetContextUUID = UUID.fromString("...");
	 * List&lt;ERElement&gt; affected = publisher.removeFromContext(service, targetContextUUID, false);
	 * 
	 * // Preview what would be affected by removing a ContactFacet from a context
	 * ContactFacet contact = publisher.read(ContactFacet.class, contactUUID);
	 * List&lt;ERElement&gt; preview = publisher.removeFromContext(contact, targetContextUUID, true);
	 * </pre>
	 * 
	 * @param er The instance to remove from the context (must contain a valid UUID in the header)
	 * @param contextUUID The UUID of the target context from which the instance should be removed
	 * @param dryRun Whether to perform a dry-run (true) or apply actual changes (false)
	 * @return List of all elements affected by this operation (including the instance itself and any related elements)
	 * @throws SchemaViolationException If the operation violates Information System model constraints (HTTP 400)
	 * @throws NotFoundException If the instance or context is not found or not accessible (HTTP 404)
	 * @throws ContextNotFoundException If the target context does not exist or is not accessible (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during the remove operation
	 */
	public List<ERElement> removeFromContext(ERElement er, UUID contextUUID, Boolean dryRun)
			throws SchemaViolationException, NotFoundException, ContextNotFoundException, ResourceRegistryException;

	/**
	 * Removes an ERElement instance from the current context.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /sharing/contexts/{type-name}/{uuid}?operation=REMOVE[&dryRun={true|false}][&includeMeta={true|false}][&allMeta={true|false}][&includeContexts={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Removes an existing ERElement instance from the current context derived from the authorization token</li>
	 * <li>The ERElement (Resource or Facet) must already exist and be accessible in the current context where the user is performing the request</li>
	 * <li>The user must have the rights to perform such a request in the current context</li>
	 * <li>Returns the list of affected elements (the ERElement itself and any related elements that are also removed)</li>
	 * <li>Supports dry-run mode to preview changes without applying them</li>
	 * <li>This is a convenience method that uses the current context instead of requiring an explicit context UUID</li>
	 * <li>This method works with both Resource and Facet types.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: ERElement successfully removed from current context (or dry-run completed)</li>
	 * <li><strong>404 Not Found</strong>: ERElement does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: User lacks permissions to perform the request in the current context</li>
	 * <li><strong>400 Bad Request</strong>: Invalid parameters or schema validation failure</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response for affected instances</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the ERElement and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The type name must be a valid ERElement type in the Information System model</li>
	 * <li>Supports all Resource types and their specializations (e.g., EService, HostingNode, Software)</li>
	 * <li>Supports all Facet types and their specializations (e.g., StateFacet, ContactFacet, AccessPointFacet)</li>
	 * <li>Type name is case-sensitive and must match exactly.</li>
	 * </ul>
	 * 
	 * <strong>Current Context:</strong>
	 * <ul>
	 * <li>The current context is automatically derived from the authorization token provided during client creation</li>
	 * <li>No explicit context UUID is required</li>
	 * <li>The user must have appropriate permissions in this context.</li>
	 * </ul>
	 * 
	 * <strong>Context Propagation Rules:</strong>
	 * <ul>
	 * <li><strong>Resources</strong>: When removed from a context, all their associated Facets (via ConsistsOf relations) are also removed from the same context if the remove propagation constraint is {@code propagate}</li>
	 * <li><strong>Facets</strong>: When removed from a context, the parent Resource that references it (via ConsistsOf relation) may also be removed from the same context if the Facet was required for the Resource's presence in that context</li>
	 * <li><strong>ConsistsOf Relations</strong>: All ConsistsOf relations connecting Resources to Facets are also removed from the context.</li>
	 * </ul>
	 * 
	 * <strong>Propagation Constraints and Cascade Effects:</strong>
	 * <ul>
	 * <li>When an ERElement is removed from a context, related elements may also be automatically removed due to propagation constraints</li>
	 * <li>The response includes all instances that were removed as part of the cascade effect</li>
	 * <li>Propagation follows the schema-defined relationships and their constraints.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation:</strong>
	 * <ul>
	 * <li>When an ERElement is removed from a context, the ERElement is automatically validated against its schema definition</li>
	 * <li>If the operation would violate schema constraints after the remove operation, a {@link SchemaViolationException} is thrown</li>
	 * <li>This results in an HTTP 400 Bad Request response with details about the schema violation</li>
	 * <li><strong>Example scenarios that violate schema constraints:</strong></li>
	 * </ul>
	 *   • Removing a Resource that would leave required Facets orphaned;
	 *   • Removing a required/mandatory Facet from a Resource;
	 *   • Operations that violate cardinality constraints (min occurrences) defined in the schema;
	 *   • Operations that make related elements incomplete or invalid according to their schema definition.
	 * <ul>
	 * <li><strong>Constraint Sources</strong>: Constraints are defined by the specific type definitions that have been installed on the system</li>
	 * <li>Schema validation ensures the integrity and consistency by enforcing the constraints defined in each type's schema during operations.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and appropriate permissions for the current context where they want to remove ERElements.</li>
	 * <li>The current context is derived from the authorization token provided during client creation.</li>
	 * </ul>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations on any ERElement in any context</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations on any ERElement in any context</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations only on ERElements accessible in their current context</li>
	 * <li>Operations are limited to contexts where they have Context-Manager role</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations only on ERElements accessible in their current context</li>
	 * <li>Operations are limited to ERElements they have access to in their authorized context</li>
	 * <li>See filtered metadata with sensitive fields obfuscated when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON array containing all instances affected by the operation</li>
	 * <li>Each instance object includes its UUID, type, and relevant metadata</li>
	 * <li>For dryRun=true: Shows what would be affected without making changes</li>
	 * <li>For dryRun=false: Shows what was actually affected by the completed operation.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Remove an EService from the current context with actual changes
	 * UUID serviceUUID = UUID.fromString("...");
	 * List&lt;ERElement&gt; affected = publisher.removeFromCurrentContext("EService", serviceUUID, false);
	 * 
	 * // Preview what would be affected by removing a StateFacet from the current context
	 * UUID stateFacetUUID = UUID.fromString("...");
	 * List&lt;ERElement&gt; preview = publisher.removeFromCurrentContext("StateFacet", stateFacetUUID, true);
	 * </pre>
	 * 
	 * @param type The exact type name of the ERElement to remove
	 * @param instanceUUID The UUID of the ERElement instance to remove from the current context
	 * @param dryRun Whether to perform a dry-run (true) or apply actual changes (false)
	 * @return List of all elements affected by this operation (including the ERElement itself and any related elements)
	 * @throws SchemaViolationException If the operation violates Information System model constraints (HTTP 400)
	 * @throws NotFoundException If the ERElement is not found or not accessible (HTTP 404)
	 * @throws ContextNotFoundException If the current context cannot be determined or accessed (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during the remove operation
	 */
	public List<ERElement> removeFromCurrentContext(String type, UUID instanceUUID, Boolean dryRun) 
			throws SchemaViolationException, NotFoundException, ContextNotFoundException, ResourceRegistryException;

	/**
	 * Removes an ERElement instance from the current context using a provided ERElement object.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /sharing/contexts/{type-name}/{uuid}?operation=REMOVE[&dryRun={true|false}][&includeMeta={true|false}][&allMeta={true|false}][&includeContexts={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Removes an existing ERElement instance from the current context derived from the authorization token</li>
	 * <li>The type and UUID are automatically extracted from the provided ERElement object</li>
	 * <li>The ERElement (Resource or Facet) must already exist and be accessible in the current context where the user is performing the request</li>
	 * <li>The user must have the rights to perform such a request in the current context</li>
	 * <li>Returns the list of affected elements (the ERElement itself and any related elements that are also removed)</li>
	 * <li>Supports dry-run mode to preview changes without applying them</li>
	 * <li>This is a convenience method that uses the current context instead of requiring an explicit context UUID</li>
	 * <li>This method provides compile-time type safety for both Resource and Facet types.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: ERElement successfully removed from current context (or dry-run completed)</li>
	 * <li><strong>404 Not Found</strong>: ERElement does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: User lacks permissions to perform the request in the current context</li>
	 * <li><strong>400 Bad Request</strong>: Invalid parameters or schema validation failure</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response for affected instances</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the ERElement and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The type name is automatically derived from the provided ERElement object using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>The UUID is extracted from the ERElement's header</li>
	 * <li>Supports all Resource types and their specializations (e.g., EService, HostingNode, Software)</li>
	 * <li>Supports all Facet types and their specializations (e.g., StateFacet, ContactFacet, AccessPointFacet).</li>
	 * </ul>
	 * 
	 * <strong>Current Context:</strong>
	 * <ul>
	 * <li>The current context is automatically derived from the authorization token provided during client creation</li>
	 * <li>No explicit context UUID is required</li>
	 * <li>The user must have appropriate permissions in this context.</li>
	 * </ul>
	 * 
	 * <strong>Context Propagation Rules:</strong>
	 * <ul>
	 * <li><strong>Resources</strong>: When removed from a context, all their associated Facets (via ConsistsOf relations) are also removed from the same context if the remove propagation constraint is {@code propagate}</li>
	 * <li><strong>Facets</strong>: When removed from a context, the parent Resource that references it (via ConsistsOf relation) may also be removed from the same context if the Facet was required for the Resource's presence in that context</li>
	 * <li><strong>ConsistsOf Relations</strong>: All ConsistsOf relations connecting Resources to Facets are also removed from the context.</li>
	 * </ul>
	 * 
	 * <strong>Propagation Constraints and Cascade Effects:</strong>
	 * <ul>
	 * <li>When an ERElement is removed from a context, related elements may also be automatically removed due to propagation constraints</li>
	 * <li>The response includes all instances that were removed as part of the cascade effect</li>
	 * <li>Propagation follows the schema-defined relationships and their constraints.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation:</strong>
	 * <ul>
	 * <li>When an ERElement is removed from a context, the ERElement is automatically validated against its schema definition</li>
	 * <li>If the operation would violate schema constraints after the remove operation, a {@link SchemaViolationException} is thrown</li>
	 * <li>This results in an HTTP 400 Bad Request response with details about the schema violation</li>
	 * <li><strong>Example scenarios that violate schema constraints:</strong></li>
	 * </ul>
	 *   • Removing a Resource that would leave required Facets orphaned;
	 *   • Removing a required/mandatory Facet from a Resource;
	 *   • Operations that violate cardinality constraints (min occurrences) defined in the schema;
	 *   • Operations that make related elements incomplete or invalid according to their schema definition.
	 * <ul>
	 * <li><strong>Constraint Sources</strong>: Constraints are defined by the specific type definitions that have been installed on the system</li>
	 * <li>Schema validation ensures the integrity and consistency by enforcing the constraints defined in each type's schema during operations.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and appropriate permissions for the current context where they want to remove ERElements.</li>
	 * <li>The current context is derived from the authorization token provided during client creation.</li>
	 * </ul>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations on any ERElement in any context</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations on any ERElement in any context</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations only on ERElements accessible in their current context</li>
	 * <li>Operations are limited to contexts where they have Context-Manager role</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations only on ERElements accessible in their current context</li>
	 * <li>Operations are limited to ERElements they have access to in their authorized context</li>
	 * <li>See filtered metadata with sensitive fields obfuscated when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON array containing all instances affected by the operation</li>
	 * <li>Each instance object includes its UUID, type, and relevant metadata</li>
	 * <li>For dryRun=true: Shows what would be affected without making changes</li>
	 * <li>For dryRun=false: Shows what was actually affected by the completed operation.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Remove an EService instance from the current context
	 * EService service = publisher.read(EService.class, serviceUUID);
	 * List&lt;ERElement&gt; affected = publisher.removeFromCurrentContext(service, false);
	 * 
	 * // Preview what would be affected by removing a StateFacet from the current context
	 * StateFacet facet = publisher.read(StateFacet.class, facetUUID);
	 * List&lt;ERElement&gt; preview = publisher.removeFromCurrentContext(facet, true);
	 * </pre>
	 * 
	 * @param er The ERElement instance to remove from the current context (must contain a valid UUID in the header)
	 * @param dryRun Whether to perform a dry-run (true) or apply actual changes (false)
	 * @return List of all elements affected by this operation (including the ERElement itself and any related elements)
	 * @throws SchemaViolationException If the operation violates Information System model constraints (HTTP 400)
	 * @throws NotFoundException If the ERElement is not found or not accessible (HTTP 404)
	 * @throws ContextNotFoundException If the current context cannot be determined or accessed (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during the remove operation
	 */
	public List<ERElement> removeFromCurrentContext(ERElement er, Boolean dryRun)
			throws SchemaViolationException, NotFoundException, ContextNotFoundException, ResourceRegistryException;
	
	/**
	 * Retrieves the list of contexts where a specific Information System element is present in the Resource Registry.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /instances/{type-name}/{instance-uuid}/contexts}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves all contexts where the specified element (Resource, Facet, ConsistsOf, or IsRelatedTo) is present</li>
	 * <li>The element must exist and be accessible to the user</li>
	 * <li>Returns a map where keys are context UUIDs and values are the full context paths</li>
	 * <li>No filtering, pagination, or metadata options are available</li>
	 * <li>Context information is determined by the element's presence across different VREs and organizational contexts.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Context list successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: Element does not exist with the specified type and UUID</li>
	 * <li><strong>403 Forbidden</strong>: Element exists but is not accessible for context listing in the current authorization scope</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager:</strong> Can retrieve context information for any element across the entire infrastructure</li>
	 * <li><strong>Infrastructure-Manager:</strong> Can retrieve context information for any element across the entire infrastructure</li>
	 * <li><strong>Context-Manager:</strong> Can retrieve context information for elements accessible within their managed contexts</li>
	 * <li><strong>Other Users:</strong> Can retrieve context information for elements accessible through their authorization token.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a Map where keys are context UUIDs (UUID) and values are full context paths (String)</li>
	 * <li>Empty map is returned if the element exists but is not present in any contexts</li>
	 * <li>Each context path represents the complete hierarchical path (e.g., "/gcube/devsec/devVRE").</li>
	 * </ul>
	 * 
	 * <strong>Element Context Rules:</strong>
	 * <ul>
	 * <li><strong>Resources:</strong> Present in all contexts that form the union of contexts of all their Facets</li>
	 * <li><strong>Facets:</strong> Present in the contexts where they are explicitly available</li>
	 * <li><strong>ConsistsOf Relations:</strong> Present in all contexts where their target Facets are present</li>
	 * <li><strong>IsRelatedTo Relations:</strong> Present based on their specific context assignments.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Get contexts for an EService by type and UUID
	 * UUID serviceUuid = UUID.fromString("48af15ad-7e56-4157-b624-71c98cea4f8f");
	 * Map&lt;UUID, String&gt; contexts = publisher.getElementContexts("EService", serviceUuid);
	 * 
	 * // Process the results
	 * for (Map.Entry&lt;UUID, String&gt; entry : contexts.entrySet()) {
	 *     System.out.println("Context UUID: " + entry.getKey() + " -> Path: " + entry.getValue());
	 * }
	 * </pre>
	 * 
	 * @param type The exact type name of the Information System element (e.g., "EService", "ContactFacet")
	 * @param instanceUUID The unique identifier of the element whose contexts to retrieve
	 * @return Map containing context UUID to context path mappings where the element is present
	 * @throws NotFoundException If no element is found with the specified type and UUID
	 * @throws ResourceRegistryException If an error occurs during context retrieval
	 */
	public Map<UUID, String> getElementContexts(String type, UUID instanceUUID)
			throws NotFoundException, ResourceRegistryException;

	/**
	 * Retrieves the list of contexts where a specific Information System element is present in the Resource Registry.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /instances/{type-name}/{instance-uuid}/contexts}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves all contexts where the specified element object (Resource, Facet, ConsistsOf, or IsRelatedTo) is present</li>
	 * <li>The element's type and UUID are automatically extracted from the provided object</li>
	 * <li>The element must exist and be accessible to the user</li>
	 * <li>Returns a map where keys are context UUIDs and values are the full context paths</li>
	 * <li>No filtering, pagination, or metadata options are available</li>
	 * <li>Context information is determined by the element's presence across different VREs and organizational contexts.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Context list successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: Element does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: Element exists but is not accessible for context listing in the current authorization scope</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager:</strong> Can retrieve context information for any element across the entire infrastructure</li>
	 * <li><strong>Infrastructure-Manager:</strong> Can retrieve context information for any element across the entire infrastructure</li>
	 * <li><strong>Context-Manager:</strong> Can retrieve context information for elements accessible within their managed contexts</li>
	 * <li><strong>Other Users:</strong> Can retrieve context information for elements accessible through their authorization token.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a Map where keys are context UUIDs (UUID) and values are full context paths (String)</li>
	 * <li>Empty map is returned if the element exists but is not present in any contexts</li>
	 * <li>Each context path represents the complete hierarchical path (e.g., "/gcube/devsec/devVRE").</li>
	 * </ul>
	 * 
	 * <strong>Element Context Rules:</strong>
	 * <ul>
	 * <li><strong>Resources:</strong> Present in all contexts that form the union of contexts of all their Facets</li>
	 * <li><strong>Facets:</strong> Present in the contexts where they are explicitly available</li>
	 * <li><strong>ConsistsOf Relations:</strong> Present in all contexts where their target Facets are present</li>
	 * <li><strong>IsRelatedTo Relations:</strong> Present based on their specific context assignments.</li>
	 * </ul>
	 * 
	 * <strong>Type Safety:</strong>
	 * <ul>
	 * <li>This method provides compile-time type safety by accepting any ERElement subtype</li>
	 * <li>The element's type is automatically determined from the object's class</li>
	 * <li>Supports all Information System element types including Resources, Facets, and Relations.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Assuming you have an EService instance
	 * EService service = ...; // Retrieved from previous operations
	 * 
	 * // Get contexts for the service object
	 * Map&lt;UUID, String&gt; contexts = publisher.getElementContexts(service);
	 * 
	 * // Process the results
	 * for (Map.Entry&lt;UUID, String&gt; entry : contexts.entrySet()) {
	 *     System.out.println("Context UUID: " + entry.getKey() + " -> Path: " + entry.getValue());
	 * }
	 * </pre>
	 * 
	 * @param <ERElem> The type of Information System element (Resource, Facet, ConsistsOf, or IsRelatedTo)
	 * @param er The Information System element whose contexts to retrieve
	 * @return Map containing context UUID to context path mappings where the element is present
	 * @throws NotFoundException If the specified element does not exist or is not accessible
	 * @throws ResourceRegistryException If an error occurs during context retrieval
	 */
	public <ERElem extends ERElement> Map<UUID, String> getElementContexts(ERElem er)
			throws NotFoundException, ResourceRegistryException;

	/**
	 * Adds a Resource instance to a specific context.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code PUT /instances/{resource-type-name}/{uuid}/contexts/{context-uuid}[?dryRun={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Adds an existing Resource instance to the specified target context</li>
	 * <li>The Resource must already exist and be accessible in the context where the user is performing the request</li>
	 * <li>The target context must exist and the user must have the rights to perform such a request</li>
	 * <li>Returns the list of affected elements (the Resource itself and any related elements that are also added)</li>
	 * <li>Supports dry-run mode to preview changes without applying them</li>
	 * <li>This method is specifically designed for Resource types and provides type safety.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Resource successfully added to context (or dry-run completed)</li>
	 * <li><strong>404 Not Found</strong>: Resource or target context does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: User lacks permissions to perform the request</li>
	 * <li><strong>400 Bad Request</strong>: Invalid parameters or schema validation failure</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>forceAddToContext</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Forces the addition of a Resource to a context even if the Resource is not present in the current context</li>
	 * <li>Default value: false (standard validation applies)</li>
	 * <li>Usage: {@code publisher.addHeader(SharingPath.FORCE_ADD_TO_CONTEXT_QUERY_PARAMETER, true)}</li>
	 * <li>Values: true | false</li>
	 * <li>When true: Allows adding a Resource to a context even if the Resource exists only in other contexts</li>
	 * <li>When false: Standard validation applies - Resource must be accessible in the current context for ADD operations</li>
	 * <li><strong>Authorization Required</strong>: Only available to IS-Manager and Infrastructure-Manager</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response for affected instances</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the Resource and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The type name must be a valid Resource type in the Information System model</li>
	 * <li>Supports all Resource types and their specializations (e.g., EService, HostingNode, Software)</li>
	 * <li>Type name is case-sensitive and must match exactly.</li>
	 * </ul>
	 * 
	 * <strong>Resource-Specific Context Propagation Rules:</strong>
	 * <ul>
	 * <li><strong>Resources</strong>: When added to a context, all their associated Facets (via ConsistsOf relations) are also added to the same context if the add propagation constraint is {@code propagate}</li>
	 * <li><strong>Associated Facets</strong>: All Facets connected to the Resource via ConsistsOf relations are automatically included</li>
	 * <li><strong>ConsistsOf Relations</strong>: All ConsistsOf relations connecting the Resource to its Facets are also added to the context.</li>
	 * </ul>
	 * 
	 * <strong>Propagation Constraints and Cascade Effects:</strong>
	 * <ul>
	 * <li>When a Resource is added to a context, all related Facets may also be automatically added due to propagation constraints</li>
	 * <li>The response includes all instances that were added as part of the cascade effect</li>
	 * <li>Propagation follows the schema-defined relationships and their constraints.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation for Resource and ConsistsOf Relations:</strong>
	 * <ul>
	 * <li>When a Resource is added to a context, the Resource is automatically validated against its schema definition</li>
	 * <li>If the Resource is no longer compliant with its schema after the add operation, a {@link SchemaViolationException} is thrown</li>
	 * <li>This results in an HTTP 400 Bad Request response with details about the schema violation</li>
	 * <li><strong>Example scenarios that violate schema constraints:</strong></li>
	 * </ul>
	 *   • Adding a Resource that violates cardinality constraints defined in the schema;
	 *   • Operations that make the Resource incomplete or invalid according to its schema definition.
	 * <ul>
	 * <li><strong>Constraint Sources</strong>: Constraints are defined by the specific Resource type definitions that have been installed on the system</li>
	 * <li>Schema validation ensures the integrity and consistency by enforcing the constraints defined in each Resource type's schema during operations.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and appropriate permissions for the specific context where they want to add Resources.</li>
	 * </ul>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD operations on any Resource in any context</li>
	 * <li>Can use forceAddToContext=true to add Resources from other contexts</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD operations on any Resource in any context</li>
	 * <li>Can use forceAddToContext=true to add Resources from other contexts</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD operations only on Resources accessible in their current context</li>
	 * <li>Cannot use forceAddToContext=true (parameter is ignored if provided)</li>
	 * <li>Operations are limited to contexts where they have Context-Manager role</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can perform ADD operations only on Resources accessible in their current context</li>
	 * <li>Cannot use forceAddToContext=true (parameter is ignored if provided)</li>
	 * <li>Operations are limited to Resources they have access to in their authorized context</li>
	 * <li>See filtered metadata with sensitive fields obfuscated when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON array containing all instances affected by the operation</li>
	 * <li>Each instance object includes its UUID, type, and relevant metadata</li>
	 * <li>For dryRun=true: Shows what would be affected without making changes</li>
	 * <li>For dryRun=false: Shows what was actually affected by the completed operation.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Add an EService to a specific context with actual changes
	 * UUID serviceUUID = UUID.fromString("...");
	 * UUID targetContextUUID = UUID.fromString("...");
	 * List&lt;ERElement&gt; affected = publisher.addResourceToContext("EService", serviceUUID, targetContextUUID, false);
	 * 
	 * // Preview what would be affected by adding a HostingNode to a context
	 * UUID hostUUID = UUID.fromString("...");
	 * List&lt;ERElement&gt; preview = publisher.addResourceToContext("HostingNode", hostUUID, targetContextUUID, true);
	 * </pre>
	 * 
	 * @param resourceType The exact type name of the Resource to add
	 * @param resourceUUID The UUID of the Resource instance to add to the context
	 * @param contextUUID The UUID of the target context where the Resource should be added
	 * @param dryRun Whether to perform a dry-run (true) or apply actual changes (false)
	 * @return List of all elements affected by this operation (including the Resource itself and any related elements)
	 * @throws SchemaViolationException If the operation violates Information System model constraints (HTTP 400)
	 * @throws ResourceNotFoundException If the Resource is not found or not accessible (HTTP 404)
	 * @throws ContextNotFoundException If the target context does not exist or is not accessible (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during the add operation
	 */
	public List<ERElement> addResourceToContext(String resourceType, UUID resourceUUID, UUID contextUUID, Boolean dryRun)
			throws SchemaViolationException, ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException;

	/**
	 * Adds a Resource instance to a specific context using a provided Resource object.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code PUT /instances/{resource-type-name}/{uuid}/contexts/{context-uuid}[?dryRun={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Adds an existing Resource instance to the specified target context</li>
	 * <li>The type and UUID are automatically extracted from the provided Resource object</li>
	 * <li>The Resource must already exist and be accessible in the context where the user is performing the request</li>
	 * <li>The target context must exist and the user must have the rights to perform such a request</li>
	 * <li>Returns the list of affected elements (the Resource itself and any related elements that are also added)</li>
	 * <li>Supports dry-run mode to preview changes without applying them</li>
	 * <li>This method provides compile-time type safety for Resource types.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Resource successfully added to context (or dry-run completed)</li>
	 * <li><strong>404 Not Found</strong>: Resource or target context does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: User lacks permissions to perform the request</li>
	 * <li><strong>400 Bad Request</strong>: Invalid parameters or schema validation failure</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>forceAddToContext</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Forces the addition of a Resource to a context even if the Resource is not present in the current context</li>
	 * <li>Default value: false (standard validation applies)</li>
	 * <li>Usage: {@code publisher.addHeader(SharingPath.FORCE_ADD_TO_CONTEXT_QUERY_PARAMETER, true)}</li>
	 * <li>Values: true | false</li>
	 * <li>When true: Allows adding a Resource to a context even if the Resource exists only in other contexts</li>
	 * <li>When false: Standard validation applies - Resource must be accessible in the current context for ADD operations</li>
	 * <li><strong>Authorization Required</strong>: Only available to IS-Manager and Infrastructure-Manager</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response for affected instances</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the Resource and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The type name is automatically derived from the provided Resource object using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>The UUID is extracted from the Resource's header</li>
	 * <li>Supports all Resource types and their specializations (e.g., EService, HostingNode, Software).</li>
	 * </ul>
	 * 
	 * <strong>Resource-Specific Context Propagation Rules:</strong>
	 * <ul>
	 * <li><strong>Resources</strong>: When added to a context, all their associated Facets (via ConsistsOf relations) are also added to the same context if the add propagation constraint is {@code propagate}</li>
	 * <li><strong>Associated Facets</strong>: All Facets connected to the Resource via ConsistsOf relations are automatically included</li>
	 * <li><strong>ConsistsOf Relations</strong>: All ConsistsOf relations connecting the Resource to its Facets are also added to the context.</li>
	 * </ul>
	 * 
	 * <strong>Propagation Constraints and Cascade Effects:</strong>
	 * <ul>
	 * <li>When a Resource is added to a context, all related Facets may also be automatically added due to propagation constraints</li>
	 * <li>The response includes all instances that were added as part of the cascade effect</li>
	 * <li>Propagation follows the schema-defined relationships and their constraints.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation for Resource and ConsistsOf Relations:</strong>
	 * <ul>
	 * <li>When a Resource is added to a context, the Resource is automatically validated against its schema definition</li>
	 * <li>If the Resource is no longer compliant with its schema after the add operation, a {@link SchemaViolationException} is thrown</li>
	 * <li>This results in an HTTP 400 Bad Request response with details about the schema violation</li>
	 * <li><strong>Example scenarios that violate schema constraints:</strong></li>
	 * </ul>
	 *   • Adding a Resource that violates cardinality constraints defined in the schema;
	 *   • Operations that make the Resource incomplete or invalid according to its schema definition.
	 * <ul>
	 * <li><strong>Constraint Sources</strong>: Constraints are defined by the specific Resource type definitions that have been installed on the system</li>
	 * <li>Schema validation ensures the integrity and consistency by enforcing the constraints defined in each Resource type's schema during operations.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and appropriate permissions for the specific context where they want to add Resources.</li>
	 * </ul>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD operations on any Resource in any context</li>
	 * <li>Can use forceAddToContext=true to add Resources from other contexts</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD operations on any Resource in any context</li>
	 * <li>Can use forceAddToContext=true to add Resources from other contexts</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD operations only on Resources accessible in their current context</li>
	 * <li>Cannot use forceAddToContext=true (parameter is ignored if provided)</li>
	 * <li>Operations are limited to contexts where they have Context-Manager role</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can perform ADD operations only on Resources accessible in their current context</li>
	 * <li>Cannot use forceAddToContext=true (parameter is ignored if provided)</li>
	 * <li>Operations are limited to Resources they have access to in their authorized context</li>
	 * <li>See filtered metadata with sensitive fields obfuscated when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON array containing all instances affected by the operation</li>
	 * <li>Each instance object includes its UUID, type, and relevant metadata</li>
	 * <li>For dryRun=true: Shows what would be affected without making changes</li>
	 * <li>For dryRun=false: Shows what was actually affected by the completed operation.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Add an EService instance to a specific context
	 * EService service = new EService();
	 * service.setHeader(new Header(serviceUUID));
	 * UUID targetContextUUID = UUID.fromString("...");
	 * List&lt;ERElement&gt; affected = publisher.addResourceToContext(service, targetContextUUID, false);
	 * 
	 * // Preview what would be affected by adding a HostingNode to a context
	 * HostingNode host = publisher.read(HostingNode.class, hostUUID);
	 * List&lt;ERElement&gt; preview = publisher.addResourceToContext(host, targetContextUUID, true);
	 * </pre>
	 * 
	 * @param <R> The specific type of Resource to add
	 * @param resource The Resource instance to add to the context (must contain a valid UUID in the header)
	 * @param contextUUID The UUID of the target context where the Resource should be added
	 * @param dryRun Whether to perform a dry-run (true) or apply actual changes (false)
	 * @return List of all elements affected by this operation (including the Resource itself and any related elements)
	 * @throws SchemaViolationException If the operation violates Information System model constraints (HTTP 400)
	 * @throws ResourceNotFoundException If the Resource is not found or not accessible (HTTP 404)
	 * @throws ContextNotFoundException If the target context does not exist or is not accessible (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during the add operation
	 */
	public <R extends Resource> List<ERElement> addResourceToContext(R resource, UUID contextUUID, Boolean dryRun)
			throws SchemaViolationException, ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException;

	/**
	 * Adds a Resource instance to the current context.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code PUT /instances/{resource-type-name}/{uuid}/contexts[?dryRun={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Adds an existing Resource instance to the current context derived from the authorization token</li>
	 * <li>The Resource must already exist and be accessible in the context where the user is performing the request</li>
	 * <li>The user must have the rights to perform such a request in the current context</li>
	 * <li>Returns the list of affected elements (the Resource itself and any related elements that are also added)</li>
	 * <li>Supports dry-run mode to preview changes without applying them</li>
	 * <li>This is a convenience method that uses the current context instead of requiring an explicit context UUID</li>
	 * <li>This method is specifically designed for Resource types and provides type safety.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Resource successfully added to current context (or dry-run completed)</li>
	 * <li><strong>404 Not Found</strong>: Resource does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: User lacks permissions to perform the request in the current context</li>
	 * <li><strong>400 Bad Request</strong>: Invalid parameters or schema validation failure</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>forceAddToContext</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Forces the addition of a Resource to a context even if the Resource is not present in the current context</li>
	 * <li>Default value: false (standard validation applies)</li>
	 * <li>Usage: {@code publisher.addHeader(SharingPath.FORCE_ADD_TO_CONTEXT_QUERY_PARAMETER, true)}</li>
	 * <li>Values: true | false</li>
	 * <li>When true: Allows adding a Resource to a context even if the Resource exists only in other contexts</li>
	 * <li>When false: Standard validation applies - Resource must be accessible in the current context for ADD operations</li>
	 * <li><strong>Authorization Required</strong>: Only available to IS-Manager and Infrastructure-Manager</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response for affected instances</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the Resource and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The type name must be a valid Resource type in the Information System model</li>
	 * <li>Supports all Resource types and their specializations (e.g., EService, HostingNode, Software)</li>
	 * <li>Type name is case-sensitive and must match exactly.</li>
	 * </ul>
	 * 
	 * <strong>Current Context:</strong>
	 * <ul>
	 * <li>The current context is automatically derived from the authorization token provided during client creation</li>
	 * <li>No explicit context UUID is required</li>
	 * <li>The user must have appropriate permissions in this context.</li>
	 * </ul>
	 * 
	 * <strong>Resource-Specific Context Propagation Rules:</strong>
	 * <ul>
	 * <li><strong>Resources</strong>: When added to a context, all their associated Facets (via ConsistsOf relations) are also added to the same context if the add propagation constraint is {@code propagate}</li>
	 * <li><strong>Associated Facets</strong>: All Facets connected to the Resource via ConsistsOf relations are automatically included</li>
	 * <li><strong>ConsistsOf Relations</strong>: All ConsistsOf relations connecting the Resource to its Facets are also added to the context.</li>
	 * </ul>
	 * 
	 * <strong>Propagation Constraints and Cascade Effects:</strong>
	 * <ul>
	 * <li>When a Resource is added to a context, all related Facets may also be automatically added due to propagation constraints</li>
	 * <li>The response includes all instances that were added as part of the cascade effect</li>
	 * <li>Propagation follows the schema-defined relationships and their constraints.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation for Resource and ConsistsOf Relations:</strong>
	 * <ul>
	 * <li>When a Resource is added to a context, the Resource is automatically validated against its schema definition</li>
	 * <li>If the Resource is no longer compliant with its schema after the add operation, a {@link SchemaViolationException} is thrown</li>
	 * <li>This results in an HTTP 400 Bad Request response with details about the schema violation</li>
	 * <li><strong>Example scenarios that violate schema constraints:</strong></li>
	 * </ul>
	 *   • Adding a Resource that violates cardinality constraints defined in the schema;
	 *   • Operations that make the Resource incomplete or invalid according to its schema definition.
	 * <ul>
	 * <li><strong>Constraint Sources</strong>: Constraints are defined by the specific Resource type definitions that have been installed on the system</li>
	 * <li>Schema validation ensures the integrity and consistency by enforcing the constraints defined in each Resource type's schema during operations.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and appropriate permissions for the current context where they want to add Resources.</li>
	 * <li>The current context is derived from the authorization token provided during client creation.</li>
	 * </ul>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD operations on any Resource in any context</li>
	 * <li>Can use forceAddToContext=true to add Resources from other contexts</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD operations on any Resource in any context</li>
	 * <li>Can use forceAddToContext=true to add Resources from other contexts</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD operations only on Resources accessible in their current context</li>
	 * <li>Cannot use forceAddToContext=true (parameter is ignored if provided)</li>
	 * <li>Operations are limited to contexts where they have Context-Manager role</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can perform ADD operations only on Resources accessible in their current context</li>
	 * <li>Cannot use forceAddToContext=true (parameter is ignored if provided)</li>
	 * <li>Operations are limited to Resources they have access to in their authorized context</li>
	 * <li>See filtered metadata with sensitive fields obfuscated when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON array containing all instances affected by the operation</li>
	 * <li>Each instance object includes its UUID, type, and relevant metadata</li>
	 * <li>For dryRun=true: Shows what would be affected without making changes</li>
	 * <li>For dryRun=false: Shows what was actually affected by the completed operation.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Add an EService to the current context with actual changes
	 * UUID serviceUUID = UUID.fromString("...");
	 * List&lt;ERElement&gt; affected = publisher.addResourceToCurrentContext("EService", serviceUUID, false);
	 * 
	 * // Preview what would be affected by adding a HostingNode to the current context
	 * UUID hostUUID = UUID.fromString("...");
	 * List&lt;ERElement&gt; preview = publisher.addResourceToCurrentContext("HostingNode", hostUUID, true);
	 * </pre>
	 * 
	 * @param resourceType The exact type name of the Resource to add
	 * @param resourceUUID The UUID of the Resource instance to add to the current context
	 * @param dryRun Whether to perform a dry-run (true) or apply actual changes (false)
	 * @return List of all elements affected by this operation (including the Resource itself and any related elements)
	 * @throws SchemaViolationException If the operation violates Information System model constraints (HTTP 400)
	 * @throws ResourceNotFoundException If the Resource is not found or not accessible (HTTP 404)
	 * @throws ContextNotFoundException If the current context cannot be determined or accessed (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during the add operation
	 */
	public List<ERElement> addResourceToCurrentContext(String resourceType, UUID resourceUUID, Boolean dryRun) 
			throws SchemaViolationException, ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException;

	/**
	 * Adds a Resource instance to the current context using a provided Resource object.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code PUT /instances/{resource-type-name}/{uuid}/contexts[?dryRun={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Adds an existing Resource instance to the current context derived from the authorization token</li>
	 * <li>The type and UUID are automatically extracted from the provided Resource object</li>
	 * <li>The Resource must already exist and be accessible in the context where the user is performing the request</li>
	 * <li>The user must have the rights to perform such a request in the current context</li>
	 * <li>Returns the list of affected elements (the Resource itself and any related elements that are also added)</li>
	 * <li>Supports dry-run mode to preview changes without applying them</li>
	 * <li>This is a convenience method that uses the current context instead of requiring an explicit context UUID</li>
	 * <li>This method provides compile-time type safety for Resource types.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Resource successfully added to current context (or dry-run completed)</li>
	 * <li><strong>404 Not Found</strong>: Resource does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: User lacks permissions to perform the request in the current context</li>
	 * <li><strong>400 Bad Request</strong>: Invalid parameters or schema validation failure</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>forceAddToContext</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Forces the addition of a Resource to a context even if the Resource is not present in the current context</li>
	 * <li>Default value: false (standard validation applies)</li>
	 * <li>Usage: {@code publisher.addHeader(SharingPath.FORCE_ADD_TO_CONTEXT_QUERY_PARAMETER, true)}</li>
	 * <li>Values: true | false</li>
	 * <li>When true: Allows adding a Resource to a context even if the Resource exists only in other contexts</li>
	 * <li>When false: Standard validation applies - Resource must be accessible in the current context for ADD operations</li>
	 * <li><strong>Authorization Required</strong>: Only available to IS-Manager and Infrastructure-Manager</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response for affected instances</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the Resource and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The type name is automatically derived from the provided Resource object using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>The UUID is extracted from the Resource's header</li>
	 * <li>Supports all Resource types and their specializations (e.g., EService, HostingNode, Software).</li>
	 * </ul>
	 * 
	 * <strong>Current Context:</strong>
	 * <ul>
	 * <li>The current context is automatically derived from the authorization token provided during client creation</li>
	 * <li>No explicit context UUID is required</li>
	 * <li>The user must have appropriate permissions in this context.</li>
	 * </ul>
	 * 
	 * <strong>Resource-Specific Context Propagation Rules:</strong>
	 * <ul>
	 * <li><strong>Resources</strong>: When added to a context, all their associated Facets (via ConsistsOf relations) are also added to the same context if the add propagation constraint is {@code propagate}</li>
	 * <li><strong>Associated Facets</strong>: All Facets connected to the Resource via ConsistsOf relations are automatically included</li>
	 * <li><strong>ConsistsOf Relations</strong>: All ConsistsOf relations connecting the Resource to its Facets are also added to the context.</li>
	 * </ul>
	 * 
	 * <strong>Propagation Constraints and Cascade Effects:</strong>
	 * <ul>
	 * <li>When a Resource is added to a context, all related Facets may also be automatically added due to propagation constraints</li>
	 * <li>The response includes all instances that were added as part of the cascade effect</li>
	 * <li>Propagation follows the schema-defined relationships and their constraints.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation for Resource and ConsistsOf Relations:</strong>
	 * <ul>
	 * <li>When a Resource is added to a context, the Resource is automatically validated against its schema definition</li>
	 * <li>If the Resource is no longer compliant with its schema after the add operation, a {@link SchemaViolationException} is thrown</li>
	 * <li>This results in an HTTP 400 Bad Request response with details about the schema violation</li>
	 * <li><strong>Example scenarios that violate schema constraints:</strong></li>
	 * </ul>
	 *   • Adding a Resource that violates cardinality constraints defined in the schema;
	 *   • Operations that make the Resource incomplete or invalid according to its schema definition.
	 * <ul>
	 * <li><strong>Constraint Sources</strong>: Constraints are defined by the specific Resource type definitions that have been installed on the system</li>
	 * <li>Schema validation ensures the integrity and consistency by enforcing the constraints defined in each Resource type's schema during operations.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and appropriate permissions for the current context where they want to add Resources.</li>
	 * <li>The current context is derived from the authorization token provided during client creation.</li>
	 * </ul>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD operations on any Resource in any context</li>
	 * <li>Can use forceAddToContext=true to add Resources from other contexts</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD operations on any Resource in any context</li>
	 * <li>Can use forceAddToContext=true to add Resources from other contexts</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD operations only on Resources accessible in their current context</li>
	 * <li>Cannot use forceAddToContext=true (parameter is ignored if provided)</li>
	 * <li>Operations are limited to contexts where they have Context-Manager role</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can perform ADD operations only on Resources accessible in their current context</li>
	 * <li>Cannot use forceAddToContext=true (parameter is ignored if provided)</li>
	 * <li>Operations are limited to Resources they have access to in their authorized context</li>
	 * <li>See filtered metadata with sensitive fields obfuscated when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON array containing all instances affected by the operation</li>
	 * <li>Each instance object includes its UUID, type, and relevant metadata</li>
	 * <li>For dryRun=true: Shows what would be affected without making changes</li>
	 * <li>For dryRun=false: Shows what was actually affected by the completed operation.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Add an EService instance to the current context
	 * EService service = new EService();
	 * service.setHeader(new Header(serviceUUID));
	 * List&lt;ERElement&gt; affected = publisher.addResourceToCurrentContext(service, false);
	 * 
	 * // Preview what would be affected by adding a HostingNode to the current context
	 * HostingNode host = publisher.read(HostingNode.class, hostUUID);
	 * List&lt;ERElement&gt; preview = publisher.addResourceToCurrentContext(host, true);
	 * </pre>
	 * 
	 * @param <R> The specific type of Resource to add
	 * @param resource The Resource instance to add to the current context (must contain a valid UUID in the header)
	 * @param dryRun Whether to perform a dry-run (true) or apply actual changes (false)
	 * @return List of all elements affected by this operation (including the Resource itself and any related elements)
	 * @throws SchemaViolationException If the operation violates Information System model constraints (HTTP 400)
	 * @throws ResourceNotFoundException If the Resource is not found or not accessible (HTTP 404)
	 * @throws ContextNotFoundException If the current context cannot be determined or accessed (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during the add operation
	 */
	public <R extends Resource> List<ERElement> addResourceToCurrentContext(R resource, Boolean dryRun) 
			throws SchemaViolationException, ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException;

	/**
	 * Removes a Resource instance from a specific context.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /sharing/contexts/{context-uuid}/{resource-type-name}/{uuid}?operation=REMOVE[&dryRun={true|false}][&includeMeta={true|false}][&allMeta={true|false}][&includeContexts={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Removes an existing Resource instance from the specified target context</li>
	 * <li>The Resource must already exist and be accessible in the target context where the user is performing the request</li>
	 * <li>The target context must exist and the user must have the rights to perform such a request</li>
	 * <li>Returns the list of affected elements (the Resource itself and any related elements that are also removed)</li>
	 * <li>Supports dry-run mode to preview changes without applying them</li>
	 * <li>This method is specifically designed for Resource types and provides type safety.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Resource successfully removed from context (or dry-run completed)</li>
	 * <li><strong>404 Not Found</strong>: Resource or target context does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: User lacks permissions to perform the request</li>
	 * <li><strong>400 Bad Request</strong>: Invalid parameters or schema validation failure</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response for affected instances</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the Resource and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The type name must be a valid Resource type in the Information System model</li>
	 * <li>Supports all Resource types and their specializations (e.g., EService, HostingNode, Software)</li>
	 * <li>Type name is case-sensitive and must match exactly.</li>
	 * </ul>
	 * 
	 * <strong>Resource-Specific Context Propagation Rules:</strong>
	 * <ul>
	 * <li><strong>Resources</strong>: When removed from a context, all their associated Facets (via ConsistsOf relations) are also removed from the same context if the remove propagation constraint is {@code propagate}</li>
	 * <li><strong>Associated Facets</strong>: All Facets connected to the Resource via ConsistsOf relations are automatically removed if they become orphaned</li>
	 * <li><strong>ConsistsOf Relations</strong>: All ConsistsOf relations connecting the Resource to its Facets are also removed from the context.</li>
	 * </ul>
	 * 
	 * <strong>Propagation Constraints and Cascade Effects:</strong>
	 * <ul>
	 * <li>When a Resource is removed from a context, all related Facets may also be automatically removed due to propagation constraints</li>
	 * <li>The response includes all instances that were removed as part of the cascade effect</li>
	 * <li>Propagation follows the schema-defined relationships and their constraints.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation for Resource and ConsistsOf Relations:</strong>
	 * <ul>
	 * <li>When a Resource is removed from a context, the Resource is automatically validated against its schema definition</li>
	 * <li>If the operation would violate schema constraints after the remove operation, a {@link SchemaViolationException} is thrown</li>
	 * <li>This results in an HTTP 400 Bad Request response with details about the schema violation</li>
	 * <li><strong>Example scenarios that violate schema constraints:</strong></li>
	 * </ul>
	 *   • Removing a Resource that would leave required Facets orphaned;
	 *   • Operations that violate cardinality constraints (min occurrences) defined in the schema;
	 *   • Operations that make related Resources incomplete or invalid according to their schema definition.
	 * <ul>
	 * <li><strong>Constraint Sources</strong>: Constraints are defined by the specific Resource type definitions that have been installed on the system</li>
	 * <li>Schema validation ensures the integrity and consistency by enforcing the constraints defined in each Resource type's schema during operations.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and appropriate permissions for the specific context where they want to remove Resources.</li>
	 * </ul>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations on any Resource in any context</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations on any Resource in any context</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations only on Resources accessible in their current context</li>
	 * <li>Operations are limited to contexts where they have Context-Manager role</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations only on Resources accessible in their current context</li>
	 * <li>Operations are limited to Resources they have access to in their authorized context</li>
	 * <li>See filtered metadata with sensitive fields obfuscated when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON array containing all instances affected by the operation</li>
	 * <li>Each instance object includes its UUID, type, and relevant metadata</li>
	 * <li>For dryRun=true: Shows what would be affected without making changes</li>
	 * <li>For dryRun=false: Shows what was actually affected by the completed operation.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Remove an EService from a specific context with actual changes
	 * UUID serviceUUID = UUID.fromString("...");
	 * UUID targetContextUUID = UUID.fromString("...");
	 * List&lt;ERElement&gt; affected = publisher.removeResourceFromContext("EService", serviceUUID, targetContextUUID, false);
	 * 
	 * // Preview what would be affected by removing a HostingNode from a context
	 * UUID hostUUID = UUID.fromString("...");
	 * List&lt;ERElement&gt; preview = publisher.removeResourceFromContext("HostingNode", hostUUID, targetContextUUID, true);
	 * </pre>
	 * 
	 * @param resourceType The exact type name of the Resource to remove
	 * @param resourceUUID The UUID of the Resource instance to remove from the context
	 * @param contextUUID The UUID of the target context from which the Resource should be removed
	 * @param dryRun Whether to perform a dry-run (true) or apply actual changes (false)
	 * @return List of all elements affected by this operation (including the Resource itself and any related elements)
	 * @throws SchemaViolationException If the operation violates Information System model constraints (HTTP 400)
	 * @throws ResourceNotFoundException If the Resource is not found or not accessible (HTTP 404)
	 * @throws ContextNotFoundException If the target context does not exist or is not accessible (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during the remove operation
	 */
	public List<ERElement> removeResourceFromContext(String resourceType, UUID resourceUUID, UUID contextUUID, Boolean dryRun)
			throws SchemaViolationException, ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException;

	/**
	 * Removes a Resource instance from a specific context using a provided Resource object.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /sharing/contexts/{context-uuid}/{resource-type-name}/{uuid}?operation=REMOVE[&dryRun={true|false}][&includeMeta={true|false}][&allMeta={true|false}][&includeContexts={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Removes an existing Resource instance from the specified target context</li>
	 * <li>The Resource must already exist and be accessible in the target context where the user is performing the request</li>
	 * <li>The target context must exist and the user must have the rights to perform such a request</li>
	 * <li>Returns the list of affected elements (the Resource itself and any related elements that are also removed)</li>
	 * <li>Supports dry-run mode to preview changes without applying them</li>
	 * <li>This method accepts a Resource instance object directly, which provides type safety and convenience</li>
	 * <li>This method is specifically designed for Resource types and provides type safety.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Resource successfully removed from context (or dry-run completed)</li>
	 * <li><strong>404 Not Found</strong>: Resource or target context does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: User lacks permissions to perform the request</li>
	 * <li><strong>400 Bad Request</strong>: Invalid parameters or schema validation failure</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response for affected instances</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the Resource and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The Resource object must contain a valid UUID in its header</li>
	 * <li>Supports all Resource types and their specializations (e.g., EService, HostingNode, Software)</li>
	 * <li>The actual type is determined from the Resource object's class.</li>
	 * </ul>
	 * 
	 * <strong>Resource-Specific Context Propagation Rules:</strong>
	 * <ul>
	 * <li><strong>Resources</strong>: When removed from a context, all their associated Facets (via ConsistsOf relations) are also removed from the same context if the remove propagation constraint is {@code propagate}</li>
	 * <li><strong>Associated Facets</strong>: All Facets connected to the Resource via ConsistsOf relations are automatically removed if they become orphaned</li>
	 * <li><strong>ConsistsOf Relations</strong>: All ConsistsOf relations connecting the Resource to its Facets are also removed from the context.</li>
	 * </ul>
	 * 
	 * <strong>Propagation Constraints and Cascade Effects:</strong>
	 * <ul>
	 * <li>When a Resource is removed from a context, all related Facets may also be automatically removed due to propagation constraints</li>
	 * <li>The response includes all instances that were removed as part of the cascade effect</li>
	 * <li>Propagation follows the schema-defined relationships and their constraints.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation for Resource and ConsistsOf Relations:</strong>
	 * <ul>
	 * <li>When a Resource is removed from a context, the Resource is automatically validated against its schema definition</li>
	 * <li>If the operation would violate schema constraints after the remove operation, a {@link SchemaViolationException} is thrown</li>
	 * <li>This results in an HTTP 400 Bad Request response with details about the schema violation</li>
	 * <li><strong>Example scenarios that violate schema constraints:</strong></li>
	 * </ul>
	 *   • Removing a Resource that would leave required Facets orphaned;
	 *   • Operations that violate cardinality constraints (min occurrences) defined in the schema;
	 *   • Operations that make related Resources incomplete or invalid according to their schema definition.
	 * <ul>
	 * <li><strong>Constraint Sources</strong>: Constraints are defined by the specific Resource type definitions that have been installed on the system</li>
	 * <li>Schema validation ensures the integrity and consistency by enforcing the constraints defined in each Resource type's schema during operations.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and appropriate permissions for the specific context where they want to remove Resources.</li>
	 * </ul>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations on any Resource in any context</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations on any Resource in any context</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations only on Resources accessible in their current context</li>
	 * <li>Operations are limited to contexts where they have Context-Manager role</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations only on Resources accessible in their current context</li>
	 * <li>Operations are limited to Resources they have access to in their authorized context</li>
	 * <li>See filtered metadata with sensitive fields obfuscated when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON array containing all instances affected by the operation</li>
	 * <li>Each instance object includes its UUID, type, and relevant metadata</li>
	 * <li>For dryRun=true: Shows what would be affected without making changes</li>
	 * <li>For dryRun=false: Shows what was actually affected by the completed operation.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Remove an EService instance from a specific context
	 * EService service = publisher.read(EService.class, serviceUUID);
	 * UUID targetContextUUID = UUID.fromString("...");
	 * List&lt;ERElement&gt; affected = publisher.removeResourceFromContext(service, targetContextUUID, false);
	 * 
	 * // Preview what would be affected by removing a HostingNode from a context
	 * HostingNode host = publisher.read(HostingNode.class, hostUUID);
	 * List&lt;ERElement&gt; preview = publisher.removeResourceFromContext(host, targetContextUUID, true);
	 * </pre>
	 * 
	 * @param <R> The type of Resource (EService, HostingNode, Software, etc.)
	 * @param resource The Resource instance to remove from the context (must contain a valid UUID in the header)
	 * @param contextUUID The UUID of the target context from which the Resource should be removed
	 * @param dryRun Whether to perform a dry-run (true) or apply actual changes (false)
	 * @return List of all elements affected by this operation (including the Resource itself and any related elements)
	 * @throws SchemaViolationException If the operation violates Information System model constraints (HTTP 400)
	 * @throws ResourceNotFoundException If the Resource is not found or not accessible (HTTP 404)
	 * @throws ContextNotFoundException If the target context does not exist or is not accessible (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during the remove operation
	 */
	public <R extends Resource> List<ERElement> removeResourceFromContext(R resource, UUID contextUUID, Boolean dryRun)
			throws SchemaViolationException, ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException;

	/**
	 * Removes a Resource instance from the current context.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /sharing/contexts/{resource-type-name}/{uuid}?operation=REMOVE[&dryRun={true|false}][&includeMeta={true|false}][&allMeta={true|false}][&includeContexts={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Removes an existing Resource instance from the current context derived from the authorization token</li>
	 * <li>The Resource must already exist and be accessible in the current context where the user is performing the request</li>
	 * <li>The user must have the rights to perform such a request in the current context</li>
	 * <li>Returns the list of affected elements (the Resource itself and any related elements that are also removed)</li>
	 * <li>Supports dry-run mode to preview changes without applying them</li>
	 * <li>This is a convenience method that uses the current context instead of requiring an explicit context UUID</li>
	 * <li>This method is specifically designed for Resource types and provides type safety.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Resource successfully removed from current context (or dry-run completed)</li>
	 * <li><strong>404 Not Found</strong>: Resource does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: User lacks permissions to perform the request in the current context</li>
	 * <li><strong>400 Bad Request</strong>: Invalid parameters or schema validation failure</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response for affected instances</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the Resource and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The type name must be a valid Resource type in the Information System model</li>
	 * <li>Supports all Resource types and their specializations (e.g., EService, HostingNode, Software)</li>
	 * <li>Type name is case-sensitive and must match exactly.</li>
	 * </ul>
	 * 
	 * <strong>Current Context:</strong>
	 * <ul>
	 * <li>The current context is automatically derived from the authorization token provided during client creation</li>
	 * <li>No explicit context UUID is required</li>
	 * <li>The user must have appropriate permissions in this context.</li>
	 * </ul>
	 * 
	 * <strong>Resource-Specific Context Propagation Rules:</strong>
	 * <ul>
	 * <li><strong>Resources</strong>: When removed from a context, all their associated Facets (via ConsistsOf relations) are also removed from the same context if the remove propagation constraint is {@code propagate}</li>
	 * <li><strong>Associated Facets</strong>: All Facets connected to the Resource via ConsistsOf relations are automatically removed if they become orphaned</li>
	 * <li><strong>ConsistsOf Relations</strong>: All ConsistsOf relations connecting the Resource to its Facets are also removed from the context.</li>
	 * </ul>
	 * 
	 * <strong>Propagation Constraints and Cascade Effects:</strong>
	 * <ul>
	 * <li>When a Resource is removed from a context, all related Facets may also be automatically removed due to propagation constraints</li>
	 * <li>The response includes all instances that were removed as part of the cascade effect</li>
	 * <li>Propagation follows the schema-defined relationships and their constraints.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation for Resource and ConsistsOf Relations:</strong>
	 * <ul>
	 * <li>When a Resource is removed from a context, the Resource is automatically validated against its schema definition</li>
	 * <li>If the operation would violate schema constraints after the remove operation, a {@link SchemaViolationException} is thrown</li>
	 * <li>This results in an HTTP 400 Bad Request response with details about the schema violation</li>
	 * <li><strong>Example scenarios that violate schema constraints:</strong></li>
	 * </ul>
	 *   • Removing a Resource that would leave required Facets orphaned;
	 *   • Operations that violate cardinality constraints (min occurrences) defined in the schema;
	 *   • Operations that make related Resources incomplete or invalid according to their schema definition.
	 * <ul>
	 * <li><strong>Constraint Sources</strong>: Constraints are defined by the specific Resource type definitions that have been installed on the system</li>
	 * <li>Schema validation ensures the integrity and consistency by enforcing the constraints defined in each Resource type's schema during operations.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and appropriate permissions for the current context where they want to remove Resources.</li>
	 * <li>The current context is derived from the authorization token provided during client creation.</li>
	 * </ul>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations on any Resource in any context</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations on any Resource in any context</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations only on Resources accessible in their current context</li>
	 * <li>Operations are limited to contexts where they have Context-Manager role</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations only on Resources accessible in their current context</li>
	 * <li>Operations are limited to Resources they have access to in their authorized context</li>
	 * <li>See filtered metadata with sensitive fields obfuscated when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON array containing all instances affected by the operation</li>
	 * <li>Each instance object includes its UUID, type, and relevant metadata</li>
	 * <li>For dryRun=true: Shows what would be affected without making changes</li>
	 * <li>For dryRun=false: Shows what was actually affected by the completed operation.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Remove an EService from the current context with actual changes
	 * UUID serviceUUID = UUID.fromString("...");
	 * List&lt;ERElement&gt; affected = publisher.removeResourceFromCurrentContext("EService", serviceUUID, false);
	 * 
	 * // Preview what would be affected by removing a HostingNode from the current context
	 * UUID hostUUID = UUID.fromString("...");
	 * List&lt;ERElement&gt; preview = publisher.removeResourceFromCurrentContext("HostingNode", hostUUID, true);
	 * </pre>
	 * 
	 * @param resourceType The exact type name of the Resource to remove
	 * @param resourceUUID The UUID of the Resource instance to remove from the current context
	 * @param dryRun Whether to perform a dry-run (true) or apply actual changes (false)
	 * @return List of all elements affected by this operation (including the Resource itself and any related elements)
	 * @throws SchemaViolationException If the operation violates Information System model constraints (HTTP 400)
	 * @throws ResourceNotFoundException If the Resource is not found or not accessible (HTTP 404)
	 * @throws ContextNotFoundException If the current context cannot be determined or accessed (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during the remove operation
	 */
	public List<ERElement> removeResourceFromCurrentContext(String resourceType, UUID resourceUUID, Boolean dryRun)
			throws SchemaViolationException, ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException;

	/**
	 * Removes a Resource instance from the current context using a provided Resource object.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /sharing/contexts/{resource-type-name}/{uuid}?operation=REMOVE[&dryRun={true|false}][&includeMeta={true|false}][&allMeta={true|false}][&includeContexts={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Removes an existing Resource instance from the current context derived from the authorization token</li>
	 * <li>The type and UUID are automatically extracted from the provided Resource object</li>
	 * <li>The Resource must already exist and be accessible in the current context where the user is performing the request</li>
	 * <li>The user must have the rights to perform such a request in the current context</li>
	 * <li>Returns the list of affected elements (the Resource itself and any related elements that are also removed)</li>
	 * <li>Supports dry-run mode to preview changes without applying them</li>
	 * <li>This is a convenience method that uses the current context instead of requiring an explicit context UUID</li>
	 * <li>This method provides compile-time type safety for Resource types.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Resource successfully removed from current context (or dry-run completed)</li>
	 * <li><strong>404 Not Found</strong>: Resource does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: User lacks permissions to perform the request in the current context</li>
	 * <li><strong>400 Bad Request</strong>: Invalid parameters or schema validation failure</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response for affected instances</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the Resource and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The type name is automatically derived from the provided Resource object using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>The UUID is extracted from the Resource's header</li>
	 * <li>Supports all Resource types and their specializations (e.g., EService, HostingNode, Software).</li>
	 * </ul>
	 * 
	 * <strong>Current Context:</strong>
	 * <ul>
	 * <li>The current context is automatically derived from the authorization token provided during client creation</li>
	 * <li>No explicit context UUID is required</li>
	 * <li>The user must have appropriate permissions in this context.</li>
	 * </ul>
	 * 
	 * <strong>Resource-Specific Context Propagation Rules:</strong>
	 * <ul>
	 * <li><strong>Resources</strong>: When removed from a context, all their associated Facets (via ConsistsOf relations) are also removed from the same context if the remove propagation constraint is {@code propagate}</li>
	 * <li><strong>Associated Facets</strong>: All Facets connected to the Resource via ConsistsOf relations are automatically removed if they become orphaned</li>
	 * <li><strong>ConsistsOf Relations</strong>: All ConsistsOf relations connecting the Resource to its Facets are also removed from the context.</li>
	 * </ul>
	 * 
	 * <strong>Propagation Constraints and Cascade Effects:</strong>
	 * <ul>
	 * <li>When a Resource is removed from a context, all related Facets may also be automatically removed due to propagation constraints</li>
	 * <li>The response includes all instances that were removed as part of the cascade effect</li>
	 * <li>Propagation follows the schema-defined relationships and their constraints.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation for Resource and ConsistsOf Relations:</strong>
	 * <ul>
	 * <li>When a Resource is removed from a context, the Resource is automatically validated against its schema definition</li>
	 * <li>If the operation would violate schema constraints after the remove operation, a {@link SchemaViolationException} is thrown</li>
	 * <li>This results in an HTTP 400 Bad Request response with details about the schema violation</li>
	 * <li><strong>Example scenarios that violate schema constraints:</strong></li>
	 * </ul>
	 *   • Removing a Resource that would leave required Facets orphaned;
	 *   • Operations that violate cardinality constraints (min occurrences) defined in the schema;
	 *   • Operations that make related Resources incomplete or invalid according to their schema definition.
	 * <ul>
	 * <li><strong>Constraint Sources</strong>: Constraints are defined by the specific Resource type definitions that have been installed on the system</li>
	 * <li>Schema validation ensures the integrity and consistency by enforcing the constraints defined in each Resource type's schema during operations.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and appropriate permissions for the current context where they want to remove Resources.</li>
	 * <li>The current context is derived from the authorization token provided during client creation.</li>
	 * </ul>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations on any Resource in any context</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations on any Resource in any context</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations only on Resources accessible in their current context</li>
	 * <li>Operations are limited to contexts where they have Context-Manager role</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations only on Resources accessible in their current context</li>
	 * <li>Operations are limited to Resources they have access to in their authorized context</li>
	 * <li>See filtered metadata with sensitive fields obfuscated when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON array containing all instances affected by the operation</li>
	 * <li>Each instance object includes its UUID, type, and relevant metadata</li>
	 * <li>For dryRun=true: Shows what would be affected without making changes</li>
	 * <li>For dryRun=false: Shows what was actually affected by the completed operation.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Remove an EService instance from the current context
	 * EService service = publisher.read(EService.class, serviceUUID);
	 * List&lt;ERElement&gt; affected = publisher.removeResourceFromCurrentContext(service, false);
	 * 
	 * // Preview what would be affected by removing a HostingNode from the current context
	 * HostingNode host = publisher.read(HostingNode.class, hostUUID);
	 * List&lt;ERElement&gt; preview = publisher.removeResourceFromCurrentContext(host, true);
	 * </pre>
	 * 
	 * @param <R> The specific type of Resource to remove
	 * @param resource The Resource instance to remove from the current context (must contain a valid UUID in the header)
	 * @param dryRun Whether to perform a dry-run (true) or apply actual changes (false)
	 * @return List of all elements affected by this operation (including the Resource itself and any related elements)
	 * @throws SchemaViolationException If the operation violates Information System model constraints (HTTP 400)
	 * @throws ResourceNotFoundException If the Resource is not found or not accessible (HTTP 404)
	 * @throws ContextNotFoundException If the current context cannot be determined or accessed (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during the remove operation
	 */
	public <R extends Resource> List<ERElement> removeResourceFromCurrentContext(R resource, Boolean dryRun) 
			throws SchemaViolationException, ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException;

	/**
	 * Retrieves the list of contexts where a specific Resource is present in the Resource Registry.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /instances/{resource-type-name}/{resource-uuid}/contexts}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves all contexts where the specified Resource is present</li>
	 * <li>The Resource must exist and be accessible to the user</li>
	 * <li>Returns a map where keys are context UUIDs and values are the full context paths</li>
	 * <li>No filtering, pagination, or metadata options are available</li>
	 * <li>Context information is determined by the Resource's presence across different VREs and organizational contexts.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Context list successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: Resource does not exist with the specified type and UUID</li>
	 * <li><strong>403 Forbidden</strong>: Resource exists but is not accessible for context listing in the current authorization scope</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager:</strong> Can retrieve context information for any Resource across the entire infrastructure</li>
	 * <li><strong>Infrastructure-Manager:</strong> Can retrieve context information for any Resource across the entire infrastructure</li>
	 * <li><strong>Context-Manager:</strong> Can retrieve context information for Resources accessible within their managed contexts</li>
	 * <li><strong>Other Users:</strong> Can retrieve context information for Resources accessible through their authorization token.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a Map where keys are context UUIDs (UUID) and values are full context paths (String)</li>
	 * <li>Empty map is returned if the Resource exists but is not present in any contexts</li>
	 * <li>Each context path represents the complete hierarchical path (e.g., "/gcube/devsec/devVRE").</li>
	 * </ul>
	 * 
	 * <strong>Resource Context Rules:</strong>
	 * <ul>
	 * <li><strong>Resources:</strong> Present in all contexts that form the union of contexts of all their Facets</li>
	 * <li>A Resource is present in a context if at least one of its associated Facets (via ConsistsOf relations) is present in that context</li>
	 * <li>If a Resource has no Facets, it may still be present in contexts based on direct context assignments.</li>
	 * </ul>
	 * 
	 * <strong>Type Safety:</strong>
	 * <ul>
	 * <li>This method is specifically designed for Resource types and provides compile-time type safety</li>
	 * <li>Supports all Resource types and their specializations (e.g., EService, HostingNode, Software, Dataset).</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Get contexts for an EService by type and UUID
	 * UUID serviceUuid = UUID.fromString("48af15ad-7e56-4157-b624-71c98cea4f8f");
	 * Map&lt;UUID, String&gt; contexts = publisher.getResourceContexts("EService", serviceUuid);
	 * 
	 * // Process the results
	 * for (Map.Entry&lt;UUID, String&gt; entry : contexts.entrySet()) {
	 *     System.out.println("Context UUID: " + entry.getKey() + " -> Path: " + entry.getValue());
	 * }
	 * </pre>
	 * 
	 * @param resourceType The exact type name of the Resource (e.g., "EService", "HostingNode")
	 * @param resourceUUID The unique identifier of the Resource whose contexts to retrieve
	 * @return Map containing context UUID to context path mappings where the Resource is present
	 * @throws ResourceNotFoundException If no Resource is found with the specified type and UUID
	 * @throws ResourceRegistryException If an error occurs during context retrieval
	 */
	public Map<UUID, String> getResourceContexts(String resourceType, UUID resourceUUID)
			throws ResourceNotFoundException, ResourceRegistryException;

	/**
	 * Retrieves the list of contexts where a specific Resource is present in the Resource Registry.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /instances/{resource-type-name}/{resource-uuid}/contexts}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves all contexts where the specified Resource object is present</li>
	 * <li>The Resource's type and UUID are automatically extracted from the provided object</li>
	 * <li>The Resource must exist and be accessible to the user</li>
	 * <li>Returns a map where keys are context UUIDs and values are the full context paths</li>
	 * <li>No filtering, pagination, or metadata options are available</li>
	 * <li>Context information is determined by the Resource's presence across different VREs and organizational contexts.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Context list successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: Resource does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: Resource exists but is not accessible for context listing in the current authorization scope</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager:</strong> Can retrieve context information for any Resource across the entire infrastructure</li>
	 * <li><strong>Infrastructure-Manager:</strong> Can retrieve context information for any Resource across the entire infrastructure</li>
	 * <li><strong>Context-Manager:</strong> Can retrieve context information for Resources accessible within their managed contexts</li>
	 * <li><strong>Other Users:</strong> Can retrieve context information for Resources accessible through their authorization token.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a Map where keys are context UUIDs (UUID) and values are full context paths (String)</li>
	 * <li>Empty map is returned if the Resource exists but is not present in any contexts</li>
	 * <li>Each context path represents the complete hierarchical path (e.g., "/gcube/devsec/devVRE").</li>
	 * </ul>
	 * 
	 * <strong>Resource Context Rules:</strong>
	 * <ul>
	 * <li><strong>Resources:</strong> Present in all contexts that form the union of contexts of all their Facets</li>
	 * <li>A Resource is present in a context if at least one of its associated Facets (via ConsistsOf relations) is present in that context</li>
	 * <li>If a Resource has no Facets, it may still be present in contexts based on direct context assignments.</li>
	 * </ul>
	 * 
	 * <strong>Type Safety:</strong>
	 * <ul>
	 * <li>This method provides compile-time type safety by accepting any Resource subtype</li>
	 * <li>The Resource's type is automatically determined from the object's class</li>
	 * <li>Supports all Resource types and their specializations (e.g., EService, HostingNode, Software, Dataset).</li>
	 * </ul>
	 * 
	 * <strong>Object-based Operation:</strong>
	 * <ul>
	 * <li>This method accepts a Resource instance object directly, which provides convenience and type safety</li>
	 * <li>The Resource object must contain a valid UUID in its header metadata</li>
	 * <li>The type is automatically inferred from the object's runtime class.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Assuming you have an EService instance
	 * EService service = ...; // Retrieved from previous operations
	 * 
	 * // Get contexts for the service object
	 * Map&lt;UUID, String&gt; contexts = publisher.getResourceContexts(service);
	 * 
	 * // Process the results
	 * for (Map.Entry&lt;UUID, String&gt; entry : contexts.entrySet()) {
	 *     System.out.println("Context UUID: " + entry.getKey() + " -> Path: " + entry.getValue());
	 * }
	 * </pre>
	 * 
	 * @param <R> The type of Resource (EService, HostingNode, Software, etc.)
	 * @param resource The Resource instance whose contexts to retrieve
	 * @return Map containing context UUID to context path mappings where the Resource is present
	 * @throws ResourceNotFoundException If the specified Resource does not exist or is not accessible
	 * @throws ResourceRegistryException If an error occurs during context retrieval
	 */
	public <R extends Resource> Map<UUID, String> getResourceContexts(R resource)
			throws ResourceNotFoundException, ResourceRegistryException;

	/**
	 * Adds a Facet instance to a specific context using type name and UUID.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code PUT /instances/{facet-type-name}/{uuid}/contexts/{context-uuid}[?dryRun={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Adds an existing Facet instance to the specified context</li>
	 * <li>The Facet must already exist and be accessible in the context where the user is performing the request</li>
	 * <li>The user must have the rights to perform such a request in the target context</li>
	 * <li>Returns the list of affected elements (the Facet itself and any related elements that are also added)</li>
	 * <li>Supports dry-run mode to preview changes without applying them.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Facet successfully added to specified context (or dry-run completed)</li>
	 * <li><strong>404 Not Found</strong>: Facet or context does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: User lacks permissions to perform the request in the target context</li>
	 * <li><strong>400 Bad Request</strong>: Invalid parameters or schema validation failure</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>forceAddToContext</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Forces the addition of a Facet to a context even if the Facet is not present in the current context</li>
	 * <li>Default value: false (standard validation applies)</li>
	 * <li>Usage: {@code publisher.addHeader(SharingPath.FORCE_ADD_TO_CONTEXT_QUERY_PARAMETER, true)}</li>
	 * <li>Values: true | false</li>
	 * <li>When true: Allows adding a Facet to a context even if the Facet exists only in other contexts</li>
	 * <li>When false: Standard validation applies - Facet must be accessible in the current context for ADD operations</li>
	 * <li><strong>Authorization Required</strong>: Only available to IS-Manager and Infrastructure-Manager</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response for affected instances</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the Facet and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Facet is present in all contexts that are part of the union of contexts of all Resources that use it via ConsistsOf relations</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their source Resources and target Facets are both present.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The facetType parameter must be a valid Facet type name registered in the Information System</li>
	 * <li>Type names are case-sensitive and must match exactly</li>
	 * <li>Supports all Facet types and their specializations (e.g., StateFacet, SoftwareFacet, AccessPointFacet).</li>
	 * </ul>
	 * 
	 * <strong>Target Context:</strong>
	 * <ul>
	 * <li>The contextUUID parameter specifies the exact context where the Facet should be added</li>
	 * <li>The user must have appropriate permissions in this context</li>
	 * <li>The context must exist and be accessible.</li>
	 * </ul>
	 * 
	 * <strong>Facet-Specific Context Propagation Rules:</strong>
	 * <ul>
	 * <li><strong>Facets</strong>: When added to a context, no automatic propagation occurs as Facets do not have dependent elements</li>
	 * <li><strong>Related Resources</strong>: Resources that use this Facet via ConsistsOf relations are NOT automatically added to the context</li>
	 * <li><strong>Consistency Check</strong>: The Facet must be associated with Resources that are already present in the target context</li>
	 * <li><strong>Important</strong>: Adding a Facet to a context where its associated Resources are not present may create inconsistencies.</li>
	 * </ul>
	 * 
	 * <strong>Propagation Constraints and Cascade Effects:</strong>
	 * <ul>
	 * <li>When a Facet is added to a context, there are typically no cascade effects since Facets are leaf elements</li>
	 * <li>However, the schema may define specific constraints for certain Facet types that could trigger validation</li>
	 * <li>The response includes the Facet instance and any validation-related information.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation for Facet:</strong>
	 * <ul>
	 * <li>When a Facet is added to a context, the Facet is automatically validated against its schema definition</li>
	 * <li>If the Facet is no longer compliant with its schema after the add operation, a {@link SchemaViolationException} is thrown</li>
	 * <li>This results in an HTTP 400 Bad Request response with details about the schema violation</li>
	 * <li><strong>Example scenarios that violate schema constraints:</strong></li>
	 * </ul>
	 *   • Adding a Facet that violates property constraints defined in the schema;
	 *   • Operations that make the Facet incomplete or invalid according to its schema definition.
	 * <ul>
	 * <li><strong>Constraint Sources</strong>: Constraints are defined by the specific Facet type definitions that have been installed on the system</li>
	 * <li>Schema validation ensures the integrity and consistency by enforcing the constraints defined in each Facet type's schema during operations.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and appropriate permissions for the target context where they want to add Facets.</li>
	 * </ul>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD operations on any Facet in any context</li>
	 * <li>Can use forceAddToContext=true to add Facets from other contexts</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD operations on any Facet in any context</li>
	 * <li>Can use forceAddToContext=true to add Facets from other contexts</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD operations only on Facets accessible in contexts where they have Context-Manager role</li>
	 * <li>Cannot use forceAddToContext=true (parameter is ignored if provided)</li>
	 * <li>Operations are limited to contexts where they have Context-Manager role</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can perform ADD operations only on Facets accessible in contexts where they have appropriate permissions</li>
	 * <li>Cannot use forceAddToContext=true (parameter is ignored if provided)</li>
	 * <li>Operations are limited to Facets they have access to in their authorized contexts</li>
	 * <li>See filtered metadata with sensitive fields obfuscated when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON array containing all instances affected by the operation</li>
	 * <li>Each instance object includes its UUID, type, and relevant metadata</li>
	 * <li>For dryRun=true: Shows what would be affected without making changes</li>
	 * <li>For dryRun=false: Shows what was actually affected by the completed operation.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Add a StateFacet instance to a specific context
	 * UUID stateFacetUUID = UUID.fromString("12345678-1234-1234-1234-123456789abc");
	 * UUID targetContextUUID = UUID.fromString("87654321-4321-4321-4321-cba987654321");
	 * List&lt;ERElement&gt; affected = publisher.addFacetToContext("StateFacet", stateFacetUUID, targetContextUUID, false);
	 * 
	 * // Preview what would be affected by adding an AccessPointFacet to a specific context
	 * UUID accessPointUUID = UUID.fromString("11223344-5566-7788-9900-aabbccddeeff");
	 * UUID contextUUID = UUID.fromString("ffeeddcc-bbaa-0099-8877-665544332211");
	 * List&lt;ERElement&gt; preview = publisher.addFacetToContext("AccessPointFacet", accessPointUUID, contextUUID, true);
	 * </pre>
	 * 
	 * @param facetType The type name of the Facet to add (must be a valid Facet type)
	 * @param facetUUID The UUID of the Facet instance to add to the specified context
	 * @param contextUUID The UUID of the target context where the Facet should be added
	 * @param dryRun Whether to perform a dry-run (true) or apply actual changes (false)
	 * @return List of all elements affected by this operation (including the Facet itself and any related elements)
	 * @throws SchemaViolationException If the operation violates Information System model constraints (HTTP 400)
	 * @throws FacetNotFoundException If the Facet is not found or not accessible (HTTP 404)
	 * @throws ContextNotFoundException If the target context cannot be determined or accessed (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during the add operation
	 */
	public List<ERElement> addFacetToContext(String facetType, UUID facetUUID, UUID contextUUID, Boolean dryRun)
			throws SchemaViolationException, FacetNotFoundException, ContextNotFoundException, ResourceRegistryException;

	/**
	 * Adds a Facet instance to a specific context using a provided Facet object.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code PUT /instances/{facet-type-name}/{uuid}/contexts/{context-uuid}[?dryRun={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Adds an existing Facet instance to the specified context</li>
	 * <li>The type and UUID are automatically extracted from the provided Facet object</li>
	 * <li>The Facet must already exist and be accessible in the context where the user is performing the request</li>
	 * <li>The user must have the rights to perform such a request in the target context</li>
	 * <li>Returns the list of affected elements (the Facet itself and any related elements that are also added)</li>
	 * <li>Supports dry-run mode to preview changes without applying them</li>
	 * <li>This method provides compile-time type safety for Facet types.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Facet successfully added to specified context (or dry-run completed)</li>
	 * <li><strong>404 Not Found</strong>: Facet or context does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: User lacks permissions to perform the request in the target context</li>
	 * <li><strong>400 Bad Request</strong>: Invalid parameters or schema validation failure</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>forceAddToContext</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Forces the addition of a Facet to a context even if the Facet is not present in the current context</li>
	 * <li>Default value: false (standard validation applies)</li>
	 * <li>Usage: {@code publisher.addHeader(SharingPath.FORCE_ADD_TO_CONTEXT_QUERY_PARAMETER, true)}</li>
	 * <li>Values: true | false</li>
	 * <li>When true: Allows adding a Facet to a context even if the Facet exists only in other contexts</li>
	 * <li>When false: Standard validation applies - Facet must be accessible in the current context for ADD operations</li>
	 * <li><strong>Authorization Required</strong>: Only available to IS-Manager and Infrastructure-Manager</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response for affected instances</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the Facet and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Facet is present in all contexts that are part of the union of contexts of all Resources that use it via ConsistsOf relations</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their source Resources and target Facets are both present.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The type name is automatically derived from the provided Facet object using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>The UUID is extracted from the Facet's header</li>
	 * <li>Supports all Facet types and their specializations (e.g., StateFacet, SoftwareFacet, AccessPointFacet).</li>
	 * </ul>
	 * 
	 * <strong>Target Context:</strong>
	 * <ul>
	 * <li>The contextUUID parameter specifies the exact context where the Facet should be added</li>
	 * <li>The user must have appropriate permissions in this context</li>
	 * <li>The context must exist and be accessible.</li>
	 * </ul>
	 * 
	 * <strong>Facet-Specific Context Propagation Rules:</strong>
	 * <ul>
	 * <li><strong>Facets</strong>: When added to a context, no automatic propagation occurs as Facets do not have dependent elements</li>
	 * <li><strong>Related Resources</strong>: Resources that use this Facet via ConsistsOf relations are NOT automatically added to the context</li>
	 * <li><strong>Consistency Check</strong>: The Facet must be associated with Resources that are already present in the target context</li>
	 * <li><strong>Important</strong>: Adding a Facet to a context where its associated Resources are not present may create inconsistencies.</li>
	 * </ul>
	 * 
	 * <strong>Propagation Constraints and Cascade Effects:</strong>
	 * <ul>
	 * <li>When a Facet is added to a context, there are typically no cascade effects since Facets are leaf elements</li>
	 * <li>However, the schema may define specific constraints for certain Facet types that could trigger validation</li>
	 * <li>The response includes the Facet instance and any validation-related information.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation for Facet:</strong>
	 * <ul>
	 * <li>When a Facet is added to a context, the Facet is automatically validated against its schema definition</li>
	 * <li>If the Facet is no longer compliant with its schema after the add operation, a {@link SchemaViolationException} is thrown</li>
	 * <li>This results in an HTTP 400 Bad Request response with details about the schema violation</li>
	 * <li><strong>Example scenarios that violate schema constraints:</strong></li>
	 * </ul>
	 *   • Adding a Facet that violates property constraints defined in the schema;
	 *   • Operations that make the Facet incomplete or invalid according to its schema definition.
	 * <ul>
	 * <li><strong>Constraint Sources</strong>: Constraints are defined by the specific Facet type definitions that have been installed on the system</li>
	 * <li>Schema validation ensures the integrity and consistency by enforcing the constraints defined in each Facet type's schema during operations.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and appropriate permissions for the target context where they want to add Facets.</li>
	 * </ul>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD operations on any Facet in any context</li>
	 * <li>Can use forceAddToContext=true to add Facets from other contexts</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD operations on any Facet in any context</li>
	 * <li>Can use forceAddToContext=true to add Facets from other contexts</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD operations only on Facets accessible in contexts where they have Context-Manager role</li>
	 * <li>Cannot use forceAddToContext=true (parameter is ignored if provided)</li>
	 * <li>Operations are limited to contexts where they have Context-Manager role</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can perform ADD operations only on Facets accessible in contexts where they have appropriate permissions</li>
	 * <li>Cannot use forceAddToContext=true (parameter is ignored if provided)</li>
	 * <li>Operations are limited to Facets they have access to in their authorized contexts</li>
	 * <li>See filtered metadata with sensitive fields obfuscated when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON array containing all instances affected by the operation</li>
	 * <li>Each instance object includes its UUID, type, and relevant metadata</li>
	 * <li>For dryRun=true: Shows what would be affected without making changes</li>
	 * <li>For dryRun=false: Shows what was actually affected by the completed operation.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Add a StateFacet instance to a specific context
	 * StateFacet facet = new StateFacet();
	 * facet.setHeader(new Header(facetUUID));
	 * UUID targetContextUUID = UUID.fromString("87654321-4321-4321-4321-cba987654321");
	 * List&lt;ERElement&gt; affected = publisher.addFacetToContext(facet, targetContextUUID, false);
	 * 
	 * // Preview what would be affected by adding an AccessPointFacet to a specific context
	 * AccessPointFacet accessPoint = publisher.read(AccessPointFacet.class, accessPointUUID);
	 * UUID contextUUID = UUID.fromString("ffeeddcc-bbaa-0099-8877-665544332211");
	 * List&lt;ERElement&gt; preview = publisher.addFacetToContext(accessPoint, contextUUID, true);
	 * </pre>
	 * 
	 * @param <F> The specific type of Facet to add
	 * @param facet The Facet instance to add to the specified context (must contain a valid UUID in the header)
	 * @param contextUUID The UUID of the target context where the Facet should be added
	 * @param dryRun Whether to perform a dry-run (true) or apply actual changes (false)
	 * @return List of all elements affected by this operation (including the Facet itself and any related elements)
	 * @throws SchemaViolationException If the operation violates Information System model constraints (HTTP 400)
	 * @throws FacetNotFoundException If the Facet is not found or not accessible (HTTP 404)
	 * @throws ContextNotFoundException If the target context cannot be determined or accessed (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during the add operation
	 */
	public <F extends Facet> List<ERElement> addFacetToContext(F facet, UUID contextUUID, Boolean dryRun)
			throws SchemaViolationException, FacetNotFoundException, ContextNotFoundException, ResourceRegistryException;

	/**
	 * Adds a Facet instance to the current context using type name and UUID.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code PUT /instances/{facet-type-name}/{uuid}/contexts[?dryRun={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Adds an existing Facet instance to the current context derived from the authorization token</li>
	 * <li>The Facet must already exist and be accessible in the context where the user is performing the request</li>
	 * <li>The user must have the rights to perform such a request in the current context</li>
	 * <li>Returns the list of affected elements (the Facet itself and any related elements that are also added)</li>
	 * <li>Supports dry-run mode to preview changes without applying them</li>
	 * <li>This is a convenience method that uses the current context instead of requiring an explicit context UUID.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Facet successfully added to current context (or dry-run completed)</li>
	 * <li><strong>404 Not Found</strong>: Facet does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: User lacks permissions to perform the request in the current context</li>
	 * <li><strong>400 Bad Request</strong>: Invalid parameters or schema validation failure</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>forceAddToContext</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Forces the addition of a Facet to a context even if the Facet is not present in the current context</li>
	 * <li>Default value: false (standard validation applies)</li>
	 * <li>Usage: {@code publisher.addHeader(SharingPath.FORCE_ADD_TO_CONTEXT_QUERY_PARAMETER, true)}</li>
	 * <li>Values: true | false</li>
	 * <li>When true: Allows adding a Facet to a context even if the Facet exists only in other contexts</li>
	 * <li>When false: Standard validation applies - Facet must be accessible in the current context for ADD operations</li>
	 * <li><strong>Authorization Required</strong>: Only available to IS-Manager and Infrastructure-Manager</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response for affected instances</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the Facet and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Facet is present in all contexts that are part of the union of contexts of all Resources that use it via ConsistsOf relations</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their source Resources and target Facets are both present.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The facetType parameter must be a valid Facet type name registered in the Information System</li>
	 * <li>Type names are case-sensitive and must match exactly</li>
	 * <li>Supports all Facet types and their specializations (e.g., StateFacet, SoftwareFacet, AccessPointFacet).</li>
	 * </ul>
	 * 
	 * <strong>Current Context:</strong>
	 * <ul>
	 * <li>The current context is automatically derived from the authorization token provided during client creation</li>
	 * <li>No explicit context UUID is required</li>
	 * <li>The user must have appropriate permissions in this context.</li>
	 * </ul>
	 * 
	 * <strong>Facet-Specific Context Propagation Rules:</strong>
	 * <ul>
	 * <li><strong>Facets</strong>: When added to a context, no automatic propagation occurs as Facets do not have dependent elements</li>
	 * <li><strong>Related Resources</strong>: Resources that use this Facet via ConsistsOf relations are NOT automatically added to the context</li>
	 * <li><strong>Consistency Check</strong>: The Facet must be associated with Resources that are already present in the target context</li>
	 * <li><strong>Important</strong>: Adding a Facet to a context where its associated Resources are not present may create inconsistencies.</li>
	 * </ul>
	 * 
	 * <strong>Propagation Constraints and Cascade Effects:</strong>
	 * <ul>
	 * <li>When a Facet is added to a context, there are typically no cascade effects since Facets are leaf elements</li>
	 * <li>However, the schema may define specific constraints for certain Facet types that could trigger validation</li>
	 * <li>The response includes the Facet instance and any validation-related information.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation for Facet:</strong>
	 * <ul>
	 * <li>When a Facet is added to a context, the Facet is automatically validated against its schema definition</li>
	 * <li>If the Facet is no longer compliant with its schema after the add operation, a {@link SchemaViolationException} is thrown</li>
	 * <li>This results in an HTTP 400 Bad Request response with details about the schema violation</li>
	 * <li><strong>Example scenarios that violate schema constraints:</strong></li>
	 * </ul>
	 *   • Adding a Facet that violates property constraints defined in the schema;
	 *   • Operations that make the Facet incomplete or invalid according to its schema definition.
	 * <ul>
	 * <li><strong>Constraint Sources</strong>: Constraints are defined by the specific Facet type definitions that have been installed on the system</li>
	 * <li>Schema validation ensures the integrity and consistency by enforcing the constraints defined in each Facet type's schema during operations.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and appropriate permissions for the current context where they want to add Facets.</li>
	 * <li>The current context is derived from the authorization token provided during client creation.</li>
	 * </ul>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD operations on any Facet in any context</li>
	 * <li>Can use forceAddToContext=true to add Facets from other contexts</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD operations on any Facet in any context</li>
	 * <li>Can use forceAddToContext=true to add Facets from other contexts</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD operations only on Facets accessible in their current context</li>
	 * <li>Cannot use forceAddToContext=true (parameter is ignored if provided)</li>
	 * <li>Operations are limited to contexts where they have Context-Manager role</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can perform ADD operations only on Facets accessible in their current context</li>
	 * <li>Cannot use forceAddToContext=true (parameter is ignored if provided)</li>
	 * <li>Operations are limited to Facets they have access to in their authorized context</li>
	 * <li>See filtered metadata with sensitive fields obfuscated when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON array containing all instances affected by the operation</li>
	 * <li>Each instance object includes its UUID, type, and relevant metadata</li>
	 * <li>For dryRun=true: Shows what would be affected without making changes</li>
	 * <li>For dryRun=false: Shows what was actually affected by the completed operation.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Add a StateFacet instance to the current context
	 * UUID stateFacetUUID = UUID.fromString("12345678-1234-1234-1234-123456789abc");
	 * List&lt;ERElement&gt; affected = publisher.addFacetToCurrentContext("StateFacet", stateFacetUUID, false);
	 * 
	 * // Preview what would be affected by adding an AccessPointFacet to the current context
	 * UUID accessPointUUID = UUID.fromString("87654321-4321-4321-4321-cba987654321");
	 * List&lt;ERElement&gt; preview = publisher.addFacetToCurrentContext("AccessPointFacet", accessPointUUID, true);
	 * </pre>
	 * 
	 * @param facetType The type name of the Facet to add (must be a valid Facet type)
	 * @param facetUUID The UUID of the Facet instance to add to the current context
	 * @param dryRun Whether to perform a dry-run (true) or apply actual changes (false)
	 * @return List of all elements affected by this operation (including the Facet itself and any related elements)
	 * @throws SchemaViolationException If the operation violates Information System model constraints (HTTP 400)
	 * @throws FacetNotFoundException If the Facet is not found or not accessible (HTTP 404)
	 * @throws ContextNotFoundException If the current context cannot be determined or accessed (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during the add operation
	 */
	public List<ERElement> addFacetToCurrentContext(String facetType, UUID facetUUID, Boolean dryRun)
			throws SchemaViolationException, FacetNotFoundException, ContextNotFoundException, ResourceRegistryException;

	/**
	 * Adds a Facet instance to the current context using a provided Facet object.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code PUT /instances/{facet-type-name}/{uuid}/contexts[?dryRun={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Adds an existing Facet instance to the current context derived from the authorization token</li>
	 * <li>The type and UUID are automatically extracted from the provided Facet object</li>
	 * <li>The Facet must already exist and be accessible in the context where the user is performing the request</li>
	 * <li>The user must have the rights to perform such a request in the current context</li>
	 * <li>Returns the list of affected elements (the Facet itself and any related elements that are also added)</li>
	 * <li>Supports dry-run mode to preview changes without applying them</li>
	 * <li>This is a convenience method that uses the current context instead of requiring an explicit context UUID</li>
	 * <li>This method provides compile-time type safety for Facet types.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Facet successfully added to current context (or dry-run completed)</li>
	 * <li><strong>404 Not Found</strong>: Facet does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: User lacks permissions to perform the request in the current context</li>
	 * <li><strong>400 Bad Request</strong>: Invalid parameters or schema validation failure</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>forceAddToContext</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Forces the addition of a Facet to a context even if the Facet is not present in the current context</li>
	 * <li>Default value: false (standard validation applies)</li>
	 * <li>Usage: {@code publisher.addHeader(SharingPath.FORCE_ADD_TO_CONTEXT_QUERY_PARAMETER, true)}</li>
	 * <li>Values: true | false</li>
	 * <li>When true: Allows adding a Facet to a context even if the Facet exists only in other contexts</li>
	 * <li>When false: Standard validation applies - Facet must be accessible in the current context for ADD operations</li>
	 * <li><strong>Authorization Required</strong>: Only available to IS-Manager and Infrastructure-Manager</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response for affected instances</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the Facet and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Facet is present in all contexts that are part of the union of contexts of all Resources that use it via ConsistsOf relations</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their source Resources and target Facets are both present.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The type name is automatically derived from the provided Facet object using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>The UUID is extracted from the Facet's header</li>
	 * <li>Supports all Facet types and their specializations (e.g., StateFacet, SoftwareFacet, AccessPointFacet).</li>
	 * </ul>
	 * 
	 * <strong>Current Context:</strong>
	 * <ul>
	 * <li>The current context is automatically derived from the authorization token provided during client creation</li>
	 * <li>No explicit context UUID is required</li>
	 * <li>The user must have appropriate permissions in this context.</li>
	 * </ul>
	 * 
	 * <strong>Facet-Specific Context Propagation Rules:</strong>
	 * <ul>
	 * <li><strong>Facets</strong>: When added to a context, no automatic propagation occurs as Facets do not have dependent elements</li>
	 * <li><strong>Related Resources</strong>: Resources that use this Facet via ConsistsOf relations are NOT automatically added to the context</li>
	 * <li><strong>Consistency Check</strong>: The Facet must be associated with Resources that are already present in the target context</li>
	 * <li><strong>Important</strong>: Adding a Facet to a context where its associated Resources are not present may create inconsistencies.</li>
	 * </ul>
	 * 
	 * <strong>Propagation Constraints and Cascade Effects:</strong>
	 * <ul>
	 * <li>When a Facet is added to a context, there are typically no cascade effects since Facets are leaf elements</li>
	 * <li>However, the schema may define specific constraints for certain Facet types that could trigger validation</li>
	 * <li>The response includes the Facet instance and any validation-related information.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation for Facet:</strong>
	 * <ul>
	 * <li>When a Facet is added to a context, the Facet is automatically validated against its schema definition</li>
	 * <li>If the Facet is no longer compliant with its schema after the add operation, a {@link SchemaViolationException} is thrown</li>
	 * <li>This results in an HTTP 400 Bad Request response with details about the schema violation</li>
	 * <li><strong>Example scenarios that violate schema constraints:</strong></li>
	 * </ul>
	 *   • Adding a Facet that violates property constraints defined in the schema;
	 *   • Operations that make the Facet incomplete or invalid according to its schema definition.
	 * <ul>
	 * <li><strong>Constraint Sources</strong>: Constraints are defined by the specific Facet type definitions that have been installed on the system</li>
	 * <li>Schema validation ensures the integrity and consistency by enforcing the constraints defined in each Facet type's schema during operations.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and appropriate permissions for the current context where they want to add Facets.</li>
	 * <li>The current context is derived from the authorization token provided during client creation.</li>
	 * </ul>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD operations on any Facet in any context</li>
	 * <li>Can use forceAddToContext=true to add Facets from other contexts</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD operations on any Facet in any context</li>
	 * <li>Can use forceAddToContext=true to add Facets from other contexts</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD operations only on Facets accessible in their current context</li>
	 * <li>Cannot use forceAddToContext=true (parameter is ignored if provided)</li>
	 * <li>Operations are limited to contexts where they have Context-Manager role</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can perform ADD operations only on Facets accessible in their current context</li>
	 * <li>Cannot use forceAddToContext=true (parameter is ignored if provided)</li>
	 * <li>Operations are limited to Facets they have access to in their authorized context</li>
	 * <li>See filtered metadata with sensitive fields obfuscated when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON array containing all instances affected by the operation</li>
	 * <li>Each instance object includes its UUID, type, and relevant metadata</li>
	 * <li>For dryRun=true: Shows what would be affected without making changes</li>
	 * <li>For dryRun=false: Shows what was actually affected by the completed operation.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Add a StateFacet instance to the current context
	 * StateFacet facet = new StateFacet();
	 * facet.setHeader(new Header(facetUUID));
	 * List&lt;ERElement&gt; affected = publisher.addFacetToCurrentContext(facet, false);
	 * 
	 * // Preview what would be affected by adding an AccessPointFacet to the current context
	 * AccessPointFacet accessPoint = publisher.read(AccessPointFacet.class, accessPointUUID);
	 * List&lt;ERElement&gt; preview = publisher.addFacetToCurrentContext(accessPoint, true);
	 * </pre>
	 * 
	 * @param <F> The specific type of Facet to add
	 * @param facet The Facet instance to add to the current context (must contain a valid UUID in the header)
	 * @param dryRun Whether to perform a dry-run (true) or apply actual changes (false)
	 * @return List of all elements affected by this operation (including the Facet itself and any related elements)
	 * @throws SchemaViolationException If the operation violates Information System model constraints (HTTP 400)
	 * @throws FacetNotFoundException If the Facet is not found or not accessible (HTTP 404)
	 * @throws ContextNotFoundException If the current context cannot be determined or accessed (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during the add operation
	 */
	public <F extends Facet> List<ERElement> addFacetToCurrentContext(F facet, Boolean dryRun)
			throws SchemaViolationException, FacetNotFoundException, ContextNotFoundException, ResourceRegistryException;

	/**
	 * Removes a Facet instance from a specific context.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /sharing/contexts/{context-uuid}/{facet-type-name}/{uuid}?operation=REMOVE[&dryRun={true|false}][&includeMeta={true|false}][&allMeta={true|false}][&includeContexts={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Removes an existing Facet instance from the specified target context</li>
	 * <li>The Facet must already exist and be accessible in the target context where the user is performing the request</li>
	 * <li>The target context must exist and the user must have the rights to perform such a request</li>
	 * <li>Returns the list of affected elements (the Facet itself and any related elements that are also removed)</li>
	 * <li>Supports dry-run mode to preview changes without applying them</li>
	 * <li>This method is specifically designed for Facet types and provides type safety.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Facet successfully removed from context (or dry-run completed)</li>
	 * <li><strong>404 Not Found</strong>: Facet or target context does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: User lacks permissions to perform the request</li>
	 * <li><strong>400 Bad Request</strong>: Invalid parameters or schema validation failure</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response for affected instances</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, parent Resources, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the Facet and its related elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The type name must be a valid Facet type in the Information System model</li>
	 * <li>Supports all Facet types and their specializations (e.g., StateFacet, ContactFacet, AccessPointFacet)</li>
	 * <li>Type name is case-sensitive and must match exactly.</li>
	 * </ul>
	 * 
	 * <strong>Facet-Specific Context Propagation Rules:</strong>
	 * <ul>
	 * <li><strong>Facets</strong>: When removed from a context, the parent Resource that references it (via ConsistsOf relation) may also be removed from the same context if the Facet was required for the Resource's presence in that context</li>
	 * <li><strong>Associated Resources</strong>: Parent Resources connected to the Facet via ConsistsOf relations are automatically affected based on propagation constraints</li>
	 * <li><strong>ConsistsOf Relations</strong>: All ConsistsOf relations connecting parent Resources to this Facet are also removed from the context.</li>
	 * </ul>
	 * 
	 * <strong>Propagation Constraints and Cascade Effects:</strong>
	 * <ul>
	 * <li>When a Facet is removed from a context, related Resources may also be automatically removed due to propagation constraints</li>
	 * <li>The response includes all instances that were removed as part of the cascade effect</li>
	 * <li>Propagation follows the schema-defined relationships and their constraints.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation for Facets and ConsistsOf Relations:</strong>
	 * <ul>
	 * <li>When a Facet is removed from a context, the parent Resources are automatically validated against their schema definitions</li>
	 * <li>If any parent Resource is no longer compliant with its schema after the remove operation, a {@link SchemaViolationException} is thrown</li>
	 * <li>This results in an HTTP 400 Bad Request response with details about the schema violation</li>
	 * <li><strong>Example scenarios that violate schema constraints:</strong></li>
	 * </ul>
	 *   • Removing a required/mandatory Facet from a Resource;
	 *   • Removing Facets that violate cardinality constraints (min occurrences) defined in the schema;
	 *   • Operations that make parent Resources incomplete or invalid according to their schema definition.
	 * <ul>
	 * <li><strong>Constraint Sources</strong>: Constraints are defined by the specific Resource type definitions that have been installed on the system</li>
	 * <li>Schema validation ensures the integrity and consistency by enforcing the constraints defined in each Resource type's schema during operations.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and appropriate permissions for the specific context where they want to remove Facets.</li>
	 * </ul>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations on any Facet in any context</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations on any Facet in any context</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations only on Facets accessible in their current context</li>
	 * <li>Operations are limited to contexts where they have Context-Manager role</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations only on Facets accessible in their current context</li>
	 * <li>Operations are limited to Facets they have access to in their authorized context</li>
	 * <li>See filtered metadata with sensitive fields obfuscated when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON array containing all instances affected by the operation</li>
	 * <li>Each instance object includes its UUID, type, and relevant metadata</li>
	 * <li>For dryRun=true: Shows what would be affected without making changes</li>
	 * <li>For dryRun=false: Shows what was actually affected by the completed operation.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Remove a StateFacet from a specific context with actual changes
	 * UUID stateFacetUUID = UUID.fromString("...");
	 * UUID targetContextUUID = UUID.fromString("...");
	 * List&lt;ERElement&gt; affected = publisher.removeFacetFromContext("StateFacet", stateFacetUUID, targetContextUUID, false);
	 * 
	 * // Preview what would be affected by removing an AccessPointFacet from a context
	 * UUID accessPointUUID = UUID.fromString("...");
	 * List&lt;ERElement&gt; preview = publisher.removeFacetFromContext("AccessPointFacet", accessPointUUID, targetContextUUID, true);
	 * </pre>
	 * 
	 * @param facetType The exact type name of the Facet to remove
	 * @param facetUUID The UUID of the Facet instance to remove from the context
	 * @param contextUUID The UUID of the target context from which the Facet should be removed
	 * @param dryRun Whether to perform a dry-run (true) or apply actual changes (false)
	 * @return List of all elements affected by this operation (including the Facet itself and any related elements)
	 * @throws SchemaViolationException If the operation violates Information System model constraints (HTTP 400)
	 * @throws FacetNotFoundException If the Facet is not found or not accessible (HTTP 404)
	 * @throws ContextNotFoundException If the target context does not exist or is not accessible (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during the remove operation
	 */
	public List<ERElement> removeFacetFromContext(String facetType, UUID facetUUID, UUID contextUUID, Boolean dryRun)
			throws SchemaViolationException, FacetNotFoundException, ContextNotFoundException, ResourceRegistryException;

	/**
	 * Removes a Facet instance from a specific context using a provided Facet object.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /sharing/contexts/{context-uuid}/{facet-type-name}/{uuid}?operation=REMOVE[&dryRun={true|false}][&includeMeta={true|false}][&allMeta={true|false}][&includeContexts={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Removes an existing Facet instance from the specified target context</li>
	 * <li>The Facet must already exist and be accessible in the target context where the user is performing the request</li>
	 * <li>The target context must exist and the user must have the rights to perform such a request</li>
	 * <li>Returns the list of affected elements (the Facet itself and any related elements that are also removed)</li>
	 * <li>Supports dry-run mode to preview changes without applying them</li>
	 * <li>This method accepts a Facet instance object directly, which provides type safety and convenience</li>
	 * <li>This method is specifically designed for Facet types and provides type safety.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Facet successfully removed from context (or dry-run completed)</li>
	 * <li><strong>404 Not Found</strong>: Facet or target context does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: User lacks permissions to perform the request</li>
	 * <li><strong>400 Bad Request</strong>: Invalid parameters or schema validation failure</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response for affected instances</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, parent Resources, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the Facet and its related elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The Facet object must contain a valid UUID in its header</li>
	 * <li>Supports all Facet types and their specializations (e.g., StateFacet, ContactFacet, AccessPointFacet)</li>
	 * <li>The actual type is determined from the Facet object's class.</li>
	 * </ul>
	 * 
	 * <strong>Facet-Specific Context Propagation Rules:</strong>
	 * <ul>
	 * <li><strong>Facets</strong>: When removed from a context, the parent Resource that references it (via ConsistsOf relation) may also be removed from the same context if the Facet was required for the Resource's presence in that context</li>
	 * <li><strong>Associated Resources</strong>: Parent Resources connected to the Facet via ConsistsOf relations are automatically affected based on propagation constraints</li>
	 * <li><strong>ConsistsOf Relations</strong>: All ConsistsOf relations connecting parent Resources to this Facet are also removed from the context.</li>
	 * </ul>
	 * 
	 * <strong>Propagation Constraints and Cascade Effects:</strong>
	 * <ul>
	 * <li>When a Facet is removed from a context, related Resources may also be automatically removed due to propagation constraints</li>
	 * <li>The response includes all instances that were removed as part of the cascade effect</li>
	 * <li>Propagation follows the schema-defined relationships and their constraints.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation for Facets and ConsistsOf Relations:</strong>
	 * <ul>
	 * <li>When a Facet is removed from a context, the parent Resources are automatically validated against their schema definitions</li>
	 * <li>If any parent Resource is no longer compliant with its schema after the remove operation, a {@link SchemaViolationException} is thrown</li>
	 * <li>This results in an HTTP 400 Bad Request response with details about the schema violation</li>
	 * <li><strong>Example scenarios that violate schema constraints:</strong></li>
	 * </ul>
	 *   • Removing a required/mandatory Facet from a Resource;
	 *   • Removing Facets that violate cardinality constraints (min occurrences) defined in the schema;
	 *   • Operations that make parent Resources incomplete or invalid according to their schema definition.
	 * <ul>
	 * <li><strong>Constraint Sources</strong>: Constraints are defined by the specific Resource type definitions that have been installed on the system</li>
	 * <li>Schema validation ensures the integrity and consistency by enforcing the constraints defined in each Resource type's schema during operations.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and appropriate permissions for the specific context where they want to remove Facets.</li>
	 * </ul>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations on any Facet in any context</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations on any Facet in any context</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations only on Facets accessible in their current context</li>
	 * <li>Operations are limited to contexts where they have Context-Manager role</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations only on Facets accessible in their current context</li>
	 * <li>Operations are limited to Facets they have access to in their authorized context</li>
	 * <li>See filtered metadata with sensitive fields obfuscated when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON array containing all instances affected by the operation</li>
	 * <li>Each instance object includes its UUID, type, and relevant metadata</li>
	 * <li>For dryRun=true: Shows what would be affected without making changes</li>
	 * <li>For dryRun=false: Shows what was actually affected by the completed operation.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Remove a StateFacet instance from a specific context
	 * StateFacet facet = publisher.read(StateFacet.class, facetUUID);
	 * UUID targetContextUUID = UUID.fromString("...");
	 * List&lt;ERElement&gt; affected = publisher.removeFacetFromContext(facet, targetContextUUID, false);
	 * 
	 * // Preview what would be affected by removing an AccessPointFacet from a context
	 * AccessPointFacet accessPoint = publisher.read(AccessPointFacet.class, accessPointUUID);
	 * List&lt;ERElement&gt; preview = publisher.removeFacetFromContext(accessPoint, targetContextUUID, true);
	 * </pre>
	 * 
	 * @param <F> The type of Facet (StateFacet, ContactFacet, AccessPointFacet, etc.)
	 * @param facet The Facet instance to remove from the context (must contain a valid UUID in the header)
	 * @param contextUUID The UUID of the target context from which the Facet should be removed
	 * @param dryRun Whether to perform a dry-run (true) or apply actual changes (false)
	 * @return List of all elements affected by this operation (including the Facet itself and any related elements)
	 * @throws SchemaViolationException If the operation violates Information System model constraints (HTTP 400)
	 * @throws FacetNotFoundException If the Facet is not found or not accessible (HTTP 404)
	 * @throws ContextNotFoundException If the target context does not exist or is not accessible (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during the remove operation
	 */
	public <F extends Facet> List<ERElement> removeFacetFromContext(F facet, UUID contextUUID, Boolean dryRun)
			throws SchemaViolationException, FacetNotFoundException, ContextNotFoundException, ResourceRegistryException;

	/**
	 * Removes a Facet instance from the current context.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /sharing/contexts/{facet-type-name}/{uuid}?operation=REMOVE[&dryRun={true|false}][&includeMeta={true|false}][&allMeta={true|false}][&includeContexts={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Removes an existing Facet instance from the current context derived from the authorization token</li>
	 * <li>The Facet must already exist and be accessible in the current context where the user is performing the request</li>
	 * <li>The user must have the rights to perform such a request in the current context</li>
	 * <li>Returns the list of affected elements (the Facet itself and any related elements that are also removed)</li>
	 * <li>Supports dry-run mode to preview changes without applying them</li>
	 * <li>This is a convenience method that uses the current context instead of requiring an explicit context UUID</li>
	 * <li>This method is specifically designed for Facet types and provides type safety.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Facet successfully removed from current context (or dry-run completed)</li>
	 * <li><strong>404 Not Found</strong>: Facet does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: User lacks permissions to perform the request in the current context</li>
	 * <li><strong>400 Bad Request</strong>: Invalid parameters or schema validation failure</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response for affected instances</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, parent Resources, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the Facet and its related elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The type name must be a valid Facet type in the Information System model</li>
	 * <li>Supports all Facet types and their specializations (e.g., StateFacet, ContactFacet, AccessPointFacet)</li>
	 * <li>Type name is case-sensitive and must match exactly.</li>
	 * </ul>
	 * 
	 * <strong>Current Context:</strong>
	 * <ul>
	 * <li>The current context is automatically derived from the authorization token provided during client creation</li>
	 * <li>No explicit context UUID is required</li>
	 * <li>The user must have appropriate permissions in this context.</li>
	 * </ul>
	 * 
	 * <strong>Facet-Specific Context Propagation Rules:</strong>
	 * <ul>
	 * <li><strong>Facets</strong>: When removed from a context, the parent Resource that references it (via ConsistsOf relation) may also be removed from the same context if the Facet was required for the Resource's presence in that context</li>
	 * <li><strong>Associated Resources</strong>: Parent Resources connected to the Facet via ConsistsOf relations are automatically affected based on propagation constraints</li>
	 * <li><strong>ConsistsOf Relations</strong>: All ConsistsOf relations connecting parent Resources to this Facet are also removed from the context.</li>
	 * </ul>
	 * 
	 * <strong>Propagation Constraints and Cascade Effects:</strong>
	 * <ul>
	 * <li>When a Facet is removed from a context, related Resources may also be automatically removed due to propagation constraints</li>
	 * <li>The response includes all instances that were removed as part of the cascade effect</li>
	 * <li>Propagation follows the schema-defined relationships and their constraints.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation for Facets and ConsistsOf Relations:</strong>
	 * <ul>
	 * <li>When a Facet is removed from a context, the parent Resources are automatically validated against their schema definitions</li>
	 * <li>If any parent Resource is no longer compliant with its schema after the remove operation, a {@link SchemaViolationException} is thrown</li>
	 * <li>This results in an HTTP 400 Bad Request response with details about the schema violation</li>
	 * <li><strong>Example scenarios that violate schema constraints:</strong></li>
	 * </ul>
	 *   • Removing a required/mandatory Facet from a Resource;
	 *   • Removing Facets that violate cardinality constraints (min occurrences) defined in the schema;
	 *   • Operations that make parent Resources incomplete or invalid according to their schema definition.
	 * <ul>
	 * <li><strong>Constraint Sources</strong>: Constraints are defined by the specific Resource type definitions that have been installed on the system</li>
	 * <li>Schema validation ensures the integrity and consistency by enforcing the constraints defined in each Resource type's schema during operations.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and appropriate permissions for the current context where they want to remove Facets.</li>
	 * <li>The current context is derived from the authorization token provided during client creation.</li>
	 * </ul>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations on any Facet in any context</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations on any Facet in any context</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations only on Facets accessible in their current context</li>
	 * <li>Operations are limited to contexts where they have Context-Manager role</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations only on Facets accessible in their current context</li>
	 * <li>Operations are limited to Facets they have access to in their authorized context</li>
	 * <li>See filtered metadata with sensitive fields obfuscated when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON array containing all instances affected by the operation</li>
	 * <li>Each instance object includes its UUID, type, and relevant metadata</li>
	 * <li>For dryRun=true: Shows what would be affected without making changes</li>
	 * <li>For dryRun=false: Shows what was actually affected by the completed operation.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Remove a StateFacet from the current context with actual changes
	 * UUID stateFacetUUID = UUID.fromString("...");
	 * List&lt;ERElement&gt; affected = publisher.removeFacetFromCurrentContext("StateFacet", stateFacetUUID, false);
	 * 
	 * // Preview what would be affected by removing an AccessPointFacet from the current context
	 * UUID accessPointUUID = UUID.fromString("...");
	 * List&lt;ERElement&gt; preview = publisher.removeFacetFromCurrentContext("AccessPointFacet", accessPointUUID, true);
	 * </pre>
	 * 
	 * @param facetType The exact type name of the Facet to remove
	 * @param facetUUID The UUID of the Facet instance to remove from the current context
	 * @param dryRun Whether to perform a dry-run (true) or apply actual changes (false)
	 * @return List of all elements affected by this operation (including the Facet itself and any related elements)
	 * @throws SchemaViolationException If the operation violates Information System model constraints (HTTP 400)
	 * @throws FacetNotFoundException If the Facet is not found or not accessible (HTTP 404)
	 * @throws ContextNotFoundException If the current context cannot be determined or accessed (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during the remove operation
	 */
	public List<ERElement> removeFacetFromCurrentContext(String facetType, UUID facetUUID, Boolean dryRun) 
			throws SchemaViolationException, FacetNotFoundException, ContextNotFoundException, ResourceRegistryException;

	/**
	 * Removes a Facet instance from the current context using a provided Facet object.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /sharing/contexts/{facet-type-name}/{uuid}?operation=REMOVE[&dryRun={true|false}][&includeMeta={true|false}][&allMeta={true|false}][&includeContexts={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Removes an existing Facet instance from the current context derived from the authorization token</li>
	 * <li>The type and UUID are automatically extracted from the provided Facet object</li>
	 * <li>The Facet must already exist and be accessible in the current context where the user is performing the request</li>
	 * <li>The user must have the rights to perform such a request in the current context</li>
	 * <li>Returns the list of affected elements (the Facet itself and any related elements that are also removed)</li>
	 * <li>Supports dry-run mode to preview changes without applying them</li>
	 * <li>This is a convenience method that uses the current context instead of requiring an explicit context UUID</li>
	 * <li>This method provides compile-time type safety for Facet types.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Facet successfully removed from current context (or dry-run completed)</li>
	 * <li><strong>404 Not Found</strong>: Facet does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: User lacks permissions to perform the request in the current context</li>
	 * <li><strong>400 Bad Request</strong>: Invalid parameters or schema validation failure</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response for affected instances</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code publisher.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, parent Resources, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code publisher.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the Facet and its related elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code publisher.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The type name is automatically derived from the provided Facet object using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>The UUID is extracted from the Facet's header</li>
	 * <li>Supports all Facet types and their specializations (e.g., StateFacet, ContactFacet, AccessPointFacet).</li>
	 * </ul>
	 * 
	 * <strong>Current Context:</strong>
	 * <ul>
	 * <li>The current context is automatically derived from the authorization token provided during client creation</li>
	 * <li>No explicit context UUID is required</li>
	 * <li>The user must have appropriate permissions in this context.</li>
	 * </ul>
	 * 
	 * <strong>Facet-Specific Context Propagation Rules:</strong>
	 * <ul>
	 * <li><strong>Facets</strong>: When removed from a context, the parent Resource that references it (via ConsistsOf relation) may also be removed from the same context if the Facet was required for the Resource's presence in that context</li>
	 * <li><strong>Associated Resources</strong>: Parent Resources connected to the Facet via ConsistsOf relations are automatically affected based on propagation constraints</li>
	 * <li><strong>ConsistsOf Relations</strong>: All ConsistsOf relations connecting parent Resources to this Facet are also removed from the context.</li>
	 * </ul>
	 * 
	 * <strong>Propagation Constraints and Cascade Effects:</strong>
	 * <ul>
	 * <li>When a Facet is removed from a context, related Resources may also be automatically removed due to propagation constraints</li>
	 * <li>The response includes all instances that were removed as part of the cascade effect</li>
	 * <li>Propagation follows the schema-defined relationships and their constraints.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation for Facets and ConsistsOf Relations:</strong>
	 * <ul>
	 * <li>When a Facet is removed from a context, the parent Resources are automatically validated against their schema definitions</li>
	 * <li>If any parent Resource is no longer compliant with its schema after the remove operation, a {@link SchemaViolationException} is thrown</li>
	 * <li>This results in an HTTP 400 Bad Request response with details about the schema violation</li>
	 * <li><strong>Example scenarios that violate schema constraints:</strong></li>
	 * </ul>
	 *   • Removing a required/mandatory Facet from a Resource;
	 *   • Removing Facets that violate cardinality constraints (min occurrences) defined in the schema;
	 *   • Operations that make parent Resources incomplete or invalid according to their schema definition.
	 * <ul>
	 * <li><strong>Constraint Sources</strong>: Constraints are defined by the specific Resource type definitions that have been installed on the system</li>
	 * <li>Schema validation ensures the integrity and consistency by enforcing the constraints defined in each Resource type's schema during operations.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and appropriate permissions for the current context where they want to remove Facets.</li>
	 * <li>The current context is derived from the authorization token provided during client creation.</li>
	 * </ul>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations on any Facet in any context</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations on any Facet in any context</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations only on Facets accessible in their current context</li>
	 * <li>Operations are limited to contexts where they have Context-Manager role</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can perform REMOVE operations only on Facets accessible in their current context</li>
	 * <li>Operations are limited to Facets they have access to in their authorized context</li>
	 * <li>See filtered metadata with sensitive fields obfuscated when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON array containing all instances affected by the operation</li>
	 * <li>Each instance object includes its UUID, type, and relevant metadata</li>
	 * <li>For dryRun=true: Shows what would be affected without making changes</li>
	 * <li>For dryRun=false: Shows what was actually affected by the completed operation.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Remove a StateFacet instance from the current context
	 * StateFacet facet = publisher.read(StateFacet.class, facetUUID);
	 * List&lt;ERElement&gt; affected = publisher.removeFacetFromCurrentContext(facet, false);
	 * 
	 * // Preview what would be affected by removing an AccessPointFacet from the current context
	 * AccessPointFacet accessPoint = publisher.read(AccessPointFacet.class, accessPointUUID);
	 * List&lt;ERElement&gt; preview = publisher.removeFacetFromCurrentContext(accessPoint, true);
	 * </pre>
	 * 
	 * @param <F> The specific type of Facet to remove
	 * @param facet The Facet instance to remove from the current context (must contain a valid UUID in the header)
	 * @param dryRun Whether to perform a dry-run (true) or apply actual changes (false)
	 * @return List of all elements affected by this operation (including the Facet itself and any related elements)
	 * @throws SchemaViolationException If the operation violates Information System model constraints (HTTP 400)
	 * @throws FacetNotFoundException If the Facet is not found or not accessible (HTTP 404)
	 * @throws ContextNotFoundException If the current context cannot be determined or accessed (HTTP 404)
	 * @throws ResourceRegistryException If an error occurs during the remove operation
	 */
	public <F extends Facet> List<ERElement> removeFacetFromCurrentContext(F facet, Boolean dryRun) 
			throws SchemaViolationException, FacetNotFoundException, ContextNotFoundException, ResourceRegistryException;

	/**
	 * Retrieves the list of contexts where a specific Facet is present in the Resource Registry.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /instances/{facet-type-name}/{facet-uuid}/contexts}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves all contexts where the specified Facet is present</li>
	 * <li>The Facet must exist and be accessible to the user</li>
	 * <li>Returns a map where keys are context UUIDs and values are the full context paths</li>
	 * <li>No filtering, pagination, or metadata options are available</li>
	 * <li>Context information is determined by the Facet's presence across different VREs and organizational contexts.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Context list successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: Facet does not exist with the specified type and UUID</li>
	 * <li><strong>403 Forbidden</strong>: Facet exists but is not accessible for context listing in the current authorization scope</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager:</strong> Can retrieve context information for any Facet across the entire infrastructure</li>
	 * <li><strong>Infrastructure-Manager:</strong> Can retrieve context information for any Facet across the entire infrastructure</li>
	 * <li><strong>Context-Manager:</strong> Can retrieve context information for Facets accessible within their managed contexts</li>
	 * <li><strong>Other Users:</strong> Can retrieve context information for Facets accessible through their authorization token.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a Map where keys are context UUIDs (UUID) and values are full context paths (String)</li>
	 * <li>Empty map is returned if the Facet exists but is not present in any contexts</li>
	 * <li>Each context path represents the complete hierarchical path (e.g., "/gcube/devsec/devVRE").</li>
	 * </ul>
	 * 
	 * <strong>Facet Context Rules:</strong>
	 * <ul>
	 * <li><strong>Facets:</strong> Present in the contexts where they are explicitly available</li>
	 * <li>A Facet's context availability depends on the Resources that include it via ConsistsOf relations</li>
	 * <li>Facets are present in contexts based on their direct context assignments and their association with Resources.</li>
	 * </ul>
	 * 
	 * <strong>Type Safety:</strong>
	 * <ul>
	 * <li>This method is specifically designed for Facet types and provides compile-time type safety</li>
	 * <li>Supports all Facet types and their specializations (e.g., StateFacet, ContactFacet, AccessPointFacet, SoftwareFacet).</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Get contexts for a ContactFacet by type and UUID
	 * UUID contactUuid = UUID.fromString("4023d5b2-8601-47a5-83ef-49ffcbfc7d86");
	 * Map&lt;UUID, String&gt; contexts = publisher.getFacetContexts("ContactFacet", contactUuid);
	 * 
	 * // Process the results
	 * for (Map.Entry&lt;UUID, String&gt; entry : contexts.entrySet()) {
	 *     System.out.println("Context UUID: " + entry.getKey() + " -> Path: " + entry.getValue());
	 * }
	 * </pre>
	 * 
	 * @param facetType The exact type name of the Facet (e.g., "ContactFacet", "StateFacet")
	 * @param facetUUID The unique identifier of the Facet whose contexts to retrieve
	 * @return Map containing context UUID to context path mappings where the Facet is present
	 * @throws FacetNotFoundException If no Facet is found with the specified type and UUID
	 * @throws ResourceRegistryException If an error occurs during context retrieval
	 */
	public Map<UUID, String> getFacetContexts(String facetType, UUID facetUUID)
			throws FacetNotFoundException, ResourceRegistryException;

	/**
	 * Retrieves the list of contexts where a specific Facet is present in the Resource Registry.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /instances/{facet-type-name}/{facet-uuid}/contexts}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves all contexts where the specified Facet object is present</li>
	 * <li>The Facet's type and UUID are automatically extracted from the provided object</li>
	 * <li>The Facet must exist and be accessible to the user</li>
	 * <li>Returns a map where keys are context UUIDs and values are the full context paths</li>
	 * <li>No filtering, pagination, or metadata options are available</li>
	 * <li>Context information is determined by the Facet's presence across different VREs and organizational contexts.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Context list successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: Facet does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: Facet exists but is not accessible for context listing in the current authorization scope</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager:</strong> Can retrieve context information for any Facet across the entire infrastructure</li>
	 * <li><strong>Infrastructure-Manager:</strong> Can retrieve context information for any Facet across the entire infrastructure</li>
	 * <li><strong>Context-Manager:</strong> Can retrieve context information for Facets accessible within their managed contexts</li>
	 * <li><strong>Other Users:</strong> Can retrieve context information for Facets accessible through their authorization token.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a Map where keys are context UUIDs (UUID) and values are full context paths (String)</li>
	 * <li>Empty map is returned if the Facet exists but is not present in any contexts</li>
	 * <li>Each context path represents the complete hierarchical path (e.g., "/gcube/devsec/devVRE").</li>
	 * </ul>
	 * 
	 * <strong>Facet Context Rules:</strong>
	 * <ul>
	 * <li><strong>Facets:</strong> Present in the contexts where they are explicitly available</li>
	 * <li>A Facet's context availability depends on the Resources that include it via ConsistsOf relations</li>
	 * <li>Facets are present in contexts based on their direct context assignments and their association with Resources.</li>
	 * </ul>
	 * 
	 * <strong>Type Safety:</strong>
	 * <ul>
	 * <li>This method provides compile-time type safety by accepting any Facet subtype</li>
	 * <li>The Facet's type is automatically determined from the object's class</li>
	 * <li>Supports all Facet types and their specializations (e.g., StateFacet, ContactFacet, AccessPointFacet, SoftwareFacet).</li>
	 * </ul>
	 * 
	 * <strong>Object-based Operation:</strong>
	 * <ul>
	 * <li>This method accepts a Facet instance object directly, which provides convenience and type safety</li>
	 * <li>The Facet object must contain a valid UUID in its header metadata</li>
	 * <li>The type is automatically inferred from the object's runtime class.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryPublisher publisher = ResourceRegistryPublisherFactory.create();
	 * 
	 * // Assuming you have a ContactFacet instance
	 * ContactFacet contact = ...; // Retrieved from previous operations
	 * 
	 * // Get contexts for the facet object
	 * Map&lt;UUID, String&gt; contexts = publisher.getFacetContexts(contact);
	 * 
	 * // Process the results
	 * for (Map.Entry&lt;UUID, String&gt; entry : contexts.entrySet()) {
	 *     System.out.println("Context UUID: " + entry.getKey() + " -> Path: " + entry.getValue());
	 * }
	 * </pre>
	 * 
	 * @param <F> The type of Facet (ContactFacet, StateFacet, AccessPointFacet, etc.)
	 * @param facet The Facet instance whose contexts to retrieve
	 * @return Map containing context UUID to context path mappings where the Facet is present
	 * @throws FacetNotFoundException If the specified Facet does not exist or is not accessible
	 * @throws ResourceRegistryException If an error occurs during context retrieval
	 */
	public <F extends Facet> Map<UUID, String> getFacetContexts(F facet)
			throws FacetNotFoundException, ResourceRegistryException;

}
