package org.gcube.informationsystem.resourceregistry.rest;

import java.util.UUID;

import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;

import org.gcube.informationsystem.model.entity.Context;
import org.gcube.informationsystem.resourceregistry.ResourceInitializer;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.context.ContextAlreadyPresentException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.context.ContextException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.context.ContextNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.rest.ContextPath;
import org.gcube.informationsystem.resourceregistry.context.ContextManagement;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Luca Frosini (ISTI - CNR)
 */
@Path(ContextPath.CONTEXT_PATH_PART)
public class ContextManager {

	/**
	 * Logger
	 */
	private static Logger logger = LoggerFactory
			.getLogger(ContextManager.class);

	public static final String ID_PATH_PARAM = "id";

	/**
	 * e.g. PUT /resource-registry/context
	 * 
	 * BODY: {...}
	 * 
	 */
	@PUT
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	public Response create(String json)
			throws ContextAlreadyPresentException, ResourceRegistryException {
		logger.info("Requested to create {} with json {}", Context.NAME, json);
		ContextManagement contextManagement = new ContextManagement();
		contextManagement.setJSON(json);
		String ret = contextManagement.create();  
		return Response.status(Status.CREATED).entity(ret).type(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8).build();
	}
	
	/**
	 * e.g. GET /resource-registry/context/c0f314e7-2807-4241-a792-2a6c79ed4fd0
	 * @param uuid
	 * @return
	 * @throws ContextException
	 */
	@GET
	@Path("{" + ID_PATH_PARAM + "}")
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	public String read(@PathParam(ID_PATH_PARAM) String uuid)
			throws ContextNotFoundException, ResourceRegistryException {
		ContextManagement contextManagement = new ContextManagement();
		if(uuid.compareTo(ContextPath.ALL_PATH_PART)==0) {
			logger.info("Requested to read all {}s", Context.NAME);
			return contextManagement.all(false);
		}else {
			logger.info("Requested to read {} with id {} ", Context.NAME, uuid);
			contextManagement.setUUID(UUID.fromString(uuid));
			return contextManagement.read();
		}
	}
	
	/**
	 * e.g. POST /resource-registry/context
	 * 
	 * BODY: {...}
	 * 
	 */
	@POST
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	public String update(String json)
			throws ContextNotFoundException, ResourceRegistryException {
		logger.info("Requested to update {} with json {} ", Context.NAME, json);
		ContextManagement contextManagement = new ContextManagement();
		contextManagement.setJSON(json);
		return contextManagement.update();
	}
	
	/**
	 * e.g. DELETE /resource-registry/context/c0f314e7-2807-4241-a792-2a6c79ed4fd0
	 * @param uuid
	 * @return
	 * @throws ContextException
	 */
	@DELETE
	@Path("{" + ID_PATH_PARAM + "}")
	public boolean delete(@PathParam(ID_PATH_PARAM) String uuid)
			throws ContextNotFoundException, ResourceRegistryException {
		logger.info("Requested to delete {} with id {} ", Context.NAME, uuid);
		ContextManagement contextManagement = new ContextManagement();
		contextManagement.setUUID(UUID.fromString(uuid));
		return contextManagement.delete();
	}
	
}
