package org.gcube.informationsystem.resourceregistry.instances.base.properties;

import java.security.Key;
import java.util.HashSet;
import java.util.Set;

import org.gcube.com.fasterxml.jackson.databind.JsonNode;
import org.gcube.com.fasterxml.jackson.databind.ObjectMapper;
import org.gcube.com.fasterxml.jackson.databind.node.ObjectNode;
import org.gcube.informationsystem.base.reference.AccessType;
import org.gcube.informationsystem.base.reference.Element;
import org.gcube.informationsystem.model.impl.properties.EncryptedImpl;
import org.gcube.informationsystem.model.reference.properties.Encrypted;
import org.gcube.informationsystem.model.reference.properties.Header;
import org.gcube.informationsystem.model.reference.properties.Property;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.schema.SchemaNotFoundException;
import org.gcube.informationsystem.resourceregistry.dbinitialization.DatabaseEnvironment;
import org.gcube.informationsystem.resourceregistry.instances.base.ElementManagement;
import org.gcube.informationsystem.resourceregistry.instances.base.ElementManagementUtility;
import org.gcube.informationsystem.resourceregistry.utils.EncryptedOrient;
import org.gcube.informationsystem.resourceregistry.utils.HeaderUtility;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.orientechnologies.orient.core.metadata.schema.OClass;
import com.orientechnologies.orient.core.record.impl.ODocument;

/**
 * @author Luca Frosini (ISTI - CNR)
 */
public class PropertyElementManagement {
	
	private static Logger logger = LoggerFactory.getLogger(PropertyElementManagement.class);
	
	public static final Set<String> PROPERTY_IGNORE_KEYS;
	public static final Set<String> PROPERTY_IGNORE_START_WITH_KEYS;
	
	public static final String AT = "@";
	public static final String UNDERSCORE = "_";
	
	static {
		PROPERTY_IGNORE_KEYS = new HashSet<String>();
		
		PROPERTY_IGNORE_START_WITH_KEYS = new HashSet<String>();
		PROPERTY_IGNORE_START_WITH_KEYS.add(AT);
		PROPERTY_IGNORE_START_WITH_KEYS.add(UNDERSCORE);
		
	}
	
	public static ODocument getPropertyDocument(JsonNode jsonNode) throws ResourceRegistryException {
		ODocument oDocument = null;
		if(jsonNode.has(Element.CLASS_PROPERTY)) {
			// Complex type
			String type = ElementManagement.getClassProperty(jsonNode);
			OClass oClass = null;
			
			try {
				oClass = ElementManagementUtility.getTypeSchema(type, AccessType.PROPERTY_ELEMENT);
			} catch(SchemaNotFoundException e) {
				throw e;
			}
			
			try {
				Header header = HeaderUtility.getHeader(jsonNode, false);
				if(header != null) {
					throw new ResourceRegistryException("A property object cannot have an Header");
				}
			} catch(Exception e) {
				logger.warn("An invalid Header has been provided. Anyway property object cannot have an Header.");
				throw new ResourceRegistryException("An property object cannot have an Header");
			}
			
			
			/*
			 * In case it is an Encrypted type the Value received arrives encrypted with the Context Key
			 * Resource Registry must decrypt the value with the Context Key and Encrypt it with DB key.
			 * The opposite operation is done when the value is read by clients. 
			 */
			if(oClass.isSubClassOf(Encrypted.NAME)) {
				EncryptedOrient encrypted = new EncryptedOrient();
				oDocument = encrypted;
				oDocument.fromJSON(jsonNode.toString());
				try {
					String contextEncryptedValue = encrypted.getEncryptedValue();
					
					// Decrypting with Context Key (default key)
					String decryptedValue = EncryptedImpl.decrypt(contextEncryptedValue); 
					
					encrypted.setDecryptedValue(decryptedValue, false);
					
					/*
					// Encrypting with DB Key
					Key databaseKey = DatabaseEnvironment.getDatabaseKey();
					String dbEncryptedValue = EncryptedImpl.encrypt(decryptedValue, databaseKey);
					
					// Setting the value encrypted with DB key
					encrypted.setEncryptedValue(dbEncryptedValue);
					*/
					
				} catch(Exception e) {
					throw new ResourceRegistryException("Unable to manage "+Encrypted.NAME+" "+org.gcube.informationsystem.model.reference.properties.Property.NAME);
				}
				return oDocument;
			}
			
			oDocument = new ODocument(type);
		} else {
			oDocument = new ODocument();
		}
		return oDocument.fromJSON(jsonNode.toString());
	}
	
	
	public static JsonNode getJsonNode(ODocument oDocument) throws ResourceRegistryException {
		try {
			String type = oDocument.getClassName();
			String json = oDocument.toJSON("class");
			
			ObjectMapper objectMapper = new ObjectMapper();
			JsonNode jsonNode = objectMapper.readTree(json);
			
			
			if(type==null) {
				return jsonNode;
			}
			
			OClass oClass = ElementManagementUtility.getTypeSchema(type, AccessType.PROPERTY_ELEMENT);
			
			/*
			 * In case it is an Encrypted type the value is encrypted with the DB Key
			 * Resource Registry must decrypt the value with the DB Key and Encrypt it with Context key.
			 * The opposite operation is done when the value is set from clients.
			 * see {@link PropertyManagement#getPropertyDocument(JsonNode) getPropertyDocument()} 
			 */
			if(oClass.isSubClassOf(Encrypted.NAME)) {
				try {
					
					EncryptedOrient encrypted = null;
					String encryptedValue = (String) oDocument.getProperty(Encrypted.VALUE);
					
					if(oDocument instanceof EncryptedOrient) {
						encrypted = (EncryptedOrient) oDocument;
						if(encrypted.getDbEncryptedValue().compareTo(encryptedValue)==0) {
							// encrypted.setEncryptedValue(encrypted.getContextEncryptedValue());
							((ObjectNode) jsonNode).put(Encrypted.VALUE, encrypted.getContextEncryptedValue());
						}
					}else {
						encrypted = new EncryptedOrient();
						oDocument = (ODocument) encrypted;
					
						// Decrypting with DB Key
						Key databaseKey = DatabaseEnvironment.getDatabaseKey();
						String decryptedValue = EncryptedImpl.decrypt(encryptedValue, databaseKey); 
						
						// encrypted.setDecryptedValue(decryptedValue, true);
						
						// Encrypting with Context Key (default key)
						String contextEncryptedValue = EncryptedImpl.encrypt(decryptedValue);
						
						// Setting the value encrypted with DB key
						//encrypted.setEncryptedValue(contextEncryptedValue);
						((ObjectNode) jsonNode).put(Encrypted.VALUE, contextEncryptedValue);
					}
					
				}catch (Exception e) {
					throw new ResourceRegistryException("Errror while managing " + Encrypted.NAME+ " "+ Property.NAME, e);
				}
			}

			return jsonNode;
			
		} catch (ResourceRegistryException e) {
			throw e;
		} catch (Exception e) {
			throw new ResourceRegistryException(e);
		}
	}
}
