package org.gcube.informationsystem.resourceregistry.query;

import java.util.HashMap;
import java.util.Map;

import org.gcube.com.fasterxml.jackson.databind.JsonNode;
import org.gcube.com.fasterxml.jackson.databind.ObjectMapper;
import org.gcube.com.fasterxml.jackson.databind.node.ArrayNode;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.query.InvalidQueryException;
import org.gcube.informationsystem.resourceregistry.api.rest.AccessPath;
import org.gcube.informationsystem.resourceregistry.contexts.ContextUtility;
import org.gcube.informationsystem.resourceregistry.contexts.security.SecurityContext;
import org.gcube.informationsystem.resourceregistry.contexts.security.SecurityContext.PermissionMode;
import org.gcube.informationsystem.resourceregistry.instances.base.ElementManagement;
import org.gcube.informationsystem.resourceregistry.instances.base.ElementManagementUtility;
import org.gcube.informationsystem.resourceregistry.utils.Utility;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.orientechnologies.orient.core.db.document.ODatabaseDocument;
import com.orientechnologies.orient.core.record.OElement;
import com.orientechnologies.orient.core.sql.executor.OResult;
import com.orientechnologies.orient.core.sql.executor.OResultSet;

/**
 * @author Luca Frosini (ISTI - CNR)
 */
public class QueryImpl implements Query {
	
	private static Logger logger = LoggerFactory.getLogger(QueryImpl.class);
	
	@Override
	public String query(String query, Integer limit, String fetchPlan, boolean raw) throws InvalidQueryException {
		if(limit == null) {
			limit = AccessPath.DEFAULT_LIMIT;
		}
		limit = (limit <= 0) ? AccessPath.UNBOUNDED : limit;
		
		ODatabaseDocument oDatabaseDocument = null;
		ODatabaseDocument current = ContextUtility.getCurrentODatabaseDocumentFromThreadLocal();
		
		try {
			SecurityContext securityContext = ContextUtility.getCurrentSecurityContext();
			
			oDatabaseDocument = securityContext.getDatabaseDocument(PermissionMode.READER);
			oDatabaseDocument.begin();
			
			StringBuffer stringBuffer = new StringBuffer();
			stringBuffer.append(query);
			stringBuffer.append(" limit :limit");

			Map<String, Object> map = new HashMap<>();
			map.put("limit", limit);
			
			if(fetchPlan!=null) {
				stringBuffer.append(" fetchplan ");
				stringBuffer.append(fetchPlan);
				logger.debug("Going to execute query '{} limit {} fetchPlan {}'", query, limit, fetchPlan);
			}else {
				logger.debug("Going to execute query '{} limit {}'", query, limit);
			}
			
			OResultSet resultSet = oDatabaseDocument.query(stringBuffer.toString(), map);
			
			ObjectMapper objectMapper = new ObjectMapper();
			ArrayNode arrayNode = objectMapper.createArrayNode();
			
			while(resultSet.hasNext()) {
				OResult oResult = resultSet.next();
				OElement element = ElementManagement.getElementFromOptional(oResult.getElement());
				
				try {
					JsonNode jsonNode = null;
					if(raw) {
						jsonNode = Utility.toJsonNode(element, false);
					} else {
						@SuppressWarnings("rawtypes")
						ElementManagement erManagement = ElementManagementUtility.getERManagement(securityContext, oDatabaseDocument,
								element);
						jsonNode = erManagement.serializeAsJson();
					}
					arrayNode.add(jsonNode);
					
				} catch(ResourceRegistryException e) {
					logger.error("Unable to correctly serialize {}. It will be excluded from results. {}",
							element.toString(), Utility.SHOULD_NOT_OCCUR_ERROR_MESSAGE);
				}
			}
			
			return objectMapper.writeValueAsString(arrayNode);
			
		} catch(Exception e) {
			throw new InvalidQueryException(e.getMessage());
		} finally {
			if(oDatabaseDocument != null) {
				oDatabaseDocument.close();
			}
			if(current!=null) {
				current.activateOnCurrentThread();
			}
		}
		
	}
	
	@Override
	public String gremlinQuery(String query) throws InvalidQueryException {
		throw new UnsupportedOperationException();
		
		/*
		OGremlinHelper.global().create();
		
		ODatabaseDocumentTx oDatabaseDocumentTx = null;
		try {
			oDatabaseDocumentTx = ContextUtility.getActualSecurityContextDatabaseTx(PermissionMode.READER);
			
			String finalQuery = String.format("select gremlin('%s')", query);
			OCommandSQL OCommandSQL = new OCommandSQL(finalQuery);
			OCommandRequest oCommandRequest = oDatabaseDocumentTx.command(OCommandSQL);
			OBasicResultSet res = oCommandRequest.execute();
			
			Iterator iterator = res.iterator();
			
			while(iterator.hasNext()) {
				ODocument oDocument = (ODocument) iterator.next();
				logger.debug("{}", oDocument);
			}
			
			return res.toString();
			
		} catch(Exception e) {
			throw new InvalidQueryException(e.getMessage());
		} finally {
			if(oDatabaseDocumentTx != null) {
				oDatabaseDocumentTx.close();
			}
		}
		*/
	}
	
}
