package org.gcube.informationsystem.resourceregistry.rest;

import java.util.UUID;

import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;

import org.gcube.common.authorization.library.provider.CalledMethodProvider;
import org.gcube.informationsystem.contexts.reference.entities.Context;
import org.gcube.informationsystem.resourceregistry.ResourceInitializer;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.contexts.ContextNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.rest.ContextPath;
import org.gcube.informationsystem.resourceregistry.contexts.ContextUtility;
import org.gcube.informationsystem.resourceregistry.contexts.entities.ContextManagement;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Luca Frosini (ISTI - CNR)
 */
@Path(ContextPath.CONTEXTS_PATH_PART)
public class ContextManager {
	
	public static final String CONTEXT_UUID_PATH_PARAMETER = "CONTEXT_UUID";
	
	/**
	 * Logger
	 */
	private static Logger logger = LoggerFactory.getLogger(ContextManager.class);

	public ContextManager() {
		ContextUtility.getHierarchicalMode().set(false);
		ContextUtility.getIncludeInstanceContexts().set(false);
	}
	
	/*
	 * GET /contexts
	 * 
	 */
	@GET
	@Consumes({MediaType.TEXT_PLAIN, ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8})
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	public String all() throws ContextNotFoundException, ResourceRegistryException {
		logger.info("Requested to read all {}s", Context.NAME);
		CalledMethodProvider.instance.set("listContexts");
		
		ContextManagement contextManagement = new ContextManagement();
		return contextManagement.all(false);
	}
	
	/*
	 * GET /contexts/{UUID}
	 * e.g. GET /contexts/c0f314e7-2807-4241-a792-2a6c79ed4fd0
	 * 
	 */
	@GET
	@Path("{" + ContextManager.CONTEXT_UUID_PATH_PARAMETER + "}")
	@Consumes({MediaType.TEXT_PLAIN, ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8})
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	public String read(@PathParam(ContextManager.CONTEXT_UUID_PATH_PARAMETER) String uuid)
			throws ContextNotFoundException, ResourceRegistryException {
		if(uuid.compareTo(ContextPath.CURRENT_CONTEXT_PATH_PART)==0){
			uuid = ContextUtility.getCurrentSecurityContext().getUUID().toString();
		}
		logger.info("Requested to read {} with id {} ", Context.NAME, uuid);
		CalledMethodProvider.instance.set("readContext");
		
		ContextManagement contextManagement = new ContextManagement();
		contextManagement.setUUID(UUID.fromString(uuid));
		return contextManagement.readAsString();
	}
	
	/*
	 * PUT /contexts/{UUID}
	 * e.g. PUT /contexts/c0f314e7-2807-4241-a792-2a6c79ed4fd0
	 * 
	 * BODY: {...}
	 * 
	 */
	@PUT
	@Path("{" + ContextManager.CONTEXT_UUID_PATH_PARAMETER + "}")
	@Consumes({MediaType.TEXT_PLAIN, ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8})
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	public String updateCreate(@PathParam(ContextManager.CONTEXT_UUID_PATH_PARAMETER) String uuid, String json)
			throws ResourceRegistryException {
		logger.info("Requested to update/create {} with json {} ", Context.NAME, json);
		CalledMethodProvider.instance.set("updateContext");
		
		ContextManagement contextManagement = new ContextManagement();
		contextManagement.setUUID(UUID.fromString(uuid));
		contextManagement.setJson(json);
		return contextManagement.createOrUpdate();
	}
	
	/*
	 * DELETE /contexts/{UUID}
	 * e.g. DELETE /contexts/c0f314e7-2807-4241-a792-2a6c79ed4fd0
	 */
	@DELETE
	@Consumes({MediaType.TEXT_PLAIN, ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8})
	@Path("{" + ContextManager.CONTEXT_UUID_PATH_PARAMETER + "}")
	public Response delete(@PathParam(ContextManager.CONTEXT_UUID_PATH_PARAMETER) String uuid)
			throws ContextNotFoundException, ResourceRegistryException {
		logger.info("Requested to delete {} with id {} ", Context.NAME, uuid);
		CalledMethodProvider.instance.set("deleteContext");
		
		ContextManagement contextManagement = new ContextManagement();
		contextManagement.setUUID(UUID.fromString(uuid));
		contextManagement.delete();
		
		return Response.status(Status.NO_CONTENT).build();
	}
	
}
