package org.gcube.informationsystem.resourceregistry.utils;

import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.UUID;

import org.gcube.com.fasterxml.jackson.core.JsonParseException;
import org.gcube.com.fasterxml.jackson.databind.JsonMappingException;
import org.gcube.com.fasterxml.jackson.databind.JsonNode;
import org.gcube.common.authorization.utils.manager.SecretManagerProvider;
import org.gcube.informationsystem.base.reference.IdentifiableElement;
import org.gcube.informationsystem.model.reference.properties.Header;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.serialization.ElementMapper;
import org.gcube.informationsystem.utils.UUIDManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.orientechnologies.orient.core.record.OElement;
import com.orientechnologies.orient.core.record.impl.ODocument;

/**
 * @author Luca Frosini (ISTI - CNR)
 */
public class HeaderUtility {
	
	private static final Logger logger = LoggerFactory.getLogger(HeaderUtility.class);
	
	public static String getUser() {
		String user = Header.UNKNOWN_USER;
		try {
			user = SecretManagerProvider.instance.get().getUser().getUsername();
		} catch(Exception e) {
			logger.error("Unable to retrieve user. {} will be used", user);
		}
		return user;
	}
	
	public static Header createHeader(UUID uuid) {
		HeaderOrient header = new HeaderOrient();
		
		if(uuid == null) {
			uuid = UUIDManager.generateValidRandomUUID();
		}
		
		header.setUUID(uuid);
		
		String creator = getUser();
		header.setCreatedBy(creator);
		header.setLastUpdateBy(creator);
		
		Date date = Calendar.getInstance().getTime();
		SimpleDateFormat ft = new SimpleDateFormat("E yyyy.MM.dd 'at' hh:mm:ss a zzz");
		logger.trace("Setting Last Update and Creation Time to " + ft.format(date));
		
		header.setCreationTime(date);
		header.setLastUpdateTime(date);
		
		return header;
	}
	
	public static Header getHeader(JsonNode jsonNode, boolean creation)
			throws JsonParseException, JsonMappingException, IOException, ResourceRegistryException {
		if(jsonNode.has(IdentifiableElement.HEADER_PROPERTY)) {
			JsonNode headerNode = jsonNode.get(IdentifiableElement.HEADER_PROPERTY);
			if(headerNode.isNull()) {
				return null;
			}
			HeaderOrient header = null;
			if(creation) {
				// If an header is provided, it MUST contains an UUID otherwise is
				// an invalid request so that let that an exception is raised
				UUID uuid = UUID.fromString(headerNode.get(Header.UUID_PROPERTY).asText());
				if(UUIDManager.isReservedUUID(uuid)) {
					throw new ResourceRegistryException("The provided UUID " + uuid.toString() + "is reserved. The reserved UUID are : " + UUIDManager.getAllReservedUUIDAsStrings());
				}
				header = (HeaderOrient) createHeader(uuid);
			} else {
				header = new HeaderOrient();
				header.fromJSON(headerNode.toString());
			}
			return header;
		}
		return null;
	}
	
	public static HeaderOrient getHeaderOrient(ODocument oDocument) throws ResourceRegistryException {
		if(oDocument instanceof HeaderOrient) {
			return (HeaderOrient) oDocument;
		} else {
			try {
				HeaderOrient headerOrient = new HeaderOrient();
				String json = oDocument.toJSON();
				Header header = ElementMapper.unmarshal(Header.class, json);
				headerOrient.setUUID(header.getUUID());
				headerOrient.setCreatedBy(header.getCreatedBy());
				headerOrient.setCreationTime(header.getCreationTime());
				headerOrient.setLastUpdateBy(header.getLastUpdateBy());
				headerOrient.setLastUpdateTime(header.getLastUpdateTime());
				return headerOrient;
			} catch(Exception e) {
				throw new ResourceRegistryException(
						"Unable to recreate Header. " + Utility.SHOULD_NOT_OCCUR_ERROR_MESSAGE);
			}
		}
	}
	
	public static Header addHeader(OElement element, UUID uuid) {
		Header header = createHeader(uuid);
		element.setProperty(IdentifiableElement.HEADER_PROPERTY, header);
		return header;
	}
	
	/*
	public static Header addHeader(Edge edge, UUID uuid) {
		Header header = createHeader(uuid);
		edge.setProperty(IdentifiableElement.HEADER_PROPERTY, header);
		return header;
	}
	*/
	
	public static Header getHeader(OElement element) throws ResourceRegistryException {
		return Utility.getPropertyDocument(Header.class, element, IdentifiableElement.HEADER_PROPERTY);
	}
	
	public static void updateModifiedByAndLastUpdate(OElement element) throws ResourceRegistryException {
		ODocument oDocument = element.getProperty(IdentifiableElement.HEADER_PROPERTY);
		String lastUpdateBy = getUser();
		oDocument.field(Header.LAST_UPDATE_BY_PROPERTY, lastUpdateBy);
		Date lastUpdateTime = Calendar.getInstance().getTime();
		oDocument.field(Header.LAST_UPDATE_TIME_PROPERTY, lastUpdateTime);
		element.setProperty(IdentifiableElement.HEADER_PROPERTY, oDocument);
	}
	
}
