package org.gcube.informationsystem.resourceregistry.requests;

import java.util.List;

import javax.ws.rs.core.UriInfo;

import org.gcube.informationsystem.base.reference.IdentifiableElement;
import org.gcube.informationsystem.model.reference.properties.Metadata;
import org.gcube.informationsystem.resourceregistry.api.request.RequestInfo;
import org.gcube.informationsystem.resourceregistry.api.rest.InstancePath;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Luca Frosini (ISTI - CNR)
 */
public class ServerRequestInfo implements RequestInfo {
	
	protected static Logger logger = LoggerFactory.getLogger(ServerRequestInfo.class);
	
	protected UriInfo uriInfo;
	
	/**
	 * Track if the request requested to include {@link Metadata} 
	 */
	protected boolean includeMeta;
	
	/**
	 * Track if the request requested to include {@link Metadata} in all
	 * {@link IdentifiableElement} or just in the root instance 
	 */
	protected boolean allMeta;
	
	
	/**
	 * Track if hierarchicalMode has been requested 
	 */
	protected boolean hierarchicalMode;
	
	/**
	 * Track if the request requested to include contexts 
	 */	
	protected boolean includeContexts;

	public ServerRequestInfo() {
		this.includeMeta = false;
		this.allMeta = false;
		this.hierarchicalMode = false;
		this.includeContexts = false;
		this.uriInfo = null;
	}
	
	public UriInfo getUriInfo() {
		return uriInfo;
	}

	/**
	 * -check if the user is allowed to set such a request
	 * @param queryParameterKey requested query parameter
	 * @return true is the user is allowed to set such a request
	 */
	public boolean isAllowed(String queryParameterKey) {
		switch (queryParameterKey) {
			case InstancePath.INCLUDE_META_QUERY_PARAMETER:
				// TODO check is the user has the role to request such parameter
				return true;
			
			case InstancePath.INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER:
				// TODO check is the user has the role to request such parameter
				return true;
				
			case InstancePath.INCLUDE_CONTEXTS_QUERY_PARAMETER:
				// TODO check is the user has the role to request such parameter
				return true;
	
			case InstancePath.HIERARCHICAL_MODE_QUERY_PARAMETER:
				// TODO check is the user has the role to request such parameter
				return true;
				
			default:
				break;
		}
		
		return true;
	}
	
	/**
	 * Set the parameter if the user is allowed otherwise the default is maintained 
	 * @param queryParameterKey requested query parameter
	 * @param bool the value to set
	 * @return the value of variable corresponding the request parameter independetly if
	 * the value has been set.
	 */
	public boolean setIfAllowed(String queryParameterKey, boolean bool) {
		boolean toBeSet = bool && isAllowed(queryParameterKey);
		
		switch (queryParameterKey) {
			case InstancePath.INCLUDE_META_QUERY_PARAMETER:
				includeMeta = toBeSet;
				return includeMeta;
			
			case InstancePath.INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER:
				allMeta =  toBeSet;
				return allMeta;
				
			case InstancePath.INCLUDE_CONTEXTS_QUERY_PARAMETER:
				includeContexts = toBeSet;
				return includeContexts;
	
			case InstancePath.HIERARCHICAL_MODE_QUERY_PARAMETER:
				hierarchicalMode = toBeSet;
				return hierarchicalMode;
				
			default:
				break;
		}
		
		return false;
	}
	
	public void checkQueryParameter(String queryParameterKey) {
		try {
			List<String> queryParameterList = uriInfo.getQueryParameters().get(queryParameterKey);
			if(queryParameterList!=null && queryParameterList.size()>0) {
				String booleanString = queryParameterList.get(0);
				boolean bool = Boolean.valueOf(booleanString);
				setIfAllowed(queryParameterKey, bool);
			}
		}catch (Throwable t) {
			logger.warn("Unable to properly set the Hierarchical Mode is set", t.getMessage());
		}
	}
	
	public void setUriInfo(UriInfo uriInfo) {
		this.uriInfo = uriInfo;
	}
	
	public void checkAllQueryParameters() {
		checkIncludeQueryParameters();
		checkQueryParameter(InstancePath.HIERARCHICAL_MODE_QUERY_PARAMETER);
	}
	
	public void checkIncludeQueryParameters() {
		checkIncludeAllMetaQueryParameters();
		checkQueryParameter(InstancePath.INCLUDE_CONTEXTS_QUERY_PARAMETER);
	}

	public void checkIncludeAllMetaQueryParameters() {
		checkQueryParameter(InstancePath.INCLUDE_META_QUERY_PARAMETER);
		checkQueryParameter(InstancePath.INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER);
	}
	
	@Override
	public boolean includeMeta() {
		return includeMeta;
	}

	@Override
	public void setIncludeMeta(boolean includeMeta) {
		this.includeMeta = includeMeta;
	}

	@Override
	public boolean allMeta() {
		return allMeta;
	}

	@Override
	public void setAllMeta(boolean allMeta) {
		this.allMeta = allMeta;
	}
	
	@Override
	public boolean isHierarchicalMode() {
		return hierarchicalMode;
	}
	
	@Override
	public void setHierarchicalMode(boolean hierarchicalMode) {
		this.hierarchicalMode = hierarchicalMode;
	}

	@Override
	public boolean includeContexts() {
		return includeContexts;
	}
	
	@Override
	public void setIncludeContexts(boolean includeContexts) {
		this.includeContexts = includeContexts;
	}
	
}
