package org.gcube.informationsystem.resourceregistry.rest;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import org.eclipse.microprofile.openapi.annotations.Operation;
import org.eclipse.microprofile.openapi.annotations.enums.ParameterIn;
import org.eclipse.microprofile.openapi.annotations.enums.SchemaType;
import org.eclipse.microprofile.openapi.annotations.media.Content;
import org.eclipse.microprofile.openapi.annotations.media.ExampleObject;
import org.eclipse.microprofile.openapi.annotations.media.Schema;
import org.eclipse.microprofile.openapi.annotations.parameters.Parameter;
import org.eclipse.microprofile.openapi.annotations.parameters.RequestBody;
import org.eclipse.microprofile.openapi.annotations.responses.APIResponse;
import org.eclipse.microprofile.openapi.annotations.tags.Tag;
import org.gcube.informationsystem.model.reference.entities.Resource;
import org.gcube.informationsystem.queries.templates.reference.entities.QueryTemplate;
import org.gcube.informationsystem.resourceregistry.ResourceInitializer;
import org.gcube.informationsystem.resourceregistry.api.exceptions.AvailableInAnotherContextException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.NotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.contexts.ContextNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.queries.InvalidQueryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.types.SchemaNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.request.BaseRequestInfo;
import org.gcube.informationsystem.resourceregistry.api.rest.AccessPath;
import org.gcube.informationsystem.resourceregistry.api.rest.ContextPath;
import org.gcube.informationsystem.resourceregistry.api.rest.InstancePath;
import org.gcube.informationsystem.resourceregistry.api.rest.QueryTemplatePath;
import org.gcube.informationsystem.resourceregistry.api.rest.TypePath;
import org.gcube.informationsystem.resourceregistry.base.ElementManagement;
import org.gcube.informationsystem.resourceregistry.contexts.ContextUtility;
import org.gcube.informationsystem.resourceregistry.contexts.entities.ContextManagement;
import org.gcube.informationsystem.resourceregistry.instances.model.ERManagementUtility;
import org.gcube.informationsystem.resourceregistry.instances.model.entities.ResourceManagement;
import org.gcube.informationsystem.resourceregistry.queries.Query;
import org.gcube.informationsystem.resourceregistry.queries.QueryImpl;
import org.gcube.informationsystem.resourceregistry.queries.json.JsonQuery;
import org.gcube.informationsystem.resourceregistry.queries.templates.QueryTemplateManagement;
import org.gcube.informationsystem.resourceregistry.rest.requests.ServerRequestInfo;
import org.gcube.informationsystem.resourceregistry.types.TypeManagement;
import org.gcube.informationsystem.types.TypeMapper;
import org.gcube.informationsystem.types.reference.Type;

import com.orientechnologies.orient.core.record.ODirection;

import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.DefaultValue;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.HEAD;
import jakarta.ws.rs.POST;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.PathParam;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.QueryParam;
import jakarta.ws.rs.core.Context;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.MultivaluedMap;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.core.Response.Status;
import jakarta.ws.rs.core.UriInfo;

/**
 * @author Luca Frosini (ISTI - CNR)
 */
@Path(AccessPath.ACCESS_PATH_PART)
@Tag(name = "Access", description = "Resource Registry Access API - Provides read-only access to contexts, types, instances, and query execution")
public class Access extends BaseRest {
	
	/**
	 * Constant representing the resource type name path parameter.
	 */
	public static final String RESOURCE_TYPE_PATH_PARAMETER = "resource-type-name";
	/**
	 * Constant representing the relation type name path parameter.
	 */
	public static final String RELATION_TYPE_PATH_PARAMETER = "relation-type-name";
	/**
	 * Constant representing the reference type name path parameter.
	 */
	public static final String REFERENCE_TYPE_PATH_PARAMETER = "reference-type-name";
	
	/**
	 * Constant representing the method name for executing raw queries.
	 */
	public static final String RAW_QUERY_METHOD = "raw";
	/**
	 * Constant representing the method name for executing graph queries.
	 */
	public static final String GRAPH_QUERY_METHOD = "graph";
	/**
	 * Constant representing the method name for executing json queries.
	 */
	public static final String JSON_QUERY_METHOD = "json";
	/**
	 * Constant representing the method name for executing prepared queries.
	 */
	public static final String PREPARED_QUERY_METHOD = "prepared";
	
	/**
	 * Default constructor for the Access class.
	 * Initializes a new instance of the Access class.
	 */
	public Access() {
		super();
	}
	
	/**
	 * Retrieves the list of contexts in the system.
	 * The response can include metadata and provides pagination based on query parameters and user authorization.
	 * 
	 * <strong>REST Endpoint:</strong> {@code GET /access/contexts[?limit={limit}&offset={offset}&includeMeta={true|false}]}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>{@code GET /access/contexts} (returns first 10 contexts - default pagination)</li>
	 * <li>{@code GET /access/contexts?limit=20} (returns first 20 contexts)</li>
	 * <li>{@code GET /access/contexts?offset=10} (returns 10 contexts starting from the 11th - default limit with offset)</li>
	 * <li>{@code GET /access/contexts?limit=5&offset=15} (returns 5 contexts starting from the 16th)</li>
	 * <li>{@code GET /access/contexts?includeMeta=true} (returns first 10 contexts with metadata - for non-admin users)</li>
	 * <li>{@code GET /access/contexts?limit=20&includeMeta=true} (returns first 20 contexts with metadata).</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>limit</strong> (optional):
	 * <ul>
	 * <li>Maximum number of contexts to return in a single response</li>
	 * <li>Default value: 10</li>
	 * <li>Example: ?limit=50 (returns at most 50 contexts).</li>
	 * </ul>
	 * 
	 * <strong>offset</strong> (optional):
	 * <ul>
	 * <li>Number of contexts to skip from the beginning of the result set</li>
	 * <li>Default value: 0 (starts from the first context)</li>
	 * <li>Example: ?offset=100 (skips the first 100 contexts)</li>
	 * <li>Commonly used with limit for pagination: ?limit=50&amp;offset=100.</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (optional):
	 * Whether to include metadata in the response (for non-admin users).
	 * <ul>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeMeta=true (includes metadata with obfuscated sensitive fields)</li>
	 * <li>Note: IS-Manager and Infrastructure-Manager always receive not filtered metadata regardless of this parameter.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Automatically receive full context information including:</li>
	 *   <li>Complete metadata (createdBy, lastUpdateBy, creation/modification dates)</li>
	 *   <li>Context state information</li>
	 *   <li>All context properties without obfuscation.</li>
	 * <li>Metadata is included by default regardless of request parameters.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Automatically receive full context information including:</li>
	 *   <li>Complete metadata (createdBy, lastUpdateBy, creation/modification dates)</li>
	 *   <li>Context state information</li>
	 *   <li>All context properties without obfuscation.</li>
	 * <li>Metadata is included by default regardless of request parameters.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Receive basic context information by default</li>
	 * <li>Can explicitly request metadata via includeMeta=true query parameter</li>
	 * <li>When metadata is requested, receive metadata with sensitive information filtered:</li>
	 *   <li>Date fields (creation, modification) are visible</li>
	 *   <li>User identifiers (createdBy, lastUpdateBy) are obfuscated or hidden</li>
	 *   <li>Other administrative details may be filtered.</li>
	 *   </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li>200 OK: Contexts successfully retrieved</li>
	 * <li>403 Forbidden: User lacks authorization to access context information</li>
	 * <li>500 Internal Server Error: Server error during context retrieval.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Content-Type: application/json</li>
	 * <li>Body: JSON array containing context objects with role-appropriate detail level</li>
	 * <li>Pagination information may be included in response headers.</li>
	 * </ul>
	 * 
	 * @return JSON array containing context objects with role-appropriate detail level
	 * @throws ResourceRegistryException if contexts cannot be retrieved
	 */
	@GET
	@Path(AccessPath.CONTEXTS_PATH_PART)
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	@Operation(
		summary = "List Contexts",
		description = """
		Retrieves the list of contexts in the system.

		The response can include metadata and provides pagination based on query parameters and user authorization.


		**Request Examples:**
		- GET /access/contexts (returns first 10 contexts - default pagination);
		- GET /access/contexts?limit=20 (returns first 20 contexts);
		- GET /access/contexts?offset=10 (returns 10 contexts starting from the 11th - default limit with offset);
		- GET /access/contexts?limit=5&offset=15 (returns 5 contexts starting from the 16th);
		- GET /access/contexts?includeMeta=true (returns first 10 contexts with metadata - for non-admin users);
		- GET /access/contexts?limit=20&includeMeta=true (returns first 20 contexts with metadata).


		**Authorization Requirements:**
		- **IS-Manager:**
			- Automatically receive full context information including:
				- Complete metadata (createdBy, lastUpdateBy, creation/modification dates);
				- Context state information;
				- All context properties without obfuscation.
			- Metadata is included by default regardless of request parameters.

		- **Infrastructure-Manager:**
			- Automatically receive full context information including:
				- Complete metadata (createdBy, lastUpdateBy, creation/modification dates);
				- Context state information;
				- All context properties without obfuscation.
			- Metadata is included by default regardless of request parameters.

		- **Other Users:**
			- Receive basic context information by default;
			- Can explicitly request metadata via includeMeta=true query parameter;
			- When metadata is requested, receive metadata with sensitive information filtered:
				- Date fields (creation, modification) are visible;
				- User identifiers (createdBy, lastUpdateBy) are obfuscated or hidden;
				- Other administrative details may be filtered.
		"""
	)
	@Parameter(
		name = ContextPath.LIMIT_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Maximum number of contexts to return in a single response.
		- Default: 10 (returns at most 10 contexts);
		- Example: ?limit=50 (returns at most 50 contexts).
		""",
		required = false,
		schema = @Schema(type = SchemaType.INTEGER, defaultValue = "10", example = "10")
	)
	@Parameter(
		name = ContextPath.OFFSET_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Number of contexts to skip from the beginning of the result set.
		- Default: 0 (starts from the first context);
		- Example: ?offset=100 (skips the first 100 contexts);
		- Commonly used with limit for pagination: ?limit=50&offset=100.
		""",
		required = false,
		schema = @Schema(type = SchemaType.INTEGER, defaultValue = "0", example = "0")
	)
	@Parameter(
		name = ContextPath.INCLUDE_META_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include metadata in the response (for non-admin users). 
		- Default: false (basic information only);
		- Example: ?includeMeta=true (includes metadata with obfuscated sensitive fields);
		- Note: IS-Manager and Infrastructure-Manager always receive not filtered metadata regardless of this parameter.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@APIResponse(
		responseCode = "200",
		description = "Contexts successfully retrieved",
		content = @Content(mediaType = "application/json")
	)
	@APIResponse(
		responseCode = "403",
		description = "User lacks authorization to access context information"
	)
	@APIResponse(
		responseCode = "500",
		description = "Server error during context retrieval"
	)
	public String listContexts() throws ResourceRegistryException {
		logger.info("Requested to read all {}s", org.gcube.informationsystem.contexts.reference.entities.Context.NAME);
		setAccountingMethod(Method.LIST, org.gcube.informationsystem.contexts.reference.entities.Context.NAME);
		
		ServerRequestInfo serverRequestInfo = initRequestInfo(BaseRequestInfo.DEFAULT_OFFSET, BaseRequestInfo.UNBOUNDED_LIMIT);
		/*
		 * We never provide meta information 
		 * for context via this REST API.
		 * Contexts collection should be used instead.
		 */
		serverRequestInfo.setIncludeMeta(false);
		serverRequestInfo.setAllMeta(false);
		/*
		 * serverRequestInfo.checkBooleanQueryParameter(ContextPath.INCLUDE_META_QUERY_PARAMETER);
		 * serverRequestInfo.setAllMeta(true);
		 */
		
		serverRequestInfo.checkLimitOffset();
		
		ContextManagement contextManagement = new ContextManagement();
		return contextManagement.all(false);
	}

	/**
	 * Retrieves a specific context by its UUID.
	 * The response includes metadata and detail level based on query parameters and user authorization.
	 * 
	 * <strong>REST Endpoint:</strong> {@code GET /access/contexts/{context-uuid}[?includeMeta={true|false}]}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>{@code GET /access/contexts/c0f314e7-2807-4241-a792-2a6c79ed4fd0} (returns basic context information)</li>
	 * <li>{@code GET /access/contexts/c0f314e7-2807-4241-a792-2a6c79ed4fd0?includeMeta=true} (returns context with metadata - for non-admin users)</li>
	 * <li>{@code GET /access/contexts/CURRENT_CONTEXT} (returns the current context - retrieved from client's authorization token).</li>
	 * </ul>
	 * 
	 * <strong>Path Parameters:</strong>
	 * 
	 * <strong>context-uuid</strong> (required):
	 * <ul>
	 * <li>The unique identifier of the context to retrieve</li>
	 * <li>Must be a valid UUID string format (e.g., "c0f314e7-2807-4241-a792-2a6c79ed4fd0")</li>
	 * <li><strong>Special keyword</strong>: "CURRENT_CONTEXT" to retrieve the context from client's authorization token</li>
	 * <li>When using "CURRENT_CONTEXT", the actual context is extracted from the JWT token sent by the client</li>
	 * <li>Example: GET /access/contexts/CURRENT_CONTEXT returns the context the client is authorized for.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (optional):
	 * Whether to include metadata in the response (for non-admin users).
	 * <ul>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeMeta=true (includes metadata with obfuscated sensitive fields)</li>
	 * <li>Note: IS-Manager and Infrastructure-Manager always receive metadata regardless of this parameter.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can retrieve context information for any context without restrictions</li>
	 * <li>Automatically receive full context information including complete metadata</li>
	 * <li>All context properties and state information without obfuscation</li>
	 * <li>Metadata is included by default regardless of request parameters.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can retrieve context information for any context without restrictions</li>
	 * <li>Automatically receive full context information including complete metadata</li>
	 * <li>All context properties and state information without obfuscation</li>
	 * <li>Metadata is included by default regardless of request parameters.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>When operating in their own context (using a token from that context):</li>
	 *   <li>Receive full context information like IS-Manager and Infrastructure-Manager</li>
	 *   <li>Complete metadata without obfuscation</li>
	 *   <li>All context properties and state information.</li>
	 * <li>When operating outside their context or requesting other contexts:</li>
	 *   <li>Receive basic context information by default</li>
	 *   <li>Can request metadata via includeMeta=true with obfuscated sensitive fields.</li>
	 *   </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Receive basic context information by default</li>
	 * <li>Can explicitly request metadata via includeMeta=true query parameter</li>
	 * <li>When metadata is requested, receive metadata with sensitive information filtered:</li>
	 *   <li>Date fields (creation, modification) are visible</li>
	 *   <li>User identifiers (createdBy, lastUpdateBy) are obfuscated or hidden</li>
	 *   <li>Other administrative details may be filtered.</li>
	 *   </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li>200 OK: Context successfully retrieved</li>
	 * <li>404 Not Found: Context with the specified UUID does not exist</li>
	 * <li>403 Forbidden: User lacks authorization to access the specific context</li>
	 * <li>400 Bad Request: Invalid UUID format.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Content-Type: application/json</li>
	 * <li>Body: JSON object containing the context with role-appropriate detail level</li>
	 * <li>Includes context properties, metadata (if authorized), and state information.</li>
	 * </ul>
	 *
	 * @param uuid the UUID of the context to retrieve, or "CURRENT_CONTEXT" to get the context from client's authorization token
	 * @return JSON object containing the context with role-appropriate detail level
	 * @throws ContextNotFoundException if the context with the specified UUID is not found
	 * @throws ResourceRegistryException if there is an error accessing the resource registry
	 */
	@GET
	@Path(AccessPath.CONTEXTS_PATH_PART + "/{" + ContextManager.CONTEXT_UUID_PATH_PARAMETER + "}")
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	@Operation(
		summary = "Get Context by UUID",
		description = """
		Retrieves a specific context by its UUID.

		The response includes metadata and detail level based on query parameters and user authorization.


		**Request Examples:**
		- GET /access/contexts/c0f314e7-2807-4241-a792-2a6c79ed4fd0 (returns basic context information);
		- GET /access/contexts/c0f314e7-2807-4241-a792-2a6c79ed4fd0?includeMeta=true (returns context with metadata - for non-admin users);
		- GET /access/contexts/CURRENT_CONTEXT (returns the current context - retrieved from client's authorization token).


		**Authorization Requirements:**
		- **IS-Manager:**
			- Can retrieve context information for any context without restrictions;
			- Automatically receive full context information including complete metadata;
			- All context properties and state information without obfuscation;
			- Metadata is included by default regardless of request parameters.

		- **Infrastructure-Manager:**
			- Can retrieve context information for any context without restrictions;
			- Automatically receive full context information including complete metadata;
			- All context properties and state information without obfuscation;
			- Metadata is included by default regardless of request parameters.

		- **Context-Manager:**
			- When operating in their own context (using a token from that context):
				- Receive full context information like IS-Manager and Infrastructure-Manager;
				- Complete metadata without obfuscation;
				- All context properties and state information.
			- When operating outside their context or requesting other contexts:
				- Receive basic context information by default;
				- Can request metadata via includeMeta=true with obfuscated sensitive fields.

		- **Other Users:**
			- Receive basic context information by default;
			- Can explicitly request metadata via includeMeta=true query parameter;
			- When metadata is requested, receive metadata with sensitive information filtered:
				- Date fields (creation, modification) are visible;
				- User identifiers (createdBy, lastUpdateBy) are obfuscated or hidden;
				- Other administrative details may be filtered.
		"""
	)
	@Parameter(
		name = ContextPath.INCLUDE_META_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include metadata in the response (for non-admin users)
		- Default value: false (basic information only);
		- Values: true|false;
		- Example: ?includeMeta=true (includes metadata with obfuscated sensitive fields);
		- Note: IS-Manager and Infrastructure-Manager always receive metadata regardless of this parameter.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@APIResponse(
		responseCode = "200",
		description = "Context successfully retrieved",
		content = @Content(mediaType = "application/json")
	)
	@APIResponse(
		responseCode = "404",
		description = "Context with the specified UUID does not exist"
	)
	@APIResponse(
		responseCode = "403",
		description = "User lacks authorization to access the specific context"
	)
	@APIResponse(
		responseCode = "400",
		description = "Invalid UUID format"
	)
	public String readContext(
			@PathParam(ContextManager.CONTEXT_UUID_PATH_PARAMETER)
			@Parameter(
				name = ContextManager.CONTEXT_UUID_PATH_PARAMETER,
				in = ParameterIn.PATH,
				description = """
				The unique identifier of the context to retrieve.
				- Must be a valid UUID string format (e.g., "c0f314e7-2807-4241-a792-2a6c79ed4fd0");
				- Special keyword: "CURRENT_CONTEXT" to retrieve the context from client's authorization token;
				- When using "CURRENT_CONTEXT", the actual context is extracted from the JWT token sent by the client;
				- Example: GET /access/contexts/CURRENT_CONTEXT returns the context the client is authorized for.
				""",
				required = true,
				schema = @Schema(type = SchemaType.STRING, example = "c0f314e7-2807-4241-a792-2a6c79ed4fd0")
			)
			String uuid)
			throws ContextNotFoundException, ResourceRegistryException {
		if(uuid.compareTo(ContextPath.CURRENT_CONTEXT_PATH_PART)==0){
			uuid = ContextUtility.getCurrentRequestEnvironment().getUUID().toString();
		}
		logger.info("Requested to read {} with id {} ", org.gcube.informationsystem.contexts.reference.entities.Context.NAME, uuid);
		setAccountingMethod(Method.READ, org.gcube.informationsystem.contexts.reference.entities.Context.NAME);
		
		ServerRequestInfo serverRequestInfo = initRequestInfo();
		/*
		 * We have to think if never provide meta information 
		 * for context via this REST API.
		 * Contexts collection should be used instead.
		 */
		serverRequestInfo.setAllMeta(true);
		serverRequestInfo.checkBooleanQueryParameter(ContextPath.INCLUDE_META_QUERY_PARAMETER);
		
		ContextManagement contextManagement = new ContextManagement();
		contextManagement.setUUID(UUID.fromString(uuid));
		return contextManagement.readAsString();
	}

	/**
	 * Retrieves the schema definition(s) for the specified type.
	 * The response can include subtypes and metadata based on query parameters and user authorization.
	 * 
	 * <strong>REST Endpoint:</strong> {@code GET /access/types/{type-name}[?polymorphic={true|false}&includeMeta={true|false}]}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>{@code GET /access/types/ContactFacet} (returns basic ContactFacet type definition only)</li>
	 * <li>{@code GET /access/types/ContactFacet?polymorphic=true} (returns ContactFacet and all its subtypes)</li>
	 * <li>{@code GET /access/types/ContactFacet?includeMeta=true} (returns ContactFacet with metadata for authorized users)</li>
	 * <li>{@code GET /access/types/ContactFacet?polymorphic=true&includeMeta=true} (returns ContactFacet, subtypes, and metadata).</li>
	 * </ul>
	 * 
	 * <strong>Path Parameters:</strong>
	 * <ul>
	 * <li><strong>type-name</strong>: The name of the type to retrieve (case-sensitive).</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>polymorphic</strong> (optional):
	 * Whether to include subtypes in the response.
	 * <ul>
	 * <li>Default value: false (returns only the specified type)</li>
	 * <li>When true: returns the specified type AND all its existing subtypes</li>
	 * <li>When false: returns only the specified type definition</li>
	 * <li>Note: No pagination is applied - ALL subtypes are returned when polymorphic=true</li>
	 * <li>Example: ?polymorphic=true (includes all subtypes of the requested type).</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (optional):
	 * Whether to include metadata in the response.
	 * <ul>
	 * <li>Default value: false (basic type information only)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeMeta=true (includes metadata based on user authorization).</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Receive basic type information by default</li>
	 * <li>Can explicitly request metadata via includeMeta=true query parameter</li>
	 * <li>When metadata is requested, receive complete type definitions with full metadata</li>
	 * <li>All metadata fields are included when includeMeta=true</li>
	 * <li>No obfuscation or filtering of sensitive information.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Receive basic type information by default</li>
	 * <li>Can explicitly request metadata via includeMeta=true query parameter</li>
	 * <li>When metadata is requested, receive complete type definitions with full metadata</li>
	 * <li>All metadata fields are included when includeMeta=true</li>
	 * <li>No obfuscation or filtering of sensitive information.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Receive basic type information by default</li>
	 * <li>Can explicitly request metadata via includeMeta=true query parameter</li>
	 * <li>When metadata is requested, receive metadata with sensitive information filtered:</li>
	 *   <li>Date fields (creation, modification) are visible</li>
	 *   <li>User identifiers (createdBy, lastUpdateBy) are obfuscated or hidden</li>
	 *   <li>Other administrative details may be filtered.</li>
	 *   </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li>200 OK: Type successfully retrieved</li>
	 * <li>404 Not Found: Type with the specified name does not exist</li>
	 * <li>403 Forbidden: User lacks authorization to access type information.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Content-Type: application/json</li>
	 * <li>Body: Type schema definition(s) with authorization-appropriate detail level</li>
	 * <li>When polymorphic=false: single type definition object</li>
	 * <li>When polymorphic=true: array of type definitions (parent + all subtypes).</li>
	 * </ul>
	 * 
	 * @param type the name of the type to retrieve (case-sensitive)
	 * @param polymorphic whether to include subtypes in the response (default: false)
	 * @return JSON containing type schema definition(s) with authorization-appropriate detail level
	 * @throws SchemaNotFoundException if the specified type is not found in the system
	 * @throws ResourceRegistryException if there is an error retrieving the type definitions
	 */
	@GET
	@Path(AccessPath.TYPES_PATH_PART + "/{" + TypeManager.TYPE_PATH_PARAMETER + "}")
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	@Operation(
		summary = "Read Type Schema(s)",
		description = """
		Get Type Schema Definition(s)
		
		Retrieves the schema definition(s) for the specified type. The response can include subtypes and metadata based on query parameters and user authorization.

		**Request Examples:**
		- GET /access/types/ContactFacet (returns basic ContactFacet type definition only);
		- GET /access/types/ContactFacet?polymorphic=true (returns ContactFacet and all its subtypes);
		- GET /access/types/ContactFacet?includeMeta=true (returns ContactFacet with metadata for authorized users);
		- GET /access/types/ContactFacet?polymorphic=true&includeMeta=true (returns ContactFacet, subtypes, and metadata).


		**Authorization Requirements:**
		- **IS-Manager:**
			- Receive basic type information by default;
			- Can explicitly request metadata via includeMeta=true query parameter;
			- When metadata is requested, receive complete type definitions with full metadata;
			- All metadata fields are included when includeMeta=true;
			- No obfuscation or filtering of sensitive information.

		- **Infrastructure-Manager:**
			- Receive basic type information by default;
			- Can explicitly request metadata via includeMeta=true query parameter;
			- When metadata is requested, receive complete type definitions with full metadata;
			- All metadata fields are included when includeMeta=true;
			- No obfuscation or filtering of sensitive information.

		- **Other Users:**
			- Receive basic type information by default;
			- Can explicitly request metadata via includeMeta=true query parameter;
			- When metadata is requested, receive metadata with sensitive information filtered:
				- Date fields (creation, modification) are visible;
				- User identifiers (createdBy, lastUpdateBy) are obfuscated or hidden;
				- Other administrative details may be filtered.
		"""
	)		
	@Parameter(
		name = TypePath.INCLUDE_META_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include metadata in the response.
		- Default value: false (basic type information only);
		- Example: ?includeMeta=true (includes metadata based on user authorization).
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@APIResponse(
		responseCode = "200",
		description = "Type successfully retrieved",
		content = @Content(mediaType = "application/json")
	)
	@APIResponse(
		responseCode = "404",
		description = "Type with the specified name does not exist"
	)
	@APIResponse(
		responseCode = "403",
		description = "User lacks authorization to access type information"
	)
	public String readType(
			@PathParam(TypeManager.TYPE_PATH_PARAMETER)
			@Parameter(
				name = TypeManager.TYPE_PATH_PARAMETER,
				in = ParameterIn.PATH,
				description = "The name of the type to retrieve (case-sensitive)",
				required = true,
				schema = @Schema(type = SchemaType.STRING, example = "ContactFacet")
			)
			String type,
			@QueryParam(TypePath.POLYMORPHIC_QUERY_PARAMETER) @DefaultValue("false") 
			@Parameter(
				name = TypePath.POLYMORPHIC_QUERY_PARAMETER,
				in = ParameterIn.QUERY,
				description = """
				Whether to include subtypes in the response.
				- Default value: false (returns only the specified type);
				- When true: returns the specified type AND all its existing subtypes;
				- When false: returns only the specified type definition;
				- Note: No pagination is applied - ALL subtypes are returned when polymorphic=true;
				- Example: ?polymorphic=true (includes all subtypes of the requested type).
				""",
				required = false,
				schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
			)
			Boolean polymorphic)
			throws SchemaNotFoundException, ResourceRegistryException {
		logger.info("Requested Schema for type {}", type);
		setAccountingMethod(Method.READ, Type.NAME);
		
		ServerRequestInfo serverRequestInfo = initRequestInfo();
		serverRequestInfo.setAllMeta(true);
		serverRequestInfo.checkBooleanQueryParameter(TypePath.INCLUDE_META_QUERY_PARAMETER);
		
		TypeManagement typeManagement = new TypeManagement();
		typeManagement.setTypeName(type);
		List<Type> types = typeManagement.read(polymorphic);
		try {
			return TypeMapper.serializeTypeDefinitions(types);
		}catch (Exception e) {
			throw new ResourceRegistryException(e);
		}
	}

	/**
	 * Retrieves all query templates from the Resource Registry.
	 * 
	 * <strong>REST Endpoint:</strong> {@code GET /access/query-templates}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>Basic query: {@code GET /access/query-templates}</li>
	 * <li>With pagination: {@code GET /access/query-templates?limit=20&offset=40}</li>
	 * <li>With metadata: {@code GET /access/query-templates?includeMeta=true}</li>
	 * <li>Combined parameters: {@code GET /access/query-templates?limit=15&offset=30&includeMeta=true}.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>limit</strong> (optional):
	 * <ul>
	 * <li>Maximum number of query templates to return in a single response</li>
	 * <li>Default value: 10</li>
	 * <li>Example: ?limit=50 (returns at most 50 query templates).</li>
	 * </ul>
	 * 
	 * <strong>offset</strong> (optional):
	 * <ul>
	 * <li>Number of query templates to skip from the beginning of the result set</li>
	 * <li>Default value: 0 (starts from the first query template)</li>
	 * <li>Example: ?offset=100 (skips the first 100 query templates).</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (optional):
	 * Whether to include metadata in the response query templates.
	 * <ul>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeMeta=true (includes metadata with appropriate filtering based on user role)</li>
	 * <li><strong>Restriction:</strong> IS-Manager and Infrastructure-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Full access to query templates</li>
	 * <li>Receives complete, unfiltered metadata including all administrative fields.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Full access to query templates</li>
	 * <li>Receives complete, unfiltered metadata including all administrative fields.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Basic access with metadata filtering</li>
	 * <li>Receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Query templates successfully retrieved</li>
	 * <li><strong>400 Bad Request</strong>: Invalid query parameters (e.g., invalid limit/offset values)</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to access query templates.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON array containing the requested query templates</li>
	 * <li>Each query template includes its name, description, template definition, template variables, and metadata (if requested)</li>
	 * <li>Empty array is returned if no query templates exist in the accessible contexts.</li>
	 * </ul>
	 * 
	 * <strong>Example Metadata (filtered for non-admin users):</strong>
	 * ``{@code json
	 * "metadata": {
	 *     "type": "Metadata",
	 *     "creationTime": "2025-03-18 17:13:40.952 +0100",
	 *     "createdBy": "HIDDEN_FOR_PRIVACY",
	 *     "lastUpdateBy": "HIDDEN_FOR_PRIVACY",
	 *     "lastUpdateTime": "2025-03-19 16:21:16.805 +0100"
	 * }
	 * }``
	 * 
	 * @return JSON array containing the requested query templates with their properties and metadata (if requested)
	 * @throws NotFoundException If query templates cannot be retrieved
	 * @throws ResourceRegistryException If an error occurs during query template retrieval
	 */
	@GET
	@Path(AccessPath.QUERY_TEMPLATES_PATH_PART)
	@Consumes({MediaType.TEXT_PLAIN, ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8})
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	@Operation(
		summary = "List Query Templates",
		description = """
		Retrieves all query templates from the Resource Registry.
		
		The response can include metadata and provides pagination based on query parameters and user authorization.


		**Request Examples:**
		- GET /access/query-templates;
		- GET /access/query-templates?limit=20&offset=40;
		- GET /access/query-templates?includeMeta=true;
		- GET /access/query-templates?limit=15&offset=30&includeMeta=true.


		**Authorization Requirements:**
		- **IS-Manager:**
			- Full access to query templates;
			- Receives complete, unfiltered metadata including all administrative fields.

		- **Infrastructure-Manager:**
			- Full access to query templates;
			- Receives complete, unfiltered metadata including all administrative fields.

		- **Other Users:**
			- Basic access with metadata filtering;
			- Receive metadata with sensitive information filtered when includeMeta=true.
		"""
	)
	@Parameter(
		name = QueryTemplatePath.LIMIT_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Maximum number of query templates to return in a single response.
		- Default value: 10;
		- Example: ?limit=50 (returns at most 50 query templates).
		""",
		required = false,
		schema = @Schema(type = SchemaType.INTEGER, defaultValue = "10", example = "10")
	)
	@Parameter(
		name = QueryTemplatePath.OFFSET_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Number of query templates to skip from the beginning of the result set.
		- Default value: 0 (starts from the first query template);
		- Example: ?offset=100 (skips the first 100 query templates).
		""",
		required = false,
		schema = @Schema(type = SchemaType.INTEGER, defaultValue = "0", example = "0")
	)
	@Parameter(
		name = QueryTemplatePath.INCLUDE_META_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include metadata in the response query templates.
		- Default value: false (basic information only);
		- Example: ?includeMeta=true (includes metadata with appropriate filtering based on user role);
		- Restriction: IS-Manager and Infrastructure-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)		
	@APIResponse(
		responseCode = "200",
		description = "Query templates successfully retrieved",
		content = @Content(mediaType = "application/json")
	)
	@APIResponse(
		responseCode = "400",
		description = "Invalid query parameters (e.g., invalid limit/offset values)"
	)
	@APIResponse(
		responseCode = "403",
		description = "Insufficient permissions to access query templates"
	)	
	public String listQueryTemplates() throws NotFoundException, ResourceRegistryException {
		logger.info("Requested to read all {}s", QueryTemplate.NAME);
		setAccountingMethod(Method.LIST, QueryTemplate.NAME);
		
		ServerRequestInfo serverRequestInfo = initRequestInfo();
		serverRequestInfo.setAllMeta(true);
		serverRequestInfo.checkBooleanQueryParameter(ContextPath.INCLUDE_META_QUERY_PARAMETER);
		serverRequestInfo.checkLimitOffset();
		
		QueryTemplateManagement queryTemplateManagement = new QueryTemplateManagement();
		return queryTemplateManagement.all(false);
	}

	/**
	 * Retrieves a specific query template by its name from the Resource Registry.
	 * 
	 * <strong>REST Endpoint:</strong> {@code GET /access/query-templates/{query-template-name}}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>Basic retrieval: {@code GET /access/query-templates/GetAllEServiceWithState}</li>
	 * <li>With metadata: {@code GET /access/query-templates/GetAllEServiceWithState?includeMeta=true}.</li>
	 * </ul>
	 * 
	 * <strong>Path Parameters:</strong>
	 * <ul>
	 * <li><strong>query-template-name</strong>: The name of the query template to retrieve (e.g., "GetAllEServiceWithState").</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (optional):
	 * Whether to include metadata in the response query template.
	 * <ul>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeMeta=true (includes metadata with appropriate filtering based on user role)</li>
	 * <li><strong>Restriction:</strong> IS-Manager and Infrastructure-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Full access to query template</li>
	 * <li>Receives complete, unfiltered metadata including all administrative fields.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Full access to query template</li>
	 * <li>Receives complete, unfiltered metadata including all administrative fields.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Basic access with metadata filtering</li>
	 * <li>Receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Query template successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: Query template with the specified name does not exist</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to access the query template.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON object containing the requested query template</li>
	 * <li>Includes name, description, template definition, template variables, and metadata (if requested).</li>
	 * </ul>
	 * 
	 * <strong>Example Metadata (filtered for non-admin users):</strong>
	 * ``{@code json
	 * "metadata": {
	 *     "type": "Metadata",
	 *     "creationTime": "2025-03-18 17:13:40.952 +0100",
	 *     "createdBy": "HIDDEN_FOR_PRIVACY",
	 *     "lastUpdateBy": "HIDDEN_FOR_PRIVACY",
	 *     "lastUpdateTime": "2025-03-19 16:21:16.805 +0100"
	 * }
	 * }``
	 * 
	 * @param queryTemplateName The name of the query template to retrieve
	 * @return JSON object containing the requested query template with its properties and metadata (if requested)
	 * @throws NotFoundException If the query template with the specified name does not exist
	 * @throws ResourceRegistryException If an error occurs during query template retrieval
	 */
	@GET
	@Path(AccessPath.QUERY_TEMPLATES_PATH_PART + "/{" + QueryTemplateManager.QUERY_TEMPLATE_NAME_PATH_PARAMETER + "}")
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	@Operation(
		summary = "Read Query Template",
		description = """
		Retrieves a specific query template by its name from the Resource Registry.
		
		The response includes metadata based on query parameters and user authorization.


		**Request Examples:**
		- GET /access/query-templates/GetAllEServiceWithState;
		- GET /access/query-templates/GetAllEServiceWithState?includeMeta=true.


		**Authorization Requirements:**
		- **IS-Manager:**
			- Full access to query template;
			- Receives complete, unfiltered metadata including all administrative fields.

		- **Infrastructure-Manager:**
			- Full access to query template;
			- Receives complete, unfiltered metadata including all administrative fields.

		- **Other Users:**
			- Basic access with metadata filtering;
			- Receive metadata with sensitive information filtered when includeMeta=true.
		"""
	)		
	@Parameter(
		name = QueryTemplatePath.INCLUDE_META_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include metadata in the response query template
		- Default value: false (basic information only);
		- Example: ?includeMeta=true (includes metadata with appropriate filtering based on user role);
		- Restriction: IS-Manager and Infrastructure-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@APIResponse(
		responseCode = "200",
		description = "Query template successfully retrieved",
		content = @Content(mediaType = "application/json")
	)
	@APIResponse(
		responseCode = "404",
		description = "Query template with the specified name does not exist"
	)
	@APIResponse(
		responseCode = "403",
		description = "Insufficient permissions to access the query template"
	)
	public String readQueryTemplate(
			@PathParam(QueryTemplateManager.QUERY_TEMPLATE_NAME_PATH_PARAMETER) 
			@Parameter(
				name = QueryTemplateManager.QUERY_TEMPLATE_NAME_PATH_PARAMETER,
				in = ParameterIn.PATH,
				description = "The name of the query template to retrieve (e.g., \"GetAllEServiceWithState\")",
				required = true,
				schema = @Schema(type = SchemaType.STRING, example = "GetAllEServiceWithState")
			)
			String queryTemplateName)
			throws NotFoundException, ResourceRegistryException {
		logger.info("Requested {} with name", QueryTemplate.NAME, queryTemplateName);
		setAccountingMethod(Method.READ, QueryTemplate.NAME);
		
		ServerRequestInfo serverRequestInfo = initRequestInfo();
		serverRequestInfo.setAllMeta(true);
		serverRequestInfo.checkBooleanQueryParameter(ContextPath.INCLUDE_META_QUERY_PARAMETER);
		
		QueryTemplateManagement queryTemplateManagement = new QueryTemplateManagement();
		queryTemplateManagement.setName(queryTemplateName);
		return queryTemplateManagement.read();
	}

	/**
	 * Executes a query template with the specified name and parameters in the Resource Registry.
	 * 
	 * <strong>REST Endpoint:</strong> {@code POST /access/query-templates/{query-template-name}}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>Basic execution: {@code POST /access/query-templates/GetAllEServiceWithState}</li>
	 * <li>With polymorphic disabled: {@code POST /access/query-templates/GetAllEServiceWithState?polymorphic=false}</li>
	 * <li>With metadata: {@code POST /access/query-templates/GetAllEServiceWithState?includeMeta=true}</li>
	 * <li>Hierarchical (admin only): {@code POST /access/query-templates/GetAllEServiceWithState?hierarchical=true}</li>
	 * <li>Full metadata context: {@code POST /access/query-templates/GetAllEServiceWithState?includeMeta=true&amp;allMeta=true}</li>
	 * <li>With context information: {@code POST /access/query-templates/GetAllEServiceWithState?includeContexts=true}.</li>
	 * </ul>
	 * 
	 * <strong>Path Parameters:</strong>
	 * <ul>
	 * <li><strong>query-template-name</strong>: The name of the query template to execute (e.g., "GetAllEServiceWithState").</li>
	 * </ul>
	 * 
	 * <strong>Request Body:</strong>
	 * <ul>
	 * <li>JSON object containing parameter values to substitute in the query template</li>
	 * <li>Parameters not provided in the body will use their default values from the template definition</li>
	 * <li>Parameter names must match those defined in the template's templateVariables section.</li>
	 * </ul>
	 * 
	 * <strong>Example Request Body for GetAllEServiceWithState:</strong>
	 * ``{@code json
	 * {
	 *     "$state" : "down"
	 * }
	 * }`{@code 
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>polymorphic</strong> (optional):
	 * Whether the query should consider instances polymorphically (include subtypes).
	 * <ul>
	 * <li>Default value: true (includes all subtypes)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?polymorphic=false (exact type match only, may slow down the query)</li>
	 * <li>When false, only instances of the exact types defined in the query are returned</li>
	 * <li>When true, instances of derived types are also included.</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (optional):
	 * Whether to include metadata in the response instances.
	 * <ul>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeMeta=true (includes metadata with appropriate filtering based on user role)</li>
	 * <li><strong>Effect</strong>: Only applies when query returns complete instances of defined types</li>
	 * <li><strong>No Effect</strong>: When query uses _emit to return individual fields instead of complete instances</li>
	 * <li><strong>Restriction:</strong> IS-Manager and Infrastructure-Manager see complete metadata; other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (optional):
	 * Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.).
	 * <ul>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instances, more human-readable)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeMeta=true&amp;allMeta=true (includes metadata for all nested elements)</li>
	 * <li><strong>Effect</strong>: Only applies when query returns complete instances of defined types</li>
	 * <li><strong>No Effect</strong>: When query uses _emit to return individual fields instead of complete instances.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (optional, admin-only):
	 * Whether to execute the query across child contexts of the current context.
	 * <ul>
	 * <li>Default value: false (current context only)</li>
	 * <li>Values: true|false</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles (Context-Manager must be of the current context derived from the authorization token)</li>
	 * <li>Example: ?hierarchical=true (executes query in current and child contexts)</li>
	 * <li>Current context is determined from the authorization token.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (optional):
	 * Whether to include the list of contexts where each returned instance is available.
	 * <ul>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeContexts=true (shows context availability for each instance)</li>
	 * <li><strong>Effect</strong>: Only applies when query returns complete instances of defined types</li>
	 * <li><strong>No Effect</strong>: When query uses _emit to return individual fields instead of complete instances</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Full access to execute query template</li>
	 * <li>Can use hierarchical querying across all contexts</li>
	 * <li>Receives complete, unfiltered metadata including all administrative fields.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Full access to execute query template</li>
	 * <li>Can use hierarchical querying across all contexts</li>
	 * <li>Receives complete, unfiltered metadata including all administrative fields.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can execute query templates with hierarchical querying within their managed context hierarchy</li>
	 * <li>Must be Context-Manager of the current context (derived from the authorization token)</li>
	 * <li>Receives complete, unfiltered metadata including all administrative fields.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can execute query templates in the current context only</li>
	 * <li>Cannot use hierarchical querying (hierarchical parameter ignored)</li>
	 * <li>Receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li><strong>Parameter Substitution</strong>: Parameters in the template query (prefixed with $) are replaced with values from the request body</li>
	 * <li><strong>Default Values</strong>: Parameters not provided in the request body use their default values from the template definition</li>
	 * <li><strong>Query Execution</strong>: The resulting query is executed against the Resource Registry</li>
	 * <li><strong>Response Type</strong>: Can return complete instances or individual fields (when using _emit in the query)</li>
	 * <li><strong>Metadata Effect</strong>: includeMeta, allMeta, and includeContexts only affect queries returning complete instances, not _emit queries</li>
	 * <li><strong>Polymorphic Behavior</strong>: When disabled (polymorphic=false), may impact query performance but ensures exact type matching</li>
	 * <li><strong>No Pagination</strong>: Pagination is not currently supported for query template execution.</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Query template successfully executed</li>
	 * <li><strong>400 Bad Request</strong>: Invalid query parameters or malformed parameter values</li>
	 * <li><strong>404 Not Found</strong>: Query template with the specified name does not exist</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to execute the query template.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns query results based on the template definition</li>
	 * <li>For complete instances: JSON array containing instances with properties, relations, and metadata (if requested)</li>
	 * <li>For _emit queries: JSON array containing individual field values as specified in the query</li>
	 * <li>Response structure depends on the query template and parameter values provided.</li>
	 * </ul>
	 * 
	 * <strong>Example Metadata (filtered for non-admin users, only for complete instances):</strong>
	 * }`{@code json
	 * "metadata": {
	 *     "type": "Metadata",
	 *     "creationTime": "2025-03-18 17:13:40.952 +0100",
	 *     "createdBy": "HIDDEN_FOR_PRIVACY",
	 *     "lastUpdateBy": "HIDDEN_FOR_PRIVACY",
	 *     "lastUpdateTime": "2025-03-19 16:21:16.805 +0100"
	 * }
	 * }``
	 * 
	 * @param queryTemplateName The name of the query template to execute
	 * @param params JSON object containing parameter values for template substitution
	 * @return JSON array containing query results based on the template definition and parameters
	 * @throws NotFoundException If the query template with the specified name does not exist
	 * @throws InvalidQueryException If the query execution fails due to invalid parameters or query structure
	 * @throws ResourceRegistryException If an error occurs during query template execution
	 */
	@POST
	@Path(AccessPath.QUERY_TEMPLATES_PATH_PART + "/{" + QueryTemplateManager.QUERY_TEMPLATE_NAME_PATH_PARAMETER + "}")
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	@Operation(
		summary = "Run Query Template",
		description = """
		Executes a query template with the specified name and parameters in the Resource Registry.
		
		The response includes metadata and supports advanced query features based on parameters and user authorization.


		**Request Examples:**
		- POST /access/query-templates/GetAllEServiceWithState;
		- POST /access/query-templates/GetAllEServiceWithState?polymorphic=false;
		- POST /access/query-templates/GetAllEServiceWithState?includeMeta=true;
		- POST /access/query-templates/GetAllEServiceWithState?hierarchical=true;
		- POST /access/query-templates/GetAllEServiceWithState?includeMeta=true&allMeta=true;
		- POST /access/query-templates/GetAllEServiceWithState?includeContexts=true.


		**Operation Behavior:**
		- Parameter Substitution: Parameters in the template query (prefixed with $) are replaced with values from the request body;
		- Default Values: Parameters not provided in the request body use their default values from the template definition;
		- Query Execution: The resulting query is executed against the Resource Registry;
		- Response Type: Can return complete instances or individual fields (when using _emit in the query);
		- Metadata Effect: includeMeta, allMeta, and includeContexts only affect queries returning complete instances, not _emit queries;
		- Polymorphic Behavior: When disabled (polymorphic=false), may impact query performance but ensures exact type matching;
		- No Pagination: Pagination is not currently supported for query template execution.


		**Authorization Requirements:**
		- **IS-Manager:**
			- Full access to execute query template;
			- Can use hierarchical querying across all contexts;
			- Receives complete, unfiltered metadata including all administrative fields.

		- **Infrastructure-Manager:**
			- Full access to execute query template;
			- Can use hierarchical querying across all contexts;
			- Receives complete, unfiltered metadata including all administrative fields.

		- **Context-Manager:**
			- Can execute query templates with hierarchical querying within their managed context hierarchy;
			- Must be Context-Manager of the current context (derived from the authorization token);
			- Receives complete, unfiltered metadata including all administrative fields.

		- **Other Users:**
			- Can execute query templates in the current context only;
			- Cannot use hierarchical querying (hierarchical parameter ignored);
			- Receive metadata with sensitive information filtered when includeMeta=true.
		"""
	)		
	@Parameter(
		name = QueryTemplatePath.POLYMORPHIC_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether the query should consider instances polymorphically (include subtypes).
		- Default value: true (includes all subtypes);
		- Example: ?polymorphic=false (exact type match only, may slow down the query);
		- When false, only instances of the exact types defined in the query are returned;
		- When true, instances of derived types are also included.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "true", example = "true")
	)
	@Parameter(
		name = QueryTemplatePath.INCLUDE_META_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include metadata in the response instances.
		- Default value: false (basic information only);
		- Example: ?includeMeta=true (includes metadata with appropriate filtering based on user role);
		- Effect: Only applies when query returns complete instances of defined types;
		- No Effect: When query uses _emit to return individual fields instead of complete instances;
		- Restriction: IS-Manager and Infrastructure-Manager see complete metadata; other users see filtered metadata with sensitive fields obfuscated.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@Parameter(
		name = QueryTemplatePath.INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.).
		- Must be used in conjunction with includeMeta=true;
		- Default value: false (metadata only for main instances, more human-readable);
		- Example: ?includeMeta=true&allMeta=true (includes metadata for all nested elements);
		- Effect: Only applies when query returns complete instances of defined types;
		- No Effect: When query uses _emit to return individual fields instead of complete instances.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@Parameter(
		name = QueryTemplatePath.HIERARCHICAL_MODE_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to execute the query across child contexts of the current context.
		- Default value: false (current context only);
		- Restriction: Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles (Context-Manager must be of the current context derived from the authorization token);
		- Example: ?hierarchical=true (executes query in current and child contexts);
		- Current context is determined from the authorization token.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@Parameter(
		name = QueryTemplatePath.INCLUDE_CONTEXTS_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include the list of contexts where each returned instance is available.
		- Default value: false (context information not included);
		- Example: ?includeContexts=true (shows context availability for each instance);
		- Effect: Only applies when query returns complete instances of defined types;
		- No Effect: When query uses _emit to return individual fields instead of complete instances;
		- Note: A Resource is present in all contexts that form the union of contexts of all its Facets.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@RequestBody(
		description = """
		JSON object containing parameter values for template substitution.
		- Parameters not provided in the body will use their default values from the template definition;
		- Parameter names must match those defined in the template's templateVariables section.

			Example Request Body:
			{
				"$state" : "down"
			}
		""",
		required = false,
		content = @Content(mediaType = "application/json")
	)
	@APIResponse(
		responseCode = "200",
		description = "Query template successfully executed",
		content = @Content(mediaType = "application/json")
	)
	@APIResponse(
		responseCode = "400",
		description = "Invalid query parameters or malformed parameter values"
	)
	@APIResponse(
		responseCode = "404",
		description = "Query template with the specified name does not exist"
	)
	@APIResponse(
		responseCode = "403",
		description = "Insufficient permissions to execute the query template"
	)

	public String runQueryTemplate(
			@PathParam(QueryTemplateManager.QUERY_TEMPLATE_NAME_PATH_PARAMETER) 
			@Parameter(
				name = QueryTemplateManager.QUERY_TEMPLATE_NAME_PATH_PARAMETER,
				in = ParameterIn.PATH,
				description = "The name of the query template to execute (e.g., \"GetAllEServiceWithState\")",
				required = true,
				schema = @Schema(type = SchemaType.STRING, example = "GetAllEServiceWithState")
			)
			String queryTemplateName, 
			String params) throws NotFoundException, InvalidQueryException, ResourceRegistryException {
		logger.info("Requested {} with name", QueryTemplate.NAME, queryTemplateName);
		setAccountingMethod(Method.RUN, QueryTemplate.NAME);
		
		ServerRequestInfo serverRequestInfo = initRequestInfo();
		serverRequestInfo.checkAllBooleanQueryParameters();
		
		QueryTemplateManagement queryTemplateManagement = new QueryTemplateManagement();
		queryTemplateManagement.setName(queryTemplateName);
		queryTemplateManagement.setParams(params);
		return queryTemplateManagement.run();
	}

	/**
	 * Retrieves all instances of a specified type from the Resource Registry.
	 * 
	 * <strong>REST Endpoint:</strong> {@code GET /access/instances/{type-name}}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>Basic query: {@code GET /access/instances/EService}</li>
	 * <li>With pagination: {@code GET /access/instances/EService?limit=20&offset=40}</li>
	 * <li>With subtypes and metadata: {@code GET /access/instances/EService?polymorphic=true&includeMeta=true}</li>
	 * <li>Full metadata context: {@code GET /access/instances/EService?includeMeta=true&allMeta=true}</li>
	 * <li>With context information: {@code GET /access/instances/EService?includeContexts=true}</li>
	 * <li>Hierarchical (admin only): {@code GET /access/instances/EService?hierarchical=true}</li>
	 * <li>Hierarchical with contexts (admin only): {@code GET /access/instances/EService?hierarchical=true&includeContexts=true}.</li>
	 * </ul>
	 * 
	 * <strong>Path Parameters:</strong>
	 * <ul>
	 * <li><strong>type-name</strong>: The name of the Information System type whose instances should be retrieved (e.g., "EService", "ContactFacet").</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>polymorphic</strong> (optional):
	 * Whether to include instances of subtypes in the response.
	 * <ul>
	 * <li>Default value: true (includes all subtypes)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?polymorphic=false (returns only exact type instances).</li>
	 * </ul>
	 * 
	 * <strong>limit</strong> (optional):
	 * <ul>
	 * <li>Maximum number of instances to return in a single response</li>
	 * <li>Default value: 10</li>
	 * <li>Example: ?limit=50 (returns at most 50 instances).</li>
	 * </ul>
	 * 
	 * <strong>offset</strong> (optional):
	 * <ul>
	 * <li>Number of instances to skip from the beginning of the result set</li>
	 * <li>Default value: 0 (starts from the first instance)</li>
	 * <li>Example: ?offset=100 (skips the first 100 instances).</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (optional):
	 * Whether to include metadata in the response instances.
	 * <ul>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeMeta=true (includes metadata with appropriate filtering based on user role)</li>
	 * <li>Metadata includes only the requested type and subtypes (if polymorphic=true)</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (optional):
	 * Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.).
	 * <ul>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instances, more human-readable)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeMeta=true&amp;allMeta=true (includes metadata for all nested elements)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (optional, admin-only):
	 * Whether to include instances from child contexts of the current context.
	 * <ul>
	 * <li>Default value: false (current context only)</li>
	 * <li>Values: true|false</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles (Context-Manager must be of the current context derived from the authorization token)</li>
	 * <li>Example: ?hierarchical=true (includes instances from current and child contexts)</li>
	 * <li>Current context is determined from the authorization token.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (optional):
	 * Whether to include the list of contexts where each instance is available.
	 * <ul>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeContexts=true (shows context availability for each instance)</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Full access to all metadata and hierarchical querying capabilities</li>
	 * <li>Can access instances across all contexts</li>
	 * <li>Receives complete, unfiltered metadata including all administrative fields.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Full access to all metadata and hierarchical querying capabilities</li>
	 * <li>Can access instances across all contexts</li>
	 * <li>Receives complete, unfiltered metadata including all administrative fields.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Full access to all metadata and hierarchical querying capabilities</li>
	 * <li>Must be Context-Manager of the current context derived from the authorization token</li>
	 * <li>Can access hierarchical querying capabilities</li>
	 * <li>Receives complete, unfiltered metadata including all administrative fields.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Basic access with metadata filtering and no hierarchical querying</li>
	 * <li>Cannot use hierarchical querying (hierarchical parameter ignored)</li>
	 * <li>Receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Instances successfully retrieved</li>
	 * <li><strong>400 Bad Request</strong>: Invalid query parameters (e.g., invalid limit/offset values)</li>
	 * <li><strong>404 Not Found</strong>: The specified type does not exist</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to access instances in the current context.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON array containing the requested instances</li>
	 * <li>Each instance includes its properties, relations, and metadata (if requested)</li>
	 * <li>Empty array is returned if no instances of the specified type exist in the accessible contexts.</li>
	 * </ul>
	 * 
	 * <strong>Example Metadata (filtered for non-admin users):</strong>
	 * ``{@code json
	 * "metadata": {
	 *     "type": "Metadata",
	 *     "creationTime": "2025-03-18 17:13:40.952 +0100",
	 *     "createdBy": "HIDDEN_FOR_PRIVACY",
	 *     "lastUpdateBy": "HIDDEN_FOR_PRIVACY",
	 *     "lastUpdateTime": "2025-03-19 16:21:16.805 +0100"
	 * }
	 * }``
	 * 
	 * @param type The name of the Information System type whose instances to retrieve
	 * @param polymorphic Whether to include instances of subtypes (default: true)
	 * @return JSON array containing the requested instances with their properties and relations
	 * @throws NotFoundException If the specified type does not exist
	 * @throws ResourceRegistryException If an error occurs during instance retrieval
	 */
	@GET
	@Path(AccessPath.INSTANCES_PATH_PART + "/{" + TypeManager.TYPE_PATH_PARAMETER + "}")
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	@Operation(
		summary = "List Instances by specified Type",
		description = """
		Retrieves all instances of a specified type from the Resource Registry.
		The response can include metadata, hierarchical querying, and context information based on query parameters and user authorization.


		**Request Examples:**
		- GET /access/instances/EService;
		- GET /access/instances/EService?limit=20&offset=40;
		- GET /access/instances/EService?polymorphic=true&includeMeta=true;
		- GET /access/instances/EService?includeMeta=true&allMeta=true;
		- GET /access/instances/EService?includeContexts=true;
		- GET /access/instances/EService?hierarchical=true;
		- GET /access/instances/EService?hierarchical=true&includeContexts=true.


		**Authorization Requirements:**
		- **IS-Manager:**
			- Full access to all metadata and hierarchical querying capabilities;
			- Can access instances across all contexts;
			- Receives complete, unfiltered metadata including all administrative fields.

		- **Infrastructure-Manager:**
			- Full access to all metadata and hierarchical querying capabilities;
			- Can access instances across all contexts;
			- Receives complete, unfiltered metadata including all administrative fields.

		- **Context-Manager:**
			- Full access to all metadata and hierarchical querying capabilities;
			- Must be Context-Manager of the current context derived from the authorization token;
			- Can access hierarchical querying capabilities;
			- Receives complete, unfiltered metadata including all administrative fields.

		- **Other Users:**
			- Basic access with metadata filtering and no hierarchical querying;
			- Cannot use hierarchical querying (hierarchical parameter ignored);
			- Receive metadata with sensitive information filtered when includeMeta=true.
		"""
	)		
	@Parameter(
		name = InstancePath.LIMIT_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Maximum number of instances to return in a single response.
		- Default value: 10;
		- Example: ?limit=50 (returns at most 50 instances).
		""",
		required = false,
		schema = @Schema(type = SchemaType.INTEGER, defaultValue = "10", example = "10")
	)
	@Parameter(
		name = InstancePath.OFFSET_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Number of instances to skip from the beginning of the result set.
		- Default value: 0 (starts from the first instance);
		- Example: ?offset=100 (skips the first 100 instances).
		""",
		required = false,
		schema = @Schema(type = SchemaType.INTEGER, defaultValue = "0", example = "0")
	)
	@Parameter(
		name = InstancePath.INCLUDE_META_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include metadata in the response instances.
		- Default value: false (basic information only);
		- Example: ?includeMeta=true (includes metadata with appropriate filtering based on user role);
		- Metadata includes only the requested type and subtypes (if polymorphic=true);
		- Restriction: IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@Parameter(
		name = InstancePath.INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.).
		- Must be used in conjunction with includeMeta=true;
		- Default value: false (metadata only for main instances, more human-readable);
		- Example: ?includeMeta=true&allMeta=true (includes metadata for all nested elements);
		- Purpose: When false, produces more human-readable responses with less JSON to process.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@Parameter(
		name = InstancePath.HIERARCHICAL_MODE_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include instances from child contexts of the current context.
		- Default value: false (current context only);
		- Restriction: Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles (Context-Manager must be of the current context derived from the authorization token);
		- Example: ?hierarchical=true (includes instances from current and child contexts);
		- Current context is determined from the authorization token.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@Parameter(
		name = InstancePath.INCLUDE_CONTEXTS_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include the list of contexts where each instance is available.
		- Default value: false (context information not included);
		- Example: ?includeContexts=true (shows context availability for each instance);
		- Note: A Resource is present in all contexts that form the union of contexts of all its Facets;
		- Note: ConsistsOf relations are present in all contexts where their target Facets are present.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@APIResponse(
		responseCode = "200",
		description = "Instances successfully retrieved",
		content = @Content(mediaType = "application/json")
	)
	@APIResponse(
		responseCode = "400",
		description = "Invalid query parameters (e.g., invalid limit/offset values)"
	)
	@APIResponse(
		responseCode = "404",
		description = "The specified type does not exist"
	)
	@APIResponse(
		responseCode = "403",
		description = "Insufficient permissions to access instances in the current context"
	)
	public String listInstances(
			@PathParam(TypeManager.TYPE_PATH_PARAMETER)
			@Parameter(
				name = TypeManager.TYPE_PATH_PARAMETER,
				in = ParameterIn.PATH,
				description = "The name of the Information System type whose instances should be retrieved (e.g., 'EService', 'ContactFacet')",
				required = true,
				schema = @Schema(type = SchemaType.STRING, example = "EService")
			)
			String type,
			@QueryParam(InstancePath.POLYMORPHIC_QUERY_PARAMETER) @DefaultValue("true") 
			@Parameter(
				name = InstancePath.POLYMORPHIC_QUERY_PARAMETER,
				in = ParameterIn.QUERY,
				description = """
				Whether to include instances of subtypes in the response.
				- Default value: true (includes all subtypes);
				- Example: ?polymorphic=false (returns only exact type instances).
				""",
				required = false,
				schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "true", example = "true")
			)
			Boolean polymorphic)
			throws NotFoundException, ResourceRegistryException {
		logger.info("Requested all {}instances of {}", polymorphic ? InstancePath.POLYMORPHIC_QUERY_PARAMETER + " " : "", type);
		setAccountingMethod(Method.LIST, InstancesManager.INSTANCE);
		
		ServerRequestInfo serverRequestInfo = initRequestInfo();
		serverRequestInfo.checkAllBooleanQueryParameters();
		serverRequestInfo.checkLimitOffset();
		
		ElementManagement<?,?> erManagement = ERManagementUtility.getERManagement(type);
		return erManagement.all(polymorphic);
	}

	/**
	 * Checks if an instance with the specified type and UUID exists in the Resource Registry.
	 * 
	 * <strong>REST Endpoint:</strong> {@code HEAD /access/instances/{type-name}/{instance-uuid}}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>Basic check: {@code HEAD /access/instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f}</li>
	 * <li>Hierarchical check (admin only): {@code HEAD /access/instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f?hierarchical=true}.</li>
	 * </ul>
	 * 
	 * <strong>Path Parameters:</strong>
	 * <ul>
	 * <li><strong>type-name</strong>: The name of the Information System type (e.g., "EService", "ContactFacet")</li>
	 * <li><strong>instance-uuid</strong>: The unique identifier of the instance to check.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>hierarchical</strong> (optional, admin-only):
	 * Whether to check for instance existence in child contexts of the current context.
	 * <ul>
	 * <li>Default value: false (current context only)</li>
	 * <li>Values: true|false</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles (Context-Manager must be of the current context derived from the authorization token)</li>
	 * <li>Example: ?hierarchical=true (checks current context and all child contexts)</li>
	 * <li>Current context is determined from the authorization token.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can check existence of any instance type in any context</li>
	 * <li>Can use hierarchical querying to check across context hierarchies</li>
	 * <li>Has full administrative access to existence checking.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can check existence of any instance type in any context</li>
	 * <li>Can use hierarchical querying to check across context hierarchies</li>
	 * <li>Has full administrative access to existence checking.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can check existence of instances in the current context</li>
	 * <li>Can use hierarchical querying within their managed context hierarchy</li>
	 * <li>Must be Context-Manager of the current context (derived from the authorization token)</li>
	 * <li>Cannot check instances in contexts where they don't have Context-Manager role.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can check existence of instances in the current context only</li>
	 * <li>Cannot use hierarchical querying (parameter is ignored)</li>
	 * <li>Only within contexts where they have authorized access through their token.</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li><strong>204 No Content</strong>: Instance exists and is accessible in the current context</li>
	 * <li><strong>404 Not Found</strong>: Instance does not exist</li>
	 * <li><strong>403 Forbidden</strong>: Instance exists but is available in another context (not accessible in current context).</li>
	 * </ul>
	 * 
	 * @param type The name of the Information System type
	 * @param uuid The unique identifier of the instance to check
	 * @return HTTP response with appropriate status code (204, 404, or 403)
	 * @throws NotFoundException If the specified type or instance does not exist
	 * @throws ResourceRegistryException If an error occurs during the existence check
	 */
	@HEAD
	@Path(AccessPath.INSTANCES_PATH_PART + "/{" + TypeManager.TYPE_PATH_PARAMETER + "}" + "/{" + InstancesManager.INSTANCE_UUID_PATH_PARAMETER + "}")
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	@Operation(
		summary = "Check Instance Existence",
		description = """
		Checks if an instance with the specified type and UUID exists in the Resource Registry.
		The response status code indicates existence and accessibility based on context and user authorization.


		**Request Examples:**
		- HEAD /access/instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f;
		- HEAD /access/instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f?hierarchical=true.


		**Authorization Requirements:**
		- **IS-Manager:**
			- Can check existence of any instance type in any context;
			- Can use hierarchical querying to check across context hierarchies;
			- Has full administrative access to existence checking.

		- **Infrastructure-Manager:**
			- Can check existence of any instance type in any context;
			- Can use hierarchical querying to check across context hierarchies;
			- Has full administrative access to existence checking.

		- **Context-Manager:**
			- Can check existence of instances in the current context;
			- Can use hierarchical querying within their managed context hierarchy;
			- Must be Context-Manager of the current context (derived from the authorization token);
			- Cannot check instances in contexts where they don't have Context-Manager role.

		- **Other Users:**
			- Can check existence of instances in the current context only;
			- Cannot use hierarchical querying (parameter is ignored);
			- Only within contexts where they have authorized access through their token.
		"""
	)		
	@Parameter(
		name = InstancePath.HIERARCHICAL_MODE_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to check for instance existence in child contexts of the current context.
		- Default value: false (current context only);
		- Restriction: Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles (Context-Manager must be of the current context derived from the authorization token);
		- Example: ?hierarchical=true (checks current context and all child contexts);
		- Current context is determined from the authorization token.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@APIResponse(
		responseCode = "204",
		description = "Instance exists and is accessible in the current context"
	)
	@APIResponse(
		responseCode = "404",
		description = "Instance does not exist"
	)
	@APIResponse(
		responseCode = "403",
		description = "Instance exists but is available in another context (not accessible in current context)"
	)
	public Response existsInstance(
			@PathParam(TypeManager.TYPE_PATH_PARAMETER)
			@Parameter(
				name = TypeManager.TYPE_PATH_PARAMETER,
				in = ParameterIn.PATH,
				description = "The name of the Information System type (e.g., 'EService', 'ContactFacet')",
				required = true,
				schema = @Schema(type = SchemaType.STRING, example = "EService")
			)
			String type,
			@PathParam(InstancesManager.INSTANCE_UUID_PATH_PARAMETER) 
			@Parameter(
				name = "UUID",
				in = ParameterIn.PATH,
				description = "The unique identifier of the instance to check",
				required = true,
				schema = @Schema(type = SchemaType.STRING, example = "48af15ad-7e56-4157-b624-71c98cea4f8f")
			)
			String uuid) throws NotFoundException, ResourceRegistryException {
		logger.info("Requested to check if {} with id {} exists", type, uuid);
		setAccountingMethod(Method.EXIST, InstancesManager.INSTANCE);
		
		ServerRequestInfo serverRequestInfo = initRequestInfo();
		serverRequestInfo.checkBooleanQueryParameter(InstancePath.HIERARCHICAL_MODE_QUERY_PARAMETER);
		
		ElementManagement<?,?> erManagement = ERManagementUtility.getERManagement(type);
		
		try {
			erManagement.setUUID(UUID.fromString(uuid));
			boolean found = erManagement.exists();
			if(found) {
				return Response.status(Status.NO_CONTENT).build();
			} else {
				// This code should never be reached due to exception management
				// anyway adding it for safety reason
				return Response.status(Status.NOT_FOUND).build();
			}
		} catch(NotFoundException e) {
			return Response.status(Status.NOT_FOUND).build();
		} catch(AvailableInAnotherContextException e) {
			return Response.status(Status.FORBIDDEN).build();
		} catch(ResourceRegistryException e) {
			throw e;
		}
	}

	/**
	 * Retrieves a single instance with the specified type and UUID.
	 * 
	 * <strong>REST Endpoint:</strong> {@code GET /access/instances/{type-name}/{instance-uuid}}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>Basic query: {@code GET /access/instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f}</li>
	 * <li>With metadata: {@code GET /access/instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f?includeMeta=true}</li>
	 * <li>Full metadata context: {@code GET /access/instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f?includeMeta=true&amp;allMeta=true}</li>
	 * <li>With context information: {@code GET /access/instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f?includeContexts=true}</li>
	 * <li>Hierarchical (admin only): {@code GET /access/instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f?hierarchical=true}</li>
	 * <li>Hierarchical with contexts (admin only): {@code GET /access/instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f?hierarchical=true&includeContexts=true}.</li>
	 * </ul>
	 * 
	 * <strong>Path Parameters:</strong>
	 * <ul>
	 * <li><strong>type-name</strong>: The name of the Information System type (e.g., "EService", "ContactFacet")</li>
	 * <li><strong>instance-uuid</strong>: The unique identifier of the instance to retrieve (e.g., "48af15ad-7e56-4157-b624-71c98cea4f8f").</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (optional):
	 * Whether to include metadata in the response instance.
	 * <ul>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeMeta=true (includes metadata with appropriate filtering based on user role)</li>
	 * <li>Metadata includes the requested instance and nested elements based on polymorphic setting</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (optional):
	 * Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.).
	 * <ul>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeMeta=true&amp;allMeta=true (includes metadata for all nested elements)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (optional, admin-only):
	 * Whether to retrieve the instance from child contexts of the current context.
	 * <ul>
	 * <li>Default value: false (current context only)</li>
	 * <li>Values: true|false</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles (Context-Manager must be of the current context derived from the authorization token)</li>
	 * <li>Example: ?hierarchical=true (searches current and child contexts for the instance)</li>
	 * <li>Current context is determined from the authorization token.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (optional):
	 * Whether to include the list of contexts where the instance and its nested elements are available.
	 * <ul>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeContexts=true (shows context availability for the instance and nested elements)</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Full access to retrieve any instance in any context</li>
	 * <li>Can use hierarchical querying to search across context hierarchies</li>
	 * <li>Receive complete, unfiltered metadata for the instance and all nested elements</li>
	 * <li>Metadata includes all administrative fields without obfuscation.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Full access to retrieve any instance in any context</li>
	 * <li>Can use hierarchical querying to search across context hierarchies</li>
	 * <li>Receive complete, unfiltered metadata for the instance and all nested elements</li>
	 * <li>Metadata includes all administrative fields without obfuscation.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can retrieve instances in the current context and child contexts</li>
	 * <li>Can use hierarchical querying within their managed context hierarchy</li>
	 * <li>Must be Context-Manager of the current context (derived from the authorization token)</li>
	 * <li>Receive complete, unfiltered metadata for the instance and all nested elements</li>
	 * <li>Metadata includes all administrative fields without obfuscation.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can retrieve instances in the current context only</li>
	 * <li>Cannot use hierarchical querying (hierarchical parameter ignored)</li>
	 * <li>Receive metadata with sensitive information filtered when includeMeta=true:</li>
	 *   <li>Date fields (creation, modification) are visible</li>
	 *   <li>User identifiers (createdBy, lastUpdateBy) are obfuscated or hidden</li>
	 *   <li>Other administrative details may be filtered.</li>
	 *   </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Instance successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: Instance does not exist</li>
	 * <li><strong>403 Forbidden</strong>: Instance exists but is available in another context (not accessible in current context).</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON object containing the requested instance</li>
	 * <li>Includes its properties, relations, and metadata (if requested)</li>
	 * <li>Nested elements are included based on polymorphic and metadata settings.</li>
	 * </ul>
	 * 
	 * <strong>Example Metadata (filtered for non-admin users):</strong>
	 * ``{@code json
	 * "metadata": {
	 *     "type": "Metadata",
	 *     "creationTime": "2025-03-18 17:13:40.952 +0100",
	 *     "createdBy": "HIDDEN_FOR_PRIVACY",
	 *     "lastUpdateBy": "HIDDEN_FOR_PRIVACY",
	 *     "lastUpdateTime": "2025-03-19 16:21:16.805 +0100"
	 * }
	 * }``
	 * 
	 * @param type The name of the Information System type
	 * @param uuid The unique identifier of the instance to retrieve
	 * @return JSON object containing the requested instance with its properties and relations
	 * @throws NotFoundException If the specified type or instance does not exist
	 * @throws ResourceRegistryException If an error occurs during instance retrieval
	 */
	@GET
	@Path(AccessPath.INSTANCES_PATH_PART + "/{" + TypeManager.TYPE_PATH_PARAMETER + "}" + "/{" + InstancesManager.INSTANCE_UUID_PATH_PARAMETER + "}")
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	@Operation(
		summary = "Read Instance by type and UUID",
		description = """
		Retrieves a single instance with the specified type and UUID.
		The response can include metadata, hierarchical querying, and context information based on query parameters and user authorization.


		**Request Examples:**
		- GET /access/instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f;
		- GET /access/instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f?includeMeta=true;
		- GET /access/instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f?includeMeta=true&allMeta=true;
		- GET /access/instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f?includeContexts=true;
		- GET /access/instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f?hierarchical=true;
		- GET /access/instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f?hierarchical=true&includeContexts=true.


		**Authorization Requirements:**
		- **IS-Manager:**
			- Full access to retrieve any instance in any context;
			- Can use hierarchical querying to search across context hierarchies;
			- Receive complete, unfiltered metadata for the instance and all nested elements;
			- Metadata includes all administrative fields without obfuscation.

		- **Infrastructure-Manager:**
			- Full access to retrieve any instance in any context;
			- Can use hierarchical querying to search across context hierarchies;
			- Receive complete, unfiltered metadata for the instance and all nested elements;
			- Metadata includes all administrative fields without obfuscation.

		- **Context-Manager:**
			- Can retrieve instances in the current context and child contexts;
			- Can use hierarchical querying within their managed context hierarchy;
			- Must be Context-Manager of the current context (derived from the authorization token);
			- Receive complete, unfiltered metadata for the instance and all nested elements;
			- Metadata includes all administrative fields without obfuscation.

		- **Other Users:**
			- Can retrieve instances in the current context only;
			- Cannot use hierarchical querying (hierarchical parameter ignored);
			- Receive metadata with sensitive information filtered when includeMeta=true:
				- Date fields (creation, modification) are visible;
				- User identifiers (createdBy, lastUpdateBy) are obfuscated or hidden;
				- Other administrative details may be filtered.
		"""
	)		
	@Parameter(
		name = InstancePath.INCLUDE_META_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include metadata in the response instance.
		- Default value: false (basic information only);
		- Example: ?includeMeta=true (includes metadata with appropriate filtering based on user role);
		- Metadata includes the requested instance and nested elements based on polymorphic setting;
		- Restriction: IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@Parameter(
		name = InstancePath.INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.).
		- Must be used in conjunction with includeMeta=true;
		- Default value: false (metadata only for main instance, more human-readable);
		- Example: ?includeMeta=true&allMeta=true (includes metadata for all nested elements);
		- Purpose: When false, produces more human-readable responses with less JSON to process.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@Parameter(
		name = InstancePath.HIERARCHICAL_MODE_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to retrieve the instance from child contexts of the current context.
		- Default value: false (current context only);
		- Restriction: Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles (Context-Manager must be of the current context derived from the authorization token);
		- Example: ?hierarchical=true (searches current and child contexts for the instance);
		- Current context is determined from the authorization token.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@Parameter(
		name = InstancePath.INCLUDE_CONTEXTS_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include the list of contexts where the instance and its nested elements are available.
		- Default value: false (context information not included);
		- Example: ?includeContexts=true (shows context availability for the instance and nested elements);
		- Note: A Resource is present in all contexts that form the union of contexts of all its Facets;
		- Note: ConsistsOf relations are present in all contexts where their target Facets are present.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@APIResponse(
		responseCode = "200",
		description = "Instance successfully retrieved",
		content = @Content(mediaType = "application/json")
	)
	@APIResponse(
		responseCode = "404",
		description = "Instance does not exist"
	)
	@APIResponse(
		responseCode = "403",
		description = "Instance exists but is available in another context (not accessible in current context)"
	)
	public String readInstance(
			@PathParam(TypeManager.TYPE_PATH_PARAMETER) 
			@Parameter(
				name = TypeManager.TYPE_PATH_PARAMETER,
				in = ParameterIn.PATH,
				description = "The name of the Information System type (e.g., 'EService', 'ContactFacet')",
				required = true,
				schema = @Schema(type = SchemaType.STRING, example = "EService")
			)
			String type,
			@PathParam(InstancesManager.INSTANCE_UUID_PATH_PARAMETER) 
			@Parameter(
				name = InstancesManager.INSTANCE_UUID_PATH_PARAMETER,
				in = ParameterIn.PATH,
				description = "The unique identifier of the instance to retrieve",
				required = true,
				schema = @Schema(type = SchemaType.STRING, example = "48af15ad-7e56-4157-b624-71c98cea4f8f")
			)
			String uuid) throws NotFoundException, ResourceRegistryException {
		logger.info("Requested to read {} with id {}", type, uuid);
		setAccountingMethod(Method.READ, InstancesManager.INSTANCE);
		
		ServerRequestInfo serverRequestInfo = initRequestInfo();
		serverRequestInfo.checkAllBooleanQueryParameters();
		
		@SuppressWarnings("rawtypes")
		ElementManagement erManagement = ERManagementUtility.getERManagement(type);
		
		erManagement.setElementType(type);
		erManagement.setUUID(UUID.fromString(uuid));
		return erManagement.read().toString();
	}

	/**
	 * Retrieves the list of contexts where a specific instance is present in the Resource Registry.
	 * 
	 * <strong>REST Endpoint:</strong> {@code GET /access/instances/{type-name}/{instance-uuid}/contexts}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>Get contexts for a resource: {@code GET /access/instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f/contexts}</li>
	 * <li>Get contexts for a facet: {@code GET /access/instances/ContactFacet/4023d5b2-8601-47a5-83ef-49ffcbfc7d86/contexts}.</li>
	 * </ul>
	 * 
	 * <strong>Path Parameters:</strong>
	 * <ul>
	 * <li><strong>type-name</strong>: The name of the Information System type (e.g., "EService", "ContactFacet")</li>
	 * <li><strong>instance-uuid</strong>: The unique identifier of the instance whose contexts to retrieve.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * <ul>
	 * <li>This endpoint does not accept any query parameters</li>
	 * <li>No filtering, pagination, or metadata options are available.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can retrieve context information for any instance</li>
	 * <li>Full access to view contexts across the entire infrastructure.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can retrieve context information for any instance</li>
	 * <li>Full access to view contexts across the entire infrastructure.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can retrieve context information for any accessible instance</li>
	 * <li>Access depends on their Context-Manager role in relevant contexts.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can retrieve context information for any accessible instance</li>
	 * <li>Access depends on their permissions and context availability through their token.</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Context list successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: Instance does not exist</li>
	 * <li><strong>403 Forbidden</strong>: Instance exists but is available in another context (not accessible for context listing).</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON object containing the contexts where the instance is present</li>
	 * <li>Each key is a context UUID and each value is the current full context path</li>
	 * <li>Empty object is returned if the instance exists but is not present in any contexts.</li>
	 * </ul>
	 * 
	 * <strong>Example Response:</strong>
	 * ``{@code json
	 * {
	 *   "3d846e2f-3582-4344-b03a-629d4cd34d53": "/gcube/devsec/devVRE",
	 *   "4eb849d5-efbb-4430-9eb6-1968123921b4": "/gcube",
	 *   "a7bc333f-84df-498d-ae9d-748b358267b5": "/gcube/devsec"
	 * }
	 * }``
	 * 
	 * @param type The name of the Information System type
	 * @param instanceId The unique identifier of the instance
	 * @return JSON object containing context UUID to context path mappings where the instance is present
	 * @throws NotFoundException If the instance does not exist
	 * @throws ResourceRegistryException If an error occurs during context retrieval
	 */
	@GET
	@Path(AccessPath.INSTANCES_PATH_PART + "/{" + TypeManager.TYPE_PATH_PARAMETER + "}" + "/{" + InstancesManager.INSTANCE_UUID_PATH_PARAMETER + "}/" + AccessPath.CONTEXTS_PATH_PART)
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	@Operation(
		summary = "Get Contexts for Instance",
		description = """
		Retrieves the list of contexts where a specific instance is present in the Resource Registry.
		This endpoint does not accept any query parameters.


		**Request Examples:**
		- GET /access/instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f/contexts;
		- GET /access/instances/ContactFacet/4023d5b2-8601-47a5-83ef-49ffcbfc7d86/contexts.


		**Authorization Requirements:**
		- **IS-Manager:**
			- Can retrieve context information for any instance;
			- Full access to view contexts across the entire infrastructure.

		- **Infrastructure-Manager:**
			- Can retrieve context information for any instance;
			- Full access to view contexts across the entire infrastructure.

		- **Context-Manager:**
			- Can retrieve context information for any accessible instance;
			- Access depends on their Context-Manager role in relevant contexts.

		- **Other Users:**
			- Can retrieve context information for any accessible instance;
			- Access depends on their permissions and context availability through their token.
		"""
	)
	@APIResponse(
		responseCode = "200",
		description = "Context list successfully retrieved",
		content = @Content(mediaType = "application/json")
	)
	@APIResponse(
		responseCode = "404",
		description = "Instance does not exist"
	)
	@APIResponse(
		responseCode = "403",
		description = "Instance exists but is available in another context (not accessible for context listing)"
	)
	public String getInstanceContexts(
			@PathParam(TypeManager.TYPE_PATH_PARAMETER)
			@Parameter(
				name = TypeManager.TYPE_PATH_PARAMETER,
				in = ParameterIn.PATH,
				description = "The name of the Information System type (e.g., 'EService', 'ContactFacet')",
				required = true,
				schema = @Schema(type = SchemaType.STRING, example = "EService")
			)
			String type,
			@PathParam(InstancesManager.INSTANCE_UUID_PATH_PARAMETER) 
			@Parameter(
				name = InstancesManager.INSTANCE_UUID_PATH_PARAMETER,
				in = ParameterIn.PATH,
				description = "The unique identifier of the instance whose contexts to retrieve",
				required = true,
				schema = @Schema(type = SchemaType.STRING, example = "48af15ad-7e56-4157-b624-71c98cea4f8f")
			)
			String instanceId) throws NotFoundException, ResourceRegistryException {
		logger.info("Requested to get contexts of {} with UUID {}", type, instanceId);
		setAccountingMethod(InstancesManager.GET_INSTANCE_CONTEXTS_METHOD);
		
		ElementManagement<?,?> erManagement = ERManagementUtility.getERManagement(type);
		erManagement.setUUID(UUID.fromString(instanceId));
		return erManagement.getContexts();
	}
	
	/**
	 * Executes a graph query directly against the Resource Registry backend database.
	 * 
	 * <strong>REST Endpoint:</strong> {@code GET /access/query?q={raw-query}[&raw={true|false}]}
	 * 
	 * <strong>⚠️ PRODUCTION WARNING:</strong>
	 * <ul>
	 * <li>⚠️ <strong>NOT FOR PRODUCTION USE</strong>: This endpoint is intended for development, debugging, and administrative purposes only</li>
	 * <li>⚠️ <strong>NO COMPATIBILITY GUARANTEE</strong>: The query syntax and behavior may change without notice</li>
	 * <li>⚠️ <strong>BACKEND DEPENDENCY</strong>: Currently based on OrientDB which may be replaced at any time without prior warning</li>
	 * <li>⚠️ <strong>BREAKING CHANGES</strong>: Future backend changes will break existing queries without migration support</li>
	 * <li>⚠️ <strong>USE AT YOUR OWN RISK</strong>: Production applications should use type-specific endpoints instead.</li>
	 * </ul>
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>Basic vertex query: {@code GET /access/query?q=SELECT FROM V}</li>
	 * <li>Count resources: {@code GET /access/query?q=SELECT count(*) FROM Resource}</li>
	 * <li>Raw backend response: {@code GET /access/query?q=SELECT FROM EService&raw=true}</li>
	 * <li>With pagination: {@code GET /access/query?q=SELECT FROM EService SKIP 10 LIMIT 20}</li>
	 * <li>Complex query: {@code GET /access/query?q=SELECT FROM EService WHERE out('ConsistsOf').@class = 'StateFacet'}</li>
	 * <li>With metadata (raw=false): {@code GET /access/query?q=SELECT FROM EService&includeMeta=true}</li>
	 * <li>Hierarchical (admin only): {@code GET /access/query?q=SELECT FROM EService&hierarchical=true}.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>q</strong> (required):
	 * <ul>
	 * <li>The OrientDB SQL query to execute against the backend</li>
	 * <li>Must be URL-encoded for special characters</li>
	 * <li>Only idempotent queries are accepted (SELECT statements)</li>
	 * <li>For query syntax reference: <a href="https://orientdb.com/docs/last/SQL-Syntax.html" target="_blank">OrientDB SQL Documentation</a></li>
	 * <li>Example: ?q=SELECT FROM EService WHERE out('ConsistsOf').@class = 'StateFacet'.</li>
	 * </ul>
	 * 
	 * <strong>raw</strong> (optional):
	 * Whether to return the raw backend response or format it according to the Information System model.
	 * <ul>
	 * <li>Default value: false (format results according to IS model)</li>
	 * <li>Values: true|false</li>
	 * <li>When false: Results are converted to standard Information System model format with proper typing and structure</li>
	 * <li>When true: Results are returned exactly as provided by the OrientDB backend without any formatting or conversion</li>
	 * <li>Example: ?raw=true (returns raw OrientDB response format).</li>
	 * </ul>
	 * 
	 * <strong>polymorphic</strong> (optional, when raw=false):
	 * Whether to include instances of subtypes both main type instance as well as nested instances.
	 * <ul>
	 * <li>Default value: true (includes all subtypes both main type instance as well as nested instances)</li>
	 * <li>Values: true|false</li>
	 * <li>Only effective when raw=false (ignored when raw=true)</li>
	 * <li>Example: ?polymorphic=false (exact type matching both main type instance as well as nested instances).</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (optional, when raw=false):
	 * Whether to include metadata in the response instances.
	 * <ul>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Values: true|false</li>
	 * <li>Only effective when raw=false (ignored when raw=true)</li>
	 * <li>Example: ?includeMeta=true (includes complete metadata for admin users).</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (optional, when raw=false):
	 * Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.).
	 * <ul>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instances)</li>
	 * <li>Values: true|false</li>
	 * <li>Only effective when raw=false (ignored when raw=true)</li>
	 * <li>Example: ?includeMeta=true&amp;allMeta=true (includes metadata for all nested elements).</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (optional, admin-only):
	 * Whether to execute the query across child contexts of the current context.
	 * <ul>
	 * <li>Default value: false (current context only)</li>
	 * <li>Values: true|false</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager and Infrastructure-Manager roles</li>
	 * <li>Example: ?hierarchical=true (executes query in current and child contexts)</li>
	 * <li>When raw=true: Only this parameter is checked (other boolean parameters ignored)</li>
	 * <li>When raw=false: All boolean parameters are validated and applied.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (optional, when raw=false):
	 * Whether to include the list of contexts where each returned instance is available.
	 * <ul>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Values: true|false</li>
	 * <li>Only effective when raw=false (ignored when raw=true)</li>
	 * <li>Example: ?includeContexts=true (shows context availability for each instance).</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Full access to execute any graph query</li>
	 * <li>Can use hierarchical querying across all contexts</li>
	 * <li>Receives complete, unfiltered metadata when raw=false and includeMeta=true</li>
	 * <li>Can access raw backend responses when raw=true.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Full access to execute any graph query</li>
	 * <li>Can use hierarchical querying across all contexts</li>
	 * <li>Receives complete, unfiltered metadata when raw=false and includeMeta=true</li>
	 * <li>Can access raw backend responses when raw=true.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Full access to execute any graph query</li>
	 * <li>Can use hierarchical querying across all contexts</li>
	 * <li>Receives complete, unfiltered metadata when raw=false and includeMeta=true</li>
	 * <li>Can access raw backend responses when raw=true.</li>
	 * </ul>
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li><strong>Query Validation</strong>: Only idempotent SELECT queries are accepted; INSERT/UPDATE/DELETE operations are rejected</li>
	 * <li><strong>Pagination Responsibility</strong>: Client must handle pagination using OrientDB SQL SKIP and LIMIT clauses</li>
	 * <li><strong>Context Scope</strong>: Queries execute within the current context unless hierarchical=true is specified by authorized users</li>
	 * <li><strong>Response Format</strong>: Depends on raw parameter - either OrientDB native format or Information System model format</li>
	 * <li><strong>Performance</strong>: Raw queries may be faster but sacrifice type safety and model consistency.</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Query successfully executed</li>
	 * <li><strong>400 Bad Request</strong>: Invalid query syntax or non-idempotent query attempted</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to execute graph queries (only IS-Manager and Infrastructure-Manager allowed).</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>When raw=false: JSON array containing instances formatted according to Information System model with proper typing, metadata (if requested), and context information (if requested)</li>
	 * <li>When raw=true: JSON response exactly as returned by the OrientDB backend without any formatting or model conversion</li>
	 * <li>Empty array/response is returned if no results match the query criteria.</li>
	 * </ul>
	 * 
	 * @param query The OrientDB SQL query to execute (must be idempotent SELECT statement)
	 * @param raw Whether to return raw backend response (true) or Information System model formatted response (false)
	 * @return JSON containing query results - either raw OrientDB format or IS model format based on raw parameter
	 * @throws InvalidQueryException if the query is invalid, not idempotent, or contains syntax errors
	 */
	@GET
	@Path(AccessPath.QUERY_PATH_PART)
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	@Operation(
		summary = "Execute Graph Query (OrientDB SQL)",
		description = """
		Executes a graph query directly against the Resource Registry backend database using OrientDB SQL syntax.
		
		This endpoint is intended for development, debugging, and administrative purposes only. Not for production use.


		⚠️ **PRODUCTION WARNING**:
		- ⚠️ **NOT FOR PRODUCTION USE**: This endpoint is intended for development, debugging, and administrative purposes only;
		- ⚠️ **NO COMPATIBILITY GUARANTEE**: The query syntax and behavior may change without notice;
		- ⚠️ **BACKEND DEPENDENCY**: Currently based on OrientDB which may be replaced at any time without prior warning;
		- ⚠️ **BREAKING CHANGES**: Future backend changes will break existing queries without migration support;
		- ⚠️ **USE AT YOUR OWN RISK**: Production applications should use type-specific endpoints instead.


		**Request Examples:**
		- GET /access/query?q=SELECT FROM V;
		- GET /access/query?q=SELECT count(*) FROM Resource;
		- GET /access/query?q=SELECT FROM EService&raw=true;
		- GET /access/query?q=SELECT FROM EService SKIP 10 LIMIT 20;
		- GET /access/query?q=SELECT FROM EService WHERE out('ConsistsOf').@class = 'StateFacet';
		- GET /access/query?q=SELECT FROM EService&includeMeta=true;
		- GET /access/query?q=SELECT FROM EService&hierarchical=true.


		**Authorization Requirements:**
		- **IS-Manager:**
			- Full access to execute any graph query;
			- Can use hierarchical querying across all contexts;
			- Receives complete, unfiltered metadata when raw=false and includeMeta=true;
			- Can access raw backend responses when raw=true.

		- **Infrastructure-Manager:**
			- Full access to execute any graph query;
			- Can use hierarchical querying across all contexts;
			- Receives complete, unfiltered metadata when raw=false and includeMeta=true;
			- Can access raw backend responses when raw=true.
		"""
	)		
	@Parameter(
		name = AccessPath.POLYMORPHIC_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include instances of subtypes both main type instance as well as nested instances.
		- Default value: true (includes all subtypes both main type instance as well as nested instances);
		- Only effective when raw=false (ignored when raw=true);
		- Example: ?polymorphic=false (exact type matching both main type instance as well as nested instances).
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "true", example = "true")
	)
	@Parameter(
		name = AccessPath.INCLUDE_META_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include metadata in the response instances.
		- Default value: false (basic information only);
		- Only effective when raw=false (ignored when raw=true);
		- Example: ?includeMeta=true (includes complete metadata for admin users).
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@Parameter(
		name = AccessPath.INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.).
		- Must be used in conjunction with includeMeta=true;
		- Default value: false (metadata only for main instances);
		- Only effective when raw=false (ignored when raw=true);
		- Example: ?includeMeta=true&allMeta=true (includes metadata for all nested elements).
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@Parameter(
		name = AccessPath.HIERARCHICAL_MODE_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to execute the query across child contexts of the current context.
		- Default value: false (current context only);
		- Restriction: Only available to IS-Manager and Infrastructure-Manager roles;
		- Example: ?hierarchical=true (executes query in current and child contexts);
		- When raw=true: Only this parameter is checked (other boolean parameters ignored);
		- When raw=false: All boolean parameters are validated and applied.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@Parameter(
		name = AccessPath.INCLUDE_CONTEXTS_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include the list of contexts where each returned instance is available.
		- Default value: false (context information not included);
		- Only effective when raw=false (ignored when raw=true);
		- Example: ?includeContexts=true (shows context availability for each instance).
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@APIResponse(
		responseCode = "200",
		description = "Query successfully executed",
		content = @Content(mediaType = "application/json")
	)
	@APIResponse(
		responseCode = "400",
		description = "Invalid query syntax or non-idempotent query attempted"
	)
	@APIResponse(
		responseCode = "403",
		description = "Insufficient permissions to execute graph queries (only IS-Manager and Infrastructure-Manager allowed)"
	)
	public String graphQuery(
			@QueryParam(AccessPath.Q_QUERY_PARAMETER) 
			@Parameter(
				name = AccessPath.Q_QUERY_PARAMETER,
				in = ParameterIn.QUERY,
				description = """
				The OrientDB SQL query to execute against the backend.
				- Must be URL-encoded for special characters;
				- Only idempotent queries are accepted (SELECT statements);
				- For query syntax reference: https://orientdb.com/docs/last/SQL-Syntax.html;
				- Example: SELECT FROM EService WHERE out('ConsistsOf').@class = 'StateFacet'.
				""",
				required = true,
				schema = @Schema(type = SchemaType.STRING, example = "SELECT FROM EService WHERE out('ConsistsOf').@class = 'StateFacet'")
			)
			String query,
			@QueryParam(AccessPath.RAW_QUERY_PARAMETER) @DefaultValue(AccessPath.RAW_QUERY_PARAMETER_DEFAULT_VALUE) 
			@Parameter(
				name = AccessPath.RAW_QUERY_PARAMETER,
				in = ParameterIn.QUERY,
				description = """
				Whether to return the raw backend response or format it according to the Information System model.
				- Default value: false (format results according to IS model);
				- When false: Results are converted to standard Information System model format with proper typing and structure;
				- When true: Results are returned exactly as provided by the OrientDB backend without any formatting or conversion;
				- Example: ?raw=true (returns raw OrientDB response format).
				""",
				required = false,
				schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = AccessPath.RAW_QUERY_PARAMETER_DEFAULT_VALUE, example = "true")
			)
			Boolean raw)
			throws InvalidQueryException {
		logger.info("Requested query (Raw {}):\n{}", raw, query);
		if(raw) {
			setAccountingMethod(Method.QUERY, RAW_QUERY_METHOD);
		}else {
			setAccountingMethod(Method.QUERY, GRAPH_QUERY_METHOD);
		}

		// TODO Allow only IS-Manager e Infrastructure-Manager to invoke this method 

		ServerRequestInfo serverRequestInfo = initRequestInfo();
		if(raw) {
			// TODO Check if the role allow to request raw data 
			serverRequestInfo.checkBooleanQueryParameter(InstancePath.HIERARCHICAL_MODE_QUERY_PARAMETER);
		}else {
			serverRequestInfo.checkAllBooleanQueryParameters();
		}
		
		Query queryManager = new QueryImpl();
		return queryManager.query(query, raw);
	}

	/**
	 * Executes a JSON-based query to find instances matching the specified structure in the Resource Registry.
	 * 
	 * <strong>REST Endpoint:</strong> {@code POST /access/query}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>Basic JSON query: {@code POST /access/query}</li>
	 * <li>With pagination: {@code POST /access/query?limit=20&offset=40}</li>
	 * <li>With subtypes and metadata: {@code POST /access/query?polymorphic=true&includeMeta=true}</li>
	 * <li>Full metadata context: {@code POST /access/query?includeMeta=true&amp;allMeta=true}</li>
	 * <li>With context information: {@code POST /access/query?includeContexts=true}</li>
	 * <li>Hierarchical (admin only): {@code POST /access/query?hierarchical=true}</li>
	 * <li>Hierarchical with contexts (admin only): {@code POST /access/query?hierarchical=true&includeContexts=true}.</li>
	 * </ul>
	 * 
	 * <strong>Request Body:</strong>
	 * <ul>
	 * <li>JSON object describing the structure to search for</li>
	 * <li>Must specify the target type and optionally its relations and properties</li>
	 * <li>Relations can include ConsistsOf, IsRelatedTo, and other defined relation types</li>
	 * <li>Properties can specify exact values or patterns to match.</li>
	 * </ul>
	 * 
	 * <strong>Example Request Body - Find EService with specific state and software:</strong>
	 * ``{@code json
	 * {
	 *   "type": "EService",
	 *   "consistsOf": [
	 *     {
	 *       "type": "ConsistsOf",
	 *       "propagationConstraint": {
	 *         "add": "propagate"
	 *       },
	 *       "target": {
	 *         "type": "StateFacet",
	 *         "value": "down"
	 *       }
	 *     },
	 *     {
	 *       "type": "IsIdentifiedBy",
	 *       "target": {
	 *         "type": "SoftwareFacet",
	 *         "name": "data-transfer-service",
	 *         "group": "DataTransfer"
	 *       }
	 *     }
	 *   ]
	 * }
	 * }`{@code 
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>polymorphic</strong> (optional):
	 * Whether to include instances of subtypes in the response and nested relations.
	 * <ul>
	 * <li>Default value: true (includes all subtypes)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?polymorphic=false (returns only exact type instances).</li>
	 * </ul>
	 * 
	 * <strong>limit</strong> (optional):
	 * <ul>
	 * <li>Maximum number of instances to return in a single response</li>
	 * <li>Default value: 10</li>
	 * <li>Example: ?limit=50 (returns at most 50 instances).</li>
	 * </ul>
	 * 
	 * <strong>offset</strong> (optional):
	 * <ul>
	 * <li>Number of instances to skip from the beginning of the result set</li>
	 * <li>Default value: 0 (starts from the first instance)</li>
	 * <li>Example: ?offset=100 (skips the first 100 instances).</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (optional):
	 * Whether to include metadata in the response instances.
	 * <ul>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeMeta=true (includes metadata with appropriate filtering based on user role)</li>
	 * <li>Metadata includes only the requested type and subtypes (if polymorphic=true)</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (optional):
	 * Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.).
	 * <ul>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instances, more human-readable)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeMeta=true&amp;allMeta=true (includes metadata for all nested elements)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (optional, admin-only):
	 * Whether to include instances from child contexts of the current context.
	 * <ul>
	 * <li>Default value: false (current context only)</li>
	 * <li>Values: true|false</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles (Context-Manager must be of the current context derived from the authorization token)</li>
	 * <li>Example: ?hierarchical=true (includes instances from current and child contexts)</li>
	 * <li>Current context is determined from the authorization token.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (optional):
	 * Whether to include the list of contexts where each instance is available.
	 * <ul>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeContexts=true (shows context availability for each instance)</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Full access to execute any JSON query</li>
	 * <li>Can use hierarchical querying across all contexts</li>
	 * <li>Receives complete, unfiltered metadata including all administrative fields.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Full access to execute any JSON query</li>
	 * <li>Can use hierarchical querying across all contexts</li>
	 * <li>Receives complete, unfiltered metadata including all administrative fields.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can execute JSON queries with hierarchical querying within their managed context hierarchy</li>
	 * <li>Must be Context-Manager of the current context (derived from the authorization token)</li>
	 * <li>Receives complete, unfiltered metadata including all administrative fields.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can execute JSON queries in the current context only</li>
	 * <li>Cannot use hierarchical querying (hierarchical parameter ignored)</li>
	 * <li>Receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li><strong>Structure Matching</strong>: The JSON query describes the structure to match - instances must have the specified properties and relations</li>
	 * <li><strong>Flexible Matching</strong>: Properties not specified in the query are ignored; only specified properties must match</li>
	 * <li><strong>Relation Traversal</strong>: ConsistsOf and IsRelatedTo relations are traversed to find matching nested structures</li>
	 * <li><strong>Type Polymorphism</strong>: When polymorphic=true, subtypes are included in both main results and nested relations</li>
	 * <li><strong>Pagination Support</strong>: Results can be paginated using limit and offset parameters</li>
	 * <li><strong>Context Scope</strong>: Queries execute within the current context unless hierarchical=true is specified by authorized users.</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: JSON query successfully executed</li>
	 * <li><strong>400 Bad Request</strong>: Invalid query structure, syntax errors, or invalid parameters</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to execute JSON queries or access hierarchical data.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON array containing instances that match the query structure</li>
	 * <li>Each instance includes its properties, relations, and metadata (if requested)</li>
	 * <li>Empty array is returned if no instances match the specified structure in the accessible contexts.</li>
	 * </ul>
	 * 
	 * <strong>Example Metadata (filtered for non-admin users):</strong>
	 * }`{@code json
	 * "metadata": {
	 *     "type": "Metadata",
	 *     "creationTime": "2025-03-18 17:13:40.952 +0100",
	 *     "createdBy": "HIDDEN_FOR_PRIVACY",
	 *     "lastUpdateBy": "HIDDEN_FOR_PRIVACY",
	 *     "lastUpdateTime": "2025-03-19 16:21:16.805 +0100"
	 * }
	 * }``
	 * 
	 * @param jsonQuery JSON object describing the structure to search for
	 * @return JSON array containing instances that match the query structure with their properties and relations
	 * @throws InvalidQueryException if the provided query structure is invalid or contains syntax errors
	 * @throws ResourceRegistryException if there is an error in the resource registry during query execution
	 */
	@POST
	@Path(AccessPath.QUERY_PATH_PART)
	@Operation(
		summary = "Execute JSON Query",
		description = """
		Executes a JSON-based query to find instances matching the specified structure in the Resource Registry.
		
		The response can include metadata, hierarchical querying, and context information based on query parameters and user authorization.


		**Request Examples:**
		- POST /access/query;
		- POST /access/query?limit=20&offset=40;
		- POST /access/query?polymorphic=true&includeMeta=true;
		- POST /access/query?includeMeta=true&allMeta=true;
		- POST /access/query?includeContexts=true;
		- POST /access/query?hierarchical=true;
		- POST /access/query?hierarchical=true&includeContexts=true.


		**Operation Behavior:**
		- Structure Matching: The JSON query describes the structure to match - instances must have the specified properties and relations;
		- Flexible Matching: Properties not specified in the query are ignored; only specified properties must match;
		- Relation Traversal: ConsistsOf and IsRelatedTo relations are traversed to find matching nested structures;
		- Type Polymorphism: When polymorphic=true, subtypes are included in both main results and nested relations;
		- Pagination Support: Results can be paginated using limit and offset parameters;
		- Context Scope: Queries execute within the current context unless hierarchical=true is specified by authorized users.


		**Authorization Requirements:**
		- **IS-Manager:**
			- Full access to execute any JSON query;
			- Can use hierarchical querying across all contexts;
			- Receives complete, unfiltered metadata including all administrative fields.

		- **Infrastructure-Manager:**
			- Full access to execute any JSON query;
			- Can use hierarchical querying across all contexts;
			- Receives complete, unfiltered metadata including all administrative fields.

		- **Context-Manager:**
			- Can execute JSON queries with hierarchical querying within their managed context hierarchy;
			- Must be Context-Manager of the current context (derived from the authorization token);
			- Receives complete, unfiltered metadata including all administrative fields.

		- **Other Users:**
			- Can execute JSON queries in the current context only;
			- Cannot use hierarchical querying (hierarchical parameter ignored);
			- Receive metadata with sensitive information filtered when includeMeta=true.
		"""
	)
	@RequestBody(
		description = """
		JSON object describing the structure to search for.
		- Must specify the target type and optionally its relations and properties;
		- Relations can include ConsistsOf, IsRelatedTo, and other defined relation types;
		- Properties can specify exact values or patterns to match.
		""",
		required = true,
		content = @Content(
			mediaType = "application/json",
			examples = {
				@ExampleObject(
					name = "Find EService with specific state and software",
					value = """
						{
							"type": "EService",
							"consistsOf": [
								{
									"type": "ConsistsOf",
									"propagationConstraint": {
										"add": "propagate"
									},
									"target": {
										"type": "StateFacet",
										"value": "down"
									}
								},
								{
									"type": "IsIdentifiedBy",
									"target": {
										"type": "SoftwareFacet",
										"name": "data-transfer-service",
										"group": "DataTransfer"
									}
								}
							]
						}
					"""
				)
			}
		)
	)
	@Parameter(
		name = AccessPath.POLYMORPHIC_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include instances of subtypes in the response and nested instances.
		- Default value: true (includes all subtypes);
		- Example: ?polymorphic=false (returns only exact type instances).
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "true", example = "true")
	)
	@Parameter(
		name = AccessPath.LIMIT_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Maximum number of instances to return in a single response.
		- Default value: 10;
		- Example: ?limit=50 (returns at most 50 instances).
		""",
		required = false,
		schema = @Schema(type = SchemaType.INTEGER, defaultValue = "10", example = "10")
	)
	@Parameter(
		name = AccessPath.OFFSET_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Number of instances to skip from the beginning of the result set.
		- Default value: 0 (starts from the first instance);
		- Example: ?offset=100 (skips the first 100 instances).
		""",
		required = false,
		schema = @Schema(type = SchemaType.INTEGER, defaultValue = "0", example = "0")
	)
	@Parameter(
		name = AccessPath.INCLUDE_META_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include metadata in the response instances.
		- Default value: false (basic information only);
		- Example: ?includeMeta=true (includes metadata with appropriate filtering based on user role);
		- Metadata includes only the requested type and subtypes (if polymorphic=true);
		- Restriction: IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@Parameter(
		name = AccessPath.INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.).
		- Must be used in conjunction with includeMeta=true;
		- Default value: false (metadata only for main instances, more human-readable);
		- Example: ?includeMeta=true&allMeta=true (includes metadata for all nested elements);
		- Purpose: When false, produces more human-readable responses with less JSON to process.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@Parameter(
		name = AccessPath.HIERARCHICAL_MODE_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include instances from child contexts of the current context.
		- Default value: false (current context only);
		- Restriction: Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles (Context-Manager must be of the current context derived from the authorization token);
		- Example: ?hierarchical=true (includes instances from current and child contexts);
		- Current context is determined from the authorization token.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@Parameter(
		name = AccessPath.INCLUDE_CONTEXTS_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include the list of contexts where each instance is available.
		- Default value: false (context information not included);
		- Example: ?includeContexts=true (shows context availability for each instance);
		- Note: A Resource is present in all contexts that form the union of contexts of all its Facets;
		- Note: ConsistsOf relations are present in all contexts where their target Facets are present.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)	
	@APIResponse(
		responseCode = "200",
		description = "JSON query successfully executed",
		content = @Content(mediaType = "application/json")
	)
	@APIResponse(
		responseCode = "400",
		description = "Invalid query structure, syntax errors, or invalid parameters"
	)
	@APIResponse(
		responseCode = "403",
		description = "Insufficient permissions to execute JSON queries or access hierarchical data"
	)
	public String jsonQuery(String jsonQuery) throws InvalidQueryException, ResourceRegistryException {
		logger.info("Requested json query \n{}", jsonQuery);
		setAccountingMethod(Method.QUERY, JSON_QUERY_METHOD);
		
		ServerRequestInfo serverRequestInfo = initRequestInfo();
		serverRequestInfo.checkAllBooleanQueryParameters();
		serverRequestInfo.checkLimitOffset();
		
		JsonQuery jsonQueryManager = new JsonQuery();
		jsonQueryManager.setJsonQuery(jsonQuery);
		return jsonQueryManager.query();
	}
	
	/**
	 * Executes a prepared query to find resources connected to specific entities through defined relations.
	 * This method provides a structured way to query the Resource Registry using path and query parameters.
	 * 
	 * <strong>REST Endpoint:</strong> {@code GET /access/query/{resource-type-name}/{relation-type-name}/{reference-type-name}}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>All EServices identified by a SoftwareFacet: {@code GET /access/query/EService/IsIdentifiedBy/SoftwareFacet?_polymorphic=true&_direction=out}</li>
	 * <li>All EServices identified by specific SoftwareFacet: {@code GET /access/query/EService/IsIdentifiedBy/SoftwareFacet?_reference=7bc997c3-d005-40ff-b9ed-c4b6a35851f1&_polymorphic=true&_direction=out}</li>
	 * <li>All Resources with a ContactFacet: {@code GET /access/query/Resource/ConsistsOf/ContactFacet?_polymorphic=true&_direction=out}</li>
	 * <li>All EServices hosted by HostingNodes: {@code GET /access/query/EService/Hosts/HostingNode?_polymorphic=true&_direction=in}</li>
	 * <li>All EServices hosted by specific HostingNode: {@code GET /access/query/EService/Hosts/HostingNode?_reference=16032d09-3823-444e-a1ff-a67de4f350a8&_polymorphic=true&_direction=in}</li>
	 * <li>With pagination: {@code GET /access/query/EService/IsIdentifiedBy/SoftwareFacet?_polymorphic=true&_direction=out&limit=20&offset=40}</li>
	 * <li>With metadata: {@code GET /access/query/EService/IsIdentifiedBy/SoftwareFacet?_polymorphic=true&_direction=out&includeMeta=true}</li>
	 * <li>Hierarchical (admin only): {@code GET /access/query/EService/IsIdentifiedBy/SoftwareFacet?_polymorphic=true&_direction=out&hierarchical=true}.</li>
	 * </ul>
	 * 
	 * <strong>Path Parameters:</strong>
	 * <ul>
	 * <li><strong>resource-type-name</strong>: The name of the Resource type to search for (e.g., "EService", "HostingNode", "Software")</li>
	 * <li><strong>relation-type-name</strong>: The name of the relation connecting resources to entities (e.g., "IsIdentifiedBy", "ConsistsOf", "Hosts")</li>
	 * <li><strong>reference-type-name</strong>: The name of the Resource/Facet type connected through the relation (e.g., 'SoftwareFacet', 'ContactFacet', 'EService', 'HostingNode').</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>_polymorphic</strong> (optional):
	 * Whether to include instances of subtypes in the response and nested relations.
	 * <ul>
	 * <li>Default value: false (exact type matching only)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?_polymorphic=true (includes all subtypes).</li>
	 * </ul>
	 * 
	 * <strong>_direction</strong> (optional):
	 * <ul>
	 * <li>Direction of the relation traversal from the resource perspective</li>
	 * <li>Default value: out (outgoing relations from resource)</li>
	 * <li>Values: out|in</li>
	 * <li>out: Resources having outgoing relations to the specified entity type (e.g., Resource ConsistsOf Facet)</li>
	 * <li>in: Resources having incoming relations from the specified entity type (e.g., EService hosted by HostingNode)</li>
	 * <li>Example: ?_direction=in (incoming relations).</li>
	 * </ul>
	 * 
	 * <strong>_reference</strong> (optional):
	 * <ul>
	 * <li>UUID of a specific entity instance to filter by</li>
	 * <li>When specified: only resources connected to that specific entity instance are returned</li>
	 * <li>When not specified: resources connected to any entity of the specified type are returned</li>
	 * <li>Must be a valid UUID format</li>
	 * <li>Example: ?_reference=7bc997c3-d005-40ff-b9ed-c4b6a35851f1.</li>
	 * </ul>
	 * 
	 * <strong>Additional Constraint Parameters</strong> (optional):
	 * <ul>
	 * <li>Any other query parameters are treated as property constraints on the connected entities</li>
	 * <li>Parameter names must match property names in the target entity type</li>
	 * <li>Only entities with matching property values will be considered</li>
	 * <li>Example: ?name=data-transfer-service&amp;group=DataTransfer (filters SoftwareFacet by name and group).</li>
	 * </ul>
	 * 
	 * <strong>Standard Query Parameters (same as JSON query):</strong>
	 * 
	 * <strong>limit</strong> (optional):
	 * <ul>
	 * <li>Maximum number of resources to return in a single response</li>
	 * <li>Default value: 10</li>
	 * <li>Example: ?limit=50 (returns at most 50 resources).</li>
	 * </ul>
	 * 
	 * <strong>offset</strong> (optional):
	 * <ul>
	 * <li>Number of resources to skip from the beginning of the result set</li>
	 * <li>Default value: 0 (starts from the first resource)</li>
	 * <li>Example: ?offset=100 (skips the first 100 resources).</li>
	 * </ul>
	 * 
	 * <strong>polymorphic</strong> (optional, different from _polymorphic):
	 * Whether to include instances of subtypes in nested relations within returned resources.
	 * <ul>
	 * <li>Default value: true (includes all subtypes in nested relations)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?polymorphic=false (exact type matching in nested relations).</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (optional):
	 * Whether to include metadata in the response resources.
	 * <ul>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeMeta=true (includes metadata with appropriate filtering based on user role)</li>
	 * <li>Metadata includes only the requested type and subtypes (if polymorphic=true)</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (optional):
	 * Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.).
	 * <ul>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main resources, more human-readable)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeMeta=true&amp;allMeta=true (includes metadata for all nested elements)</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (optional, admin-only):
	 * Whether to include resources from child contexts of the current context.
	 * <ul>
	 * <li>Default value: false (current context only)</li>
	 * <li>Values: true|false</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles (Context-Manager must be of the current context derived from the authorization token)</li>
	 * <li>Example: ?hierarchical=true (includes resources from current and child contexts).</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (optional):
	 * Whether to include the list of contexts where each resource is available.
	 * <ul>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeContexts=true (shows context availability for each resource).</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Full access to execute any prepared query</li>
	 * <li>Can use hierarchical querying across all contexts</li>
	 * <li>Receives complete, unfiltered metadata including all administrative fields</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Full access to execute any prepared query</li>
	 * <li>Can use hierarchical querying across all contexts</li>
	 * <li>Receives complete, unfiltered metadata including all administrative fields.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can execute prepared queries with hierarchical querying within their managed context hierarchy</li>
	 * <li>Must be Context-Manager of the current context (derived from the authorization token)</li>
	 * <li>Receives complete, unfiltered metadata including all administrative fields.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can execute prepared queries in the current context only</li>
	 * <li>Cannot use hierarchical querying (hierarchical parameter ignored)</li>
	 * <li>Receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li><strong>Prepared Query Pattern</strong>: Uses path parameters to define the query structure (resource-relation-entity pattern)</li>
	 * <li><strong>Relation Traversal</strong>: Follows the specified relation in the given direction to find connected resources</li>
	 * <li><strong>Constraint Application</strong>: Additional query parameters are applied as property filters on the target entities</li>
	 * <li><strong>Type Validation</strong>: Resource type must be a valid Resource type; relation and entity types must exist in the model</li>
	 * <li><strong>Reference Filtering</strong>: When _reference is specified, only resources connected to that specific entity instance are returned</li>
	 * <li><strong>Pagination Support</strong>: Results can be paginated using limit and offset parameters</li>
	 * <li><strong>Context Scope</strong>: Queries execute within the current context unless hierarchical=true is specified by authorized users.</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Prepared query successfully executed</li>
	 * <li><strong>400 Bad Request</strong>: Invalid resource type, relation type, entity type, UUID format, or query parameters</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to execute prepared queries or access hierarchical data</li>
	 * <li><strong>404 Not Found</strong>: Specified types do not exist in the model.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON array containing resources that match the prepared query criteria</li>
	 * <li>Each resource includes its properties, relations, and metadata (if requested)</li>
	 * <li>Empty array is returned if no resources match the specified criteria in the accessible contexts</li>
	 * <li>Resources are returned with their complete structure based on polymorphic and metadata settings.</li>
	 * </ul>
	 * 
	 * <strong>Example Metadata (filtered for non-admin users):</strong>
	 * ``{@code json
	 * "metadata": {
	 *     "type": "Metadata",
	 *     "creationTime": "2025-03-18 17:13:40.952 +0100",
	 *     "createdBy": "HIDDEN_FOR_PRIVACY",
	 *     "lastUpdateBy": "HIDDEN_FOR_PRIVACY",
	 *     "lastUpdateTime": "2025-03-19 16:21:16.805 +0100"
	 * }
	 * }``
	 * 
	 * @param resourceTypeName The name of the Resource type to search for
	 * @param relationTypeName The name of the relation connecting resources to entities
	 * @param referenceTypeName The name of the Entity/Facet type connected through the relation
	 * @param reference UUID of a specific entity instance to filter by (optional)
	 * @param polymorphic Whether to include subtypes in the main query (default: false)
	 * @param direction Direction of relation traversal (out|in, default: out)
	 * @param uriInfo URI information containing additional constraint parameters
	 * @return JSON array containing resources that match the prepared query criteria
	 * @throws ResourceRegistryException if there is an error during query execution, invalid types, or invalid parameters
	 */
	@SuppressWarnings({"rawtypes"})
	@GET
	@Path(AccessPath.QUERY_PATH_PART + "/" + "{" + Access.RESOURCE_TYPE_PATH_PARAMETER + "}" + "/" + "{"
			+ Access.RELATION_TYPE_PATH_PARAMETER + "}" + "/" + "{" + Access.REFERENCE_TYPE_PATH_PARAMETER + "}")
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	@Operation(
		summary = "Execute Prepared Query (Resource-Relation-Entity)",
		description = """
		Executes a prepared query to find resources connected to specific entities through defined relations.
		
		This method provides a structured way to query the Resource Registry using path and query parameters.


		**Request Examples:**
		- GET /access/query/EService/IsIdentifiedBy/SoftwareFacet?_polymorphic=true&_direction=out;
		- GET /access/query/EService/IsIdentifiedBy/SoftwareFacet?_reference=7bc997c3-d005-40ff-b9ed-c4b6a35851f1&_polymorphic=true&_direction=out;
		- GET /access/query/Resource/ConsistsOf/ContactFacet?_polymorphic=true&_direction=out;
		- GET /access/query/EService/Hosts/HostingNode?_polymorphic=true&_direction=in;
		- GET /access/query/EService/Hosts/HostingNode?_reference=16032d09-3823-444e-a1ff-a67de4f350a8&_polymorphic=true&_direction=in;
		- GET /access/query/EService/IsIdentifiedBy/SoftwareFacet?_polymorphic=true&_direction=out&limit=20&offset=40;
		- GET /access/query/EService/IsIdentifiedBy/SoftwareFacet?_polymorphic=true&_direction=out&includeMeta=true;
		- GET /access/query/EService/IsIdentifiedBy/SoftwareFacet?_polymorphic=true&_direction=out&hierarchical=true.


		**Operation Behavior:**
		- Prepared Query Pattern: Uses path parameters to define the query structure (resource-relation-entity pattern);
		- Relation Traversal: Follows the specified relation in the given direction to find connected resources;
		- Constraint Application: Additional query parameters are applied as property filters on the target entities;
		- Type Validation: Resource type must be a valid Resource type; relation and entity types must exist in the model;
		- Reference Filtering: When _reference is specified, only resources connected to that specific entity instance are returned;
		- Pagination Support: Results can be paginated using limit and offset parameters;
		- Context Scope: Queries execute within the current context unless hierarchical=true is specified by authorized users.

		**Authorization Requirements:**
		- **IS-Manager:**
			- Full access to execute any prepared query;
			- Can use hierarchical querying across all contexts;
			- Receives complete, unfiltered metadata including all administrative fields.

		- **Infrastructure-Manager:**
			- Full access to execute any prepared query;
			- Can use hierarchical querying across all contexts;
			- Receives complete, unfiltered metadata including all administrative fields.

		- **Context-Manager:**
			- Can execute prepared queries with hierarchical querying within their managed context hierarchy;
			- Must be Context-Manager of the current context (derived from the authorization token);
			- Receives complete, unfiltered metadata including all administrative fields.

		- **Other Users:**
			- Can execute prepared queries in the current context only;
			- Cannot use hierarchical querying (hierarchical parameter ignored);
			- Receive metadata with sensitive information filtered when includeMeta=true.
		"""
	)		
	@Parameter(
		name = AccessPath.LIMIT_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Maximum number of instances to return in a single response.
		- Default value: 10;
		- Example: ?limit=50 (returns at most 50 instances).
		""",
		required = false,
		schema = @Schema(type = SchemaType.INTEGER, defaultValue = "10", example = "10")
	)
	@Parameter(
		name = AccessPath.OFFSET_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Number of instances to skip from the beginning of the result set.
		- Default value: 0 (starts from the first instance);
		- Example: ?offset=100 (skips the first 100 instances).
		""",
		required = false,
		schema = @Schema(type = SchemaType.INTEGER, defaultValue = "0", example = "0")
	)
	@Parameter(
		name = AccessPath.INCLUDE_META_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include metadata in the response instances.
		- Default value: false (basic information only);
		- Example: ?includeMeta=true (includes metadata with appropriate filtering based on user role);
		- Metadata includes only the requested type and subtypes (if polymorphic=true);
		- Restriction: IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@Parameter(
		name = AccessPath.INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.).
		- Must be used in conjunction with includeMeta=true;
		- Default value: false (metadata only for main instances, more human-readable);
		- Example: ?includeMeta=true&allMeta=true (includes metadata for all nested elements);
		- Purpose: When false, produces more human-readable responses with less JSON to process.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@Parameter(
		name = AccessPath.HIERARCHICAL_MODE_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include instances from child contexts of the current context.
		- Default value: false (current context only);
		- Restriction: Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles (Context-Manager must be of the current context derived from the authorization token);
		- Example: ?hierarchical=true (includes instances from current and child contexts);
		- Current context is determined from the authorization token.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@Parameter(
		name = AccessPath.INCLUDE_CONTEXTS_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include the list of contexts where each instance is available.
		- Default value: false (context information not included);
		- Example: ?includeContexts=true (shows context availability for each instance);
		- Note: A Resource is present in all contexts that form the union of contexts of all its Facets;
		- Note: ConsistsOf relations are present in all contexts where their target Facets are present.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@APIResponse(
		responseCode = "200",
		description = "Prepared query successfully executed",
		content = @Content(mediaType = "application/json")
	)
	@APIResponse(
		responseCode = "400",
		description = "Invalid resource type, relation type, entity type, UUID format, or query parameters"
	)
	@APIResponse(
		responseCode = "403",
		description = "Insufficient permissions to execute prepared queries or access hierarchical data"
	)
	@APIResponse(
		responseCode = "404",
		description = "Specified types do not exist in the model"
	)
	public String getAllResourcesHavingFacet(
			@PathParam(Access.RESOURCE_TYPE_PATH_PARAMETER) 
			@Parameter(
				name = Access.RESOURCE_TYPE_PATH_PARAMETER,
				in = ParameterIn.PATH,
				description = "The name of the Resource type to search for (e.g., 'EService', 'HostingNode', 'Software')",
				required = true,
				schema = @Schema(type = SchemaType.STRING, example = "EService")
			)
			String resourceTypeName,
			@PathParam(Access.RELATION_TYPE_PATH_PARAMETER) 
			@Parameter(
				name = Access.RELATION_TYPE_PATH_PARAMETER,
				in = ParameterIn.PATH,
				description = "The name of the relation connecting resources to entities (e.g., 'IsIdentifiedBy', 'ConsistsOf', 'Hosts')",
				required = true,
				schema = @Schema(type = SchemaType.STRING, example = "IsIdentifiedBy")
			)
			String relationTypeName,
			@PathParam(Access.REFERENCE_TYPE_PATH_PARAMETER) 
			@Parameter(
				name = Access.REFERENCE_TYPE_PATH_PARAMETER,
				in = ParameterIn.PATH,
				description = "The name of the Resource/Facet type connected through the relation (e.g., 'SoftwareFacet', 'ContactFacet', 'EService', 'HostingNode')",
				required = true,
				schema = @Schema(type = SchemaType.STRING, example = "SoftwareFacet")
			)
			String referenceTypeName,
			@QueryParam(AccessPath._REFERENCE_QUERY_PARAMETER) 
			@Parameter(
				name = AccessPath._REFERENCE_QUERY_PARAMETER,
				in = ParameterIn.QUERY,
				description = """
				- UUID of a specific entity instance to filter by;
				- When specified: only resources connected to that specific entity instance are returned;
				- When not specified: resources connected to any entity of the specified type are returned;
				- Must be a valid UUID format;
				- Example: ?_reference=7bc997c3-d005-40ff-b9ed-c4b6a35851f1.
				""",
				required = false,
				schema = @Schema(type = SchemaType.STRING, example = "7bc997c3-d005-40ff-b9ed-c4b6a35851f1")
			)
			String reference,
			@QueryParam(AccessPath._POLYMORPHIC_QUERY_PARAMETER) @DefaultValue("false")
			@Parameter(
				name = AccessPath._POLYMORPHIC_QUERY_PARAMETER,
				in = ParameterIn.QUERY,
				description = """
				Whether to include instances of subtypes in the response and nested relations.
				- Default value: false (exact type matching only);
				- Example: ?_polymorphic=true (includes all subtypes).
				""",
				required = false,
				schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
			)
			Boolean polymorphic,
			@QueryParam(AccessPath._DIRECTION_QUERY_PARAMETER) @DefaultValue("out") 
			@Parameter(
				name = AccessPath._DIRECTION_QUERY_PARAMETER,
				in = ParameterIn.QUERY,
				description = """
				- Direction of the relation traversal from the resource perspective;
				- Default value: out (outgoing relations from resource);
				- Values: out|in;
				- out: Resources having outgoing relations to the specified entity type (e.g., Resource ConsistsOf Facet);
				- in: Resources having incoming relations from the specified entity type (e.g., EService hosted by HostingNode);
				- Example: ?_direction=in (incoming relations).
				""",
				required = false,
				schema = @Schema(type = SchemaType.STRING, defaultValue = "out", example = "in")
			)
			String direction,
			/*@QueryParam(AccessPath._INCLUDE_RELATION_PARAM) @DefaultValue("false") Boolean includeRelation,*/
			@Context UriInfo uriInfo) throws ResourceRegistryException {
		
		logger.info("Requested {} instances having a(n) {} ({}={}} with {} ({}={}). Request URI is {})", resourceTypeName, relationTypeName,
				AccessPath._DIRECTION_QUERY_PARAMETER, direction, referenceTypeName, AccessPath._POLYMORPHIC_QUERY_PARAMETER, polymorphic, uriInfo.getRequestUri());
		
		setAccountingMethod(Method.QUERY, PREPARED_QUERY_METHOD);
		
		ServerRequestInfo serverRequestInfo = initRequestInfo();
		serverRequestInfo.checkAllBooleanQueryParameters();
		serverRequestInfo.checkLimitOffset();
		
		ElementManagement erManagement = ERManagementUtility.getERManagement(resourceTypeName);
		
		if(erManagement instanceof ResourceManagement management) {
			UUID refereceUUID = null;
			ODirection directionEnum = ODirection.OUT;
			
			Map<String,String> constraint = new HashMap<>();
			
			MultivaluedMap<String,String> multivaluedMap = uriInfo.getQueryParameters();
			for(String key : multivaluedMap.keySet()) {
				switch (key) {
					case AccessPath._POLYMORPHIC_QUERY_PARAMETER:
						break;
						
					case AccessPath._DIRECTION_QUERY_PARAMETER:
						break;
					
					case AccessPath._REFERENCE_QUERY_PARAMETER:
						break;
					
					/*
					case AccessPath._INCLUDE_RELATION_PARAM:
						break;
					*/
					
					case "gcube-token":
						break;
						
					case "gcube-scope":
						break;
	
					default:
						constraint.put(key, multivaluedMap.getFirst(key));
						break;
				}
			}
			
			if(reference != null) {
				try {
					refereceUUID = UUID.fromString(reference);
				} catch(Exception e) {
					String error = "%s is not a valid %s".formatted(reference, UUID.class.getSimpleName());
					throw new InvalidQueryException(error);
				}
			}
			try {
				directionEnum = ODirection.valueOf(direction.toUpperCase());
			} catch(Exception e) {
				String error = "%s is not a valid. Allowed values are %s".formatted(direction, ODirection.values());
				throw new InvalidQueryException(error);
			}
			
			return management.query(relationTypeName, referenceTypeName, refereceUUID, directionEnum,
					polymorphic, constraint); /*, includeRelation*/
		}
		
		String error = "%s is not a %s type".formatted(resourceTypeName, Resource.NAME);
		throw new InvalidQueryException(error);
	}
	
}