package org.gcube.informationsystem.resourceregistry.rest;

import org.eclipse.microprofile.openapi.annotations.Operation;
import org.eclipse.microprofile.openapi.annotations.enums.ParameterIn;
import org.eclipse.microprofile.openapi.annotations.enums.SchemaType;
import org.eclipse.microprofile.openapi.annotations.media.Content;
import org.eclipse.microprofile.openapi.annotations.media.Schema;
import org.eclipse.microprofile.openapi.annotations.parameters.Parameter;
import org.eclipse.microprofile.openapi.annotations.responses.APIResponse;
import org.eclipse.microprofile.openapi.annotations.tags.Tag;
import org.gcube.informationsystem.queries.templates.reference.entities.QueryTemplate;
import org.gcube.informationsystem.resourceregistry.ResourceInitializer;
import org.gcube.informationsystem.resourceregistry.api.exceptions.NotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.queries.InvalidQueryException;
import org.gcube.informationsystem.resourceregistry.api.rest.ContextPath;
import org.gcube.informationsystem.resourceregistry.api.rest.InstancePath;
import org.gcube.informationsystem.resourceregistry.api.rest.QueryTemplatePath;
import org.gcube.informationsystem.resourceregistry.queries.templates.QueryTemplateManagement;
import org.gcube.informationsystem.resourceregistry.rest.requests.ServerRequestInfo;

import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.DELETE;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.POST;
import jakarta.ws.rs.PUT;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.PathParam;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.core.Response.Status;

/**
 * @author Luca Frosini (ISTI - CNR)
 */
@Path(QueryTemplatePath.QUERY_TEMPLATES_PATH_PART)
@Tag(name = "Query Templates", description = "Operations for managing query templates in the Resource Registry.")
public class QueryTemplateManager extends BaseRest {

	public static final String QUERY_TEMPLATE_NAME_PATH_PARAMETER = "query-template-name";
	
	public QueryTemplateManager() {
		super();
	}
	
	/**
	 * Retrieves all query templates from the Resource Registry.
	 * 
	 * <strong>REST Endpoint:</strong> {@code GET /query-templates}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>Basic query: {@code GET /query-templates}</li>
	 * <li>With pagination: {@code GET /query-templates?limit=20&offset=40}</li>
	 * <li>With metadata: {@code GET /query-templates?includeMeta=true}</li>
	 * <li>Combined parameters: {@code GET /query-templates?limit=15&offset=30&includeMeta=true}.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>limit</strong> (optional):
	 * <ul>
	 * <li>Maximum number of query templates to return in a single response</li>
	 * <li>Default value: 10</li>
	 * <li>Example: ?limit=50 (returns at most 50 query templates).</li>
	 * </ul>
	 * 
	 * <strong>offset</strong> (optional):
	 * <ul>
	 * <li>Number of query templates to skip from the beginning of the result set</li>
	 * <li>Default value: 0 (starts from the first query template)</li>
	 * <li>Example: ?offset=100 (skips the first 100 query templates).</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (optional):
	 * <ul>
	 * <li>Whether to include metadata in the response query templates</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeMeta=true (includes metadata with appropriate filtering based on user role)</li>
	 * <li><strong>Restriction:</strong> IS-Manager and Infrastructure-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Full access to all query templates</li>
	 * <li>Can access query templates across all contexts</li>
	 * <li>Receives complete, unfiltered metadata including all administrative fields.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Full access to all query templates</li>
	 * <li>Can access query templates across all contexts</li>
	 * <li>Receives complete, unfiltered metadata including all administrative fields.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Basic access with metadata filtering</li>
	 * <li>Receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Query templates successfully retrieved</li>
	 * <li><strong>400 Bad Request</strong>: Invalid query parameters (e.g., invalid limit/offset values)</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to access query templates.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON array containing the requested query templates</li>
	 * <li>Each query template includes its name, description, template definition, template variables, and metadata (if requested)</li>
	 * <li>Empty array is returned if no query templates exist in the accessible contexts.</li>
	 * </ul>
	 * 
	 * <strong>Example Metadata (filtered for non-admin users):</strong>
	 * ``{@code json
	 * "metadata": {
	 *     "type": "Metadata",
	 *     "creationTime": "2025-03-18 17:13:40.952 +0100",
	 *     "createdBy": "HIDDEN_FOR_PRIVACY",
	 *     "lastUpdateBy": "HIDDEN_FOR_PRIVACY",
	 *     "lastUpdateTime": "2025-03-19 16:21:16.805 +0100"
	 * }
	 * }``
	 * 
	 * @return JSON array containing the requested query templates with their properties and metadata (if requested)
	 * @throws NotFoundException If query templates cannot be retrieved
	 * @throws ResourceRegistryException If an error occurs during query template retrieval
	 */
	@GET
	@Consumes({MediaType.TEXT_PLAIN, ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8})
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	@Operation(
		summary = "List Query Templates",
		description = """
		Retrieves all query templates from the Resource Registry.

		The response can include metadata and provides pagination based on query parameters and user authorization.


		**Request Examples:**
		- GET /query-templates;
		- GET /query-templates?limit=20&offset=40;
		- GET /query-templates?includeMeta=true;
		- GET /query-templates?limit=15&offset=30&includeMeta=true.


		**Authorization Requirements:**
		- **IS-Manager:**
			- Full access to all query templates;
			- Can access query templates across all contexts;
			- Receives complete, unfiltered metadata including all administrative fields.

		- **Infrastructure-Manager:**
			- Full access to all query templates;
			- Can access query templates across all contexts;
			- Receives complete, unfiltered metadata including all administrative fields.

		- **Other Users:**
			- Basic access with metadata filtering;
			- Receive metadata with sensitive information filtered when includeMeta=true.
		"""
	)
	@Parameter(
		name = QueryTemplatePath.LIMIT_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Maximum number of query templates to return in a single response.
		- Default: 10;
		- Example: ?limit=50 (returns at most 50 query templates).
		""",
		required = false,
		schema = @Schema(type = SchemaType.INTEGER, defaultValue = "10", example = "10")
	)
	@Parameter(
		name = QueryTemplatePath.OFFSET_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Number of query templates to skip from the beginning of the result set.
		- Default: 0 (starts from the first query template);
		- Example: ?offset=100 (skips the first 100 query templates).
		""",
		required = false,
		schema = @Schema(type = SchemaType.INTEGER, defaultValue = "0", example = "0")
	)
	@Parameter(
		name = QueryTemplatePath.INCLUDE_META_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include metadata in the response query templates.
		- Default: false (basic information only);
		- Example: ?includeMeta=true (includes metadata with appropriate filtering based on user role);
		- Restriction: IS-Manager and Infrastructure-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@APIResponse(
		responseCode = "200",
		description = "Query templates successfully retrieved",
		content = @Content(mediaType = "application/json")
	)
	@APIResponse(
		responseCode = "400",
		description = "Invalid query parameters (e.g., invalid limit/offset values)"
	)
	@APIResponse(
		responseCode = "403",
		description = "Insufficient permissions to access query templates"
	)
	public String all() throws NotFoundException, ResourceRegistryException {
		logger.info("Requested to read all {}s", QueryTemplate.NAME);
		setAccountingMethod(Method.LIST, QueryTemplate.NAME);
		
		ServerRequestInfo serverRequestInfo = initRequestInfo();
		serverRequestInfo.setAllMeta(true);
		serverRequestInfo.checkBooleanQueryParameter(ContextPath.INCLUDE_META_QUERY_PARAMETER);
		serverRequestInfo.checkLimitOffset();
		
		QueryTemplateManagement queryTemplateManagement = new QueryTemplateManagement();
		return queryTemplateManagement.all(false);
	}
	
	/**
	 * Creates a new query template or updates an existing query template with the specified name in the Resource Registry.
	 * 
	 * <strong>REST Endpoint:</strong> {@code PUT /query-templates/{query-template-name}}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>Basic create/update: {@code PUT /query-templates/GetAllEServiceWithState}</li>
	 * <li>With metadata in response: {@code PUT /query-templates/GetAllEServiceWithState?includeMeta=true}.</li>
	 * </ul>
	 * 
	 * <strong>Path Parameters:</strong>
	 * <ul>
	 * <li><strong>query-template-name</strong>: The name of the query template to create or update (e.g., "GetAllEServiceWithState")</li>
	 * <li><strong>Important</strong>: This name must exactly match the "name" field in the JSON body, otherwise a 400 Bad Request error will be returned.</li>
	 * </ul>
	 * 
	 * <strong>Request Body:</strong>
	 * <ul>
	 * <li>JSON representation of the query template to create or update</li>
	 * <li>The "name" field in the JSON must match the <strong>query-template-name</strong> path parameter.</li>
	 * </ul>
	 * 
	 * <strong>Example Request Body:</strong>
	 * ``{@code json
	 * {
	 * 		"name" : "GetAllEServiceWithState",
	 * 		"description" : "The following query return all the EService having the state provided as parameters, e.g. down, ready. 
	 * 					The content of the request to run this query template will be something like {\"$state\": "ready"}",
	 * 		"template": {
	 * 			"type": "EService",
	 * 			"consistsOf": [{
	 * 				"type": "ConsistsOf",
	 * 				"target": {
	 * 					"type": "StateFacet",
	 * 					"value": "$state"
	 * 				}
	 * 			}]
	 * 		},
	 * 		"templateVariables" = {
	 * 			"$state" : {
	 * 				"name": "$state",
	 * 				"description": "The state of the EService, e.g. down, ready.",
	 * 				"defaultValue": "ready"
	 * 			}
	 * 		}
	 * }
	 * }`{@code 
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (optional):
	 * <ul>
	 * <li>Whether to include metadata in the response query template</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeMeta=true (includes metadata with appropriate filtering based on user role)</li>
	 * <li><strong>Restriction:</strong> IS-Manager and Infrastructure-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Full access to create/update any query template</li>
	 * <li>Can create/update query templates across all contexts</li>
	 * <li>Receives complete, unfiltered metadata including all administrative fields in the response.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Full access to create/update any query template</li>
	 * <li>Can create/update query templates across all contexts</li>
	 * <li>Receives complete, unfiltered metadata including all administrative fields in the response.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Limited create/update permissions based on context access</li>
	 * <li>Receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li><strong>Create</strong>: If the query template with the specified name does not exist, creates a new query template</li>
	 * <li><strong>Update</strong>: If the query template with the specified name exists, updates the existing query template</li>
	 * <li><strong>Name Validation</strong>: The path parameter <strong>query-template-name</strong> must exactly match the "name" field in the JSON body</li>
	 * <li><strong>Query Validation</strong>: The template query is automatically tested during creation/update</li>
	 * <li><strong>Parameter Declaration</strong>: Every parameter used in the query (prefixed with $) must have a corresponding declaration in templateVariables</li>
	 * <li><strong>Default Value Substitution</strong>: During validation, parameters are replaced with their default values before executing the query</li>
	 * <li><strong>Valid Default Values</strong>: Default values must be valid for the query execution - the query must be able to run on the system</li>
	 * <li><strong>Result Independence</strong>: The query doesn't need to return results during validation (empty results are acceptable), but it must execute without errors</li>
	 * <li>The operation is idempotent and atomic.</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Query template successfully updated (existing template modified)</li>
	 * <li><strong>201 Created</strong>: Query template successfully created (new template)</li>
	 * <li><strong>400 Bad Request</strong>: Invalid request body, malformed JSON, or path parameter name does not match JSON name field</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to create/update the query template.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON object containing the created or updated query template</li>
	 * <li>Includes name, description, template definition, template variables, and metadata (if requested).</li>
	 * </ul>
	 * 
	 * <strong>Example Metadata (filtered for non-admin users):</strong>
	 * }`{@code json
	 * "metadata": {
	 *     "type": "Metadata",
	 *     "creationTime": "2025-03-18 17:13:40.952 +0100",
	 *     "createdBy": "HIDDEN_FOR_PRIVACY",
	 *     "lastUpdateBy": "HIDDEN_FOR_PRIVACY",
	 *     "lastUpdateTime": "2025-03-19 16:21:16.805 +0100"
	 * }
	 * }``
	 * 
	 * @param queryTemplateName The name of the query template to create or update (must match the "name" field in the JSON body)
	 * @param json JSON representation of the query template to create or update
	 * @return JSON object containing the created or updated query template with its properties and metadata (if requested)
	 * @throws InvalidQueryException If the query template definition is invalid
	 * @throws ResourceRegistryException If an error occurs during query template creation or update
	 */
	@PUT
	@Path("{" + QueryTemplateManager.QUERY_TEMPLATE_NAME_PATH_PARAMETER + "}")
	@Consumes({MediaType.TEXT_PLAIN, ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8})
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	@Operation(
		summary = "Create or Update Query Template",
		description = """
		Creates a new query template or updates an existing query template with the specified name in the Resource Registry.


		**Request Examples:**
		- PUT /query-templates/GetAllEServiceWithState;
		- PUT /query-templates/GetAllEServiceWithState?includeMeta=true.


		**Authorization Requirements:**
		- **IS-Manager:**
			- Full access to create/update any query template;
			- Can create/update query templates across all contexts;
			- Receives complete, unfiltered metadata including all administrative fields in the response.

		- **Infrastructure-Manager:**
			- Full access to create/update any query template;
			- Can create/update query templates across all contexts;
			- Receives complete, unfiltered metadata including all administrative fields in the response.

		- **Other Users:**
			- Limited create/update permissions based on context access;
			- Receive metadata with sensitive information filtered when includeMeta=true.


		**Operation Behavior:**
		- **Create**: If the query template with the specified name does not exist, creates a new query template;
		- **Update**: If the query template with the specified name exists, updates the existing query template;
		- **Name Validation**: The path parameter **query-template-name** must exactly match the "name" field in the JSON body;
		- **Query Validation**: The template query is automatically tested during creation/update;
		- **Parameter Declaration**: Every parameter used in the query (prefixed with $) must have a corresponding declaration in templateVariables;
		- **Default Value Substitution**: During validation, parameters are replaced with their default values before executing the query;
		- **Valid Default Values**: Default values must be valid for the query execution - the query must be able to run on the system;
		- **Result Independence**: The query doesn't need to return results during validation (empty results are acceptable), but it must execute without errors;
		- The operation is idempotent and atomic.
		"""
	)
	@Parameter(
		name = QueryTemplatePath.INCLUDE_META_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include metadata in the response query template.
		- Default: false (basic information only);
		- Values: true|false;
		- Example: ?includeMeta=true (includes metadata with appropriate filtering based on user role);
		- Restriction: IS-Manager and Infrastructure-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@APIResponse(
		responseCode = "200",
		description = "Query template successfully updated (existing template modified)",
		content = @Content(mediaType = "application/json")
	)
	@APIResponse(
		responseCode = "201",
		description = "Query template successfully created (new template)",
		content = @Content(mediaType = "application/json")
	)
	@APIResponse(
		responseCode = "400",
		description = "Invalid request body, malformed JSON, or path parameter name does not match JSON name field"
	)
	@APIResponse(
		responseCode = "403",
		description = "Insufficient permissions to create/update the query template"
	)
	public String updateCreate(
			@PathParam(QueryTemplateManager.QUERY_TEMPLATE_NAME_PATH_PARAMETER)
			@Parameter(
				name = QueryTemplateManager.QUERY_TEMPLATE_NAME_PATH_PARAMETER,
				in = ParameterIn.PATH,
				description = """
				The name of the query template to create or update.
				- Must match the "name" field in the JSON body (e.g., "GetAllEServiceWithState");
				- Important: This name must exactly match the "name" field in the JSON body, otherwise a 400 Bad Request error will be returned.
				""",
				required = true,
				schema = @Schema(type = SchemaType.STRING, example = "GetAllEServiceWithState")
			)
			String queryTemplateName, String json)
			throws InvalidQueryException, ResourceRegistryException {
		logger.info("Requested {} creation with name {} and content {}", QueryTemplate.NAME, queryTemplateName, json);
		setAccountingMethod(Method.UPDATE, QueryTemplate.NAME);
		
		ServerRequestInfo serverRequestInfo = initRequestInfo();
		serverRequestInfo.setAllMeta(true);
		serverRequestInfo.checkBooleanQueryParameter(ContextPath.INCLUDE_META_QUERY_PARAMETER);
		
		QueryTemplateManagement queryTemplateManagement = new QueryTemplateManagement();
		queryTemplateManagement.setName(queryTemplateName);
		queryTemplateManagement.setJson(json);
		return queryTemplateManagement.createOrUpdate();
	}
	
	/**
	 * Retrieves a specific query template by its name from the Resource Registry.
	 * 
	 * <strong>REST Endpoint:</strong> {@code GET /query-templates/{query-template-name}}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>Basic query: {@code GET /query-templates/GetAllEServiceWithState}</li>
	 * <li>With metadata: {@code GET /query-templates/GetAllEServiceWithState?includeMeta=true}.</li>
	 * </ul>
	 * 
	 * <strong>Path Parameters:</strong>
	 * <ul>
	 * <li><strong>query-template-name</strong>: The name of the query template to retrieve (e.g., "GetAllEServiceWithState").</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (optional):
	 * <ul>
	 * <li>Whether to include metadata in the response query template</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeMeta=true (includes metadata with appropriate filtering based on user role)</li>
	 * <li><strong>Restriction:</strong> IS-Manager and Infrastructure-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Full access to retrieve any query template</li>
	 * <li>Can access query templates across all contexts</li>
	 * <li>Receives complete, unfiltered metadata including all administrative fields.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Full access to retrieve any query template</li>
	 * <li>Can access query templates across all contexts</li>
	 * <li>Receives complete, unfiltered metadata including all administrative fields.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Basic access with metadata filtering</li>
	 * <li>Receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Query template successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: Query template with the specified name does not exist</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to access the query template.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON object containing the requested query template</li>
	 * <li>Includes name, description, template definition, template variables, and metadata (if requested).</li>
	 * </ul>
	 * 
	 * <strong>Example Metadata (filtered for non-admin users):</strong>
	 * ``{@code json
	 * "metadata": {
	 *     "type": "Metadata",
	 *     "creationTime": "2025-03-18 17:13:40.952 +0100",
	 *     "createdBy": "HIDDEN_FOR_PRIVACY",
	 *     "lastUpdateBy": "HIDDEN_FOR_PRIVACY",
	 *     "lastUpdateTime": "2025-03-19 16:21:16.805 +0100"
	 * }
	 * }``
	 * 
	 * @param queryTemplateName The name of the query template to retrieve
	 * @return JSON object containing the requested query template with its properties and metadata (if requested)
	 * @throws NotFoundException If the query template with the specified name does not exist
	 * @throws ResourceRegistryException If an error occurs during query template retrieval
	 */
	@GET
	@Path("{" + QueryTemplateManager.QUERY_TEMPLATE_NAME_PATH_PARAMETER + "}")
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	@Operation(
		summary = "Read Query Template",
		description = """
		Retrieves a specific query template by its name from the Resource Registry.

		The response includes metadata based on query parameters and user authorization.


		**Request Examples:**
		- GET /query-templates/GetAllEServiceWithState;
		- GET /query-templates/GetAllEServiceWithState?includeMeta=true.


		**Authorization Requirements:**
		- **IS-Manager:**
			- Full access to retrieve any query template;
			- Can access query templates across all contexts;
			- Receives complete, unfiltered metadata including all administrative fields.

		- **Infrastructure-Manager:**
			- Full access to retrieve any query template;
			- Can access query templates across all contexts;
			- Receives complete, unfiltered metadata including all administrative fields.

		- **Other Users:**
			- Basic access with metadata filtering;
			- Receive metadata with sensitive information filtered when includeMeta=true.
		"""
	)
	@Parameter(
		name = QueryTemplatePath.INCLUDE_META_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include metadata in the response query template.
		- Default: false (basic information only);
		- Example: ?includeMeta=true (includes metadata with appropriate filtering based on user role);
		- Restriction: IS-Manager and Infrastructure-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@APIResponse(
		responseCode = "200",
		description = "Query template successfully retrieved",
		content = @Content(mediaType = "application/json")
	)
	@APIResponse(
		responseCode = "404",
		description = "Query template with the specified name does not exist"
	)
	@APIResponse(
		responseCode = "403",
		description = "Insufficient permissions to access the query template"
	)
	public String read(
			@PathParam(QueryTemplateManager.QUERY_TEMPLATE_NAME_PATH_PARAMETER)
			@Parameter(
				name = QueryTemplateManager.QUERY_TEMPLATE_NAME_PATH_PARAMETER,
				in = ParameterIn.PATH,
				description = "The name of the query template to retrieve (e.g., \"GetAllEServiceWithState\")",
				required = true,
				schema = @Schema(type = SchemaType.STRING, example = "GetAllEServiceWithState")
			)
			String queryTemplateName)
			throws NotFoundException, ResourceRegistryException {
		logger.info("Requested {} with name", QueryTemplate.NAME, queryTemplateName);
		setAccountingMethod(Method.READ, QueryTemplate.NAME);
		
		ServerRequestInfo serverRequestInfo = initRequestInfo();
		serverRequestInfo.setAllMeta(true);
		serverRequestInfo.checkBooleanQueryParameter(ContextPath.INCLUDE_META_QUERY_PARAMETER);
		
		QueryTemplateManagement queryTemplateManagement = new QueryTemplateManagement();
		queryTemplateManagement.setName(queryTemplateName);
		return queryTemplateManagement.read();
	}
	
	
	/**
	 * Executes a query template with the specified name and parameters in the Resource Registry.
	 * 
	 * <strong>REST Endpoint:</strong> {@code POST /query-templates/{query-template-name}}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>Basic execution: {@code POST /query-templates/GetAllEServiceWithState}</li>
	 * <li>With polymorphic disabled: {@code POST /query-templates/GetAllEServiceWithState?polymorphic=false}</li>
	 * <li>With metadata: {@code POST /query-templates/GetAllEServiceWithState?includeMeta=true}</li>
	 * <li>Hierarchical (admin only): {@code POST /query-templates/GetAllEServiceWithState?hierarchical=true}</li>
	 * <li>Full metadata context: {@code POST /query-templates/GetAllEServiceWithState?includeMeta=true&allMeta=true}</li>
	 * <li>With context information: {@code POST /query-templates/GetAllEServiceWithState?includeContexts=true}.</li>
	 * </ul>
	 * 
	 * <strong>Path Parameters:</strong>
	 * <ul>
	 * <li><strong>query-template-name</strong>: The name of the query template to execute (e.g., "GetAllEServiceWithState").</li>
	 * </ul>
	 * 
	 * <strong>Request Body:</strong>
	 * <ul>
	 * <li>JSON object containing parameter values to substitute in the query template</li>
	 * <li>Parameters not provided in the body will use their default values from the template definition</li>
	 * <li>Parameter names must match those defined in the template's templateVariables section.</li>
	 * </ul>
	 * 
	 * <strong>Example Request Body for GetAllEServiceWithState:</strong>
	 * ``{@code json
	 * {
	 *     "$state" : "down"
	 * }
	 * }`{@code 
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>polymorphic</strong> (optional):
	 * <ul>
	 * <li>Whether the query should consider instances polymorphically (include subtypes)</li>
	 * <li>Default value: true (includes all subtypes)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?polymorphic=false (exact type match only, may slow down the query)</li>
	 * <li>When false, only instances of the exact types defined in the query are returned</li>
	 * <li>When true, instances of derived types are also included.</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (optional):
	 * <ul>
	 * <li>Whether to include metadata in the response instances</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeMeta=true (includes metadata with appropriate filtering based on user role)</li>
	 * <li><strong>Effect</strong>: Only applies when query returns complete instances of defined types</li>
	 * <li><strong>No Effect</strong>: When query uses _emit to return individual fields instead of complete instances</li>
	 * <li><strong>Restriction:</strong> IS-Manager and Infrastructure-Manager see complete metadata; other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (optional):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.)</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instances, more human-readable)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeMeta=true&allMeta=true (includes metadata for all nested elements)</li>
	 * <li><strong>Effect</strong>: Only applies when query returns complete instances of defined types</li>
	 * <li><strong>No Effect</strong>: When query uses _emit to return individual fields instead of complete instances.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (optional, admin-only):
	 * <ul>
	 * <li>Whether to execute the query across child contexts of the current context</li>
	 * <li>Default value: false (current context only)</li>
	 * <li>Values: true|false</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles (Context-Manager must be of the current context derived from the authorization token)</li>
	 * <li>Example: ?hierarchical=true (executes query in current and child contexts)</li>
	 * <li>Current context is determined from the authorization token.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (optional):
	 * <ul>
	 * <li>Whether to include the list of contexts where each returned instance is available</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeContexts=true (shows context availability for each instance)</li>
	 * <li><strong>Effect</strong>: Only applies when query returns complete instances of defined types</li>
	 * <li><strong>No Effect</strong>: When query uses _emit to return individual fields instead of complete instances</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Full access to execute any query template</li>
	 * <li>Can use hierarchical querying across all contexts</li>
	 * <li>Receives complete, unfiltered metadata including all administrative fields.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Full access to execute any query template</li>
	 * <li>Can use hierarchical querying across all contexts</li>
	 * <li>Receives complete, unfiltered metadata including all administrative fields.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can execute query templates with hierarchical querying within their managed context hierarchy</li>
	 * <li>Must be Context-Manager of the current context (derived from the authorization token)</li>
	 * <li>Receives complete, unfiltered metadata including all administrative fields.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can execute query templates in the current context only</li>
	 * <li>Cannot use hierarchical querying (hierarchical parameter ignored)</li>
	 * <li>Receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li><strong>Parameter Substitution</strong>: Parameters in the template query (prefixed with $) are replaced with values from the request body</li>
	 * <li><strong>Default Values</strong>: Parameters not provided in the request body use their default values from the template definition</li>
	 * <li><strong>Query Execution</strong>: The resulting query is executed against the Resource Registry</li>
	 * <li><strong>Response Type</strong>: Can return complete instances or individual fields (when using _emit in the query)</li>
	 * <li><strong>Metadata Effect</strong>: includeMeta, allMeta, and includeContexts only affect queries returning complete instances, not _emit queries</li>
	 * <li><strong>Polymorphic Behavior</strong>: When disabled (polymorphic=false), may impact query performance but ensures exact type matching</li>
	 * <li><strong>No Pagination</strong>: Pagination is not currently supported for query template execution</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Query template successfully executed</li>
	 * <li><strong>400 Bad Request</strong>: Invalid query parameters or malformed parameter values</li>
	 * <li><strong>404 Not Found</strong>: Query template with the specified name does not exist</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to execute the query template.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns query results based on the template definition</li>
	 * <li>For complete instances: JSON array containing instances with properties, relations, and metadata (if requested)</li>
	 * <li>For _emit queries: JSON array containing individual field values as specified in the query</li>
	 * <li>Response structure depends on the query template and parameter values provided.</li>
	 * </ul>
	 * 
	 * <strong>Example Metadata (filtered for non-admin users, only for complete instances):</strong>
	 * }`{@code json
	 * "metadata": {
	 *     "type": "Metadata",
	 *     "creationTime": "2025-03-18 17:13:40.952 +0100",
	 *     "createdBy": "HIDDEN_FOR_PRIVACY",
	 *     "lastUpdateBy": "HIDDEN_FOR_PRIVACY",
	 *     "lastUpdateTime": "2025-03-19 16:21:16.805 +0100"
	 * }
	 * }``
	 * 
	 * @param queryTemplateName The name of the query template to execute
	 * @param params JSON object containing parameter values for template substitution
	 * @return JSON array containing query results based on the template definition and parameters
	 * @throws NotFoundException If the query template with the specified name does not exist
	 * @throws InvalidQueryException If the query execution fails due to invalid parameters or query structure
	 * @throws ResourceRegistryException If an error occurs during query template execution
	 */
	@POST
	@Path("{" + QueryTemplateManager.QUERY_TEMPLATE_NAME_PATH_PARAMETER + "}")
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	@Operation(
		summary = "Execute Query Template",
		description = """
		Executes a query template with the specified name and parameters in the Resource Registry.


		**Request Examples:**
		- POST /query-templates/GetAllEServiceWithState;
		- POST /query-templates/GetAllEServiceWithState?polymorphic=false;
		- POST /query-templates/GetAllEServiceWithState?includeMeta=true;
		- POST /query-templates/GetAllEServiceWithState?hierarchical=true;
		- POST /query-templates/GetAllEServiceWithState?includeMeta=true&allMeta=true;
		- POST /query-templates/GetAllEServiceWithState?includeContexts=true.


		**Authorization Requirements:**
		- **IS-Manager:**
			- Full access to execute any query template;
			- Can use hierarchical querying across all contexts;
			- Receives complete, unfiltered metadata including all administrative fields.

		- **Infrastructure-Manager:**
			- Full access to execute any query template;
			- Can use hierarchical querying across all contexts;
			- Receives complete, unfiltered metadata including all administrative fields.

		- **Context-Manager:**
			- Can execute query templates with hierarchical querying within their managed context hierarchy;
			- Must be Context-Manager of the current context (derived from the authorization token);
			- Receives complete, unfiltered metadata including all administrative fields.

		- **Other Users:**
			- Can execute query templates in the current context only;
			- Cannot use hierarchical querying (hierarchical parameter ignored);
			- Receive metadata with sensitive information filtered when includeMeta=true.


		**Operation Behavior:**
		- **Parameter Substitution**: Parameters in the template query (prefixed with $) are replaced with values from the request body;
		- **Default Values**: Parameters not provided in the request body use their default values from the template definition;
		- **Query Execution**: The resulting query is executed against the Resource Registry;
		- **Response Type**: Can return complete instances or individual fields (when using _emit in the query);
		- **Metadata Effect**: includeMeta, allMeta, and includeContexts only affect queries returning complete instances, not _emit queries;
		- **Polymorphic Behavior**: When disabled (polymorphic=false), may impact query performance but ensures exact type matching;
		- **No Pagination**: Pagination is not currently supported for query template execution.
		"""
	)
	@Parameter(
		name = QueryTemplatePath.POLYMORPHIC_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether the query should consider instances polymorphically (include subtypes).
		- Default: true (includes all subtypes);
		- Values: true|false;
		- Example: ?polymorphic=false (exact type match only, may slow down the query);
		- When false, only instances of the exact types defined in the query are returned;
		- When true, instances of derived types are also included.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "true", example = "true")
	)
	@Parameter(
		name = QueryTemplatePath.INCLUDE_META_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include metadata in the response instances.
		- Default: false (basic information only);
		- Values: true|false;
		- Example: ?includeMeta=true (includes metadata with appropriate filtering based on user role);
		- Effect: Only applies when query returns complete instances of defined types;
		- No Effect: When query uses _emit to return individual fields instead of complete instances;
		- Restriction: IS-Manager and Infrastructure-Manager see complete metadata; other users see filtered metadata with sensitive fields obfuscated.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@Parameter(
		name = InstancePath.INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.).
		- Must be used in conjunction with includeMeta=true;
		- Default: false (metadata only for main instances, more human-readable);
		- Values: true|false;
		- Example: ?includeMeta=true&allMeta=true (includes metadata for all nested elements);
		- Effect: Only applies when query returns complete instances of defined types;
		- No Effect: When query uses _emit to return individual fields instead of complete instances.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@Parameter(
		name = InstancePath.HIERARCHICAL_MODE_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to execute the query across child contexts of the current context.
		- Default: false (current context only);
		- Values: true|false;
		- Restriction: Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles (Context-Manager must be of the current context derived from the authorization token);
		- Example: ?hierarchical=true (executes query in current and child contexts);
		- Current context is determined from the authorization token.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@Parameter(
		name = QueryTemplatePath.INCLUDE_CONTEXTS_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include the list of contexts where each returned instance is available.
		- Default: false (context information not included);
		- Values: true|false;
		- Example: ?includeContexts=true (shows context availability for each instance);
		- Effect: Only applies when query returns complete instances of defined types;
		- No Effect: When query uses _emit to return individual fields instead of complete instances;
		- Note: A Resource is present in all contexts that form the union of contexts of all its Facets.
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@APIResponse(
		responseCode = "200",
		description = "Query template successfully executed",
		content = @Content(mediaType = "application/json")
	)
	@APIResponse(
		responseCode = "400",
		description = "Invalid query parameters or malformed parameter values"
	)
	@APIResponse(
		responseCode = "404",
		description = "Query template with the specified name does not exist"
	)
	@APIResponse(
		responseCode = "403",
		description = "Insufficient permissions to execute the query template"
	)
	public String run(
			@PathParam(QueryTemplateManager.QUERY_TEMPLATE_NAME_PATH_PARAMETER)
			@Parameter(
				name = QueryTemplateManager.QUERY_TEMPLATE_NAME_PATH_PARAMETER,
				in = ParameterIn.PATH,
				description = "The name of the query template to execute (e.g., \"GetAllEServiceWithState\")",
				required = true,
				schema = @Schema(type = SchemaType.STRING, example = "GetAllEServiceWithState")
			)
			String queryTemplateName, String params)
			throws NotFoundException, InvalidQueryException, ResourceRegistryException {
		logger.info("Requested {} with name", QueryTemplate.NAME, queryTemplateName);
		setAccountingMethod(Method.RUN, QueryTemplate.NAME);
		
		ServerRequestInfo serverRequestInfo = initRequestInfo();
		serverRequestInfo.checkAllBooleanQueryParameters();
		
		QueryTemplateManagement queryTemplateManagement = new QueryTemplateManagement();
		queryTemplateManagement.setName(queryTemplateName);
		queryTemplateManagement.setParams(params);
		return queryTemplateManagement.run();
	}
	
	
	/**
	 * Deletes a query template with the specified name from the Resource Registry.
	 * 
	 * <strong>REST Endpoint:</strong> {@code DELETE /query-templates/{query-template-name}}
	 * 
	 * <strong>IMPORTANT: This is an irreversible operation that permanently removes the query template from the system.</strong>
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>Delete query template: {@code DELETE /query-templates/GetAllEServiceWithState}.</li>
	 * </ul>
	 * 
	 * <strong>Path Parameters:</strong>
	 * <ul>
	 * <li><strong>query-template-name</strong>: The name of the query template to delete (e.g., "GetAllEServiceWithState").</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * <ul>
	 * <li>This endpoint does not accept any query parameters</li>
	 * <li>No filtering or metadata options are available for delete operations.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Full access to delete any query template</li>
	 * <li>Can delete query templates across all contexts.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Full access to delete any query template</li>
	 * <li>Can delete query templates across all contexts.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Basic access to delete query templates</li>
	 * <li>Same deletion permissions as administrative users.</li>
	 * </ul>
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li><strong>Permanent Deletion</strong>: The query template is permanently removed from the system</li>
	 * <li><strong>Irreversible Operation</strong>: This operation cannot be undone</li>
	 * <li><strong>Template Unavailability</strong>: Once deleted, the query template name becomes available for reuse</li>
	 * <li><strong>Atomicity</strong>: The operation is atomic - either the template is completely deleted or the operation fails.</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li><strong>204 No Content</strong>: Query template successfully deleted</li>
	 * <li><strong>404 Not Found</strong>: Query template with the specified name does not exist</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to delete the query template.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Content-Type: N/A (no response body)</li>
	 * <li>Body: Empty (204 No Content response)</li>
	 * <li>No response body is returned upon successful deletion</li>
	 * <li>The query template is permanently removed from the system.</li>
	 * </ul>
	 * 
	 * @param queryTemplateName The name of the query template to delete
	 * @return HTTP 204 No Content response upon successful deletion
	 * @throws NotFoundException If the query template with the specified name does not exist
	 * @throws ResourceRegistryException If an error occurs during query template deletion
	 */
	@DELETE
	@Consumes({MediaType.TEXT_PLAIN, ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8})
	@Path("{" + QueryTemplateManager.QUERY_TEMPLATE_NAME_PATH_PARAMETER + "}")
	@Operation(
		summary = "Delete Query Template",
		description = """
		Deletes a query template with the specified name from the Resource Registry.


		**IMPORTANT: This is an irreversible operation that permanently removes the query template from the system.**


		**Request Examples:**
		- DELETE /query-templates/GetAllEServiceWithState.


		**Authorization Requirements:**
		- **IS-Manager:**
			- Full access to delete any query template;
			- Can delete query templates across all contexts.

		- **Infrastructure-Manager:**
			- Full access to delete any query template;
			- Can delete query templates across all contexts.

		- **Other Users:**
			- Basic access to delete query templates;
			- Same deletion permissions as administrative users.


		**Operation Behavior:**
		- **Permanent Deletion**: The query template is permanently removed from the system;
		- **Irreversible Operation**: This operation cannot be undone;
		- **Template Unavailability**: Once deleted, the query template name becomes available for reuse;
		- **Atomicity**: The operation is atomic - either the template is completely deleted or the operation fails.


		**Query Parameters:**
		- This endpoint does not accept any query parameters;
		- No filtering or metadata options are available for delete operations.
		"""
	)
	@APIResponse(
		responseCode = "204",
		description = "Query template successfully deleted"
	)
	@APIResponse(
		responseCode = "404",
		description = "Query template with the specified name does not exist"
	)
	@APIResponse(
		responseCode = "403",
		description = "Insufficient permissions to delete the query template"
	)
	public Response delete(
			@PathParam(QueryTemplateManager.QUERY_TEMPLATE_NAME_PATH_PARAMETER)
			@Parameter(
				name = QueryTemplateManager.QUERY_TEMPLATE_NAME_PATH_PARAMETER,
				in = ParameterIn.PATH,
				description = "The name of the query template to delete (e.g., \"GetAllEServiceWithState\")",
				required = true,
				schema = @Schema(type = SchemaType.STRING, example = "GetAllEServiceWithState")
			)
			String queryTemplateName)
			throws NotFoundException, ResourceRegistryException {
		logger.info("Requested to delete {} with name {} ", QueryTemplate.NAME, queryTemplateName);
		setAccountingMethod(Method.DELETE, QueryTemplate.NAME);
		
		QueryTemplateManagement queryTemplateManagement = new QueryTemplateManagement();
		queryTemplateManagement.setName(queryTemplateName);
		queryTemplateManagement.delete();
		
		return Response.status(Status.NO_CONTENT).build();
	}
	
}
