package org.gcube.opensearch.opensearchdatasource;

import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.opensearch.opensearchlibrary.utils.FactoryClassNamePair;
import org.globus.wsrf.jndi.Initializable;

/**
 * The configuration parameters of the {@link OpenSearchDataSource}
 * 
 * @author NKUA
 *
 */
public class OpenSearchDataSourceConfig implements Initializable{
    static GCUBELog logger = new GCUBELog(OpenSearchDataSourceConfig.class);
    
    private Boolean clearCacheOnStartup;
    private Long cacheRefreshIntervalMillis;
    private String openSearchLibraryFactories;
    private Map<String, FactoryClassNamePair> factories = new HashMap<String, FactoryClassNamePair>();
    
    /**
     * Whether cache clear when first loading resources is enabled
     * @return true if cache clear on startup is enabled, false otherwise
     */
    public Boolean getClearCacheOnStartup() {
    	return clearCacheOnStartup;
    }
    
    /**
     * The time interval between cache refresh cycles
     * @return The time interval in milliseconds
     */
    public Long getCacheRefreshIntervalMillis() {
    	return cacheRefreshIntervalMillis;
    }
    
    /**
     * The namespace-to-factory class name mappings that the OpenSearch Library will use
     * @return A {@link Map} containing all the mappings from OpenSearch extension namespaces to factory class name pairs
     */
    public Map<String, FactoryClassNamePair> getFactories() {
    	return factories;
    }
    
    /**
     * The namespace-to-factory class name mappings that the OpenSearch Library will use, in string format
     * @return A {@link Map} containing all the mappings from OpenSearch extension namespaces to factory class name pairs
     */
    public String getOpenSearchLibraryFactories() {
    	return openSearchLibraryFactories;
    }
    /**
     * Enables or disables cache clearing on startup
     * @param clearCacheOnStartup true if cache clearing should be enabled, false otherwise
     */
    public void setClearCacheOnStartup(Boolean clearCacheOnStartup) {
    	this.clearCacheOnStartup = clearCacheOnStartup;
    }
    
    /**
     * Sets the time interval between cache refresh cycles
     * @param cacheRefreshIntervalMillis The time interval in millisecods
     */
    public void setCacheRefreshIntervalMillis(Long cacheRefreshIntervalMillis) {
    	this.cacheRefreshIntervalMillis = cacheRefreshIntervalMillis;
    }
    
    public void setOpenSearchLibraryFactories(String openSearchLibraryFactories) {
    	this.openSearchLibraryFactories = openSearchLibraryFactories;
    }
    /**
     * Called on initialization. Parses openSearchLibraryFactories and prints the current configutation
     */
    public void initialize() throws Exception {
    	Pattern factoriesPattern = Pattern.compile("\\[[^=]*=\\([^,]*,[^\\)]*\\)\\]");
		Matcher factoriesMatcher = factoriesPattern.matcher(openSearchLibraryFactories);
		while(factoriesMatcher.find()) {
			String factoryEntry = factoriesMatcher.group().trim();
			factoryEntry = factoryEntry.substring(1).substring(0, factoryEntry.length()-2); //trim enclosing braces
			String keyValue[] = factoryEntry.split("=");
			if(keyValue.length != 2) {
				System.out.println("Failed to parse factory entry: " + factoryEntry + ". Ignoring entry");
				continue;
			}
			
			try {
				factories.put(keyValue[0].trim(), new FactoryClassNamePair(keyValue[1]));
			}catch(Exception e) {
				logger.debug("Failed to parse factory pair: " + keyValue[1] + ". Ignoring entry");
				continue;
			}
		}
  //  	factories.put(OpenSearchConstants.OpenSearchNS, new FactoryClassNamePair("org.gcube.opensearch.opensearchlibrary.urlelements.BasicURLElementFactory", "org.gcube.opensearch.opensearchlibrary.queryelements.BasicQueryElementFactory"));
  //  	factories.put(TimeConstants.TimeExtensionsNS, new FactoryClassNamePair("org.gcube.opensearch.opensearchlibrary.urlelements.extensions.time.TimeURLElementFactory", "org.gcube.opensearch.opensearchlibrary.queryelements.extensions.time.TimeQueryElementFactory"));
  //  	factories.put(GeoConstants.GeoExtensionsNS, new FactoryClassNamePair("org.gcube.opensearch.opensearchlibrary.urlelements.extensions.geo.GeoURLElementFactory", "org.gcube.opensearch.opensearchlibrary.queryelements.extensions.geo.GeoQueryElementFactory"));
    	
    	logger.debug("Initialized OpenSearchDataSource Config:" +
                "\n   clearCacheOnStartup: " + clearCacheOnStartup +
                "\n   cacheRefreshIntervalMillis: " + cacheRefreshIntervalMillis + 
                "\n   factories: " + factories
        );
    }

}
