package org.gcube.opensearch.opensearchdatasource;


import java.rmi.RemoteException;
import java.util.LinkedList;
import java.util.List;

import javax.xml.namespace.QName;

import org.apache.axis.components.uuid.UUIDGen;
import org.apache.axis.components.uuid.UUIDGenFactory;
import org.apache.axis.message.addressing.EndpointReferenceType;
import org.gcube.common.core.contexts.GCUBEServiceContext;
import org.gcube.common.core.contexts.GCUBEStatefulPortTypeContext;
import org.gcube.common.core.contexts.GHNContext;
import org.gcube.common.core.faults.GCUBEFault;
import org.gcube.common.core.faults.GCUBERetryEquivalentException;
import org.gcube.common.core.faults.GCUBERetrySameException;
import org.gcube.common.core.faults.GCUBERetrySameFault;
import org.gcube.common.core.faults.GCUBEUnrecoverableException;
import org.gcube.common.core.informationsystem.client.ISClient;
import org.gcube.common.core.informationsystem.client.QueryParameter;
import org.gcube.common.core.informationsystem.client.RPDocument;
import org.gcube.common.core.informationsystem.client.XMLResult;
import org.gcube.common.core.informationsystem.client.queries.GCUBEGenericQuery;
import org.gcube.common.core.informationsystem.client.queries.WSResourceQuery;
import org.gcube.common.core.porttypes.GCUBEPortType;
import org.gcube.common.core.resources.GCUBEGenericResource;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.state.GCUBEWSResourceKey;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.opensearch.opensearchdatasource.stubs.CreateResourceParams;
import org.gcube.opensearch.opensearchdatasource.stubs.CreateResourceResponse;
import org.gcube.opensearch.opensearchdatasource.stubs.Provider;

/**
 * OpenSearchDataSourceFactory creates OpenSearchDatasourceResource instances and returns
 * references to the created objects.
 * 
 * @author gerasimos.farantatos, NKUA
 */
public class OpenSearchDataSourceFactory extends GCUBEPortType {

        GCUBELog logger = new GCUBELog(OpenSearchDataSourceFactory.class);

        private static final UUIDGen uuidGen = UUIDGenFactory.getUUIDGen();

		private static final int ISRETRY = 3;
        

        /**
         * Creates an instance of the OpenSearchDataSourceResource class and returns a reference to it
         * @param request <code>CreateResource</code> - Object containing the set of searchable and presentable fields to be published and the set of OpenSearch
         * and a set of OpenSearch providers that will be handled by the DataSource. Each provider is tied to a collection id, a generic resource id and 
         * possibly some fixed parameters that will be used while querying the first-level broker OpenSearch provider
         * @return <code>CreateResourceResponse</code> - object containing the epr of the created OpenSearchDataSourceResource instance. The created resource will expose
         * a number of query templates which can be used to query an OpenSearch {@link Provider}
         * @throws GCUBEFault In case of error
         * @see {@link org.gcube.opensearch.opensearchserice.stubs.OpenSearchDataSourceFactoryPortType}
         */
        public CreateResourceResponse createResource(CreateResourceParams request)
                        throws GCUBEFault{
            try{
                logger.debug("CreateResource called... with params: " + request.getProviders() + ", " 
                		+ request.getFields());                
                CreateResourceResponse response = create(request.getFields().getArray(), 
                		request.getProviders() != null ? request.getProviders() : new Provider[0], request.getResourceKey());
                return response;

            } catch (Exception e) {
                    logger.error("Unable to create lookup resource", e);
                    throw new GCUBEUnrecoverableException(e).toFault("*** In OpenSearchDataSource.createResource ***"); 
            }
        }  
        
        
        /**
         * Creates an instance of the OpenSearchDataSourceResource class and returns a reference to it. 
         * This method is used to create a new WS resource while staging the service for a new external collection
         * @param <code>fields</code> the set of searchable and presentable fields that are to be published
         * @param <code>providers</code> the set of OpenSearch providers that will be supported by the DataSource. Each provider
         * is tied to the ID of a collection, the id of a generic resource which describes the OpenSearch provider and a number of fixed parameters which are
         * query parameters that will remain fixed when querying a top-level provider which is a broker. The ID of the generic resource will be used to retrieve
         * the generic resource from the IS.
         * @return <code>CreateResourceResponse</code> - object containing the epr of the created OpenSearchDataSourceResource instance
         * @see {@link org.gcube.opensearch.opensearchdatasource.stubs.OpenSearchDataSourceFactoryPortType}
         */
        private CreateResourceResponse create(String[] fields, Provider[] providers, String resourceKey)
                        throws RemoteException{
            OpenSearchDataSourceResource resource = null;
        	CreateResourceResponse response = new CreateResourceResponse();
              
        	//TODO: see if the adaptor ID can be removed
    		String[] collectionIDs = new String[providers.length];
    		String[] openSearchResourceIDs = new String[providers.length];
    		String[][] fixedParameters = new String[providers.length][];
    		for(int i = 0; i < providers.length; i++) {
    			collectionIDs[i] = providers[i].getCollectionID();
    			openSearchResourceIDs[i] = providers[i].getOpenSearchResourceID();
    			fixedParameters[i] = (providers[i].getFixedParameters().getArray() != null ? providers[i].getFixedParameters().getArray() : new String[0]);
    		}
    		
        	if(resourceKey != null) {
        		try {
        			logger.debug("Adding providers to resource " + resourceKey);
	        		OpenSearchDataSourceResourceHome wsHome = (OpenSearchDataSourceResourceHome)StatefulContext.getPortTypeContext().getWSHome();
	    			 resource = (OpenSearchDataSourceResource)wsHome.find(new GCUBEWSResourceKey(new QName("resource"), resourceKey));
	    			 resource.addProviders(fields, collectionIDs, openSearchResourceIDs, fixedParameters);
	    			 resource.store();
	    			 response.setEndpointReference(resource.getEPR());
	    			 return response;
        		}catch(Exception e) {
        			throw new RemoteException("Could not add providers", e);
        		}
        	}
        	
    		XMLResult[] genericResources = new XMLResult[providers.length];
    		//retrieve the openSearch generic resources from the IS
    		for(int i =0; i < providers.length; i++) {
	    		ISClient client = null;
	    		GCUBEGenericQuery query = null;
	    		try{
	    			client =  GHNContext.getImplementation(ISClient.class);
	    			query = client.getQuery("GCUBEResourceQuery");
	    		}catch (Exception e) {
					throw new GCUBERetryEquivalentException("Could not get ISClient or GCUBEGenericQuery");
				}
	    		
				query.addParameters(new QueryParameter("FILTER", "$result/ID/string() eq '" + openSearchResourceIDs[i] + "'"),
						new QueryParameter("TYPE", GCUBEGenericResource.TYPE),
						new QueryParameter("RESULT", "$result/Profile/Body"));
	
				List<XMLResult> res = null;
				for(int retryOnError=0;retryOnError<ISRETRY;retryOnError+=1){
					try{
						res = client.execute(query, ServiceContext.getContext().getScope());
					}catch(Exception e){
						logger.error("Could not retrieve generic resources. continuing");
						continue;
					}
					break;
				}
				
				if(res == null || res.size() == 0)
					throw new GCUBERetrySameException("The generic resource with ID: " + openSearchResourceIDs[i] + " was not found.");
				if(res.size() > 1)
					throw new GCUBERetrySameException("The generic resource with ID: " + openSearchResourceIDs[i] + " was found " + res.size() + " times published.");
				genericResources[i] = res.get(0);    		
			}
    		
            try {
                GCUBEStatefulPortTypeContext ptctx = StatefulContext.getPortTypeContext();
                GCUBEWSResourceKey key = ptctx.makeKey(uuidGen.nextUUID());
                //		0 GCUBEWSResourceKey key, 1 String[] collectionIDs, 2 XMLResult genericResource,
                //      3  String openSearchResource                    
                
                resource = (OpenSearchDataSourceResource) ptctx.getWSHome().create(
                		key, key, fields, collectionIDs, genericResources,
                		openSearchResourceIDs, fixedParameters);
                
                /* Finally, return the endpoint reference in a CreateResourceResponse */
                response.setEndpointReference(resource.getEPR());
                
                resource.store();
            } catch (Exception e) {
                logger.error("*** Unable to create lookup resource ***", e);
                throw new GCUBERetrySameFault(" *** Unable to create lookup resource *** " + e.getMessage());
            }

            return response;
            

        }
        
    	/**
         * Finds WS-Resources based on given property values and a given namespace
         * 
         * @param properties the (name,value) pairs of the properties
         * @param namespace the namespace of the properties
         * @param scope the scope to look at
         * @return a list of found WS-Resource EPRs
         * @throws Exception
         */
        private List<EndpointReferenceType> getWSResourceEPRsFromPropValuesAndNamespace(List<String[]> properties, String namespace, GCUBEScope scope) throws Exception {
            String filter = "$result/child::*[local-name()='"+properties.get(0)[0].trim()+"' and namespace-uri(.)='"+namespace.trim()+"']/string() eq '"+properties.get(0)[1].trim()+"'";
            for (int i=1; i<properties.size(); i++)
                    filter += " and $result/child::*[local-name()='"+properties.get(i)[0].trim()+"' and namespace-uri(.)='"+namespace.trim()+"']/string() eq '"+properties.get(i)[1].trim()+"'";
            
            ISClient client = GHNContext.getImplementation(ISClient.class);
            WSResourceQuery gquery = client.getQuery(WSResourceQuery.class);
            gquery.addGenericCondition(filter);
            
            List<EndpointReferenceType> ret = new LinkedList<EndpointReferenceType>();
            for(RPDocument d : client.execute(gquery, scope)){
                    ret.add(d.getEndpoint());
            }
            return ret;
        }

        /* (non-Javadoc)
    	 * @see org.gcube.common.core.porttypes.GCUBEStartupPortType#getServiceContext()
    	 */
    	@Override
    	protected GCUBEServiceContext getServiceContext() {
    		return ServiceContext.getContext();
    	}

}
