package org.gcube.portal.event.publisher.lr74.model;

import com.liferay.portal.kernel.events.LifecycleEvent;
import com.liferay.portal.kernel.log.Log;
import com.liferay.portal.kernel.log.LogFactoryUtil;
import com.liferay.portal.kernel.theme.ThemeDisplay;
import com.liferay.portal.kernel.util.WebKeys;
import javax.servlet.http.HttpServletRequest;

/**
 * Facade to preserve legacy event publishing semantics, bridging LR6.2 logic to LR7.4.
 */
public class EventPublisherFacade {

    private static final Log _log = LogFactoryUtil.getLog(EventPublisherFacade.class);
    private static final EventPublisherFacade INSTANCE = new EventPublisherFacade();

    public static EventPublisherFacade getInstance() {
        return INSTANCE;
    }

    public void onStartup() throws Exception {
        // Example: trigger orchestrator init or audit event
        publish("STARTUP", null, null);
    }

    public void onShutdown() throws Exception {
        publish("SHUTDOWN", null, null);
    }

    public void onPostLogin(LifecycleEvent event) throws Exception {
        HttpServletRequest request = event.getRequest();
        ThemeDisplay themeDisplay = (ThemeDisplay) request.getAttribute(WebKeys.THEME_DISPLAY);
        // Guard against null ThemeDisplay on early login flows
        if (themeDisplay == null || themeDisplay.getUser() == null) {
            _log.debug("ThemeDisplay not available during post-login; skipping event publish");
            return;
        }
        String username = themeDisplay.getUser().getScreenName();
        String scope = (themeDisplay.getScopeGroup() != null) ? themeDisplay.getScopeGroup().getGroupKey() : null;
        String token = TokenUtil.resolveUserToken(username, scope);
        publish("LOGIN", username, token);
    }

    public void onPreLogout(LifecycleEvent event) throws Exception {
        HttpServletRequest request = event.getRequest();
        ThemeDisplay themeDisplay = (ThemeDisplay) request.getAttribute(WebKeys.THEME_DISPLAY);
        String username = themeDisplay.getUser().getScreenName();
        publish("LOGOUT", username, null);
    }

    private void publish(String type, String username, String token) {
        try {
            _log.info("Publishing event type=" + type + " user=" + username);
            // Try direct API first if library is available on classpath
            try {
                Class<?> processorClass = Class.forName("org.gcube.event.publisher.EventProcessor");
                processorClass.getMethod("process", String.class, String.class, String.class)
                    .invoke(null, type, username, token);
                _log.debug("Event published successfully via EventProcessor");
                return;
            } catch (ClassNotFoundException cnf) {
                _log.debug("EventProcessor not on classpath, trying EventPublisher");
            }

            // Fallback: attempt EventPublisher API
            try {
                Class<?> publisherClass = Class.forName("org.gcube.event.publisher.EventPublisher");
                Object publisher = publisherClass.getDeclaredConstructor().newInstance();
                // Assume simple publish(String type, String user, String token)
                publisherClass.getMethod("publish", String.class, String.class, String.class)
                    .invoke(publisher, type, username, token);
                _log.debug("Event published successfully via EventPublisher");
                return;
            } catch (ClassNotFoundException cnf2) {
                _log.info("gCube event publisher libraries not available - event logged only: type=" + type + " user=" + username);
                return; // This is expected when gCube libraries are not in the bundle
            }
        } catch (Throwable t) {
            _log.warn("Event publish attempt failed for type=" + type + " user=" + username + ": " + t.getMessage());
        }
    }

    /**
     * Generic publish for model events. Attempts to call publisher methods that accept a payload map.
     */
    public void publish(String eventName, java.util.Map<String, Object> payload) {
        try {
            _log.info("Publishing event name=" + eventName + " payload keys=" + (payload != null ? payload.keySet() : "none"));
            // Try EventProcessor with (String, Map)
            try {
                Class<?> processorClass = Class.forName("org.gcube.event.publisher.EventProcessor");
                processorClass.getMethod("process", String.class, java.util.Map.class)
                    .invoke(null, eventName, payload);
                _log.debug("Model event published successfully via EventProcessor");
                return;
            } catch (ClassNotFoundException cnf) {
                _log.debug("EventProcessor not on classpath for Map payload, trying EventPublisher");
            }

            // Fallback: EventPublisher with (String, Map)
            try {
                Class<?> publisherClass = Class.forName("org.gcube.event.publisher.EventPublisher");
                Object publisher = publisherClass.getDeclaredConstructor().newInstance();
                publisherClass.getMethod("publish", String.class, java.util.Map.class)
                    .invoke(publisher, eventName, payload);
                _log.debug("Model event published successfully via EventPublisher");
                return;
            } catch (ClassNotFoundException cnf2) {
                _log.info("gCube event publisher libraries not available - model event logged only: " + eventName);
                return; // This is expected when gCube libraries are not in the bundle
            }
        } catch (Throwable t) {
            _log.warn("Model event publish attempt failed for event=" + eventName + ": " + t.getMessage());
        }
    }
}