package org.gcube.portlets.admin.gcubereleases.server;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.gcube.application.framework.core.session.ASLSession;
import org.gcube.portlets.admin.gcubereleases.client.rpc.GcubeReleasesService;
import org.gcube.portlets.admin.gcubereleases.server.converter.EticsReportConverter;
import org.gcube.portlets.admin.gcubereleases.server.database.DaoGcubeBuilderReportDBManager;
import org.gcube.portlets.admin.gcubereleases.server.exception.DatabaseServiceException;
import org.gcube.portlets.admin.gcubereleases.server.persistence.AbstractPersistence.SQL_ORDER;
import org.gcube.portlets.admin.gcubereleases.server.persistence.PackagePersistence;
import org.gcube.portlets.admin.gcubereleases.server.util.HttpCallerUtil;
import org.gcube.portlets.admin.gcubereleases.server.util.LiferayUserUtil;
import org.gcube.portlets.admin.gcubereleases.server.util.StringUtils;
import org.gcube.portlets.admin.gcubereleases.server.util.WsUtil;
import org.gcube.portlets.admin.gcubereleases.shared.Package;
import org.gcube.portlets.admin.gcubereleases.shared.Release;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.gwt.user.server.rpc.RemoteServiceServlet;


/**
 * The Class GcubeReleasesServiceImpl.
 *
 * @author Francesco Mangiacrapa francesco.mangiacrapa@isti.cnr.it
 * Feb 19, 2015
 */
@SuppressWarnings("serial")
public class GcubeReleasesServiceImpl extends RemoteServiceServlet implements GcubeReleasesService {

	protected static Logger logger = LoggerFactory.getLogger(GcubeReleasesServiceImpl.class);
	

	/**
	 * Gets the DB manager.
	 *
	 * @return the DB manager
	 */
	public DaoGcubeBuilderReportDBManager<Release> getDBManager(){
		ASLSession asl = WsUtil.getAslSession(this.getThreadLocalRequest().getSession());
		return WsUtil.getDbMangerForRelease(asl);
	}
	
	/* (non-Javadoc)
	 * @see org.gcube.portlets.admin.gcubereleases.client.rpc.GcubeReleasesService#getReleases(boolean)
	 */
	@Override
	public List<Release> getReleases(boolean onlyOnline) throws Exception{
		
		DaoGcubeBuilderReportDBManager<Release> daoManager = getDBManager();
		
		try {
			if(daoManager==null)
				throw new Exception("DaoManger is null");
			
			List<Release> releases = daoManager.getDaoViewer().getRowsOrdered("insertTime", SQL_ORDER.DESC);
			
			//TO FIX Type 'org.eclipse.persistence.indirection.IndirectList' 
			//was not included in the set of types which can be serialized 
			//by this SerializationPolicy or its Class object could not be loaded. 
			//For security purposes, this type will not be serialized.: instance = {IndirectList: not instantiated}

			List<Release> results = new ArrayList<Release>(releases.size());
			for (Release release2 : releases) {
				release2.setListPackages(null);
				if(onlyOnline){
					if(release2.isOnLine())
						results.add(release2);
				}else
					results.add(release2);
			}
			
//			logger.trace("Returning releases: ");
//			for (Release release : results) {
//				logger.trace(release.toString());
//			}
			return results;
		} catch (DatabaseServiceException e) {
			logger.error("getReleases error: ",e);
			throw new Exception("Sorry, an error occurred on reconvering releases, try again later");
		}
	}
	
	/* (non-Javadoc)
	 * @see org.gcube.portlets.admin.buildreportmng.client.rpc.BuildReportService#getSubsystemsForReleaseID(java.lang.String)
	 */
	@Override
	public Map<String, Long> getSubsystemsForReleaseID(String releaseID)
			throws Exception {
		
		DaoGcubeBuilderReportDBManager<Release> daoManager = getDBManager();
		
		try {
			if(daoManager==null)
				throw new Exception("DaoManger is null");
			
			return daoManager.getReleasePersistenceEntity().getPackagesPersistence().getMapFieldGroupedBy(releaseID, "groupID");
		} catch (DatabaseServiceException e) {
			logger.error("getSubsystemsForReleaseID error: ",e);
			throw new Exception("Sorry, an error occurred on reconvering releases, try again later");
		}
	}

	
	/* (non-Javadoc)
	 * @see org.gcube.portlets.admin.buildreportmng.client.rpc.BuildReportService#getReleaseByID(java.lang.String)
	 */
	@Override
	public Release getReleaseByID(String releaseID) throws Exception {
		
		DaoGcubeBuilderReportDBManager<Release> daoManager = getDBManager();
		
		try {
			if(daoManager==null)
				throw new Exception("DaoManger is null");
			
			HashMap<String, String> where = new HashMap<String, String>();
			where.put(Release.ID_FIELD, releaseID);
			List<Release> releases = daoManager.getReleasePersistenceEntity().getRowsFiltered(where);
			
			if(releases==null || releases.size()==0)
				throw new Exception("Release with id "+releaseID +" not found");
			
			Release release = releases.get(0);
//			release.getListPackages().size();
			
			//TO FIX Type 'org.eclipse.persistence.indirection.IndirectList' 
			//was not included in the set of types which can be serialized 
			//by this SerializationPolicy or its Class object could not be loaded. 
			//For security purposes, this type will not be serialized.: instance = {IndirectList: not instantiated}
			release.setListPackages(null);
			
//			release.setDescription(StringUtils.toTabularHTML(release.getDescription()));
			return release;
			
		} catch (DatabaseServiceException e) {
			logger.error("getReleaseByID error: ",e);
			throw new Exception("Sorry, an error occurred on reconvering releases, try again later");
		}
	}
	
	/* (non-Javadoc)
	 * @see org.gcube.portlets.admin.gcubereleases.client.rpc.GcubeReleasesService#getPackagesForReleaseID(org.gcube.portlets.admin.gcubereleases.shared.Release)
	 */
	@Override
	public List<Package> getPackagesForReleaseID(Release release) throws Exception{
		
		DaoGcubeBuilderReportDBManager<Release> daoManager = getDBManager();
		
		try {
			if(daoManager==null)
				throw new Exception("DaoManger is null");
			
			List<Package> result = daoManager.getReleasePersistenceEntity().getPackagesPersistence().getPackageOrdered(release.getId(), "groupID", null);
			
			
			//FIX Type 'org.eclipse.persistence.indirection.IndirectList'on Release
			for (Package package1 : result) {
				package1.setRelease(null);
			}
			
			return result;
		} catch (DatabaseServiceException e) {
			logger.error("getPackagesByRelease error: ",e);
			throw new Exception("Sorry, an error occurred on reconvering packages, try again later");
		}
	}

	
	/* (non-Javadoc)
	 * @see org.gcube.portlets.admin.gcubereleases.client.rpc.GcubeReleasesService#insertNewRelease(org.gcube.portlets.admin.gcubereleases.shared.Release)
	 */
	@Override
	public boolean insertNewRelease(Release release) throws Exception{

		if(StringUtils.isStringEmpty(release.getId()))
			throw new Exception("Error, mandatory field 'release ID' not found");
		
		if(StringUtils.isStringEmpty(release.getName()))
			throw new Exception("Error, mandatory field 'release Name' not found");
		
		if(StringUtils.isStringEmpty(release.getUrl()))
			throw new Exception("Error, mandatory field 'release URL' not found");

		return storeReleaseIntoDB(release);
	}
	
	/**
	 * Store release into db.
	 *
	 * @param release the release
	 * @return true, if successful
	 */
	public boolean storeReleaseIntoDB(Release release) {

		//UNCOMMENT THIS FOR DEBUG
//		DaoGcubeBuilderReportDBManager<Release> daoManager = new DaoGcubeBuilderReportDBManager<Release>();
//		daoManager.instanceReleaseEntity();
		
		DaoGcubeBuilderReportDBManager<Release> daoManager = getDBManager();

		HttpCallerUtil httpCaller = new HttpCallerUtil(release.getUrl(), "", "");
		String response = "";

		try {

			response = httpCaller.callGet("", null);
//			Release release = new Release(releaseID, releaseName, distributioXmlURL);
			
			//SETTINGS INSERT TIME AND LATEST UPDATE
			release.setInsertTime(Calendar.getInstance().getTimeInMillis());
			release.setLatestUpdate(Calendar.getInstance().getTimeInMillis());

			logger.info("Converting packages...");
			EticsReportConverter converter = new EticsReportConverter(response, release.getId());
			List<org.gcube.portlets.admin.gcubereleases.shared.Package> listPackage = converter.convertToListPackage(release);

			logger.info("Converted "+listPackage.size()+ " packages");
			logger.info("Storing data into DB...");
			release.setListPackages(listPackage);
			daoManager.getDaoUpdater().create(release);
			logger.info("Data stored correctly for: "+release.toString());
			
			List<Release> releases = daoManager.getDaoViewer().getRows();
			logger.info("Numb release: " + releases.size());

			logger.trace("Releases are: "
					+ daoManager.getDaoViewer().countItems());
			logger.trace("PackageRows are: "
					+ daoManager.getReleasePersistenceEntity()
							.getPackagesPersistence().countItems());
			logger.trace("AccountingPackageRows are: "
					+ daoManager.getReleasePersistenceEntity()
							.getPackagesPersistence()
							.getAccountingPersistence().countItems());

		} catch (Exception e) {
			logger.error("An error occurred when stroring data for: "+release.getUrl(), e);
			return false;
		}
		
		return true;
	}
	
	/* (non-Javadoc)
	 * @see org.gcube.portlets.admin.buildreportmng.client.rpc.BuildReportService#loadPackagesForSubsystem(java.lang.String, java.lang.String)
	 */
	@Override
	public List<Package> loadPackagesForSubsystem(String releaseID, String subsystemID)
			throws Exception {

		DaoGcubeBuilderReportDBManager<Release> daoManager = getDBManager();
		
		try {
			if(daoManager==null)
				throw new Exception("DaoManger is null");
			
			Map<String, String> mapFilter = new HashMap<String, String>();
			mapFilter.put("groupID", subsystemID);
			
			List<Package> packages = daoManager.getReleasePersistenceEntity().getPackagesPersistence().getPackageOrdered(releaseID, "artifactID", mapFilter);
			
			//FIX Type 'org.eclipse.persistence.indirection.IndirectList'on Release
			for (Package package1 : packages) {
				package1.setRelease(null);
			}
			logger.trace("Returning "+packages.size() +" packages for releaseID: "+releaseID +" subsystemID: "+subsystemID);
			return packages;
		} catch (DatabaseServiceException e) {
			logger.error("getPackagesByRelease error: ",e);
			throw new Exception("Sorry, an error occurred on reconvering packages, try again later");
		}
	}
	
	/* (non-Javadoc)
	 * @see org.gcube.portlets.admin.gcubereleases.client.rpc.GcubeReleasesService#updateReleaseInfo(org.gcube.portlets.admin.gcubereleases.shared.Release)
	 */
	@Override
	public Release updateReleaseInfo(Release release) throws Exception{
		
		DaoGcubeBuilderReportDBManager<Release> daoManager = getDBManager();
		
		try {
			if(daoManager==null)
				throw new Exception("DaoManger is null");
			
			if(release==null)
				throw new Exception("Release is null");
			
			logger.info("Trying to update release for key: "+release.getInternalId());
//			logger.info("Recovering release for key: "+release.getInternalId());
			
			//UPDATING LATEST UPDATE
			release.setLatestUpdate(Calendar.getInstance().getTimeInMillis());
			
			Release daoRelease = daoManager.getReleasePersistenceEntity().updateReleaseInfo(release);
			
//			Release daoRelease = daoManager.getReleasePersistenceEntity().update(release);
			
			daoRelease.setListPackages(null); //FIX INDERCT LIST
			logger.info("DaoRelease updated into DB, returning..");
			return daoRelease;
			
		} catch (DatabaseServiceException e) {
			logger.error("updateReleaseInfo error: ",e);
			throw new Exception("Sorry, an error occurred on update Release, try again later");
		}
	}
	
	/* (non-Javadoc)
	 * @see org.gcube.portlets.admin.gcubereleases.client.rpc.GcubeReleasesService#deletePackage(org.gcube.portlets.admin.gcubereleases.shared.Package)
	 */
	@Override
	public boolean deletePackage(Package pck) throws Exception{
		
		DaoGcubeBuilderReportDBManager<Release> daoManager = getDBManager();
		
		try {
			if(daoManager==null)
				throw new Exception("DaoManger is null");
			
			if(pck==null)
				throw new Exception("Package is null");
			
			logger.info("Trying to delete: "+pck.toString());
			
			PackagePersistence pckPeristence = daoManager.getReleasePersistenceEntity().getPackagesPersistence();	
//			int deleted = pckPeristence.deletePackageForID(pck.getID(), true);
			int deleted = pckPeristence.deletePackageForInternalId(pck.getInternalId(), true);
			
			logger.info("Deleted? "+(deleted>0));
			return deleted>0;
		} catch (DatabaseServiceException e) {
			logger.error("deletePackage error: ",e);
			throw new Exception("Sorry, an error occurred on delete Package, try again later");
		}
	}
	
	/* (non-Javadoc)
	 * @see org.gcube.portlets.admin.gcubereleases.client.rpc.GcubeReleasesService#deleteRelease(org.gcube.portlets.admin.gcubereleases.shared.Release)
	 */
	@Override
	public boolean deleteRelease(Release release) throws Exception{
		
		DaoGcubeBuilderReportDBManager<Release> daoManager = getDBManager();
		
		//TODO USE TO DEBUG
//		DaoGcubeBuilderReportDBManager daoManager = new DaoGcubeBuilderReportDBManager<Release>();
//		daoManager.instanceReleaseEntity();
		
		try {
			if(daoManager==null)
				throw new Exception("DaoManger is null");
			
			if(release==null)
				throw new Exception("release is null");
			
			logger.info("Trying to delete: "+release);
			
			int deleted = 0;
			
			if(release !=null){
				
				deleted = daoManager.getReleasePersistenceEntity().removeRelations(release);
//				logger.info("Removed relation (packages)? "+deleted);
				deleted = daoManager.getReleasePersistenceEntity().deleteItemByInternalId(release.getInternalId());
				logger.info("Removed release? "+(deleted>0));
			}
			
			return deleted>0;

		} catch (DatabaseServiceException e) {
			logger.error("deleteRelease error: ",e);
			throw new Exception("Sorry, an error occurred on delete Release, try again later");
		}
	}
	
	/* (non-Javadoc)
	 * @see org.gcube.portlets.admin.gcubereleases.client.rpc.GcubeReleasesService#deletePackages(java.util.List)
	 */
	@Override
	public List<Package> deletePackages(List<Package> listPcks) throws Exception{
		
		DaoGcubeBuilderReportDBManager<Release> daoManager = getDBManager();
		
		try {
			if(daoManager==null)
				throw new Exception("DaoManger is null");
			
			if(listPcks==null || listPcks.size()==0)
				throw new Exception("Package is null or empty");
			
			List<Package> errors = new ArrayList<Package>();
			
			for (Package pck : listPcks) {
				try{
					boolean deleted = deletePackage(pck);
					if(!deleted)
						errors.add(pck);
				}catch(Exception e){
					errors.add(pck);
				}
			}
			return errors;
			
		} catch (DatabaseServiceException e) {
			logger.error("deletePackages error: ",e);
			throw new Exception("Sorry, an error occurred on delete Packages, try again later");
		}
	}
	
	/* (non-Javadoc)
	 * @see org.gcube.portlets.admin.buildreportmng.client.rpc.BuildReportService#filterPackagesForValue(java.lang.String)
	 */
	@Override
	public List<Package> filterPackagesForValue(String releaseID, String filter) throws Exception {
		
		DaoGcubeBuilderReportDBManager<Release> daoManager = getDBManager();
		
		try {
			if(daoManager==null)
				throw new Exception("DaoManger is null");
			
			Map<String, String> likeMapFilter = new HashMap<String, String>();
			likeMapFilter.put("artifactID", filter);

			List<Package> packages = daoManager.getReleasePersistenceEntity().getPackagesPersistence().getPackageOrderedLikeFilter(releaseID, "artifactID", likeMapFilter);
			
			//FIX Type 'org.eclipse.persistence.indirection.IndirectList'on Release
			for (Package package1 : packages) {
				package1.setRelease(null);
			}
			logger.trace("Returning "+packages.size() +" packages for releaseID: "+releaseID +" LIKE filter: "+filter);
			return packages;
		} catch (DatabaseServiceException e) {
			logger.error("filterPackagesForValue error: ",e);
			throw new Exception("Sorry, an error occurred on reconvering packages, try again later");
		}
	}
	
	/**
	 * Checks if is management mode.
	 *
	 * @return true, if is management mode
	 */
	@Override
	public boolean isManagementMode(){
		logger.trace("Checking is Management mode..");
		try{
			logger.info("LiferayUserManager is instantiable?");
			LiferayUserUtil luu = new LiferayUserUtil();
			logger.info("LiferayUserUtil class created");
			
			//TODO ADD USER RULE CHECK
			logger.trace("is Management mode returning TRUE");
			return true;
		}catch(Exception e){
			logger.warn("LiferayUserManager is not instantiable!!: "+e.getMessage());
			logger.trace("is Management mode returning FALSE");
			return false;
		}
	}
}
