package org.gcube.portlets.user.gisviewer.server.datafeature.wfs;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.io.IOUtils;
import org.apache.log4j.Logger;
import org.gcube.portlets.user.gisviewer.client.Constants;
import org.gcube.portlets.user.gisviewer.client.commons.beans.LayerItem;
import org.gcube.portlets.user.gisviewer.client.commons.utils.URLMakers;
import org.gcube.portlets.user.gisviewer.server.util.CSVReader;
import org.gcube.portlets.user.gisviewer.shared.CSVFile;
import org.gcube.portlets.user.gisviewer.shared.CSVRow;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;


// TODO: Auto-generated Javadoc
/**
 * The Class WFSExporter.
 *
 * @author Francesco Mangiacrapa at ISTI-CNR Pisa (Italy)
 * Apr 22, 2020
 */
public class WFSExporter {
	
	private static Logger log = Logger.getLogger(WFSExporter.class);
	
//	public static final String WFS_DATA_EXPORT_CONFIG_FILE = "wfs_data_export_config.csv";
//	public static final String WFS_DATA_VIEW_CONFIG_FILE = "wfs_data_view_config.csv";
	
	/**
	 * The Enum WFSGetFeatureFields.
	 *
	 * @author Francesco Mangiacrapa at ISTI-CNR Pisa (Italy)
	 * Apr 23, 2020
	 */
	public static enum WFSGetFeatureFields{type,id,geometry_type,geometry_coordinates,geometry_name};
	
	private FeatureExporterFileConfig toWFSViewFileConfig;
	private FeatureExporterFileConfig toWFSExportFileConfig;
	
	/**
	 * The Enum WFS_TO.
	 *
	 * @author Francesco Mangiacrapa at ISTI-CNR Pisa (Italy)
	 * Apr 23, 2020
	 */
	public static enum WFS_TO {VIEW, EXPORT};
	
	/**
	 * Instantiates a new WFS exporter.
	 */
	public WFSExporter() {
		
		WFSExporterSystemDir wfsSystemDir = new WFSExporterSystemDir();
		
		try {
			CSVReader toWFSExport = wfsSystemDir.getCSVFileConfig(WFS_TO.EXPORT);
			toWFSExportFileConfig = getConfig(toWFSExport);
			log.info("To WFS export file config has the map: "+toWFSExportFileConfig.getMap());
		} catch (FileNotFoundException e) {
			log.error("Error on reading WFS config for "+WFS_TO.EXPORT, e);
		}
		
		try {
			CSVReader toWFSView = wfsSystemDir.getCSVFileConfig(WFS_TO.VIEW);
			toWFSViewFileConfig = getConfig(toWFSView);
			log.info("To WFS export file config has the map: "+toWFSViewFileConfig.getMap());
		} catch (FileNotFoundException e) {
			log.error("Error on reading WFS config for "+WFS_TO.VIEW, e);
		}
	}
	
	/**
	 * Gets the config.
	 *
	 * @param reader the reader
	 * @return the config
	 */
	private FeatureExporterFileConfig getConfig(CSVReader reader) {
		
		CSVFile file = reader.getCsvFile();
		log.debug("Building config from: "+file.getFileName());
		FeatureExporterFileConfig fileConfig = new FeatureExporterFileConfig();
		fileConfig.setFile(file);
		Map<String, FeatureExportViewConfig> map = new HashMap<String, FeatureExportViewConfig>(file.getValueRows().size());
		for (CSVRow row : file.getValueRows()) {
			List<String> values = row.getListValues();
			String featureName = values.get(0).replace("\"", "");
			boolean export = true;
			try {
				export = Boolean.parseBoolean(values.get(1).replace("\"", ""));
			}catch (Exception e) {
				log.error("Error on parsing as Boolean the property: "+values.get(1), e);
			}
			String exportAs = values.get(2).replace("\"", "");
			FeatureExportViewConfig featureExportViewConfig = new FeatureExportViewConfig(featureName, export, exportAs);
			log.trace("Adding key: "+featureName + " with config: "+featureExportViewConfig);
			map.put(featureName, featureExportViewConfig);
		}
		fileConfig.setMap(map);
		
		return fileConfig;
	}
	

	/**
	 * Gets the table from json to export.
	 *
	 * @param layerItem the layer item
	 * @param bbox the bbox
	 * @param maxWFSFeature the max WFS feature
	 * @param toViewExport the to view export
	 * @return the table from json to export
	 */
	public CSVFile getTableFromJsonToExport(LayerItem layerItem, String bbox, int maxWFSFeature, WFS_TO toViewExport) {
		
		toViewExport = toViewExport == null ? WFS_TO.VIEW : toViewExport; //default is to VIEW

		switch (toViewExport) {
		case EXPORT: {
			return getTableFromJsonToExport(layerItem, bbox, maxWFSFeature, toWFSExportFileConfig);
			}
		case VIEW: 
		default: {
			return getTableFromJsonToExport(layerItem, bbox, maxWFSFeature, toWFSViewFileConfig);
			}
		}
	}
	
	private static void printMap(Map<String, FeatureExportViewConfig> theMap) {
		for (String key : theMap.keySet()) {
			System.out.println("Key: "+key + " - value: "+theMap.get(key));
			
		}
	}
	
	
	/**
	 * Gets the table from json to export.
	 *
	 * @param layerItem the layer item
	 * @param bbox the bbox
	 * @param maxWFSFeature the max WFS feature
	 * @param exporterConfig the exporter config
	 * @return the table from json to export
	 */
	@SuppressWarnings("unchecked")
	public CSVFile getTableFromJsonToExport(LayerItem layerItem, String bbox, int maxWFSFeature, FeatureExporterFileConfig exporterConfig) {
		CSVFile tableFile = new CSVFile();
		tableFile.setFileName(layerItem.getName());
		CSVRow headerRow = new CSVRow();
		headerRow.setListValues(new ArrayList<String>()); //to manage export as false on all WFS fields
		tableFile.setHeaderRow(headerRow);
		List<CSVRow> valueRows = new ArrayList<CSVRow>();
		tableFile.setValueRows(valueRows);

		log.info("getTableFromJson -> Creating WfsTable to layerItem:  "+layerItem.getLayer());
		InputStream is = null;
		try {
			String url = URLMakers.getWfsFeatureUrl(layerItem, bbox, maxWFSFeature, Constants.JSON);
			log.info("getTableFromJson -> WFS URL:  "+url);
			is = new URL(url).openStream();
			String jsonTxt = IOUtils.toString(is);

			if(jsonTxt==null || jsonTxt.isEmpty()){
				jsonTxt = "{\"type\":\"FeatureCollection\",\"features\":[]}";
			}
			
			//validating export map config
			Map<String, FeatureExportViewConfig> exporterMapConfig = (exporterConfig == null || exporterConfig.getMap() == null)?
					new HashMap<String, FeatureExportViewConfig>():exporterConfig.getMap();
			
			log.debug("Using config to tranfrom: "+exporterMapConfig);
			List<WFSGetFeatureFields> toTransform = new ArrayList<WFSExporter.WFSGetFeatureFields>();
			
			for (WFSGetFeatureFields fieldName : WFSGetFeatureFields.values()) {
				
				FeatureExportViewConfig toHeader = exporterMapConfig.get(fieldName.name());
				log.debug("Searched field '"+fieldName.name()+"' into map has value: "+toHeader);
				if(toHeader!=null && toHeader.isExport()) {
					String exportAs = (toHeader.getExportAs() != null && !toHeader.getExportAs().isEmpty())
							? toHeader.getExportAs()
							: fieldName.name();
					headerRow.addValue(exportAs);
					toTransform.add(fieldName);
				}
			}
			
			log.debug("Detected the WFS fields to view/export: "+toTransform);

			// get json object
			JSONObject json = new JSONObject(jsonTxt);
			// iterate features
			JSONArray features = json.getJSONArray("features");
			if(features.length()==0) {
				log.info("No features detected in the response, returning empty csv file");
				return tableFile;
			}

			for (int i=0; i<features.length(); i++) {
				final CSVRow row = new CSVRow();
				//List<String> listValues = new ArrayList<String>();
				JSONObject theFeature = ((JSONObject)features.get(i));
				log.debug("Building feature at index: "+i);
				for (WFSGetFeatureFields featureToExport : toTransform) {
					switch (featureToExport) {
					case geometry_type:
						JSONObject geometry = theFeature.getJSONObject("geometry");
						String typeValue = geometry.getString("type");
						row.addValue(typeValue);
						break;
					case geometry_coordinates:
						JSONObject geometry2 = theFeature.getJSONObject("geometry");
						JSONArray coordinates = geometry2.getJSONArray("coordinates");
						row.addValue(coordinates.toString());
						break;
					default:
						String value = theFeature.getString(featureToExport.name());
						row.addValue(value);
						log.debug("Added value: "+value+ " to row");
					}
				}
				
				//parse the properties
				FeatureExportViewConfig defaultBehav = exporterMapConfig.get("properties_*");
				
				//if the default config "properties_* is missing I will export all properties found
				boolean exportProperties = defaultBehav==null?true:defaultBehav.isExport();
				log.debug("Default behaviour to export/view the properties is add: "+exportProperties);
	
//				// iterate properties
				JSONObject properties = theFeature.getJSONObject("properties");
				@SuppressWarnings("unchecked")
				Iterator<String> ii = properties.keys();
				while (ii.hasNext()) {
					String key = ii.next();
					String value = properties.optString(key,"");
					String propertykeyConfig = String.format("%s_%s", "properties",key);
					log.trace("Searching config for property: "+propertykeyConfig);
					FeatureExportViewConfig thePropertyConfig = exporterMapConfig.get(propertykeyConfig);
					boolean toAdd = true;
					if(thePropertyConfig!=null) {
						//read specific configuration for the property
						toAdd = thePropertyConfig.isExport();
						log.trace("Specific config for the property "+propertykeyConfig+" is export/view: "+toAdd);
					}else {
						//applying default behavior to export/view the properties
						toAdd = exportProperties;
						log.trace("No specific config found for the property: "+propertykeyConfig+". Applying default export/view: "+toAdd);
					}
					
					if(toAdd) {
//						//Adding the properties name to header row only if it is not already inserted
						if(!headerRow.getListValues().contains(key))
							headerRow.addValue(key);
						
						row.addValue(value);
						log.debug("Added couple ("+key+","+value+") to exported properties");
					}
				}
				
				tableFile.addValueRow(row);
				log.info("Added row "+row+" to exported properties");
			}
			
			tableFile.setHeaderRow(headerRow);


		} catch (IOException e) {
			log.error("IOException in getTableFromJson for layerItem UUID: "+layerItem.getUUID() + ", name: "+layerItem.getName(),e);
		} catch (JSONException e) {
			log.error("JSONException in getTableFromJson for layerItem UUID: "+layerItem.getUUID() + ", name: "+layerItem.getName(),e);
		}finally{
			IOUtils.closeQuietly(is);
		}

		return tableFile;
	}
	
	public FeatureExporterFileConfig getToWFSExportFileConfig() {
		return toWFSExportFileConfig;
	}
	
	public FeatureExporterFileConfig getToWFSViewFileConfig() {
		return toWFSViewFileConfig;
	}
	
}
