package org.gcube.portlets.user.homelibrary.jcr;

import java.io.File;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import org.gcube.common.core.contexts.GHNContext;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.utils.logging.GCUBEClientLog;
import org.gcube.portlets.user.homelibrary.home.Home;
import org.gcube.portlets.user.homelibrary.home.HomeManager;
import org.gcube.portlets.user.homelibrary.home.HomeManagerFactory;
import org.gcube.portlets.user.homelibrary.home.User;
import org.gcube.portlets.user.homelibrary.home.exceptions.HomeNotFoundException;
import org.gcube.portlets.user.homelibrary.home.exceptions.InternalErrorException;
import org.gcube.portlets.user.homelibrary.home.exceptions.UserNotFoundException;
import org.gcube.portlets.user.homelibrary.jcr.home.JCRHome;
import org.gcube.portlets.user.homelibrary.util.accesslog.AccessLogUtil;
import org.gcube.vomanagement.usermanagement.GroupManager;
import org.gcube.vomanagement.usermanagement.UserManager;
import org.gcube.vomanagement.usermanagement.impl.liferay.LiferayGroupManager;
import org.gcube.vomanagement.usermanagement.impl.liferay.LiferayUserManager;
import org.gcube.vomanagement.usermanagement.model.UserModel;

public class JCRHomeManager implements HomeManager{
	
	private Map<String, JCRUser> users = new LinkedHashMap<String, JCRUser>();
	private Map<String, JCRHome> userHomesLogged = new LinkedHashMap<String, JCRHome>();

	private HomeManagerFactory factory;
	private final File persistenceFoolder;
	private static GCUBEClientLog logger;
	
	
	public JCRHomeManager(HomeManagerFactory factory, File persistenceFolder) {
		
		 logger = new GCUBEClientLog(this);
		
		this.persistenceFoolder = persistenceFolder;
		this.factory = factory;
		
		createUsers();
	} 

	private void createUsers() {
		
		 GHNContext ctx = GHNContext.getContext();
         String rootOrgName= (String) ctx.getProperty(GHNContext.INFRASTRUCTURE_NAME, true);                       
         GroupManager gm = new LiferayGroupManager();
         UserManager um = new LiferayUserManager();
         
         try {
        	 
        	 logger.debug("Start import portal users");
        	 for (UserModel model : um.listUsersByGroup(gm.getGroupId(rootOrgName))) {
        		  createUser(model.getScreenName());
        		  logger.debug("User " + model.getScreenName() + " imported");
        	 }
         
         } catch (Exception e) {
        	 logger.error("Users not retrieved",e);
		}
	}

	@Override
	public HomeManagerFactory getHomeManagerFactory() {
		return factory;
	}

	@Override
	public List<User> getUsers() {
		
		return new LinkedList<User>(users.values());
	}

	@Override
	public User getUser(String portalLogin) throws InternalErrorException {
		
		logger.info("getUser portalLogin: "+portalLogin);

		return createUser(portalLogin);
	}
	
	@Override
	public synchronized boolean existUser(String portalLogin) throws InternalErrorException {

		logger.trace("existUser portalLogin: "+portalLogin);

		if (portalLogin == null){
			logger.error("portalLogin null");
			throw new IllegalArgumentException("The portalLogin value is null");
		}

		return users.containsKey(portalLogin);
	}

	@Override
	public synchronized User createUser(String portalLogin) throws InternalErrorException {
		
		
		// TODO check if is a really user of the infrastructure
		JCRUser user = users.get(portalLogin);
		if (user == null){
			logger.info("User "+portalLogin+" not found, creating a new one.");
			
			// Create a new user with scope = null
			user = new JCRUser(UUID.randomUUID().toString(),portalLogin,null);
			logger.info("User created: "+user);
			users.put(portalLogin,user);	

		}
		
		return user;
	}

	@Override
	public synchronized Home getHome(User user, GCUBEScope scope) throws InternalErrorException,
			HomeNotFoundException {
	
		logger.info("getHome user: "+user);

		if (userHomesLogged.containsKey(user.getPortalLogin())) {
			
			logger.debug(" User is already logged");
			
			JCRHome home = userHomesLogged.get(user.getPortalLogin());
			if(scope != null)
				home.setScope(scope);
			
			return home;

		}
		
		JCRHome home;
		try {
			((JCRUser)user).setScope(scope);
			home = new JCRHome(this, (JCRUser)user, persistenceFoolder);
		} catch (Exception e) {
			throw new InternalErrorException(e);
		}
		
		
		userHomesLogged.put(user.getPortalLogin(), home);
		
		logger.trace("User loaded.");
		
		if (user.getScope() != null) {
			AccessLogUtil.logWorkspaceCreated(user);
		}
		return home;
	}
	
	@Override
	public void logout(User user) throws InternalErrorException {
		// TODO
		
	}

	@Override
	public Home getHome(String portalLogin, GCUBEScope scope) throws InternalErrorException,
			HomeNotFoundException, UserNotFoundException {
		
		logger.info("getHome portalLogin: "+portalLogin);
		
		User user = getUser(portalLogin);
		return getHome(user,scope);
	}

	@Override
	public synchronized void removeUser(User user) throws InternalErrorException {
		 //TODO
		
	}

	public synchronized void homeLogout(User user) throws InternalErrorException {
		// TODO
	}

	
}
