package org.gcube.portlets.user.homelibrary.jcr.workspace;


import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.UUID;

import javax.jcr.ItemExistsException;
import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.PathNotFoundException;
import javax.jcr.Property;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Value;

import org.apache.commons.lang.Validate;
import org.apache.jackrabbit.util.Text;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.utils.logging.GCUBEClientLog;
import org.gcube.portlets.user.homelibrary.home.User;
import org.gcube.portlets.user.homelibrary.home.exceptions.InternalErrorException;
import org.gcube.portlets.user.homelibrary.home.workspace.Properties;
import org.gcube.portlets.user.homelibrary.home.workspace.WorkspaceFolder;
import org.gcube.portlets.user.homelibrary.home.workspace.WorkspaceItem;
import org.gcube.portlets.user.homelibrary.home.workspace.WorkspaceItemAction;
import org.gcube.portlets.user.homelibrary.home.workspace.WorkspaceItemType;
import org.gcube.portlets.user.homelibrary.home.workspace.accounting.AccountingEntry;
import org.gcube.portlets.user.homelibrary.home.workspace.accounting.AccountingEntryRead;
import org.gcube.portlets.user.homelibrary.home.workspace.accounting.AccountingEntryType;
import org.gcube.portlets.user.homelibrary.home.workspace.acl.Capabilities;
import org.gcube.portlets.user.homelibrary.home.workspace.exceptions.InsufficientPrivilegesException;
import org.gcube.portlets.user.homelibrary.home.workspace.exceptions.ItemAlreadyExistException;
import org.gcube.portlets.user.homelibrary.home.workspace.exceptions.ItemNotFoundException;
import org.gcube.portlets.user.homelibrary.home.workspace.exceptions.WorkspaceFolderNotFoundException;
import org.gcube.portlets.user.homelibrary.home.workspace.exceptions.WrongDestinationException;
import org.gcube.portlets.user.homelibrary.home.workspace.exceptions.WrongItemTypeException;
import org.gcube.portlets.user.homelibrary.jcr.JCRUser;
import org.gcube.portlets.user.homelibrary.jcr.repository.JCRRepository;
import org.gcube.portlets.user.homelibrary.jcr.workspace.accounting.JCRAccountingEntry;
import org.gcube.portlets.user.homelibrary.jcr.workspace.accounting.JCRAccountingEntryPaste;
import org.gcube.portlets.user.homelibrary.jcr.workspace.accounting.JCRAccountingEntryRead;
import org.gcube.portlets.user.homelibrary.jcr.workspace.accounting.JCRAccountingEntryRenaming;
import org.gcube.portlets.user.homelibrary.jcr.workspace.accounting.JCRAccountingEntryType;
import org.gcube.portlets.user.homelibrary.jcr.workspace.accounting.JCRAccountingFolderEntryCut;
import org.gcube.portlets.user.homelibrary.jcr.workspace.accounting.JCRAccountingFolderEntryRemoval;

public abstract class JCRWorkspaceItem implements WorkspaceItem {
	
	public static final String TITLE 				= "jcr:title"; 
	public static final String CREATED 				= "jcr:created";
	
	protected static final String LAST_MODIFIED_BY 	= "jcr:lastModifiedBy";
	protected static final String DESCRIPTION 		= "jcr:description";
	protected static final String LAST_ACTION 		= "hl:lastAction";
	protected static final String LAST_MODIFIED 	= "jcr:lastModified";
	
	protected static final String READERS 			= "hl:readers";
	protected static final String NT_READERS		= "nthl:readersSet";
	
	protected static final String ACCOUNTING		= "hl:accounting";
	protected static final String NT_ACCOUNTING		= "nthl:accountingSet";
	
	private static final String OWNER 				= "hl:owner";
	private static final String SCOPE               = "hl:scope";
	private static final String PORTAL_LOGIN  		= "hl:portalLogin";
	private static final String USER_ID 			= "hl:uuid";
	private static final String NT_USER				= "nthl:user";
	

	
	
	protected final JCRWorkspace workspace;
	protected String identifier;
	protected Calendar creationDate;

	
	protected static GCUBEClientLog logger = new GCUBEClientLog(JCRWorkspaceItem.class);

	public JCRWorkspaceItem(JCRWorkspace workspace, Node node) throws RepositoryException {

		this.workspace = workspace;
		this.identifier = node.getIdentifier();
		this.creationDate = node.getProperty(CREATED).getDate();

	}
	
	public JCRWorkspaceItem(JCRWorkspace workspace, Node node, String name,
			String description) throws  RepositoryException  {			

		Validate.notNull(name, "Name must be not null");
		Validate.notNull(description, "Description must be not null");
				
		this.workspace = workspace;
		
		node.setProperty(LAST_MODIFIED_BY,workspace.getOwner().getPortalLogin());
		node.setProperty(DESCRIPTION, description);
		node.setProperty(TITLE, name);
		node.setProperty(LAST_ACTION,WorkspaceItemAction.CREATED.toString());
		
		Node nodeOwner = null;
		try {
			nodeOwner = node.getNode(OWNER);
		} catch (PathNotFoundException e) {
			nodeOwner = node.addNode(OWNER,NT_USER);
		}
		
		setOwnerNode(nodeOwner);
	
	}
	
	protected void setOwnerNode(Node nodeOwner) throws RepositoryException {
		
		nodeOwner.setProperty(SCOPE,(workspace.getOwner().getScope() != null)?workspace.getOwner().getScope().toString():null);
		nodeOwner.setProperty(PORTAL_LOGIN,workspace.getOwner().getPortalLogin());
		nodeOwner.setProperty(USER_ID,workspace.getOwner().getId());
	}
	
	@Override
	public User getOwner() {
		
		
		Session session = null;
		Node node = null;
		try {
			session = JCRRepository.getSession();
			node = session.getNodeByIdentifier(identifier);
			Node nodeOwner = node.getNode(OWNER);
			return new JCRUser(nodeOwner.getProperty(USER_ID).getString(),
					nodeOwner.getProperty(PORTAL_LOGIN).getString(),GCUBEScope.getScope(nodeOwner.getProperty(SCOPE).getString()));
		} catch (PathNotFoundException e) {
			try {
				Node nodeOwner = node.addNode(OWNER,NT_USER);
				setOwnerNode(nodeOwner);
				return workspace.getOwner();
			} catch (RepositoryException e1) {
				return null;
			}
		} catch (Exception e) {
			return null;
		} finally {
			session.logout();
		}
		
	}
	
	public void save(Node node) throws RepositoryException {
		
		node.getSession().save();		
		this.identifier = node.getIdentifier();
		this.creationDate = node.getProperty(CREATED).getDate();

	}
	
	@Override
	public String getId() throws InternalErrorException {
		return identifier;
	}

	@Override
	public String getName() throws InternalErrorException {
		
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(identifier);
			String name = node.getProperty(TITLE).getString();
			return name;
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public String getDescription() throws InternalErrorException {
		
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(identifier);
			return node.getProperty(DESCRIPTION).getString();
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public void setDescription(String description)
			throws InternalErrorException {	
		
		Session session = JCRRepository.getSession();
		try {
			internalDescription(session.getNodeByIdentifier(identifier), description);
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public void rename(String name) throws InternalErrorException,
			InsufficientPrivilegesException, ItemAlreadyExistException {

		try {
			workspace.renameItem(getId(), name);
		} catch (ItemNotFoundException e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public Calendar getCreationTime() throws InternalErrorException {
		return creationDate;
	}

	@Override
	public Calendar getLastModificationTime() throws InternalErrorException {
		
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(identifier);
			return node.getProperty(LAST_MODIFIED).getDate();
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public WorkspaceItemAction getLastAction() throws InternalErrorException {
		
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(identifier);
			return WorkspaceItemAction.valueOf(node.getProperty(LAST_ACTION).getString());
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public Capabilities getCapabilities() {
		return null;
	}

	@Override
	public Properties getProperties() throws InternalErrorException {
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(identifier);
			return new JCRProperties(node);
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}
	
	public void addAccountingEntry(AccountingEntry entry) throws InternalErrorException {
		
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(identifier);
			
			if (!node.hasNode(ACCOUNTING)){
				node.addNode(ACCOUNTING, NT_ACCOUNTING);
				session.save();
			}
			Node accountingNode = node.getNode(ACCOUNTING);
			JCRAccountingEntryType nodeType = JCRAccountingEntryType.valueOf(entry.getEntryType().toString());
			logger.debug("Accountin node type " + nodeType.getNodeTypeDefinition());
			Node entryNode = accountingNode.addNode(UUID.randomUUID().toString(),
					nodeType.getNodeTypeDefinition());
			
			((JCRAccountingEntry) entry).save(entryNode);
			
			session.save();	
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		}
	}
	
	
	@Override
	public List<AccountingEntry> getAccounting() {
		List<AccountingEntry> list = new ArrayList<AccountingEntry>();
		
		Session session = null;
		try {
			session = JCRRepository.getSession();
			Node node = session.getNodeByIdentifier(identifier);
			Node accountingNode = node.getNode(ACCOUNTING);
		
			for(NodeIterator iterator = accountingNode.getNodes();iterator.hasNext();) {
				Node entryNode = (Node)iterator.next();
				try {
					switch (JCRAccountingEntryType.getEnum(
							entryNode.getPrimaryNodeType().getName())) {
							case CUT:
								list.add(new JCRAccountingFolderEntryCut(entryNode));
								break;
							case PASTE:
								list.add(new JCRAccountingEntryPaste(entryNode));
								break;
							case REMOVAL:
								list.add(new JCRAccountingFolderEntryRemoval(entryNode));
								break;
							case RENAMING:
								list.add(new JCRAccountingEntryRenaming(entryNode));
								break;
							default:
								break;
					}
				} catch (Exception e) {
					logger.error("Accounting entry skipped ",e);
				}
				
			}
			return list;

		} catch (Exception e) {
			logger.error("Error to retrieve accounting entries ",e);
			return list;
		} finally {
			if (session != null)
				session.logout();
		}

	}

	@Override
	public WorkspaceFolder getParent() throws InternalErrorException {
	
		Session session = JCRRepository.getSession();
		try {
			return workspace.getParent(session.
					getNodeByIdentifier(identifier));
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}
	
	protected JCRAbstractWorkspaceFolder getParent(Node node) throws InternalErrorException {
		try {
			return workspace.getParent(node);
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} 
	}
	
	@Override
	public boolean isShared () throws InternalErrorException {
		return (getIdSharedFolder() != null)?true:false;
	}
	
	@Override
	public String getIdSharedFolder() throws InternalErrorException {
		
		if (isRoot()) {
			return null;
		}
		
		if (getType() == WorkspaceItemType.SHARED_FOLDER) {
			return getId();
		}
		
		return  ((JCRWorkspaceItem)getParent()).getIdSharedFolder();
	}
	
	@Override
	public String getPath() throws InternalErrorException {
		
		Session session = JCRRepository.getSession();
		try {
			return getPath(session.getNodeByIdentifier(identifier));
		} catch (Exception e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
		
	}
	
	public String getPath(Node node) throws RepositoryException, InternalErrorException {
		
		if (isRoot(node)) return workspace.getPathSeparator() + node.getProperty(TITLE).getString();
		return getParent(node).getPath(node.getParent()) + workspace.getPathSeparator() 
		+ node.getProperty(TITLE).getString();		
	}

	@Override
	public boolean isRoot() throws InternalErrorException { 
		return getParent() == null;
	}
	
	public boolean isRoot(Node node) throws RepositoryException, InternalErrorException {
		return workspace.getParent(node) == null;
	}

	@Override
	public void remove() throws InternalErrorException,
			InsufficientPrivilegesException {
		try {
			workspace.removeItem(getId());
		} catch (ItemNotFoundException e) {
			throw new InternalErrorException(e);
		} 
		
	}

	@Override
	public void move(WorkspaceFolder destination)
			throws InternalErrorException, WrongDestinationException,
			InsufficientPrivilegesException, ItemAlreadyExistException {
		
		try {
			workspace.moveItem(getId(), destination.getId());
		} catch (ItemNotFoundException e) {
			throw new InternalErrorException(e);
		} catch (WorkspaceFolderNotFoundException e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public WorkspaceItem cloneItem(String cloneName)
			throws InternalErrorException, InsufficientPrivilegesException,
			ItemAlreadyExistException {
		
		try {
			return workspace.cloneItem(getId(), cloneName);
		} catch (ItemNotFoundException e) {
			throw new InternalErrorException(e);
		} catch (WrongDestinationException e) {
			throw new InternalErrorException(e);
		} catch (WorkspaceFolderNotFoundException e) {
			throw new InternalErrorException(e);
		}
	}
	
	public Node internalCopy(Node nodeFolder, String newName) throws InternalErrorException,
	ItemAlreadyExistException, WrongDestinationException, RepositoryException{
		
		Node node = nodeFolder.getSession().getNodeByIdentifier(identifier);
		
		String pathNewNode = nodeFolder.getPath()
		+ workspace.getPathSeparator() + Text.escapeIllegalJcrChars(newName);
		
		try {
			
			if(node.getSession().getNode(pathNewNode) != null)
				throw new ItemAlreadyExistException(newName + " already exist");
		} catch(RepositoryException e) {
			
		}
		
		try {
			node.getSession().getWorkspace().copy(node.getPath(),pathNewNode);
			
			Node newNode = node.getSession().getNode(pathNewNode);
			newNode.setProperty(LAST_MODIFIED,Calendar.getInstance());
			newNode.setProperty(LAST_MODIFIED_BY,workspace.getOwner().getPortalLogin());
			newNode.setProperty(TITLE, newName);
			newNode.setProperty(LAST_ACTION,WorkspaceItemAction.CLONED.toString());
			newNode.getSession().save();
			return newNode;
		} catch (ItemExistsException e) {
			throw new ItemAlreadyExistException(e.getMessage());
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		}
		
	}
	
	public void internalMove(Node destinationFolderNode) throws ItemAlreadyExistException,
	InternalErrorException, RepositoryException {
			
		try {
			logger.debug("Start internal move item with id " 
					+ getId() + " to destination item with id " + destinationFolderNode.getIdentifier());
			
			Node node = destinationFolderNode.getSession().getNodeByIdentifier(identifier);
			
			if (workspace.exists(node.getName(), destinationFolderNode.getIdentifier())) {
				logger.error("Item with name " + getName() + " exists");
				throw new ItemAlreadyExistException("Item " + node.getName() + " already exists");
			}
			
			node.setProperty(LAST_MODIFIED,Calendar.getInstance());
			node.setProperty(LAST_MODIFIED_BY,workspace.getOwner().getPortalLogin());
			node.setProperty(LAST_ACTION,WorkspaceItemAction.MOVED.toString());
			node.getSession().save();
			
			node.getSession().getWorkspace().move(node.getPath(), destinationFolderNode.getPath() 
					+ workspace.getPathSeparator() + node.getName());
		} catch (RepositoryException e) {
			logger.error("Repository exception thrown by move operation",e);
			throw new InternalErrorException(e);
		} catch (WrongItemTypeException e) {
			logger.fatal("Unhandled Exception ");
			throw new InternalErrorException(e);
		} catch (ItemNotFoundException e) {
			logger.fatal("Unhandled Exception ");
			throw new InternalErrorException(e);
		} 
	}

	public void internalRename(Node node, String newName) throws ItemAlreadyExistException, InternalErrorException {
		
		String nodeNewName = Text.escapeIllegalJcrChars(newName);
		try {
			
			logger.debug("Internal rename item with id " 
					+ getId() + " to destination item with id " + node.getParent().getIdentifier());
			if (workspace.exists(nodeNewName, node.getParent().getIdentifier())) {
				logger.error("Item with name " + nodeNewName + " exists");
				throw new ItemAlreadyExistException("Item " + nodeNewName + " already exists");
			}
			String newPath = node.getParent().getPath() 
			+ workspace.getPathSeparator() + nodeNewName;
			
			node.setProperty(LAST_MODIFIED,Calendar.getInstance());
			node.setProperty(LAST_MODIFIED_BY,workspace.getOwner().getPortalLogin());
			node.setProperty(LAST_ACTION,WorkspaceItemAction.RENAMED.toString());
			node.setProperty(TITLE,newName);
			node.getSession().save();
			
			String path = node.getPath();
			node.getSession().getWorkspace().move(path, newPath);
			
			
			
		} catch (RepositoryException e) {
			logger.error("Repository exception thrown by move operation",e);
			throw new InternalErrorException(e);
		} catch (WrongItemTypeException e) {
			logger.fatal("Unhandled Exception ");
			throw new InternalErrorException(e);
		} catch (ItemNotFoundException e) {
			logger.fatal("Unhandled Exception ");
			throw new InternalErrorException(e);
		}
		
	}

	public void internalDescription(Node node, String newDescription) throws InternalErrorException {
		
		Validate.notNull(newDescription, "Description must be not null");
		try {
			node.setProperty(DESCRIPTION, newDescription);
			node.setProperty(LAST_MODIFIED, Calendar.getInstance());
			node.setProperty(LAST_MODIFIED_BY, workspace.getOwner().getPortalLogin());
			node.getSession().save();
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} catch (Exception e) {
			throw new InternalErrorException(e);
		} 
	}
	
	@Override
	public boolean isMarkedAsRead() throws InternalErrorException  {
		
		Session session = JCRRepository.getSession();
		try {
			return existReader(session);
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}
	
	@Override
	public List<AccountingEntryRead> getReaders() throws InternalErrorException {
		List<AccountingEntryRead> list = new ArrayList<AccountingEntryRead>();
		
		Session session = JCRRepository.getSession();
		try {

			Node node = session.getNodeByIdentifier(identifier);
			Node readersNode = node.getNode(READERS);
			for (NodeIterator iterator = readersNode.getNodes(); iterator.hasNext();) {
				Node reader = iterator.nextNode();
				
				if ( JCRAccountingEntryType.getEnum(
						reader.getPrimaryNodeType().getName()) != null)
					list.add(new JCRAccountingEntryRead(reader));
			}
			return list; 
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
		
	}
	

	private boolean existReader(Session session) throws  RepositoryException,
	InternalErrorException{
			
		Node node = session.getNodeByIdentifier(getId());
		Node readersNode = null;
		try {
			 readersNode = node.getNode(READERS);
		} catch(PathNotFoundException e) {
			readersNode = node.addNode(READERS, NT_READERS );
			session.save();
			return false;
		}
		
		if (readersNode.hasNode(workspace.getOwner().getPortalLogin()))
			return true;
		
		return false;
	}

	@Override
	public void markAsRead(boolean read) throws InternalErrorException {
		
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(getId());
			String user = workspace.getOwner().getPortalLogin();
			
			if (!existReader(session)) {
				Node readers = node.getNode(READERS);
				if (read) {
					JCRAccountingEntryRead entry = new JCRAccountingEntryRead(
							user, Calendar.getInstance());
					entry.save(readers.addNode(user, 
							JCRAccountingEntryType.READ.getNodeTypeDefinition()));
				}
			} else {
				Node readers = node.getNode(READERS);
				if (!read)
					readers.getNode(user).remove();
			}
			session.save();
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
		
	}
}
