package org.gcube.portlets.user.homelibrary.jcr.workspace;

import java.io.IOException;
import java.io.InputStream;
import java.util.Calendar;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.UUID;

import javax.jcr.ItemExistsException;
import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.PathNotFoundException;
import javax.jcr.Property;
import javax.jcr.PropertyIterator;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.query.QueryManager;
import javax.jcr.query.QueryResult;
import javax.print.attribute.standard.MediaSize.ISO;

import org.apache.commons.lang.Validate;
import org.apache.jackrabbit.util.ISO9075;
import org.apache.jackrabbit.util.Text;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.utils.logging.GCUBEClientLog;
import org.gcube.contentmanagement.blobstorage.transport.backend.RemoteBackendException;
import org.gcube.portlets.user.homelibrary.home.Home;
import org.gcube.portlets.user.homelibrary.home.User;
import org.gcube.portlets.user.homelibrary.home.exceptions.HomeNotFoundException;
import org.gcube.portlets.user.homelibrary.home.exceptions.InternalErrorException;
import org.gcube.portlets.user.homelibrary.home.workspace.Workspace;
import org.gcube.portlets.user.homelibrary.home.workspace.WorkspaceFolder;
import org.gcube.portlets.user.homelibrary.home.workspace.WorkspaceItem;
import org.gcube.portlets.user.homelibrary.home.workspace.WorkspaceItemAction;
import org.gcube.portlets.user.homelibrary.home.workspace.WorkspaceItemType;
import org.gcube.portlets.user.homelibrary.home.workspace.WorkspaceSharedFolder;
import org.gcube.portlets.user.homelibrary.home.workspace.WorkspaceSmartFolder;
import org.gcube.portlets.user.homelibrary.home.workspace.acl.Capabilities;
import org.gcube.portlets.user.homelibrary.home.workspace.events.AbstractWorkspaceEventSource;
import org.gcube.portlets.user.homelibrary.home.workspace.exceptions.InsufficientPrivilegesException;
import org.gcube.portlets.user.homelibrary.home.workspace.exceptions.ItemAlreadyExistException;
import org.gcube.portlets.user.homelibrary.home.workspace.exceptions.ItemNotFoundException;
import org.gcube.portlets.user.homelibrary.home.workspace.exceptions.WorkspaceFolderNotFoundException;
import org.gcube.portlets.user.homelibrary.home.workspace.exceptions.WrongDestinationException;
import org.gcube.portlets.user.homelibrary.home.workspace.exceptions.WrongItemTypeException;
import org.gcube.portlets.user.homelibrary.home.workspace.exceptions.WrongParentTypeException;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.FolderBulkCreator;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.FolderBulkCreatorManager;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.FolderItem;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.FolderItemType;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.ExternalFile;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.ExternalImage;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.ExternalPDFFile;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.ExternalResourceLink;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.ExternalUrl;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.Query;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.QueryType;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.Report;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.ReportTemplate;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.TabularDataLink;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.WorkflowReport;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.WorkflowTemplate;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.gcube.Annotation;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.gcube.Document;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.gcube.ImageDocument;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.gcube.Metadata;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.gcube.PDFDocument;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.gcube.UrlDocument;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.gcube.link.DocumentLink;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.gcube.link.ImageDocumentLink;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.gcube.link.PDFDocumentLink;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.ts.TimeSeries;
import org.gcube.portlets.user.homelibrary.home.workspace.search.SearchFolderItem;
import org.gcube.portlets.user.homelibrary.home.workspace.search.SearchItem;
import org.gcube.portlets.user.homelibrary.home.workspace.sharing.WorkspaceMessageManager;
import org.gcube.portlets.user.homelibrary.jcr.home.JCRHome;
import org.gcube.portlets.user.homelibrary.jcr.repository.JCRRepository;
import org.gcube.portlets.user.homelibrary.jcr.sharing.JCRWorkspaceMessageManager;
import org.gcube.portlets.user.homelibrary.jcr.workspace.accounting.JCRAccountingEntryPaste;
import org.gcube.portlets.user.homelibrary.jcr.workspace.accounting.JCRAccountingEntryRenaming;
import org.gcube.portlets.user.homelibrary.jcr.workspace.accounting.JCRAccountingFolderEntryCut;
import org.gcube.portlets.user.homelibrary.jcr.workspace.accounting.JCRAccountingFolderEntryRemoval;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.ContentType;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.JCRAquaMapsItem;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.JCRExternalFile;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.JCRExternalImage;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.JCRExternalPDFFile;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.JCRExternalUrl;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.JCRFile;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.JCRImage;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.JCRPDFFile;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.JCRQuery;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.JCRReport;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.JCRReportTemplate;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.JCRTabularDataLink;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.JCRTimeSeries;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.JCRWorkflowReport;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.JCRWorkflowTemplate;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.JCRWorkspaceFolderItem;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.gcube.JCRAnnotation;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.gcube.JCRDocument;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.gcube.JCRExternalResourceLink;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.gcube.JCRImageDocument;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.gcube.JCRMetadata;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.gcube.JCRPDFDocument;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.gcube.JCRUrlDocument;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.gcube.link.JCRDocumentLink;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.gcube.link.JCRImageDocumentLink;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.gcube.link.JCRPDFDocumentLink;
import org.gcube.portlets.user.homelibrary.jcr.workspace.search.JCRSearchFolder;
import org.gcube.portlets.user.homelibrary.jcr.workspace.search.JCRSearchFolderItem;
import org.gcube.portlets.user.homelibrary.util.Util;
import org.gcube.portlets.user.homelibrary.util.WorkspaceUtil;

public class JCRWorkspace extends AbstractWorkspaceEventSource implements
		Workspace {
	
	private static final String WORKSPACE_ROOT_FOLDER 			= "Workspace";
	
	public static final String NT_WORKSPACE_FOLDER 			= "nthl:workspaceItem";
	public static final String NT_WORKSPACE_SHARED_FOLDER		= "nthl:workspaceSharedItem";
	private static final String NT_WORKSPACE_FOLDER_ITEM		= "nthl:workspaceLeafItem";
	private static final String NT_WORKSPACE_FILE 				= "nthl:externalFile";
	private static final String NT_WORKSPACE_IMAGE 				= "nthl:externalImage";
	private static final String NT_WORKSPACE_PDF_FILE 			= "nthl:externalPdf";
	private static final String NT_WORKSPACE_URL 				= "nthl:externalUrl";
	
	private static final String NT_WORKSPACE_REPORT				= "nthl:report";
	private static final String NT_WORKSPACE_REPORT_TEMPLATE	= "nthl:reportTemplate"; 
	private static final String NT_WORKSPACE_WORKFLOW_REPORT	= "nthl:workflowReport";
	private static final String NT_WORKSPACE_WORKFLOW_TEMPLATE	= "nthl:workflowTemplate";
	
	private static final String NT_ANNOTATION_ITEM				= "nthl:gCubeAnnotation";
	private static final String NT_ANNOTATION_LINK_ITEM			= "nthl:gCubeAnnotationLink";
	
	private static final String NT_DOCUMENT_ITEM 				= "nthl:gCubeDocument";
	private static final String NT_DOCUMENT_LINK_ITEM			= "nthl:gCubeDocumentLink";
	private static final String NT_IMAGE_DOCUMENT_ITEM 			= "nthl:gCubeImageDocument";
	private static final String NT_IMAGE_DOCUMENT_LINK_ITEM 	= "nthl:gCubeImageDocumentLink";
	private static final String NT_PDF_DOCUMENT_ITEM 			= "nthl:gCubePDFDocument";
	private static final String NT_PDF_DOCUMENT_LINK_ITEM		= "nthl:gCubePDFDocumentLink";
	private static final String NT_URL_DOCUMENT_ITEM 			= "nthl:gCubeURLDocument";
	private static final String NT_METADATA_ITEM				= "nthl:gCubeMetadata";				
	private static final String NT_METADATA_LINK_ITEM			= "nthl:gCubeMetadataLink";
	private static final String NT_AQUAMAPS_ITEM 				= "nthl:aquamapsItem";
	private static final String NT_TIMESERIES_ITEM 				= "nthl:timeSeriesItem";
	private static final String NT_QUERY 						= "nthl:query";
	private static final String NT_EXTERNAL_RESOURCE_LINK_ITEM   = "nthl:externalResourceLink";
	
	private static final String NT_FILE 						= "nt:file";
	private static final String CONTENT							= "jcr:content";

	
	private String scope;
	private final Home home;
	private final JCRRepository repository;
	private final JCRFolderBulkCreatorManager folderBulkCreatorsManager;
	private JCRWorkspaceFolder root;
	private JCRWorkspaceMessageManager sendRequestManager;
	
	
	private GCUBEClientLog logger;
	
	public JCRWorkspace(Home home, JCRRepository repository) throws InternalErrorException {
		
		this.scope = null;
		this.home = home;
		this.repository = repository;
		this.folderBulkCreatorsManager = new JCRFolderBulkCreatorManager(this);

		this.logger = new GCUBEClientLog(this);
		
		Session session = JCRRepository.getSession();
		String pathRoot = null;
		try {
			Node userHome = repository.getUserHome(session);
			Node node = null;
			try {
				node = userHome.addNode(WORKSPACE_ROOT_FOLDER,NT_WORKSPACE_FOLDER);
				this.root = new JCRWorkspaceFolder(this, node, WORKSPACE_ROOT_FOLDER, "The root");
				root.save(node);
			} catch (ItemExistsException e) {
				node = userHome.getNode(WORKSPACE_ROOT_FOLDER);
				this.root = new JCRWorkspaceFolder(this, node);
			}
			pathRoot = node.getPath();
			
		} catch (RepositoryException e) {
			logger.error("Root WorkspaceFolder failed creation", e);
			throw new InternalErrorException(e);
		} finally {
			if(session != null)
				session.logout();
		}
		
		JCRRepository.setACL(home.getOwner().getPortalLogin(), pathRoot);

	}
	
	public JCRWorkspace(JCRHome home) {
		this.home = home;
		this.repository = null;
		this.folderBulkCreatorsManager = null;
	}
	
	
	private Node addChildNode(Session session, String parentId, String nodeName, String nodeType) throws ItemAlreadyExistException,
	WorkspaceFolderNotFoundException, InternalErrorException, WrongDestinationException {
		
		Validate.notNull(parentId, "Destination folder must be not null");
		Validate.notNull(nodeName, "Name must be not null");
		
		if (!isValidName(nodeName)){
			
			logger.error("The name  " + nodeName + "contains illegal chars or is empty");
			throw new IllegalArgumentException("The name contains illegal chars or is empty");
		}
		
		
		Node parent;
		try {
			parent = session.getNodeByIdentifier(parentId);
			
			if(!parent.getPrimaryNodeType().getName().equals(NT_WORKSPACE_FOLDER) &&
					!parent.getPrimaryNodeType().getName().equals(NT_WORKSPACE_SHARED_FOLDER) &&
					// FIXME item send functionality: copied NT_WORKSPACE_SHARED_FOLDER
					// items on "hl:attachments" node with type "nt:folder"
					!parent.getPrimaryNodeType().getName().equals("nt:folder")) {
			
				
				throw new WrongDestinationException("Not is a folder");
			}
		} catch (RepositoryException e) {
			logger.error("Destination folder not found");
			throw new WorkspaceFolderNotFoundException(e.getMessage());
		}
		
		try {
			String name = Text.escapeIllegalJcrChars(nodeName);
			Node node = parent.addNode(name,nodeType);
			
			return node;
		} catch (ItemExistsException e) {
			logger.error("Item already exist");
			throw new ItemAlreadyExistException(e.getMessage());
		} catch (RepositoryException e) {
			logger.fatal(e);
			throw new InternalErrorException(e);
		}
	}
	
	
	@Override
	public String getPathSeparator() {
		return JCRRepository.PATH_SEPARATOR;
	}

	@Override
	public Home getHome() {
		return home;
	}

	public JCRRepository getRepository() {
		return repository;
	}
	
	@Override
	public User getOwner() {
		return home.getOwner();
	}

	@Override
	public WorkspaceFolder getRoot() {
		return getRoot(null);
	}

	@Override
	public  WorkspaceFolder getRoot(GCUBEScope scope) {
		
		this.scope = (scope == null) ? null : scope.toString();
		return root;
	}
	
	
	@Override
	public WorkspaceFolder createFolder(String name, String description,
			String destinationFolderId) throws InternalErrorException,
			InsufficientPrivilegesException, ItemAlreadyExistException,
			WrongDestinationException, ItemNotFoundException,
			WorkspaceFolderNotFoundException {
		
		logger.trace("Create workspace folder");
		
		Session session = JCRRepository.getSession();
		try {
			Node node = addChildNode(session,destinationFolderId,name,NT_WORKSPACE_FOLDER);		
			JCRWorkspaceFolder folder = new JCRWorkspaceFolder(this, node, name, 
					description);
			folder.save(node);
			
			fireItemCreatedEvent(folder);
				
			return folder;
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
		
		
	}

	@Override
	public ExternalImage createExternalImage(String name, String description,
			String mimeType, InputStream imageData, String destinationFolderId)
			throws InsufficientPrivilegesException,
			WorkspaceFolderNotFoundException, InternalErrorException,
			ItemAlreadyExistException, WrongDestinationException {
		
		logger.trace("Create external image");
		
		Session session = JCRRepository.getSession();
		try {
			Node node = addChildNode(session, destinationFolderId, name, NT_WORKSPACE_IMAGE);
			JCRExternalImage item =  new JCRExternalImage(this, node, name, description, mimeType,
					imageData);
			item.save(node);
						
			fireItemCreatedEvent(item);
			
			return item;
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} catch (IOException e) {
			throw new InternalErrorException(e);
		} catch (RemoteBackendException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public ExternalFile createExternalFile(String name, String description,
			String mimeType, InputStream fileData, String destinationFolderId)
			throws InsufficientPrivilegesException,
			WorkspaceFolderNotFoundException, InternalErrorException,
			ItemAlreadyExistException, WrongDestinationException {
			
		logger.trace("Create external file");
		Session session = JCRRepository.getSession();
		try {
			Node node = addChildNode(session, destinationFolderId, name, NT_WORKSPACE_FILE);
			JCRExternalFile item = new JCRExternalFile(this,node,name,description,mimeType,fileData);
			item.save(node);
						
			fireItemCreatedEvent(item);
			
			return item;
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} catch (IOException e) {
			throw new InternalErrorException(e);
		} catch (RemoteBackendException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public ExternalPDFFile createExternalPDFFile(String name,
			String description, String mimeType, InputStream fileData,
			String destinationFolderId) throws InsufficientPrivilegesException,
			WorkspaceFolderNotFoundException, InternalErrorException,
			ItemAlreadyExistException, WrongDestinationException {
		
		logger.trace("Create external pdf file");
		
		Session session = JCRRepository.getSession();
		try {
			Node node = addChildNode(session, destinationFolderId, name, NT_WORKSPACE_PDF_FILE);
			JCRExternalPDFFile item =  new JCRExternalPDFFile(this, node, name, description, mimeType,
					fileData);
			item.save(node);
			
			fireItemCreatedEvent(item);
			
			return item;
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} catch (IOException e) {
			throw new InternalErrorException(e);
		} catch (RemoteBackendException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public ExternalUrl createExternalUrl(String name, String description,
			String url, String destinationFolderId)
			throws InsufficientPrivilegesException,
			WorkspaceFolderNotFoundException, InternalErrorException,
			ItemAlreadyExistException, WrongDestinationException {
		
		logger.trace("Create external url");
		Session session = JCRRepository.getSession();
		try {
			Node node = addChildNode(session, destinationFolderId, name, NT_WORKSPACE_URL);
			JCRExternalUrl item = new JCRExternalUrl(this, node, name, description, url);
			item.save(node);
						
			fireItemCreatedEvent(item);
			
			return item;
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} catch (RemoteBackendException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
		
	}

	@Override
	public ExternalUrl createExternalUrl(String name, String description,
			InputStream url, String destinationFolderId)
			throws InsufficientPrivilegesException, InternalErrorException,
			ItemAlreadyExistException, WrongDestinationException,
			WorkspaceFolderNotFoundException {
		
		String urlString;
		try {
			urlString = Util.readStreamAsString(url);
		} catch (IOException e) {
			throw new InternalErrorException("Error converting url from" +
					" input stream to string.");
		}
		return createExternalUrl(name, description, urlString, destinationFolderId);
	}

	@Override
	public ReportTemplate createReportTemplate(String name, String description,
			Calendar created, Calendar lastEdit, String author,
			String lastEditBy, int numberOfSections, String status,
			InputStream templateData, String destinationfolderId)
			throws InsufficientPrivilegesException, InternalErrorException,
			ItemAlreadyExistException, WrongDestinationException,
			WorkspaceFolderNotFoundException {
		
		logger.trace("Created report template");
		
		Session session = JCRRepository.getSession();
		try {
			Node node = addChildNode(session, destinationfolderId, name, NT_WORKSPACE_REPORT_TEMPLATE);
			JCRReportTemplate reportTemplate = new JCRReportTemplate(this, node, name, description,
					created, lastEdit, author, lastEditBy, numberOfSections, status, templateData);
			reportTemplate.save(node);
						
			fireItemCreatedEvent(reportTemplate);
			return reportTemplate;
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} catch (RemoteBackendException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	
	}

	@Override
	public Report createReport(String name, String description,
			Calendar created, Calendar lastEdit, String author,
			String lastEditBy, String templateName, int numberOfSections,
			String status, InputStream reportData, String destinationfolderId)
			throws InsufficientPrivilegesException, InternalErrorException,
			ItemAlreadyExistException, WrongDestinationException,
			WorkspaceFolderNotFoundException {
		
		logger.trace("Create report");
	
		Session session = JCRRepository.getSession();
		try {
			Node node = addChildNode(session, destinationfolderId, name, NT_WORKSPACE_REPORT);
			JCRReport item = new JCRReport(this,node,name,description,created,lastEdit,
					author,lastEditBy,templateName,numberOfSections,status,reportData);
			item.save(node);
						
			fireItemCreatedEvent(item);
			
			return item;
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} catch (RemoteBackendException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	
	}

	@Override
	public Query createQuery(String name, String description, String query,
			QueryType queryType, String destinationFolderId)
			throws InsufficientPrivilegesException, InternalErrorException,
			ItemAlreadyExistException, WrongDestinationException,
			WorkspaceFolderNotFoundException {
		
		logger.trace("Create query");
		
		Session session = JCRRepository.getSession();
		try {
			Node node = addChildNode(session, destinationFolderId, name, NT_QUERY);
			JCRQuery item = new JCRQuery(this, node, name, description, query, queryType);
			item.save(node);
						
			fireItemCreatedEvent(item);
			
			return item;
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	
	}

	@Override
	public Query createQuery(String name, String description,
			InputStream query, QueryType queryType, String destinationFolderId)
			throws InsufficientPrivilegesException, InternalErrorException,
			ItemAlreadyExistException, WrongDestinationException,
			WorkspaceFolderNotFoundException {
		
		try {
			return createQuery(name,description,Util.readStreamAsString(query),queryType,destinationFolderId);
		} catch (IOException e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public WorkspaceFolder createAquaMapsItem(String name, String description,
			String mapName, String mapType, String author, int numberOfSpecies,
			String boundingBox, float psoThreshold,
			int numberOfGeneratedImages, InputStream metadata,
			Map<String, InputStream> images, String destinationFolderId)
			throws InsufficientPrivilegesException, InternalErrorException,
			ItemAlreadyExistException, WrongDestinationException,
			WorkspaceFolderNotFoundException {
		
		logger.trace("Create aquamaps item");

		Session session = JCRRepository.getSession();		
		JCRWorkspaceFolder folder;
		try {
			Node node = addChildNode(session,destinationFolderId,name,NT_WORKSPACE_FOLDER);		
			folder = new JCRWorkspaceFolder(this, node, name, 
					description);
			folder.save(node);
			
		
			
						
			String fileName = WorkspaceUtil.getUniqueName("metadata.xml", folder);
			
			Node nodeFile = addChildNode(session, node.getIdentifier(),fileName, NT_WORKSPACE_FILE);
			JCRExternalFile file = new JCRExternalFile(this,nodeFile,fileName,name + " metadata","text/xml",metadata);
			file.save(nodeFile);
			
		
						
			fireItemCreatedEvent(file);
			for(Entry<String, InputStream> entry : images.entrySet()) {
				
				String imageName = Text.escapeIllegalJcrChars(entry.getKey());
				Node imageNode = addChildNode(session, node.getIdentifier(), imageName, NT_WORKSPACE_IMAGE);
				JCRExternalImage image = new JCRExternalImage(this, imageNode, imageName, description, "", entry.getValue());
				image.save(imageNode);
				
		
				
				fireItemCreatedEvent(image);
			}
			
			
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} catch (IOException e) {
			throw new InternalErrorException(e);
		} catch (RemoteBackendException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
		
		return folder;		
	
	}

	@Override
	public Annotation createAnnotation(String name, String description,
			String oid, Map<String, String> data, String destinationFolderId)
			throws InsufficientPrivilegesException, InternalErrorException,
			ItemAlreadyExistException, WorkspaceFolderNotFoundException {
		
		logger.trace("Create annotation item");
		
		Session session = JCRRepository.getSession();
		try {
			Node node = addChildNode(session ,destinationFolderId, name, NT_ANNOTATION_ITEM);
			JCRAnnotation item = new JCRAnnotation(this, node, name, description,
					oid,data);
			item.save(node);
			
			fireItemCreatedEvent(item);
			
			
			return item;
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} catch (WrongDestinationException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public Metadata createMetadata(String name, String description, String oid,
			String schema, String language, String data, String collectionName,
			String destinationFolderId) throws InsufficientPrivilegesException,
			InternalErrorException, ItemAlreadyExistException,
			WrongDestinationException, WorkspaceFolderNotFoundException {
		
		logger.trace("Create metadata item");
		
		Session session = JCRRepository.getSession();
		try {
			Node node = addChildNode(session, destinationFolderId, name, NT_METADATA_ITEM);
			JCRMetadata item = new JCRMetadata(this,node,name,description,oid,schema,
					language,collectionName,data);
			item.save(node);
			
			fireItemCreatedEvent(item);
						
			return item;
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} catch (RemoteBackendException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public Document createDocument(String name, String description, String oid,
			String mimeType, InputStream documentData,
			Map<String, String> metadata, Map<String, String> annotations,
			String collectionName, String destinationFolderId)
			throws InsufficientPrivilegesException, InternalErrorException,
			ItemAlreadyExistException, WrongDestinationException,
			WorkspaceFolderNotFoundException {
		
		logger.trace("Create document item");
		
		Session session = JCRRepository.getSession();
		try {
			Node node = addChildNode(session, destinationFolderId, name, NT_DOCUMENT_ITEM);
			JCRDocument item = new JCRDocument(this, node, name, description, oid, mimeType,
					documentData, metadata, annotations, collectionName);
			item.save(node);
						
			fireItemCreatedEvent(item);
			
			return item;
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public ImageDocument createImageDocument(String name, String description,
			String oid, String mimeType, InputStream imageData,
			Map<String, String> metadata, Map<String, String> annotations,
			String collectionName, String destinationFolderId)
			throws InsufficientPrivilegesException, InternalErrorException,
			ItemAlreadyExistException, WrongDestinationException,
			WorkspaceFolderNotFoundException {
		
		logger.trace("Create image document item");
		
		Session session = JCRRepository.getSession();
		try {
			Node node = addChildNode(session, destinationFolderId, name, NT_IMAGE_DOCUMENT_ITEM);
			JCRImageDocument item = new JCRImageDocument(this, node, name, description, 
					oid, mimeType,imageData, metadata, annotations, collectionName);
			item.save(node);
						
			fireItemCreatedEvent(item);
			
			return item;
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public PDFDocument createPDFDocument(String name, String description,
			String oid, String mimeType, InputStream data,
			Map<String, String> metadata, Map<String, String> annotations,
			String collectionName, String destinationFolderId)
			throws InsufficientPrivilegesException, InternalErrorException,
			ItemAlreadyExistException, WrongDestinationException,
			WorkspaceFolderNotFoundException {
		
		logger.trace("Create pdf document item");
		
		Session session = JCRRepository.getSession();
		try {
			Node node = addChildNode(session, destinationFolderId, name, NT_PDF_DOCUMENT_ITEM);
			JCRPDFDocument	item = new JCRPDFDocument(this,node,name,description,oid,
					mimeType,data,metadata,annotations,collectionName);
			item.save(node);
						
			fireItemCreatedEvent(item);
			
			return item;
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public UrlDocument createUrlDocument(String name, String description,
			String oid, String mimeType, InputStream documentData,
			Map<String, String> metadata, Map<String, String> annotations,
			String collectionName, String destinationFolderId)
			throws InsufficientPrivilegesException,
			WorkspaceFolderNotFoundException, InternalErrorException,
			ItemAlreadyExistException, WrongDestinationException {
		
		logger.trace("Create url document item");
		
		Session session = JCRRepository.getSession();
		try {
			Node node = addChildNode(session, destinationFolderId, name, NT_URL_DOCUMENT_ITEM);
			JCRUrlDocument	item = new JCRUrlDocument(this, node, name, description, oid, mimeType,
					documentData, metadata, annotations, collectionName);
			item.save(node);
						
			fireItemCreatedEvent(item);
			
			return item;
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public DocumentLink createDocumentLink(String name, String description,
			String oid, Map<String, String> metadata,
			Map<String, String> annotations, String collectionName,
			String mimeType, String destinationFolderId)
			throws InsufficientPrivilegesException,
			WorkspaceFolderNotFoundException, InternalErrorException,
			ItemAlreadyExistException, WrongDestinationException {
		
		logger.trace("Create document link item");
		Session session = JCRRepository.getSession();
		try {
			Node node = addChildNode(session, destinationFolderId, name, NT_DOCUMENT_LINK_ITEM);
			JCRDocumentLink item = new JCRDocumentLink(this, node, name, description, oid,
					mimeType, metadata, annotations, collectionName);
			item.save(node);
						
			fireItemCreatedEvent(item);
			
			return item;
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public ImageDocumentLink createImageDocumentLink(String name,
			String description, String oid, Map<String, String> metadata,
			Map<String, String> annotations, String collectionName,
			String mimeType, String destinationFolderId)
			throws InsufficientPrivilegesException,
			WorkspaceFolderNotFoundException, InternalErrorException,
			ItemAlreadyExistException, WrongDestinationException {
		
		Session session = JCRRepository.getSession();
		try {
			Node node = addChildNode(session, destinationFolderId, name, NT_IMAGE_DOCUMENT_LINK_ITEM);
			JCRImageDocumentLink item = new JCRImageDocumentLink(this, node, name, description, oid,
					mimeType, metadata, annotations, collectionName);
			item.save(node);
						
			fireItemCreatedEvent(item);
			
			return item;
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public PDFDocumentLink createPDFDocumentLink(String name,
			String description, String oid, Map<String, String> metadata,
			Map<String, String> annotations, String collectionName,
			String mimeType, String destinationFolderId)
			throws InsufficientPrivilegesException,
			WorkspaceFolderNotFoundException, InternalErrorException,
			ItemAlreadyExistException, WrongDestinationException {
	
		Session session = JCRRepository.getSession();
		try {
			Node node = addChildNode(session, destinationFolderId, name, NT_PDF_DOCUMENT_LINK_ITEM);
			JCRPDFDocumentLink item = new JCRPDFDocumentLink(this, node, name, description, oid,
					mimeType, metadata, annotations, collectionName);
			item.save(node);
						
			fireItemCreatedEvent(item);
			
			return item;
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	
	}

	@Override
	public TimeSeries createTimeSeries(String name, String description,
			String timeseriesId, String title, String creator,
			String timeseriesDescription, String timeseriesCreationDate,
			String publisher, String sourceId, String sourceName,
			String rights, long dimension, List<String> headerLabels,
			InputStream compressedCSV, String destinationFolderId)
			throws InsufficientPrivilegesException, InternalErrorException,
			ItemAlreadyExistException, WorkspaceFolderNotFoundException,
			WrongDestinationException {
		
		logger.trace("Create TimeSeries item");
		Session session = JCRRepository.getSession();
		try {
			Node node = addChildNode(session, destinationFolderId, name, NT_TIMESERIES_ITEM);
			JCRTimeSeries item = new JCRTimeSeries(this, node, name, description,
					timeseriesId, title, creator, timeseriesDescription,
					timeseriesCreationDate, publisher, sourceId, sourceName,
					rights, dimension, headerLabels, compressedCSV);
			item.save(node);
					
			fireItemCreatedEvent(item);
			
			return item;
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} catch (RemoteBackendException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public WorkflowReport createWorkflowReport(String name, String description,
			String workflowId, String workflowStatus, String workflowData,
			String destinationFolderId) throws InsufficientPrivilegesException,
			InternalErrorException, ItemAlreadyExistException,
			WorkspaceFolderNotFoundException, WrongDestinationException {
		
		logger.trace("Create WorkflowReport item");
		
		Session session = JCRRepository.getSession();
		try {
			Node node = addChildNode(session, destinationFolderId, name, NT_WORKSPACE_WORKFLOW_REPORT);
			JCRWorkflowReport item = new JCRWorkflowReport(this, node, name,
					description, workflowId, workflowStatus, workflowData);
			item.save(node);
						
			fireItemCreatedEvent(item);
			
			return item;
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public WorkflowTemplate createWorkflowTemplate(String name,
			String description, String workflowId, String workflowStatus,
			String workflowData, String destinationFolderId)
			throws InsufficientPrivilegesException, InternalErrorException,
			ItemAlreadyExistException, WorkspaceFolderNotFoundException,
			WrongDestinationException {
		
		logger.trace("Create WorkflowTemplate item");
		
		Session session = JCRRepository.getSession();
		try {
			Node node = addChildNode(session, destinationFolderId, name, NT_WORKSPACE_WORKFLOW_TEMPLATE);
			JCRWorkflowTemplate item = new JCRWorkflowTemplate(this, node, name,
					description, workflowId, workflowStatus, workflowData);
			item.save(node);
						
			fireItemCreatedEvent(item);
			
			return item;
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}
	
	@Override
	public ExternalResourceLink createExternalResourceLink(String name, String description,
			String mimeType, String resourceId, String pluginName, String destinationFolderId)
					throws  WorkspaceFolderNotFoundException, WrongDestinationException,
					InternalErrorException, ItemAlreadyExistException, InsufficientPrivilegesException{
		
		logger.trace("Create service resource link");
		
		Session session = JCRRepository.getSession();
		try {
			Node node = addChildNode(session, destinationFolderId, name, NT_EXTERNAL_RESOURCE_LINK_ITEM);
			JCRExternalResourceLink item = new JCRExternalResourceLink(this, node, name, mimeType,
					description, resourceId, pluginName);
			item.save(node);
						
			fireItemCreatedEvent(item);
			
			return item;
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
		
	}
	
	public void removeRemoteContent(Node node) throws RepositoryException, InternalErrorException, RemoteBackendException {
		
		Session session = node.getSession();
		
		WorkspaceItem item = getWorkspaceItem(node);
		for (WorkspaceItem child : item.getChildren()) {
			removeRemoteContent(session.getNodeByIdentifier(child.getId()));
		}
		
		if (item.getType() == WorkspaceItemType.FOLDER_ITEM) {
			((JCRWorkspaceFolderItem)item).removeRemoteContent(node);
		}
	}


	@Override
	public void removeItem(String itemId) throws ItemNotFoundException,
			InternalErrorException, InsufficientPrivilegesException {
		
		Validate.notNull(itemId , "Item id must be not null");
		
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(itemId);
			
			JCRWorkspaceItem item = (JCRWorkspaceItem) getItem(session,itemId);
			 
				
			
			
			JCRWorkspaceItem parent = (JCRWorkspaceItem)item.getParent();
			
			
			
			//Add removal accounting entry to folder item parent
			parent.addAccountingEntry(new JCRAccountingFolderEntryRemoval(getOwner().getPortalLogin(),
					Calendar.getInstance(),
					item.getType(),
					(item.getType() == WorkspaceItemType.FOLDER_ITEM)?((FolderItem)item).getFolderItemType():null,
							item.getName(),
							(item.getType() == WorkspaceItemType.FOLDER_ITEM)?((FolderItem)item).getMimeType():null));
			
			if (item.getType() == WorkspaceItemType.SHARED_FOLDER) {
				item.remove();
				return;
			}
			
			// TODO temporarily solution to remove all FOLDER_ITEM
			// remote contents in GCUBEStorage.
			removeRemoteContent(node);
			
			fireItemRemovedEvent(getWorkspaceItem(node));
			node.remove();
			session.save();
			
		} catch (javax.jcr.ItemNotFoundException e) {
			throw new ItemNotFoundException(e.getMessage());
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} catch (RemoteBackendException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}
	

	private void checkDestination(Node node, Node destinationNode)  
			throws WrongDestinationException, InternalErrorException, InsufficientPrivilegesException {
		
		try {	
			Session session = node.getSession();
			
			if (!destinationNode.getPrimaryNodeType().getName().equals(NT_WORKSPACE_FOLDER) &&
					!destinationNode.getPrimaryNodeType().getName().equals(NT_WORKSPACE_SHARED_FOLDER)) {
				logger.error("Destination is not a folder");
				throw new WrongDestinationException("Destination is not a folder");
			} 
			
			JCRWorkspaceItem item = getWorkspaceItem(node);
			JCRWorkspaceItem itemDestination = getWorkspaceItem(destinationNode);
						
			QueryManager queryManager = session.getWorkspace().getQueryManager();
			javax.jcr.query.Query q = queryManager.createQuery("/jcr:root/Home/" + getOwner().getPortalLogin()
					+ ISO9075.encodePath(item.getPath()) +
					"//element(*,nthl:workspaceSharedItem)",
					javax.jcr.query.Query.XPATH);
			QueryResult result = q.execute();
			if (itemDestination.isShared() && !item.isShared() && result.getNodes().hasNext()
					|| (itemDestination.isShared() && (item.getType() == WorkspaceItemType.SHARED_FOLDER ))) {
				throw new WrongDestinationException("Not allowed to move in an other destination folder already shared");	 
					
			}			
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		}
				
	}
	
	@Override
	public void moveItem(String itemId, String destinationFolderId)
			throws ItemNotFoundException, WrongDestinationException,
			InsufficientPrivilegesException, InternalErrorException,
			ItemAlreadyExistException, WorkspaceFolderNotFoundException {
		
		logger.debug("Move item with id " + itemId + "to destination with id " + destinationFolderId);
		
		Validate.notNull(itemId , "Item id must be not null");
		Validate.notNull(destinationFolderId, "Destination folder id must be not null");
		
		Session session = JCRRepository.getSession();
		try {
			Node nodeFolder;
			try {
				nodeFolder = session.getNodeByIdentifier(destinationFolderId);			
			
			} catch (RepositoryException e) {
				logger.error("Destination not found");
				throw new WorkspaceFolderNotFoundException(e.getMessage());
			}

			Node nodeItem;
			try {
				nodeItem = session.getNodeByIdentifier(itemId);
			} catch (javax.jcr.ItemNotFoundException e) {
				logger.error("Item with id "+ itemId + " not found");
				throw new ItemNotFoundException(e.getMessage());
			} catch (RepositoryException e) {
				logger.fatal("Fatal error retrieving item with id " + itemId);
				throw new InternalErrorException(e);
			}

			
			try {
				checkDestination(nodeItem, nodeFolder);
				
				JCRWorkspaceItem item = ((JCRWorkspaceItem)getWorkspaceItem(nodeItem));
				JCRWorkspaceItem parent = (JCRWorkspaceItem) item.getParent();
				
				// Set cut accounting entry to folder parent item
				logger.debug("Adding accounting entry ....");
				parent.addAccountingEntry(new JCRAccountingFolderEntryCut(getOwner().getPortalLogin(),
						Calendar.getInstance(),
						item.getType(),
						(item.getType() == WorkspaceItemType.FOLDER_ITEM)?((FolderItem)item).getFolderItemType():null,
								item.getName(),
								(item.getType() == WorkspaceItemType.FOLDER_ITEM)?((FolderItem)item).getMimeType():null));
				// Set paste accounting entry to item
				item.addAccountingEntry(new JCRAccountingEntryPaste(getOwner().getPortalLogin(),
						Calendar.getInstance(), item.getPath()));
				
				item.internalMove(nodeFolder);
					
			} catch (RepositoryException e) {
				logger.fatal("Fatal error moving item with id " + itemId 
						+ " to WorkspaceFolder with id " + destinationFolderId);

				throw new InternalErrorException(e);
			}
			
			session.save();
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
		
	}

	@Override
	public void renameItem(String itemId, String newName)
			throws ItemNotFoundException, InternalErrorException,
			ItemAlreadyExistException {
				
		Validate.notNull(itemId, "Item id must be not null");
		
		if (!isValidName(newName))
			throw new IllegalArgumentException("Invalid item name");
	
		Session session = JCRRepository.getSession();
		try {
			Node nodeItem;
			try {
				nodeItem = session.getNodeByIdentifier(itemId);
			} catch (RepositoryException e) {
				throw new ItemNotFoundException(e.getMessage());
			}

			JCRWorkspaceItem item = null;
			try {
				item = getWorkspaceItem(nodeItem);
				
				//Set renaming accounting entry to item
				item.addAccountingEntry(new JCRAccountingEntryRenaming(getOwner().getPortalLogin(),
						Calendar.getInstance(), item.getName()));
				
				item.internalRename(nodeItem, newName);
				
			} catch (RepositoryException e) {
				throw new InternalErrorException(e);
			} 
			session.save();
			fireItemRenamedEvent(item);
			
			
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public void changeDescription(String itemId, String newDescription)
			throws ItemNotFoundException, InternalErrorException {
		
		Validate.notNull(itemId, "Item id must be not null");
		Session session = JCRRepository.getSession();
		try {
			Node nodeItem = session.getNodeByIdentifier(itemId);
			getWorkspaceItem(nodeItem).internalDescription(nodeItem, newDescription);
		} catch (RepositoryException e) {
			throw new ItemNotFoundException(e.getMessage());
		} finally {
			session.logout();
		}
	}

	@Override
	public WorkspaceItem getItem(String itemId) throws ItemNotFoundException {
		
		Validate.notNull(itemId, "Item id must be not null");
		
		Session session = null;
		try {
			session = JCRRepository.getSession();
			return getItem(session, itemId);
		} catch (InternalErrorException e) {
			throw new ItemNotFoundException(e.getMessage());
		} catch (RepositoryException e) {
			throw new ItemNotFoundException(e.getMessage());
		} finally {
			if (session != null)
				session.logout();
		}
	}
	
	public WorkspaceItem getItem(Session session, String itemId) throws ItemNotFoundException,
	InternalErrorException, RepositoryException {
		
		Node nodeItem = null;
		try {
			nodeItem = session.getNodeByIdentifier(itemId);
		} catch (javax.jcr.ItemNotFoundException e) {
			throw new ItemNotFoundException(e.getMessage());
		} 
		return getWorkspaceItem(nodeItem);
	}
	
	@Override
	public Capabilities getCapabilities(String itemId)
			throws ItemNotFoundException, InternalErrorException {
		return null;
	}

	@Override
	public void removeChild(String childId, String folderId)
			throws ItemNotFoundException, InternalErrorException,
			InsufficientPrivilegesException, WrongParentTypeException {
		
		Validate.notNull(childId, "Child Id must be not null");
		Validate.notNull(folderId, "Folder Id must be not null");
		
		Session session = JCRRepository.getSession();
		try {

			Node parent = session.getNodeByIdentifier(folderId);
			if (!parent.getPrimaryNodeType().getName().equals(NT_WORKSPACE_FOLDER)) {
				throw new WrongParentTypeException("Item with id " + folderId + " isn't a folder item");
			}
			removeItem(childId);
		} catch (javax.jcr.ItemNotFoundException e) {
			throw new ItemNotFoundException("Folder is not present");
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
			
	}

	@Override
	public void remove(String itemName, String folderId)
			throws ItemNotFoundException, InternalErrorException,
			InsufficientPrivilegesException, WrongItemTypeException {
		
		Session session = JCRRepository.getSession();
		try {
			Node nodeFolder = null;
			try {
				nodeFolder = session.getNodeByIdentifier(folderId);

			} catch (javax.jcr.ItemNotFoundException e) {
				throw new ItemNotFoundException(e.getMessage());
			} catch (RepositoryException e) {
				throw new InternalErrorException(e);
			} 

			try {
				Node childNode = nodeFolder.getNode(Text.escapeIllegalJcrChars(itemName));
				removeItem(childNode.getIdentifier());
				
			} catch (PathNotFoundException e) {
				throw new ItemNotFoundException(e.getMessage());
			} catch (RepositoryException e) {
				throw new InternalErrorException(e);
			}
			session.save();
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public WorkspaceItem copy(String itemId, String newName,
			String destinationFolderId) throws ItemNotFoundException,
			WrongDestinationException, InternalErrorException,
			ItemAlreadyExistException, InsufficientPrivilegesException,
			WorkspaceFolderNotFoundException {
		
		Validate.notNull(itemId, "ItemId must be not null");
		Validate.notNull(newName, "NewName must be not null");
		Validate.notNull(destinationFolderId, "Destination Folder id must be not null");
		
		if (!isValidName(newName)){
			logger.error("The name contains illegal chars or is empty");
			throw new IllegalArgumentException("The name contains illegal chars or is empty");
		}
		
		return internalCopy(itemId, newName, destinationFolderId);
		
		
	}

	@Override
	public WorkspaceItem copy(String itemId, String destinationFolderId)
			throws ItemNotFoundException, WrongDestinationException,
			InternalErrorException, ItemAlreadyExistException,
			InsufficientPrivilegesException, WorkspaceFolderNotFoundException {
		
		Validate.notNull(itemId,"Item id must be not null");
		Validate.notNull(destinationFolderId, "destinationFolder id must be not null");
		
		return internalCopy(itemId, null, destinationFolderId); 
	}
	
	public void copyRemoteContent(Node node) throws RepositoryException, InternalErrorException, RemoteBackendException  {
		
		Session session = node.getSession();
		
		WorkspaceItem item = getWorkspaceItem(node);
		for (WorkspaceItem child : item.getChildren()) {
			copyRemoteContent(session.getNodeByIdentifier(child.getId()));
		}
		
		if (item.getType() == WorkspaceItemType.FOLDER_ITEM) {
			((JCRWorkspaceFolderItem)item).copyRemoteContent(node);
		}
		
	}
	
	private WorkspaceItem internalCopy(String itemId, String newName, String destinationFolderId) throws ItemNotFoundException,
	WrongDestinationException, WorkspaceFolderNotFoundException, ItemAlreadyExistException, InternalErrorException {
		
		Session session = JCRRepository.getSession();
		try {
			Node nodeItem = null;
			try {
				nodeItem = session.getNodeByIdentifier(itemId);
			} catch (RepositoryException e) {
				throw new ItemNotFoundException(e.getMessage());
			}

			Node nodeDestinationFolder = null;
			try {
				if(destinationFolderId == null)
					destinationFolderId = nodeItem.getParent().getIdentifier();
				
				nodeDestinationFolder = session.getNodeByIdentifier(destinationFolderId);
				if(!nodeDestinationFolder.getPrimaryNodeType().getName().equals(NT_WORKSPACE_FOLDER)
						&& !nodeDestinationFolder.getPrimaryNodeType().getName().equals(NT_WORKSPACE_SHARED_FOLDER)) {
					throw new WrongDestinationException("Destination is not a folder");
				}
			} catch (RepositoryException e) {
				throw new WorkspaceFolderNotFoundException(e.getMessage());
			}
			
			WorkspaceItem item = getWorkspaceItem(nodeItem);
			WorkspaceItem itemDestination = getWorkspaceItem(nodeDestinationFolder);
			QueryManager queryManager = session.getWorkspace().getQueryManager();
			javax.jcr.query.Query q = queryManager.createQuery("/jcr:root/Home/" + getOwner().getPortalLogin()
					+ ISO9075.encodePath(item.getPath()) +
					"//element(*,nthl:workspaceSharedItem)",
					javax.jcr.query.Query.XPATH);
			QueryResult result = q.execute();
			if (!item.isShared() && result.getNodes().hasNext()) {
				throw new WrongDestinationException("Not allowed to copy a folder with some discendents item shared ");	 
			}

			if (newName == null) 
				newName = item.getName();
			
			Node newNode = ((JCRWorkspaceItem)getWorkspaceItem(nodeItem)).internalCopy(
					nodeDestinationFolder, newName);
			session.save();
			JCRWorkspaceItem newItem = getWorkspaceItem(newNode);
			
			// Set paste accounting property
			newItem.addAccountingEntry(new JCRAccountingEntryPaste(getOwner().getPortalLogin(),
					Calendar.getInstance(), item.getPath()));
			
			//TODO temporarily solution to copy all remote content item child nodes.
			copyRemoteContent(newNode);
			
			session.save();
			
			return newItem;

		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} catch (RemoteBackendException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public WorkspaceItem cloneItem(String itemId, String cloneName)
			throws ItemNotFoundException, ItemAlreadyExistException,
			InsufficientPrivilegesException, InternalErrorException,
			WrongDestinationException, WorkspaceFolderNotFoundException {
		
		Validate.notNull(itemId, "itemId must be not null");
		if(!isValidName(cloneName)) 
			throw new IllegalArgumentException("cloneName is a not valid name");

		return internalCopy(itemId, cloneName, null);
	}

	@Override
	public boolean exists(String name, String folderId)
			throws InternalErrorException, ItemNotFoundException,
			WrongItemTypeException {
		
		Validate.notNull(name, "Name must be not null");
		Validate.notNull(folderId, "Name must be not null");
		
		Session session = JCRRepository.getSession();
		try {
			Node folderNode;
			try {
				folderNode = session.getNodeByIdentifier(folderId);
			} catch (RepositoryException e) {
				throw new ItemNotFoundException(e.getMessage());
			}

			if (!isValidName(name)){
				return false;
			}

			try {
				folderNode.getNode(Text.escapeIllegalJcrChars(name));
			} catch (RepositoryException e) {
				logger.debug("Item not exists");
				return false; 
			}

			logger.debug("Item exists");
			return true;
		} finally {
			session.logout();
		}
	}

	@Override
	public boolean exists(String itemId) throws InternalErrorException {
		
		Validate.notNull(itemId, "Item id must be not null");
		Session session = JCRRepository.getSession();
		try {
			try {
				session.getNodeByIdentifier(itemId);
			} catch (RepositoryException e) {
				return false; 
			}
			return true;
		} finally {
			session.logout();
		}
	}

	@Override
	public WorkspaceItem find(String name, String folderId)
			throws InternalErrorException, ItemNotFoundException,
			WrongItemTypeException {

		Session session = JCRRepository.getSession();
		try {
			Node nodeFolder = session.getNodeByIdentifier(folderId);
			Node node = nodeFolder.getNode(Text.escapeIllegalJcrChars(name));
			return getWorkspaceItem(node);
		} catch (javax.jcr.ItemNotFoundException e) {
			throw new ItemNotFoundException(e.getMessage());
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public WorkspaceItem find(String path) throws InternalErrorException {
		
		Session session = JCRRepository.getSession();
		try {
			try {
				String[] strings = path.split("/");
				String pathCleaned = "";
				for(String string : strings) {
					pathCleaned += "/" + Text.escapeIllegalJcrChars(string);
				}
				Node rootNode = session.getNodeByIdentifier(root.getId());	
				Node node = session.getNode(rootNode.getPath() + pathCleaned);
				return getWorkspaceItem(node);
			} catch (RepositoryException e) {
				throw new InternalErrorException(e);
			}
		} finally {
			session.logout();
		}
	}

	@Override
	public boolean isValidName(String name) {
		if(name == null || name.length() == 0)
			return false;
		return !name.contains(JCRRepository.PATH_SEPARATOR);
	}

	@Override
	public FolderBulkCreator getNewFolderBulkCreator(String folderId)
			throws WorkspaceFolderNotFoundException, WrongItemTypeException,
			InternalErrorException {
		
		Validate.notNull(folderId, "Folder id must be not null");
		
		Session session = JCRRepository.getSession();
		try {
			Node folderNode;
			try {
				folderNode = session.getNodeByIdentifier(folderId);
			} catch (RepositoryException e) {
				throw new WorkspaceFolderNotFoundException(e.getMessage());
			}

			try {		
				if (!folderNode.getPrimaryNodeType().getName().equals(NT_WORKSPACE_FOLDER))
					throw new WrongItemTypeException("A FolderBulkCreator can be created " +
					" only for a folder");
			} catch (RepositoryException e) {
				throw new InternalErrorException(e); 
			}
			
			JCRWorkspaceFolder folder = new JCRWorkspaceFolder(this, folderNode);

			return folderBulkCreatorsManager.getFolderBulk(folder);
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public FolderBulkCreatorManager getFolderBulkCreatorManager() {
		return folderBulkCreatorsManager;
	}

	@Override
	public WorkspaceMessageManager getWorkspaceMessageManager() {
		
		if(sendRequestManager == null)
			this.sendRequestManager = new JCRWorkspaceMessageManager(this);
		return sendRequestManager;
	}

	@Override
	public WorkspaceFolder decomposeAquaMapsItem(String itemId,
			String folderName, String destinationWorkspaceId)
			throws WrongItemTypeException, WorkspaceFolderNotFoundException,
			WrongDestinationException, InternalErrorException,
			ItemAlreadyExistException, InsufficientPrivilegesException,
			ItemNotFoundException {
		
		
		return null;
	}
	
	public JCRWorkspaceItem getWorkspaceItem(Node node) throws RepositoryException, InternalErrorException {
		
		
		if (node.getPrimaryNodeType().getName().equals(NT_WORKSPACE_FOLDER)) {
			return new JCRWorkspaceFolder(this, node);
		} else if (node.getPrimaryNodeType().getName().equals(NT_WORKSPACE_SHARED_FOLDER)) {
			return new JCRWorkspaceSharedFolder(this, node);
		} else if (node.getPrimaryNodeType().getName().equals(NT_WORKSPACE_FILE)) {
			JCRWorkspaceItem item = new JCRExternalFile(this, node);
			return item;
		} else if (node.getPrimaryNodeType().getName().equals(NT_WORKSPACE_IMAGE)) {
			JCRWorkspaceItem item = new JCRExternalImage(this, node);
			return item; 
		} else if (node.getPrimaryNodeType().getName().equals(NT_WORKSPACE_PDF_FILE)) {
			JCRWorkspaceItem item =new JCRExternalPDFFile(this,node);
			return item; 
		} else if (node.getPrimaryNodeType().getName().equals(NT_WORKSPACE_URL)) {
			return new JCRExternalUrl(this, node);
		} else if (node.getPrimaryNodeType().getName().equals(NT_WORKSPACE_REPORT)) {
			return new JCRReport(this, node);
		} else if (node.getPrimaryNodeType().getName().equals(NT_WORKSPACE_REPORT_TEMPLATE)) {
			return new JCRReportTemplate(this, node);
		} else if (node.getPrimaryNodeType().getName().equals(NT_WORKSPACE_WORKFLOW_REPORT)) {
			return new JCRWorkflowReport(this, node);
		} else if (node.getPrimaryNodeType().getName().equals(NT_WORKSPACE_WORKFLOW_TEMPLATE)) {
			return new JCRWorkflowTemplate(this, node);
		} else if (node.getPrimaryNodeType().getName().equals(NT_AQUAMAPS_ITEM)) {
			return new JCRAquaMapsItem(this, node);
		} else if (node.getPrimaryNodeType().getName().equals(NT_QUERY)) {
			return new JCRQuery(this,node);
		} else if (node.getPrimaryNodeType().getName().equals(NT_TIMESERIES_ITEM)) {
			return new JCRTimeSeries(this, node);
		} else if (node.getPrimaryNodeType().getName().equals(NT_DOCUMENT_ITEM)) {
			return new JCRDocument(this, node);
		} else if (node.getPrimaryNodeType().getName().equals(NT_DOCUMENT_LINK_ITEM)) {
			return new JCRDocumentLink(this, node);
		} else if (node.getPrimaryNodeType().getName().equals(NT_IMAGE_DOCUMENT_ITEM)) {
			return new JCRImageDocument(this, node);
		} else if (node.getPrimaryNodeType().getName().equals(NT_IMAGE_DOCUMENT_LINK_ITEM)) {
			return new JCRImageDocumentLink(this, node);
		} else if (node.getPrimaryNodeType().getName().equals(NT_PDF_DOCUMENT_ITEM)) {
			return new JCRPDFDocument(this, node);
		} else if (node.getPrimaryNodeType().getName().equals(NT_PDF_DOCUMENT_LINK_ITEM)) {
			return new JCRPDFDocumentLink(this, node);	
		} else if (node.getPrimaryNodeType().getName().equals(NT_URL_DOCUMENT_ITEM)) {
			return new JCRUrlDocument(this, node);
		} else if (node.getPrimaryNodeType().getName().equals(NT_ANNOTATION_ITEM)) {
			return new JCRAnnotation(this, node);
		} else if (node.getPrimaryNodeType().getName().equals(NT_METADATA_ITEM)) {
			return new JCRMetadata(this, node);
		} else if (node.getPrimaryNodeType().getName().equals(NT_WORKSPACE_FOLDER_ITEM)) {
			return new JCRWorkspaceSmartFolder(this,node);
		} else if (node.getPrimaryNodeType().getName().equals(NT_EXTERNAL_RESOURCE_LINK_ITEM)) {
			return new JCRExternalResourceLink(this, node);
		}
	
		throw new InternalErrorException("JCR node type unknow");
	}
	
	
	public FolderItemType getFolderItemType(Node node) throws RepositoryException {
		
		String nodeType = node.getPrimaryNodeType().getName();
		
		if (nodeType.equals(NT_WORKSPACE_FILE)) {
			return FolderItemType.EXTERNAL_FILE;
		} else if (nodeType.equals(NT_WORKSPACE_IMAGE)) {
			return FolderItemType.EXTERNAL_IMAGE; 
		} else if (nodeType.equals(NT_WORKSPACE_PDF_FILE)) {
			return FolderItemType.EXTERNAL_PDF_FILE; 
		} else if (nodeType.equals(NT_WORKSPACE_URL)) {
			return FolderItemType.EXTERNAL_URL;
		} else if (nodeType.equals(NT_WORKSPACE_REPORT)) {
			return FolderItemType.REPORT;
		} else if (nodeType.equals(NT_WORKSPACE_REPORT_TEMPLATE)) {
			return FolderItemType.REPORT_TEMPLATE;
		} else if (nodeType.equals(NT_WORKSPACE_WORKFLOW_REPORT)) {
			return FolderItemType.WORKFLOW_REPORT;
		} else if (nodeType.equals(NT_WORKSPACE_WORKFLOW_TEMPLATE)) {
			return FolderItemType.WORKFLOW_TEMPLATE;
		} else if (nodeType.equals(NT_AQUAMAPS_ITEM)) {
			return FolderItemType.AQUAMAPS_ITEM;
		} else if (nodeType.equals(NT_QUERY)) {
			return FolderItemType.QUERY;
		} else if (nodeType.equals(NT_TIMESERIES_ITEM)) {
			return FolderItemType.TIME_SERIES;
		} else if (nodeType.equals(NT_DOCUMENT_ITEM)) {
			return FolderItemType.DOCUMENT;
		} else if (nodeType.equals(NT_DOCUMENT_LINK_ITEM)) {
			return FolderItemType.DOCUMENT_LINK;
		} else if (nodeType.equals(NT_IMAGE_DOCUMENT_ITEM)) {
			return FolderItemType.IMAGE_DOCUMENT;
		} else if (nodeType.equals(NT_IMAGE_DOCUMENT_LINK_ITEM)) {
			return FolderItemType.IMAGE_DOCUMENT_LINK;
		} else if (nodeType.equals(NT_PDF_DOCUMENT_ITEM)) {
			return FolderItemType.PDF_DOCUMENT;
		} else if (nodeType.equals(NT_PDF_DOCUMENT_LINK_ITEM)) {
			return FolderItemType.PDF_DOCUMENT_LINK;	
		} else if (nodeType.equals(NT_URL_DOCUMENT_ITEM)) {
			return FolderItemType.URL_DOCUMENT;
		} else if (nodeType.equals(NT_ANNOTATION_ITEM)) {
			return FolderItemType.ANNOTATION;
		} else if (nodeType.equals(NT_METADATA_ITEM)) {
			return FolderItemType.METADATA_LINK;
		} else if (nodeType.equals(NT_EXTERNAL_RESOURCE_LINK_ITEM))
			return FolderItemType.EXTERNAL_RESOURCE_LINK;
		
		return null; 
	}

	public JCRAbstractWorkspaceFolder getParent(Node node) throws RepositoryException, InternalErrorException {
		
		Node parent = node.getParent();
		if (parent.getPrimaryNodeType().getName().equals(NT_WORKSPACE_FOLDER)) 
			return new JCRWorkspaceFolder(this, parent);
		else if (parent.getPrimaryNodeType().getName().equals(NT_WORKSPACE_SHARED_FOLDER))
			return new JCRWorkspaceSharedFolder(this, parent);
		else 
			return null;
	}
	
	public JCRFile getGCUBEDocumentContent(Session session, String oid, ContentType contentType) throws RepositoryException {
		
		Node node = JCRRepository.getGCubeRoot(session).getNode(Text.escapeIllegalJcrChars(oid));
		switch(contentType) {
		case GENERAL:
			return new JCRFile(node.getNode(CONTENT));
		case IMAGE: 
			return new JCRImage(node.getNode(CONTENT));
		case PDF:
			return new JCRPDFFile(node.getNode(CONTENT));
		default:
			return null;
		}
	}
	
	public JCRFile setGCUBEDocumentContent(Session session, String oid,InputStream data, String mimeType,
			ContentType contentType) throws IOException, ItemExistsException, RepositoryException, RemoteBackendException{
		
		Node parent = JCRRepository.getGCubeRoot(session);
		Node nodeFile = parent.addNode(Text.escapeIllegalJcrChars(oid),NT_FILE);
		
		JCRFile file = null;
		switch(contentType) {
		case GENERAL:
			file = new JCRFile(nodeFile.addNode(CONTENT,ContentType.GENERAL.toString()),mimeType,data);
			break;
		case IMAGE: 
			file = new JCRImage(nodeFile.addNode(CONTENT,ContentType.IMAGE.toString()),data);
			break;
		case PDF:
			file = new JCRPDFFile(nodeFile.addNode(CONTENT,ContentType.PDF.toString()),mimeType,data);
			break;
		default:
			break;
		}
		
		return file;
	}
	
	private String isValidSearchResult(Node node) {
		
		String portalLogin = getHome().getOwner().getPortalLogin();
		String sharePath = JCRRepository.PATH_SEPARATOR + JCRRepository.SHARED_FOLDER;
		String userPath = JCRRepository.PATH_SEPARATOR + JCRRepository.HOME_FOLDER +
				JCRRepository.PATH_SEPARATOR + portalLogin;
			
		try {
			String nodePath = node.getPath();
			if (nodePath.startsWith(userPath))
				return node.getProperty(JCRWorkspaceItem.TITLE).getString();
			
			if (nodePath.startsWith(sharePath)) {
				
				Node sharedNode = (Node) node.getAncestor(2);
				Node usersNode = sharedNode.getNode(JCRWorkspaceSharedFolder.USERS);
				for (PropertyIterator iterator = usersNode.getProperties(); iterator.hasNext();) {
					Property property  = iterator.nextProperty();
					String name = property.getName();
					if (name.equals(portalLogin)) {
						if (node.getPath().equals(sharedNode.getPath())) {
							String[] values = property.getValue().getString().
									split(JCRRepository.PATH_SEPARATOR);
							return values[1];
									
						} else {
							return node.getName();
						}
					}
				} 
			}	
			return null;
		} catch (RepositoryException e) {
			return null;
		}
	}


	@Override
	public List<SearchItem> searchByName(String name) throws InternalErrorException {
		
		Session session = JCRRepository.getSession();
		NodeIterator iterator;
		List<SearchItem> list = null;
		try {
			QueryManager queryManager = session.getWorkspace().getQueryManager();	
			
			String userPath = "/Home/" + getHome().getOwner().getPortalLogin();

			
			String path = userPath +"/Workspace";
			String sql2= "SELECT * FROM [nthl:workspaceItem] AS node WHERE ISDESCENDANTNODE('" + path + "')" +
					" AND (LOCALNAME(node) LIKE '%"+ name + "%')";
			javax.jcr.query.Query q = queryManager.createQuery(sql2,javax.jcr.query.Query.JCR_SQL2);
			
			QueryResult result = q.execute();
			iterator = result.getNodes();

			list = new LinkedList<SearchItem>();
			while (iterator != null && iterator.hasNext()) {
				Node node = iterator.nextNode();
				
				String itemName = isValidSearchResult(node);
				if (itemName == null || !itemName.contains(name)) {
					logger.trace("Search result is not valid :" + node.getPath());
					continue;
				}
				
				try {
					if (node.getPrimaryNodeType().getName().equals(NT_WORKSPACE_FOLDER) ||
							node.getPrimaryNodeType().getName().equals(NT_WORKSPACE_SHARED_FOLDER)) {
						list.add(new JCRSearchFolder(node,itemName));
					} else {
						list.add(new JCRSearchFolderItem(node, getFolderItemType(node), itemName));
					}
				} catch (Exception e) {
					
				}
			}

		} catch (Exception e) {
			logger.error("Error searchByName ",e);
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
		
		return list;
	}

	
	
	@Override
	public List<WorkspaceItem> getWorkspaceTree(WorkspaceItem item) throws InternalErrorException {
		List<WorkspaceItem> listItems = new LinkedList<WorkspaceItem>();
		listItems.addAll(item.getChildren());
		for(WorkspaceItem child : item.getChildren() ) {
			listItems.addAll(getWorkspaceTree(child));
		}
		return listItems;
	}

	@Override
	public WorkspaceSmartFolder createSmartFolder(String name, String description,
			String query) throws ItemAlreadyExistException, InternalErrorException  {
		
		Session session = JCRRepository.getSession();
		try {
			Node node = repository.getRootSmartFolders(session).addNode(name, 
					NT_WORKSPACE_FOLDER_ITEM);
			JCRWorkspaceSmartFolder  folder = new JCRWorkspaceSmartFolder(this, node, name, description, query);
			folder.save(node);
			return folder;
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
		
	}
	
	@Override
	public List<WorkspaceSmartFolder> getAllSmartFolders() throws InternalErrorException {
		
		Session session = JCRRepository.getSession();
		List<WorkspaceSmartFolder> folders = new LinkedList<WorkspaceSmartFolder>();
		try {
			for (NodeIterator iterator = repository.getRootSmartFolders(session)
					.getNodes(); iterator.hasNext();) {
				Node node = iterator.nextNode();
				folders.add(new JCRWorkspaceSmartFolder(this, node));
			}
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
		return folders;
		
	}
	
	public WorkspaceSmartFolder getSmartFolder(String folderId) throws ItemNotFoundException,
	InternalErrorException  {
		
		Session session = JCRRepository.getSession();
		try {
			return new JCRWorkspaceSmartFolder(this, session.getNodeByIdentifier(folderId));
		} catch (javax.jcr.ItemNotFoundException e) {
			throw new ItemNotFoundException(e.getMessage());
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}
	

	@Override
	public List<SearchItem> getFolderItems(FolderItemType... types) throws InternalErrorException{
		
		List<SearchItem> list = new LinkedList<SearchItem>();
		for (FolderItemType folderItemType : types) {
			list.addAll(getFolderItems(folderItemType));
		}
		return list;
	}
	
	@Override
	public List<SearchItem> getFolderItems(FolderItemType type)
			throws InternalErrorException {
		
		Session session = JCRRepository.getSession();
		NodeIterator iterator = null;
		List<SearchItem> list = null;
		try {
			QueryManager queryManager = session.getWorkspace().getQueryManager();	
			javax.jcr.query.Query q = queryManager.createQuery("/jcr:root/Home/" + getHome().getOwner().getPortalLogin() +
					"/Workspace//element()[@hl:workspaceItemType = '"+ type.toString() +"']",javax.jcr.query.Query.XPATH);
			QueryResult result = q.execute();
			iterator = result.getNodes();
		
			list = new LinkedList<SearchItem>();
			while (iterator != null && iterator.hasNext()) {

				Node node = iterator.nextNode();
				String itemName = isValidSearchResult(node);
				if (itemName == null) {
					logger.trace("Search result is not valid :" + node.getPath());
					continue;
				}
				
				try {
					if (node.getPrimaryNodeType().getName().equals(NT_WORKSPACE_FOLDER) ||
							node.getPrimaryNodeType().getName().equals(NT_WORKSPACE_SHARED_FOLDER)) {
						list.add(new JCRSearchFolder(node,itemName));
					} else {
						list.add(new JCRSearchFolderItem(node, getFolderItemType(node), itemName));
					}
				} catch (Exception e) {
					
				}
			}
		} catch (Exception e) {
			logger.error("Error getFolderItems", e);
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
		return list;
	}

	@Override
	public List<SearchFolderItem> searchByMimeType(String mimeType)
	throws InternalErrorException {

		Session session = JCRRepository.getSession();
		NodeIterator iterator = null;
		List<SearchFolderItem> list = null;
		try {
			QueryManager queryManager = session.getWorkspace().getQueryManager();	
			javax.jcr.query.Query q = queryManager.createQuery("/jcr:root/Home/" + getHome().getOwner().getPortalLogin() +
					"/Workspace//element()[@jcr:mimeType = '"+ mimeType+"']",javax.jcr.query.Query.XPATH);
			QueryResult result = q.execute();
			iterator = result.getNodes();

			list = new LinkedList<SearchFolderItem>();
			while (iterator != null && iterator.hasNext()) {
				
				Node node = iterator.nextNode();
				String itemName = isValidSearchResult(node);
				if (itemName == null) {
					logger.trace("Search result is not valid :" + node.getPath());
					continue;
				}
				
				try {
						list.add(new JCRSearchFolderItem(node.getParent(), 
								getFolderItemType(node.getParent()),itemName));		
				} catch (RepositoryException e) {
					try {
						logger.error("Item " + node.getName() + " unknow");
					} catch (RepositoryException e1) {
						logger.error(e1);
					}
				}

			}
		} catch (Exception e) {
			logger.error("Error getFolderItems", e);
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
		return list;
	}
	
	private String getJCRWorkspaceItemType(FolderItemType type) {
		
		switch (type) {
		case ANNOTATION:
			return NT_ANNOTATION_ITEM;
		case ANNOTATION_LINK:
			return NT_ANNOTATION_LINK_ITEM;
		case AQUAMAPS_ITEM:
			return NT_AQUAMAPS_ITEM;
		case DOCUMENT:
			return NT_DOCUMENT_ITEM;
		case DOCUMENT_LINK:
			return NT_DOCUMENT_LINK_ITEM;
		case EXTERNAL_FILE:
			return NT_WORKSPACE_FILE;
		case EXTERNAL_IMAGE:
			return NT_WORKSPACE_IMAGE;
		case EXTERNAL_PDF_FILE:
			return NT_WORKSPACE_PDF_FILE;
		case EXTERNAL_URL:
			return NT_WORKSPACE_URL;
		case IMAGE_DOCUMENT:
			return NT_IMAGE_DOCUMENT_ITEM;
		case IMAGE_DOCUMENT_LINK:
			return NT_IMAGE_DOCUMENT_LINK_ITEM;
		case METADATA:
			return NT_METADATA_ITEM;
		case METADATA_LINK:
			return NT_METADATA_LINK_ITEM;
		case PDF_DOCUMENT:
			return NT_PDF_DOCUMENT_ITEM;
		case PDF_DOCUMENT_LINK:
			return NT_PDF_DOCUMENT_LINK_ITEM;
		case QUERY:
			return NT_QUERY;
		case REPORT:
			return NT_WORKSPACE_REPORT;
		case REPORT_TEMPLATE:
			return NT_WORKSPACE_REPORT_TEMPLATE;
		case TIME_SERIES:
			return NT_TIMESERIES_ITEM;
		case URL_DOCUMENT:
			return NT_URL_DOCUMENT_ITEM;
		case WORKFLOW_REPORT:
			return NT_WORKSPACE_WORKFLOW_REPORT;
		case WORKFLOW_TEMPLATE:
			return NT_WORKSPACE_WORKFLOW_TEMPLATE;	
		default:
			return null;
		}
		
	}


	@Override
	public GCUBEScope getScope() {
		if (scope == null)
			return null;
		
		return GCUBEScope.getScope(scope);
	}

	@Override
	public String getUrlWebDav() throws InternalErrorException {
		
		return repository.getUserHomeUrl(home.getOwner().getPortalLogin()) 
		+  getPathSeparator() + WORKSPACE_ROOT_FOLDER;
	}

	@Override
	public WorkspaceSharedFolder createSharedFolder(String name,
			String description, List<String> users, String destinationFolderId)
			throws InternalErrorException, InsufficientPrivilegesException,
			ItemAlreadyExistException, WrongDestinationException,
			ItemNotFoundException, WorkspaceFolderNotFoundException {
		
		logger.trace("Create workspace shared folder");
		
		Session session = JCRRepository.getSession();
		try {
			
			if (exists(name, destinationFolderId))
				throw new ItemAlreadyExistException("The item already exists");
			
			JCRWorkspaceItem item = (JCRWorkspaceItem)getItem(session,destinationFolderId);
			if (item.getType() != WorkspaceItemType.FOLDER)
				throw new WrongDestinationException("Destination is not a folder");
			
			if (item.isShared())
				throw new WrongDestinationException("Destination folder is already shared");
			
			Node sharedFolder = JCRRepository.getSharedRoot(session);
			Node node = sharedFolder.addNode(UUID.randomUUID().toString(),NT_WORKSPACE_SHARED_FOLDER);		
			
			JCRWorkspaceSharedFolder folder = new JCRWorkspaceSharedFolder(this, node, name, 
					description, destinationFolderId, users);
			folder.save(node);
			
			
			fireItemCreatedEvent(folder);
			return folder;
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} catch (WrongItemTypeException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}	
	}
	
	public WorkspaceSharedFolder shareFolder(List<String> users,
			String folderId)
			throws InternalErrorException, InsufficientPrivilegesException,
			WrongDestinationException, ItemNotFoundException,
			WorkspaceFolderNotFoundException {

					
		Session session = JCRRepository.getSession();
		try {

			Node nodeItemToShare = session.getNodeByIdentifier(folderId);
			
			JCRWorkspaceItem itemToShare = (JCRWorkspaceItem)getItem(session, folderId);
			if (itemToShare.getType() == WorkspaceItemType.SHARED_FOLDER) {
				WorkspaceSharedFolder sharedFolder = (WorkspaceSharedFolder)itemToShare; 
				sharedFolder.share(users);
				return sharedFolder;
			}
			
			if (itemToShare.getType() != WorkspaceItemType.FOLDER) {
				throw new WorkspaceFolderNotFoundException("The item to share is not a folder");
			}
			
			WorkspaceItem parentItem = itemToShare.getParent(nodeItemToShare);
			if (parentItem == null)
				throw new WrongDestinationException("The root can't be shared");
			
			String parentId = parentItem.getId();
			JCRWorkspaceItem destinationFolder = (JCRWorkspaceItem)getItem(session,parentId);
			if (destinationFolder.isShared())
				throw new WrongDestinationException("Destination folder is already shared");
			
			
			QueryManager queryManager = session.getWorkspace().getQueryManager();
			javax.jcr.query.Query q = queryManager.createQuery("/jcr:root/Home/" + getOwner().getPortalLogin()
					+ ISO9075.encodePath(itemToShare.getPath()) +
					"//element(*,nthl:workspaceSharedItem)",
					javax.jcr.query.Query.XPATH);
			QueryResult result = q.execute();
			if (result.getNodes().hasNext())
				throw new WrongDestinationException("Folder contains descendants already shared");
			
						
			String sharedFolderName = itemToShare.getName();
			String sharedFolderDescription = itemToShare.getDescription();
			
			Node sharedRootFolder = JCRRepository.getSharedRoot(session);
			Node sharedNode = sharedRootFolder.addNode(UUID.randomUUID().toString(),NT_WORKSPACE_SHARED_FOLDER);
			JCRWorkspaceSharedFolder sharedFolder = new JCRWorkspaceSharedFolder(this, sharedNode, sharedFolderName, 
					sharedFolderDescription, parentId, users);
			
			Node folderNode = session.getNodeByIdentifier(folderId);
			
			for (NodeIterator iterator = folderNode.getNodes();iterator.hasNext();) {
				Node node = iterator.nextNode();
				if (!node.getName().startsWith(JCRRepository.JCR_NAMESPACE)
						&& !node.getName().startsWith(JCRRepository.HL_NAMESPACE)) {
					session.getWorkspace().copy(node.getPath(), sharedNode.getPath() + 
							getPathSeparator() + node.getName());
				}
			}
			
			folderNode.remove();
			session.save();
			
			sharedFolder.save(sharedNode);
			
			fireItemCreatedEvent(sharedFolder);
			
			return sharedFolder;
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}	

		
		
	}

	@Override
	public TabularDataLink createTabularDataLink(String name,
			String description, String tableId, String template,
			String provenance, String runtimeResourceName, String destinationFolderId)
			throws InsufficientPrivilegesException, InternalErrorException,
			ItemAlreadyExistException, WrongDestinationException,
			WorkspaceFolderNotFoundException {
		
		logger.trace("Create tabular data link");
		
		Session session = JCRRepository.getSession();
		try {
			Node node = addChildNode(session, destinationFolderId, name, NT_EXTERNAL_RESOURCE_LINK_ITEM);
			JCRTabularDataLink item = new JCRTabularDataLink(this, node, name, description,
					tableId, template, provenance, runtimeResourceName);
			item.save(node);
						
			fireItemCreatedEvent(item);
			return item;
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	
}
