package org.gcube.portlets.user.homelibrary.jcr.sharing;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.LinkedList;
import java.util.List;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Value;

import org.apache.jackrabbit.util.Text;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.portlets.user.homelibrary.home.User;
import org.gcube.portlets.user.homelibrary.home.exceptions.InternalErrorException;
import org.gcube.portlets.user.homelibrary.home.workspace.WorkspaceFolder;
import org.gcube.portlets.user.homelibrary.home.workspace.WorkspaceItem;
import org.gcube.portlets.user.homelibrary.home.workspace.WorkspaceItemType;
import org.gcube.portlets.user.homelibrary.home.workspace.exceptions.ItemAlreadyExistException;
import org.gcube.portlets.user.homelibrary.home.workspace.exceptions.ItemNotFoundException;
import org.gcube.portlets.user.homelibrary.home.workspace.exceptions.WrongDestinationException;
import org.gcube.portlets.user.homelibrary.home.workspace.sharing.WorkspaceMessage;
import org.gcube.portlets.user.homelibrary.jcr.JCRUser;
import org.gcube.portlets.user.homelibrary.jcr.repository.JCRRepository;
import org.gcube.portlets.user.homelibrary.jcr.workspace.JCRWorkspace;
import org.gcube.portlets.user.homelibrary.jcr.workspace.JCRWorkspaceItem;
import org.gcube.portlets.user.homelibrary.util.WorkspaceUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JCRWorkspaceMessage implements WorkspaceMessage {

	private static Logger logger = LoggerFactory.getLogger(JCRWorkspaceMessage.class);

	public enum WorkspaceMessageType {
		RECEIVED,
		SENT
	};

	private static final String CREATED 		= "jcr:created";
	private static final String SUBJECT 		= "hl:subject";
	private static final String BODY 	  		= "hl:body";	
	private static final String ATTACHMENTS   	= "hl:attachments";
	private static final String ADDRESSES		= "hl:addresses";

	private static final String TITLE 			= "jcr:title"; 
	private static final String OWNER			= "hl:owner";
	private static final String SCOPE           = "hl:scope";
	private static final String PORTAL_LOGIN    = "hl:portalLogin";
	private static final String USER_ID 		= "hl:uuid";
	private static final String READ			= "hl:read";
	private static final String OPEN			= "hl:open";
	private static final String NT_USER			= "nthl:user";

	private static final String NT_WORKSPACE_FOLDER_ITEM			= "nthl:workspaceLeafItem";

	private final JCRWorkspace workspace;
	private final String id;
	private final String subject;
	private final String body;
	private final User user;
	private final Calendar date;
	private List<String> attachments;
	private List<String> copyAttachmentIds;
	private final List<String> addresses;
	private boolean isRead;
	private boolean isOpened;
	private final WorkspaceMessageType type;


	public JCRWorkspaceMessage(JCRWorkspace workspace, Node node, WorkspaceMessageType type) throws RepositoryException {

		super();

		this.type = type;
		this.workspace = workspace;
		this.id = node.getName();


		this.subject = node.getProperty(SUBJECT).getString();
		this.body = node.getProperty(BODY).getString();
		this.isRead = node.getProperty(READ).getBoolean();
		this.isOpened = node.getProperty(OPEN).getBoolean();

		this.date = node.getProperty(CREATED).getDate();
		Node userNode = node.getNode(OWNER);
		this.user = new JCRUser(userNode.getProperty(USER_ID).getString(),
				userNode.getProperty(PORTAL_LOGIN).getString(),
				GCUBEScope.getScope(userNode.getProperty(SCOPE).getString()));


		List<String> attachments = new LinkedList<String>();
		Node attachmentsNode = node.getNode(ATTACHMENTS);
		for(NodeIterator iterator = attachmentsNode.getNodes(); iterator.hasNext();) {
			attachments.add(iterator.nextNode().getIdentifier());
		}

		this.attachments = attachments;

		List<String> addresses = new LinkedList<String>();
		for(Value user : node.getProperty(ADDRESSES).getValues()) {
			addresses.add(user.getString());
		}

		this.addresses = addresses;

	}



	public JCRWorkspaceMessage(JCRWorkspace workspace, Node node, WorkspaceMessageType type, String messageId, String subject, String body,
			User sender, List<String> attachmentIds, List<String> addresses, Node nodeHiddenFolder) throws  RepositoryException,
			InternalErrorException {

		this.type = type;
		this.workspace = workspace;
		this.id = messageId;
		this.subject = subject;
		this.body = body;
		this.user = sender;
		this.isRead = false;
		this.isOpened = false;
		this.date = Calendar.getInstance();
		this.addresses = addresses;

		this.copyAttachmentIds = new ArrayList<String>();

		node.setProperty(SUBJECT, subject);
		node.setProperty(BODY, body);
		node.setProperty(READ, false);
		node.setProperty(OPEN, false);

		Node ownerNode =  node.getNode(OWNER);
		ownerNode.setProperty(USER_ID,user.getId());
		ownerNode.setProperty(SCOPE, user.getScope().toString());
		ownerNode.setProperty(PORTAL_LOGIN, user.getPortalLogin());

		node.setProperty(ADDRESSES, addresses.toArray(new String[addresses.size()]));
		Node rootAttachments = node.getNode(ATTACHMENTS);

		node.getSession().save();

		Session session = node.getSession();

		List<User> users = new LinkedList<User>();
		for (String address : addresses) {
			User user = workspace.getHome().getHomeManager().getUser(address);
			if(user != null)
				users.add(user);
		}

		logger.info("attachmentIds.size() " + attachmentIds.size());
		for(String attachmentId : attachmentIds) {
			Node nodeItem = session.getNodeByIdentifier(attachmentId);
			WorkspaceItem item = workspace.getWorkspaceItem(nodeItem);			
			if(item.getType().equals(WorkspaceItemType.FOLDER_ITEM)){
				try {
					logger.info("nodeItem.getName() " + nodeItem.getName());
					Node newNode = ((JCRWorkspaceItem)item).internalCopy(rootAttachments, nodeItem.getName());
					session.save();
					if (this.type.equals(WorkspaceMessageType.SENT)){			
						//copy all remote content item child nodes.
						workspace.copyRemoteContent(newNode, null);
						session.save();		
						logger.info("add new id to copyAttachmentIds " + newNode.getIdentifier());
						copyAttachmentIds.add(newNode.getIdentifier());

					} else if (this.type.equals(WorkspaceMessageType.RECEIVED)){


						//create hardlink

						String hardLinkRemotePath = node.getPath() + "/"+ nodeItem.getName();	
						//						logger.info("WorkspaceMessageType.RECEIVED, messageId: " + messageId);
						workspace.setHardLink(newNode, hardLinkRemotePath);
						session.save();
					}

					//					System.out.println("nodeHiddenFolder "+ nodeHiddenFolder.getPrimaryNodeType().getName());					
					//					nodeHiddenFolder.addNode(newNode.getIdentifier(),"nt:hiddenFolderContent" );

					session.save();

					workspace.fireItemSentEvent(item, users);	

				} catch (ItemAlreadyExistException e) {
					throw new InternalErrorException(e);

				} catch (WrongDestinationException e) {
					throw new InternalErrorException(e);

				}
			}
		}
	}


	@Override
	public String getId() {
		return id;
	}

	@Override
	public User getSender() {
		return user;
	}

	@Override
	public Calendar getSendTime() {
		return date;
	}

	@Override
	public String getSubject() {
		return subject;
	}

	@Override
	public String getBody() {
		return body;
	}

	@Override
	public List<String> getAttachmentsIds() {
		return attachments;
	}

	@Override
	public List<String> getCopyAttachmentsIds() {
		return copyAttachmentIds;
	}

	@Override
	public boolean isRead() {
		return isRead;
	}

	@Override
	public void open() throws InternalErrorException {

		this.isOpened = true;

		Session session = JCRRepository.getSession();
		Node root = null;
		try {

			switch (type) {
			case RECEIVED:
				root = workspace.getRepository().
				getOwnInBoxFolder(session);
				break;
			case SENT:
				root = workspace.getRepository().
				getOutBoxFolder(session);
				break;
			}

			Node node = root.getNode(id);
			node.setProperty(OPEN, true);
			session.save();
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public void setStatus(boolean status) throws InternalErrorException {
		this.isRead = status;

		Session session = JCRRepository.getSession();
		Node root = null;
		try {

			switch (type) {
			case RECEIVED:
				root = workspace.getRepository().
				getOwnInBoxFolder(session);
				break;
			case SENT:
				root = workspace.getRepository().
				getOutBoxFolder(session);
				break;
			}
			Node node = root.getNode(id);
			node.setProperty(READ, status);
			session.save();
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}

	}

	@Override
	public void saveAttachments(String destinationFolderId) throws InternalErrorException, 
	WrongDestinationException, ItemNotFoundException {

		Session session = JCRRepository.getSession();
		try {

			Node folderNode;
			WorkspaceFolder folder;
			try {
				folderNode = session.getNodeByIdentifier(destinationFolderId);
				folder = (WorkspaceFolder)workspace.getItem(session, destinationFolderId);
			} catch (RepositoryException e) {
				throw new ItemNotFoundException(e.getMessage());
			} catch (InternalErrorException e) {
				throw new WrongDestinationException(e.getMessage());
			}

			for(String attachmentId : attachments) {
				saveAttachment(session, attachmentId, folder, folderNode);
			}		
		} finally {
			session.logout();
		}

	}

	@Override
	public WorkspaceItem saveAttachment(String attachmentId, String destinationFolderId) throws InternalErrorException, 
	WrongDestinationException, ItemNotFoundException {

		Session session = JCRRepository.getSession();
		try {
			return saveAttachment(session, attachmentId, destinationFolderId);
		} finally {
			session.logout();
		}

	}

	private WorkspaceItem saveAttachment(Session session, String attachmentId,
			String destinationFolderId) throws ItemNotFoundException,
			WrongDestinationException, InternalErrorException {

		Node folderNode; 
		WorkspaceFolder folder;
		try {
			folderNode = session.getNodeByIdentifier(destinationFolderId);	
			folder = (WorkspaceFolder)workspace.getItem(session, destinationFolderId);
		} catch (RepositoryException e) {
			throw new ItemNotFoundException(e.getMessage());
		} catch (InternalErrorException e) {
			throw new WrongDestinationException(e.getMessage());
		}
		return saveAttachment(session, attachmentId, folder, folderNode);
	}

	private WorkspaceItem saveAttachment(Session session, String attachmentId,
			WorkspaceFolder folder, Node folderNode) throws ItemNotFoundException,
			WrongDestinationException, InternalErrorException {

		try {

			Node attachment = session.getNodeByIdentifier(attachmentId);
			// TODO should implement a new method getUniqueName without open a new session
			String name = WorkspaceUtil.getUniqueName(attachment.getProperty(TITLE).getString(),
					folder);
			String pathDestination =  folderNode.getPath() 
					+ workspace.getPathSeparator() + Text.escapeIllegalJcrChars(name);

			session.getWorkspace().copy(attachment.getPath(), pathDestination);

			//Save the new name to attachment node, useful at the method getUniqueName()
			Node itemSaved = session.getNode(pathDestination);
			itemSaved.setProperty(TITLE, name);
			session.save();

			//every download creates a new copy
			//copy all remote content item child nodes.
			workspace.copyRemoteContent(itemSaved,folderNode);
			session.save();

			
			//update owner node
			Node nodeOwner = null;
			try{
				nodeOwner = itemSaved.getNode(OWNER);
			}catch (Exception e) {
				nodeOwner = itemSaved.addNode(OWNER,NT_USER);

			}
			nodeOwner.setProperty(SCOPE,(workspace.getOwner().getScope() != null)?workspace.getOwner().getScope().toString():null);
			nodeOwner.setProperty(PORTAL_LOGIN,workspace.getOwner().getPortalLogin());
			nodeOwner.setProperty(USER_ID,workspace.getOwner().getId());
			session.save();
			
			return workspace.getItem(session,itemSaved.getIdentifier());
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} 
	}

	@Override
	public List<WorkspaceItem> getAttachments() throws InternalErrorException {

		List<WorkspaceItem> list = new LinkedList<WorkspaceItem>();
		Session session = JCRRepository.getSession();
		try {
			for(String id : attachments) {
				list.add(workspace.getItem(session, id));
			}
		} catch (Exception e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
		return list;
	}

	@Override
	public List<String> getAddresses() {
		return this.addresses;
	}

	public boolean isOpened() {
		return isOpened;
	}






}
