/**
 * 
 */
package org.gcube.portlets.user.td.gwtservice.server;

import java.io.InputStream;
import java.nio.charset.Charset;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpSession;

import org.gcube.application.framework.core.session.ASLSession;
import org.gcube.common.homelibrary.home.Home;
import org.gcube.common.homelibrary.home.HomeLibrary;
import org.gcube.common.homelibrary.home.HomeManager;
import org.gcube.common.homelibrary.home.HomeManagerFactory;
import org.gcube.common.homelibrary.home.exceptions.InternalErrorException;
import org.gcube.common.homelibrary.home.workspace.Workspace;
import org.gcube.common.homelibrary.home.workspace.WorkspaceItem;
import org.gcube.data.analysis.tabulardata.metadata.NoSuchMetadataException;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.metadata.common.DescriptionsMetadata;
import org.gcube.data.analysis.tabulardata.model.metadata.common.LocalizedText;
import org.gcube.data.analysis.tabulardata.model.metadata.common.NamesMetadata;
import org.gcube.data.analysis.tabulardata.model.metadata.table.ExportMetadata;
import org.gcube.data.analysis.tabulardata.model.metadata.table.GenericMapMetadata;
import org.gcube.data.analysis.tabulardata.model.metadata.table.ImportMetadata;
import org.gcube.data.analysis.tabulardata.model.metadata.table.TableMetadata;
import org.gcube.data.analysis.tabulardata.model.metadata.table.VersionMetadata;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.model.table.TableId;
import org.gcube.data.analysis.tabulardata.operation.OperationDescriptor;
import org.gcube.data.analysis.tabulardata.operation.invocation.InvocationCreator;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;

import org.gcube.data.analysis.tabulardata.service.TabularDataService;
import org.gcube.data.analysis.tabulardata.service.exception.InvalidTabularResourceException;
import org.gcube.data.analysis.tabulardata.service.exception.NoSuchTabularResourceException;
import org.gcube.data.analysis.tabulardata.service.impl.TabularDataServiceFactory;
import org.gcube.data.analysis.tabulardata.service.operation.OperationInterface;
import org.gcube.data.analysis.tabulardata.service.operation.Task;
import org.gcube.data.analysis.tabulardata.service.operation.TaskStatus;
import org.gcube.data.analysis.tabulardata.service.tabular.TabularResource;
import org.gcube.data.analysis.tabulardata.service.tabular.TabularResourceId;
import org.gcube.data.analysis.tabulardata.service.tabular.TabularResourceInterface;
import org.gcube.data.analysis.tabulardata.service.tabular.metadata.AgencyMetadata;
import org.gcube.data.analysis.tabulardata.service.tabular.metadata.CreationDateMetadata;
import org.gcube.data.analysis.tabulardata.service.tabular.metadata.DescriptionMetadata;
import org.gcube.data.analysis.tabulardata.service.tabular.metadata.NameMetadata;
import org.gcube.data.analysis.tabulardata.service.tabular.metadata.RightsMetadata;
import org.gcube.datapublishing.sdmx.api.model.SDMXRegistryDescriptor;
import org.gcube.datapublishing.sdmx.api.model.SDMXRegistryInterfaceType;
import org.gcube.datapublishing.sdmx.impl.model.GCubeSDMXRegistryDescriptor;
import org.gcube.portlets.user.td.gwtservice.client.rpc.TDGWTService;
import org.gcube.portlets.user.td.gwtservice.server.file.FileUploadSession;
import org.gcube.portlets.user.td.gwtservice.server.file.FileUtil;
import org.gcube.portlets.user.td.gwtservice.server.storage.FilesStorage;
import org.gcube.portlets.user.td.gwtservice.server.trservice.OperationsId;
import org.gcube.portlets.user.td.gwtservice.shared.csv.AvailableCharsetList;
import org.gcube.portlets.user.td.gwtservice.shared.csv.CSVFileUtil;
import org.gcube.portlets.user.td.gwtservice.shared.csv.CSVImportMonitor;
import org.gcube.portlets.user.td.gwtservice.shared.csv.CSVImportSession;
import org.gcube.portlets.user.td.gwtservice.shared.csv.CSVParserConfiguration;
import org.gcube.portlets.user.td.gwtservice.shared.csv.CSVRowError;
import org.gcube.portlets.user.td.gwtservice.shared.exception.TDGWTServiceException;
import org.gcube.portlets.user.td.gwtservice.shared.file.FileUploadMonitor;
import org.gcube.portlets.user.td.gwtservice.shared.file.FileUploadState;
import org.gcube.portlets.user.td.gwtservice.shared.file.HeaderPresence;
import org.gcube.portlets.user.td.gwtservice.shared.sdmx.SDMXImportMonitor;
import org.gcube.portlets.user.td.gwtservice.shared.sdmx.SDMXImportSession;
import org.gcube.portlets.user.td.gwtservice.shared.tr.ColumnData;
import org.gcube.portlets.user.td.gwtservice.shared.tr.TRId;
import org.gcube.portlets.user.td.gwtservice.shared.tr.TabResource;
import org.gcube.portlets.user.td.gwtservice.shared.tr.TableData;
import org.gcube.portlets.user.td.gwtservice.shared.tr.metadata.TRDescriptionsMetadata;
import org.gcube.portlets.user.td.gwtservice.shared.tr.metadata.TRExportMetadata;
import org.gcube.portlets.user.td.gwtservice.shared.tr.metadata.TRGenericMapMetadata;
import org.gcube.portlets.user.td.gwtservice.shared.tr.metadata.TRImportMetadata;
import org.gcube.portlets.user.td.gwtservice.shared.tr.metadata.TRLocalizedText;
import org.gcube.portlets.user.td.gwtservice.shared.tr.metadata.TRMetadata;
import org.gcube.portlets.user.td.gwtservice.shared.tr.metadata.TRNamesMetadata;
import org.gcube.portlets.user.td.gwtservice.shared.tr.metadata.TRVersionMetadata;
import org.gcube.portlets.user.td.gwtservice.shared.tr.open.TDOpenSession;
import org.gcube.portlets.user.td.gwtservice.shared.tr.type.Agencies;
import org.gcube.portlets.user.td.gwtservice.shared.tr.type.Codelist;
import org.gcube.portlets.user.td.gwtservice.shared.tr.type.Dataset;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.gwt.user.server.rpc.RemoteServiceServlet;

/**
 * 
 * @author "Giancarlo Panichi" <a
 *         href="mailto:g.panichi@isti.cnr.it">g.panichi@isti.cnr.it</a>
 * 
 */
public class TDGWTServiceImpl extends RemoteServiceServlet implements
		TDGWTService {

	private static final long serialVersionUID = -5707400086333186368L;
	protected static Logger logger = LoggerFactory
			.getLogger(TDGWTServiceImpl.class);

	protected static SimpleDateFormat sdf = new SimpleDateFormat("yyyy/MM/dd");
	protected TabularDataService service;
	protected HttpSession session;
	protected ASLSession aslSession;

	/**
	 * 
	 * {@inheritDoc}
	 */
	@Override
	public void setTabResource(TabResource tabResource)
			throws TDGWTServiceException {
		try {
			session = this.getThreadLocalRequest().getSession();
			if (tabResource == null) {
				logger.error("Error setting TabResource: null");
				throw new TDGWTServiceException(
						"Error setting TabResource: null");
			}

			SessionUtil.setTabResource(session, tabResource);
			return;

		} catch (Exception e) {
			logger.error(
					"Error setting TabResource parameter: "
							+ e.getLocalizedMessage(), e);
			throw new TDGWTServiceException(
					"Error setting TabResource parameter: "
							+ e.getLocalizedMessage());
		}

	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TabResource getTabResourceInformation() throws TDGWTServiceException {
		try {
			session = this.getThreadLocalRequest().getSession();
			aslSession = SessionUtil.getAslSession(session);

			TabResource currentTR = SessionUtil.getTabResource(session);
			if (currentTR == null) {
				logger.error("CURRENT_TABULAR_RESOURCE is null");
				throw new TDGWTServiceException(
						"CURRENT_TABULAR_RESOURCE is null");
			}

			return currentTR;

		} catch (Exception e) {
			logger.error(
					"Error setting TabResource parameter: "
							+ e.getLocalizedMessage(), e);
			throw new TDGWTServiceException(
					"Error setting TabResource parameter: "
							+ e.getLocalizedMessage());
		}
	}

	@Override
	public TableData getLastTable(TRId trId) throws TDGWTServiceException {
		try {
			session = this.getThreadLocalRequest().getSession();
			aslSession = SessionUtil.getAslSession(session);

			service = TabularDataServiceFactory.getService(aslSession
					.getUsername());

			Table table = service.getLastTable(new TabularResourceId(Long
					.valueOf(trId.getId())));

			TableData tData = new TableData();
			trId.setTableId(String.valueOf(table.getId().getValue()));
			tData.setTrId(trId);
			tData.setName(table.getName());
			tData.setTypeName(table.getTableType().getName());
			tData.setTypeCode(table.getTableType().getCode());
			Collection<TableMetadata> cMeta=table.getAllMetadata();
			
			tData.setMetaData(cMeta.toString());
			
			
			ArrayList<ColumnData> lColumnData = new ArrayList<ColumnData>();
			for (Column column : table.getColumns()) {
				ColumnData colData = new ColumnData();
				colData.setName(column.getName());
				colData.setTypeName(column.getColumnType().getName());
				colData.setTypeCode(column.getColumnType().getCode());
				lColumnData.add(colData);
			}

			tData.setListColumnData(lColumnData);

			return tData;

		} catch (Exception e) {
			logger.error("Error in getLastTable(): " + e.getLocalizedMessage(),
					e);
			throw new TDGWTServiceException("Error in getLastTable(): "
					+ e.getLocalizedMessage());
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setTDOpenSession(TDOpenSession s) throws TDGWTServiceException {
		try {
			session = this.getThreadLocalRequest().getSession();
			SessionUtil.setTDOpenSession(session, s);
			return;

		} catch (Exception e) {
			logger.error("Error setting TDOpenSession parameter: "
					+ e.getLocalizedMessage());
			throw new TDGWTServiceException(
					"Error setting TDOpenSession parameter: "
							+ e.getLocalizedMessage());
		}
	}

	/**
	 * 
	 * 
	 * @param tr
	 * @param tabResource
	 */
	protected void syncTRMetaData(TabularResource tr, TabResource tabResource) {
		Date date = Calendar.getInstance().getTime();
		tabResource.setDate(sdf.format(date));
		tr.setMetadata(new NameMetadata(tabResource.getName()));
		tr.setMetadata(new DescriptionMetadata(tabResource.getDescription()));
		tr.setMetadata(new CreationDateMetadata(date));
		if (tabResource.getAgency() != null
				&& !tabResource.getAgency().isEmpty()) {
			tr.setMetadata(new AgencyMetadata(tabResource.getAgency()));
		}
		tr.setMetadata(new RightsMetadata(tabResource.getRight()));

	}

	/**
	 * 
	 * @param tr
	 * @throws TDGWTServiceException
	 */
	protected void retriveTRMetadataFromService(TabularResource tr)
			throws TDGWTServiceException {
		retriveTRMetadataFromService(tr, 0);
	}

	/**
	 * Retrieves the informations for a specific Tabular Resource from service
	 * 
	 * @param tr
	 *            TabularResource on service
	 * @param i
	 *            Index on Grid
	 * @return TabResource for Grid
	 * @throws TDGWTServiceException
	 */
	protected TabResource retriveTRMetadataFromService(TabularResource tr, int i)
			throws TDGWTServiceException {

		NameMetadata nameMeta;
		RightsMetadata rightsMeta;
		CreationDateMetadata creationDateMeta;
		AgencyMetadata agencyMeta;
		DescriptionMetadata descriptionMeta;
		try {
			nameMeta = tr.getMetadata(NameMetadata.class);
			creationDateMeta = tr.getMetadata(CreationDateMetadata.class);
			agencyMeta = tr.getMetadata(AgencyMetadata.class);
			descriptionMeta = tr.getMetadata(DescriptionMetadata.class);
			rightsMeta = tr.getMetadata(RightsMetadata.class);
		} catch (NoSuchMetadataException e) {
			e.printStackTrace();
			throw new TDGWTServiceException(
					"Error retriving metadata from service: "
							+ e.getLocalizedMessage());
		}

		if (nameMeta == null) {
			throw new TDGWTServiceException(
					"Error retriving metadata from service: TR nameMeta is null");
		}
		String nameTR = nameMeta.getValue();
		String descriptionTR = "";
		String agencyTR = "";
		String rightsTR = "";
		String dateS = "";

		if (descriptionMeta != null) {
			descriptionTR = descriptionMeta.getValue();
		}
		if (agencyMeta != null) {
			agencyTR = agencyMeta.getValue();
		}
		if (rightsMeta != null) {
			rightsTR = rightsMeta.getValue();
		}
		if (creationDateMeta != null) {
			Date dateTR = creationDateMeta.getValue();
			dateS = sdf.format(dateTR);
		}

		Table table = null;
		try {
			table = service.getLastTable(tr.getId());
		} catch (NoSuchTabularResourceException e) {
			e.printStackTrace();
		} catch (InvalidTabularResourceException e) {
			e.printStackTrace();
		}

		TableId tableId = null;

		if (table == null) {
			logger.info("Tabular resource " + tr.getId() + " has no table.");
		} else {
			tableId = table.getId();
		}

		TRId trId = null;
		if (tableId == null) {
			logger.info("Tabular Resource " + tr.getId()
					+ " has last table with id null.");
		} else {
			trId = new TRId(String.valueOf(tr.getId().getValue()),
					String.valueOf(tableId.getValue()));
		}

		TabResource t = new TabResource(i, nameTR, descriptionTR, agencyTR,
				dateS, rightsTR, trId);
		return t;
	}

	/**
	 * 
	 * {@inheritDoc}
	 */
	@Override
	public ArrayList<TabResource> getTabularResources()
			throws TDGWTServiceException {
		try {
			session = this.getThreadLocalRequest().getSession();
			aslSession = SessionUtil.getAslSession(session);
			service = TabularDataServiceFactory.getService(aslSession
					.getUsername());

			List<TabularResource> trs = service.getTabularResources();
			SessionUtil.setTabularResources(session, trs);

			ArrayList<TabResource> ltr = new ArrayList<TabResource>();

			int i;
			for (i = 0; i < trs.size(); i++) {
				TabularResource tr = trs.get(i);
				TabResource t = retriveTRMetadataFromService(tr, i);
				if (t.getTrId() != null) {
					ltr.add(t);
				}
			}

			logger.debug("Tabular Resources retrived: " + ltr);
			return ltr;
		} catch (Exception e) {
			e.printStackTrace();
			logger.error("Error retrieving TabularResources: "
					+ e.getLocalizedMessage());
			throw new TDGWTServiceException(
					"Error retrieving TabularResources: "
							+ e.getLocalizedMessage());
		}

	}

	/**
	 * 
	 * @param tabResource
	 * @throws TDGWTServiceException
	 */
	public void removeTabularResource(TabResource tabResource)
			throws TDGWTServiceException {
		try {
			session = this.getThreadLocalRequest().getSession();
			aslSession = SessionUtil.getAslSession(session);

			if (tabResource == null) {
				logger.error("Error removing TabularResource: tabResource is null");
				throw new TDGWTServiceException(
						"Error removing TabularResource no parameters set");
			}
			service = TabularDataServiceFactory.getService(aslSession
					.getUsername());
			TabularResourceId id = new TabularResourceId(
					Long.valueOf(tabResource.getTrId().getId()));
			service.removeTabularResource(id);
			return;
		} catch (Exception e) {
			logger.error(
					"Error removing TabularResource: "
							+ e.getLocalizedMessage(), e);
			throw new TDGWTServiceException("Error removing TabularResource: "
					+ e.getLocalizedMessage());
		}
	}

	/**
	 * 
	 * {@inheritDoc}
	 */
	@Override
	public TabResource createTabularResource(TabResource tabResource)
			throws TDGWTServiceException {
		try {
			session = this.getThreadLocalRequest().getSession();
			aslSession = SessionUtil.getAslSession(session);

			if (tabResource == null) {
				logger.error("Error creating new TabularResource: tabResource is null");
				throw new TDGWTServiceException(
						"Error creating new TabularResource no parameters set");
			}
			service = TabularDataServiceFactory.getService(aslSession
					.getUsername());
			TabularResource serviceTR = service.createTabularResource();
			Table table = service.getLastTable(serviceTR.getId());
			syncTRMetaData(serviceTR, tabResource);
			TRId trId = new TRId(String.valueOf(serviceTR.getId().getValue()),
					String.valueOf(table.getId().getValue()));
			tabResource.setTrId(trId);
			return tabResource;
		} catch (Exception e) {
			logger.error(
					"Error creating new TabularResource: "
							+ e.getLocalizedMessage(), e);
			throw new TDGWTServiceException(
					"Error creating new TabularResource: "
							+ e.getLocalizedMessage());
		}

	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ArrayList<Codelist> getCodelists() throws TDGWTServiceException {
		try {
			session = this.getThreadLocalRequest().getSession();
			return SessionUtil.retrieveCodelists(session);
		} catch (Exception e) {
			e.printStackTrace();
			throw new TDGWTServiceException("Error retrieving codelists: "
					+ e.getLocalizedMessage());
		}

	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ArrayList<Dataset> getDatasets() throws TDGWTServiceException {
		try {
			session = this.getThreadLocalRequest().getSession();
			return SessionUtil.retrieveDatasets(session);
		} catch (Exception e) {
			e.printStackTrace();
			throw new TDGWTServiceException("Error retrieving datasets: "
					+ e.getLocalizedMessage());
		}

	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ArrayList<Agencies> getAgencies() throws TDGWTServiceException {
		try {
			session = this.getThreadLocalRequest().getSession();
			return SessionUtil.retrieveAgencies(session);
		} catch (Exception e) {
			e.printStackTrace();
			throw new TDGWTServiceException("Error retrieving datasets: "
					+ e.getLocalizedMessage());
		}

	}

	/**
	 * 
	 * @param sdmxImportSession
	 * @return
	 */
	protected Map<String, Object> retrieveOperationParameters(
			SDMXImportSession sdmxImportSession) {

		Map<String, Object> map = new HashMap<String, Object>();

		Codelist codelist = sdmxImportSession.getSelectedCodelist();

		map.put("agency", codelist.getAgencyId());
		map.put("id", codelist.getId());
		map.put("version", codelist.getVersion());

		// TODO-GP: Get registry url from client

		SDMXRegistryDescriptor descriptor = new GCubeSDMXRegistryDescriptor();
		map.put("registryBaseUrl",
				descriptor.getUrl(SDMXRegistryInterfaceType.RESTV2_1));

		return map;

	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void startSDMXImport(SDMXImportSession sdmxImportSession)
			throws TDGWTServiceException {

		try {
			session = this.getThreadLocalRequest().getSession();
			SessionUtil.setSDMXImportSession(session, sdmxImportSession);
			aslSession = SessionUtil.getAslSession(session);

			service = TabularDataServiceFactory.getService(aslSession
					.getUsername());
			OperationInterface oService = service;
			TabularResourceInterface trService = service;

			List<OperationDescriptor> capabilities = service.getCapabilities();

			// Import SDMX Codelist takes id 200
			OperationDescriptor importSDMXCodelistOperation = getOperationDescriptorWithId(
					OperationsId.SDMXCodelistImport.toString(), capabilities);

			TabResource importSDMXTabResource = sdmxImportSession
					.getTabResource();

			TabularResource serviceTR = trService.createTabularResource();

			syncTRMetaData(serviceTR, importSDMXTabResource);
			TRId trId = new TRId(String.valueOf(serviceTR.getId().getValue()));
			importSDMXTabResource.setTrId(trId);

			SessionUtil
					.setSDMXImportTabResource(session, importSDMXTabResource);

			Map<String, Object> parameterInstance = retrieveOperationParameters(sdmxImportSession);

			InvocationCreator ic=InvocationCreator.getCreator(importSDMXCodelistOperation);			
			OperationInvocation invocation = ic.setParameters(parameterInstance).create();
			
			logger.debug("OperationInvocation: \n" + invocation.toString());
			Task trTask = oService.execute(invocation, serviceTR.getId());
			logger.debug("Start Task on service: TaskId " + trTask.getId());
			SessionUtil.setSDMXImportTask(session, trTask);
			return;
		} catch (Exception e) {
			e.printStackTrace();
			throw new TDGWTServiceException("Error in Client Library Request: "
					+ e.getLocalizedMessage());
		}
	}

	/**
	 * 
	 * @param op
	 * @param capabilities
	 * @return
	 * @throws TDGWTServiceException
	 */
	protected OperationDescriptor getOperationDescriptorWithId(String op,
			List<OperationDescriptor> capabilities) throws TDGWTServiceException {
		for (OperationDescriptor operation : capabilities) {
			if (Long.valueOf(op) == operation.getOperationId().getValue()) {
				return operation;
			}
			

		}
		throw new TDGWTServiceException("OperationDescriptor not found");

	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public SDMXImportMonitor getSDMXImportMonitor()
			throws TDGWTServiceException {
		try {
			session = this.getThreadLocalRequest().getSession();
			aslSession = SessionUtil.getAslSession(session);
			Task task = SessionUtil.getSDMXImportTask(session);
			TabResource sdmxImportTabResource = SessionUtil
					.getSDMXImportTabResource(session);
			TRId trId = sdmxImportTabResource.getTrId();

			SDMXImportMonitor importMonitor = new SDMXImportMonitor();
			if (trId == null) {
				throw new TDGWTServiceException(
						"Error in importSDMX TabularResource Id null");
			} else {
				if (task == null) {
					throw new TDGWTServiceException(
							"Error in importSDMX task null");
				} else {
					logger.debug("Service task: " + task.toString());
					TaskStatus status = task.getStatus();
					if (status == null) {
						throw new TDGWTServiceException(
								"Error in importSDMX Status null");
					} else {
						logger.debug("Service Task.getStatus(): "
								+ task.getStatus());
						importMonitor.setStatus(task.getStatus().ordinal());
						switch (importMonitor.getStatus()) {
						case SUCCEDED:
							importMonitor.setProgress(task.getProgress());
							logger.debug("Task Result: " + task.getResult());
							trId.setTableId(String.valueOf(task.getResult().getPrimaryTable().getId().getValue()));
							sdmxImportTabResource.setTrId(trId);
							SessionUtil.setSDMXImportTabResource(session,
									sdmxImportTabResource);
							SessionUtil.setTabResource(session,
									sdmxImportTabResource);

							SessionUtil.setTRId(session, trId);
							break;
						case FAILED:
							if (task.getResult() != null) {
								logger.debug("Task exception:"
										+ task.getErrorMessage());
								importMonitor
										.setError(new Throwable(task.getErrorMessage()));
							} else {
								logger.debug("Task exception: Error In Import");
								importMonitor.setError(new Throwable(
										"Error In Import"));
							}
							importMonitor.setProgress(task.getProgress());
							break;
						case ABORTED:
							break;
						case WAITING:
							break;
						case IN_PROGRESS:
							importMonitor.setProgress(task.getProgress());
							break;
						case INITIALIZING:
							break;
						default:
							break;
						}
					}
					SessionUtil.setSDMXImportTask(session, task);
				}

				importMonitor.setTrId(trId);
			}
			logger.debug("getSDMXImportMonitor(): " + importMonitor);
			return importMonitor;
		} catch (Exception e) {
			e.printStackTrace();
			throw new TDGWTServiceException(
					"Error in importSDMX ImportMonitor: "
							+ e.getLocalizedMessage());

		}

	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setSDMXSession(SDMXImportSession sdmxImportSession)
			throws TDGWTServiceException {
		try {
			session = this.getThreadLocalRequest().getSession();
			SessionUtil.setSDMXImportSession(session, sdmxImportSession);
			return;

		} catch (Exception e) {
			logger.error(
					"Error setting SDMXImportSession parameter: "
							+ e.getLocalizedMessage(), e);
			throw new TDGWTServiceException(
					"Error setting SDMXImportSession parameter: "
							+ e.getLocalizedMessage());
		}

	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setCSVSession(CSVImportSession importSession)
			throws TDGWTServiceException {
		try {
			session = this.getThreadLocalRequest().getSession();
			SessionUtil.setCSVImportSession(session, importSession);
			return;

		} catch (Exception e) {
			logger.error(
					"Error setting SDMXImportSession parameter: "
							+ e.getLocalizedMessage(), e);
			throw new TDGWTServiceException(
					"Error setting SDMXImportSession parameter: "
							+ e.getLocalizedMessage());
		}

	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FileUploadMonitor getFileUploadMonitor()
			throws TDGWTServiceException {

		session = this.getThreadLocalRequest().getSession();
		if (session == null) {
			throw new TDGWTServiceException(
					"Error retrieving the session: null");
		}
		FileUploadSession fileUploadSession = SessionUtil
				.getFileUploadSession(session);
		if (fileUploadSession == null) {
			throw new TDGWTServiceException(
					"Error retrieving the fileUploadSession: null");
		}
		return fileUploadSession.getFileUploadMonitor();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public AvailableCharsetList getAvailableCharset()
			throws TDGWTServiceException {
		session = this.getThreadLocalRequest().getSession();
		if (session == null) {
			throw new TDGWTServiceException(
					"Error retrieving the session: null");
		}
		logger.info("Session:" + session.getId());
		String defaultEncoding = Charset.defaultCharset().displayName();
		ArrayList<String> charsetList = new ArrayList<String>(Charset
				.availableCharsets().keySet());
		return new AvailableCharsetList(charsetList, defaultEncoding);
	}

	@Override
	public ArrayList<String> configureCSVParser(String encoding,
			HeaderPresence headerPresence, char delimiter, char comment)
			throws TDGWTServiceException {
		session = this.getThreadLocalRequest().getSession();
		if (session == null) {
			throw new TDGWTServiceException(
					"Error retrieving the session: null");
		}
		logger.info("Session:" + session.getId());
		logger.info("configureCSVParser  encoding: " + encoding
				+ " headerPresence: " + headerPresence + " delimiter: "
				+ delimiter + " comment: " + comment);

		FileUploadSession fileUploadSession = SessionUtil
				.getFileUploadSession(session);
		if (fileUploadSession == null) {
			throw new TDGWTServiceException(
					"Error retrieving the fileUploadSession: null");
		}
		CSVParserConfiguration parserConfiguration = fileUploadSession
				.getParserConfiguration();
		if (parserConfiguration == null) {
			parserConfiguration = new CSVParserConfiguration(
					Charset.forName(encoding), delimiter, comment,
					headerPresence);
			fileUploadSession.setParserConfiguration(parserConfiguration);
		} else {
			parserConfiguration.update(encoding, delimiter, comment,
					headerPresence);
		}
		SessionUtil.setFileUploadSession(session, fileUploadSession);
		try {
			return CSVFileUtil.getHeader(fileUploadSession.getCsvFile(),
					fileUploadSession.getParserConfiguration());
		} catch (Exception e) {
			logger.error("Error retrieving the CSV header", e);
			throw new TDGWTServiceException(
					"Error calculating the CSV header: "
							+ e.getLocalizedMessage());
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ArrayList<CSVRowError> checkCSV(long errorsLimit)
			throws TDGWTServiceException {

		session = this.getThreadLocalRequest().getSession();
		if (session == null) {
			throw new TDGWTServiceException(
					"Error retrieving the session: null");
		}
		logger.info("Session:" + session.getId());
		FileUploadSession fileUploadSession = SessionUtil
				.getFileUploadSession(session);
		if (fileUploadSession == null) {
			throw new TDGWTServiceException(
					"Error retrieving the fileUploadSession: null");
		}

		try {
			return CSVFileUtil.checkCSV(fileUploadSession.getCsvFile(),
					fileUploadSession.getParserConfiguration(), errorsLimit);
		} catch (Exception e) {
			logger.error("Error checking the CSV file", e);
			throw new TDGWTServiceException("Error checking the CSV file: "
					+ e.getLocalizedMessage());
		}

	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void startCSVImport(CSVImportSession csvImportSession)
			throws TDGWTServiceException {
		session = this.getThreadLocalRequest().getSession();
		if (session == null) {
			throw new TDGWTServiceException(
					"Error retrieving the session: null");
		}
		logger.info("Session:" + session.getId());

		aslSession = SessionUtil.getAslSession(session);
		if (aslSession == null) {
			throw new TDGWTServiceException(
					"Error retrieving the asl session: null");
		}
		String user = aslSession.getUsername();
		logger.info("Session User:" + user);

		FileUploadSession fileUploadSession = SessionUtil
				.getFileUploadSession(session);
		if (fileUploadSession == null) {
			throw new TDGWTServiceException(
					"Error retrieving the fileUploadSession: null");
		}

		importCSVFileOnService(user, fileUploadSession, csvImportSession);

	}

	/**
	 * 
	 * @param fileUrlOnStorage
	 * @param fileUploadSession
	 * @param csvImportSession
	 * @return
	 */
	protected Map<String, Object> csvImportFileParameter(
			String fileUrlOnStorage, FileUploadSession fileUploadSession,
			CSVImportSession csvImportSession) {
		final String ENCODING = "encoding";
		final String HASHEADER = "hasHeader";
		final String SEPARATOR = "separator";
		final String URL = "url";

		Map<String, Object> parameterInstances = new HashMap<String, Object>();
		parameterInstances.put(URL, fileUrlOnStorage);
		parameterInstances.put(SEPARATOR, String.valueOf(fileUploadSession
				.getParserConfiguration().getDelimiter()));// ','
		parameterInstances.put(ENCODING, fileUploadSession
				.getParserConfiguration().getCharset().name());// "UTF-8"
		boolean hasHeader = true;
		if (fileUploadSession.getParserConfiguration().getHeaderPresence() == HeaderPresence.NONE) {
			hasHeader = false;
		}
		parameterInstances.put(HASHEADER, hasHeader);// true
		return parameterInstances;
	}

	/**
	 * 
	 * @param user
	 * @param fileUploadSession
	 * @param csvImportSession
	 * @throws TDGWTServiceException
	 */
	protected void importCSVFileOnService(String user,
			FileUploadSession fileUploadSession,
			CSVImportSession csvImportSession) throws TDGWTServiceException {
		FilesStorage filesStorage = new FilesStorage();

		String fileUrlOnStorage = filesStorage.storageCSVTempFile(user,
				fileUploadSession.getCsvFile());

		Map<String, Object> parameterInstance = csvImportFileParameter(
				fileUrlOnStorage, fileUploadSession, csvImportSession);

		service = TabularDataServiceFactory
				.getService(aslSession.getUsername());
		List<OperationDescriptor> capabilities = service.getCapabilities();

		// Import CSV file
		
		OperationDescriptor importCSVFileOperation = getOperationDescriptorWithId(
				OperationsId.CSVImport.toString(), capabilities);

		InvocationCreator ic=InvocationCreator.getCreator(importCSVFileOperation);			
		OperationInvocation invocation = ic.setParameters(parameterInstance).create();
		
		logger.debug("OperationInvocation: \n" + invocation.toString());
		TabularResource tabularResource = service.createTabularResource();
		TabResource csvImportTabResource = csvImportSession.getTabResource();
		syncTRMetaData(tabularResource, csvImportTabResource);
		TRId trId = new TRId(String.valueOf(tabularResource.getId().getValue()));
		csvImportTabResource.setTrId(trId);
		SessionUtil.setCSVImportTabResource(session, csvImportTabResource);
		Task trTask;
		try {
			trTask = service.execute(invocation, tabularResource.getId());
		} catch (Exception e) {
			e.printStackTrace();
			throw new TDGWTServiceException(
					"Tabular Data Service error creating TabularResource: "
							+ e.getLocalizedMessage());
		}

		logger.debug("Start Task on service: TaskId " + trTask.getId());
		SessionUtil.setCSVImportFileTask(session, trTask);

	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public CSVImportMonitor getCSVImportMonitor() throws TDGWTServiceException {
		try {
			session = this.getThreadLocalRequest().getSession();
			aslSession = SessionUtil.getAslSession(session);
			Task task = SessionUtil.getCSVImportFileTask(session);
			TabResource csvImportTabResource = SessionUtil
					.getCSVImportTabResource(session);
			TRId trId = csvImportTabResource.getTrId();

			CSVImportMonitor importMonitor = new CSVImportMonitor();
			if (trId == null) {
				throw new TDGWTServiceException(
						"Error in importCSV TabularResource Id null");
			} else {
				if (task == null) {
					logger.debug("Task null");
					throw new TDGWTServiceException(
							"Error in importCSV task null");
				} else {
					TaskStatus status = task.getStatus();
					if (status == null) {
						throw new TDGWTServiceException(
								"Error in importCSV Status null");
					} else {
						logger.debug("Status: " + task.getStatus());
						importMonitor.setStatus(task.getStatus().ordinal());
						switch (importMonitor.getStatus()) {
						case FAILED:
							if (task.getResult() != null) {
								logger.debug("Task exception:"
										+ task.getErrorMessage());
								importMonitor
										.setError(new Throwable(task.getErrorMessage()));
							} else {
								logger.debug("Task exception: Error In Import");
								importMonitor.setError(new Throwable(
										"Error In Import"));
							}
							importMonitor.setProgress(task.getProgress());
							break;
						case SUCCEDED:
							logger.debug("Task Result:" + task.getResult());
							importMonitor.setProgress(task.getProgress());
							Table table = task.getResult().getPrimaryTable();

							logger.debug("Table retrived: " + table.toString());
							trId.setTableId(String.valueOf(table.getId()
									.getValue()));
							csvImportTabResource.setTrId(trId);
							SessionUtil.setCSVImportTabResource(session,
									csvImportTabResource);
							SessionUtil.setTabResource(session,
									csvImportTabResource);
							SessionUtil.setTRId(session, trId);
							break;
						case IN_PROGRESS:
							importMonitor.setProgress(task.getProgress());
							break;
						case ABORTED:
							break;
						case INITIALIZING:
							break;
						case WAITING:
							break;
						default:
							break;
						}
					}
					SessionUtil.setCSVImportFileTask(session, task);
				}

				importMonitor.setTrId(trId);
			}
			logger.info("getImportMonitor(): " + importMonitor);
			return importMonitor;
		} catch (Exception e) {
			e.printStackTrace();
			throw new TDGWTServiceException(
					"Error in importCSV CSVImportMonitor: "
							+ e.getLocalizedMessage());

		}

	}

	@Override
	public void getFileFromWorkspace(CSVImportSession csvImportSession)
			throws TDGWTServiceException {
		// TODO Auto-generated method stub
		session = this.getThreadLocalRequest().getSession();
		aslSession = SessionUtil.getAslSession(session);
		Workspace w = null;
		WorkspaceItem wi = null;

		try {
			HomeManagerFactory factory = HomeLibrary.getHomeManagerFactory();

			HomeManager manager = factory.getHomeManager();

			Home home = manager.getHome(aslSession.getUsername());

			w = home.getWorkspace();
			wi = w.getItem(csvImportSession.getItemId());
		} catch (Exception e) {
			e.printStackTrace();
			throw new TDGWTServiceException(
					"Error in importCSV getFileFromWorkspace accessing the workspace: "
							+ e.getLocalizedMessage(), e);
		}

		if (wi == null) {
			logger.error("Error retrieving the item on workspace"
					+ csvImportSession.getItemId());
			throw new TDGWTServiceException(
					"Error retrieving the item on workspace"
							+ csvImportSession.getItemId());
		}

		try {
			logger.debug("WorkspaceItem [id:" + wi.getId() + " name:"
					+ wi.getName() + " remotePath:" + wi.getRemotePath() + "]");
		} catch (InternalErrorException e1) {
			e1.printStackTrace();
			throw new TDGWTServiceException(
					"Error retrieving the item on workspace" + wi);
		}

		FileUploadSession fileUploadSession = new FileUploadSession();
		CSVImportMonitor csvImportMonitor = new CSVImportMonitor();
		FileUploadMonitor fileUploadMonitor = new FileUploadMonitor();

		fileUploadSession.setId(session.getId());
		fileUploadSession.setFileUploadState(FileUploadState.STARTED);
		fileUploadSession.setCsvImportMonitor(csvImportMonitor);
		fileUploadSession.setFileUploadMonitor(fileUploadMonitor);

		SessionUtil.setFileUploadSession(session, fileUploadSession);

		try {
			FilesStorage filesStorage = new FilesStorage();
			InputStream is = filesStorage.retriveImputStream(
					aslSession.getUsername(), wi);

			FileUtil.setImportFile(fileUploadSession, is, wi.getName(),
					"text/csv");
		} catch (Exception e) {
			fileUploadSession.getFileUploadMonitor().setFailed(
					"An error occured elaborating the file",
					FileUtil.exceptionDetailMessage(e));
			fileUploadSession.setFileUploadState(FileUploadState.FAILED);
			SessionUtil.setFileUploadSession(session, fileUploadSession);
			logger.error("Error elaborating the imput stream", e);
			throw new TDGWTServiceException(
					"Error in importCSV getFileFromWorkspace: "
							+ e.getLocalizedMessage(), e);
		}

		logger.trace("changing state");
		fileUploadSession.getFileUploadMonitor().setState(
				FileUploadState.COMPLETED);
		SessionUtil.setFileUploadSession(session, fileUploadSession);
	}

	@Override
	public ArrayList<TRMetadata> getTableMetadata(TRId trId)
			throws TDGWTServiceException {
		try {
			session = this.getThreadLocalRequest().getSession();
			aslSession = SessionUtil.getAslSession(session);

			service = TabularDataServiceFactory.getService(aslSession
					.getUsername());

			Table table = service.getTable(new TableId(Long.valueOf(trId.getTableId())));
			
			Collection<TableMetadata> cMeta=table.getAllMetadata();
			
			ArrayList<TRMetadata> listTRMetadata=new ArrayList<TRMetadata>();
			
			for(TableMetadata tMetadata:cMeta){
				if(tMetadata instanceof DescriptionsMetadata){
					TRDescriptionsMetadata trDescriptionsMetadata=new TRDescriptionsMetadata();
					ArrayList<TRLocalizedText> listTRLocalizedText=new ArrayList<TRLocalizedText>();
					List<LocalizedText> lLocalizedText=((DescriptionsMetadata) tMetadata).getTexts();
					int i=0;
					for(LocalizedText lt:lLocalizedText){
						TRLocalizedText trLocalizedText=new TRLocalizedText();
						trLocalizedText.setId(i);
						trLocalizedText.setValue(lt.getValue());
						trLocalizedText.setLocaleCode(lt.getLocale());
						listTRLocalizedText.add(trLocalizedText);
						i++;
					}
					trDescriptionsMetadata.setListTRLocalizedText(listTRLocalizedText);
					listTRMetadata.add(trDescriptionsMetadata);
				
				} else {
					if(tMetadata instanceof NamesMetadata){
						TRNamesMetadata trNamesMetadata=new TRNamesMetadata();
						ArrayList<TRLocalizedText> listTRLocalizedText=new ArrayList<TRLocalizedText>();
						List<LocalizedText> lLocalizedText=((NamesMetadata) tMetadata).getTexts();
						int i=0;
						for(LocalizedText lt:lLocalizedText){
							TRLocalizedText trLocalizedText=new TRLocalizedText();
							trLocalizedText.setId(i);
							trLocalizedText.setValue(lt.getValue());
							trLocalizedText.setLocaleCode(lt.getLocale());
							listTRLocalizedText.add(trLocalizedText);
							i++;
						}
						trNamesMetadata.setListTRLocalizedText(listTRLocalizedText);
						listTRMetadata.add(trNamesMetadata);
					
					} else {
						if(tMetadata instanceof VersionMetadata){
							TRVersionMetadata trVersionMetadata=new TRVersionMetadata();
							trVersionMetadata.setVersion(((VersionMetadata)tMetadata).getVersion());
							listTRMetadata.add(trVersionMetadata);
						}else{
							if(tMetadata instanceof ExportMetadata){
								TRExportMetadata trExportMetadata=new TRExportMetadata();
								trExportMetadata.setDestinationType(((ExportMetadata)tMetadata).getDestinationType());
								trExportMetadata.setExportDate(sdf.format(((ExportMetadata)tMetadata).getExportDate()));
								trExportMetadata.setUrl(((ExportMetadata)tMetadata).getUri());
								listTRMetadata.add(trExportMetadata);
							}else{
								if(tMetadata instanceof ImportMetadata){
									TRImportMetadata trImportMetadata=new TRImportMetadata();
									trImportMetadata.setSourceType(((ImportMetadata)tMetadata).getSourceType());
									trImportMetadata.setImportDate(sdf.format(((ImportMetadata)tMetadata).getImportDate()));
									trImportMetadata.setUrl(((ImportMetadata)tMetadata).getUri());
									listTRMetadata.add(trImportMetadata);
								}else{
									if(tMetadata instanceof GenericMapMetadata){
										TRGenericMapMetadata trGenericMapMetadata=new TRGenericMapMetadata();
										trGenericMapMetadata.setMetadataMap(((GenericMapMetadata)tMetadata).getMetadataMap());
										listTRMetadata.add(trGenericMapMetadata);
									}else{
										
									}
									
								}
							}
						}
					}
				}
			}
			
			
			return listTRMetadata;

		} catch (Exception e) {
			logger.error("Error in getTableMetadata(): " + e.getLocalizedMessage(),
					e);
			throw new TDGWTServiceException("Error in getTableMetadata(): "
					+ e.getLocalizedMessage());
		}
	}

}
