/**
 * 
 */
package org.gcube.portlets.user.uriresolvermanager;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;

import org.gcube.common.scope.api.ScopeProvider;
import org.gcube.portlets.user.uriresolvermanager.entity.ServiceParameter;
import org.gcube.portlets.user.uriresolvermanager.exception.IllegalArgumentException;
import org.gcube.portlets.user.uriresolvermanager.exception.UriResolverMapException;
import org.gcube.portlets.user.uriresolvermanager.readers.RuntimeResourceReader;
import org.gcube.portlets.user.uriresolvermanager.readers.UriResolverMapReader;
import org.gcube.portlets.user.uriresolvermanager.util.UrlEncoderUtil;
import org.gcube.portlets.user.urlshortener.UrlShortener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Francesco Mangiacrapa francesco.mangiacrapa@isti.cnr.it
 * @Oct 14, 2014
 *
 */
public class UriResolverManager {
	
	
	/**
	 * Time to reload Runtime Resource Configuration
	 */
	public static int RESET_DELAY = 10*60*1000; //10 MINUTES
	
	/**
	 * Time to reload Runtime Resource Configuration
	 */
	public static int RESET_TIME = RESET_DELAY; //10 MINUTES
	
	private UriResolverMapReader uriResolverMapReader;
	private Map<String, String> applicationTypes;
	private String scope;
	private String applicationType;
	private RuntimeResourceReader reader;
	
	/**
	 * A lock to prevent reader = null;
	 */
	private int usingReader = 0;
	
	public synchronized void lockReader() {
		usingReader++;
	}

	public synchronized void releaseReader() {
		usingReader--;
	}

	public synchronized int countReaders() {
		return usingReader;
	}
	
	public static final Logger logger = LoggerFactory.getLogger(UriResolverManager.class);
	
	/**
	 * Instance a UriResolverManager
	 * Precondition: set the scope provider {@link  ScopeProvider.instance.get()} 
	 * The scope is used to look up the generic resource {@link UriResolverMapReader#URI_RESOLVER_MAP} available in the infrastructure to map ApplicationType with its Resolver
	 * @param applicationType a key Application Type {@link  UriResolverManager#getApplicationTypes()} 
	 * @throws Exception 
	 */
	public UriResolverManager(String applicationType) throws UriResolverMapException, IllegalArgumentException{
		try {
			
			this.scope = ScopeProvider.instance.get();
			
			if(this.scope == null)
				throw new UriResolverMapException("Scope is null, set scope provider!");
					
			this.uriResolverMapReader = new UriResolverMapReader(this.scope);
			this.applicationTypes = uriResolverMapReader.getApplicationTypes();
			
		} catch (Exception e) {
			throw new UriResolverMapException("Map Application Type - Resources not found in IS");
		}
		
		if(!this.applicationTypes.containsKey(applicationType)){
			throw new IllegalArgumentException("Application type '"+applicationType +"' not found in Application Types: "+getApplicationTypes());
		}
		
		this.applicationType = applicationType;
		this.reloadRuntimeResourceParameter(RESET_DELAY, RESET_TIME);
	}
	
	/**
	 * 
	 * @param parameters the map of the parameters sent with HTTP request (link) generated
	 * @param shortLink if true the link is shorted otherwise none
	 * @return
	 * @throws IllegalArgumentException
	 * @throws UriResolverMapException
	 */
	public String getLink(Map<String, String> parameters, boolean shortLink) throws IllegalArgumentException, UriResolverMapException{
		
		String resourceName = this.applicationTypes.get(applicationType);
		String link;
		
		if(parameters==null){
			throw new IllegalArgumentException("Input Map parameters is null");
		}
		
		try {
			
			lockReader();
			
			if(reader==null){
				logger.info("Runtime Resource Reader is null, istancing...");
				ScopeProvider.instance.set(this.scope);
				reader = new RuntimeResourceReader(this.scope, resourceName);
			}
			
			List<ServiceParameter> resourceParameters = reader.getServiceParameters();
			
			//CHECK PARAMETERS
			for (ServiceParameter serviceParameter : resourceParameters) {	
				if(serviceParameter.isMandatory()){
					if(!parameters.containsKey(serviceParameter.getKey())){
						throw new IllegalArgumentException("Mandatory service key (parameter) '"+serviceParameter.getKey() +"' not found into input map");
					}
				}
			}
			
			String baseURI = reader.getServiceBaseURI();
			
			releaseReader();
			
			String params = UrlEncoderUtil.encodeQuery(parameters);
			link = baseURI+"?"+params;
			logger.info("Created HTTP URI request (link): "+link);
			
			if(shortLink){
				try{
					logger.info("Shortner start..");
					UrlShortener shortener = new UrlShortener(this.scope);
					link = shortener.shorten(link);
					logger.info("Shorted link is: "+link);
				}catch(Exception e){
					logger.warn("An error occurred on shortener the link ",e);
				}
			}
		} catch (IllegalArgumentException e){
			throw e;
		} catch (Exception e) {
			logger.error("Uri Resolver error: ", e);
			throw new UriResolverMapException("Uri Resolver error: " +e.getMessage());
		}	
		return link;
	}
	
	/**
	 * 
	 * @return the Application Types available
	 */
	public Set<String> getApplicationTypes(){
		return this.applicationTypes.keySet();
	}
	
	/**
	 * 
	 * @return a map Application Type - Resource Name
	 */
	public Map<String, String> getCapabilities(){
		return this.applicationTypes;
	}
	
	/**
	 * 
	 */
	public void reloadRuntimeResourceParameter(long delay, long period) {
		Timer timer = new Timer(true);
		
		timer.schedule(new TimerTask() {
			@Override
			public void run() {
				logger.info("Timer Reset Runtime Resource running..");
				int counters = countReaders();
				if(counters==0){
					logger.info("Reader not locked, resetting");
					reader = null;
				}else
					logger.info("Reader locked, counters is/are:"+counters+", skipping");
			
			}
		}, delay, period);
	}

}
