package org.gcube.portlets.user.vreappintegration.action;

import javax.portlet.ActionRequest;
import javax.portlet.ActionResponse;
import javax.portlet.PortletConfig;
import javax.portlet.PortletPreferences;

import org.gcube.portlets.user.vreappintegration.constants.VREAppIntegrationPortletKeys;
import org.osgi.service.component.annotations.Component;

import com.liferay.portal.configuration.metatype.bnd.util.ConfigurableUtil;
import com.liferay.portal.kernel.log.Log;
import com.liferay.portal.kernel.log.LogFactoryUtil;
import com.liferay.portal.kernel.portlet.DefaultConfigurationAction;
import com.liferay.portal.kernel.portlet.ConfigurationAction;
import com.liferay.portal.kernel.util.ParamUtil;

/**
 * Configuration Action for VRE App Integration Portlet
 * 
 * 📌 6.2 Code Analysis:
 * Original class extended DefaultConfigurationAction and used basic preference handling
 * 
 * 🔄 Differences in Liferay 7.4:
 * - Uses OSGi @Component annotation for service registration
 * - Must register as ConfigurationAction service, not DefaultConfigurationAction
 * - Enhanced parameter validation and error handling
 * - Updated to use ParamUtil for parameter extraction
 * - Improved logging with structured messages
 */
@Component(
    immediate = true,
    property = {
        "javax.portlet.name=" + VREAppIntegrationPortletKeys.VREAPPINTEGRATION
    },
    service = ConfigurationAction.class
)
public class VREAppConfigurationAction extends DefaultConfigurationAction {
    
    private static final Log _log = LogFactoryUtil.getLog(VREAppConfigurationAction.class);
    
    @Override
    public void processAction(
        PortletConfig portletConfig, ActionRequest actionRequest,
        ActionResponse actionResponse) throws Exception {
        
        try {
            // Call parent implementation first
            super.processAction(portletConfig, actionRequest, actionResponse);
            
            // Get portlet preferences
            PortletPreferences preferences = actionRequest.getPreferences();
            
            // Extract configuration parameters using ParamUtil (Liferay 7.4 best practice)
            // Note: JSP form fields use "preferences--paramName--" format
            String appURL = ParamUtil.getString(actionRequest, "preferences--" + VREAppIntegrationPortletKeys.PREF_APP_URL + "--", "");
            String appURLTokenParam = ParamUtil.getString(actionRequest, "preferences--" + VREAppIntegrationPortletKeys.PREF_APP_URL_TOKEN_PARAM + "--", "");
            String iFrameHeight = ParamUtil.getString(actionRequest, "preferences--" + VREAppIntegrationPortletKeys.PREF_IFRAME_HEIGHT + "--", VREAppIntegrationPortletKeys.DEFAULT_IFRAME_HEIGHT);
            String applicationName = ParamUtil.getString(actionRequest, "preferences--" + VREAppIntegrationPortletKeys.PREF_APPLICATION_NAME + "--", VREAppIntegrationPortletKeys.DEFAULT_APPLICATION_NAME);
            boolean newAuthVersion = ParamUtil.getBoolean(actionRequest, "preferences--" + VREAppIntegrationPortletKeys.PREF_NEW_AUTH_VERSION + "--", false);
            boolean newWindow = ParamUtil.getBoolean(actionRequest, "preferences--" + VREAppIntegrationPortletKeys.PREF_NEW_WINDOW + "--", false);
            
            // Validate parameters
            if (appURL != null && !appURL.trim().isEmpty()) {
                preferences.setValue(VREAppIntegrationPortletKeys.PREF_APP_URL, appURL.trim());
                _log.debug("Set appURL preference: " + appURL);
            }
            
            if (appURLTokenParam != null && !appURLTokenParam.trim().isEmpty()) {
                preferences.setValue(VREAppIntegrationPortletKeys.PREF_APP_URL_TOKEN_PARAM, appURLTokenParam.trim());
                _log.debug("Set appURLTokenParam preference: " + appURLTokenParam);
            }
            
            // Set iframe height with validation
            try {
                int height = Integer.parseInt(iFrameHeight);
                if (height > 0 && height <= 5000) { // Reasonable limits
                    preferences.setValue(VREAppIntegrationPortletKeys.PREF_IFRAME_HEIGHT, String.valueOf(height));
                    _log.debug("Set iFrameHeight preference: " + height);
                } else {
                    _log.warn("Invalid iframe height: " + height + ". Using default: " + VREAppIntegrationPortletKeys.DEFAULT_IFRAME_HEIGHT);
                    preferences.setValue(VREAppIntegrationPortletKeys.PREF_IFRAME_HEIGHT, VREAppIntegrationPortletKeys.DEFAULT_IFRAME_HEIGHT);
                }
            } catch (NumberFormatException e) {
                _log.warn("Invalid iframe height format: " + iFrameHeight + ". Using default: " + VREAppIntegrationPortletKeys.DEFAULT_IFRAME_HEIGHT);
                preferences.setValue(VREAppIntegrationPortletKeys.PREF_IFRAME_HEIGHT, VREAppIntegrationPortletKeys.DEFAULT_IFRAME_HEIGHT);
            }
            
            // Set application name
            if (applicationName != null && !applicationName.trim().isEmpty()) {
                preferences.setValue(VREAppIntegrationPortletKeys.PREF_APPLICATION_NAME, applicationName.trim());
                _log.debug("Set applicationName preference: " + applicationName);
            }
            
            // Set boolean preferences
            preferences.setValue(VREAppIntegrationPortletKeys.PREF_NEW_AUTH_VERSION, String.valueOf(newAuthVersion));
            preferences.setValue(VREAppIntegrationPortletKeys.PREF_NEW_WINDOW, String.valueOf(newWindow));
            
            _log.debug("Set newAuthVersion preference: " + newAuthVersion);
            _log.debug("Set newWindow preference: " + newWindow);
            
            // Store preferences
            preferences.store();
            
            _log.info("VRE App Integration portlet configuration updated successfully");
            
        } catch (Exception e) {
            _log.error("Error processing VRE App Integration portlet configuration", e);
            throw e;
        }
    }
    

}