package org.gcube.resourcemanagement.resource;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.gcube.com.fasterxml.jackson.annotation.JsonIgnore;
import org.gcube.com.fasterxml.jackson.databind.ObjectMapper;
import org.gcube.informationsystem.model.reference.entities.Resource;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.resourcemanagement.analyser.InstanceAnalyser;
import org.gcube.resourcemanagement.analyser.InstanceAnalyserFactory;

import jakarta.ws.rs.WebApplicationException;

/**
 * @author Luca Frosini (ISTI - CNR)
 * 
 * Represents a group of related resources that are derived from a common ancestor.
 * This class implements the Comparable interface to allow comparison based on the name of the resource group.
 *
 * @param <I> the type of instances contained in the resource group, which extends the Instance class.
 */
public class DerivatedRelatedResourceGroup<I extends Instance> implements Comparable<DerivatedRelatedResourceGroup<I>> {

    /**
     * The ancestor resource from which this group is derived.
     */
    @JsonIgnore
    protected Resource ancestor;

    /**
     * The ObjectMapper used for JSON serialization and deserialization.
     */
    @JsonIgnore
    protected final ObjectMapper mapper;
    
    /**
     * The class type of the instances contained in the resource group.
     */
    @JsonIgnore
    protected final Class<I> instanceClass;

    /**
     * The name of the resource group.
     */
    protected String name;

    /**
     * The description of the resource group.
     */
    protected String description;

    /**
     * The minimum number of instances in the resource group.
     */
    protected int min;

    /**
     * The maximum number of instances in the resource group.
     */
    protected Integer max;
    
    /**
     * The list of instances contained in the resource group.
     */
    protected Set<I> instances;

    /**
     * The list of related resources associated with this resource group.
     */
    @JsonIgnore
    protected List<Resource> relatedResources;

    /**
     * Constructs a new DerivatedRelatedResourceGroup with the specified instance class type.
     *
     * @param instanceClass the class type of the instances contained in the resource group
     */
    public DerivatedRelatedResourceGroup(Class<I> instanceClass) {
        this.instanceClass = instanceClass;
        this.mapper = new ObjectMapper();
    }

    /**
     * Returns the class type of the instances contained in the resource group.
     *
     * @return the class type of the instances
     */
    @JsonIgnore
    public Class<I> getInstanceClass() {
        return instanceClass;
    }
    
    /**
     * Returns the ancestor resource from which this group is derived.
     *
     * @return the ancestor resource
     */
    @JsonIgnore
    public Resource getAncestor() {
        return ancestor;
    }

    /**
     * Sets the ancestor resource from which this group is derived.
     *
     * @param ancestor the ancestor resource
     */
    @JsonIgnore
    public void setAncestor(Resource ancestor) {
        this.ancestor = ancestor;
    }

    /**
     * Returns the name of the resource group.
     *
     * @return the name of the resource group
     */
    public String getName() {
        return name;
    }

    /**
     * Sets the name of the resource group.
     *
     * @param name the name of the resource group
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * Returns the description of the resource group.
     *
     * @return the description of the resource group
     */
    public String getDescription() {
        return description;
    }

    /**
     * Sets the description of the resource group.
     *
     * @param description the description of the resource group
     */
    public void setDescription(String description) {
        this.description = description;
    }

    /**
     * Returns the minimum number of instances in the resource group.
     *
     * @return the minimum number of instances
     */
    public int getMin() {
        return min;
    }

    /**
     * Sets the minimum number of instances in the resource group.
     *
     * @param min the minimum number of instances
     */
    public void setMin(int min) {
        this.min = min;
    }

    /**
     * Returns the maximum number of instances in the resource group.
     *
     * @return the maximum number of instances
     */
    public Integer getMax() {
        return max;
    }

    /**
     * Sets the maximum number of instances in the resource group.
     *
     * @param max the maximum number of instances
     */
    public void setMax(Integer max) {
        this.max = max;
    }

    /**
     * Returns the list of related resources associated with this resource group.
     * @return the list of related resources
     */
    @JsonIgnore
    public List<Resource> getRelatedResources() {
        return relatedResources;
    }

    /**
     * Sets the list of related resources associated with this resource group.
     * @param relatedResources the list of related resources
     */
    @JsonIgnore
    public void setRelatedResources(List<Resource> relatedResources) {
        this.relatedResources = relatedResources;
    }

    /**
     * Returns the list of instances contained in the resource group.
     * If the instances are not already initialized, they are derived from the related resources.
     * Please note that Instance is a format to represent the summary of a resource, 
     * so the instances are just a minimal representation of the related resources.
     * In other words, the instances and related resources are two sides of the same coin.
     * 
     * @return the list of instances
     * @throws WebApplicationException if there is an error during the web application process
     * @throws ResourceRegistryException if there is an error with the resource registry
     */
    @SuppressWarnings("unchecked")
    public Set<I> getInstances() throws WebApplicationException, ResourceRegistryException {
        if(instances==null && relatedResources!=null){
            this.instances = Instance.getAsInstances(instanceClass, relatedResources);
            if(min==1 && max==1 && relatedResources.size()==1){
                Resource r = relatedResources.iterator().next();
                InstanceAnalyser<Resource, Instance> analyser = InstanceAnalyserFactory.getInstanceAnalyser(r.getTypeName(), r.getID());
                analyser.setAncestor(ancestor);
                analyser.setResource(r);
                I inst =  (I) analyser.getInstance();
                instances.add(inst);
            }
        }
        return instances;
    }


    public void setInstances(Set<I> instances) {
        this.instances = instances;
    }

    public void addInstance(I instance) {
        if(this.instances==null){
            this.instances = new HashSet<>();
        }
        this.instances.add(instance);
    }

    public void addInstances(Collection<I> instances) {
        if(this.instances==null){
            this.instances = new HashSet<>();
        }
        this.instances.addAll(instances);
    }

    /**
     * {@inheritDoc}
     * Returns a JSON representation of the DerivatedRelatedResourceGroup.
     * @return the JSON representation
     */
    @Override
    public String toString(){
        try {
            return mapper.writeValueAsString(this);
        } catch (Exception e) {
            return super.toString();
        }
    }

    /**
     * {@inheritDoc}
     * Compare two DerivatedRelatedResourceGroup by name
     * @param o
     * @return the comparison result
     */
    @Override
    public int compareTo(DerivatedRelatedResourceGroup<I> o) {
        return this.getName().compareTo(o.getName());
    }

    /**
     * {@inheritDoc}
     * Returns the hash code of the DerivatedRelatedResourceGroup based on the name.
     */
    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((name == null) ? 0 : name.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        @SuppressWarnings("rawtypes")
        DerivatedRelatedResourceGroup other = (DerivatedRelatedResourceGroup) obj;
        if (name == null) {
            if (other.name != null)
                return false;
        } else if (!name.equals(other.name))
            return false;
        return true;
    }

}
